/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot;

import java.io.File;

import edu.duke.donaldLab.jdshot.context.DistanceRestraintsContext;
import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.grid.PointIterator;
import edu.duke.donaldLab.jdshot.grid.PointReader;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.minimize.MinimizerClient;
import edu.duke.donaldLab.jdshot.minimize.MinimizerServer;
import edu.duke.donaldLab.share.io.ArgumentType;
import edu.duke.donaldLab.share.io.ArgumentsProcessor;
import edu.duke.donaldLab.share.io.Logging;
import edu.duke.donaldLab.share.mpi.Client;
import edu.duke.donaldLab.share.mpi.MPIEndpoint;
import edu.duke.donaldLab.share.mpi.Server;
import edu.duke.donaldLab.share.mpi.ClientServerCreator;

public class MinimizeMain
{
	/**************************
	 *   Static Methods
	 **************************/
	
	public static void main( String[] args )
	throws Exception
	{
		Logging.Mpi.init();
		
		// process the arguments
		ArgumentsProcessor argproc = new ArgumentsProcessor();
		argproc.add( "workDir", ArgumentType.Folder, "path to folder in which temporary files will be written" );
		argproc.add( "symmetry", ArgumentType.String, "symmetry type (e.g., Cn, Dn)" );
		argproc.add( "monomerPath", ArgumentType.InFile, "path to the monomer protein" );
		argproc.add( "numSubunits", ArgumentType.String, "number of subunits in the oligomer" );
		argproc.add( "inPath", ArgumentType.InFile, "path to input points/cells" );
		argproc.add( "noesPath", ArgumentType.InFile, "path to the noes" );
		argproc.add( "scoresPath", ArgumentType.OutFile, "path to output scores file" );
		argproc.add( "logPath", ArgumentType.OutFile, "path for log" );
		argproc.add( "structuresDir", ArgumentType.Folder, "path to dir to hold minimized PDB files" );
		argproc.process( args );
		
		// read in arguments
		final File workDir = argproc.getFile( "workDir" );
		final Symmetry symmetry = Symmetry.valueOf( argproc.getString( "symmetry" ) );
		final File monomerFile = argproc.getFile( "monomerPath" );
		final int numSubunits = argproc.getInteger( "numSubunits" );
		final File inFile = argproc.getFile( "inPath" );
		final File noesFile = argproc.getFile( "noesPath" );
		final File scoresFile = argproc.getFile( "scoresPath" );
		final File logFile = argproc.getFile( "logPath" );
		final File structuresDir = argproc.getFile( "structuresDir" );
		
		// closures could make this prettier, but oh well
		MPIEndpoint.initClientServer(
			args,
			logFile,
			new ClientServerCreator( )
			{
				public Server createServer( )
				throws Exception
				{
					return new MinimizerServer(
						new PointIterator( new PointReader( inFile, symmetry ) ),
						scoresFile
					);
				}
				
				public Client createClient( )
				throws Exception
				{
					SymmetryContext symmetryContext = new SymmetryContext(
						symmetry,
						monomerFile,
						numSubunits
					);
					DistanceRestraintsContext noesContext = new DistanceRestraintsContext( symmetryContext, noesFile );
					return new MinimizerClient( workDir, structuresDir, symmetryContext, noesContext );
				}
			}
		);
	}
}
