/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.analyze;

import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.StructureGenerator;
import edu.duke.donaldLab.jdshot.search.SearchContext;
import edu.duke.donaldLab.share.analysis.RestraintCalculator;
import edu.duke.donaldLab.share.analysis.RmsdCalculator;
import edu.duke.donaldLab.share.analysis.StructureAligner;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Subunit;

public class SCSAnalyzer
{
	/**************************
	 *   Data Members
	 **************************/
	
	private SearchContext m_searchContext;
	private AnalysisContext m_analysisContext;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public SCSAnalyzer( SearchContext searchContext, AnalysisContext analysisContext )
	{
		m_searchContext = searchContext;
		m_analysisContext = analysisContext;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public double getReferenceRmsd( Protein referenceBackbone, GridPoint point )
	{
		int numSubunits = referenceBackbone.getSubunits().size();
		
		// generate the computed structure and swap the subunits
		StructureGenerator structureGenerator = m_searchContext.getSymmetry().newStructureGenerator( m_searchContext.getMonomer() );
		Protein computedBackbone = structureGenerator.getStructure( point, numSubunits ).getBackbone();
		m_searchContext.getSubunitOrder().convertComputedToReference( computedBackbone );
		
		// just in case...
		assert( referenceBackbone.getNumAtoms() == computedBackbone.getNumAtoms() );
		
		// align the two backbones optimally
		StructureAligner.alignOptimally( referenceBackbone, computedBackbone );
		
		// compute the rmsd
		return RmsdCalculator.getRmsd( referenceBackbone, computedBackbone );
	}
	
	public double getNoeRmsd( GridPoint point )
	{
		// setup
		Subunit monomer = m_analysisContext.getSparseMonomer();
		StructureGenerator structureGenerator = m_searchContext.getSymmetry().newStructureGenerator( monomer );
		RestraintCalculator restraintCalculator = new RestraintCalculator();
		
		// do eeet!
		Protein structure = structureGenerator.getStructure( point, m_searchContext.getNumSubunits() );
		m_searchContext.getSubunitOrder().convertComputedToReference( structure );
		return restraintCalculator.getRmsd( structure, m_searchContext.getRestraints() );
	}
}
