/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.chart;

import java.awt.Graphics2D;
import java.awt.geom.Arc2D;
import java.awt.geom.Rectangle2D;
import java.util.List;

import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.Range;

import edu.duke.donaldLab.jdshot.disco.FaceInfo;
import edu.duke.donaldLab.share.geom.CircularArc;

public class FaceSubrenderer implements Subrenderer
{
	@Override
	@SuppressWarnings( "unchecked" )
	public Range findDomainBounds( GeometryDataset dataset, int series )
	{
		List<FaceInfo> faces = (List<FaceInfo>)dataset.getData( series );
		
		// for each face...
		double lower = Double.POSITIVE_INFINITY;
		double upper = Double.NEGATIVE_INFINITY;
		for( FaceInfo face : faces )
		{
			lower = Math.min( lower, face.boundingBox.getXMin() );
			upper = Math.max( upper, face.boundingBox.getXMax() );
		}
        return new Range( lower, upper );
	}

	@Override
	@SuppressWarnings( "unchecked" )
	public Range findRangeBounds( GeometryDataset dataset, int series )
	{
		List<FaceInfo> faces = (List<FaceInfo>)dataset.getData( series );
		
		// for each face...
		double lower = Double.POSITIVE_INFINITY;
		double upper = Double.NEGATIVE_INFINITY;
		for( FaceInfo face : faces )
		{
			lower = Math.min( lower, face.boundingBox.getYMin() );
			upper = Math.max( upper, face.boundingBox.getYMax() );
		}
        return new Range( lower, upper );
	}
	
	@Override
	public void drawItem( Graphics2D g2, Rectangle2D dataArea, XYPlot plot,
		ValueAxis xAxis, ValueAxis yAxis, GeometryDataset dataset,
		int series, int item, GeometryRenderer renderer )
	{
		FaceInfo face = (FaceInfo)dataset.getData( series ).get( item );
		for( CircularArc arc : face.edges )
		{
			// calculate the coordinate transform
			double tranX = xAxis.valueToJava2D( arc.circle.center.x, dataArea, plot.getDomainAxisEdge() );
			double tranXZero = xAxis.valueToJava2D( 0.0, dataArea, plot.getDomainAxisEdge() );
			double tranXRadius = Math.abs( xAxis.valueToJava2D( arc.circle.radius, dataArea, plot.getDomainAxisEdge() ) - tranXZero );
			
			double tranY = yAxis.valueToJava2D( arc.circle.center.y, dataArea, plot.getRangeAxisEdge() );
			double tranYZero = yAxis.valueToJava2D( 0.0, dataArea, plot.getRangeAxisEdge() );
			double tranYRadius = Math.abs( yAxis.valueToJava2D( arc.circle.radius, dataArea, plot.getRangeAxisEdge() ) - tranYZero );
			
			double length = arc.thetaStop - arc.thetaStart;
			
			// HACKHACK: if the length is a semi-circle, it should always be drawn counter-clockwise
			if( Math.abs( length ) == Math.PI )
			{
				length = Math.abs( length );
			}
			
			Arc2D drawArc = new Arc2D.Double(
				tranX - tranXRadius,
				tranY - tranYRadius,
				tranXRadius * 2,
				tranYRadius * 2,
				Math.toDegrees( arc.thetaStart ),
				Math.toDegrees( length ),
				Arc2D.OPEN
			);
			
			// do we need to draw the arc?
			if( !drawArc.intersects( dataArea ) )
	        {
	        	continue;
	        }
			
			// draw the arc
			if( renderer.getItemOutlinePaint( series, item ) != null )
			{
				g2.setPaint( renderer.getItemOutlinePaint( series, item ) );
				g2.setStroke( renderer.getItemOutlineStroke( series, item ) );
				g2.draw( drawArc );
			}
			
			/* actually, don't draw the vertices
			// draw the vertex
			if( renderer.getItemOutlinePaint( series, item ) != null )
			{
				Vector2 start = arc.getStart();
				tranX = xAxis.valueToJava2D( start.x, dataArea, plot.getDomainAxisEdge() );
				tranY = yAxis.valueToJava2D( start.y, dataArea, plot.getRangeAxisEdge() );
				double radius = 2.0 * ((BasicStroke)renderer.getItemOutlineStroke( series, item )).getLineWidth();
				g2.setPaint( renderer.getItemOutlinePaint( series, item ) );
				g2.fill( new Ellipse2D.Double(
					tranX - radius,
					tranY - radius,
					radius * 2.0,
					radius * 2.0
				) );
			}
			*/
		}
	}
}
