/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

public class CellReader
{
	/**************************
	 *   Data Members
	 **************************/
	
	private DataInputStream m_in;
	private long m_numCells;
	private long m_numCellsRead;
	private Symmetry m_symmetry;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public CellReader( String path, Symmetry symmetry )
	throws IOException
	{
		this( new File( path ), symmetry );
	}
	
	public CellReader( File file, Symmetry symmetry )
	throws IOException
	{
		m_symmetry = symmetry;
		
		// this is kind of a hack, but determine the number of cells by looking at how big the file is
		long fileLen = file.length();
		int bytesPerCell = getBytesPerCell();
		assert( fileLen % bytesPerCell == 0 );
		m_numCells = fileLen / bytesPerCell;
		
		// open the file for reading
		m_in = new DataInputStream( new BufferedInputStream( new FileInputStream( file ) ) );
		
		m_numCellsRead = 0;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public long getNumCells( )
	{
		return m_numCells;
	}
	
	public long getNumCellsRead( )
	{
		return m_numCellsRead;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public GridCell readCell( )
	throws IOException
	{
		// read the min point
		GridPoint min = m_symmetry.newGridPoint();
		for( int i=0; i<m_symmetry.getNumDimensions(); i++ )
		{
			min.set( i, m_in.readDouble() );
		}
		
		// read the max point
		GridPoint max = m_symmetry.newGridPoint();
		for( int i=0; i<m_symmetry.getNumDimensions(); i++ )
		{
			max.set( i, m_in.readDouble() );
		}
		
		// build the cell
		GridCell cell = m_symmetry.newGridCell();
		cell.setMin( min );
		cell.setMax( max );
		
		m_numCellsRead++;
		
		return cell;
	}
	
	public boolean hasNext( )
	{
		return m_numCellsRead < m_numCells;
	}
	
	public void close( )
	throws IOException
	{
		m_in.close();
		m_in = null;
	}
	

	/**************************
	 *   Functions
	 **************************/
	
	private int getBytesPerCell( )
	{
		return m_symmetry.getNumDimensions() * 8 * 2;
	}
}
