/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.dn;

import java.io.Serializable;

import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.share.math.MultiVector;

public class DnGridPoint implements Serializable, GridPoint
{
	/**************************
	 *   Definitions
	 **************************/
	
	private static final long serialVersionUID = 5538757590775225465L;
	public static final Symmetry SymmetryType = Symmetry.Dn;
	
	
	/**************************
	 *   Fields
	 **************************/
	
	public double x;
	public double y;
	public double z;
	public double phi;
	public double theta;
	public double rho;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DnGridPoint( )
	{
		x = 0.0;
		y = 0.0;
		z = 0.0;
		phi = 0.0;
		theta = 0.0;
		rho = 0.0;
	}
	
	public DnGridPoint( DnGridPoint other )
	{
		set( other );
	}
	
	public DnGridPoint( double a, double b, double c, double d, double e, double f )
	{
		x = a;
		y = b;
		z = c;
		phi = d;
		theta = e;
		rho = f;
	}
	
	public DnGridPoint( MultiVector point )
	{
		this(
			point.get( 0 ),
			point.get( 1 ),
			point.get( 2 ),
			point.get( 3 ),
			point.get( 4 ),
			point.get( 5 )
		);
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	@Override
	public double get( int i )
	{
		switch( i )
		{
			case 0: return x;
			case 1: return y;
			case 2: return z;
			case 3: return phi;
			case 4: return theta;
			case 5: return rho;
		}
		
		assert( false ) : "Invalid index: " + i;
		
		// just to make the compiler happy
		return Double.NaN;
	}
	
	@Override
	public void set( int i, double value )
	{
		switch( i )
		{
			case 0: x = value; return;
			case 1: y = value; return;
			case 2: z = value; return;
			case 3: phi = value; return;
			case 4: theta = value; return;
			case 5: rho = value; return;
		}

		assert( false ) : "Invalid index: " + i;
	}
	
	public void set( DnGridPoint other )
	{
		x = other.x;
		y = other.y;
		z = other.z;
		phi = other.phi;
		theta = other.theta;
		rho = other.rho;
	}
	
	@Override
	public double[] getArray( )
	{
		return new double[] { x, y, z, phi, theta, rho };
	}
	
	public void setArray( double[] values )
	{
		x = values[0];
		y = values[1];
		z = values[2];
		phi = values[3];
		theta = values[4];
		rho = values[5];
	}
	
	@Override
	public Symmetry getSymmetry( )
	{
		return SymmetryType;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public String toString( )
	{
		return "[DnGridPoint] (" + x + "," + y + "," + z + "),(" + phi + "," + theta + ")," + rho;
	}
	
	public String toCode( )
	{
		return "new DnGridPoint( " + x + ", " + y + ", " + z + ", " + phi + ", " + theta + ", " + rho + " )";
	}
	
	public void add( DnGridPoint other )
	{
		x += other.x;
		y += other.y;
		z += other.z;
		phi += other.phi;
		theta += other.theta;
		rho += other.rho;
	}
	
	public void subtract( DnGridPoint other )
	{
		x -= other.x;
		y -= other.y;
		z -= other.z;
		phi -= other.phi;
		theta -= other.theta;
		rho -= other.rho;
	}
	
	public void scale( double scalar )
	{
		x *= scalar;
		y *= scalar;
		z *= scalar;
		phi *= scalar;
		theta *= scalar;
		rho *= scalar;
	}
}
