/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.stericFilter;

import java.io.IOException;
import java.util.ArrayList;

import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.PointIterator;
import edu.duke.donaldLab.jdshot.grid.PointWriter;
import edu.duke.donaldLab.share.perf.Progress;

public class StericFilterSynchronizer
{
	/**************************
	 *   Data Members
	 **************************/
	
	private PointIterator m_iterPoint;
	private Progress m_progress;
	private PointWriter m_writer;
	private int m_batchSize;
	private int m_numProcessed;
	private int m_numAccepted;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public StericFilterSynchronizer( PointIterator iterPoint, Progress progress, PointWriter writer, int batchSize )
	{
		// save parameters
		m_iterPoint = iterPoint;
		m_progress = progress;
		m_writer = writer;
		m_batchSize = batchSize;
		
		// init defaults
		m_numProcessed = 0;
		m_numAccepted = 0;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public int getNumProcessed( )
	{
		return m_numProcessed;
	}
	
	public int getNumAccepted( )
	{
		return m_numAccepted;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public synchronized ArrayList<GridPoint> getPoints( )
	{
		ArrayList<GridPoint> points = new ArrayList<GridPoint>();
		for( int i=0; i<m_batchSize && m_iterPoint.hasNext(); i++ )
		{
			points.add( m_iterPoint.next() );
		}
		
		return points;
	}
	
	public synchronized void acceptPoint( GridPoint point )
	throws IOException
	{
		// write out the point
		m_writer.writePoint( point );
		
		// update counters
		m_numProcessed++;
		m_numAccepted++;
		m_progress.incrementProgress();
	}
	
	public synchronized void rejectPoint( GridPoint point )
	{
		// update counters
		m_numProcessed++;
		m_progress.incrementProgress();
	}
}
