/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.test.disco;

import java.io.File;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import edu.duke.donaldLab.jdshot.context.DistanceRestraintsContext;
import edu.duke.donaldLab.jdshot.context.RdcsContext;
import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.disco.AnnulusCalculator;
import edu.duke.donaldLab.jdshot.disco.AnnulusIndex;
import edu.duke.donaldLab.jdshot.disco.FaceInfo;
import edu.duke.donaldLab.jdshot.disco.OrientationCalculator;
import edu.duke.donaldLab.jdshot.disco.PositionCalculator;
import edu.duke.donaldLab.jdshot.disco.ResultsReader;
import edu.duke.donaldLab.jdshot.disco.cgal.Arrangement;
import edu.duke.donaldLab.jdshot.disco.cgal.ShotCgal;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.test.ExtendedTestCase;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.nmr.AlignmentTensor;

public class TestPositionCalculator extends ExtendedTestCase
{
	public void testDagkArrangement( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/2KDC.monomer.protein" ), 3 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/2KDC.experimental.rdc" );
		DistanceRestraintsContext distanceRestraintsContext = new DistanceRestraintsContext( symmetryContext, "src/resources/test/2KDC.experimental.onlyDisulfideBonds.noe" );
		
		ShotCgal.init();
		try
		{
			// previous steps
			AlignmentTensor tensor = AlignmentTensor.compute( symmetryContext.getMonomer(), rdcsContext.getInternalRdcs() );
			Vector3 orientation = OrientationCalculator.computeOrientations( 3, tensor ).get( 0 );
			OrientationCalculator.normalizeProtein( symmetryContext.getMonomer(), orientation );
			AnnulusIndex annulusIndex = AnnulusCalculator.computeAnnuli( symmetryContext.getMonomer(), distanceRestraintsContext.getInternalRestraints(), 3 );
			
			// build and check the arrangement
			assertEquals( 1458, PositionCalculator.computeArrangementIncremental( Arrays.asList( annulusIndex ) ).getNumFaces() );
		}
		finally
		{
			ShotCgal.cleanup();
		}
	}
	
	public void testMergeFaces( )
	throws Exception
	{
		// load the faces we're going to merge
		List<FaceInfo> msrs = new ResultsReader().readMsrs( new File( "src/resources/test/2KDC.facesToMerge.dat" ) );
		
		assertEquals( 62, msrs.size() );
		
		// call mergeFaces() using reflection since it's private
		Method method = PositionCalculator.class.getDeclaredMethod( "mergeFaces", ArrayList.class );
		method.setAccessible( true );
		method.invoke( null, msrs );
		
		assertEquals( 1, msrs.size() );
	}
	
	public void testMsrsSingle( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/2KDC.monomer.protein" ), 3 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/2KDC.experimental.rdc" );
		DistanceRestraintsContext distanceRestraintsContext = new DistanceRestraintsContext( symmetryContext, "src/resources/test/2KDC.experimental.onlyDisulfideBonds.noe" );
		
		ShotCgal.init();
		try
		{
			// previous steps
			AlignmentTensor tensor = AlignmentTensor.compute( symmetryContext.getMonomer(), rdcsContext.getInternalRdcs() );
			Vector3 orientation = OrientationCalculator.computeOrientations( 3, tensor ).get( 0 );
			OrientationCalculator.normalizeProtein( symmetryContext.getMonomer(), orientation );
			AnnulusIndex annulusIndex = AnnulusCalculator.computeAnnuli( symmetryContext.getMonomer(), distanceRestraintsContext.getInternalRestraints(), 3 );
			Arrangement arrangement = PositionCalculator.computeArrangementIncremental( Arrays.asList( annulusIndex ) );
			
			// compute the MSRs
			List<FaceInfo> msrs = PositionCalculator.computeMsrs( arrangement, Arrays.asList( annulusIndex ) );
			assertEquals( 1, msrs.size() );
		}
		finally
		{
			ShotCgal.cleanup();
		}
	}
}
