/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.analysis;

import java.util.Iterator;
import java.util.List;

import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Subunit;

public class RmsdCalculator
{
	/**************************
	 *   Static Methods
	 **************************/
	
	// UNDONE: the HasAtoms interface would work well here!
	
	public static double getRmsd( Protein a, Protein b )
	{
		// just in case...
		assert( a.getNumAtoms() == b.getNumAtoms() );
		
		return getRmsd( a, b, a.getAtomIndex(), b.getAtomIndex() );
	}
	
	public static double getBackboneRmsd( Protein a, Protein b )
	{
		// just in case...
		assert( a.getNumAtoms() == b.getNumAtoms() );
		
		return getRmsd( a, b, a.getBackboneAtomIndex(), b.getBackboneAtomIndex() );
	}
	
	public static double getRangeRmsd( Protein a, Protein b, int startResidueId, int stopResidueId )
	{
		return getRmsd(
			a,
			b,
			a.getAtomsByResidueRange( startResidueId, stopResidueId ),
			b.getAtomsByResidueRange( startResidueId, stopResidueId )
		);
	}
	
	public static double getBackboneRangeRmsd( Protein a, Protein b, int startResidueId, int stopResidueId )
	{
		return getRmsd(
			a,
			b,
			a.getBackboneAtomsByResidueRange( startResidueId, stopResidueId ),
			b.getBackboneAtomsByResidueRange( startResidueId, stopResidueId )
		);
	}
	
	public static double getRmsd( Protein a, Protein b, Iterator<AtomAddressInternal> iterAtomA, Iterator<AtomAddressInternal> iterAtomB )
	{
		double sum = 0.0;
		int count = 0;
		
		// for each atom...
		while( iterAtomA.hasNext() && iterAtomB.hasNext() )
		{
			// get atoms
			Vector3 posA = new Vector3( a.getAtom( iterAtomA.next() ).getPosition() );
			Vector3 posB = new Vector3( b.getAtom( iterAtomB.next() ).getPosition() );
			
			// add the sd
			sum += posB.getSquaredDistance( posA );
			count++;
		}
		
		// just in case...
		assert( iterAtomA.hasNext() == iterAtomB.hasNext() );
		
		// do the rm part
		return Math.sqrt( sum / (double)count );
	}
	
	public static double getRmsd( Subunit a, Subunit b )
	{
		// just in case...
		assert( a.getAtomIndex().size() == b.getAtomIndex().size() );
		
		return getRmsd( a, b, a.getAtomIndex().iterator(), b.getAtomIndex().iterator() );
	}
	
	public static double getBackboneRmsd( Subunit a, Subunit b )
	{
		// just in case...
		assert( a.getBackboneAtomIndex().size() == b.getBackboneAtomIndex().size() );
		
		return getRmsd( a, b, a.getBackboneAtomIndex().iterator(), b.getBackboneAtomIndex().iterator() );
	}
	
	public static double getRangeRmsd( Subunit a, Subunit b, int startResidueId, int stopResidueId )
	{
		List<AtomAddressInternal> addressesA = a.getAtomsByResidueRange( startResidueId, stopResidueId );
		List<AtomAddressInternal> addressesB = b.getAtomsByResidueRange( startResidueId, stopResidueId );
		assert( addressesA.size() == addressesB.size() );
		return getRmsd( a, b, addressesA.iterator(), addressesB.iterator() );
	}
	
	public static double getBackboneRangeRmsd( Subunit a, Subunit b, int startResidueId, int stopResidueId )
	{
		List<AtomAddressInternal> addressesA = a.getBackboneAtomsByResidueRange( startResidueId, stopResidueId );
		List<AtomAddressInternal> addressesB = b.getBackboneAtomsByResidueRange( startResidueId, stopResidueId );
		assert( addressesA.size() == addressesB.size() );
		return getRmsd( a, b, addressesA.iterator(), addressesB.iterator() );
	}
	
	public static double getRmsd( Subunit a, Subunit b, Iterator<AtomAddressInternal> iterAtomA, Iterator<AtomAddressInternal> iterAtomB )
	{
		double sum = 0.0;
		int count = 0;
		
		// for each atom...
		while( iterAtomA.hasNext() && iterAtomB.hasNext() )
		{
			// get atoms
			Vector3 posA = new Vector3( a.getAtom( iterAtomA.next() ).getPosition() );
			Vector3 posB = new Vector3( b.getAtom( iterAtomB.next() ).getPosition() );
			
			// add the sd
			sum += posB.getSquaredDistance( posA );
			count++;
		}
		
		// just in case...
		assert( iterAtomA.hasNext() == iterAtomB.hasNext() );
		
		// do the rm part
		return Math.sqrt( sum / (double)count );
	}
}
