/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.atomType;

import java.io.IOException;
import java.io.InputStream;
import java.util.TreeMap;

import edu.duke.donaldLab.share.protein.Atom;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Residue;
import edu.duke.donaldLab.share.protein.ResidueType;
import edu.duke.donaldLab.share.protein.Subunit;

public class AtomTypeMap
{
	/**************************
	 *   Definitions
	 **************************/
	
	private static final String BackbonePath = "/resources/backbone.atomTypes";
	private static final String CanonicalPath = "/resources/canonical.atomTypes";
	
	
	/**************************
	 *   Data Members
	 **************************/
	
	private TreeMap<String,TreeMap<String,AtomType>> m_canonicalMap;
	private TreeMap<String,TreeMap<String,AtomType>> m_backboneMap;
	
	private static AtomTypeMap m_instance;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	static
	{
		m_instance = null;
	}
	
	protected AtomTypeMap( TreeMap<String,TreeMap<String,AtomType>> backboneMap, TreeMap<String,TreeMap<String,AtomType>> canonicalMap )
	{
		m_backboneMap = backboneMap;
		m_canonicalMap = canonicalMap;
	}
	
	
	/**************************
	 *   Static Methods
	 **************************/
	
	public static AtomTypeMap getInstance( )
	throws IOException
	{
		if( m_instance == null )
		{
			AtomTypeReader reader = new AtomTypeReader();
			
			// read the backbone atom types
			InputStream in = AtomTypeMap.class.getResourceAsStream( BackbonePath );
			TreeMap<String,TreeMap<String,AtomType>> backboneMap = reader.read( in );
			
			// read the canonical atom types
			in = AtomTypeMap.class.getResourceAsStream( CanonicalPath );
			TreeMap<String,TreeMap<String,AtomType>> canonicalMap = reader.read( in );
				
			// return the map
			m_instance = new AtomTypeMap( backboneMap, canonicalMap );
		}
		
		return m_instance;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public AtomType getAtomType( Subunit subunit, Residue residue, Atom atom )
	{
		// gather the necessary info
		String aminoAcidAbbreviation = residue.getAminoAcid().getAbbreviation();
		String atomName = atom.getName();
		String residueTypeName = ResidueType.valueOf( subunit, residue ).name();
		
		// lookup the atom type
		AtomType atomType = null;
		atomType = m_canonicalMap.get( aminoAcidAbbreviation ).get( atomName );
		if( atomType == null )
		{
			atomType = m_backboneMap.get( residueTypeName ).get( atomName );
		}
		return atomType;
	}
	
	public AtomType getAtomType( Protein protein, AtomAddressInternal address )
	{
		return getAtomType(
			protein.getSubunit( address.getSubunitId() ),
			protein.getResidue( address.getSubunitId(), address.getResidueId() ),
			protein.getAtom( address )
		);
	}
}
