/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot;

import java.io.File;

import edu.duke.donaldLab.jdshot.context.DistanceRestraintsContext;
import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.search.SearchClient;
import edu.duke.donaldLab.jdshot.search.SearchServer;
import edu.duke.donaldLab.share.io.ArgumentType;
import edu.duke.donaldLab.share.io.ArgumentsProcessor;
import edu.duke.donaldLab.share.io.Logging;
import edu.duke.donaldLab.share.mpi.Client;
import edu.duke.donaldLab.share.mpi.ClientServerCreator;
import edu.duke.donaldLab.share.mpi.MPIEndpoint;
import edu.duke.donaldLab.share.mpi.Server;
import edu.duke.donaldLab.share.nmr.DistanceRestraintFilterer;

public class SearchMpiMain
{
	public static void main( String[] args )
	throws Exception
	{
		Logging.Mpi.init();
		
		// process the arguments
		ArgumentsProcessor argproc = new ArgumentsProcessor();
		argproc.add( "symmetry", ArgumentType.String, "symmetry type (e.g., Cn, Dn)" );
		argproc.add( "monomerPath", ArgumentType.InFile, "path to the monomer protein" );
		argproc.add( "distanceRestraintsPath", ArgumentType.InFile, "path to the distance restraints" );
		argproc.add( "numSubunits", ArgumentType.String, "number of subunits in the oligomer" );
		argproc.add( "outCellsPath", ArgumentType.OutFile, "path to output cells" );
		argproc.add( "logPath", ArgumentType.OutFile, "path for log" );
		argproc.add( "inCellsPath", ArgumentType.InFile, "c", null, "path to input cells" );
		argproc.add( "numAllowedInconsistencies", ArgumentType.Integer, "i", "0", "number of allowed inconsistencies per cell" );
		argproc.add( "batchSize", ArgumentType.Integer, "b", "100", "number of cells to process in one batch" );
		argproc.process( args );
		
		// get the arguments
		Symmetry symmetry = Symmetry.valueOf( argproc.getString( "symmetry" ) );
		File monomerFile = argproc.getFile( "monomerPath" );
		File distanceRestraintsFile = argproc.getFile( "distanceRestraintsPath" );
		final int numSubunits = argproc.getInteger( "numSubunits" );
		final File outCellsFile = argproc.getFile( "outCellsPath" );
		final File inCellsFile = argproc.getFile( "inCellsPath" );
		File logFile = argproc.getFile( "logPath" );
		final int numAllowedInconsistencies = argproc.getInteger( "numAllowedInconsistencies" );
		final int batchSize = argproc.getInteger( "batchSize" );
		
		// init our contexts
		final SymmetryContext symmetryContext = new SymmetryContext( symmetry, monomerFile, numSubunits );
		symmetryContext.normalizePosition();
		final DistanceRestraintsContext distanceRestraintsContext = new DistanceRestraintsContext( symmetryContext, distanceRestraintsFile );
		
		// normalize the distance restraints so the main subunit is on the left
		DistanceRestraintFilterer.sortSides( distanceRestraintsContext.getInternalRestraints() );		
		
		MPIEndpoint.initClientServer( args, logFile,
			new ClientServerCreator()
			{
				@Override
				public Client createClient( ) throws Exception
				{
					SearchClient client = new SearchClient( symmetryContext, distanceRestraintsContext );
					if( batchSize > 0 )
					{
						client.setBatchSize( batchSize );
					}
					return client;
				}

				@Override
				public Server createServer( ) throws Exception
				{
					return new SearchServer( symmetryContext, distanceRestraintsContext, outCellsFile, inCellsFile, numAllowedInconsistencies );
				}
			}
		);
	}
}
