/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.analyze;

import edu.duke.donaldLab.jdshot.grid.GridCell;
import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.tools.ProteinGeometry;

public class AnalysisProcessor implements Runnable
{
	/**************************
	 *   Data Members
	 **************************/
	
	private SCSAnalyzer m_scsAnalyzer;
	private AnalysisProcessorSynchronizer m_synch;
	private Protein m_referenceBackbone;
	private double m_totalVolume;
	private double m_minReferenceRmsd;
	private double m_maxReferenceRmsd;
	private double m_sumReferenceRmsd;
	private double m_minNoeRmsd;
	private double m_maxNoeRmsd;
	private double m_sumNoeRmsd;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public AnalysisProcessor( SCSAnalyzer scsAnalyzer, AnalysisProcessorSynchronizer synch, Protein referenceBackbone )
	{
		m_scsAnalyzer = scsAnalyzer;
		m_synch = synch;
		m_referenceBackbone = null;
		if( referenceBackbone != null )
		{
			m_referenceBackbone = new Protein( referenceBackbone );
			ProteinGeometry.center( m_referenceBackbone );
		}
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public double getTotalVolume( )
	{
		return m_totalVolume;
	}
	
	public double getMinReferenceRmsd( )
	{
		return m_minReferenceRmsd;
	}
	
	public double getMaxReferenceRmsd( )
	{
		return m_maxReferenceRmsd;
	}
	
	public double getSumReferenceRmsd( )
	{
		return m_sumReferenceRmsd;
	}
	
	public double getMinNoeRmsd( )
	{
		return m_minNoeRmsd;
	}
	
	public double getMaxNoeRmsd( )
	{
		return m_maxNoeRmsd;
	}
	
	public double getSumNoeRmsd( )
	{
		return m_sumNoeRmsd;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public void run( )
	{
		// TEMP
		System.out.println( "Thread starting..." );
		
		// reset stats vars
		m_totalVolume = 0.0;
		m_minReferenceRmsd = Double.POSITIVE_INFINITY;
		m_maxReferenceRmsd = Double.NEGATIVE_INFINITY;
		m_sumReferenceRmsd = 0.0;
		m_minNoeRmsd = Double.POSITIVE_INFINITY;
		m_maxNoeRmsd = Double.NEGATIVE_INFINITY;
		m_sumNoeRmsd = 0.0;
		
		int numProcessed = 0;
		boolean keepGoing = true;
		while( keepGoing )
		{
			// calculate stuff
			double volume = 0.0;
			double referenceRmsd = 0.0;
			double noeRmsd = 0.0;
			
			// do the calculations on a per-mode basis
			GridCell cell = null;
			GridPoint point = null;
			switch( m_synch.getMode() )
			{
				case Cells:
					// get the next cell if there is one
					cell = m_synch.getNextCell();
					if( cell == null )
					{
						keepGoing = false;
					}
					else
					{
						// get the point from the cell
						point = m_synch.getAdapter().getPoint( cell );
						
						volume = cell.getVolume();
						if( m_referenceBackbone != null )
						{
							referenceRmsd = m_scsAnalyzer.getReferenceRmsd( m_referenceBackbone, point );
						}
						noeRmsd = m_scsAnalyzer.getNoeRmsd( point );
					}
				break;
				
				case Points:
					// get the next point if there is one
					point = m_synch.getNextPoint();
					if( point == null )
					{
						keepGoing = false;
					}
					else
					{
						if( m_referenceBackbone != null )
						{
							referenceRmsd = m_scsAnalyzer.getReferenceRmsd( m_referenceBackbone, point );
						}
						noeRmsd = m_scsAnalyzer.getNoeRmsd( point );
					}
				break;
			}
			
			if( !keepGoing )
			{
				break;
			}
			
			// update stats
			m_totalVolume += volume;
			m_sumReferenceRmsd += referenceRmsd;
			if( referenceRmsd < m_minReferenceRmsd )
			{
				m_minReferenceRmsd = referenceRmsd;
			}
			if( referenceRmsd > m_maxReferenceRmsd )
			{
				m_maxReferenceRmsd = referenceRmsd;
			}
			m_sumNoeRmsd += noeRmsd;
			if( noeRmsd < m_minNoeRmsd )
			{
				m_minNoeRmsd = noeRmsd;
			}
			if( noeRmsd > m_maxNoeRmsd )
			{
				m_maxNoeRmsd = noeRmsd;
			}
			
			numProcessed++;
			m_synch.incrementProgress();
		}
		
		// TEMP
		System.out.println( "Thread finished! processed " + numProcessed + " " + m_synch.getMode() );
	}
}
