/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.cluster;

import java.util.ArrayList;

import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.StructureGenerator;
import edu.duke.donaldLab.share.analysis.RmsdCalculator;
import edu.duke.donaldLab.share.analysis.StructureAligner;
import edu.duke.donaldLab.share.math.IndexPair;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.tools.ProteinGeometry;

public class DistanceProcessor implements Runnable
{
	/**************************
	 *   Definitions
	 **************************/
	
	private static final int BatchSize = 600;
	
	
	/**************************
	 *   Data Memebers
	 **************************/
	
	private DistanceSynchronizer m_synch;
	private StructureGenerator m_structureGenerator;
	private int m_numSubunits;
	private ArrayList<GridPoint> m_points;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DistanceProcessor( DistanceSynchronizer synch, StructureGenerator structureGenerator, int numSubunits, ArrayList<GridPoint> points )
	{
		m_synch = synch;
		m_structureGenerator = structureGenerator;
		m_numSubunits = numSubunits;
		m_points = points;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public void run( )
	{
		// ALERT
		System.out.println( "Starting thread..." );
		
		IndexPair[] pairs = new IndexPair[BatchSize];
		double[] rmsds = new double[BatchSize];
		int actualBatchSize = 0;
		int i = 0;
		int numProcessed = 0;
		
		Protein leftStructure = null;
		int lastLeft = -1;
		
		while( true )
		{
			// get some pairs from the synch
			actualBatchSize = m_synch.getBatch( pairs, BatchSize );
			
			// are we done?
			if( actualBatchSize == 0 )
			{
				break;
			}
			
			// for each pair
			for( i=0; i<actualBatchSize; i++ )
			{
				// generate the left structure only if needed
				if( pairs[i].left != lastLeft )
				{
					leftStructure = m_structureGenerator.getStructure( m_points.get( pairs[i].left ), m_numSubunits );
					ProteinGeometry.center( leftStructure );
					lastLeft = pairs[i].left;
				}
				
				// generate the right structure
				Protein rightStructure = m_structureGenerator.getStructure( m_points.get( pairs[i].right ), m_numSubunits );
				
				// align them
				StructureAligner.alignOptimally( leftStructure, rightStructure );
				//structureAligner.align( leftStructure, rightStructure );
				
				// compute the RMSD
				rmsds[i] = RmsdCalculator.getRmsd( leftStructure, rightStructure );
				
				numProcessed++;
			}
			
			// report the batch
			m_synch.reportBatch( pairs, rmsds, actualBatchSize );
		}
		
		// ALERT
		System.out.println( "Thread processed " + numProcessed + " pairs" );
	}
}
