/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.context;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import edu.duke.donaldLab.share.io.Logging;
import edu.duke.donaldLab.share.mapping.NameMapper;
import edu.duke.donaldLab.share.mapping.NameScheme;
import edu.duke.donaldLab.share.nmr.DistanceRestraint;
import edu.duke.donaldLab.share.nmr.DistanceRestraintFilterer;
import edu.duke.donaldLab.share.nmr.DistanceRestraintMapper;
import edu.duke.donaldLab.share.nmr.DistanceRestraintReader;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.AtomAddressReadable;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.pseudoatoms.PseudoatomBuilder;

public class DistanceRestraintsContext
{
	/**************************
	 *   Data Members
	 **************************/
	
	private static final Logger m_log = Logging.getLog( DistanceRestraintsContext.class );
	
	private List<DistanceRestraint<AtomAddressReadable>> m_readableRestraints;
	private List<DistanceRestraint<AtomAddressInternal>> m_internalRestraints;
	private DistanceRestraint<AtomAddressInternal> m_maxRestraint;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DistanceRestraintsContext( SymmetryContext symmetryContext, String noesPath )
	throws IOException
	{
		this( symmetryContext, new File( noesPath ) );
	}
	
	public DistanceRestraintsContext( SymmetryContext symmetryContext, File noesFile )
	throws IOException
	{
		// copy the monomer so we can pretend we have an oligomer
		this( symmetryContext.getClonedMonomer(), noesFile );
	}
	
	public DistanceRestraintsContext( Protein protein, String noesPath )
	throws IOException
	{
		this( protein, new File( noesPath ) );
	}
	
	public DistanceRestraintsContext( Protein protein, File noesFile )
	throws IOException
	{
		// read in and name map the restraints
		ArrayList<DistanceRestraint<AtomAddressReadable>> readableRestraints = new DistanceRestraintReader().read( noesFile );
		assert( !readableRestraints.isEmpty() );
		m_log.info( String.format( "Num distance restraints read: %d", readableRestraints.size() ) );
		NameMapper.ensureDistanceRestraints( protein, readableRestraints, NameScheme.New );
		
		// apply pseudoatoms
		int numBuilt = PseudoatomBuilder.getInstance().buildDistanceRestraints( protein, readableRestraints );
		m_log.info( String.format( "%d distance restraints interpreted with pseudoatoms", numBuilt ) );
		
		// map the readable restraints to internal restraints
		ArrayList<DistanceRestraint<AtomAddressInternal>> internalRestraints = DistanceRestraintMapper.mapReadableToInternal( readableRestraints, protein );
		
		/*
			run the standard filtering suite
			ie, Filter the restraints so they're all:
				intersubunit
				involve subunit A
				contain different information
				unique
		*/
		m_log.info( String.format( "Filtering %d restraints...", internalRestraints.size() ) );
		internalRestraints = DistanceRestraintFilterer.pickIntersubunit( internalRestraints );
		m_log.info( String.format( "\tIntersubunit Filter: %d", internalRestraints.size() ) );
		internalRestraints = DistanceRestraintFilterer.pickSubunitEitherSide( internalRestraints, 0 );
		m_log.info( String.format( "\tFirst Subunit Filter: %d", internalRestraints.size() ) );
		internalRestraints = DistanceRestraintFilterer.pickOneFromSymmetricGroup( internalRestraints );
		m_log.info( String.format( "\tUnique Filter: %d", internalRestraints.size() ) );
		internalRestraints = DistanceRestraintFilterer.pickUnique( internalRestraints );
		m_log.info( String.format( "\tSymmetry Filter: %d", internalRestraints.size() ) );
		assert( !internalRestraints.isEmpty() );
		
		setInternalRestraints( internalRestraints, protein );
		m_log.info( String.format( "Num internal distance restraints used: %d", m_internalRestraints.size() ) );
		m_log.info( String.format( "Num readable distance restraints used: %d", m_readableRestraints.size() ) );
		
		// just in case...
		assert( m_readableRestraints.size() == m_internalRestraints.size() );
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public List<DistanceRestraint<AtomAddressReadable>> getReadableRestraints( )
	{
		return m_readableRestraints;
	}
	
	public List<DistanceRestraint<AtomAddressInternal>> getInternalRestraints( )
	{
		return m_internalRestraints;
	}
	
	public DistanceRestraint<AtomAddressInternal> getMaxRestraint( )
	{
		return m_maxRestraint;
	}
	
	public void setInternalRestraints( List<DistanceRestraint<AtomAddressInternal>> restraints, Protein protein )
	{
		m_internalRestraints = restraints;
		m_readableRestraints = DistanceRestraintMapper.mapInternalToReadable( m_internalRestraints, protein, true );
		m_maxRestraint = calculateMaxDistanceRestraint();
	}
	
	
	/**************************
	 *   Functions
	 **************************/
	
	private DistanceRestraint<AtomAddressInternal> calculateMaxDistanceRestraint( )
	{
		DistanceRestraint<AtomAddressInternal> maxRestraint = null;
		
		for( DistanceRestraint<AtomAddressInternal> restraint : m_internalRestraints )
		{
			if( maxRestraint == null || restraint.getMaxDistance() > maxRestraint.getMaxDistance() )
			{
				maxRestraint = restraint;
			}
		}
		
		// just in case
		assert( maxRestraint != null );
		
		return maxRestraint;
	}
}
