/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.context;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;

import edu.duke.donaldLab.share.io.Logging;
import edu.duke.donaldLab.share.mapping.NameMapper;
import edu.duke.donaldLab.share.mapping.NameScheme;
import edu.duke.donaldLab.share.nmr.Rdc;
import edu.duke.donaldLab.share.nmr.RdcFilterer;
import edu.duke.donaldLab.share.nmr.RdcMapper;
import edu.duke.donaldLab.share.nmr.RdcReader;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.AtomAddressReadable;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.tools.MonomerCloner;

public class RdcsContext
{
	/**************************
	 *   Data Members
	 **************************/
	
	private static final Logger m_log = Logging.getLog( RdcsContext.class );
	
	private List<Rdc<AtomAddressReadable>> m_readableRdcs;
	private List<Rdc<AtomAddressInternal>> m_internalRdcs;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public RdcsContext( SymmetryContext symmetryContext, String rdcsPath )
	throws IOException
	{
		this( symmetryContext, new File( rdcsPath ) );
	}
	
	public RdcsContext( SymmetryContext symmetryContext, File rdcsFile )
	throws IOException
	{
		this(
			MonomerCloner.clone(
				symmetryContext.getPseudoMonomer(),
				symmetryContext.getNumSubunits()
			),
			rdcsFile
		);
	}
	
	public RdcsContext( Protein protein, String rdcsPath )
	throws IOException
	{
		this( protein, new File( rdcsPath ) );
	}
	
	public RdcsContext( Protein protein, File rdcsFile )
	throws IOException
	{
		// read in RDCs
		List<Rdc<AtomAddressReadable>> readableRdcs = new RdcReader().read( rdcsFile );
		assert( !readableRdcs.isEmpty() );
		
		// make sure the RDCs have subunit names
		for( Rdc<AtomAddressReadable> rdc : readableRdcs )
		{
			if( !rdc.getFrom().hasSubunitName() )
			{
				rdc.getFrom().setSubunitName( 'A' );
			}
			if( !rdc.getTo().hasSubunitName() )
			{
				rdc.getTo().setSubunitName( 'A' );
			}
		}
		
		// map the RDCs
		NameMapper.ensureRdcs( protein, readableRdcs, NameScheme.New );
		m_internalRdcs = RdcMapper.mapReadableToInternal( protein, readableRdcs );
		
		// dump some stats
		m_log.info( String.format( "Read %d RDCs from:\n\t%s", readableRdcs.size(), rdcsFile.getAbsolutePath() ) );
		
		/*
			run the standard filtering suite
			ie, Filter the RDCs so they're all:
				involving subunit A
				unique
		*/
		m_log.info( String.format( "\tFiltering %d RDCs...", m_internalRdcs.size() ) );
		m_internalRdcs = RdcFilterer.pickUnique( m_internalRdcs );
		m_log.info( String.format( "\t\tUnique Filter: %d", m_internalRdcs.size() ) );
		m_internalRdcs = RdcFilterer.pickFromSubunit( m_internalRdcs, 0 );
		m_log.info( String.format( "\t\tFrom First Subunit Filter: %d", m_internalRdcs.size() ) );
		
		// map back to readable RDCs
		m_log.info( String.format( "\tNum Internal RDCs used: %d", m_internalRdcs.size() ) );
		m_readableRdcs = RdcMapper.mapInternalToReadable( protein, m_internalRdcs );
		m_log.info( String.format( "\tNum Readable RDCs used: %d", m_readableRdcs.size() ) );
		
		// just in case...
		assert( m_internalRdcs.size() == m_readableRdcs.size() );
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public List<Rdc<AtomAddressReadable>> getReadableRdcs( )
	{
		return m_readableRdcs;
	}
	
	public List<Rdc<AtomAddressInternal>> getInternalRdcs( )
	{
		return m_internalRdcs;
	}

}
