/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.test.disco;

import java.io.File;
import java.util.List;

import edu.duke.donaldLab.jdshot.context.RdcsContext;
import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.disco.OrientationCalculator;
import edu.duke.donaldLab.jdshot.disco.ReferenceAnalyzer;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.test.ExtendedTestCase;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.mapping.NameMapper;
import edu.duke.donaldLab.share.mapping.NameScheme;
import edu.duke.donaldLab.share.nmr.AlignmentTensor;
import edu.duke.donaldLab.share.nmr.Rdc;
import edu.duke.donaldLab.share.nmr.RdcFilterer;
import edu.duke.donaldLab.share.nmr.RdcMapper;
import edu.duke.donaldLab.share.nmr.RdcReader;
import edu.duke.donaldLab.share.pdb.ProteinReader;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.AtomAddressReadable;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.tools.MonomerCloner;

public class TestOrientationCalculator extends ExtendedTestCase
{
	private static final double Epsilon = 1e-3;
	
	public void testDagk( )
	throws Exception
	{
		// read the subunit structure and the RDCs
		Protein monomer = new ProteinReader().read( "src/resources/test/2KDC.monomer.protein" );
		List<Rdc<AtomAddressReadable>> readableRdcs = new RdcReader().read( "src/resources/test/2KDC.experimental.rdc" );
		Protein oligomer = MonomerCloner.clone( monomer.getSubunit( 0 ), 3 );
		NameMapper.ensureRdcs( oligomer, readableRdcs, NameScheme.New );
		List<Rdc<AtomAddressInternal>> rdcs = RdcMapper.mapReadableToInternal( oligomer, readableRdcs );
		rdcs = RdcFilterer.pickFromSubunit( rdcs, 0 );
		
		// compute the alignment tensor
		AlignmentTensor tensor = AlignmentTensor.compute( monomer, rdcs );
		
		// compute the orientations
		List<Vector3> orientations = OrientationCalculator.computeOrientations( 3, tensor );
		assertEquals( 1, orientations.size() );
		assertEquals( new Vector3( -0.42511, -0.41622, 0.80377 ), orientations.get( 0 ), Epsilon );
	}
	
	public void testGb1( )
	throws Exception
	{
		// read the subunit structure and the RDCs
		Protein monomer = new ProteinReader().read( "src/resources/test/1Q10.monomer.protein" );
		List<Rdc<AtomAddressReadable>> readableRdcs = new RdcReader().read( "src/resources/test/1Q10.experimental.rdc" );
		Protein oligomer = MonomerCloner.clone( monomer.getSubunit( 0 ), 2 );
		NameMapper.ensureRdcs( oligomer, readableRdcs, NameScheme.New );
		List<Rdc<AtomAddressInternal>> rdcs = RdcMapper.mapReadableToInternal( oligomer, readableRdcs );
		rdcs = RdcFilterer.pickFromSubunit( rdcs, 0 );
		
		// compute the alignment tensor
		AlignmentTensor tensor = AlignmentTensor.compute( monomer, rdcs );
		
		// compute the orientations
		List<Vector3> orientations = OrientationCalculator.computeOrientations( 2, tensor );
		assertEquals( 3, orientations.size() );
		assertEquals( new Vector3( 1.0, -0.0015901, 0.0024668 ), orientations.get( 0 ), Epsilon );
	}
	
	public void testDagkNormalize( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/2KDC.monomer.protein" ), 3 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/2KDC.experimental.rdc" );
		
		// previous steps
		AlignmentTensor tensor = AlignmentTensor.compute( symmetryContext.getMonomer(), rdcsContext.getInternalRdcs() );
		Vector3 orientation = OrientationCalculator.computeOrientations( 3, tensor ).get( 0 );
		
		// normalize the protein
		OrientationCalculator.normalizeProtein( symmetryContext.getMonomer(), orientation );
		
		// check atom positions
		assertEquals( new Vector3( -1.011023, 5.922894, -2.276559 ), getPos( symmetryContext, 33, "CB" ), Epsilon );
		assertEquals( new Vector3( -1.124400, -17.794971, -6.170632 ), getPos( symmetryContext, 96, "CB" ), Epsilon );
		
		assertEquals( new Vector3( -10.174453, -1.224457, -6.125492 ), getPos( symmetryContext, 70, "CB" ), Epsilon );
		assertEquals( new Vector3( 1.537237, -13.517542, -2.024960 ), getPos( symmetryContext, 99, "CB" ), Epsilon );
		
		assertEquals( new Vector3( -8.665511, -3.985224, 19.544434 ), getPos( symmetryContext, 52, "CB" ), Epsilon );
		assertEquals( new Vector3( -5.662176, -18.544356, 21.011362 ), getPos( symmetryContext, 117, "CB" ), Epsilon );
	}
	
	public void testGb1Normalize( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/1Q10.monomer.protein" ), 2 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/1Q10.experimental.rdc" );
		
		// previous steps
		AlignmentTensor tensor = AlignmentTensor.compute( symmetryContext.getMonomer(), rdcsContext.getInternalRdcs() );
		Vector3 orientation = OrientationCalculator.computeOrientations( 2, tensor ).get( 0 );
		
		// normalize the protein
		OrientationCalculator.normalizeProtein( symmetryContext.getMonomer(), orientation );
		
		// check atom positions
		// HACKHACK: 0.1 is a large value for epsilon. Why are the GB1 numbers less precise?
		final double Epsilon = 1e-1;
		assertEquals( new Vector3( 0.600, -15.963, 6.109 ), getPos( symmetryContext, 1, "CA" ), Epsilon );
		assertEquals( new Vector3( -1.904, -2.817, -4.818 ), getPos( symmetryContext, 33, "CA" ), Epsilon );
		assertEquals( new Vector3( 0.051, -17.822, 1.072 ), getPos( symmetryContext, 21, "CA" ), Epsilon );
		assertEquals( new Vector3( -7.928, -0.866, -4.143 ), getPos( symmetryContext, 56, "CA" ), Epsilon );
	}
	
	public void testDagkNormalizeAxis( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/2KDC.monomer.protein" ), 3 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/2KDC.experimental.rdc" );
		Protein oligomer = new ProteinReader().read( new File( "src/resources/test/2KDC.oligomer.protein" ) );
		
		// previous steps
		AlignmentTensor tensor = AlignmentTensor.compute( symmetryContext.getMonomer(), rdcsContext.getInternalRdcs() );
		Vector3 orientation = OrientationCalculator.computeOrientations( 3, tensor ).get( 0 );
		Vector3 referencePosition = ReferenceAnalyzer.getSymmetryAxisPosition( oligomer );
		
		// normalize
		OrientationCalculator.normalizeProteinAndAxis(
			symmetryContext.getMonomer(),
			referencePosition,
			orientation
		);
		
		// check the x,y coords of the reference position
		assertEquals( -10.173282, referencePosition.x, 1e-2 );
		assertEquals( -8.375934, referencePosition.y, 1e-2 );
	}
	
	public void testGb1NormalizeAxis( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/1Q10.monomer.protein" ), 2 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/1Q10.experimental.rdc" );
		Protein oligomer = new ProteinReader().read( new File( "src/resources/test/1Q10.oligomer.protein" ) );
		
		// previous steps
		AlignmentTensor tensor = AlignmentTensor.compute( symmetryContext.getMonomer(), rdcsContext.getInternalRdcs() );
		Vector3 orientation = OrientationCalculator.computeOrientations( 2, tensor ).get( 0 );
		Vector3 referencePosition = ReferenceAnalyzer.getSymmetryAxisPosition( oligomer );
		
		// normalize
		OrientationCalculator.normalizeProteinAndAxis(
			symmetryContext.getMonomer(),
			referencePosition,
			orientation
		);
		
		// check the x,y coords of the reference position
		assertEquals( 1.739201, referencePosition.x, 1e-2 );
		assertEquals( 2.221133, referencePosition.y, 1e-2 );
	}
	
	private Vector3 getPos( SymmetryContext symmetryContext, int residueNumber, String atomName )
	{
		AtomAddressReadable address = new AtomAddressReadable( 'A', residueNumber, atomName );
		return symmetryContext.getMonomer().getAtom( address ).getPosition();
	}
}
