/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.analysis;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Protein;

public class VarianceCalculator
{
	/**************************
	 *   Definitions
	 **************************/
	
	public static enum Metric
	{
		Variance
		{
			@Override
			public double evaluate( Collection<Vector3> points )
			{
				// compute the geometric mean
				Vector3 mean = Vector3.getOrigin();
				for( Vector3 pos : points )
				{
					mean.add( pos );
				}
				mean.scale( 1.0 / points.size() );
				
				// compute the average squared deviation from the mean
				double sum = 0.0;
				for( Vector3 pos : points )
				{
					sum += pos.getSquaredDistance( mean );
				}
				return sum / (double) points.size();
			}
		},
		Rmsd
		{
			@Override
			public double evaluate( Collection<Vector3> points )
			{
				return Math.sqrt( Variance.evaluate( points ) );
			}
		};
		
		public abstract double evaluate( Collection<Vector3> points );
	}
	
	
	/**************************
	 *   Static Methods
	 **************************/
	
	public static double getAverageRmsd( Collection<Protein> proteins )
	{
		return getAverageMetric( proteins, Metric.Rmsd );
	}
	
	public static double getAverageVariance( Collection<Protein> proteins )
	{
		return getAverageMetric( proteins, Metric.Variance );
	}
	
	
	/**************************
	 *   Static Functions
	 **************************/
	
	public static double getAverageMetric( Collection<Protein> proteins, Metric metric )
	{
		double sum = 0.0;
		int count = 0;
		
		// get an atom iterator for each protein
		ArrayList<Iterator<AtomAddressInternal>> atomIters = new ArrayList<Iterator<AtomAddressInternal>>();
		for( Protein protein : proteins )
		{
			atomIters.add( protein.getAtomIndex() );
		}
		
		// for each atom...
		while( atomIters.get( 0 ).hasNext() )
		{
			// collect the atom positions into a list
			LinkedList<Vector3> atomPositions = new LinkedList<Vector3>();
			
			// for each protein...
			int proteinIndex = 0;
			for( Protein protein : proteins )
			{
				// get the atom position for this protein
				atomPositions.add( protein.getAtom( atomIters.get( proteinIndex ).next() ).getPosition() );
				
				proteinIndex++;
			}
			
			sum += metric.evaluate( atomPositions );
			count++;
		}
		
		// DEBUG: double-check all iters are empty
		for( Iterator<AtomAddressInternal> iter : atomIters )
		{
			assert( !iter.hasNext() );
		}
		
		// return the average
		return sum / (double)count;
	}
}
