/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.test.nmr;

import java.util.List;

import edu.duke.donaldLab.share.mapping.NameMapper;
import edu.duke.donaldLab.share.mapping.NameScheme;
import edu.duke.donaldLab.share.nmr.AlignmentTensor;
import edu.duke.donaldLab.share.nmr.Rdc;
import edu.duke.donaldLab.share.nmr.RdcFilterer;
import edu.duke.donaldLab.share.nmr.RdcMapper;
import edu.duke.donaldLab.share.nmr.RdcReader;
import edu.duke.donaldLab.share.pdb.ProteinReader;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.AtomAddressReadable;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.tools.MonomerCloner;
import edu.duke.donaldLab.share.test.ExtendedTestCase;

public class TestAlignmentTensor extends ExtendedTestCase
{
	private static final double Epsilon = 1e-3;
	
	public void testDAGK( )
	throws Exception
	{
		// read the subunit structure and the RDCs
		Protein monomer = new ProteinReader().read( "src/resources/test/2KDC.monomer.protein" );
		List<Rdc<AtomAddressReadable>> readableRdcs = new RdcReader().read( "src/resources/test/2KDC.experimental.rdc" );
		Protein oligomer = MonomerCloner.clone( monomer.getSubunit( 0 ), 3 );
		NameMapper.ensureRdcs( oligomer, readableRdcs, NameScheme.New );
		List<Rdc<AtomAddressInternal>> rdcs = RdcMapper.mapReadableToInternal( oligomer, readableRdcs );
		rdcs = RdcFilterer.pickFromSubunit( rdcs, 0 );
		
		// compute the alignment tensor
		AlignmentTensor tensor = AlignmentTensor.compute( monomer, rdcs );
		
		assertEquals( -6.653, tensor.getDxx(), Epsilon );
		assertEquals( -7.049, tensor.getDyy(), Epsilon );
		assertEquals( 13.702, tensor.getDzz(), Epsilon );
		assertEqualsReal( 0.0, tensor.getDxx() + tensor.getDyy() + tensor.getDzz() );
		assertEqualsReal( 0.028861, tensor.getAssymmetry() );
		assertEqualsReal( 0.019241, tensor.getRhombicity() );
		assertEquals( 0.276, tensor.getRmsd( monomer, rdcs ), Epsilon );
	}
	
	public void testGB1( )
	throws Exception
	{
		// read the subunit structure and the RDCs
		Protein monomer = new ProteinReader().read( "src/resources/test/1Q10.monomer.protein" );
		List<Rdc<AtomAddressReadable>> readableRdcs = new RdcReader().read( "src/resources/test/1Q10.experimental.rdc" );
		Protein oligomer = MonomerCloner.clone( monomer.getSubunit( 0 ), 2 );
		NameMapper.ensureRdcs( oligomer, readableRdcs, NameScheme.New );
		List<Rdc<AtomAddressInternal>> rdcs = RdcMapper.mapReadableToInternal( oligomer, readableRdcs );
		rdcs = RdcFilterer.pickFromSubunit( rdcs, 0 );
		
		// compute the alignment tensor
		AlignmentTensor tensor = AlignmentTensor.compute( monomer, rdcs );
		
		assertEquals( 4.848, tensor.getDxx(), Epsilon );
		assertEquals( 30.980, tensor.getDyy(), Epsilon );
		assertEquals( -35.829, tensor.getDzz(), Epsilon );
		assertEqualsReal( 0.0, tensor.getDxx() + tensor.getDyy() + tensor.getDzz() );
		assertEqualsReal( 0.729363, tensor.getAssymmetry() );
		assertEqualsReal( 0.486242, tensor.getRhombicity() );
		assertEquals( 0.565, tensor.getRmsd( monomer, rdcs ), Epsilon );
	}
}
