/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


#include <jni.h>

#include "cgal.h"
#include "shotCgal.h"
#include "global.h"
#include "types.h"
#include "Storage.h"


#define CLASS					Java_edu_duke_donaldLab_jdshot_disco_cgal_FaceIterator
#define FaceIterator_init		CLASSFN( CLASS, init )
#define FaceIterator_cleanup	CLASSFN( CLASS, cleanup )
#define FaceIterator_hasNext	CLASSFN( CLASS, hasNext )
#define FaceIterator_next		CLASSFN( CLASS, next )


static Storage<Arrangement::Face_iterator> g_faceIterators( CGALCLASS( "FaceIterator" ), true );
static jclass g_faceIteratorClass;
static jfieldID g_faceIteratorArrangementId;


void faceIteratorsCleanup( JNIEnv *jvm )
{
	g_faceIterators.cleanupAll( jvm );

	if( g_faceIteratorClass != NULL )
	{
		jvm->DeleteGlobalRef( g_faceIteratorClass );
		g_faceIteratorClass = NULL;
	}
}

Arrangement *getArrangementFromIterator( JNIEnv *jvm, jobject self )
{
	// lookup meta info if needed
	if( g_faceIteratorClass == NULL )
	{
		g_faceIteratorClass = (jclass)jvm->NewGlobalRef( jvm->FindClass( CGALCLASS( "FaceIterator" ) ) );
	}
	if( g_faceIteratorArrangementId == NULL )
	{
		g_faceIteratorArrangementId = jvm->GetFieldID( g_faceIteratorClass, "m_arrangement", "L" CGALCLASS( "Arrangement" ) ";" );
		checkException( jvm );
	}

	jobject arrangement = jvm->GetObjectField( self, g_faceIteratorArrangementId );
	checkException( jvm );
	return getArrangement( jvm, arrangement );
}

Arrangement::Face_iterator *getFaceIterator( JNIEnv *jvm, jobject self )
{
	return g_faceIterators.get( jvm, self );
}


JNIEXPORT void JNICALL FaceIterator_init( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement *pArrangement = getArrangementFromIterator( jvm, self );
		g_faceIterators.add( jvm, self, new Arrangement::Face_iterator( pArrangement->faces_begin() ) );
	}
	STOP_SIGNAL_HANDLING
}

JNIEXPORT void JNICALL FaceIterator_cleanup( JNIEnv *jvm, jclass c, jint id )
{
	START_SIGNAL_HANDLING
	{
		if( !g_faceIterators.cleanup( id ) )
		{
			throwException( jvm, "FaceIterator cleanup failure!" );
		}
	}
	STOP_SIGNAL_HANDLING
}

JNIEXPORT jboolean JNICALL FaceIterator_hasNext( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement *pArrangement = getArrangementFromIterator( jvm, self );
		Arrangement::Face_iterator iter = *getFaceIterator( jvm, self );
		return iter != pArrangement->faces_end();
	}
	STOP_SIGNAL_HANDLING
	return false;
}

JNIEXPORT jobject JNICALL FaceIterator_next( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement::Face_iterator *pIter = getFaceIterator( jvm, self );
		jobject face = newFace( jvm, &**pIter );
		(*pIter)++;

		return face;
	}
	STOP_SIGNAL_HANDLING
	return NULL;
}
