/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot;

import java.io.File;

import edu.duke.donaldLab.jdshot.analyze.AnalysisContext;
import edu.duke.donaldLab.jdshot.analyze.Analyzer;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.search.SearchContext;
import edu.duke.donaldLab.share.io.ArgumentType;
import edu.duke.donaldLab.share.io.ArgumentsProcessor;
import edu.duke.donaldLab.share.perf.Timer;
import edu.duke.donaldLab.share.protein.SubunitOrder;

public class AnalyzeMain
{
	public static void main( String[] args )
	throws Exception
	{
		// process the arguments
		ArgumentsProcessor argproc = new ArgumentsProcessor();
		argproc.add( "symmetry", ArgumentType.String, "symmetry type (e.g., Cn, Dn)" );
		argproc.add( "monomerPath", ArgumentType.InFile, "path to the monomer protein" );
		argproc.add( "oligomerPath", ArgumentType.InFile, "o", null, "path to the oligomer protein" );
		argproc.add( "noesPath", ArgumentType.InFile, "path to the noes" );
		argproc.add( "subunitOrder", ArgumentType.String, "ring order of the subunits e.g., ABCD" );
		argproc.add( "inPath", ArgumentType.InFile, "path to input points/cells" );
		argproc.add( "numThreads", ArgumentType.Integer, "t", "1", "number of threads to use" );
		argproc.process( args );
		
		// read the arguments for the monomer path and the noes path
		Symmetry symmetry = Symmetry.valueOf( argproc.getString( "symmetry" ) );
		File monomerFile = argproc.getFile( "monomerPath" );
		File oligomerFile = argproc.getFile( "oligomerPath" );
		File noesFile = argproc.getFile( "noesPath" );
		SubunitOrder subunitOrder = new SubunitOrder( argproc.getString( "subunitOrder" ) );
		File inFile = argproc.getFile( "inPath" );
		Integer numThreads = argproc.getInteger( "numThreads" );
		
		// create a search context
		System.out.println( "Loading search context..." );
		SearchContext searchContext = new SearchContext();
		searchContext.load( symmetry, monomerFile, noesFile, subunitOrder );
		
		// create an analysis context
		System.out.println( "Loading analysis context..." );
		AnalysisContext analysisContext = new AnalysisContext();
		analysisContext.load( oligomerFile, searchContext );
		System.out.println( "Loaded analysis context!" );
		
		// do stuff
		System.out.println( "Analysis starting..." );
		Timer timer = new Timer( "Analysis" );
		timer.start();
		Analyzer analyzer = new Analyzer();
		analyzer.init( searchContext, analysisContext, inFile );
		int numCellsAnalyzed = analyzer.analyze( numThreads );
		timer.stop();
		System.out.println( "Analysis complete!" );
		System.out.println( timer );
		System.out.println( "Cells analyzed: " + numCellsAnalyzed );
		System.out.println( "Cells analyzed per second: " + (float)numCellsAnalyzed/timer.getElapsedSeconds() );
	}
}
