/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot;

import java.io.File;

import edu.duke.donaldLab.jdshot.grid.PointIterator;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.search.SearchContext;
import edu.duke.donaldLab.jdshot.stericFilter.FilterResults;
import edu.duke.donaldLab.jdshot.stericFilter.StericFilterer;
import edu.duke.donaldLab.share.io.ArgumentType;
import edu.duke.donaldLab.share.io.ArgumentsProcessor;
import edu.duke.donaldLab.share.perf.Timer;
import edu.duke.donaldLab.share.protein.SubunitOrder;

public class StericFilterMain
{
	public static void main( String[] args )
	throws Exception
	{
		// process the arguments
		ArgumentsProcessor argproc = new ArgumentsProcessor();
		argproc.add( "symmetry", ArgumentType.String, "symmetry type (e.g., Cn, Dn)" );
		argproc.add( "monomerPath", ArgumentType.InFile, "path to the monomer protein" );
		argproc.add( "subunitOrder", ArgumentType.String, "ring order of the subunits e.g., ABCD" );
		argproc.add( "outPointsPath", ArgumentType.OutFile, "path to output points" );
		argproc.add( "inPath", ArgumentType.InFile, "path to input points/cells" );
		argproc.add( "epsilon", ArgumentType.Double, "allowed steric overlap in Angstroms" );
		argproc.add( "numThreads", ArgumentType.Integer, "t", "1", "number of threads to use" );
		argproc.process( args );
		
		// get the arguments
		Symmetry symmetry = Symmetry.valueOf( argproc.getString( "symmetry" ) );
		File monomerFile = argproc.getFile( "monomerPath" );
		SubunitOrder subunitOrder = new SubunitOrder( argproc.getString( "subunitOrder" ) );
		File outPointsFile = argproc.getFile( "outPointsPath" );
		Integer numThreads = argproc.getInteger( "numThreads" );
		File inFile = argproc.getFile( "inPath" );
		Double epsilon = argproc.getDouble( "epsilon" );
		
		// create a search context
		System.out.println( "Loading search context..." );
		SearchContext searchContext = new SearchContext();
		searchContext.load( symmetry, monomerFile, null, subunitOrder );
		System.out.println( "Loaded search context!" );
		
		// perform the filtering
		System.out.println( "Filtering started!" );
		PointIterator iterPoint = PointIterator.newFromPath( inFile, symmetry );
		System.out.println( "Processing " + iterPoint.getNumPoints() + " points..." );
		Timer timer = new Timer( "Search" );
		timer.start();
		StericFilterer filterer = new StericFilterer( searchContext );
		FilterResults results = filterer.filter( outPointsFile, iterPoint, epsilon, numThreads );
		timer.stop();
		System.out.println( "Filtering complete!" );
		System.out.println( timer );
		System.out.println( "Num processsed: " + results.numProcessed );
		System.out.println( "Num accepted: " + results.numAccepted );
		System.out.println( "Percent Accepted: " + String.format( "%.2f", 100.0 * results.numAccepted / results.numProcessed ) + "%" );
		System.out.println( "Num processed per second: " + (float)results.numProcessed/timer.getElapsedSeconds() );
	}
}
