/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.analyze;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import edu.duke.donaldLab.jdshot.search.SearchContext;
import edu.duke.donaldLab.share.bond.BondGraph;
import edu.duke.donaldLab.share.bond.BondGraphBuilder;
import edu.duke.donaldLab.share.mapping.NameMapper;
import edu.duke.donaldLab.share.mapping.NameScheme;
import edu.duke.donaldLab.share.nmr.SparseMonomerGenerator;
import edu.duke.donaldLab.share.pdb.ProteinReader;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Subunit;
import edu.duke.donaldLab.share.pseudoatoms.PseudoatomBuilder;

public class AnalysisContext
{
	/**************************
	 *   Definitions
	 **************************/
	
	private static final String CanonicalBondsPath = "/resources/canonical.bonds";
	private static final String BackboneBondsPath = "/resources/backbone.bonds";
	
	
	/**************************
	 *   Data Members
	 **************************/
	
	private Subunit m_sparseMonomer;
	private Protein m_oligomer;
	private ArrayList<BondGraph> m_bondGraphs;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public AnalysisContext(  )
	{
		m_sparseMonomer = null;
		m_oligomer = null;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public Subunit getSparseMonomer( )
	{
		return m_sparseMonomer;
	}
	
	public Protein getOligomer( )
	{
		return m_oligomer;
	}
	
	public BondGraph getBondGraph( Subunit subunit )
	{
		return m_bondGraphs.get( subunit.getId() );
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public void load( File oligomerFile, SearchContext searchContext )
	throws IOException
	{
		// read the protein if needed
		if( oligomerFile != null )
		{
			ProteinReader proteinReader = new ProteinReader();
			m_oligomer = proteinReader.read( oligomerFile );
			NameMapper.ensureProtein( m_oligomer, NameScheme.New );
			
			// add pseudoatoms if needed
			if( PseudoatomBuilder.distanceRestraintsHavePseudoatoms( searchContext.getNoes() ) )
			{
				PseudoatomBuilder.getInstance().build( m_oligomer );
			}
			
			// create the bond graphs
			m_bondGraphs = new BondGraphBuilder( CanonicalBondsPath, BackboneBondsPath ).build( m_oligomer ); 
		}
		
		// generate the sparse monomer
		m_sparseMonomer = SparseMonomerGenerator.getSparseMonomer(
			searchContext.getMonomer(),
			searchContext.getRestraints()
		);
	}
}
