/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid;

import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.grid.cn.CnCellChecker;
import edu.duke.donaldLab.jdshot.grid.cn.CnCellSplitter;
import edu.duke.donaldLab.jdshot.grid.cn.CnGridCell;
import edu.duke.donaldLab.jdshot.grid.cn.CnGridPoint;
import edu.duke.donaldLab.jdshot.grid.cn.CnNoeReplicator;
import edu.duke.donaldLab.jdshot.grid.cn.CnSearchSpace;
import edu.duke.donaldLab.jdshot.grid.cn.CnStructureGenerator;
import edu.duke.donaldLab.jdshot.grid.dn.DnCellChecker;
import edu.duke.donaldLab.jdshot.grid.dn.DnCellSplitter;
import edu.duke.donaldLab.jdshot.grid.dn.DnGridCell;
import edu.duke.donaldLab.jdshot.grid.dn.DnGridPoint;
import edu.duke.donaldLab.jdshot.grid.dn.DnNoeReplicator;
import edu.duke.donaldLab.jdshot.grid.dn.DnSearchSpace;
import edu.duke.donaldLab.jdshot.grid.dn.DnStructureGenerator;
import edu.duke.donaldLab.jdshot.search.CellChecker;
import edu.duke.donaldLab.jdshot.search.CellSplitter;
import edu.duke.donaldLab.share.protein.Subunit;

public enum Symmetry
{
	/**************************
	 *   Values
	 **************************/
	
	Cn( 4 )
	{
		@Override
		public GridPoint newGridPoint( )
		{
			return new CnGridPoint();
		}
		
		@Override
		public GridCell newGridCell( )
		{
			return new CnGridCell();
		}

		@Override
		public CellSplitter newCellSplitter( SearchSpace searchSpace )
		{
			return new CnCellSplitter( searchSpace );
		}

		@Override
		public CellChecker newCellChecker( SymmetryContext symmetryContext )
		{
			return new CnCellChecker( symmetryContext );
		}

		@Override
		public SearchSpace newSearchSpace( double monomerRadius, double maxDistance, int numSubunits )
		{
			return new CnSearchSpace( monomerRadius, maxDistance, numSubunits );
		}

		@Override
		public StructureGenerator newStructureGenerator( Subunit monomer )
		{
			return new CnStructureGenerator( monomer );
		}

		@Override
		public DistanceRestraintReplicator newNoeReplicator( )
		{
			return new CnNoeReplicator();
		}
		
		@Override
		public String toString( int numSubunits )
		{
			return "C" + Integer.toString( numSubunits );
		}
	},
	Dn( 6 )
	{
		@Override
		public GridPoint newGridPoint( )
		{
			return new DnGridPoint();
		}
		
		@Override
		public GridCell newGridCell( )
		{
			return new DnGridCell();
		}

		@Override
		public CellSplitter newCellSplitter( SearchSpace searchSpace )
		{
			return new DnCellSplitter( searchSpace );
		}

		@Override
		public CellChecker newCellChecker( SymmetryContext symmetryContext )
		{
			return new DnCellChecker( symmetryContext );
		}

		@Override
		public SearchSpace newSearchSpace( double monomerRadius, double maxDistance, int numSubunits )
		{
			return new DnSearchSpace( monomerRadius, maxDistance, numSubunits );
		}

		@Override
		public StructureGenerator newStructureGenerator( Subunit monomer )
		{
			return new DnStructureGenerator( monomer );
		}

		@Override
		public DistanceRestraintReplicator newNoeReplicator( )
		{
			return new DnNoeReplicator();
		}
		
		@Override
		public String toString( int numSubunits )
		{
			return "D" + Integer.toString( numSubunits / 2 );
		}
	};
	

	/**************************
	 *   Data Members
	 **************************/
	
	private int m_numDimensions;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	private Symmetry( int numDimensions )
	{
		m_numDimensions = numDimensions;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public int getNumDimensions( )
	{
		return m_numDimensions;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public abstract GridPoint newGridPoint( );
	public abstract GridCell newGridCell( );
	public abstract CellSplitter newCellSplitter( SearchSpace searchSpace );
	public abstract CellChecker newCellChecker( SymmetryContext symmetryContext );
	public abstract SearchSpace newSearchSpace( double monomerRadius, double maxDistance, int numSubunits );
	public abstract StructureGenerator newStructureGenerator( Subunit monomer );
	public abstract DistanceRestraintReplicator newNoeReplicator( );
	public abstract String toString( int numSubunits );
}
