/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.cn;

import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.grid.Axes;
import edu.duke.donaldLab.jdshot.grid.GridCell;
import edu.duke.donaldLab.jdshot.grid.base.CellCheckerBase;
import edu.duke.donaldLab.jdshot.search.BoundContext;
import edu.duke.donaldLab.jdshot.search.CellChecker;
import edu.duke.donaldLab.share.geom.AxisAlignedBox;
import edu.duke.donaldLab.share.geom.Intersect;
import edu.duke.donaldLab.share.geom.Sphere;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.nmr.Assignment;
import edu.duke.donaldLab.share.protein.Atom;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;

public class CnCellChecker extends CellCheckerBase implements CellChecker
{
	/**************************
	 *   Data Members
	 **************************/
	
	private SymmetryContext m_symmetryContext;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public CnCellChecker( SymmetryContext symmetryContext )
	{
		m_symmetryContext = symmetryContext;
	}
	

	/**************************
	 *   Methods
	 **************************/
	
	@Override
	public boolean isConsistent( GridCell cell, double maxDistance, Assignment<AtomAddressInternal> assignment )
	{
		// check the cast...
		assert( cell instanceof CnGridCell );
		
		return isConsistent( (CnGridCell)cell, maxDistance, assignment );
	}
	
	public boolean isConsistent( CnGridCell cell, double maxDistance, Assignment<AtomAddressInternal> assignment )
	{
		// allocate memory
		BoundContext boundContext = new BoundContext( CnGridCell.NumTCorners );
		AxisAlignedBox bound = new AxisAlignedBox();
		Sphere sphere = new Sphere();
	
		// get the atom positions
		Atom leftAtom = m_symmetryContext.getMonomer().getAtom( assignment.getLeft() );
		Atom rightAtom = m_symmetryContext.getMonomer().getAtom( assignment.getRight() );
		
		// UNDONE: update this so bound context uses the rotation axis!
		
		// UNDONE: find a way to get the type of restraint without using subunit order
		/*
		// get the appropriate bound context
		boundContext.type = getSubunitType( assignment.getRight().getSubunitId(), m_symmetryContext.getSubunitOrder() );
		getBoundContext( boundContext, cell, rightAtom);
		
		// just in case...
		assert( assignment.getLeft().getSubunitId() == getSubunitId( SubunitType.Main, m_symmetryContext.getSubunitOrder() ) );
		assert( boundContext.type != SubunitType.Unknown ) : "Restraint is not intersubunit!";
		*/
		/*Himanshu: From Shobha's code:
		 * The restraint is considered satisfied if any pair is satisfied in both 
         * directions of the fixed subunit (the closed ring constraint)
		 */
		
		/*First perform the test if the restraint is satisfied if other subunit is anticlockwise*/
		// get the bound
		getBoundingBox( bound, boundContext );
		sphere.set( leftAtom.getPosition(), maxDistance );
		// perform intersection test
		boolean anticlock = Intersect.isOverlapping( sphere, bound );
		
		boundContext.alpha = 2*Math.PI - boundContext.alpha;
		
		// get the bound
		getBoundingBox( bound, boundContext );
		boolean clock = Intersect.isOverlapping( sphere, bound );
	
		return clock & anticlock;
	}
	
	
	/**************************
	 *   Functions
	 **************************/
	
	private void getBoundContext( BoundContext boundContext, CnGridCell cell, Atom atom)
	{
		// allocate memory
		Vector3 corner = new Vector3();
		Vector3 temp = new Vector3();
		
		// calculate q
		boundContext.q.set( atom.getPosition() );
		
		// calculate a
		double avgPhi = ( cell.max.phi + cell.min.phi ) / 2.0;
		double avgTheta = ( cell.max.theta + cell.min.theta ) / 2.0;
		Axes.getRotationVector( boundContext.a, avgPhi, avgTheta );
		
		boundContext.alpha = 2 * Math.PI / (double)m_symmetryContext.getNumSubunits();
		
		// calculate epsilon (need to get the max dist of all corners of A)
		double distsq = 0.0;
		double esq = 0.0;
		for( int i=0; i<CnGridCell.NumACorners; i++ )
		{
			cell.getACorner( corner, i );
			Axes.getRotationVector( temp, corner.x, corner.y );//x and y are phi and theta
			distsq = boundContext.a.getSquaredDistance( temp );
			
			if( distsq > esq )
				esq = distsq;
		}
		boundContext.epsilon = Math.sqrt( esq );
		
		// calculate {t_i}
		for( int i=0; i<CnGridCell.NumTCorners; i++ )
		{
			cell.getTCorner( boundContext.corners[i], i );
		}
	}
}
