/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.cn;

import java.io.Serializable;

import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.share.math.MultiVector;

public class CnGridPoint implements Serializable, GridPoint
{
	/**************************
	 *   Definitions
	 **************************/
	
	private static final long serialVersionUID = -4947149197572196363L;
	public static final Symmetry SymmetryType = Symmetry.Cn;
	
	
	/**************************
	 *   Fields
	 **************************/
	
	public double x;
	public double y;
	public double phi;
	public double theta;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public CnGridPoint( )
	{
		x = 0.0;
		y = 0.0;
		phi = 0.0;
		theta = 0.0;
	}
	
	public CnGridPoint( CnGridPoint other )
	{
		set( other );
	}
	
	public CnGridPoint( double a, double b, double c, double d )
	{
		x = a;
		y = b;
		phi = c;
		theta = d;
	}
	
	public CnGridPoint( MultiVector point )
	{
		this(
			point.get( 0 ),
			point.get( 1 ),
			point.get( 2 ),
			point.get( 3 )
		);
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public void set( CnGridPoint other )
	{
        x = other.x;
        y = other.y;
        phi = other.phi;
        theta = other.theta;
	}
	
	@Override
	public double get( int i ) 
	{
		switch( i )
		{
			case 0: return x;
			case 1: return y;
			case 2: return phi;
			case 3: return theta;
		}
		
		assert( false ) : "Invalid index: " + i;
		
		// just to make the compiler happy
		return Double.NaN;
	}
	
	@Override
	public void set( int i, double value )
	{
		switch( i )
		{
			case 0: x = value; return;
			case 1: y = value; return;
			case 2: phi = value; return;
			case 3: theta = value; return;
		}
		
		assert( false ) : "Invalid index: " + i;
	}
	
	@Override
	public double[] getArray( )
	{
		return new double[] { x, y, phi, theta };
	}
	
	@Override
	public Symmetry getSymmetry( )
	{
		return SymmetryType;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public void add( CnGridPoint other )
	{
		x += other.x;
		y += other.y;
		phi += other.phi;
		theta += other.theta;
	}
	
	public void scale( double scalar )
	{
        x *= scalar;
        y *= scalar;
        phi *= scalar;
        theta *= scalar;
	}
	
	public void subtract( CnGridPoint other )
	{
        x -= other.x;
        y -= other.y;
        phi -= other.phi;
        theta -= other.theta;
	}
	
	public String toString( )
	{
		return "[CnGridPoint] (" + x + "," + y + "),(" + phi + "," + theta + ")";
	}
	
	// NOTE: we probably don't need any of this for now
	/*
	@Override
	public int getDimensions() {
		return 4;
	}

	@Override
	GridPoint getInstance() {
		return new CnGridPoint(this);
	}

	@Override
	public void setArray(double[] values) {
        x = values[0];
        y = values[1];
        phi = values[2];
        theta = values[3];

	}

	@Override
	public String toCode() {
        return "new GridPoint( " + x + ", " + y + ", " + phi + ", " + theta + " )";

	}
	*/
}
