/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.cn;

import edu.duke.donaldLab.jdshot.grid.Axes;
import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.StructureGenerator;
import edu.duke.donaldLab.jdshot.grid.cn.CnGridPoint;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.math.Matrix3;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Subunit;
import edu.duke.donaldLab.share.protein.tools.ProteinGeometry;

public class CnStructureGenerator implements StructureGenerator
{
	/**************************
	 *   Data Members
	 **************************/
	
	Subunit m_monomer;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public CnStructureGenerator( Subunit monomer )
	{
		m_monomer = monomer;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	@Override
	public Subunit getMonomer( )
	{
		return m_monomer;
	}
	@Override
	public void setMonomer( Subunit value )
	{
		m_monomer = value;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	@Override
	public Protein getStructure( GridPoint point, int numSubunits )
	{
		// make sure the cast is safe
		assert( point instanceof CnGridPoint );
		
		return getStructure( (CnGridPoint)point, numSubunits );
	}
	
	public Protein getStructure( CnGridPoint point, int numSubunits )
	{
		// build the symmetry parameters from the grid point
		Vector3 symmetryPosition = new Vector3( point.x, point.y, 0 );
		Vector3 symmetryOrientation = new Vector3();
		Axes.getRotationVector( symmetryOrientation, point.phi, point.theta );
		
		return getStructure( symmetryPosition, symmetryOrientation, numSubunits );
	}
	
	public Protein getStructure( Vector3 symmetryPosition, Vector3 symmetryOrientation, int numSubunits )
	{
		Matrix3 rotate = new Matrix3();
		
		// make a new protein with the monomer
		Subunit mainSubunit = new Subunit( m_monomer );
		mainSubunit.setName( 'A' );
		Protein protein = new Protein();
		protein.addSubunit( mainSubunit );
		
		// get the point where the vector intersects the xy plane
		double angle = 2*Math.PI/numSubunits;
		Matrix3.getRotation( rotate, symmetryOrientation, angle );
		for( int i=1; i<numSubunits; i++ )
		{
			// clone the next subunit
			Subunit previousSubunit = protein.getSubunit( i-1 );
			Subunit subunit = new Subunit( previousSubunit );
			subunit.setName( (char)( previousSubunit.getName() + 1 ) );
			protein.addSubunit( subunit );
			
			ProteinGeometry.transform( subunit, rotate, symmetryPosition );
		}
		assert( protein.getSubunits().size() == numSubunits );
		return protein;
	}
}
