/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.test.analyze;

import java.util.Iterator;

import edu.duke.donaldLab.jdshot.TestMain;
import edu.duke.donaldLab.jdshot.analyze.AnalysisContext;
import edu.duke.donaldLab.jdshot.grid.StructureGenerator;
import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.search.SearchContext;
import edu.duke.donaldLab.jdshot.test.ExtendedTestCase;
import edu.duke.donaldLab.share.analysis.RmsdCalculator;
import edu.duke.donaldLab.share.analysis.StructureAligner;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.math.CompareReal;
import edu.duke.donaldLab.share.math.Quaternion;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Subunit;
import edu.duke.donaldLab.share.protein.tools.ProteinGeometry;

public class TestStructureAligner extends ExtendedTestCase
{
	// UNDONE: this test should be moved into a ProteinGeometry test suite
	public void testCentering( )
	throws Exception
	{
		SearchContext searchContext = TestMain.getSearchContext();
		Subunit monomer = searchContext.getMonomer();
		Symmetry symmetry = TestMain.getSearchContext().getSymmetry();
		
		// get a random protein conformation and center it
		StructureGenerator generator = symmetry.newStructureGenerator( monomer );
		GridPoint point = getRandomGridPoint();
		Protein referenceOligomer = generator.getStructure( point, searchContext.getNumSubunits() );
		
		// create another oligomer but don't center it
		// the rmsd should not be zero
		Protein computedOligomer = generator.getStructure( point, searchContext.getNumSubunits() );
		assertRmsdZero( referenceOligomer, computedOligomer );
		
		// center the proteins
		ProteinGeometry.center( referenceOligomer );
		assertRmsdNotZero( referenceOligomer, computedOligomer );
		ProteinGeometry.center( computedOligomer );
		
		// rmsd should be zero
		assertRmsdZero( referenceOligomer, computedOligomer );
	}
	
	public void testTranslatedComputedOligomer( )
	throws Exception
	{
		SearchContext searchContext = TestMain.getSearchContext();
		Subunit monomer = searchContext.getMonomer();
		Symmetry symmetry = TestMain.getSearchContext().getSymmetry();
		
		// get a random protein conformation and center it
		StructureGenerator generator = symmetry.newStructureGenerator( monomer );
		Protein referenceOligomer = generator.getStructure( getRandomGridPoint(), searchContext.getNumSubunits() );
		ProteinGeometry.center( referenceOligomer );
		
		// create a copy of our oligomer (the rmsd should be zero here)
		Protein computedOligomer = new Protein( referenceOligomer );
		assertRmsdZero( referenceOligomer, computedOligomer );
		
		// translate our copy
		Vector3 randomTranslation = getRandomVector( -5.0, 5.0 );
		Iterator<AtomAddressInternal> iterAddress = computedOligomer.getAtomIndex();
		while( iterAddress.hasNext() )
		{
			Vector3 atomPos = computedOligomer.getAtom( iterAddress.next() ).getPosition();
			atomPos.add( randomTranslation );
		}
		
		// the RMSD should NOT be zero here until we find the optimal alignment
		assertRmsdNotZero( referenceOligomer, computedOligomer );
		StructureAligner.alignOptimally( referenceOligomer, computedOligomer );
		assertRmsdZero( referenceOligomer, computedOligomer );
	}
	
	public void testRecomputedOligomer( )
	throws Exception
	{
		SearchContext searchContext = TestMain.getSearchContext();
		Subunit monomer = searchContext.getMonomer();
		Symmetry symmetry = TestMain.getSearchContext().getSymmetry();
		
		// get a random protein conformation and center it
		StructureGenerator generator = symmetry.newStructureGenerator( monomer );
		GridPoint point = getRandomGridPoint();
		Protein referenceOligomer = generator.getStructure( point, searchContext.getNumSubunits() );
		ProteinGeometry.center( referenceOligomer );
		
		// create another oligomer but don't center it
		Protein computedOligomer = generator.getStructure( point, searchContext.getNumSubunits() );
		
		// the RMSD should NOT be zero here until we find the optimal alignment
		assertRmsdNotZero( referenceOligomer, computedOligomer );
		StructureAligner.alignOptimally( referenceOligomer, computedOligomer );
		assertRmsdZero( referenceOligomer, computedOligomer );
	}
	
	public void testRotatedComputedOligomer( )
	throws Exception
	{
		SearchContext searchContext = TestMain.getSearchContext();
		Subunit monomer = searchContext.getMonomer();
		Symmetry symmetry = TestMain.getSearchContext().getSymmetry();
		
		// get a random protein conformation and center it
		StructureGenerator generator = symmetry.newStructureGenerator( monomer );
		Protein referenceOligomer = generator.getStructure( getRandomGridPoint(), searchContext.getNumSubunits() );
		ProteinGeometry.center( referenceOligomer );
		
		// create a copy of our oligomer (the rmsd should be zero here)
		Protein computedOligomer = new Protein( referenceOligomer );
		assertRmsdZero( referenceOligomer, computedOligomer );
		
		// rotate our copy with a random rotation
		Quaternion randomRotation = getRandomRotation();		
		Iterator<AtomAddressInternal> iterAddress = computedOligomer.getAtomIndex();
		while( iterAddress.hasNext() )
		{
			computedOligomer.getAtom( iterAddress.next() ).getPosition().rotate( randomRotation );
		}
		
		// the RMSD should NOT be zero here until we find the optimal alignment
		assertRmsdNotZero( referenceOligomer, computedOligomer );
		StructureAligner.alignOptimally( referenceOligomer, computedOligomer );
		assertRmsdZero( referenceOligomer, computedOligomer );
	}
	
	public void testRotatedAndTranslatedReferenceOligomer( )
	throws Exception
	{
		AnalysisContext analysisContext = TestMain.getAnalysisContext();
		Protein referenceOligomer = analysisContext.getOligomer();
		
		// center the reference
		ProteinGeometry.center( referenceOligomer );
		
		// create a copy of our oligomer (the rmsd should be zero here)
		Protein computedOligomer = new Protein( referenceOligomer );
		assertRmsdZero( referenceOligomer, computedOligomer );
		
		// rotate our copy with a random rotation and translation
		Quaternion randomRotation = getRandomRotation();
		Vector3 randomTranslation = getRandomVector( -5.0, 5.0 );
		Iterator<AtomAddressInternal> iterAddress = computedOligomer.getAtomIndex();
		while( iterAddress.hasNext() )
		{
			Vector3 atomPos = computedOligomer.getAtom( iterAddress.next() ).getPosition();
			atomPos.add( randomTranslation );
			atomPos.rotate( randomRotation );
		}
		
		// the RMSD should NOT be zero here until we find the optimal alignment
		assertRmsdNotZero( referenceOligomer, computedOligomer );
		StructureAligner.alignOptimally( referenceOligomer, computedOligomer );
		assertRmsdZero( referenceOligomer, computedOligomer );
	}
	
	private void assertRmsdZero( Protein reference, Protein compare )
	{
		assertTrue( CompareReal.eq( RmsdCalculator.getRmsd( reference, compare ), 0.0 ) );
	}
	
	private void assertRmsdNotZero( Protein reference, Protein compare )
	{
		// NOTE: There's a (small) possibility that we randomly chose the identitiy
		//       transformation and this assert could trip.
		assertFalse( CompareReal.eq( RmsdCalculator.getRmsd( reference, compare ), 0.0 ) );
	}
}
