/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.nmr;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import edu.duke.donaldLab.share.io.HashCalculator;
import edu.duke.donaldLab.share.protein.AtomAddress;

public class Rdc<T extends AtomAddress<T>>
{
	/**************************
	 *   Definitions
	 **************************/
	
	// NOTE: this default is for NH RDCs
	// other internuclear vectors should pass a different value into sample()
	public static final double DefaultStandardDeviation = 1.0;
	
	
	/**************************
	 *   Data Members
	 **************************/
	
	private static Random m_random;
	
	private T m_from;
	private T m_to;
	private double m_value;
	private double m_error;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	static
	{
		m_random = new Random();
	}
	
	public Rdc( T from, T to )
	{
		m_from = from;
		m_to = to;
		m_value = Double.NaN;
		m_error = Double.NaN;
	}
	
	public Rdc( Rdc<T> other )
	{
		m_from = other.m_from.newCopy();
		m_to = other.m_to.newCopy();
		m_value = other.m_value;
		m_error = other.m_error;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public T getFrom( )
	{
		return m_from;
	}
	public void setFrom( T val )
	{
		m_from = val;
	}
	
	public T getTo( )
	{
		return m_to;
	}
	public void setTo( T val )
	{
		m_to = val;
	}
	
	public double getValue( )
	{
		return m_value;
	}
	public void setValue( double val )
	{
		m_value = val;
	}
	
	public double getError( )
	{
		return m_error;
	}
	public void setError( double val )
	{
		m_error = val;
	}

	
	/**************************
	 *   Static Methods
	 **************************/
	
	public static <T extends AtomAddress<T>> ArrayList<Rdc<T>> copyDeep( List<Rdc<T>> inRdcs )
	{
		ArrayList<Rdc<T>> outRdcs = new ArrayList<Rdc<T>>( inRdcs.size() );
		for( Rdc<T> rdc : inRdcs )
		{
			outRdcs.add( new Rdc<T>( rdc ) );
		}
		return outRdcs;
	}
	
	public static <T extends AtomAddress<T>> void sample( List<Rdc<T>> outRdcs, List<Rdc<T>> inRdcs )
	{
		sample( outRdcs, inRdcs, Rdc.DefaultStandardDeviation );
	}
	
	public static <T extends AtomAddress<T>> void sample( List<Rdc<T>> outRdcs, List<Rdc<T>> inRdcs, double stddev )
	{
		// just in case...
		assert( outRdcs.size() == inRdcs.size() );
		
		// perform the sampling
		for( int i=0; i<inRdcs.size(); i++ )
		{
			outRdcs.get( i ).setError( 0.0 );
			outRdcs.get( i ).setValue( inRdcs.get( i ).sample( stddev ) );
		}
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public double sample( )
	{
		return sample( DefaultStandardDeviation );
	}
	
	public double sample( double stddev )
	{
		// 95% of sampled values should be in the interval
		return m_random.nextGaussian() * stddev + m_value;
	}
	
	@Override
	public String toString( )
	{
		return m_from.toString() + "->" + m_to.toString() + " - " + m_value + " +- " + m_error;
	}
	
	@Override
	public boolean equals( Object other )
	{
		if( other == null )
		{
            return false;
		}
		if( other == this )
		{
			return true;
		}
		if( !( other instanceof Rdc<?> ) )
		{
			return false;
		}
		
		// UNDONE: there has to be a way to check this cast
		return equals( (Rdc<T>)other );
	}
	
	public boolean equals( Rdc<T> other )
	{
		return
			m_value == other.m_value
			&& m_error == other.m_error
			&& m_from.equals( other.m_from )
			&& m_to.equals( other.m_to );
	}
	
	@Override
	public int hashCode( )
	{
		return HashCalculator.combineHashes(
			Double.valueOf( m_value ).hashCode(),
			Double.valueOf( m_error ).hashCode(),
			m_from.hashCode(),
			m_to.hashCode()
		);
	}
}
