/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/
package libprotnmr.nmr;

import java.util.HashSet;
import java.util.Set;

import libprotnmr.io.HashCalculator;
import libprotnmr.protein.AtomAddressInternal;
import libprotnmr.protein.AtomAddressResidueAtom;


/*
	This class' purpose is to override hashCode() and equals() for different comparisons
*/
public class DistanceRestraintResidueAtom extends DistanceRestraint<AtomAddressInternal>
{
	private static final long serialVersionUID = -629824621576456759L;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DistanceRestraintResidueAtom( DistanceRestraint<AtomAddressInternal> other )
	{
		super( other );
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	@Override
	public boolean equals( Object other )
	{
		if( other == null )
		{
            return false;
		}
        if( other == this )
        {
            return true;
        }
        if( other.getClass() != getClass() )
        {
            return false;
        }
        
        return equals( (DistanceRestraintResidueAtom)other );
	}
	
	public boolean equals( DistanceRestraintResidueAtom other )
	{
		return
			getMinDistance() == other.getMinDistance()
			&& getMaxDistance() == other.getMaxDistance()
			&&
			(
				(
					residueAtomEquals( getLefts(), other.getLefts() )
					&& residueAtomEquals( getRights(), other.getRights() )
				)
				||
				(
					residueAtomEquals( getLefts(), other.getRights() )
					&& residueAtomEquals( getRights(), other.getLefts() )
				)
			);
	}
	
	@Override
	public int hashCode( )
	{
		return HashCalculator.combineHashes(
			Double.valueOf( getMinDistance() ).hashCode(),
			Double.valueOf( getMaxDistance() ).hashCode(),
			HashCalculator.combineHashesCommutative(
				residueAtomHashCode( getLefts() ),
				residueAtomHashCode( getRights() )
			)
		);
	}
	
	
	/**************************
	 *   Functions
	 **************************/
	
	private int residueAtomHashCode( Set<AtomAddressInternal> addresses )
	{
		HashSet<AtomAddressResidueAtom> set = new HashSet<AtomAddressResidueAtom>();
		for( AtomAddressInternal address : addresses )
		{
			set.add( new AtomAddressResidueAtom( address ) );
		}
		return set.hashCode();
	}
	
	private boolean residueAtomEquals( Set<AtomAddressInternal> a, Set<AtomAddressInternal> b )
	{
		HashSet<AtomAddressResidueAtom> x = new HashSet<AtomAddressResidueAtom>();
		for( AtomAddressInternal address : a )
		{
			x.add( new AtomAddressResidueAtom( address ) );
		}
		HashSet<AtomAddressResidueAtom> y = new HashSet<AtomAddressResidueAtom>();
		for( AtomAddressInternal address : b )
		{
			y.add( new AtomAddressResidueAtom( address ) );
		}
		return x.equals( y );
	}
}
