package rdcPanda;

 ///////////////////////////////////////////////////////////////////////////////////////////////
//	BackNoe.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** 
   This class provides functions that involve back-computed NOEs.
*   Written by Jianyang (Michael) Zeng (2005-2009).
*/
public class BackNoe
{
    
    /** The first res no of the bk noe. */
    private int  firstResNo;
    
    /** The second res no of the bk noe. */
    private int secondResNo;
    
    /** The first res name of the bk noe. */
    private String firstResName;
    
    /** The second res name of the bk noe. */
    private String secondResName;
    
    /** The h1 atom name of the bk noe. */
    private String h1AtomName;
    
    /** The heavy atom name of the bk noe. */
    private String heavyAtomName;
    
    /** The h2 atom name of the bk noe. */
    private String h2AtomName;
    
    /** The first chemical shift of the bk noe. */
    private double csH1; 
    
    /** The chemical shift of the first heavy atom of the bk noe. */
    private double csHeavy;
    
    /** The second chemical shift of the bk noe. */
    private double csH2;
    
    /** The distance of the bk noe. */
    private double distance;
    
    /** The intensity of the bk noe. */
    private double intensity;
    
    /** The missing probability of the bk noe. */
    private double dbMissingPr=0.0;
    
    /** The missing penalty for the bk noe. */
    private double dbMissingPenalty=0.0;
    
    /** The vector for storing all hn bk noes. */
    private Vector vecHnNoe=new Vector();
    
    /** The vector for storing all ha bk noes. */
    private Vector vecCaNoe=new Vector();
    
    /** The rmsd, not used now. */
    private double rmsd;//rmsd between itself to the nearest point in the exp peak list
    					//not used now
    
    /** The rmsd for cs1, not used now. */
					    private double rmsdH1=0.0;
    
    /** The rmsd heavy, not used now. */
    private double rmsdHeavy=0.0;
    
    /** The rmsd h2. Not used now */
    private double rmsdH2=0.0;
    
    /** whether the noe is missing. */
    private boolean isMissing=false;
    
    /**
     * Sets the missing penalty.
     * 
     * @param distThresh the dist thresh
     * @param penalty4SmallDist the penalty4 small dist
     * @param penalty4LargeDist the penalty4 large dist
     */
    public void setMissingPenalty(double distThresh, double penalty4SmallDist, double penalty4LargeDist)
    {
    	if (distance<distThresh)
    		dbMissingPenalty=penalty4SmallDist;
    	else
    		dbMissingPenalty=penalty4LargeDist;
    	
    }
   
    /**
     * Sets the missing prob.
     * 
     * @param pr the new missing prob
     */
    public void setMissingProb(double pr)
    {
    	dbMissingPr=pr;
    }
    
    /**
     * Instantiates a new back noe.
     */
    public BackNoe()
    {
    	csH1=-999.9;
    	csHeavy=-999.9;
    	csH2=-999.9;
    	firstResNo=0;
    	secondResNo=0;
    	firstResName="";
    	secondResName="";
    	h1AtomName="";
    	heavyAtomName="";
    	h2AtomName="";
    	distance=0.0;
    	intensity=0.0;	
    	
    	rmsd=0.0;
    	isMissing=false;
    	rmsdH1=0.0;
    	rmsdHeavy=0.0;
    	rmsdH2=0.0;
    }		
    
    /**
     * Instantiates a new back noe.
     * 
     * @param csh1 the csh1
     * @param csheavy the csheavy
     * @param csh2 the csh2
     */
    public BackNoe(double csh1, double csheavy, double csh2)
    {
    	csH1=csh1;
    	csHeavy=csheavy;
    	csH2=csh2;
    	firstResNo=0;
    	secondResNo=0;
    	firstResName="";
    	secondResName="";
    	h1AtomName="";
    	heavyAtomName="";
    	h2AtomName="";
    	distance=0.0;
    	intensity=0.0;	
    	
    	rmsd=0.0;
    	isMissing=false;
    	rmsdH1=0.0;
    	rmsdHeavy=0.0;
    	rmsdH2=0.0;
    }		
  
    /**
     * Instantiates a new back noe.
     * 
     * @param csh1 the csh1
     * @param csheavy the csheavy
     * @param csh2 the csh2
     * @param dist the dist
     * @param inten the inten
     * @param no1 the no1
     * @param no2 the no2
     * @param res1 the res1
     * @param res2 the res2
     * @param h1 the h1
     * @param heavy the heavy
     * @param h2 the h2
     */
    public BackNoe(double csh1, double csheavy, double csh2, double dist,double inten, 
    		int no1, int no2, String res1, String res2, String h1, String heavy, String h2)
    {
    	csH1=csh1;
    	csHeavy=csheavy;
    	csH2=csh2;
    	firstResNo=no1;
    	secondResNo=no2;
    	firstResName=res1;
    	secondResName=res2;
    	h1AtomName=h1;
    	heavyAtomName=heavy;
    	h2AtomName=h2;
    	distance=dist;
    	intensity=inten;	
    
    	rmsd=999.9;
    	isMissing=false;
    	rmsdH1=0.0;
    	rmsdHeavy=0.0;
    	rmsdH2=0.0;
    }		
    
    /**
     * Adds the hn noe.
     * 
     * @param hn_noe the hn_noe
     */
    public void addHnNoe(HnNoe hn_noe)
    {
    	vecHnNoe.add(hn_noe);    	
    }
    
    /**
     * Adds the ca noe.
     * 
     * @param ca_noe the ca_noe
     */
    public void addCaNoe(HcchTocsy ca_noe)
    {
    	vecCaNoe.add(ca_noe);
    }
    
    /**
     * Sets the rmsd.
     * 
     * @param dst the new rmsd
     */
    public void setRmsd(double dst)
    {
    	rmsd=dst;
    }
    
    /**
     * Sets the checks if is missing.
     * 
     * @param bool the new checks if is missing
     */
    public void setIsMissing(boolean bool)
    {
    	isMissing=bool; 
    }
    
    /**
     * Sets the rmsd all.
     * 
     * @param csh1 the csh1
     * @param csheavy the csheavy
     * @param csh2 the csh2
     */
    public void setRmsdAll(double csh1, double csheavy, double csh2)
    {
    	rmsdH1=csh1;
    	rmsdHeavy=csheavy;
    	rmsdH2=csh2;
    }
    
    //getting the values	
    /**
     * Gets the rmsd h1.
     * 
     * @return the rmsd h1
     */
    public double getRmsdH1()
    {
    	return rmsdH1;
    }
    
    /**
     * Gets the rmsd heavy.
     * 
     * @return the rmsd heavy
     */
    public double getRmsdHeavy()
    {
    	return rmsdHeavy;
    }
    
    /**
     * Gets the rmsd h2.
     * 
     * @return the rmsd h2
     */
    public double getRmsdH2()
    {
    	return rmsdH2;
    }
    
    /**
     * Gets the rmsd.
     * 
     * @return the rmsd
     */
    public double getRmsd()
    {
    	return rmsd;
    }
    
    /**
     * Gets the missing penalty.
     * 
     * @return the missing penalty
     */
    public double getMissingPenalty()
    {
    	return dbMissingPenalty;
    }
    
    /**
     * Gets the distance.
     * 
     * @return the distance
     */
    public double getDistance()
    {
    	return distance;
    }
    
    /**
     * Gets the intensity.
     * 
     * @return the intensity
     */
    public double getIntensity()
    {
    	return intensity;
    }
    
    /**
     * Gets the h1 name.
     * 
     * @return the h1 name
     */
    public String getH1Name()
    {
    	return h1AtomName;
    }
    
    /**
     * Gets the heavy name.
     * 
     * @return the heavy name
     */
    public String getHeavyName()
    {
    	return heavyAtomName;
    }
    
    /**
     * Gets the h2 name.
     * 
     * @return the h2 name
     */
    public String getH2Name()
    {
    	return h2AtomName;
    }
    
    /**
     * Gets the first res name.
     * 
     * @return the first res name
     */
    public String getFirstResName()
    {
    	return firstResName;
    }
    
    /**
     * Gets the second res name.
     * 
     * @return the second res name
     */
    public String getSecondResName()
    {
    	return secondResName;
    }
    
    /**
     * Gets the cS h1.
     * 
     * @return the cS h1
     */
    public double getCSH1()
    {
	return csH1;
    }	
    public void setCSH1(double cs)
    {
	  csH1=cs;
    }	
    /**
     * Gets the cS heavy.
     * 
     * @return the cS heavy
     */
    public double getCSHeavy()
    {
	return csHeavy;
    }	
    public void setCSHeavy(double cs)
    {
	 csHeavy=cs;
    }	
    /**
     * Gets the cS h2.
     * 
     * @return the cS h2
     */
    public double getCSH2()
    {
	return csH2;
    }	
    public void setCSH2(double cs)
    {
	 csH2=cs;
    }	
    /**
     * Gets the checks if is missing.
     * 
     * @return the checks if is missing
     */
    public boolean getIsMissing()
    {
    	return isMissing;
    }
    
    /**
     * Gets the first res no.
     * 
     * @return the first res no
     */
    public int getFirstResNo()
    {
    	return firstResNo;
    }

    /**
     * Gets the missing prob.
     * 
     * @return the missing prob
     */
    public double getMissingProb()
    {
    	return dbMissingPr;
    }
    
    /**
     * Gets the second res no.
     * 
     * @return the second res no
     */
    public int getSecondResNo()
    {
    	return secondResNo;
    }
   
    /**
     * Gets the hn noe vector.
     * 
     * @return the hn noe vector
     */
    public Vector getHnNoeVector()
    {
    	return vecHnNoe;
    }
    
    /**
     * Gets the ca noe vector.
     * 
     * @return the ca noe vector
     */
    public Vector getCaNoeVector()
    {
    	return vecCaNoe;
    }
    
    /**
     * delete repeated back-computed NOEs.
     * 
     * @param vecBackNoe original back-computed NOEs
     * 
     * @return new back-computed NOEs
     */
    public Vector<BackNoe> DeleteRepeat(Vector<BackNoe> vecBackNoe)
	{
    	Vector vecBkNoeNew=new Vector();
    	for(int i=0;i<vecBackNoe.size();i++)
    	{
    		BackNoe bkNoeA=(BackNoe)vecBackNoe.elementAt(i);
    		double csH1=bkNoeA.getCSH1();
    		double csH2=bkNoeA.getCSH2();
    		double csHeavy=bkNoeA.getCSHeavy();
    		boolean isExist=false;
    		
    		for(int j=0; j<vecBackNoe.size();j++)
    		{
    			if(i==j)
    				continue;
    			BackNoe bkNoeB=(BackNoe)vecBackNoe.elementAt(j);
    			double cs_H1=bkNoeB.getCSH1();
        		double cs_H2=bkNoeB.getCSH2();
        		double cs_Heavy=bkNoeB.getCSHeavy();
        		if(Math.abs(cs_H1-csH1)<0.00001 && Math.abs(cs_H2-csH2)<0.00001 && Math.abs(cs_Heavy-csHeavy)<0.00001)
        			isExist=true;
    		}//for(int j=0; j<vecBackNoe.size();j++)
    		if(isExist)
    		{
    			vecBackNoe.remove(i);
    			i--;
    		}
    		
    	}//for(int i=0;i<vecBackNoe.size();i++)
    	vecBkNoeNew.addAll(vecBackNoe);
    	return vecBkNoeNew;
	}

  




}
