package rdcPanda;

 ///////////////////////////////////////////////////////////////////////////////////////////////
//	HcchTocsy.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 *  
*   This class defines data structures related to HcchTocsy.
*   Written by Lincong Wang (2001-2005).
*/
public class HcchTocsy{
    
    /** The residue no. */
    private int    residueNo; 
    
    /** The residue. */
    private String residue;
    
    /** The cs ha. */
    private double csHA; 
    
    /** The cs c13. */
    private double csC13;
    
    /** The cs h1. */
    private double csH1;
    
    /** The intensity. */
    private double intensity;
    
    /** The uncertainty. */
    private double uncertainty=0.0;
    
    /** The assign map. */
    private Map assignMap = new TreeMap();

  /**
   * Instantiates a new hcch tocsy.
   */
  public HcchTocsy(){
	residueNo = 0;
	residue = null;
	csHA = 0.0; 
        csC13 = 0.0;
        csH1 =0.0;
	intensity = 0.0;
	assignMap = new TreeMap();
    }		
    
    /**
     * Instantiates a new hcch tocsy.
     * 
     * @param no the no
     */
    public HcchTocsy(int no){
	residueNo = no;
	residue = null;
	csHA = 0.0; 
        csC13 = 0.0;
        csH1 = 0.0;
	intensity = 0.0;
	assignMap = new TreeMap();
    }		
    
    /**
     * Instantiates a new hcch tocsy.
     * 
     * @param no the no
     * @param resid the resid
     * @param HN the hN
     * @param N15 the n15
     * @param H1 the h1
     * @param inten the inten
     */
    public HcchTocsy(int no, String resid, double HN, double N15, double H1, double inten){
    	residueNo = no;
    	residue = resid;
    	csHA = HN; 
            csC13 = N15;
            csH1 = H1;
    	intensity = inten;
    	assignMap = new TreeMap();
        }	
    
    /**
     * Instantiates a new hcch tocsy.
     * 
     * @param no the no
     * @param resid the resid
     * @param HN the hN
     * @param N15 the n15
     * @param H1 the h1
     * @param inten the inten
     * @param uncern the uncern
     */
    public HcchTocsy(int no, String resid, double HN, double N15, double H1, double inten,double uncern){
	residueNo = no;
	residue = resid;
	csHA = HN; 
        csC13 = N15;
        csH1 = H1;
	intensity = inten;
	assignMap = new TreeMap();
	uncertainty=uncern;
    }	

  //getting the values	
    /**
   * Gets the residue no.
   * 
   * @return the residue no
   */
  public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the residue type.
     * 
     * @return the residue type
     */
    public String getResidueType(){
	return residue;
    }	
    
    /**
     * Gets the hA.
     * 
     * @return the hA
     */
    public double getHA(){
	return csHA;
    }	
    
    /**
     * Gets the c13.
     * 
     * @return the c13
     */
    public double getC13(){
	return csC13;
    }	
    
    /**
     * Gets the h1.
     * 
     * @return the h1
     */
    public double getH1(){
	return csH1;
    }	
    
    /**
     * Gets the intensity.
     * 
     * @return the intensity
     */
    public double getIntensity(){
	return intensity;
    }
    
    /**
     * Gets the assign map.
     * 
     * @return the assign map
     */
    public Map getAssignMap(){
	return assignMap;
    }	

    //set the values	
    /**
     * Sets the residue no.
     * 
     * @param R the new residue no
     */
    public void setResidueNo(int R){
	residueNo = R;
    }	
    
    /**
     * Sets the residue type.
     * 
     * @param R the new residue type
     */
    public void setResidueType(String R){
	residue = R;
    }	
    
    /**
     * Sets the assign map.
     * 
     * @param aMap the new assign map
     */
    public void setAssignMap(Map aMap){
	assignMap = aMap;
    }	

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
	String desc = "";
	String str = String.valueOf(residueNo) +" "+residue+"\n";
	double csValue = 0.0;
	if (!assignMap.isEmpty()){
	    for (Iterator i = assignMap.entrySet().iterator(); i.hasNext(); ){
		Map.Entry e = (Map.Entry) i.next();
		csValue =  ((Double)e.getValue()).doubleValue();
		if( Math.abs(csValue+999.99) > 0.01)
		    str += (String)e.getKey()+" = "+ (Double)e.getValue()+"\n";
	    }
	    desc +=  str+"\n";
	    desc += String.valueOf(csHA)+"  "+String.valueOf(csC13)
		+"   "+String.valueOf(csH1)+"  "+String.valueOf(intensity);
	}else{
	    desc =  residueNo+"  "+String.valueOf(csHA) +"  "+String.valueOf(csC13)
		+"   "+String.valueOf(csH1)+"  "+String.valueOf(intensity);
	}
	return desc;	
    }

    /**
     * The Class resNoComparator.
     */
    public static class resNoComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    HcchTocsy n1 = (HcchTocsy)o1;
	    HcchTocsy n2 = (HcchTocsy)o2;
	    double d1 = n1.getResidueNo();
	    double d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class haComparator.
     */
    public static class haComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    HcchTocsy n1 = (HcchTocsy)o1;
	    HcchTocsy n2 = (HcchTocsy)o2;
	    double d1 = n1.getHA();
	    double d2 = n2.getHA();
	    if (d1 > d2)
		return -1;
	    else if (d1 < d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class h1Comparator.
     */
    public static class h1Comparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    HcchTocsy n1 = (HcchTocsy)o1;
	    HcchTocsy n2 = (HcchTocsy)o2;
	    double d1 = n1.getH1();
	    double d2 = n2.getH1();
	    if (d1 > d2)
		return -1;
	    else if (d1 < d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class c13Comparator.
     */
    public static class c13Comparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    HcchTocsy n1 = (HcchTocsy)o1;
	    HcchTocsy n2 = (HcchTocsy)o2;
	    double d1 = n1.getC13();
	    double d2 = n2.getC13();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

   /**
    * read peak list generated by ANSIG.
    * 
    * @param filename the name of the file
    * 
    * @return a vector of RDC object
    */
    public Vector hcchReader(String filename){
	Vector inputs = new Vector();
	double csHA = 0.0, csC13 = 0.0, csH1 = 0.0, intensity = 0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	String aaType = "", ss ="";
	try{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    ss = in.readLine();
	    ss = in.readLine();
	    stop:
	    while(true) {
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    csHA = new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    csH1 = new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    csC13 = (new Double(st.nextToken()).doubleValue())- 1.3; //Calibrate?
		if (st.hasMoreTokens()){
		    str = st.nextToken();
		    index1 = str.indexOf("e");
		    order = str.substring(index1+1);
		    value = new Double(str.substring(0, index1)).doubleValue();
// 		    System.out.println(value);
		    if (order.equals("+10"))
			intensity = value*10000;
		    if (order.equals("+09"))
			intensity = value*1000;
		    else if (order.equals("+08"))
			intensity = value*100;
		    else if (order.equals("+07"))
			intensity = value*10;
		    else if (order.equals("+06"))
			intensity = value;
		    else if (order.equals("+05"))
			intensity = value*0.10;
		    else System.out.println("Intensity Error");
		}
		inputs.add(new HcchTocsy(no, aaType, csHA, csC13, csH1, intensity));
		if ((ss = in.readLine()) == null){
// 		    inputs.add(new HcchTocsy(no, aaType, csHA, csC13, csH1, intensity));
		    break stop;
		}
	    }
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }
    
    /**
     * Read NOESY peaks in xeasy format
     * The required file format is:.
     * 
     * @param filename the name of the file
     * 
     * @return a vector of HnNOE object
     */
    public Vector NoesyReader(String filename)
    {
	Vector inputs = new Vector();
	double csHa = 0.0, csC13 = 0.0, csH1 = 0.0, intensity = 0.0, uncertainty=0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	String strTemp="";
	double csID=0;
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	String aaType = "", ss ="";
	int counter=0;
	int    index = -1;
	try
	{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    //ss = in.readLine();
	    //ss = in.readLine();
	    stop:
	    while(true) 
	    {
	    	index = ss.trim().indexOf("#");
	    	if (index==0)
	    	{
	    		if ((ss = in.readLine()) == null)
		    		break stop;
	    		continue;	    		
	    	}
	    	
	    	st = new StringTokenizer(ss);
	    	if (st.hasMoreTokens())
	    		csID = new Double(st.nextToken()).doubleValue();
	    	if (st.hasMoreTokens())
	    		csC13 = new Double(st.nextToken()).doubleValue();
	    	if (st.hasMoreTokens())
	    		csHa = new Double(st.nextToken()).doubleValue();	    	
	    	if (st.hasMoreTokens())
	    		csH1 = new Double(st.nextToken()).doubleValue();
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	if (st.hasMoreTokens())
	    	{
	    		intensity = new Double(st.nextToken()).doubleValue();
	    		//if (intensity < 1.0)
	    			//intensity *= 10;
	    	}
	    	if (st.hasMoreTokens())
	    		uncertainty = new Double(st.nextToken()).doubleValue();
	    	//else intensity = 3.0;
	    	counter=counter+1;
	    	inputs.add(new HnNoe(0, "", csHa, csC13, csH1, intensity,uncertainty));
	    	if ((ss = in.readLine()) == null)
	    		break stop;
	    	}
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	 System.out.println("The total number of HN-NOE frequency triples is: " + counter);
	return inputs;
    }
   
   /**
    * Read peak list generated by NMRView.
    * 
    * @param filename the name of the file
    * 
    * @return a vector of RDC object
    */
    public Vector hcchReader2(String filename){
	Vector inputs = new Vector();
	double csHA = 0.0, csC13 = 0.0, csH1 = 0.0, intensity = 0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	String aaType = "", ss ="";
	try{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    stop:
	    while(true) {
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    csHA = new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    csH1 = new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    csC13 = (new Double(st.nextToken()).doubleValue()) -0.5;
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    intensity = (new Double(st.nextToken()).doubleValue());
		inputs.add(new HcchTocsy(no, aaType, csHA, csC13, csH1, intensity));
		if ((ss = in.readLine()) == null){
		    break stop;
		}
	    }
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }

   /**
    * Read peak list generated by NMRView.
    * 
    * @param filename the name of the file
    * 
    * @return a vector of RDC object
    */
    public Vector hcchReader3(String filename){
	Vector allVec = new Vector();
	Vector inputs = new Vector();
	double csHA = 0.0, csC13 = 0.0, csH1 = 0.0, intensity = 0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	int cnt = 0;
	String aaType = "", ss ="";
	try{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    st = new StringTokenizer(ss);
	    no =  Integer.parseInt(st.nextToken());
	    stop:
	    while((ss = in.readLine()) != null) {
		st = new StringTokenizer(ss);
		cnt = st.countTokens();
		if (cnt == 2){

		    no =  Integer.parseInt(st.nextToken());
		    continue stop;
		}
		csHA = new Double(st.nextToken()).doubleValue();
		csC13 = (new Double(st.nextToken()).doubleValue());
		csH1 = new Double(st.nextToken()).doubleValue();
		intensity = (new Double(st.nextToken()).doubleValue());
		inputs.add(new HcchTocsy(no, aaType, csHA, csC13, csH1, intensity));
	    }

	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs; //allVec;
    }

  /**
   * Read the cnoe pk list generated manually.
   * 
   * @param filename the name of the file
   * 
   * @return a vector of RDC object
   */
    public Vector hcchReader4(String filename){
	Vector allVec = new Vector();
	Vector inputs = new Vector();
	double csHA = 0.0, csC13 = 0.0, csH1 = 0.0, intensity = 0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	int cnt = 0;
	String aaType = "", ss ="";
	int counter=0;
	try{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    st = new StringTokenizer(ss);
	    st.nextToken();
	    no =  Integer.parseInt(st.nextToken());
	    csHA =  new Double(st.nextToken()).doubleValue();
	    st.nextToken();
	    csC13 =  new Double(st.nextToken()).doubleValue();
	    stop:
	    while((ss = in.readLine()) != null) {
		st = new StringTokenizer(ss);
		cnt = st.countTokens();
		if (cnt == 2){
		    csH1 = new Double(st.nextToken()).doubleValue();
		    intensity = (new Double(st.nextToken()).doubleValue());
		    counter=counter+1;
		    
		    inputs.add(new HcchTocsy(no, aaType, csHA, csC13, csH1, intensity));
		    continue stop;
		}

		    if (st.hasMoreTokens())
			st.nextToken();
		    if (st.hasMoreTokens())
			no =  Integer.parseInt(st.nextToken());
		    if (st.hasMoreTokens())	
			csHA = new Double(st.nextToken()).doubleValue();
		    if (st.hasMoreTokens())
			st.nextToken();
		    if (st.hasMoreTokens())
			csC13 = (new Double(st.nextToken()).doubleValue());
		}

	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	System.out.println("The total number of HA-NOE frequency triples is: " + counter);
	return inputs; //allVec;
    }

    /**
     * Range search.
     * 
     * @param nucType the nuc type
     * @param a the a
     * @param b the b
     * @param pkVec the pk vec
     * 
     * @return the vector
     */
    public Vector rangeSearch(String nucType, double a, double b, Vector pkVec){
	Vector outVec = new Vector();
	double csValue = 0.0;
	for (int i=0; i< pkVec.size(); i++){
	    if (nucType.equals("H1"))
		csValue = ((HcchTocsy)pkVec.elementAt(i)).getHA();
	    else if (nucType.equals("N15"))
		csValue = ((HcchTocsy)pkVec.elementAt(i)).getC13();
	    if (csValue >= a && csValue <= b)
		outVec.add(pkVec.elementAt(i));
	}
	return outVec;
    }

    /**
     * Delete repeat.
     * 
     * @param hcchVec the hcch vec
     * @param epsH the eps h
     * @param epsH1 the eps h1
     * @param epsC13 the eps c13
     * 
     * @return the vector
     */
    public Vector deleteRepeat(Vector hcchVec, double epsH, double epsH1, double epsC13){    
	Vector outVec = new Vector();
	double csHa1 = 0.0,  csHa2 = 0.0;
	double csH1 = 0.0,  csH2 = 0.0;
	double csC1 = 0.0, csC2 = 0.0;
	double intensity1 = 0.0, intensity2 = 0.0;
	HcchTocsy hcch = new HcchTocsy();
	HcchTocsy hcch2 = new HcchTocsy();
	int i, j;
	int no = 0;
	outVec.add(hcchVec.elementAt(0));
	boolean bigger = false;
	boolean smaller = false;
	for(i=1; i<hcchVec.size(); i++){
	    hcch = (HcchTocsy) hcchVec.elementAt(i);
	    csHa1 = hcch.getHA();
	    csH1 = hcch.getH1();
	    csC1  = hcch.getC13();
	    intensity1 = hcch.getIntensity();
	    bigger = false;
	    smaller = false;
	    for (j=0; j<outVec.size(); j++){
		hcch2 = (HcchTocsy) outVec.elementAt(j);
		csHa2 = hcch2.getHA();
		csH2 = hcch2.getH1();
		csC2 = hcch2.getC13();
		intensity2 = hcch2.getIntensity();
		if ( Math.abs(csHa1 - csHa2) < epsH && Math.abs(csH1 - csH2) < epsH1
		     && Math.abs(csC1 - csC2) < epsC13 && intensity1 > intensity2){
		    bigger = true;
		    outVec.setElementAt(hcch, j);
		    break;
		}else  if ( Math.abs(csHa1 - csHa2) < epsH && Math.abs(csH1 - csH2) < epsH1
		     && Math.abs(csC1 - csC2) < epsC13 && intensity1 <= intensity2){
		    smaller = true;
		    break;
		}
	    }
	    if(!bigger && !smaller)
		outVec.add(hcch);

	}
	return outVec;
    }
			    
  /**
   * Merge hcch.
   * 
   * @param pkVec the pk vec
   * @param epsH the eps h
   * @param epsC the eps c
   * 
   * @return the vector
   */
    public Vector mergeHcch(Vector pkVec, double epsH, double epsC){
	double csHA = 0.0;
	double csC13 = 0.0;
	double csHATmp = 0.0;
	double csC13Tmp = 0.0;
	Vector outVec = new Vector();
	Vector allVec = new Vector();
	double csValue = 0.0;
	HcchTocsy hncc = new HcchTocsy();
	int i = 0, j = 0;
	Collections.sort(pkVec, new HcchTocsy.c13Comparator());
	int count = 0, cnt1 = 0, cnt2 = 0;
	int N = pkVec.size();
	while (!pkVec.isEmpty()){
	    hncc = (HcchTocsy)pkVec.elementAt(0);
	    csHA  = hncc.getH1();
	    csC13 = hncc.getC13();
	    outVec.add(hncc);
	    cnt1 = 0;
	    for (i=1; i< pkVec.size(); i++){
		hncc = (HcchTocsy)pkVec.elementAt(i);
		csHATmp = hncc.getH1();
		csC13Tmp = hncc.getC13();
		if ( Math.abs(csHATmp -csHA) < epsH && Math.abs(csC13Tmp -csC13) < epsC){
		    outVec.add(hncc);
		    cnt1++;
		}
	    }
	    allVec.add(outVec);
	    outVec = new Vector();
	    cnt2 = 0;   //remove the added element
	    for (j=0; j < pkVec.size();){
		hncc = (HcchTocsy)pkVec.elementAt(j);
		csHATmp = hncc.getH1();
		csC13Tmp = hncc.getC13();
		if ( Math.abs(csHATmp -csHA) < epsH && Math.abs(csC13Tmp -csC13) < epsC){
		    pkVec.removeElementAt(j);
		    cnt2++;
		}else j++;
	    }	   
	    pkVec.trimToSize();
	    N = pkVec.size();
	}
	return allVec;
    }

   
}
