package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	PdbRdc.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 *  
*   This class provides various SVD methods for best fitting RDCs and a structural fragment,
 *  and grid search method for computing the three Euler angles. Written by Lincong Wang (2001-2005) 
*/
public class PdbRdc {
    
    /** The pdb coord. */
    private Pdb pdbCoord;
    
    /** The rdc val. */
    private double rdcVal;
  
    /**
     * Instantiates a new pdb rdc.
     */
    public PdbRdc (){
	pdbCoord = null;
	rdcVal = 0.0;
    }
    
    /**
     * Instantiates a new pdb rdc.
     * 
     * @param pp the pp
     * @param rdc the rdc
     */
    public PdbRdc (Pdb pp, double rdc){
	pdbCoord = pp;
	rdcVal = rdc;
    }
    
    /**
     * Gets the pdb.
     * 
     * @return the pdb
     */
    public Pdb getPdb(){
	return pdbCoord;
    }
    
    /**
     * Gets the rdc.
     * 
     * @return the rdc
     */
    public double getRdc(){
	return rdcVal;
    }

    /**
     * cal the directioal cosines of the vector vec.
     * 
     * @param vec the vec
     * 
     * @return the double[]
     */
    public double[] dirCos(double [] vec) {
	double len = 0.0;
	double [] dirs = new double[vec.length];
	for (int i=0; i<vec.length; i++)
	    len += vec[i] * vec[i];
	for (int j=0; j<vec.length; j++)
	    dirs[j] = vec[j] / Math.sqrt(len);
	return dirs;
    }

 /**
  * Compute the internuclear vector between two atoms.
  * 
  * @param n1 the coordinate for atom 1
  * @param n2 the coordinate for atom 2
  * 
  * @return a vector from n1->n2
  */
    public double[] internuclearVec(double[] n1, double[] n2){
        return new double[]{n2[0]-n1[0], n2[1]-n1[1], n2[2]-n1[2]};
    }

    /**
     * Compute the new vector after translation.
     * 
     * @param n1 the coordinate for atom 1
     * @param vT  the translation vector
     * 
     * @return a vector from n1->n2
     */
    public double[] newVecByTranslation(double[] n1,  double[] vT)
    {
        return new double[]{n1[0] + vT[0], n1[1] + vT[1], n1[2] + vT[2]};        
    }
    
   /**
    * Computing the Saupe matrix by SVD method and then
    * diagonalizing it for computing the diagonalized elements and
    * rotation matrix to one of POF.
    * 
    * @param pdbVec  a vector of PDB objects.
    * @param rdcVec  a corresponding RDC vector.
    * @param rmsdArr for returning the five Saupe elements and the best-fitted RMSD
    * @param atom1   the nucleus type whose RDC for the internuclear vector are recorded
    * @param atom2   the nucleus type whose RDC for the internuclear vector are recorded
    * @param printResults if true, it prints the back-computed RDCs.
    * 
    * @return True if the SVD matrix is NOT singular, otherwise False
    * The order of atom1 and atom2 is important (different order == sign inversion)
    */
    public boolean bestFit(final Vector pdbVec, final Vector rdcVec, String atom1, String atom2,
			   double[] rmsdArr, boolean printResults){
	Pdb pp = new Pdb();
	Dipolar dd = new Dipolar();
	int i, j, k;
	int no;
	String resid;
	Vector atomVec = new Vector();
	double [] nToNHVec = new double[3];
	double [] amide = new double[3];
	double [] nh = new double[3];
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Vector pdbRdcVec = new Vector();
	Cartesian cc = new Cartesian();
	String atom = "";
	int index;
	for (i=0; i<rdcVec.size(); i++){
	    dd = (Dipolar)rdcVec.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
		pdbRdcVec.add(new PdbRdc((Pdb)pdbVec.elementAt(index), rdcExp)); 
	}
	int m = pdbRdcVec.size();   //number of rows
	int n = 5;                    //number of columns
	double [][] equ = new double[m][n];
	double [] b     = new double[m];
	for (i=0; i<pdbRdcVec.size(); i++){
	    pp = ((PdbRdc)pdbRdcVec.elementAt(i)).getPdb();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = dirCos(nToNHVec);
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    equ[i][0] = cosY * cosY - cosX * cosX;
	    equ[i][1] = cosZ * cosZ - cosX * cosX;
	    equ[i][2] = 2.0 * cosX * cosY;
	    equ[i][3] = 2.0 * cosX * cosZ;
	    equ[i][4] = 2.0 * cosY * cosZ;
	    b[i] = ((PdbRdc)pdbRdcVec.elementAt(i)).getRdc();
	}
	Matrix A = new Matrix(equ, m, n);
	SingularValueDecomposition SVD = A.svd();
	double [] singularvalues = SVD.getSingularValues();
	double[] singulars = new double[6];

	for (k=0; k<singularvalues.length; k++){
	    if (singularvalues[k] != 0.0)
		singulars[k] = 1 / singularvalues[k];
	    else return false;
	}
	double [][] ArrS = { {singulars[0], 0.0, 0.0, 0.0, 0.0},
			     {0.0, singulars[1], 0.0, 0.0, 0.0},
			     {0.0, 0.0, singulars[2], 0.0, 0.0},
			     {0.0, 0.0, 0.0, singulars[3], 0.0},
			     {0.0, 0.0, 0.0, 0.0, singulars[4]}};
	Matrix U = SVD.getU();
	Matrix V = SVD.getV();
	Matrix S  = SVD.getS();
	Matrix invS = new Matrix(ArrS, n, n);
	int errorCount = 0;
	/** comment out at your own risk */
	try {
	    check(A,SVD.getU().times(SVD.getS().times(SVD.getV().transpose())));
	    try_success("SingularValueDecomposition...","");
	} catch ( java.lang.RuntimeException e ) {
	    errorCount = try_failure(errorCount,"SingularValueDecomposition...",
				     "incorrect singular value decomposition calculation");
	}
	/** comment out at your own risk */

	double [] saupe = V.times(invS.times(U.transpose().times(b)));
	double Syy = saupe[0], Szz = saupe[1], Sxy = saupe[2], Sxz = saupe[3], Syz = saupe[4];
	double r = 0.0, rdcCal = 0.0;
	//back-computing RDCs	
	if (printResults)
	    System.out.println("ResidueNO,   back-computed RDC,   measured RDC,  difference");
	for (i = 0; i < pdbRdcVec.size(); i++){
	    pp = ((PdbRdc)pdbRdcVec.elementAt(i)).getPdb(); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = dirCos(nToNHVec);
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcCal =  (cosY * cosY - cosX * cosX) * Syy
		       + (cosZ * cosZ - cosX * cosX) * Szz
		       + 2.0 * cosX * cosY * Sxy
		       + 2.0 * cosX * cosZ * Sxz
		       + 2.0 * cosY * cosZ * Syz;
	    rdcExp = ((PdbRdc)pdbRdcVec.elementAt(i)).getRdc();
	    if (printResults)
		System.out.println(pp.getResidueNo()+"  "+rdcCal+"  "+rdcExp+"  "+(rdcCal-rdcExp));
	    r += (rdcCal-rdcExp) * (rdcCal-rdcExp);
	}
	if (pdbRdcVec.size() < 2){
	    System.out.println("Error in bestFit method, possible reason: either Pdb or RDC file is not read properly");
	    System.exit(1);
	}
	if (printResults)
	    System.out.println("rmsd :"+ Math.sqrt( r / pdbRdcVec.size() ) );
	for (k=0; k<saupe.length; k++)
	    rmsdArr[k] = saupe[k];
	rmsdArr[5] = Math.sqrt(r / pdbRdcVec.size() );  //The last one
	return true;
    } 
    
    /**
     * Computing the Saupe matrix by SVD method and then
     * diagonalizing it for computing the diagonalized elements and
     * rotation matrix to one of POF.
     * 
     * @param pdbVec  a vector of PDB objects.
     * @param rdcVec  a corresponding RDC vector.
     * @param rmsdArr for returning the five Saupe elements and the best-fitted RMSD
     * @param rdcArrN  back-computed RDCs
     * @param atom1   the nucleus type whose RDC for the internuclear vector are recorded
     * @param atom2   the nucleus type whose RDC for the internuclear vector are recorded
     * @param printResults if true, it prints the back-computed RDCs.
     * 
     * @return True if the SVD matrix is NOT singular, otherwise False
     * The order of atom1 and atom2 is important (different order == sign inversion)
     */
    public boolean bestFit(final Vector pdbVec, final Vector rdcVec, String atom1, String atom2,
			   double[] rmsdArr, double[] rdcArrN, boolean printResults){
	Pdb pp = new Pdb();
	Dipolar dd = new Dipolar();
	int i, j, k;
	int no;
	String resid;
	Vector atomVec = new Vector();
	double [] nToNHVec = new double[3];
	double [] amide = new double[3];
	double [] nh = new double[3];
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Vector pdbRdcVec = new Vector();
	Cartesian cc = new Cartesian();
	String atom = "";
	int index;
	for (i=0; i<rdcVec.size(); i++){
	    dd = (Dipolar)rdcVec.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
		pdbRdcVec.add(new PdbRdc((Pdb)pdbVec.elementAt(index), rdcExp)); 
	}
	int m = pdbRdcVec.size();   //number of rows
	int n = 5;                    //number of columns
	double [][] equ = new double[m][n];
	double [] b     = new double[m];
	for (i=0; i<pdbRdcVec.size(); i++){
	    pp = ((PdbRdc)pdbRdcVec.elementAt(i)).getPdb();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = dirCos(nToNHVec);
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    equ[i][0] = cosY * cosY - cosX * cosX;
	    equ[i][1] = cosZ * cosZ - cosX * cosX;
	    equ[i][2] = 2.0 * cosX * cosY;
	    equ[i][3] = 2.0 * cosX * cosZ;
	    equ[i][4] = 2.0 * cosY * cosZ;
	    b[i] = ((PdbRdc)pdbRdcVec.elementAt(i)).getRdc();
	}
	Matrix A = new Matrix(equ, m, n);
	SingularValueDecomposition SVD = A.svd();
	double [] singularvalues = SVD.getSingularValues();
	double[] singulars = new double[6];

	for (k=0; k<singularvalues.length; k++){
	    if (singularvalues[k] != 0.0)
		singulars[k] = 1 / singularvalues[k];
	    else return false;
	}
	double [][] ArrS = { {singulars[0], 0.0, 0.0, 0.0, 0.0},
			     {0.0, singulars[1], 0.0, 0.0, 0.0},
			     {0.0, 0.0, singulars[2], 0.0, 0.0},
			     {0.0, 0.0, 0.0, singulars[3], 0.0},
			     {0.0, 0.0, 0.0, 0.0, singulars[4]}};
	Matrix U = SVD.getU();
	Matrix V = SVD.getV();
	Matrix S  = SVD.getS();
	Matrix invS = new Matrix(ArrS, n, n);
	int errorCount = 0;
	/** comment out at your own risk */
	try {
	    check(A,SVD.getU().times(SVD.getS().times(SVD.getV().transpose())));
	    try_success("SingularValueDecomposition...","");
	} catch ( java.lang.RuntimeException e ) {
	    errorCount = try_failure(errorCount,"SingularValueDecomposition...",
				     "incorrect singular value decomposition calculation");
	}
	/** comment out at your own risk */

	double [] saupe = V.times(invS.times(U.transpose().times(b)));
	double Syy = saupe[0], Szz = saupe[1], Sxy = saupe[2], Sxz = saupe[3], Syz = saupe[4];
	double r = 0.0, rdcCal = 0.0;
	//back-computing RDCs	
	if (printResults)
	    System.out.println("ResidueNO,   back-computed RDC,   measured RDC,  difference");
	for (i = 0; i < pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i); 
	    no = pp.getResidueNo();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = dirCos(nToNHVec);
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcCal =  (cosY * cosY - cosX * cosX) * Syy
		       + (cosZ * cosZ - cosX * cosX) * Szz
		       + 2.0 * cosX * cosY * Sxy
		       + 2.0 * cosX * cosZ * Sxz
		       + 2.0 * cosY * cosZ * Syz;

	    index = Collections.binarySearch(rdcVec, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1)
		rdcExp = ((Dipolar)rdcVec.elementAt(index)).getRdc();
	    else
		rdcExp = rdcCal;
	    rdcArrN[i] = rdcCal;

	    if (printResults)
		System.out.println(pp.getResidueNo()+"   "+rdcCal+"   "+rdcExp+"   "+(rdcCal-rdcExp));
	    r += (rdcCal-rdcExp) * (rdcCal-rdcExp);
	}

	if (pdbRdcVec.size() < 2){
	    System.out.println("Error in bestFit method, possible reason: either Pdb or RDC file is not read properly");
	    System.exit(1);
	}
	if (printResults)
	    System.out.println("rmsd :"+ Math.sqrt( r / pdbRdcVec.size() ) );
	for (k=0; k<saupe.length; k++)
	    rmsdArr[k] = saupe[k];
	rmsdArr[5] = Math.sqrt(r / pdbRdcVec.size() );  //The last one
	return true;
    }

    /**
     * A method to generate an equation consisting all the different
     * internuclear vectors whose RDC are measured. IN our specific
     * case the N->NH and CA->HA RDCs are used
     * 
     * @param pdbVec the PDB ccordinates
     * @param rdcVec used for extracting the PDB coordiantes with the RDC data
     * @param atom1 the atom like N and H whose vector has RDC measured values
     * @param atom2 the atom like N and H whose vector has RDC measured values
     * @param rdcRatio the theoretical value of RDC strength compared with CH (which is set as 1.00)
     * Real experimental value  may differ due to Dynamics etc
     * 
     * @return A 2D array with six columns corresponding to the five terms of direction cosine and RDCs
     */  
    public double[][] equGenerate(final Vector pdbVec, final Vector rdcVec, String atom1, String atom2, 
				  double rdcRatio)
    {
    	if(atom2.equalsIgnoreCase("H"))
    		atom2="HN";
    	if(atom1.equalsIgnoreCase("H"))
    		atom1="HN";   	
		Pdb pp = new Pdb();
		Dipolar dd = new Dipolar();
		int i,j,k;
		int no;
		int index;
		String resid;
		Vector atomVec = new Vector();
		double [] nToNHVec = new double[3]; //NH vector
		double [] amide = new double[3];    // Amide Nitrogen Array
		double [] nh = new double[3];       // Amide Proton Array
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector pdbRdcVec = new Vector();
		Cartesian cc = new Cartesian();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
			pdbRdcVec.add(new PdbRdc((Pdb)pdbVec.elementAt(index), rdcExp)); 
		}
		int m = pdbRdcVec.size();   //number of rows

		int n = 6;  //Number of column: 5 for directional cosines and 1 for RDC values
		double [][] equ = new double[m][n];
		double [] b = new double[m];
		int N = 0;
	
		N = pdbRdcVec.size();

		for (i=0; i<N; i++)
		{
		    pp = ((PdbRdc)pdbRdcVec.elementAt(i)).getPdb();
		    atomVec = pp.getAtomVec();
		    
		    if (atom1.equalsIgnoreCase("CO") && atom2.equalsIgnoreCase("N"))
		    {
		    	for (j=0; j<atomVec.size(); j++)
			    {
					cc = (Cartesian)atomVec.elementAt(j);
					atom = cc.getAtom();
					if(atom.equalsIgnoreCase("H"))
						atom="HN";
					if (atom.equalsIgnoreCase(atom1))
					    amide = cc.getXYZ();		
			    }
		    	
		    	int resCurNo=pp.getResidueNo();
		    	index = Collections.binarySearch(pdbVec, new Pdb(resCurNo+1), new Pdb.PdbComparator());
		    	if(index <= -1)
		    	{
		    		System.out.println("Error in finding co-N RDC.");
		    		System.exit(1);
		    	}
		    	Pdb pp2=(Pdb)pdbVec.elementAt(index);
		    	atomVec = pp2.getAtomVec();
		    	for (j=0; j<atomVec.size(); j++)
			    {
					cc = (Cartesian)atomVec.elementAt(j);
					atom = cc.getAtom();
					if(atom.equalsIgnoreCase("H"))
						atom="HN";
					if (atom.equalsIgnoreCase(atom2))
						 nh = cc.getXYZ();		
			    }
		    }
		    else
		    {	    	
			    for (j=0; j<atomVec.size(); j++)
			    {
					cc = (Cartesian)atomVec.elementAt(j);
					atom = cc.getAtom();
					if(atom.equalsIgnoreCase("H"))
						atom="HN";
					if (atom.equalsIgnoreCase(atom1))
					    amide = cc.getXYZ();
					else if (atom.equalsIgnoreCase(atom2))
					    nh = cc.getXYZ();
			    }
		    }	    
		    
		    nToNHVec[0] = nh[0] - amide[0];
		    nToNHVec[1] = nh[1] - amide[1];
		    nToNHVec[2] = nh[2] - amide[2];
		    nToNHVec = dirCos(nToNHVec);
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    equ[i][0] = cosY * cosY - cosX * cosX;
		    equ[i][1] = cosZ * cosZ - cosX * cosX;
		    equ[i][2] = 2.0 * cosX * cosY;
		    equ[i][3] = 2.0 * cosX * cosZ;
		    equ[i][4] = 2.0 * cosY * cosZ;
		    equ[i][5] = ((PdbRdc)pdbRdcVec.elementAt(i)).getRdc() / rdcRatio; //All were converted to CH 
		}
		return equ;
    }

    /**
     * Best fit Two sets of RDC data from one medium such as NH and CH
     * please note that the rotation matrix returned by the
     * eigenValueDecomposition method is rather arbitrary so we need
     * to make sure that only the righ handedone are returned.
     * 
     * @param pdbVec Coordinates
     * @param nhRdc the measured NH rdcs
     * @param cahaRdc the measured CH rdcs
     * @param rdc1Cal for saving back-computed NH Rdcs including those of missing experimental RDCs
     * @param rdc2Cal for saving back-computed CH Rdcs including those of missing experimental RDCs
     * @param eigenValues for saving the computed {Syy, Szz, rmsd4NHRDC, rmsd4CHRDC
     * 
     * @return the rotation matrix rotating the original frame to one of four possible POFs
     */
    public Matrix  bestFit(final Vector<Pdb> pdbVec, final Vector nhRdc, final Vector<Dipolar> cahaRdc,  
			   double[] rdc1Cal, double[] rdc2Cal, double[] eigenValues){
	Pdb pp = new Pdb();
	Dipolar dd = new Dipolar();
	int i, j, k;
	int no;
	String resid;
	Vector atomVec = new Vector();
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Cartesian cc = new Cartesian();
	String atom = "";
	int index;		
	
	//The NH and CH RDC data: Thw two RDCs are concatenated together.
	double[][] nhArr   = equGenerate(pdbVec, nhRdc, "N", "H", Const.nhRatio);
	double[][] cahaArr = equGenerate(pdbVec, cahaRdc, "CA", "HA", Const.cahaRatio);
	int nhNO = nhArr.length;   //number of rows of the NH Rdcs
	int cahaNO = cahaArr.length;  
	Matrix nhMat = new Matrix(nhArr,   nhNO,  5);
	Matrix chMat = new Matrix(cahaArr, cahaNO, 5);
   	int m = nhNO + cahaNO;	//The total number of rows

	double [][] equ = new double[m][5];
	double []     b = new double[m];
	//Checking the RDC file has been read correctly
 	for(i=0; i<nhNO; i++){
	    for (j =0; j<5; j++)
		equ[i][j] = nhArr[i][j];
	    b[i] = nhArr[i][5];           //The last column has RDC values.
	}
	//For fitting one set of RDC data.
	for(i = nhNO; i< nhNO+cahaNO; i++){
	    for (j =0; j<5; j++)
		equ[i][j] = cahaArr[i-nhNO][j];
	    b[i] = cahaArr[i-nhNO][5];
	}
	int n = 5;
	Matrix A = new Matrix(equ, m, n);
	SingularValueDecomposition SVD = A.svd();
	double [] singularvalues = SVD.getSingularValues();
	double[] singulars = new double[6];
 	
	for (k=0; k<singularvalues.length; k++){
    	
	    if (singularvalues[k] != 0.0)
		singulars[k] = 1 / singularvalues[k];
	    else {
		System.out.println("Zero singluar values");
		System.exit(1);
	    }
	}
	double [][] ArrS = { {singulars[0], 0.0, 0.0, 0.0, 0.0},
			     {0.0, singulars[1], 0.0, 0.0, 0.0},
			     {0.0, 0.0, singulars[2], 0.0, 0.0},
			     {0.0, 0.0, 0.0, singulars[3], 0.0},
			     {0.0, 0.0, 0.0, 0.0, singulars[4]} };
	Matrix U = SVD.getU();
	Matrix V = SVD.getV();
	Matrix S = SVD.getS();
	Matrix invS = new Matrix(ArrS, 5, 5);
	int errorCount = 0;


/** comment out at your own risk */
	try {
	    check(A,SVD.getU().times(SVD.getS().times(SVD.getV().transpose())));
	    try_success("SingularValueDecomposition...","");
	} catch ( java.lang.RuntimeException e ) {
	    errorCount = try_failure(errorCount,"SingularValueDecomposition...",
				     "incorrect singular value decomposition calculation");
	}
/** comment out at your own risk */

	double [] saupe = V.times(invS.times(U.transpose().times(b)));
	double Syy = saupe[0], Szz = saupe[1], Sxy = saupe[2], Sxz = saupe[3], Syz = saupe[4];
	double[][] mArr = {{-Syy-Szz, Sxy, Sxz},
			   {Sxy,    Syy,   Syz},
			   {Sxz,    Syz,   Szz}};
	Matrix mm = new Matrix(mArr);
	
	///addded by Zeng
	System.out.println(" The following is the 3*3 matrix of alignment tensor ");
	System.out.println(" mm(1,1)= "+ (-Syy-Szz));
	System.out.println(" mm(1,2)= "+ Sxy);
	System.out.println(" mm(1,3)="+ Sxz);
	System.out.println(" mm(2,1)="+ Sxy);
	System.out.println(" mm(2,2)="+ Syy);
	System.out.println(" mm(2,3)="+ Syz);
	System.out.println(" mm(3,1)="+ Sxz);
	System.out.println(" mm(3,2)="+ Syz);
	System.out.println(" mm(3,3)="+ Szz);
	
	EigenvalueDecomposition eigs = new  EigenvalueDecomposition (mm, true);
	Matrix VV = eigs.getV();
	Matrix DD = eigs.getD();
  
	double	Sxx = DD.get(0,0);
	Syy = DD.get(1,1);
	Szz = DD.get(2,2);
	eigenValues[0] = Syy;
	eigenValues[1] = Szz;
	Vector<Pdb> pdbVecN = pp.newPdb(pdbVec, VV.transpose() );
	ModelRdc mr = new ModelRdc();
	Vector<PhiPsi> phiPsiVec1 = pp.PhiPsiTotalPdb(pdbVec);
	Vector<PhiPsi> phiPsiVec2 = pp.PhiPsiTotalPdb(pdbVecN);
	double eps = 1.0E-10;

//	added by zeng
	eigenValues[2]=0.0;
	eigenValues[3]=0.0;
	
	if (cahaRdc.size()>1)
	{
	
  	double chRms = BackCal(pdbVec, cahaRdc, "CA", "HA", VV.transpose(), Sxx, Syy, Szz, Const.cahaRatio, rdc2Cal);
	eigenValues[2] = chRms;
	
	}
	
	if (nhRdc.size()>1)
	{
	
 	double nhRms = BackCalNH(pdbVec, nhRdc, "N", "H",  VV.transpose(), Sxx, Syy, Szz, Const.nhRatio, rdc1Cal);
	eigenValues[3] = nhRms;
		
	}
	//Make sure what returned has the same handedness as the input PDB
	if (Math.abs( phiPsiVec1.elementAt(0).getPsi() - phiPsiVec2.elementAt(0).getPsi()) < eps){
 	   
	    return VV.transpose();
	}

 	return Const.mLeftHand.times(VV.transpose());

    }
 

    /**
     * Best fit Two sets of RDC data from one medium such as NH and CH
     * please note that the rotation matrix returned by the
     * eigenValueDecomposition method is rather arbitrary so we need
     * to make sure only the righ handedone are returned.
     * 
     * @param pdbVec Coordinates
     * @param nhRdc the measured NH rdcs
     * @param cahaRdc the measured CH rdcs
     * @param rdc1Cal for saving back-computed NH Rdcs including those of missing experimental RDCs
     * @param rdc2Cal for saving back-computed CH Rdcs including those of missing experimental RDCs
     * @param eigenValues for saving the computed {Syy, Szz, rmsd4NHRDC, rmsd4CHRDC
     * @param cacoRdc the caco rdc
     * @param conRdc the con rdc
     * 
     * @return the rotation matrix rotating the original frame to one of four possible POFs
     */
    public Matrix  bestFit4RDCs(final Vector pdbVec, final Vector nhRdc, final Vector cahaRdc, 
    		final Vector cacoRdc, final Vector conRdc, 
			   double[] rdc1Cal, double[] rdc2Cal, double[] eigenValues){
	Pdb pp = new Pdb();
	Dipolar dd = new Dipolar();
	int i, j, k;
	int no;
	String resid;
	Vector atomVec = new Vector();
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Cartesian cc = new Cartesian();
	String atom = "";
	int index;
	//The NH and CH RDC data: Thw two RDCs are concatenated together.
	double[][] nhArr   = equGenerate(pdbVec, nhRdc, "N", "H", Const.nhRatio);
	double[][] cahaArr = equGenerate(pdbVec, cahaRdc, "CA", "HA", Const.cahaRatio);
	double[][] cacoArr = equGenerate(pdbVec, cacoRdc, "CA", "CO", Const.cacoRatio);//added by zeng
	double[][] conArr = equGenerate(pdbVec, conRdc, "CO", "N", Const.conRatio);///added by zeng
	int nhNO = nhArr.length;   //number of rows of the NH Rdcs
	int cahaNO = cahaArr.length;  
	int cacoNO=cacoArr.length;//added by zeng
	int conNO=conArr.length;//added by zeng
	
	
	int m = nhNO + cahaNO+cacoNO+conNO;
	
	double [][] equ = new double[m][5];
	double []     b = new double[m];
	//Checking the RDC file has been read correctly
 	for(i=0; i<nhNO; i++){
	    for (j =0; j<5; j++)
		equ[i][j] = nhArr[i][j];
	    b[i] = nhArr[i][5];           //The last column has RDC values.
	}
	//For fitting one set of RDC data.
	for(i = nhNO; i< nhNO+cahaNO; i++){
	    for (j =0; j<5; j++)
		equ[i][j] = cahaArr[i-nhNO][j];
	    b[i] = cahaArr[i-nhNO][5];
	}
	for(i = nhNO+cahaNO; i< nhNO+cahaNO+cacoNO; i++)
	{
	    for (j =0; j<5; j++)
		equ[i][j] = cacoArr[i-nhNO-cahaNO][j];
	    b[i] = cacoArr[i-nhNO-cahaNO][5];
	}
	for(i = nhNO+cahaNO+cacoNO; i< nhNO+cahaNO+cacoNO+conNO; i++)
	{
	    for (j =0; j<5; j++)
		equ[i][j] = conArr[i-nhNO-cahaNO-cacoNO][j];
	    b[i] = conArr[i-nhNO-cahaNO-cacoNO][5];
	}
	
	
	int n = 5;
	Matrix A = new Matrix(equ, m, n);
	SingularValueDecomposition SVD = A.svd();
	double [] singularvalues = SVD.getSingularValues();
	double[] singulars = new double[6];

	for (k=0; k<singularvalues.length; k++){

	    if (singularvalues[k] != 0.0)
		singulars[k] = 1 / singularvalues[k];
	    else {
		System.out.println("Zero singluar values");
		System.exit(1);
	    }
	}
	double [][] ArrS = { {singulars[0], 0.0, 0.0, 0.0, 0.0},
			     {0.0, singulars[1], 0.0, 0.0, 0.0},
			     {0.0, 0.0, singulars[2], 0.0, 0.0},
			     {0.0, 0.0, 0.0, singulars[3], 0.0},
			     {0.0, 0.0, 0.0, 0.0, singulars[4]} };
	Matrix U = SVD.getU();
	Matrix V = SVD.getV();
	Matrix S = SVD.getS();
	Matrix invS = new Matrix(ArrS, 5, 5);
	int errorCount = 0;


/** comment out at your own risk */
	try {
	    check(A,SVD.getU().times(SVD.getS().times(SVD.getV().transpose())));
	    try_success("SingularValueDecomposition...","");
	} catch ( java.lang.RuntimeException e ) {
	    errorCount = try_failure(errorCount,"SingularValueDecomposition...",
				     "incorrect singular value decomposition calculation");
	}
/** comment out at your own risk */


	double [] saupe = V.times(invS.times(U.transpose().times(b)));
	double Syy = saupe[0], Szz = saupe[1], Sxy = saupe[2], Sxz = saupe[3], Syz = saupe[4];
	double[][] mArr = {{-Syy-Szz, Sxy, Sxz},
			   {Sxy,    Syy,   Syz},
			   {Sxz,    Syz,   Szz}};
	Matrix mm = new Matrix(mArr);
	EigenvalueDecomposition eigs = new  EigenvalueDecomposition (mm, true);
	Matrix VV = eigs.getV();
	Matrix DD = eigs.getD();
  	VV.print(5,12);
  	DD.print(5,12);
	double	Sxx = DD.get(0,0);
	Syy = DD.get(1,1);
	Szz = DD.get(2,2);
	eigenValues[0] = Syy;
	eigenValues[1] = Szz;
	Vector pdbVecN = pp.newPdb(pdbVec, VV.transpose() );
	ModelRdc mr = new ModelRdc();
	Vector phiPsiVec1 = pp.PhiPsiTotalPdb(pdbVec);
	Vector phiPsiVec2 = pp.PhiPsiTotalPdb(pdbVecN);
	double eps = 1.0E-10;

	System.out.println(" CH ");
  	double chRms = BackCal(pdbVec, cahaRdc, "CA", "HA", VV.transpose(), Sxx, Syy, Szz, Const.cahaRatio, rdc2Cal);
	eigenValues[2] = chRms;
	System.out.println(" CH "+chRms);
	System.out.println(" NH ");
 	double nhRms = BackCal(pdbVec, nhRdc, "N", "H",  VV.transpose(), Sxx, Syy, Szz, Const.nhRatio, rdc1Cal);
	eigenValues[3] = nhRms;
	System.out.println(" NH "+nhRms);	
	//Make sure what returned has the same handedness as the input PDB
	if (Math.abs( ((PhiPsi)phiPsiVec1.elementAt(0)).getPsi() - ((PhiPsi)phiPsiVec2.elementAt(0)).getPsi()) < eps){
 	    System.out.println(" The same ");
	    return VV.transpose();
	}
 	System.out.println(" Differ ");
 	return Const.mLeftHand.times(VV.transpose());

    }

    /**
     * For back-computing NH RDCs since the NH RDC of the fiest residue is NOT used in the DFS-based
     * refinement.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2  the atoms of which the internuclear vector has RDC values
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param squareRmsd the variable for computing overall Rmsd, added by zeng
     * @param size the size of experimental RDCs, added by zeng
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalNH(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc, double[] squareRmsd, int[] size){
	int i, j, no;
	int index = 0;
	Cartesian cc = new Cartesian();
	Vector<Cartesian> atomVec = new Vector<Cartesian>();
	double [] nToNHVec = new double[3];
	double [] amide = new double[3];
	double [] nh = new double[3];

	double r = 0.0, rdcCal = 0.0;
	Dipolar dd = new Dipolar();
	Pdb pp = new Pdb();
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
	String atom = "";
	for (i=0; i<rdcVec.size(); i++)
	{
	    dd = (Dipolar)rdcVec.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
		pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
	}
	for (i = 1; i < pdbRdcVec.size(); i++)
	{
	    pp = pdbRdcVec.elementAt(i).getPdb(); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
	    rdcExp = pdbRdcVec.elementAt(i).getRdc();
   	   // System.out.println(pp.getResidueNo()+"   "+rdcCal+"   "+rdcExp+"   "+(rdcCal-rdcExp));//////////////
	    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
	}
	if (pdbRdcVec.size() < 2){
	    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
	    System.exit(1);
	}
	//System.out.println("rmsd :"+ Math.sqrt(r / pdbRdcVec.size() ));
	//For back-computing RDCs of those residue with missing experimental RDCs
	for (i = 0; i < pdbVec.size() ; i++)
	{
	    pp = pdbVec.elementAt(i); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
	}
	squareRmsd[0]= r;
	size[0] =(pdbRdcVec.size() );
	
	return 	(Math.sqrt(r / (pdbRdcVec.size()  ) ));
    }
    
    /**
     * compute the RDC deviation for one residue.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2  the atoms of which the internuclear vector has RDC values
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param squareRmsd the variable for computing overall Rmsd, added by zeng
     * @param size the size of experimental RDCs, added by zeng
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalNHOne(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc, double[] squareRmsd, int[] size)
    {
    	if(atom1.equalsIgnoreCase("H"))
    		atom1="HN";
    	if(atom2.equalsIgnoreCase("H"))
    		atom2="HN";
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] amide = new double[3];
		double [] nh = new double[3];
	
		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
			pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if(atom.equalsIgnoreCase("H"))
					atom="HN";
				if (atom.equalsIgnoreCase(atom1))
				    amide = cc.getXYZ();
				else if (atom.equalsIgnoreCase(atom2))
				    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();
	   	   // System.out.println(pp.getResidueNo()+"   "+rdcCal+"   "+rdcExp+"   "+(rdcCal-rdcExp));//////////////
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}
		/*if (pdbRdcVec.size() < 2){
		    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
		    System.exit(1);
		}*/
		//System.out.println("rmsd :"+ Math.sqrt(r / pdbRdcVec.size() ));
		//For back-computing RDCs of those residue with missing experimental RDCs
		for (i = 0; i < pdbVec.size() ; i++)
		{
		    pp = pdbVec.elementAt(i); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals(atom1))
			    amide = cc.getXYZ();
			else if (atom.equals(atom2))
			    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		}
		squareRmsd[0]= r;
		size[0] =(pdbRdcVec.size() );
	
		return 	Math.sqrt(r);//(Math.sqrt(r / (pdbRdcVec.size()  ) ));
    }

    /**
     * For back-computing RDCs.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * @param squareRmsd the variable for computing overall Rmsd, added by zeng
     * @param size the size of experimental RDCs, added by zeng
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCal(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc, double[] squareRmsd, int[]  size){
	int i, j, no;
	int index = 0;
	Cartesian cc = new Cartesian();
	Vector<Cartesian> atomVec = new Vector<Cartesian>();
	double [] nToNHVec = new double[3];
	double [] amide = new double[3];
	double [] nh = new double[3];

	double r = 0.0, rdcCal = 0.0;
	Dipolar dd = new Dipolar();
	Pdb pp = new Pdb();
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
	String atom = "";
	for (i=0; i<rdcVec.size(); i++)
	{
	    dd = (Dipolar)rdcVec.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
		pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
	}
	for (i = 0; i < pdbRdcVec.size(); i++){
	    pp = pdbRdcVec.elementAt(i).getPdb(); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
	    rdcExp = pdbRdcVec.elementAt(i).getRdc();
   	   
	    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
	}
	if (pdbRdcVec.size() < 2){
	    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
	    System.exit(1);
	}
	
	//For back-computing RDCs of those residue with missing experimental RDCs
	for (i = 0; i < pdbVec.size() ; i++){
	    pp = pdbVec.elementAt(i); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
// 	    System.out.println(pp.getResidueNo()+"   "+rdcCal+"    "+1.0+"   "+0.0);
	}
	
	squareRmsd[0]=r;
	size[0]= pdbRdcVec.size();
	return 	(Math.sqrt(r / pdbRdcVec.size() ));
    }
    
    /**
     * compute rdc deviation for one residue
     * 
     * For back-computing RDCs.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * @param squareRmsd the variable for computing overall Rmsd, added by zeng
     * @param size the size of experimental RDCs, added by zeng
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalOne(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc, double[] squareRmsd, int[]  size)
    {
    	if(atom1.equalsIgnoreCase("H"))
    		atom1="HN";
    	if(atom2.equalsIgnoreCase("H"))
    		atom2="HN";
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] amide = new double[3];
		double [] nh = new double[3];
	
		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
			pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals(atom1))
			    amide = cc.getXYZ();
			else if (atom.equals(atom2))
			    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();
	   	   
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}
		/*if (pdbRdcVec.size() < 2){
		    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
		    System.exit(1);
		}*/
		
		//For back-computing RDCs of those residue with missing experimental RDCs
		for (i = 0; i < pdbVec.size() ; i++){
		    pp = pdbVec.elementAt(i); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals(atom1))
			    amide = cc.getXYZ();
			else if (atom.equals(atom2))
			    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
	// 	    System.out.println(pp.getResidueNo()+"   "+rdcCal+"    "+1.0+"   "+0.0);
		}
	
		squareRmsd[0]=r;
		size[0]= pdbRdcVec.size();
		return Math.sqrt(r);//	(Math.sqrt(r / pdbRdcVec.size() ));
    }
  
    
    
    
    
    
    /**
     * compute rdc rmsd for only one pdb.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalOnePdb(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio)
    {
    	if(atom1.equalsIgnoreCase("H"))
    		atom1="HN";
    	if(atom2.equalsIgnoreCase("H"))
    		atom2="HN";
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] amide = new double[3];
		double [] nh = new double[3];
	
		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
			pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = atomVec.elementAt(j);
			atom = cc.getAtom();
			if(atom.equalsIgnoreCase("H"))
				atom="HN";
			if (atom.equalsIgnoreCase(atom1))
			    amide = cc.getXYZ();
			else if (atom.equalsIgnoreCase(atom2))
			    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();
	   	    
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}
	
	    return 	(Math.sqrt(r / pdbRdcVec.size() ));
    }

    
    
    /**
     * For back-computing RDCs.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * @param squareRmsd the variable for computing overall Rmsd, added by zeng
     * @param size the size of experimental RDCs, added by zeng
     * @param isPrint whether the RDC deviation at each residue is printed
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalCACO(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc, double[] squareRmsd, 
			    int[]  size, boolean isPrint)
    {
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] amide = new double[3];
		double [] nh = new double[3];

		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());/////
		    if (index > -1)
			pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom1))
				    amide = cc.getXYZ();
				else if (atom.equalsIgnoreCase(atom2))
				    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();
		    if(isPrint)
		    	System.out.println(pp.getResidueNo()+"   "+rdcCal+"   "+rdcExp+"   "+(rdcCal-rdcExp));///////
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}
		if (pdbRdcVec.size() < 2)
		{
		    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
		    System.exit(1);
		}
	
		//For back-computing RDCs of those residue with missing experimental RDCs
		for (i = 0; i < pdbVec.size()-1 ; i++)
		{
		    pp = pdbVec.elementAt(i); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom1))
				    amide = cc.getXYZ();
				else if (atom.equalsIgnoreCase(atom2))
				    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		     	    //System.out.println(pp.getResidueNo()+"   "+ rdcBc[i]+"    "+1.0+"   "+0.0);
		}
	
		squareRmsd[0]=r;
		size[0]= pdbRdcVec.size();
		return 	(Math.sqrt(r / pdbRdcVec.size() ));
    }
    
    /**
     * compute rdc rmsd for only one pdb.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalCACOOnePdb(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio)
    {
    	
    	if( (pdbVec.size()<1) || (rdcVec.size()<1))
    		return 0.0;
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] amide = new double[3];
		double [] nh = new double[3];

		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());/////
		    if (index > -1)
			pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom1))
				    amide = cc.getXYZ();
				else if (atom.equalsIgnoreCase(atom2))
				    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();		  
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}		
		return 	(Math.sqrt(r / pdbRdcVec.size() ));
    }

    
    
    
    
    /**
     * For back-computing RDCs
     * from Co at residue i to N at residue i+1.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * @param squareRmsd the variable for computing overall Rmsd, added by zeng
     * @param size the size of experimental RDCs, added by zeng
     * @param isPrint whether the RDC deviation is printed
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalCON(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc, double[] squareRmsd,
			    int[]  size, boolean isPrint)
    {
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] co = new double[3];
		double [] n_next = new double[3];

		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
		    	pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom1))
				    co = cc.getXYZ();
			
		    }
		    
		    no=pp.getResidueNo();
		    int no_next=no+1;
		    index = Collections.binarySearch(pdbVec, new Pdb(no_next), new Pdb.PdbComparator());
		    if (index > -1)
		    	pp=pdbVec.elementAt(index);
		    else
		    {
		    	
		    	continue;
			 
		    }
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom2))//"N"
				    n_next = cc.getXYZ();
		    }
		    
		    nToNHVec = internuclearVec(co, n_next);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();
		    if (isPrint)
		    	System.out.println(pp.getResidueNo()+"   "+rdcCal+"   "+rdcExp+"   "+(rdcCal-rdcExp));///////
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}
		if (pdbRdcVec.size() < 2)
		{
		    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
		    System.exit(1);
		}
		
		//For back-computing RDCs of those residue with missing experimental RDCs
		for (i = 0; i < pdbVec.size() ; i++)
		{
		    pp = pdbVec.elementAt(i); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom1))
				    co = cc.getXYZ();
				//else if (atom.equalsIgnoreCase(atom2))
				  //  nh = cc.getXYZ();
		    }
		    no=pp.getResidueNo();
		    int no_next=no+1;
		    index = Collections.binarySearch(pdbVec, new Pdb(no_next), new Pdb.PdbComparator());
		    if (index > -1)
		    	pp=pdbVec.elementAt(index);
		    else
		    {
		    	
		    	continue;
			    //System.exit(1);
		    }
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom2))//"N"
				    n_next = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(co, n_next);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    // 	    System.out.println(pp.getResidueNo()+"   "+rdcCal+"    "+1.0+"   "+0.0);
		}
	
		squareRmsd[0]=r;
		size[0]= pdbRdcVec.size();
		return 	(Math.sqrt(r / pdbRdcVec.size() ));
    }
    
    /**
     * compute rdc rmsd for only one pdb
     * from Co at residue i to N at residue i+1.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalCONOnePdb(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio)
    {
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] co = new double[3];
		double [] n_next = new double[3];

		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
		    	pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom1))
				    co = cc.getXYZ();
			
		    }
		    
		    no=pp.getResidueNo();
		    int no_next=no+1;
		    index = Collections.binarySearch(pdbVec, new Pdb(no_next), new Pdb.PdbComparator());
		    if (index > -1)
		    	pp=pdbVec.elementAt(index);
		    else
		    {
		    	
		    	continue;
			   
		    }
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equalsIgnoreCase(atom2))//"N"
				    n_next = cc.getXYZ();
		    }
		    
		    nToNHVec = internuclearVec(co, n_next);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();		  
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}		
		return 	r;//(Math.sqrt(r / pdbRdcVec.size() ));
    }



    /**
     * For back-computing RDCs.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atoms of which the internuclear vector has RDC values
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCal(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc){
	int i, j, no;
	int index = 0;
	Cartesian cc = new Cartesian();
	Vector<Cartesian> atomVec = new Vector<Cartesian>();
	double [] nToNHVec = new double[3];
	double [] amide = new double[3];
	double [] nh = new double[3];

	double r = 0.0, rdcCal = 0.0;
	Dipolar dd = new Dipolar();
	Pdb pp = new Pdb();
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
	String atom = "";
	for (i=0; i<rdcVec.size(); i++){
	    dd = (Dipolar)rdcVec.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
		pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
	}
	for (i = 0; i < pdbRdcVec.size(); i++){
	    pp = pdbRdcVec.elementAt(i).getPdb(); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
	    rdcExp = pdbRdcVec.elementAt(i).getRdc();
    	
	    System.out.println(pp.getResidueNo()+" CH  "+rdcCal+"   "+rdcExp+"   "+(rdcCal-rdcExp));
	    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
	}
	if (pdbRdcVec.size() < 2){
	    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
	    System.exit(1);
	}
	
	//For back-computing RDCs of those residue with missing experimental RDCs
	for (i = 0; i < pdbVec.size() ; i++){
	    pp = pdbVec.elementAt(i); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 

	}
	return 	(Math.sqrt(r / pdbRdcVec.size() ));
    }

    /**
     * For back-computing NH RDCs since the NH RDC of the fiest residue is NOT used in the DFS-based
     * refinement.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2  the atoms of which the internuclear vector has RDC values
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param Szz the three diagonalized Saupe elements
     * @param rdcRatio the relative strength of RDCs
     * @param rdcBc an array for returning the back-computed RDCs
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double BackCalNH(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz, double rdcRatio, double[] rdcBc)
    {
    	if(atom2.equalsIgnoreCase("H"))
    		atom2="HN";
		int i, j, no;
		int index = 0;
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		double [] nToNHVec = new double[3];
		double [] amide = new double[3];
		double [] nh = new double[3];
	
		double r = 0.0, rdcCal = 0.0;
		Dipolar dd = new Dipolar();
		Pdb pp = new Pdb();
		double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbRdcVec = new Vector<PdbRdc>();
		String atom = "";
		for (i=0; i<rdcVec.size(); i++)
		{
		    dd = (Dipolar)rdcVec.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
			pdbRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		for (i = 0; i < pdbRdcVec.size(); i++)
		{
		    pp = pdbRdcVec.elementAt(i).getPdb(); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if(atom.equalsIgnoreCase("H"))
					atom="HN";
				if (atom.equals(atom1))
				    amide = cc.getXYZ();
				else if (atom.equals(atom2))
				    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcCal = (cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		    rdcExp = pdbRdcVec.elementAt(i).getRdc();
	   	    System.out.println(pp.getResidueNo()+" NH  "+rdcCal+"   "+rdcExp+"   "+(rdcCal-rdcExp));
		    r += (rdcCal - rdcExp) * (rdcCal - rdcExp);
		}
		if (pdbRdcVec.size() < 2)
		{
		    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
		    System.exit(1);
		}
	
		//For back-computing RDCs of those residue with missing experimental RDCs
		for (i = 0; i < pdbVec.size() ; i++)
		{
		    pp = pdbVec.elementAt(i); 
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++)
		    {
				cc = atomVec.elementAt(j);
				atom = cc.getAtom();
				if(atom.equalsIgnoreCase("H"))
					atom="HN";
				if (atom.equals(atom1))
				    amide = cc.getXYZ();
				else if (atom.equals(atom2))
				    nh = cc.getXYZ();
		    }
		    nToNHVec = internuclearVec(amide, nh);
		    nToNHVec = mm.times(dirCos(nToNHVec));
		    cosX = nToNHVec[0];
		    cosY = nToNHVec[1];
		    cosZ = nToNHVec[2];
		    rdcBc[i] = (cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz) * rdcRatio; 
		}
		return 	(Math.sqrt(r / (pdbRdcVec.size() ) ));
    }


    /**
     * For back-computing RDCs.
     * 
     * @param pdbVec the pdb coordinates
     * @param rdcVec the meausred RDC values
     * @param atom1 the atoms of which the internuclear vector has RDC values
     * @param atom2 the atom2
     * @param mm the matrix used to rotate the original PDB frame to one of POF
     * @param Sxx the three diagonalized Saupe elements
     * @param Syy the syy
     * @param Szz the szz
     * 
     * @return RMSDs between the computed RDCs and the experimental RDCs.
     */
    public double[] BackCalRdcs(final Vector pdbVec, final Vector rdcVec, String atom1, String atom2, 
			    Matrix mm, double Sxx, double Syy, double Szz){
	int i, j, no;
	int index = 0;
	Cartesian cc = new Cartesian();
	Vector atomVec = new Vector();
	double [] nToNHVec = new double[3];
	double [] amide = new double[3];
	double [] nh = new double[3];
	double [] rdcArr = new double[pdbVec.size()];
	double r = 0.0, rdcCal = 0.0;
	Dipolar dd = new Dipolar();
	Pdb pp = new Pdb();
	double cosX = 0.0, cosY = 0.0, cosZ = 0.0;
	double rdcExp = 0.0;
	Vector pdbRdcVec = new Vector();
	String atom = "";
	for (i=0; i<rdcVec.size(); i++){
	    dd = (Dipolar)rdcVec.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
		pdbRdcVec.add(new PdbRdc((Pdb)pdbVec.elementAt(index), rdcExp)); 
	}
	for (i = 0; i < pdbRdcVec.size(); i++){
	    pp = ((PdbRdc)pdbRdcVec.elementAt(i)).getPdb(); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcCal = cosX * cosX * Sxx+ cosY * cosY * Syy + cosZ * cosZ * Szz; 
	    rdcExp = ((PdbRdc)pdbRdcVec.elementAt(i)).getRdc();

	    r += (rdcCal-rdcExp) * (rdcCal-rdcExp);
	}
	if (pdbRdcVec.size() < 2){
	    System.out.println("Error in file Size 1401: "+pdbRdcVec.size());
	    System.exit(1);
	}
	
	//For back-computing RDCs of those residue with missing experimental RDCs
	for (i = 0; i < pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals(atom1))
		    amide = cc.getXYZ();
		else if (atom.equals(atom2))
		    nh = cc.getXYZ();
	    }
	    nToNHVec = internuclearVec(amide, nh);
	    nToNHVec = mm.times(dirCos(nToNHVec));
	    cosX = nToNHVec[0];
	    cosY = nToNHVec[1];
	    cosZ = nToNHVec[2];
	    rdcCal = cosX * cosX * Sxx + cosY * cosY * Syy + cosZ * cosZ * Szz; 

	    rdcArr[i] = rdcCal;
	}

	return 	rdcArr;
    }
    
    /**
     * print out all pdb orientations in which rdc rmsd is smaler than some threshold, in grid search.
     * 
     * @param pdbVec coordinate vector
     * @param rdcVec1 nh rdc vector
     * @param rdcVec2 ch rdc vector
     * @param resolution resolution in the grid search
     * @param RdcRmsdThreshold rdc rmsd threshold
     * @param rmsds return rdc rmsd
     * return the best rotation matrix in terms of rdc rmsd
     * @param Syy the syy
     * @param Szz the szz
     * @param debug the debug
     * 
     * @return the matrix
     */
    public Matrix eulerFitPrintEnsemble(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec1, final Vector<Dipolar> rdcVec2, 
			   double Syy, double Szz, double[] rmsds, double resolution, boolean debug,double RdcRmsdThreshold)
    {
	double alpha = 0.0, beta = 0.0, gamma = 0.0;
	double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
	int i, j, k, no, index = 0;
	final int N = rdcVec2.size();  //the number of RDC data
	Vector nhRdc= rdcVec1;	
	Vector	cahaRdc= rdcVec2;
	double Sxx=-Syy-Szz;
	double[] sRmsdCH= new double[1];
	int[] sizeCH=new int[1];
	double[] sRmsdNH=new double[1];
	int[] sizeNH=new int[1];
		
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	Dipolar dd = (Dipolar)rdcVec1.elementAt(0);
	no = dd.getResidueNo();      //The 1st residues of fragment	
	
	
	
	//added by zeng	
	double rdcExp = 0.0;
	Vector<PdbRdc> pdbNHRdcVec = new Vector<PdbRdc>();
	Vector<PdbRdc> pdbCHRdcVec = new Vector<PdbRdc>();
	String atom = "", resid = "";
	//String atom = "";
	for (i=0; i<rdcVec1.size(); i++)
	{
	    dd = (Dipolar)rdcVec1.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
	    	pdbNHRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
	}
		
	for (i=0; i<rdcVec2.size(); i++)
	{
	    dd = (Dipolar)rdcVec2.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
	    	pdbCHRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
	}
	
	double [] rdcArr1 = new double[N]; 
	double [] rdcArr2 = new double[N]; 
	double [] rdcArr1S =new double[N]; 
	double [] rdcArr2S =new double[N]; 

	double[] rdc1Cal  = new double[pdbVec.size() ];
	double[] rdc2Cal  = new double[pdbVec.size() ];

	double [] chArr = new double[3*N]; 
	double [] nhArr = new double[3*N]; 

	Matrix A = new Matrix(3,3);
	Cartesian cc = new Cartesian();
	Vector<Cartesian> atomVec = new Vector<Cartesian>();
	Pdb pp = new Pdb();
	double nhRms=0;
	double chRms=0;

	double rdc1Rms=0;
	
	double [] amide = new double[3];
	double [] nh = new double[3];
	double [] ca = new double[3];
	double [] ch = new double[3];
	double [] ha = new double[3];
	double [] nToNHVec = new double[3];
	double [] caToHAVec = new double[3];
	double [][] mat = new double[3][3];
	double r = 0.0, r1 = 0.0, r2 = 0.0, r1s = 0.0, r2s = 0.0; 
	double rT = 10000.0,  rT1 = 10000.0,  rT2 = 10000.0;
	final double eps = 1.0E-10; //for finding all four possible sets of Euler angles.
	for (i=0; i<N; i++)
	{        //extract the NH vectors based on the available RDC values.
	    pp = pdbVec.elementAt(i); 
	    no = pp.getResidueNo();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++)
	    {
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide = cc.getXYZ();
		else if (atom.equals("H"))
		    nh = cc.getXYZ();
		else if (atom.equals("CA"))
		    ca = cc.getXYZ();
		else if (atom.equals("HA"))
		    ha = cc.getXYZ();
	    }
	    nhArr[3 * i]     = (nh[0] - amide[0]) / Const.dN2H;
	    nhArr[3 * i + 1] = (nh[1] - amide[1]) / Const.dN2H;
	    nhArr[3 * i + 2] = (nh[2] - amide[2]) / Const.dN2H;
	    //N(i)->CA(i) vector
	    chArr[3 * i]      = (ha[0] - ca[0]) / Const.dCA2HA;
	    chArr[3 * i + 1]  = (ha[1] - ca[1]) / Const.dCA2HA;
	    chArr[3 * i + 2]  = (ha[2] - ca[2]) / Const.dCA2HA;
	}
	
	String userDir = System.getProperty("user.dir");////
	String src=userDir+"/inputFiles/";
	
	String H3fileName=src+"H3New.pdb";
	Vector vecH3=pp.readPdb(H3fileName);
	Vector vecH3BB=pp.OutputBackbone(vecH3);
	
	
	String fileName =src+ "H5AllGoodRdcRMSD.pdb";
	int counter=0;
	int nTemp=0;
	Vector vecEnsemblePdbs=new Vector();
	
	try{
		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
		
	//Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360) 
	for (i = 0; i < (180 / resolution); i++)
	{		
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (180 / resolution ); k++)
		{
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
		    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
		    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
		    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
		    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
		    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
		    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
		    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
		    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
		    mat[2][2] = Math.cos(beta);
		    
		    Matrix tempA;
		    A = new Matrix(mat);
		  
		    r = 0.0;
		    r1 = 0.0; 
		    r2 = 0.0;
		    nhRms=0.0;
		    chRms=0.0;		    

		    nhRms = BackCalNH(pdbVec, nhRdc, "N", "H",  A, Sxx, Syy, Szz, Const.nhRatio, rdc1Cal,sRmsdNH, sizeNH);
			r1=nhRms;
		    
			chRms = BackCal(pdbVec, cahaRdc, "CA", "HA", A, Sxx, Syy, Szz, Const.cahaRatio, rdc2Cal,sRmsdCH, sizeCH );
			r2 = chRms; 	    
		    
		    r = r1 + r2;
		  
		    if((r1<1.0) && (r2<1.0))
		    {
		    	
		    	Matrix mm_temp = A.eulerMat(alpha, beta, gamma);	
		    	Vector vecHelixNew=new Vector();
		    	vecHelixNew = pp.newPdb(pdbVec, mm_temp);//be careful here... almost the same, so we chose anyone
				
		    	//clustering
		    	boolean isInPreClusters=pp.isInPdbClusters(vecEnsemblePdbs,vecHelixNew,0.8);///////////
		    	if (isInPreClusters)
		    		continue;
		    	System.out.println("MODEL"+i+j+k);
		    	//pp.print(ppVecT);
		    	out.println("MODEL"+i+" "+ j+ " "+k);///to file
		    	out.println("REMARK  Euler angles: " + " alpha="+alpha+ " beta=" +beta+ " gamma="+gamma);
		    	out.println("REMARK  overall (ch and nh) RDC RMSD:"+ "  OverallRMSD=" + r);///to file
        		out.println("REMARK  HN RDC RMSD:"+ "  HnRMSD =" + nhRms);///to file
        		out.println("REMARK  CaHa RDC RMSD:"+ "   CaHa RMSD=" + chRms);///to file		    	
		    	
        		Vector vecHelixH35=new Vector();
        		
        		vecHelixH35.addAll(vecH3BB);
        		vecHelixH35.addAll(vecHelixNew);
		    	pp.printToFile(vecHelixH35,fileName, out);
		    	
		    	System.out.println("TER");
	         	System.out.println("END");   
	         	
	            out.println("TER");
	         	out.println("ENDMDL");  
	         	vecEnsemblePdbs.add(vecHelixNew);
		    	
		    }//if (r< RdcRmsdThreshold)
		    
		    if( r <= rT || Math.abs(r - rT) < eps) 
		    { 
		    	//print all the four possible rotation
		       	alphaS = alpha;
				betaS = beta;
				gammaS = gamma;
				rT = r;
				r1s = r1;
				r2s = r2;
				if(debug)
				{
				    System.arraycopy(rdc1Cal, 0, rdcArr1S, 0, N); //save the back computed RDC values
				    System.arraycopy(rdc2Cal, 0, rdcArr2S, 0, N);
				    System.out.println(" rT = "+rT+"  "+r1+"  "+r2);
				    rdc1Rms = r1;
				}
		    }//if( r <= rT || Math.abs(r - rT) < eps) 
		}
	    }
	}//for (i = 0; i < (180 / resolution); i++)

	out.println("END");  
	out.close();
	}catch (FileNotFoundException e)
	{
		System.out.println("File not found: " + fileName);
	}catch (IOException e)
	{
	   System.out.println("IOException: the stack trace is:");
	   e.printStackTrace();
	}
			
	if (debug)
	{
	    System.out.println(" Euler "+ alphaS + "  " + betaS + "  " + gammaS);
	    System.out.println();
	    System.out.println("RdcNH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++)
	    {
	    //	Dipolar ddnh=(Dipolar)nhRdc.elementAt(i);
	    	//System.out.println("back-computed rdc (NH) "+rdcArr1S[i]+ " exprimental rdc (NH) "+ddnh.getRdc());
	    }
	    System.out.println("RMSD = "+ r1s);
	    System.out.println();
	    System.out.println("RdcCH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++)
	    {
	    //	Dipolar ddch=(Dipolar)cahaRdc.elementAt(i);
	    //	System.out.println("back-computed rdc (NH) "+rdcArr2S[i]+" exprimental rdc (NH) "+ddch.getRdc());
	    }
	    System.out.println("RMSD = "+ r2s);
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	Matrix mm = A.eulerMat(alphaS, betaS, gammaS);
	return 	mm;
}
    
    /**
     * Note: Lincong's has a serious bug. it is fixed in this version
     * Compute three Euler angles using two sets of RDC data and the corresponding Saupe elements
     * Basic steps:
     * extract the NH vectors with the corresponding RDC data as an array (for efficiency)
     * For all the values of three Euler angles first compute the new NH directions after the rotation
     * then compute the RDCs and RMSD from the new NH directions.
     * 
     * @param pdbVec a vector of Pdb object corresponding to the structural fragment.
     * @param rdcVec1 a vector of NH RDC values
     * @param rdcVec2 a vector of CH RDC values
     * @param Szz the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param rmsds   for returning the RDC RMSDs in the order of RDC for medium 1 and medium 2
     * @param resolution the resolution in the grid-search
     * @param debug   if true, it will print the progress of the fitting
     * @param rdcVecCaCo the rdc vec ca co
     * @param rdcVecCoN the rdc vec co n
     * 
     * @return the rotation matrix which minimizing the score function (RDCs)
     * We only search through 1/4 of the region to obtain one possible orientation,
     * other three can be computed much faster by multiplying by one of the three constant
     * matrices.
     */
    public Matrix eulerFit4RDCs(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec1, final Vector<Dipolar> rdcVec2, 
    		final Vector<Dipolar> rdcVecCaCo, final Vector<Dipolar> rdcVecCoN,
    		double Syy, double Szz, double[] rmsds, double resolution, boolean debug)
    {
		double alpha = 0.0, beta = 0.0, gamma = 0.0;
		double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
		int i, j, k, no, index = 0;
		final int N = rdcVec2.size();  //the number of RDC data
		Vector nhRdc= rdcVec1;	
		Vector	cahaRdc= rdcVec2;
		double Sxx=-Syy-Szz;
		double[] sRmsdCH= new double[1];
		int[] sizeCH=new int[1];
		double[] sRmsdNH=new double[1];
		int[] sizeNH=new int[1];
		double[] sRmsdCaCo=new double[1];
		double[] sRmsdCoN=new double[1];
		int[] sizeCaCo=new int[1];
		int[] sizeCoN=new int[1];
		
		double [] rdcExp1 = new double[N]; //store the experimental values in the array
		double [] rdcExp2 = new double[N];
		Dipolar dd = (Dipolar)rdcVec1.elementAt(0);
		no = dd.getResidueNo();      //The 1st residues of fragment
		
		//added by zeng	
		double rdcExp = 0.0;
		Vector<PdbRdc> pdbNHRdcVec = new Vector<PdbRdc>();
		String atom = "", resid = "";
		//String atom = "";
		for (i=0; i<rdcVec1.size(); i++)
		{
		    dd = (Dipolar)rdcVec1.elementAt(i); 
		    rdcExp = dd.getRdc();
		    no = dd.getResidueNo();
		    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1)
		    	pdbNHRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
		}
		
		
		double [] rdcArr1 = new double[N]; 
		double [] rdcArr2 = new double[N]; 
		double [] rdcArr1S =new double[N]; 
		double [] rdcArr2S =new double[N]; 

		double[] rdc1Cal  = new double[pdbVec.size() ];
		double[] rdc2Cal  = new double[pdbVec.size() ];
		double[] rdcCoNCal  = new double[pdbVec.size() ];
		double[] rdcCaCoCal  = new double[pdbVec.size() ];
		
		double [] chArr = new double[3*N]; 
		double [] nhArr = new double[3*N]; 
	
		Matrix A = new Matrix(3,3);
		Cartesian cc = new Cartesian();
		Vector<Cartesian> atomVec = new Vector<Cartesian>();
		Pdb pp = new Pdb();
		double nhRms=0;
		double chRms=0;
	
		double rdc1Rms=0;
		
		double [] amide = new double[3];
		double [] nh = new double[3];
		double [] ca = new double[3];
		double [] ch = new double[3];
		double [] ha = new double[3];
		double [] nToNHVec = new double[3];
		double [] caToHAVec = new double[3];
		double [][] mat = new double[3][3];
		double r = 0.0, r1 = 0.0, r2 = 0.0, r1s = 0.0, r2s = 0.0; 
		double rs_caco=0.0, rs_con=0.0;
		double r_caco=0.0,r_con=0.0;
		double rT = 10000.0,  rT1 = 10000.0,  rT2 = 10000.0;
		final double eps = 1.0E-10; //for finding all four possible sets of Euler angles.
		for (i=0; i<N; i++)
		{        //extract the NH vectors based on the available RDC values.
		    pp = pdbVec.elementAt(i); 
		    no = pp.getResidueNo();
		    atomVec = pp.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("N"))
			    amide = cc.getXYZ();
			else if (atom.equals("H"))
			    nh = cc.getXYZ();
			else if (atom.equals("CA"))
			    ca = cc.getXYZ();
			else if (atom.equals("HA"))
			    ha = cc.getXYZ();
	    }
	    nhArr[3 * i]     = (nh[0] - amide[0]) / Const.dN2H;
	    nhArr[3 * i + 1] = (nh[1] - amide[1]) / Const.dN2H;
	    nhArr[3 * i + 2] = (nh[2] - amide[2]) / Const.dN2H;
	    //N(i)->CA(i) vector
	    chArr[3 * i]      = (ha[0] - ca[0]) / Const.dCA2HA;
	    chArr[3 * i + 1]  = (ha[1] - ca[1]) / Const.dCA2HA;
	    chArr[3 * i + 2]  = (ha[2] - ca[2]) / Const.dCA2HA;
	}
	//Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360) 
	for (i = 0; i < (180 / resolution); i++)
	{		
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (180 / resolution ); k++)
		{
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
		    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
		    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
		    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
		    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
		    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
		    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
		    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
		    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
		    mat[2][2] = Math.cos(beta);
		    
		    Matrix tempA;
		    A = new Matrix(mat);
		    /*if (i==0 && j==0 && k==0)
		    {
		    	System.out.println("here we print out A");
		    	A.print(5,12);
		    }*/
		    r = 0.0;
		    r1 = 0.0; 
		    r2 = 0.0;
		    nhRms=0.0;
		    chRms=0.0;
		    r_caco=0.0;
		    r_con=0.0;
		    
		   		    
		    nhRms = BackCalNH(pdbVec, nhRdc, "N", "H",  A, Sxx, Syy, Szz, Const.nhRatio, rdc1Cal,sRmsdNH, sizeNH);
			r1=nhRms;
		    
			chRms = BackCal(pdbVec, cahaRdc, "CA", "HA", A, Sxx, Syy, Szz, Const.cahaRatio, rdc2Cal,sRmsdCH, sizeCH );
			r2 = chRms; 
		    r_caco=BackCalCACO(pdbVec, nhRdc, "CA", "C",  A, Sxx, Syy, Szz, Const.cacoRatio, rdcCaCoCal,sRmsdCaCo, sizeCaCo,false);
		    r_con=BackCalCON(pdbVec, nhRdc, "C", "N",  A, Sxx, Syy, Szz, Const.conRatio, rdcCoNCal,sRmsdCoN, sizeCoN,false);
			   
		    
		    
		    
//		    r = Math.sqrt(r1 / N ) +  Math.sqrt(r2 / N );
		    r = r1 + r2 + Const.wtCaCo*r_caco + Const.wtCoN*r_con;
		    if( r <= rT || Math.abs(r - rT) < eps) {  //print all the four possible rotation
			alphaS = alpha;
			betaS = beta;
			gammaS = gamma;
			rT = r;
			r1s = r1;
			r2s = r2;
			rs_caco=r_caco;
			rs_con=r_con;
			if(debug)
			{
			    System.arraycopy(rdcArr1, 0, rdcArr1S, 0, N); //save the back computed RDC values
			    System.arraycopy(rdcArr2, 0, rdcArr2S, 0, N);
			    System.out.println(" rT = "+rT+"  "+r1+"  "+r2);
			    rdc1Rms = r1;
			}
		    }
		}
	    }
	}
	if (debug){
	    System.out.println(" Euler "+ alphaS + "  " + betaS + "  " + gammaS);
	    System.out.println();
	    System.out.println("RdcNH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr1S[i]+"   "+rdcExp1[i]);
	    }
	    System.out.println("RMSD = "+ r1s);
	    System.out.println();
	    System.out.println("RdcCH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr2S[i]+"   "+rdcExp2[i]);
	    }
	    System.out.println("RMSD = "+ r2s);
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	rmsds[2]=rs_caco;
	rmsds[3]=rs_con;
	Matrix mm = A.eulerMat(alphaS, betaS, gammaS);
	return 	mm;
 }
    

    /**
     * Note: Lincong's has a serious bug. it is fixed in this version
     * Compute three Euler angles using two sets of RDC data and the corresponding Saupe elements
     * Basic steps:
     * extract the NH vectors with the corresponding RDC data as an array (for efficiency)
     * For all the values of three Euler angles first compute the new NH directions after the rotation
     * then compute the RDCs and RMSD from the new NH directions.
     * 
     * @param pdbVec a vector of Pdb object corresponding to the structural fragment.
     * @param rdcVec1 a vector of NH RDC values
     * @param rdcVec2 a vector of CH RDC values
     * @param Szz the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param rmsds   for returning the RDC RMSDs in the order of RDC for medium 1 and medium 2
     * @param resolution the resolution in the grid-search
     * @param debug   if true, it will print the progress of the fitting
     * 
     * @return the rotation matrix which minimizing the score function (RDCs)
     * We only search through 1/4 of the region to obtain one possible orientation,
     * other three can be computed much faster by multiplying by one of the three constant
     * matrices.
     */
    public Matrix eulerFit(final Vector<Pdb> pdbVec, final Vector<Dipolar> rdcVec1, final Vector<Dipolar> rdcVec2, 
			   double Syy, double Szz, double[] rmsds, double resolution, boolean debug){
	double alpha = 0.0, beta = 0.0, gamma = 0.0;
	double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
	int i, j, k, no, index = 0;
	final int N = pdbVec.size();  //the number of RDC data
	Vector nhRdc= rdcVec1;	
	Vector	cahaRdc= rdcVec2;
	double Sxx=-Syy-Szz;
	double[] sRmsdCH= new double[1];
	int[] sizeCH=new int[1];
	double[] sRmsdNH=new double[1];
	int[] sizeNH=new int[1];
		
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	Dipolar dd = new Dipolar();
	
	//added by zeng	
	double rdcExp = 0.0;
	Vector<PdbRdc> pdbNHRdcVec = new Vector<PdbRdc>();
	String atom = "", resid = "";

	for (i=0; i<rdcVec1.size(); i++){
	    dd = (Dipolar)rdcVec1.elementAt(i); 
	    rdcExp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1)
	    	pdbNHRdcVec.add(new PdbRdc(pdbVec.elementAt(index), rdcExp)); 
	}
	
	
	double [] rdcArr1 = new double[N]; 
	double [] rdcArr2 = new double[N]; 
	double [] rdcArr1S =new double[N]; 
	double [] rdcArr2S =new double[N]; 

	
	double[] rdc1Cal  = new double[pdbVec.size() ];
	double[] rdc2Cal  = new double[pdbVec.size() ];

	double [] chArr = new double[3*N]; 
	double [] nhArr = new double[3*N]; 

	Matrix A = new Matrix(3,3);
	Cartesian cc = new Cartesian();
	Vector<Cartesian> atomVec = new Vector<Cartesian>();
	Pdb pp = new Pdb();
	double nhRms=0;
	double chRms=0;

	double rdc1Rms=0;
	
	double [] amide = new double[3];
	double [] nh = new double[3];
	double [] ca = new double[3];
	double [] ch = new double[3];
	double [] ha = new double[3];
	double [] nToNHVec = new double[3];
	double [] caToHAVec = new double[3];
	double [][] mat = new double[3][3];
	double r = 0.0, r1 = 0.0, r2 = 0.0, r1s = 0.0, r2s = 0.0; 
	double rT = 10000.0,  rT1 = 10000.0,  rT2 = 10000.0;
	final double eps = 1.0E-10; //for finding all four possible sets of Euler angles.
	for (i=0; i<pdbVec.size(); i++){        //extract the NH vectors based on the available RDC values.
	    pp = pdbVec.elementAt(i); 
	    no = pp.getResidueNo();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide = cc.getXYZ();
		else if (atom.equals("H"))
		    nh = cc.getXYZ();
		else if (atom.equals("CA"))
		    ca = cc.getXYZ();
		else if (atom.equals("HA"))
		    ha = cc.getXYZ();
	    }
	    nhArr[3 * i]     = (nh[0] - amide[0]) / Const.dN2H;
	    nhArr[3 * i + 1] = (nh[1] - amide[1]) / Const.dN2H;
	    nhArr[3 * i + 2] = (nh[2] - amide[2]) / Const.dN2H;
	    //N(i)->CA(i) vector
	    chArr[3 * i]      = (ha[0] - ca[0]) / Const.dCA2HA;
	    chArr[3 * i + 1]  = (ha[1] - ca[1]) / Const.dCA2HA;
	    chArr[3 * i + 2]  = (ha[2] - ca[2]) / Const.dCA2HA;
	}
	//Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360) 
	for (i = 0; i < (180 / resolution); i++){
		
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (180 / resolution ); k++){
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
		    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
		    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
		    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
		    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
		    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
		    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
		    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
		    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
		    mat[2][2] = Math.cos(beta);
		    
		    Matrix tempA;
		    A = new Matrix(mat);
		 
		    r = 0.0;
		    r1 = 0.0; 
		    r2 = 0.0;
		    nhRms=0.0;
		    chRms=0.0;
		    
		    nhRms = BackCalNH(pdbVec, nhRdc, "N", "H",  A, Sxx, Syy, Szz, Const.nhRatio, rdc1Cal,sRmsdNH, sizeNH);
			r1=nhRms;
		    
			chRms = BackCal(pdbVec, cahaRdc, "CA", "HA", A, Sxx, Syy, Szz, Const.cahaRatio, rdc2Cal,sRmsdCH, sizeCH );
			r2 = chRms; 
		    
		    

		    r = r1 + r2;
		    if( r <= rT || Math.abs(r - rT) < eps) {  //print all the four possible rotation
			alphaS = alpha;
			betaS = beta;
			gammaS = gamma;
			rT = r;
			r1s = r1;
			r2s = r2;
			if(debug){
			    System.arraycopy(rdcArr1, 0, rdcArr1S, 0, N); //save the back computed RDC values
			    System.arraycopy(rdcArr2, 0, rdcArr2S, 0, N);
			    System.out.println(" rT = "+rT+"  "+r1+"  "+r2);
			    rdc1Rms = r1;
			}
		    }
		}
	    }
	}

	if (debug){
	    System.out.println(" Euler "+ alphaS + "  " + betaS + "  " + gammaS);
	    System.out.println();
	    System.out.println("RdcNH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr1S[i]+"   "+rdcExp1[i]);
	    }
	    System.out.println("RMSD = "+ r1s);
	    System.out.println();
	    System.out.println("RdcCH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr2S[i]+"   "+rdcExp2[i]);
	    }
	    System.out.println("RMSD = "+ r2s);
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	Matrix mm = A.eulerMat(alphaS, betaS, gammaS);
	return 	mm;
 }
    


    /**
     * Compute three Euler angles using two sets of RDC data and the corresponding Saupe elements
     * Basic steps:
     * extract the NH vectors with the corresponding RDC data as an array (for efficiency)
     * For all the values of three Euler angles first compute the new NH directions after the rotation
     * then compute the RDCs and RMSD from the new NH directions.
     * 
     * @param pdbVec a vector of Pdb object corresponding to the structural fragment.
     * @param rdcVec1 a vector of NH RDC values
     * @param rdcVec2 a vector of CH RDC values
     * @param rmsds   for returning the RDC RMSDs in the order of RDC for medium 1 and medium 2
     * @param resolution the resolution in the grid-search
     * @param debug   if true, it will print the progress of the fitting
     * @param Syy the syy
     * @param Szz the szz
     * 
     * @return the rotation matrix which minimizing the score function (RDCs)
     * We only search through 1/4 of the region to obtain one possible orientation,
     * other three can be computed much faster by multiplying by one of the three constant
     * matrices.
     */
    public Matrix eulerFit_old(final Vector pdbVec, final Vector rdcVec1, final Vector rdcVec2, 
			   double Syy, double Szz, double[] rmsds, double resolution, boolean debug){
	double alpha = 0.0, beta = 0.0, gamma = 0.0;
	double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
	int i, j, k, no, index = 0;
	final int N = rdcVec1.size();  //the number of RDC data

	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	Dipolar dd = (Dipolar)rdcVec1.elementAt(0);
	no = dd.getResidueNo();      //The 1st residues of fragment
	for (j = 0; j < N; j++){
	    rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(j)).getRdc();
	    rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(j)).getRdc();
	}
	double [] rdcArr1 = new double[N]; 
	double [] rdcArr2 = new double[N]; 
	double [] rdcArr1S =new double[N]; 
	double [] rdcArr2S =new double[N]; 

	double rdc1Cal = 0.0;
	double rdc2Cal = 0.0;

	double [] chArr = new double[3*N]; 
	double [] nhArr = new double[3*N]; 

	Matrix A = new Matrix(3,3);
	Cartesian cc = new Cartesian();
	Vector atomVec = new Vector();
	Pdb pp = new Pdb();
	String atom = "", resid = "";

	double [] amide = new double[3];
	double [] nh = new double[3];
	double [] ca = new double[3];
	double [] ch = new double[3];
	double [] ha = new double[3];
	double [] nToNHVec = new double[3];
	double [] caToHAVec = new double[3];
	double [][] mat = new double[3][3];
	double r = 0.0, r1 = 0.0, r2 = 0.0, r1s = 0.0, r2s = 0.0; 
	double rT = 10000.0,  rT1 = 10000.0,  rT2 = 10000.0;
	final double eps = 1.0E-10; //for finding all four possible sets of Euler angles.
	for (i=0; i<N; i++){        //extract the NH vectors based on the available RDC values.
	    pp = (Pdb)pdbVec.elementAt(i); 
	    no = pp.getResidueNo();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide = cc.getXYZ();
		else if (atom.equals("H"))
		    nh = cc.getXYZ();
		else if (atom.equals("CA"))
		    ca = cc.getXYZ();
		else if (atom.equals("HA"))
		    ha = cc.getXYZ();
	    }
	    nhArr[3 * i]     = (nh[0] - amide[0]) / Const.dN2H;
	    nhArr[3 * i + 1] = (nh[1] - amide[1]) / Const.dN2H;
	    nhArr[3 * i + 2] = (nh[2] - amide[2]) / Const.dN2H;
	    //N(i)->CA(i) vector
	    chArr[3 * i]      = (ha[0] - ca[0]) / Const.dCA2HA;
	    chArr[3 * i + 1]  = (ha[1] - ca[1]) / Const.dCA2HA;
	    chArr[3 * i + 2]  = (ha[2] - ca[2]) / Const.dCA2HA;
	}
	//Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360) 
	for (i = 0; i < (180 / resolution); i++){
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (180 / resolution ); k++){
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
		    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
		    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
		    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
		    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
		    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
		    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
		    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
		    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
		    mat[2][2] = Math.cos(beta);
		    A = new Matrix(mat);
		    r = 0.0;
		    r1 = 0.0; 
		    r2 = 0.0;
		    for (int m = 0; m < N; m++){
			nh[0] = nhArr[3*m];    //Is for CH, 
			nh[1] = nhArr[3*m + 1];
			nh[2] = nhArr[3*m + 2];
			nh = A.times(nh);
			rdc1Cal = ( (nh[1] * nh[1] - nh[0] * nh[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
			         + (nh[2] * nh[2] - nh[0] * nh[0]) * Szz ) * Const.nhRatio;
			r1 += (rdc1Cal - rdcExp1[m]) * (rdc1Cal - rdcExp1[m]);
			//Is for CH, 
			ch[0] = chArr[3*m];
			ch[1] = chArr[3*m + 1];
			ch[2] = chArr[3*m + 2];
			ch = A.times(ch);
			rdc2Cal = (ch[1] * ch[1] - ch[0] * ch[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
			    + (ch[2] * ch[2] - ch[0] * ch[0]) * Szz;
			r2 += (rdc2Cal - rdcExp2[m]) * (rdc2Cal - rdcExp2[m]);
		rdcArr1[m] = rdc1Cal;  
			rdcArr2[m] = rdc2Cal;
		    }

 		    r = r1 + r2;
		    if( r <= rT || Math.abs(r - rT) < eps) {  //print all the four possible rotation
			alphaS = alpha;
			betaS = beta;
			gammaS = gamma;
			rT = r;
			r1s = r1;
			r2s = r2;
			if(debug){
			    System.arraycopy(rdcArr1, 0, rdcArr1S, 0, N); //save the back computed RDC values
			    System.arraycopy(rdcArr2, 0, rdcArr2S, 0, N);
			    System.out.println(" rT = "+rT+"  "+r1+"  "+r2);
			}
		    }
		}
	    }
	}

	if (debug){
	    System.out.println(" Euler "+ alphaS + "  " + betaS + "  " + gammaS);
	    System.out.println();
	    System.out.println("RdcNH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr1S[i]+"   "+rdcExp1[i]);
	    }
	    System.out.println("RMSD = "+ Math.sqrt(r1s/N));
	    System.out.println();
	    System.out.println("RdcCH: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr2S[i]+"   "+rdcExp2[i]);
	    }
	    System.out.println("RMSD = "+ Math.sqrt(r2s/N));
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	Matrix mm = A.eulerMat(alphaS, betaS, gammaS);
	return 	mm;
    }

  /**
   * Compute three Euler angles using two sets of RDC data (pissbile missing)
   * and the corresponding Saupe elements
   * Basic steps:
   * extract the NH vectors with the corresponding RDC data as an array (for efficiency)
   * For all the values of three Euler angles first compute the new NH directions after the rotation
   * then compute the RDCs and RMSD from the new NH directions.
   * 
   * @param pdbVec   a vector of Pdb object corresponding to the structural fragment.
   * @param rdcVec1  a vector of NH RDC
   * @param rdcVec2  a vector of CH RDC
   * @param rmsds    saving the RDC RMSDs and hbScore in the order: rmsdRDC1, rmsdRDC2, hbScore.
   * @param resolution the resolution in grid search on angle space
   * @param debug if true will print the progress of the search
   * @param Syy the syy
   * @param Szz the szz
   * 
   * @return the rotation matrix which minimizing the score function (RDCs)
   * if we use anyone of the four possible rotations this program
   * will find only one. If we do not use any rotational matrix the program will always find four
   * sets of Euler angles.
   * Is there any possiblities that r1 and r2 will rescure the 4 possibilites.
   */
    public Matrix fit4MissingRdc(final Vector pdbVec, final Vector rdcVec1, final Vector rdcVec2,
				 double Syy, double Szz, double[] rmsds, double resolution, boolean debug){
	double alpha = 0.0, beta = 0.0, gamma = 0.0;
        double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
        int i, j, k, no, index = 0;
        final int N = pdbVec.size() - 1;  //the number of RDC data
	int N1 = rdcVec1.size();
	int N2 = rdcVec2.size();

        double [] rdcExp1 = new double[N]; //store the experimental values in the array
        double [] rdcExp2 = new double[N];
        double [] rdcArr1 =new double[N];
        double [] rdcArr2 =new double[N];
        double [] rdcArr1S =new double[N];
        double [] rdcArr2S =new double[N];
	int[] rdc1MissNo = new int[N];
	int[] rdc2MissNo = new int[N];
        double [] chArr = new double[3*N];
        double [] nhArr = new double[3*N];

	PdbRmsd pR = new PdbRmsd();
	Dipolar dd = new Dipolar();
        Matrix A = new Matrix(3,3);
        Cartesian cc = new Cartesian();
        Vector atomVec = new Vector();
        Pdb pp = new Pdb();
        String atom = "", resid = "";

        double [] amide = new double[3];
        double [] nh = new double[3];
        double [] ca = new double[3];
        double [] ch = new double[3];
        double [] ha = new double[3];
        double [] nToNHVec = new double[3];
        double [] caToHAVec = new double[3];
        double [][] mat = new double[3][3];
        double r = 0.0, r1 = 0.0, r2 = 0.0;
        double rT = 10000.0,  rT1 = 10000.0,  rT2 = 10000.0;
        final double eps = 1.0E-10; //for finding all four possible sets of Euler angles.
	double r1s = 0.0,  r2s = 0.0;
	double rdc1Cal = 0.0;
	double rdc2Cal = 0.0;
	double hScore  = 0.0;
	double hScoreS = 0.0;
	Vector pdbVecN = new Vector();
	//extract the NH and CH vectors based on the available RDC values.
        for (i=0; i<N; i++){      
	    pp = (Pdb)pdbVec.elementAt(i); 
	    no = pp.getResidueNo();
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > -1){
		dd = (Dipolar)rdcVec1.elementAt(index); 
		rdcExp1[i] = dd.getRdc();
		rdc1MissNo[i] = 1;
	    }else{
		rdcExp1[i] = 0.0;
		rdc1MissNo[i] = 0;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > -1){
		dd = (Dipolar)rdcVec2.elementAt(index); 
		rdcExp2[i] = dd.getRdc();
		rdc2MissNo[i] = 1;
	    }else{
		rdcExp2[i] = 0.0;
		rdc2MissNo[i] = 0;
	    }
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide = cc.getXYZ();
		else if (atom.equals("H"))
		    nh = cc.getXYZ();
		else if (atom.equals("CA"))
                    ca = cc.getXYZ();
                else if (atom.equals("HA"))
                    ha = cc.getXYZ();
            }
            nhArr[3 * i]     = (nh[0] - amide[0] ) / Const.dN2H;
            nhArr[3 * i + 1] = (nh[1] - amide[1] ) / Const.dN2H;
            nhArr[3 * i + 2] = (nh[2] - amide[2] ) / Const.dN2H;
            //N(i)->CA(i) vector
            chArr[3 * i]      = (ha[0] - ca[0]) / Const.dCA2HA;
            chArr[3 * i + 1]  = (ha[1] - ca[1]) / Const.dCA2HA;
            chArr[3 * i + 2]  = (ha[2] - ca[2]) / Const.dCA2HA;
        }
	for (i = 0; i < (360 / resolution); i++){
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (360 / resolution ); k++){
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
                    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
                    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
                    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
                    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
                    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
                    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
                    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
                    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
                    mat[2][2] = Math.cos(beta);
                    A = new Matrix(mat);
                    r = 0.0;
                    r1 = 0.0;
                    r2 = 0.0;
                    for (int m = 0; m < N; m++){
                        nh[0] = nhArr[3*m];    //Is for CH,
                        nh[1] = nhArr[3*m + 1];
                        nh[2] = nhArr[3*m + 2];
                        nh = A.times(nh);
                        rdc1Cal = ( (nh[1] * nh[1] - nh[0] * nh[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
                                 + (nh[2] * nh[2] - nh[0] * nh[0]) * Szz ) * Const.nhRatio ;
                        r1 += (rdc1Cal - rdcExp1[m]) * (rdc1Cal - rdcExp1[m]) * rdc1MissNo[m] ;
                        //Is for CH,
                        ch[0] = chArr[3*m];
                        ch[1] = chArr[3*m + 1];
                        ch[2] = chArr[3*m + 2];
                        ch = A.times(ch);
                        rdc2Cal = (ch[1] * ch[1] - ch[0] * ch[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
                                 + (ch[2] * ch[2] - ch[0] * ch[0]) * Szz;
                        r2 += (rdc2Cal - rdcExp2[m]) * (rdc2Cal - rdcExp2[m]) * rdc2MissNo[m] ;

                        rdcArr1[m] = rdc1Cal;
                        rdcArr2[m] = rdc2Cal;
                    }
		    r = Math.sqrt(r1 / N1 ) +  Math.sqrt(r2 / N2 );
		    if( r < rT || Math.abs(r - rT) < Const.eps) {//finding all four possible sets of Euler angles.
			alphaS = alpha;
			betaS = beta;
			gammaS = gamma;
			r1s = r1;
			r2s = r2;
			rT = r;
			if(debug){
			    System.arraycopy(rdcArr1, 0, rdcArr1S, 0, N); //save the back computed RDC values
			    System.arraycopy(rdcArr2, 0, rdcArr2S, 0, N);
			    System.out.println(" rT = "+rT+"  "+r1+"  "+r2);
			}
                    }
                }
            }
        }
	if (debug){
 	    System.out.println("RDC1 RMSD = "+ Math.sqrt(r1s/rdcVec1.size()));
	    System.out.println("RDC2 RMSD = "+ Math.sqrt(r2s/rdcVec2.size()));
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	Matrix mm = A.eulerMat(alphaS, betaS, gammaS);
	return 	mm;
    }

  /**
   * Compute three Euler angles using two sets of RDC data (pissbile missing)
   * and the corresponding Saupe elements
   * Basic steps:
   * extract the NH vectors with the corresponding RDC data as an array (for efficiency)
   * For all the values of three Euler angles first compute the new NH directions after the rotation
   * then compute the RDCs and RMSD from the new NH directions.
   * 
   * @param pdbVec   a vector of Pdb object corresponding to the structural fragment.
   * @param rdcVec1  a vector of NH RDC
   * @param rdcVec2  a vector of CH RDC
   * @param rmsds    saving the RDC RMSDs and hbScore in the order: rmsdRDC1, rmsdRDC2, hbScore.
   * @param resolution the resolution in grid search on angle space
   * @param weight4HB relative weight for the H-bond in the score function
   * @param pdbVecA  the H-bond parter of current fragment being searched
   * @param hbVec  the identity of H-bonds
   * @param debug if true will print the progress of the search
   * @param Syy the syy
   * @param Szz the szz
   * 
   * @return the rotation matrix which minimizing the score function (RDCs)
   * if we use anyone of the four possible rotations this program
   * will find only one. If we do not use any rotational matrix the program will always find four
   * sets of Euler angles.
   * Is there any possiblities that r1 and r2 will rescure the 4 possibilites.
   */
    public Matrix fit4MissingRdcHB(final Vector pdbVec, final Vector rdcVec1, final Vector rdcVec2, double Syy, 
				   double Szz,  Vector pdbVecA, double[] rmsds, double resolution, 
				   double weight4HB, Vector hbVec, boolean debug){
	double alpha = 0.0, beta = 0.0, gamma = 0.0;
        double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
        int i, j, k, no, index = 0;
        final int N = pdbVec.size() - 1;  //the number of RDC data
	int N1 = rdcVec1.size();
	int N2 = rdcVec2.size();

        double [] rdcExp1 = new double[N]; //store the experimental values in the array
        double [] rdcExp2 = new double[N];
        double [] rdcArr1 =new double[N];
        double [] rdcArr2 =new double[N];
        double [] rdcArr1S =new double[N];
        double [] rdcArr2S =new double[N];
	int[] rdc1MissNo = new int[N];
	int[] rdc2MissNo = new int[N];
        double [] chArr = new double[3*N];
        double [] nhArr = new double[3*N];

	PdbRmsd pR = new PdbRmsd();
	Dipolar dd = new Dipolar();
        Matrix A = new Matrix(3,3);
        Cartesian cc = new Cartesian();
        Vector atomVec = new Vector();
        Pdb pp = new Pdb();
        String atom = "", resid = "";

        double [] amide = new double[3];
        double [] nh = new double[3];
        double [] ca = new double[3];
        double [] ch = new double[3];
        double [] ha = new double[3];
        double [] nToNHVec = new double[3];
        double [] caToHAVec = new double[3];
        double [][] mat = new double[3][3];
        double r = 0.0, r1 = 0.0, r2 = 0.0;
        double rT = 100000.0,  rT1 = 10000.0,  rT2 = 10000.0;
        final double eps = 1.0E-10; //for finding all four possible sets of Euler angles.
	double r1s = 0.0,  r2s = 0.0;
	double rdc1Cal = 0.0;
	double rdc2Cal = 0.0;
	double hScore  = 0.0;
	double hScoreS = 0.0;
	Vector pdbVecN = new Vector();
	//extract the NH and CH vectors based on the available RDC values.
        for (i=0; i<N; i++){      
	    pp = (Pdb)pdbVec.elementAt(i); 
	    no = pp.getResidueNo();
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > -1){
		dd = (Dipolar)rdcVec1.elementAt(index); 
		rdcExp1[i] = dd.getRdc();
		rdc1MissNo[i] = 1;
	    }else{
		rdcExp1[i] = 0.0;
		rdc1MissNo[i] = 0;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > -1){
		dd = (Dipolar)rdcVec2.elementAt(index); 
		rdcExp2[i] = dd.getRdc();
		rdc2MissNo[i] = 1;
	    }else{
		rdcExp2[i] = 0.0;
		rdc2MissNo[i] = 0;
	    }
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide = cc.getXYZ();
		else if (atom.equals("H"))
		    nh = cc.getXYZ();
		else if (atom.equals("CA"))
                    ca = cc.getXYZ();
                else if (atom.equals("HA"))
                    ha = cc.getXYZ();
            }
            nhArr[3 * i]     = (nh[0] - amide[0] ) / Const.dN2H;
            nhArr[3 * i + 1] = (nh[1] - amide[1] ) / Const.dN2H;
            nhArr[3 * i + 2] = (nh[2] - amide[2] ) / Const.dN2H;
            //N(i)->CA(i) vector
            chArr[3 * i]      = (ha[0] - ca[0]) / Const.dCA2HA;
            chArr[3 * i + 1]  = (ha[1] - ca[1]) / Const.dCA2HA;
            chArr[3 * i + 2]  = (ha[2] - ca[2]) / Const.dCA2HA;
        }
	for (i = 0; i < (360 / resolution); i++){
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (360 / resolution ); k++){
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
                    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
                    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
                    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
                    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
                    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
                    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
                    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
                    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
                    mat[2][2] = Math.cos(beta);
                    A = new Matrix(mat);
                    r = 0.0;
                    r1 = 0.0;
                    r2 = 0.0;
                    for (int m = 0; m < N; m++){
                        nh[0] = nhArr[3*m];    //Is for CH,
                        nh[1] = nhArr[3*m + 1];
                        nh[2] = nhArr[3*m + 2];
                        nh = A.times(nh);
                        rdc1Cal = ( (nh[1] * nh[1] - nh[0] * nh[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
                                 + (nh[2] * nh[2] - nh[0] * nh[0]) * Szz ) * Const.nhRatio ;
                        r1 += (rdc1Cal - rdcExp1[m]) * (rdc1Cal - rdcExp1[m]) * rdc1MissNo[m] ;
                        //Is for CH,
                        ch[0] = chArr[3*m];
                        ch[1] = chArr[3*m + 1];
                        ch[2] = chArr[3*m + 2];
                        ch = A.times(ch);
                        rdc2Cal = (ch[1] * ch[1] - ch[0] * ch[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
                                 + (ch[2] * ch[2] - ch[0] * ch[0]) * Szz;
                        r2 += (rdc2Cal - rdcExp2[m]) * (rdc2Cal - rdcExp2[m]) * rdc2MissNo[m] ;

                        rdcArr1[m] = rdc1Cal;
                        rdcArr2[m] = rdc2Cal;
                    }
                    pdbVecN = pp.newPdb(pdbVec, A);
		    hScore = pR.centerFit(pdbVecA, pdbVecN, hbVec, false);

		    r = weight4HB * hScore + Math.sqrt(r1 / N1 ) +  Math.sqrt(r2 / N2 );
		    if( r < rT || Math.abs(r - rT) < Const.eps) {//finding all four possible sets of Euler angles.
			alphaS = alpha;
			betaS = beta;
			gammaS = gamma;
			r1s = r1;
			r2s = r2;
			hScoreS = hScore;
			rT = r;
			if(debug){
			    System.arraycopy(rdcArr1, 0, rdcArr1S, 0, N); //save the back computed RDC values
			    System.arraycopy(rdcArr2, 0, rdcArr2S, 0, N);
			    System.out.println(" rT = "+rT+"  "+r1+"  "+r2+"  "+hScore);
			}
                    }
                }
            }
        }
	if (debug){
 	    System.out.println("RDC1 RMSD = "+ Math.sqrt(r1s/rdcVec1.size()));
	    System.out.println("RDC2 RMSD = "+ Math.sqrt(r2s/rdcVec2.size()));
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	rmsds[2] = hScoreS;
	Matrix mm = A.eulerMat(alphaS, betaS, gammaS);
	return 	mm;
    }

  /**
   * Compute three Euler angles using two sets of RDC data (pissbile missing)
   * and the corresponding Saupe elements
   * Basic steps:
   * extract the NH vectors with the corresponding RDC data as an array (for efficiency)
   * For all the values of three Euler angles first compute the new NH directions after the rotation
   * then compute the RDCs and RMSD from the new NH directions.
   * 
   * @param pdbVec   a vector of Pdb object corresponding to the structural fragment.
   * @param rdcVec1  a vector of NH RDC
   * @param rdcVec2  a vector of CH RDC
   * @param rmsds    saving the RDC RMSDs and hbScore in the order: rmsdRDC1, rmsdRDC2, hbScore.
   * @param resolution the resolution in grid search on angle space
   * @param weight4HB relative weight for the H-bond in the score function
   * @param pdbVecA1  the H-bond parter of current fragment being searched
   * @param pdbVecA2  another H-bond parter of current fragment being searched
   * @param debug if true will print the progress of the search
   * @param Syy the syy
   * @param Szz the szz
   * @param hbVec1 the hb vec1
   * @param hbVec2 the hb vec2
   * 
   * @return the rotation matrix which minimizing the score function (RDCs)
   * if we use anyone of the four possible rotations this program
   * will find only one. If we do not use any rotational matrix the program will always find four
   * sets of Euler angles.
   * Is there any possiblities that r1 and r2 will rescure the 4 possibilites.
   */
    public Matrix fit4MissingRdcHB2(final Vector pdbVec, final Vector rdcVec1, final Vector rdcVec2, 
				    double Syy, double Szz, Vector pdbVecA1, Vector pdbVecA2, 
				    double[] rmsds, double resolution, double weight4HB, Vector hbVec1, 
				    Vector hbVec2, boolean debug){
	double alpha = 0.0, beta = 0.0, gamma = 0.0;
        double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
        int i, j, k, no, index = 0;
        final int N = pdbVec.size() - 1;  //the number of RDC data
	int N1 = rdcVec1.size();
	int N2 = rdcVec2.size();

        double [] rdcExp1 = new double[N]; //store the experimental values in the array
        double [] rdcExp2 = new double[N];
        double [] rdcArr1 = new double[N];
        double [] rdcArr2 = new double[N];
        double [] rdcArr1S = new double[N];
        double [] rdcArr2S = new double[N];
	int[] rdc1MissNo = new int[N];
	int[] rdc2MissNo = new int[N];
        double [] chArr = new double[3*N];
        double [] nhArr = new double[3*N];

	PdbRmsd pR = new PdbRmsd();
	Dipolar dd = new Dipolar();
        Matrix A = new Matrix(3,3);
        Cartesian cc = new Cartesian();
        Vector atomVec = new Vector();
        Pdb pp = new Pdb();
        String atom = "", resid = "";

        double [] amide = new double[3];
        double [] nh = new double[3];
        double [] ca = new double[3];
        double [] ch = new double[3];
        double [] ha = new double[3];
        double [] nToNHVec = new double[3];
        double [] caToHAVec = new double[3];
        double [][] mat = new double[3][3];
        double r = 0.0, r1 = 0.0, r2 = 0.0;
        double rT = 100000.0,  rT1 = 10000.0,  rT2 = 10000.0;
        final double eps = 1.0E-10; //for finding all four possible sets of Euler angles.
	double r1s = 0.0,  r2s = 0.0;
	double rdc1Cal = 0.0;
	double rdc2Cal = 0.0;
	double hScore1  = 0.0;
	double hScore2  = 0.0;
	double hScore   = 0.0;
	double hScoreS = 0.0;
	Vector pdbVecN = new Vector();
	//extract the NH and CH vectors based on the available RDC values.
        for (i=0; i<N; i++){      
	    pp = (Pdb)pdbVec.elementAt(i); 
	    no = pp.getResidueNo();
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > -1){
		dd = (Dipolar)rdcVec1.elementAt(index); 
		rdcExp1[i] = dd.getRdc();
		rdc1MissNo[i] = 1;
	    }else{
		rdcExp1[i] = 0.0;
		rdc1MissNo[i] = 0;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > -1){
		dd = (Dipolar)rdcVec2.elementAt(index); 
		rdcExp2[i] = dd.getRdc();
		rdc2MissNo[i] = 1;
	    }else{
		rdcExp2[i] = 0.0;
		rdc2MissNo[i] = 0;
	    }
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide = cc.getXYZ();
		else if (atom.equals("H"))
		    nh = cc.getXYZ();
		else if (atom.equals("CA"))
                    ca = cc.getXYZ();
                else if (atom.equals("HA"))
                    ha = cc.getXYZ();
            }
            nhArr[3 * i]     = (nh[0] - amide[0] ) / Const.dN2H;
            nhArr[3 * i + 1] = (nh[1] - amide[1] ) / Const.dN2H;
            nhArr[3 * i + 2] = (nh[2] - amide[2] ) / Const.dN2H;
            //N(i)->CA(i) vector
            chArr[3 * i]      = (ha[0] - ca[0]) / Const.dCA2HA;
            chArr[3 * i + 1]  = (ha[1] - ca[1]) / Const.dCA2HA;
            chArr[3 * i + 2]  = (ha[2] - ca[2]) / Const.dCA2HA;
        }
	for (i = 0; i < (360 / resolution); i++){
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (360 / resolution ); k++){
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
                    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
                    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
                    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
                    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
                    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
                    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
                    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
                    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
                    mat[2][2] = Math.cos(beta);
                    A = new Matrix(mat);
                    r = 0.0;
                    r1 = 0.0;
                    r2 = 0.0;
                    for (int m = 0; m < N; m++){
                        nh[0] = nhArr[3*m];    //Is for CH,
                        nh[1] = nhArr[3*m + 1];
                        nh[2] = nhArr[3*m + 2];
                        nh = A.times(nh);
                        rdc1Cal = ( (nh[1] * nh[1] - nh[0] * nh[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
                                 + (nh[2] * nh[2] - nh[0] * nh[0]) * Szz ) * Const.nhRatio ;
                        r1 += (rdc1Cal - rdcExp1[m]) * (rdc1Cal - rdcExp1[m]) * rdc1MissNo[m] ;
                        //Is for CH,
                        ch[0] = chArr[3*m];
                        ch[1] = chArr[3*m + 1];
                        ch[2] = chArr[3*m + 2];
                        ch = A.times(ch);
                        rdc2Cal = (ch[1] * ch[1] - ch[0] * ch[0]) * Syy   //(cosy*cosy-cosx*cosx)*Syy1 ...
                                 + (ch[2] * ch[2] - ch[0] * ch[0]) * Szz;
                        r2 += (rdc2Cal - rdcExp2[m]) * (rdc2Cal - rdcExp2[m]) * rdc2MissNo[m] ;

                        rdcArr1[m] = rdc1Cal;
                        rdcArr2[m] = rdc2Cal;
                    }
                    pdbVecN = pp.newPdb(pdbVec, A);
 		    hScore1 = pR.centerFit(pdbVecA1, pdbVecN, hbVec1, false);
 		    hScore2 = pR.centerFit(pdbVecN, pdbVecA2, hbVec2, false);

 		    hScore = hScore1 + 4.0*hScore2;

		    r = weight4HB * hScore + Math.sqrt(r1 / N1 ) +  Math.sqrt(r2 / N2 );
		    if( r < rT || Math.abs(r - rT) < Const.eps) {//finding all four possible sets of Euler angles.
			alphaS = alpha;
			betaS = beta;
			gammaS = gamma;
			r1s = r1;
			r2s = r2;
			hScoreS = hScore;
			rT = r;
			if(debug){
			    System.arraycopy(rdcArr1, 0, rdcArr1S, 0, N); //save the back computed RDC values
			    System.arraycopy(rdcArr2, 0, rdcArr2S, 0, N);
			    System.out.println(" rT = "+rT+"  "+r1+"  "+r2+"  "+hScore);
			    System.out.println(" rT = "+alphaS+"  "+betaS+"  "+gammaS);
			}
                    }
                }
            }
        }
	if (debug){
 	    System.out.println("RDC1 RMSD = "+ Math.sqrt(r1s/rdcVec1.size()));
	    System.out.println("RDC2 RMSD = "+ Math.sqrt(r2s/rdcVec2.size()));
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	rmsds[2] = hScoreS;
	Matrix mm = A.eulerMat(alphaS, betaS, gammaS);
	return 	mm;
    }

  /**
   * Compute three Euler angles using two sets of RDC data and the corresponding Saupe elements
   * Basic steps:
   * extract the NH vectors with the corresponding RDC data as an array (for efficiency)
   * For all the values of three Euler angles first compute the new NH directions after the rotation
   * then compute the RDCs and RMSD from the new NH directions.
   * 
   * @param pdbVec   a vector of Pdb object corresponding to the structural fragment.
   * @param rdcVec1  a vector of RDC values from medium 1
   * @param rdcVec2  a vector of RDC values from medium 2
   * @param saupe    four diagnolized Saupe elements in the order of {Syy1, Szz1, Syy2, Szz2}
   * @param rot      The relative rotation matrix between the 2 POFs.
   * @param rmsds    saving the RDC RMSDs and hbScore in the order: rmsdRDC1, rmsdRDC2, hbScore.
   * @param resolution the resolution in grid search on angle space
   * @param weight4HB relative weight for the H-bond in the score function
   * @param pdbVecS1  the H-bond parter of current fragment being searched
   * @param hbVec  the identity of H-bonds
   * @param mat4Dirs the matrix specifying one of the possible orientations.
   * @param debug if true will print the progress of the search
   * 
   * @return the rotation matrix which minimizing the score function (RDCs)
   * if we use anyone of the four possible rotations this program
   * will find only one. If we do not use any rotational matrix the program will always find four
   * sets of Euler angles.
   * Is there any possiblities that r1 and r2 will rescure the 4 possibilites.
   */
    public Matrix eulerFitHB(final Vector pdbVec, final Vector rdcVec1,final Vector rdcVec2, double[] saupe, Matrix rot, 
			     double[] rmsds, double resolution, double weight4HB, Vector pdbVecS1, Vector hbVec,
			     Matrix mat4Dirs, boolean debug){
	double alpha = 0.0, beta = 0.0, gamma = 0.0;
	double alphaS = 0.0, betaS = 0.0, gammaS = 0.0;
	int i, j, k, no;
	int index = 0;
	Matrix A = new Matrix(3,3);
	Cartesian cc = new Cartesian();
	Vector atomVec = new Vector();
	Pdb pp = new Pdb();
	String atom = "", resid = "";

	Dipolar dd = new Dipolar();
	double rdc1Exp = 0.0;
	double rdc2Exp = 0.0;
	Vector pdbRdcVec = new Vector();
	// an array with three NH directional cosines and the last element as RDC value
	int N = rdcVec1.size();  //the number of RDC data
	double [][] nhRdcArr = new double[N][5]; 
	double [] rdcArr1 = new double[N]; 
	double [] rdcArr2 = new double[N]; 
	double [] rdcArr1S =new double[N]; 
	double [] rdcArr2S =new double[N]; 
	double [] amide = new double[3];
	double [] nh = new double[3];
	double [] nh2 = new double[3];
	double [] nhB = new double[3];
	double [][] mat = new double[3][3];
	double r = 0.0, r1 = 0.0, r2 = 0.0; 
	double rT = 10000.0;
	double r1s = 0.0,  r2s = 0.0;
	double rdc1Cal = 0.0;
	double rdc2Cal = 0.0;
	Vector pdbVecN = new Vector();
	double hScore = 0.0, hScoreS = 0.0;
	PdbRmsd pR = new PdbRmsd();
	//extract the NH vectors based on the available RDC values.
	for (i=0; i<N; i++){ 
	    dd = (Dipolar)rdcVec1.elementAt(i); 
	    rdc1Exp = dd.getRdc();
	    dd = (Dipolar)rdcVec2.elementAt(i); 
	    rdc2Exp = dd.getRdc();
	    no = dd.getResidueNo();
	    index = Collections.binarySearch(pdbVec, new Pdb(no), new Pdb.PdbComparator());

	    if (index > -1){
		pp = (Pdb)pdbVec.elementAt(index); 
		no = pp.getResidueNo();
		atomVec = pp.getAtomVec();
		for (j=0; j<atomVec.size(); j++){
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
			amide = cc.getXYZ();
		    else if (atom.equals("H"))
			nh = cc.getXYZ();
		}
		nhRdcArr[i][0] =(nh[0] - amide[0]) / Const.dN2H; //the first three are direction cosines.
		nhRdcArr[i][1] =(nh[1] - amide[1]) / Const.dN2H;
		nhRdcArr[i][2] =(nh[2] - amide[2]) / Const.dN2H;
		nhRdcArr[i][3] = rdc1Exp;   //the last two elements are RDC values
		nhRdcArr[i][4] = rdc2Exp;
	    }
	}
	//Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360) 
	if(debug){
	    System.out.println(" totalScore, rmsdRDC1,  rmsdRDC2,  hbScore");
	}

	for (i = 0; i < (360/resolution); i++){
	    alpha = i * resolution * Math.PI / 180.0;
	    for (j=0; j< (180 / resolution); j++){
		beta = j * resolution * Math.PI / 180.0;
		for (k=0; k< (360 / resolution ); k++){
		    gamma = k * resolution * Math.PI / 180.0;
		    //Generate the Euler matrix
		    mat[0][0] = Math.cos(alpha)*Math.cos(beta)*Math.cos(gamma) - Math.sin(alpha)*Math.sin(gamma);
		    mat[0][1] = Math.sin(alpha)*Math.cos(beta)*Math.cos(gamma) + Math.cos(alpha)*Math.sin(gamma);
		    mat[0][2] = -Math.sin(beta)*Math.cos(gamma);
		    mat[1][0] = -Math.cos(alpha)*Math.cos(beta)*Math.sin(gamma) - Math.sin(alpha)*Math.cos(gamma);
		    mat[1][1] = -Math.sin(alpha)*Math.cos(beta)*Math.sin(gamma) + Math.cos(alpha)*Math.cos(gamma);
		    mat[1][2] = Math.sin(gamma)*Math.sin(beta);
		    mat[2][0] = Math.sin(beta)*Math.cos(alpha);
		    mat[2][1] = Math.sin(alpha)*Math.sin(beta);
		    mat[2][2] = Math.cos(beta);
		    A = new Matrix(mat);
		    A = A.times(mat4Dirs);  //For H-bonds....
		    r = 0.0;
		    r1 = 0.0; 
		    r2 = 0.0;
		    for (int m = 0; m < N; m++){
			nhB[0] = nhRdcArr[m][0];
			nhB[1] = nhRdcArr[m][1];
			nhB[2] = nhRdcArr[m][2];
   			nh = A.times(nhB); 

       			nh2 = nh;             // NOT rotate the NH vector for the second RDC data set
			rdc1Cal =  (nh[1] * nh[1] - nh[0] * nh[0]) * saupe[0] //Syy1//(cosy*cosy-cosx*cosx)*Syy1 ...
			    + (nh[2] * nh[2] - nh[0] * nh[0]) * saupe[1]; //Szz1;
			rdc2Cal =  (nh2[1] * nh2[1] - nh2[0] * nh2[0]) * saupe[2] // Syy2
			    + (nh2[2] * nh2[2] - nh2[0] * nh2[0]) * saupe[3]; //Szz2;
			r1 += (rdc1Cal - nhRdcArr[m][3]) * (rdc1Cal - nhRdcArr[m][3]);
			r2 += (rdc2Cal - nhRdcArr[m][4]) * (rdc2Cal - nhRdcArr[m][4]);
			rdcArr1[m] = rdc1Cal;  
			rdcArr2[m] = rdc2Cal;
		    }
		    pdbVecN = pp.newPdb(pdbVec, A);
		    hScore = pR.centerFit(pdbVecS1, pdbVecN, hbVec, false);
		    r = weight4HB * hScore + Math.sqrt(r1 / N ) +  Math.sqrt(r2 / N );
		    if( r < rT || Math.abs(r - rT) < Const.eps) {//finding all four possible sets of Euler angles.
			alphaS = alpha;
			betaS = beta;
			gammaS = gamma;
			r1s = r1;
			r2s = r2;
			hScoreS = hScore;
			rT = r;
			if(debug){
			    System.arraycopy(rdcArr1, 0, rdcArr1S, 0, N); //save the back computed RDC values
			    System.arraycopy(rdcArr2, 0, rdcArr2S, 0, N);
			    System.out.println(" rT = "+rT+"  "+r1+"  "+r2+"  "+hScore);
			}
		    }
		}
	    }
	}
	if (debug){
	    System.out.println();
	    System.out.println("Rdc1: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr1S[i]+"   "+nhRdcArr[i][3]);
	    }
	    System.out.println("RMSD = "+ Math.sqrt(r1s/N));
	    System.out.println();
	    System.out.println("Rdc2: ");
	    System.out.println("back-computed RDCs,    measured RDCs");
	    for (i=0; i<N; i++){
		System.out.println(rdcArr2S[i]+"   "+nhRdcArr[i][4]);
	    }
	    System.out.println("RMSD = "+ Math.sqrt(r2s/N));
	}
	rmsds[0] = r1s;
	rmsds[1] = r2s;
	rmsds[2] = hScoreS;
	Matrix mm = (A.eulerMat(alphaS, betaS, gammaS)).times(mat4Dirs);
	return 	mm;
    }

    /**
     * private utility routines for checking the SVD method *.
     * 
     * @param x the x
     * @param y the y
     */
    /** Check magnitude of difference of scalars. **/
    private static void check(double x, double y) {
	double eps = Math.pow(2.0,-52.0);
	if (x == 0 & Math.abs(y) < 10*eps) return;
	if (y == 0 & Math.abs(x) < 10*eps) return;
	if (Math.abs(x-y) > 10*eps*Math.max(Math.abs(x), Math.abs(y))) {
	    throw new RuntimeException("The difference x-y is too large: x = " 
				       + Double.toString(x) + "  y = " + Double.toString(y));
	}
    }
    
    /**
     * Check norm of difference of "vectors". *
     * 
     * @param x the x
     * @param y the y
     */
    private static void check(double[] x, double[] y) {
	if (x.length == y.length ) {
	    for (int i=0;i<x.length;i++) {
		check(x[i],y[i]);
	    } 
	} else {
	    throw new RuntimeException("Attempt to compare vectors of different lengths");
	}
    }

    /**
     * Check norm of difference of arrays. *
     * 
     * @param x the x
     * @param y the y
     */
   private static void check(double[][] x, double[][] y) {
      Matrix A = new Matrix(x);
      Matrix B = new Matrix(y);
      check(A,B);
   }

   /**
    * Check norm of difference of Matrices. *
    * 
    * @param X the x
    * @param Y the y
    */
   private static void check(Matrix X, Matrix Y) {
      double eps = Math.pow(2.0,-52.0);
      if (X.norm1() == 0. & Y.norm1() < 10*eps) return;
      if (Y.norm1() == 0. & X.norm1() < 10*eps) return;
      if (X.minus(Y).norm1() > 1000*eps*Math.max(X.norm1(),Y.norm1())) {
         throw new RuntimeException("The norm of (X-Y) is too large: " +  Double.toString(X.minus(Y).norm1()));
      }
   }

    /**
     * Shorten spelling of print. *
     * 
     * @param s the s
     */
    private static void print (String s) {
	System.out.print(s);
    }

    /**
     * Print appropriate messages for successful outcome try *.
     * 
     * @param s the s
     * @param e the e
     */
    private static void try_success (String s,String e) {
	print(">    " + s + "success\n");
	if ( e != "" ) {
	    print(">      Message: " + e + "\n");
	}
    }
    
    /**
     * Print appropriate messages for unsuccessful outcome try *.
     * 
     * @param count the count
     * @param s the s
     * @param e the e
     * 
     * @return the int
     */
    private static int try_failure (int count, String s,String e) {
	print(">    " + s + "*** failure ***\n>      Message: " + e + "\n");
	return ++count;
    }
}
