package Nasca;
/*
NASCA NOE Assignment and Side-Chain Assignment  Software Version 1.0
Copyright (C) 2009-2011 Bruce Donald Lab, Duke University

NASCA is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the Free
Software Foundation, either version 3 of the License, or (at your option) any
later version.

NASCA is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details.

You should have received a copy of the GNU Lesser General Public License
along with this library; if not, see:
    <http://www.gnu.org/licenses/>.

There are additional restrictions imposed on the use and distribution of this
open-source code, including: (A) this header must be included in any
modification or extension of the code; (B) you are required to cite our
papers in any publications that use this code. The citation for the various
different modules of our software, together with a complete list of
requirements and restrictions are found in the document license.pdf enclosed
with this distribution.

Contact Info:
    Bruce R. Donald
    Duke University
    Department of Computer Science
    Levine Science Research Center (LSRC)
    Durham, NC 27708-0129
    USA
    email: www.cs.duke.edu/brd/

<signature of Bruce Donald>, 01 December, 2009
Bruce R. Donald, Professor of Computer Science and Biochemistry
*/
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.Map;
import java.util.Vector;

import Jampack.JampackException;

import rdcPanda.Assign;
import rdcPanda.H1CS;
import rdcPanda.Hsqc;
import rdcPanda.ModelRdc;
import rdcPanda.Noe;
import rdcPanda.Noesy;
import rdcPanda.Pdb;
import rdcPanda.Peak;

/**
 * This class provides functions for generating the simulated NOESY data for ILV test.
 *
 */
public class ILVTest {
	
	/** Extract the ILV chemical shift information (HN and methyl groups) from the reference chemical shift table.
     *  Note: for Leu and Val, only one methyl group is available. For Ile, only methyl group delta 1 is available.
     @param vecRefChemShifts reference chemical shift table.
     @param 
     @param 
     @param
     @return 
     */
	public Vector<H1CS> extractILVChemicalShifts(Vector<H1CS> vecRefChemShifts){
		Vector<H1CS> vecILVResons=new Vector<H1CS>();
		for(int i=0;i<vecRefChemShifts.size();i++){
			H1CS h1cs=(H1CS)vecRefChemShifts.elementAt(i);
			String atom=h1cs.getAtomName();
			String res=h1cs.getResidueType();
			
			//include  amide and amide proton
			if(atom.equalsIgnoreCase("H") || atom.equalsIgnoreCase("HN") || atom.equalsIgnoreCase("N"))
				vecILVResons.add(h1cs);
			
			//methyl group of ILE.
			if(res.equalsIgnoreCase("ILE")){
				if(atom.equalsIgnoreCase("HD1") || atom.equalsIgnoreCase("CD1"))
					vecILVResons.add(h1cs);
			}
			
			//methyl group of LEU, only extract HD1 and CD1.
			if(res.equalsIgnoreCase("LEU")){
				if(atom.equalsIgnoreCase("HD1") || atom.equalsIgnoreCase("CD1"))
					vecILVResons.add(h1cs);
			}
			
			//methyl group of VAL, only extract HD1 and CD1.
			if(res.equalsIgnoreCase("VAL")){
				if(atom.equalsIgnoreCase("HD1") || atom.equalsIgnoreCase("CD1"))
					vecILVResons.add(h1cs);
			}
		}//for(int i=0;i<vecRefChemShifts.size();i++)
		return vecILVResons;
	}
	
	/** Extract the ILV methyl and HN NOEs from the original NOESY data.
	 *note: for each NOESY peak, we check whether there exists a pair of ILV chemical shifts (HN and methyl groups) 
	 *intersect around this peak. If yes, we keep this peak, otherwise we exclude it.
     @param vecOrigNoesy original NOESY peak list
     @param vecILVResons ILV chemical shifts     
     @return subset of NOESY peaks involving in HN and methyl groups of ILV residues.
     */
	public Vector<Noesy> extractILVNoesyPeaks(Vector<Noesy> vecOrigNoesy, Vector<H1CS> vecILVResons){
		
		Peak pk=new Peak();
		double errWin_h1=0.05; //error window 
		double errWin_heavy=0.5;
		double errWin_h2=0.05;
		
		Vector<Noesy> vecILVNoesy=new Vector<Noesy>();
		for(int i=0;i<vecOrigNoesy.size();i++){
			Noesy noesy=(Noesy)vecOrigNoesy.elementAt(i);
			double csH1=noesy.getH1();
			double csHeavy=noesy.getHeavy();
			double csH2=noesy.getH2();
			
			boolean isFirstProtonIn=false;
			boolean isSecondProtonIn=false;
			
			//check the first proton
			for(int j=0;j<vecILVResons.size();j++){
				H1CS h1cs=(H1CS)vecILVResons.elementAt(j);
				double cs_proton=h1cs.getH1CS();
				int resNo=h1cs.getResidueNo();
				String atom=h1cs.getAtomName();
				String res=h1cs.getResidueType();
				String heavyName=pk.GetHeavyAtomFromProton(res,atom);
				double cs_heavy=-999.9;
				for(int t=0;t<vecILVResons.size();t++)
				{
					H1CS h1csHeavy=(H1CS)vecILVResons.elementAt(t);
					double cs_Temp=h1csHeavy.getH1CS();
					
					String atomT=h1csHeavy.getAtomName();
					int resNoT=h1csHeavy.getResidueNo();
					if( (resNoT==resNo) && (atomT.equalsIgnoreCase(heavyName)) )
						cs_heavy=cs_Temp;
				}//for(int j=0;j<vecAsg.size();j++)
				
				if(Math.abs(csH1-cs_proton)<errWin_h1 && Math.abs(csHeavy-cs_heavy)<errWin_heavy){
					isFirstProtonIn=true;
					break;
				}			
			}//for(int j=0;j<vecILVResons.size();j++)
			
			//check the second proton
			for(int j=0;j<vecILVResons.size();j++){
				H1CS h1cs=(H1CS)vecILVResons.elementAt(j);
				double cs_proton=h1cs.getH1CS();				
				if(Math.abs(csH2-cs_proton)<errWin_h2 ){
					isSecondProtonIn=true;
					break;
				}			
			}//for(int j=0;j<vecILVResons.size();j++)
			if(isFirstProtonIn && isSecondProtonIn)
				vecILVNoesy.add(noesy);
		}//for(int i=0;i<vecOrigNoesy.size();i++)
		return vecILVNoesy;
	}
	
	//main function for testing
	public  static void main (String[] args) throws JampackException
	{
		//output program information:
		//outputProgInfo();
		
		//read the NMR data:
		String userDir = System.getProperty("user.dir");
		String src=userDir+"/inputFiles/";  //input file directory		
		String strOut=userDir+"/outFiles/";
		String strParametersFileName="Nasca.input";
		
		ILVTest test=new ILVTest();
		test.doILVTest(src,strOut,strParametersFileName);	
		
	}  // End main()
	
	
	/** 
     *do the ILV test.
     @param src location of the input file
     @param strOut location of the output file
     @param strInput  input file name
     @return void
    */       
   // @SuppressWarnings("unchecked")
	public void doILVTest(String src, String strOut, String strInput)throws JampackException
	{
    	boolean isDebug=false;//whether outputing the debugging information
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	NascaNew nasca=new NascaNew();
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	
    	int nIs4DNoesy=0;
    	
    	String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="", strResNameScheme="",strIsCheckLongAA="",
    		strHaNoesy3D="", strResFormat="",strBB="",stroutName="",strNoesyFormat="",strOutOrFormat="",
    		strIsCheck="",strRefPdb="",strRefNameSchem="",strIsWholeStr="",strPdbNameScheme="",str4DNoesy="",
    		stroutResonName="",strOutResonFormat="";
    	int nIsoriginalUp=0; 
    	double noeLimit=0.0,metCor=0.0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map<String, String> paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		//get the input file name   
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("RESFORMAT"))
    			strResFormat  =  (String)paraMap.get("RESFORMAT");
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		//strNoesy2D=strNoesy2D.toLowerCase();
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		//strHnNoesy3D=strHnNoesy3D.toLowerCase();
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		//strHaNoesy3D=strHaNoesy3D.toLowerCase();    		
    		if(paraMap.containsKey("PDBNAMESCHEME"))
    			strPdbNameScheme  =  (String)paraMap.get("PDBNAMESCHEME");	
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");   	
    		if(paraMap.containsKey("BACKBONE"))
    			strBB  =  (String)paraMap.get("BACKBONE");
    		if(paraMap.containsKey("NOELIMIT"))
    			noeLimit  =  Double.parseDouble((String)paraMap.get("NOELIMIT"));
    		if(paraMap.containsKey("IS4DNOESY"))
    			nIs4DNoesy  =   Integer.parseInt((String)paraMap.get("IS4DNOESY"));
    		
    		if(paraMap.containsKey("4D-NOESY"))
    			str4DNoesy  =  (String)paraMap.get("4D-NOESY");
    		
    		if(paraMap.containsKey("ISORIGINALUP"))
    			nIsoriginalUp  =   Integer.parseInt((String)paraMap.get("ISORIGINALUP"));	
    		if(paraMap.containsKey("RESNAMESCHEME"))
    			strResNameScheme  =  (String)paraMap.get("RESNAMESCHEME");	
    		if(paraMap.containsKey("NOESY-FORMAT"))
    			strNoesyFormat  =  (String)paraMap.get("NOESY-FORMAT");   	
    		if(paraMap.containsKey("ISCHECKLONGAA"))
    			strIsCheckLongAA =  (String)paraMap.get("ISCHECKLONGAA"); 	
    		if(paraMap.containsKey("ISCHECK"))
    			strIsCheck  =  (String)paraMap.get("ISCHECK");   	
    		if(paraMap.containsKey("REFPDB"))
    			strRefPdb  =  (String)paraMap.get("REFPDB");  
    		if(paraMap.containsKey("METHYL-CORRECTION"))
    			metCor  =  Double.parseDouble((String)paraMap.get("METHYL-CORRECTION"));    		
    		if(paraMap.containsKey("REFNAMESCHEME"))
    			strRefNameSchem  =  (String)paraMap.get("REFNAMESCHEME");  
    		if(paraMap.containsKey("ISWHOLESTRUCTURE"))
    			strIsWholeStr  =  (String)paraMap.get("ISWHOLESTRUCTURE");  
    		if(paraMap.containsKey("ISOUTORFORMAT"))
    			strOutOrFormat  =  (String)paraMap.get("ISOUTORFORMAT");  
    		
    		if(paraMap.containsKey("OUTRESONANCENAME"))
    			stroutResonName  =  (String)paraMap.get("OUTRESONANCENAME"); 
    		if(paraMap.containsKey("OUTRESONANCEFORMAT"))
    			strOutResonFormat  =  (String)paraMap.get("OUTRESONANCEFORMAT");
    	}       	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------    	
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector<Assign> vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the resonance list 
    	//     	
    	//following is for the general case
    	H1CS h1CS=new H1CS();    	
    	String assignFile = src+ strReson;
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector<H1CS> assignVec=new Vector<H1CS>();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("unknown format of the resonance file...");
    		System.exit(0);
    	}
    	
    	//all sorted proton resonances
    	Vector<Peak> allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());     	
    	//-------------------------------------
    	// (1.3) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector<Noesy> hnNoeVec=new Vector<Noesy>();//for storing 3d n15 NOESY peaks
    	Vector<Noesy> cnoeVec=new Vector<Noesy>();//for storing 3d c13 noesy peaks
    	Vector<Noesy> Noe4DVec=new Vector<Noesy>();//for storing 4d NOESY peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			hnNoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			cnoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	//4D NOESY data
    	if (!str4DNoesy.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+str4DNoesy;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			Noe4DVec = noesy.NoesyReader(strNoeFile);// to be changed....
    		else
    			Noe4DVec = noesy.NoesyReaderNMRView4D(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	Vector<Noesy> vecNoesy=new Vector<Noesy>();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	
    	Vector<Noesy> vecNoesy4D=new Vector<Noesy>();
    	vecNoesy4D.addAll(Noe4DVec);//for 4D noesy    	

    	////////////////////////////////////    	    	
    	//intensity calibration    	
    	Vector<Noesy> vecNewNoesy=new Vector<Noesy>();
    	if(vecNoesy.size()>0)
    		vecNewNoesy=noesy.SetCalibration(vecNoesy); ////  	
    	Vector<Noesy> vecNewNoesy4D=new Vector<Noesy>();
    	if(vecNoesy4D.size()>0)
    		vecNewNoesy4D=noesy.SetCalibration(vecNoesy4D);   
    	double[] constant=new double[1];
    	Vector<Pdb> vecSSE=new Vector<Pdb>();//not used in the following function:
    	noesy.SetCalibrationN15(hnNoeVec,vecSSE,assignVec,constant);//change from vecNoesy to hnNoeVec
    	
    	// -------------------------------------
    	// (1.4) Read the backbone structure     	
    	//-------------------------------------    	
    	String pdbFile = src+strBB;
    	Vector<Pdb> pdbVecBB = pp.readPdb(pdbFile);
    	Vector<Pdb> pdbVecNew=new Vector<Pdb>();
    	
    	//update backbone atom names according to protein sequence input
    	//residue name in previous backbone may be all in "ALA".
    	if(strIsWholeStr.equalsIgnoreCase("1"))    	
    		pdbVecNew= pp.residueNameUpdateNoStr(vecSeq, pdbVecBB); 
    	else
    		pdbVecNew = pp.residueNameUpdate(vecSeq, pdbVecBB);
    	
    	Vector<Pdb> pdbVecNewSec=new Vector<Pdb>();
		if(strPdbNameScheme.equalsIgnoreCase("PDB-OLD"))
			pdbVecNewSec=pp.nameConvertOrder(pdbVecNew); 
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-NEW"))
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecNew);//need to double check
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-OLD"))
		{
			Vector<Pdb> pdbVecSSE22=pp.nameConvertOrder(pdbVecNew); 
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecSSE22);//need to double check
		}
		else
			pdbVecNewSec.addAll(pdbVecNew);  
    	
		Vector<Pdb> vecStructureAll=new Vector<Pdb>();
		vecStructureAll.addAll(pdbVecNewSec);
		
		Vector<Pdb> vecBBTemp=pp.OutputBackbone(pdbVecNewSec);
		pdbVecNewSec=new Vector<Pdb>();
		pdbVecNewSec.addAll(vecBBTemp);    	
    
      
    	////////////////////////////////////////
    	//extract the ILV chemical shifts
		Vector<H1CS> vecILVResons=extractILVChemicalShifts(assignVec);
    	
    	//////////////////////////////
    	//print out the ILV related resonance assignments into files:
		stroutName="/outFiles/ILV_CS.prot";
		String userDir = System.getProperty("user.dir");////
		H1CS h1cs=new H1CS();
    	String outResFileName=userDir+stroutName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outResFileName)));
    		
    		if(strOutResonFormat.equalsIgnoreCase("CYANA"))        	
    			h1cs.PrintAllAssignmentToFileCyana(vecILVResons,out);        	
        	else        	
        		h1cs.PrintAllAssignmentToFileBMRB(vecILVResons,out);      	
	       
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ outResFileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + outResFileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		//////////////////////////////////////////////////////////////////////////
		
		//////////////////////////////////////
		//extract the subset of NOESY peaks involving in ILV
		Vector<Noesy> vecILVNoesy=extractILVNoesyPeaks(vecNoesy,vecILVResons);
		
		//////////////////////////////////////////
		//output the ILV subset of NOESY peaks:
		String strOutNoesyName="";
		if(strNoesyFormat.equalsIgnoreCase("XEASY"))  
			strOutNoesyName="/outFiles/ILV_Noesy.PEAKS";
		else
			strOutNoesyName="/outFiles/ILV_Noesy.XPK";
		
		String outNoesyFileName=userDir+strOutNoesyName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outNoesyFileName)));
    		
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))        	
    			noesy.PrintAllNoesyToFileXEASY(vecILVNoesy,out);        	
        	else        	
        		noesy.PrintAllNoesyToFileNMRView(vecILVNoesy,out);      	
	       
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ outNoesyFileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + outResFileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		//////////////////////////////////////////////////////////////////////////
    	
     	///////////////////////////////////////
     	long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Time= "+ totalTime +" minutes");
	}


}
