package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Hbond.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

import java.io. *;
import java.util. *;
 
// TODO: Auto-generated Javadoc
/** * 
 *  
*   This class defines data structures related to h-bonds.
*   Written by Lincong Wang (2001-2005).
*/

public class Hbond implements Cloneable{
    
    /** The residue no a. */
    private int residueNoA;   //the Sequence No of one of the two partners of the NOE
    
    /** The residue no b. */
    private int residueNoB;   //the Sequence No of Another partner of the NOE
    
    /** The atom a. */
    private String atomA;     //the name of Atom A
    
    /** The atom b. */
    private String atomB;     //the name of Atom B
    
    /** The id. */
    private String id;        //an id assigned to the H-bond
    
    /** The type. */
    private String type;      //the name of Atom B

    /**
     * Constructors.
     */
    public Hbond() {
        residueNoA = 0;
        residueNoB = 0;
        atomA = null;
        atomB = null;
        id    = null;        
	type  = null;
    }

   /**
    * Instantiates a new hbond.
    * 
    * @param Ano the ano
    * @param A the a
    * @param Bno the bno
    * @param B the b
    * @param id the id
    * @param t the t
    */
   public Hbond(int Ano, String A, int Bno,  String B,  String id, String t) {
        residueNoA = Ano;
        residueNoB = Bno;
        this.atomA = A;
        this.atomB = B;
        this.id   = id;
        this.type = t;
    }
 
    /**
     * Gets the residue no a.
     * 
     * @return the residue no a
     */
    public int getResidueNoA() {
        return residueNoA;
    }
    
    /**
     * Gets the residue no b.
     * 
     * @return the residue no b
     */
    public int getResidueNoB() {
        return residueNoB;
    }
    
    /**
     * Gets the atom a.
     * 
     * @return the atom a
     */
    public String getAtomA()   {
        return atomA;
    }
    
    /**
     * Gets the atom b.
     * 
     * @return the atom b
     */
    public String getAtomB()   {
        return atomB;
    }
    
    /**
     * Gets the iD.
     * 
     * @return the iD
     */
    public String getID()   {
        return id;
    }
    
    /**
     * Gets the type.
     * 
     * @return the type
     */
    public String getType()   {
        return type;
    }
  
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString()    {
        String desc = residueNoA + atomA + "__"+residueNoB+atomB+"  "+id+"   "+type;
        return desc;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    protected Object clone(){
        try{
            Object s = super.clone();     // clone the stack
            return s;                           // return the clone
        } catch (CloneNotSupportedException e)  {
            throw new InternalError();
        }
    }
 
    /**
     * The Class HbondComparatorA.
     */
    public static class HbondComparatorA implements Comparator{
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Hbond n1 = (Hbond)o1;
            Hbond n2 = (Hbond)o2;
            int d1 = n1.getResidueNoA();
            int d2 = n2.getResidueNoA();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }
 
    /**
     * The Class HbondComparatorB.
     */
    public static class HbondComparatorB implements Comparator{
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Hbond n1 = (Hbond)o1;
            Hbond n2 = (Hbond)o2;
            int d1 = n1.getResidueNoB();
            int d2 = n2.getResidueNoB();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }
    
    /**
     * A very simple reader for H-bond data: see the attched file for the format.
     * 
     * @param hbFile the name of the file
     * 
     * @return a vector of Hbond object
     */
    public Vector HbondReader(String hbFile){
        Vector inputs = new Vector();
        int noA, noB;
        String atomA, atomB, id, type;
        try{
            StreamTokenizer in =
                new StreamTokenizer(
                                    new FileReader(hbFile));
            while( in.nextToken() != StreamTokenizer.TT_EOF ){ //ignore the comments
                if (in.ttype == StreamTokenizer.TT_WORD)
                    ;
                else if (in.ttype == StreamTokenizer.TT_NUMBER){
                    noA = (int)in.nval;
                    in.nextToken();
                    atomA = in.sval;
                    in.nextToken();
                    noB = (int)in.nval;
                    in.nextToken();
                    atomB = in.sval;
                    in.nextToken();
                    id = in.sval;
                    in.nextToken();
                    type = in.sval;
                    inputs.add(new Hbond(noA, atomA, noB, atomB, id, type));
                }
            }
        }catch (FileNotFoundException e){
            System.out.println("File not found: " + hbFile);
        }catch (IOException e) {
            System.out.println("IOException: the stack trace is:");
            e.printStackTrace();
        }
        return inputs;
    }

    /**
     * Hb of two strands.
     * 
     * @param hbVec the hb vec
     * @param ids the ids
     * 
     * @return the vector
     */
    public Vector hbOfTwoStrands(Vector hbVec, String ids)
    {
	String id;        //an id assigned to the H-bond
	Hbond hb = new  Hbond();
	Vector hb2Strands = new Vector();
	for (int i = 0; i < hbVec.size(); i++){
	    hb =  (Hbond)hbVec.elementAt(i);
	    if (hb.getID().equals(ids))
		hb2Strands.add(hb);
	}
	return hb2Strands;
    }
}
