package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Hncacb.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////

/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc

/** * 
 *  
*   This class defines data structures related to HSQC peaks.  Written by Lincong Wang (2001-2005).
*/
public class Hsqc{
    
    /** The peak no. */
    private int peakNo;
    
    /** The residue no. */
    private int    residueNo; 
    
    /** The residue. */
    private String residue;
    
    /** The cs h1. */
    private double csH1; 
    
    /** The cs n15. */
    private double csN15;
    
    /** The intensity. */
    private double intensity;

  /**
   * Instantiates a new hsqc.
   */
  public Hsqc(){
	peakNo = 0;
	residueNo = 0;
	residue = null;
	csH1 = 0.0; 
        csN15 = 0.0;
	intensity = 0.0;
    }		
    
    /**
     * Instantiates a new hsqc.
     * 
     * @param no the no
     */
    public Hsqc(int no){
	peakNo = 0;
	residueNo = no;
	residue = null;
	csH1 = 0.0; 
        csN15 = 0.0;
	intensity = 0.0;
    }		
    
    /**
     * Instantiates a new hsqc.
     * 
     * @param pkNo the pk no
     * @param no the no
     * @param resid the resid
     * @param H1 the h1
     * @param N15 the n15
     * @param inten the inten
     */
    public Hsqc(int pkNo, int no,String resid, double H1, double N15, double inten){
	peakNo = pkNo;
	residueNo = no;
	residue = resid;
	csH1 = H1; 
        csN15 = N15;
	intensity = inten;
    }	

  //getting the values	
    /**
   * Gets the peak no.
   * 
   * @return the peak no
   */
  public int getPeakNo(){
	return peakNo;
    }	
    
    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the residue type.
     * 
     * @return the residue type
     */
    public String getResidueType(){
	return residue;
    }	
    
    /**
     * Gets the hN.
     * 
     * @return the hN
     */
    public double getHN(){
	return csH1;
    }	
    
    /**
     * Gets the n15.
     * 
     * @return the n15
     */
    public double getN15(){
	return csN15;
    }	
    
    /**
     * Gets the int.
     * 
     * @return the int
     */
    public double getInt(){
	return intensity;
    }	
    //set the values	
    /**
     * Sets the residue no.
     * 
     * @param R the new residue no
     */
    public void setResidueNo(int R){
	residueNo = R;
    }	

    //set the values	
    /**
     * Sets the residue type.
     * 
     * @param type the new residue type
     */
    public void setResidueType(String type){
	residue = type;
    }	

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
	String desc = String.valueOf(peakNo) +"     "+String.valueOf(residueNo) +"    "
	    + residue+"      "+String.valueOf(csH1)
	    +"     "+String.valueOf(csN15)+"     "+ (2.0*48.00 - csN15)+"    "+String.valueOf(intensity);
	return desc;	
    }

    /**
     * The Class resNoComparator.
     */
    public static class resNoComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hsqc n1 = (Hsqc)o1;
	    Hsqc n2 = (Hsqc)o2;
	    double d1 = n1.getResidueNo();
	    double d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class peakNoComparator.
     */
    public static class peakNoComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hsqc n1 = (Hsqc)o1;
	    Hsqc n2 = (Hsqc)o2;
	    double d1 = n1.getPeakNo();
	    double d2 = n2.getPeakNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

  /**
   * The Class hnComparator.
   */
  public static class hnComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hsqc n1 = (Hsqc)o1;
	    Hsqc n2 = (Hsqc)o2;
	    double d1 = n1.getHN();
	    double d2 = n2.getHN();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class n15Comparator.
     */
    public static class n15Comparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hsqc n1 = (Hsqc)o1;
	    Hsqc n2 = (Hsqc)o2;
	    double d1 = n1.getN15();
	    double d2 = n2.getN15();
	    if (d1 > d2)
		return -1;
	    else if (d1 < d2)
		return 1;
	    else return 0;
	}
    }

   /**
    * A method for reading files with RDC restraints.
    * The required file format is: residue No. RDC value, experimental error.
    * Must have only three items per line. Otherwise, errors.
    * 
    * @param filename the name of the file
    * 
    * @return a vector of RDC object
    */
    public Vector hsqcRead(String filename){
	double foldCorrect =0.0;// 2.0*48.00 for matching fold and unfold C13HSQC
	Vector inputs = new Vector();
	int pkNo;
	int resNo =0;
	double csH1, csN15, intensity;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    pkNo = (int)in.nval;
		    in.nextToken();
 		    csH1 = in.nval; 
		    in.nextToken();
		    in.nextToken();
		    in.nextToken();
 		    csN15 =  in.nval; //foldCorrect - in.nval;// -0.50; //for c13HSqc
		    in.nextToken();
		    in.nextToken();
		    in.nextToken();
 		    intensity = in.nval;
		    in.nextToken();
		    in.nextToken();
// 		    System.out.println(pkNo+"  "+ csH1+" "+ csN15+"  "+intensity);
		    inputs.add(new Hsqc(pkNo, 0, "", csH1, csN15, intensity));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }
  
  /**
   * A method for reading files with RDC restraints.
   * The required file format is: residue No. RDC value, experimental error.
   * Must have only three items per line. Otherwise, errors.
   * 
   * @param filename the name of the file
   * 
   * @return a vector of RDC object
   */
    public Vector hsqcReader2(String filename){
	Vector inputs = new Vector();
	int pkNo;
	int resNo =0;
	double csH1, csN15, intensity;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    resNo = (int)in.nval;
		    in.nextToken();
		    pkNo = (int)in.nval;
		    in.nextToken();
		    in.nextToken();
 		    csH1 = in.nval; 
		    in.nextToken();
 		    csN15 = in.nval;// -0.50; //for c13HSqc
		    in.nextToken();
 		    intensity = in.nval;
// 		    System.out.println(pkNo+"  "+ csH1+" "+ csN15+"  "+intensity);
		    inputs.add(new Hsqc(pkNo, resNo, "", csH1, csN15, intensity));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }

    /**
     * Hsqc reader3.
     * 
     * @param filename the filename
     * 
     * @return the vector
     */
    public Vector hsqcReader3(String filename){
	double foldCorrect = 2.0*48.00;// for matching fold and unfold C13HSQC
	Vector inputs = new Vector();
	double csHA = 0.0, csC13 = 0.0, intensity = 0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int pkNo  = 0;
	int resNo = 0;
	String aaType = "", ss ="";
	try{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    stop:
	    while(true) {
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    pkNo = Integer.parseInt(st.nextToken());
		if (st.hasMoreTokens())
		    resNo = Integer.parseInt(st.nextToken());
		if (st.hasMoreTokens())
		    csHA = new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    csC13 = foldCorrect -  new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    intensity = (new Double(st.nextToken()).doubleValue());
		inputs.add(new Hsqc(pkNo, resNo, "", csHA, csC13, intensity));
		if ((ss = in.readLine()) == null){
		    break stop;
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }

    /**
     * Range search.
     * 
     * @param nucType the nuc type
     * @param a the a
     * @param b the b
     * @param pkVec the pk vec
     * 
     * @return the vector
     */
    public Vector rangeSearch(String nucType, double a, double b, Vector pkVec){
	Vector outVec = new Vector();
	double csValue = 0.0;
	for (int i=0; i< pkVec.size(); i++){
	    if (nucType.equals("H1"))
		csValue = ((Hsqc)pkVec.elementAt(i)).getHN();
	    else if (nucType.equals("N15"))
		csValue = ((Hsqc)pkVec.elementAt(i)).getN15();
	    if (csValue >= a && csValue <= b)
		outVec.add(pkVec.elementAt(i));
	}
	return outVec;
    }

   
}
