package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Rotamer.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;
import java.text.NumberFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.FieldPosition;

import Jampack.JampackException;

// TODO: Auto-generated Javadoc
/** * 
 * 
*   This class provides data structures and functions related to rotamers. 
*   
*    Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
*/
public class Rotamer
{
    
    /** The AA name. */
    private String AAName;
    
    /** The pdb vec. */
    private Vector pdbVec = new Vector(); //all the atoms belong to the residue

    /**
     * Instantiates a new rotamer.
     */
    public Rotamer()
    {
    	AAName = null;
    	pdbVec = null;
    }
    
    /**
     * Instantiates a new rotamer.
     * 
     * @param res the res
     */
    public Rotamer(String res)
    {
    	AAName = res;
    	pdbVec = null;
    }
    
    /**
     * Instantiates a new rotamer.
     * 
     * @param res the res
     * @param aa the aa
     */
    public Rotamer(String res, Vector aa)
    {	
    	AAName = res;
    	pdbVec = aa;
    }

    //getting the values	
    /**
     * Gets the aA name.
     * 
     * @return the aA name
     */
    public String getAAName()
    {
    	return AAName;
    }	
    
    /**
     * Gets the pdb vec.
     * 
     * @return the pdb vec
     */
    public Vector getPdbVec()
    {
    	return pdbVec;
    }	
    
    //Set method
    /**
     * Sets the aA name.
     * 
     * @param res the new aA name
     */
    public void setAAName(String res)
    {
    	AAName = res;
    }	
  
    /**
     * Sets the atom vec.
     * 
     * @param aVec the new atom vec
     */
    public void setAtomVec(Vector aVec){
	pdbVec = aVec;
    }	

   

    /**
     * prune rotamers based on:
     * 1: collisions with backbone atoms;
     * 2: collisions between pairwise rotamer atoms;
     * 3: NOE constraints, including pairwise and local NOEs
     * 
     * Note: a recurrsive approach is used.
     * 
     * @param vecPdb the pdb structure including backbone and those residue in which rotamers have been decided.
     * @param rotSrc the file path of the rotamer library
     * @param noeVec vector of ALL NOE  restraints
     * @param curNo the cur no
     * @param lastNo the last no
     * @param vecEmsemblePdbSave the vec emsemble pdb save
     * 
     * @return true, if prune rotamers
     * 
     * @throws JampackException the jampack exception
     */
    public boolean pruneRotamers(Vector vecPdb, String rotSrc, final Vector noeVec,int curNo, int lastNo, Vector vecEmsemblePdbSave)throws JampackException
    {
        System.out.println("curNo=: "+ curNo);
    	if(curNo>lastNo)
    	{
    		//1. check whether the pdb satisfies all NOE restraints:
    		 for (int i = 0; i < noeVec.size(); i++)
			 {			    
    			 Pdb pp=new Pdb();
			 	 double [][] noeDistance = new double[noeVec.size()][2];	
		    	 int noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
		    	 String atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
		    	
		    	 int noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
		 	     String atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
		 	     noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
				
		 	     double [] dis=new double[1];
		 	     double [] distance=new double[1];
		 	 	 pp.measurePackDisAllRotamers(vecPdb,vecPdb,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
		 	     if (dis[0]>0 )
		 	 		return false;
		 	 
			  }// for (int i = 0; i < noeVec.size(); i++)
    		 
    		 //2. check the steric clash:
    		 vdw vander = new vdw();
    		 Vector vdwVec = new Vector();
    		 double[] vdwValue = new double[1];
		     boolean hasVDW = false;
		     double vdwLevel = 0.05;
		     boolean printVDWViolation = false;
		     
		     vdwVec = vander.convert2VDW(vecPdb);
	         int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,Const.vdwStericThreshold);
		     System.out.println("the number of steric clashes: "+ numClashes);
	         if(numClashes>0)
		    		return false;
	         
	         //3. otherwise, save the structure:
	         vecEmsemblePdbSave.add(vecPdb);
	         return true;
    		
    	}//if(curNo>lastNo)
    	else //otherwise recurrsively put rotamers on the backbone
    	{    		
    		Vector pdbRotVec=new Vector();
    		Pdb pp=new Pdb();
    		int ind = Collections.binarySearch(vecPdb, new Pdb(curNo), new Pdb.PdbComparator());
    		if(ind<0)//when curNo is in the loop
    		{
    			pruneRotamers(vecPdb,rotSrc,noeVec,curNo+1,lastNo,vecEmsemblePdbSave);
    			return false;
    		}
    		
    		Pdb pdbA = (Pdb)vecPdb.elementAt(ind); 
    		String resName=pdbA.getResidue();
    		if( resName.equalsIgnoreCase("GLY"))
        	{        		   			
        		pdbRotVec.add(pdbA);    
        		//pdbRotamNewStr.addAll(pdbRotVec);
        	}
        	else
        	{
        		String rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
        		Vector pdbRotam=pp.readPdb(rotamFile);
        		//atom names are consistet with structure, namely HB1,HB2...
        		pdbRotVec=pp.nameConvert4SmallRotLibStr(pdbRotam);           	   		
        	}   
    		
    		boolean isAtLeaseOne=false;
    		for (int i=0;i<pdbRotVec.size();i++)
    		{
    			System.out.println("curNo=: "+ curNo);
    			System.out.println("rotamer id=: "+ i);
    			Vector vecRotNew=new Vector();
    			Pdb pdb_temp=(Pdb)pdbRotVec.elementAt(i);
    			vecRotNew.add(pdb_temp);
    			//Vector vecTempBB=pp.OutputBackbone(vecPdb);
    			Vector vecTemp=pp.CombinRotBB(vecRotNew,vecPdb,curNo);
    			Model md=new Model();
    			Vector vecGrowPdbNew=new Vector();
    			vecGrowPdbNew=  md.BackonbeRotation(vecTemp,vecPdb); 
    			
    			// check whether the pdb satisfies all NOE restraints:
    			 boolean isContinue=false;
	       		 for (int j = 0; j < noeVec.size(); j++)
	   			 {		       			
	   			 	 double [][] noeDistance = new double[noeVec.size()][2];	
	   		    	 int noA = ((Noe)noeVec.elementAt(j)).getResidueNoA();
	   		    	 String atomA = ((Noe)noeVec.elementAt(j)).getAtomA();
	   		    	
	   		    	 int noB = ((Noe)noeVec.elementAt(j)).getResidueNoB();
	   		 	     String atomB = ((Noe)noeVec.elementAt(j)).getAtomB();
	   		 	     noeDistance[j] = ((Noe)noeVec.elementAt(j)).getRange();
	   				
	   		 	     double [] dis=new double[1];
	   		 	     double [] distance=new double[1];
	   		 	 	 boolean isMeasure=pp.measurePackDisAllRotamers(vecGrowPdbNew,vecGrowPdbNew,noA,atomA,noB, atomB,noeDistance[j][0], noeDistance[j][1],dis,distance);
	   		 	     if (isMeasure && (dis[0]>0) )
	   		 	     {
	   		 	    	isContinue=true;
	   		 	 		break;
	   		 	     }
	   		 	 	 
	   			  }// for (int j = 0; j < noeVec.size(); j++)
	       		 if (isContinue)
	       			 continue;
	       		 
	       		 //	check the steric clash:
	    		 vdw vander = new vdw();
	    		 Vector vdwVec = new Vector();
	    		 double[] vdwValue = new double[1];
			     boolean hasVDW = false;
			     double vdwLevel = 0.05;
			     boolean printVDWViolation = false;
			     
			     vdwVec = vander.convert2VDW(vecGrowPdbNew);
		         int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,Const.vdwStericThreshold);
			     //System.out.println("the number of steric clashes: "+ numClashes);
		         if(numClashes>0)
			    		continue;  			
    			
    			pruneRotamers(vecGrowPdbNew,rotSrc,noeVec,curNo+1,lastNo,vecEmsemblePdbSave);
    		}//for (int i=0;i<pdbRotVec.size();i++)
    			
    			
    	}//else of if(curNo>lastNo)  	
    	
    	return false;
    }
    
    /**
     * Read rotamer.
     * 
     * @param rtamerFile the rtamer file
     * 
     * @return the vector
     */
    public Vector readRotamer(String rtamerFile)
    {	
    	Pdb pp=new Pdb();   
    	return pdbVec = pp.readPdb(rtamerFile);
    	
    }
    //for test
    /**
     * The main method.
     * 
     * @param argv the arguments
     * 
     * @throws JampackException the jampack exception
     */
    public static void main(String[] argv)  throws JampackException
	{
    	Pdb  pp = new Pdb();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String rotSrc=  src+ "rotasamp-small/";   
    	String pdbFile = src+"H3.pdb"; //"one-structure-pei-eta-2I5O.pdb";//"2A7O.pdb";   
    	Vector vecTempPdb=pp.readPdb(pdbFile);
    	Vector vecTempPdbBB=pp.OutputBackbone(vecTempPdb);
      
    	
    	//read the NOE assignment table from cyana
    	String strManualAsg="final.upl";
    	String manualAsgFile = src+ strManualAsg;//"noe-H123.txt";//strManualAsg; //"resonance.prot";  
    	Noe noe_temp=new Noe();
    	Vector vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");//noe_temp.noeReader(manualAsgFile);
    	
    	Rotamer rt=new Rotamer();
    	Vector vecEnsemblePdbs=new Vector();
    	rt.pruneRotamers(vecTempPdbBB,rotSrc,vecManAsg,10,76,vecEnsemblePdbs);
    	System.out.println("the ensemble size is: "+ vecEnsemblePdbs.size());
	}//public static void main(String[] argv) 
    
}   

  
