package rdcPanda;

 ///////////////////////////////////////////////////////////////////////////////////////////////
//	ScNoe.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 *  
 * represeting the tetrahedral of the sp3 Carbon configuration
 * IN order to compute the coordinates we need only to know the coordinates of the top atom
 * then using the information provided by the sp3Carbon object we should be able to compute 
 * all the coordinates of other atoms.

         atomOfChi
             |
             |
	    Top
             |
             |
          center
          /  |  \
         /   |   \
        /    |    \
  vertex1 vertex2 vertex3

The angles are defined as shown in the above figure.

Written by Lincong Wang (2001-2005).
*/
public class sp3Carbon{
    //the atom and its three Cartesian coodinates
    /** The theta1. */
    private double theta1; //The plane angle for three bottom atoms defined as the angle between two vectors
    
    /** The theta2. */
    private double theta2; // such as the vector (Top-->center) and the vector (center-->vertex1)
    
    /** The theta3. */
    private double theta3; 
    
    /** The chi. */
    private double chi; //di-angle between the plane (atomOfChi-->Top-->center) and the plane (Top-->center-->vertex1)
    
    /** The theta2d. */
    private double theta2d; //di-angle for defining the 2nd atom
    
    /** The theta3d. */
    private double theta3d; //di-angle for defining the 3rd atom
    
    /** The d1. */
    private double d1;      //distance from the top to the bottom
    
    /** The d2. */
    private double d2;
    
    /** The d3. */
    private double d3;
    
    /** The terminal. */
    private boolean terminal = false;
    
    /** The Constant cst. */
    static final double cst = 180.0 / Math.PI;
   
    //Constructors
    /**
     * Instantiates a new sp3 carbon.
     */
    public sp3Carbon(){
	theta1 = 0.0;
	theta2 = 0.0;
	theta3 = 0.0;
	chi    = 0.0;
	theta2d = 0.0;
	theta3d = 0.0;
	d1 = 0.0;
	d2 = 0.0;
	d3 = 0.0;
	terminal = false;
    }

    /**
     * Instantiates a new sp3 carbon.
     * 
     * @param theta1a the theta1a
     * @param theta2a the theta2a
     * @param theta3a the theta3a
     * @param chia the chia
     * @param theta2da the theta2da
     * @param theta3da the theta3da
     * @param d1a the d1a
     * @param d2a the d2a
     * @param d3a the d3a
     * @param ter the ter
     */
    public sp3Carbon(double theta1a, double theta2a, double theta3a, double chia,double theta2da,
		     double theta3da, double d1a, double d2a, double d3a, boolean ter){
	theta1 = theta1a;
	theta2 = theta2a;
	theta3 = theta3a;
	chi    = chia;
	theta2d = theta2da;
	theta3d = theta3da;
	d1 = d1a;
	d2 = d2a;
	d3 = d3a;
	terminal = ter;
    }

    /**
     * The Constructor.
     * 
     * @param anglesAndLen has all the data for specifying a sp3Carbon object except whether
     * it is a termial one
     * @param ter the ter
     */
    public sp3Carbon(double[] anglesAndLen, boolean ter){
	theta1 = anglesAndLen[0];
	theta2 = anglesAndLen[1];
	theta3 = anglesAndLen[2];
	chi    = anglesAndLen[3];
	theta2d =anglesAndLen[4];
	theta3d =anglesAndLen[5];
	d1 = anglesAndLen[6];
	d2 = anglesAndLen[7];
	d3 = anglesAndLen[8];
	terminal = ter;
    }

    /**
     * Instantiates a new sp3 carbon.
     * 
     * @param atomOfChi the atom of chi
     * @param top the top
     * @param center the center
     * @param vertex1 the vertex1
     * @param vertex2 the vertex2
     * @param vertex3 the vertex3
     * @param ter the ter
     */
    public sp3Carbon(double[] atomOfChi, double[] top, double[] center, double[] vertex1, 
		     double[] vertex2, double[] vertex3, boolean ter){
	double[] anglesAndLen = anglesAndLenOfSp3(atomOfChi, top, center, vertex1, vertex2, vertex3);
	theta1 = anglesAndLen[0];
	theta2 = anglesAndLen[1];
	theta3 = anglesAndLen[2];
	chi    = anglesAndLen[3];
	theta2d =anglesAndLen[4];
	theta3d =anglesAndLen[5];
	d1 = anglesAndLen[6];
	d2 = anglesAndLen[7];
	d3 = anglesAndLen[8];
	terminal = ter;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
        String desc = String.valueOf(theta1*cst) +"  "+String.valueOf(theta2*cst) +"  "+String.valueOf(theta3*cst)+"\n";
	desc += String.valueOf(chi*cst) +"  "+String.valueOf(theta2d*cst) +"  "+String.valueOf(theta3d*cst)+"\n";
// 	desc += String.valueOf(d1)+"  "+String.valueOf(d2)+"  "+String.valueOf(d3);
        return desc;
    }

    /**
     * Checks if is terminal.
     * 
     * @return true, if is terminal
     */
    public boolean isTerminal(){
	return terminal;
    }
    
    /**
     * Gets the all.
     * 
     * @return the all
     */
    public double[] getAll(){
	return new double[]{theta1, theta2, theta3, chi, theta2d, theta3d, d1, d2, d3};
    }

   /**
    * Add two coordinates.
    * 
    * @param n1 the coordinate for the nucleus of the atom 1
    * @param n2 the coordinate for the nucleus of the atom 2
    * 
    * @return a vector from n1->n2
    */
    public double[] addCoords(double[] n1, double[] n2){
        return new double[]{n2[0]+n1[0], n2[1]+n1[1], n2[2]+n1[2]};
    }

    /**
     * Prints the array.
     * 
     * @param n1 the n1
     */
    public void printArray(double[] n1){
	for (int i=0; i<n1.length; i++)
	    System.out.print(n1[i]+"  ");
	System.out.println();
    }

    /**
     * Compute the coordinates of sp3 atoms from a sp3Carbon object.
     * The program may have bugs if there exists gap in the sequence. We
     * want to compute all the relavant angles. Six plus angles for HA
     * and CO, CACB vectors
     * The rotation is specifed as:
     * rotate along +y to the plane as specified by CH1 angle
     * then rotate along +x to the vector from the center to one of the bottom atom.
     * at this stage the +y axis is on this vector
     * 
     * @param mat    the Matrix to the center ("top") of the tetrahedral, that is: a matrix
     * specifying the plane defined by the three atoms (atomOfChi, top, center) with
     * the top-->center vector on the +y axis and z-axis also in the plane,
     * This is also the first plane defining the Chi angle
     * @param center coordinate of the center, really the top atom (vertice) of tetrahedral.
     * @param atomC  the carbon atom of SP3 class with all the informs about bond angles
     * 
     * @return an array of Matrix in the  order
     * The coordinates for the 3 bottom atoms.
     * and if it is not a terminal atom the method also return
     * The rotation matrix to the plane specfied by (top, center, vertex1) with
     * vector (center->vectex1) in +y-axis and z-axis in the plane (x-axis perpendicular to it)
     * The similar rotation matrix to the vector center->vectex2 (+y-axis)
     * The similar rotation matrix to the vector center->vectex3 (+y-axis)
     */
    public Matrix [] tetrahedral(final double[] center, final Matrix mat, sp3Carbon atomC){
	double[] allElements = atomC.getAll();
	double theta1 = allElements[0];
	double theta2 = allElements[1];
	double theta3 = allElements[2];
	double chi    = allElements[3];
	double theta2d = allElements[4]; 
	double theta3d = allElements[5];
	double d1 = allElements[6];
	double d2 = allElements[7];
	double d3 = allElements[8];

	Matrix rx1Inv   = mat.rotationMat(theta1, "-x"); //Since we only need inverse
	Matrix rx2Inv   = mat.rotationMat(theta2, "-x");
	Matrix rx3Inv   = mat.rotationMat(theta3, "-x");
	Matrix ryChiInv = mat.rotationMat(chi, "-y");
	Matrix ry2dInv  = mat.rotationMat(theta2d, "-y");
	Matrix ry3dInv  = mat.rotationMat(theta3d, "-y");

	//Begin to compute coordinates
	double[] coord1 = new double[]{0.0, d1, 0.0};
	double[] v1 = mat.times(ryChiInv.times(rx1Inv.times(coord1)));
	coord1 = addCoords(center, v1);
	double[] coord2 = new double[]{0.0, d2, 0.0};
	double[] v2 = mat.times(ryChiInv.times(ry2dInv.times(rx2Inv.times(coord2))));
	coord2 = addCoords(center, v2);
	double[] coord3 = new double[]{0.0, d3, 0.0};
	double[] v3 = mat.times(ryChiInv.times(ry3dInv.times(rx3Inv.times(coord3))));
	coord3 = addCoords(center, v3);
// 	System.out.println("Back Computed: ");
//  	printArray(coord1);
//  	printArray(coord2);
//  	printArray(coord3);
// 	System.out.println();
	if (!atomC.isTerminal()){ //if is NOT a terminal
	    Matrix mat1 = mat.times(ryChiInv.times(rx1Inv));
	    Matrix mat2 = mat.times(ryChiInv.times(ry2dInv.times(rx2Inv)));
	    Matrix mat3 = mat.times(ryChiInv.times(ry3dInv.times(rx3Inv)));
	    double[][] coords = new double[][]{{coord1[0], coord1[1], coord1[2]},
					       {coord2[0], coord2[1], coord2[2]},
					       {coord3[0], coord3[1], coord3[2]}};
	    Matrix matCoords = new Matrix(coords);
	    return new Matrix[]{matCoords, mat1, mat2, mat3};
	}else{
	    double[][] coords = new double[][]{{coord1[0], coord1[1], coord1[2]},
					       {coord2[0], coord2[1], coord2[2]},
					       {coord3[0], coord3[1], coord3[2]}};
	    Matrix matCoords = new Matrix(coords);
	    return new Matrix[]{matCoords};
	}
    }

    /**
     * Compute the side-chain angles as represented by SP3 object from the coordinates of 4 atoms.
     * The Chi di-angle is specified by atomOfChi, top, center and vertex1. The structure is as shown
     * 
     * atomOfChi
     * |
     * |
     * Top
     * |
     * |
     * center
     * /  |  \
     * /   |   \
     * /    |    \
     * vertex1 vertex2 vertex3
     * 
     * @param atomOfChi the coord for the atom leading to the top, needed for computing Chi di-angle
     * @param top       the coord for top atom
     * @param center    the coord for center of the tetrahedral, really its "top" atom
     * @param vertex1   the coord for 1st bottom atom of the tetrahedral
     * @param vertex2   the coord for 2nd bottom atom of the tetrahedral
     * @param vertex3   the coord for 3rd bottom atom of the tetrahedral
     * 
     * @return an object of sp3Carbon class
     */
    public double [] anglesAndLenOfSp3(final double[] atomOfChi, double[] top, double[] center, 
				       double[] vertex1, double[] vertex2, double[] vertex3){
	Pdb pp = new Pdb();
	double [] atom2Top = pp.internuclearVec(atomOfChi, top);
	double [] top2Center = pp.internuclearVec(top, center);
	double [] center2V1  = pp.internuclearVec(center, vertex1);
	double [] center2V2  = pp.internuclearVec(center, vertex2);
	double [] center2V3  = pp.internuclearVec(center, vertex3);
	double theta1 = pp.interAngle(top2Center, center2V1);
	double theta2 = pp.interAngle(top2Center, center2V2);
	double theta3 = pp.interAngle(top2Center, center2V3);
	double chi = Math.PI + pp.dihedralAngle(atom2Top, top2Center, center2V1); 
	if (chi >= Math.PI)
	    chi -= 2.0*Math.PI;
	double theta2d = pp.dihedralAngle(center2V1, top2Center, center2V2);
	double theta3d = pp.dihedralAngle(center2V1, top2Center, center2V3);
	double d1 = pp.length(center2V1);
	double d2 = pp.length(center2V2);
	double d3 = pp.length(center2V3);
// 	System.out.println(" input ");
//  	pp.printArray(vertex1);  //used for checking the code
//  	pp.printArray(vertex2);
//  	pp.printArray(vertex3);
// 	System.out.println();
// 	return (new double[]{cst*theta1,cst*theta2,cst*theta3,cst*chi,cst*theta2d,cst*theta3d, d1, d2, d3});
	return (new double[]{theta1,theta2,theta3,chi,theta2d,theta3d, d1, d2, d3});
    }
}
