package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	vdw.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

import java.io. *;
import java.util. *;

import Jampack.JampackException;

// TODO: Auto-generated Javadoc
/** * 
 *  
*   A class for computing the vdw repellsion.
*   
*   Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
*   */
public class vdw {
    
    /** The _id. */
    String _id; //id is a String "resid+residueNo+"_+atomName" which should be unique
    
    /** The _coord. */
    double[] _coord = new double[3]; //the coordinate for this atom
    
    /** The vec pdb. */
    Vector vecPdb=new Vector();
    
    /** The vdw_score. */
    double vdw_score=0.0;
    
    /** The nh rmsd. */
    double nhRmsd=0.0;
    
    /** The ch rmsd. */
    double chRmsd=0.0;
    
    /** The noe rmsd. */
    double noeRmsd=0.0;
    
    /** The ensemble id. */
    int ensembleID=0;
    
    /** The model id. */
    int modelID=0;
    
   
    /**
     * Instantiates a new vdw.
     */
    public vdw(){ //an empty Constructor used for initializing
    }
    
    /**
     * Instantiates a new vdw.
     * 
     * @param vecPp the vec pp
     * @param sc the sc
     */
    public vdw( Vector vecPp,double sc)
    {
    	vecPdb=vecPp;
    	vdw_score=sc;   	
    }
    
    /**
     * Instantiates a new vdw.
     * 
     * @param vecPp the vec pp
     * @param sc the sc
     * @param noe_sc the noe_sc
     */
    public vdw( Vector vecPp,double sc, double noe_sc)
    {
    	vecPdb=vecPp;
    	vdw_score=sc;   
    	noeRmsd=noe_sc;
    }
    
    /**
     * Instantiates a new vdw.
     * 
     * @param vecPp the vec pp
     * @param sc the sc
     * @param noe_sc the noe_sc
     * @param modID the mod id
     */
    public vdw( Vector vecPp,double sc, double noe_sc,int modID)
    {
    	vecPdb=vecPp;
    	vdw_score=sc;   
    	noeRmsd=noe_sc;
    	modelID=modID;
    }
    
    /**
     * Instantiates a new vdw.
     * 
     * @param vecPp the vec pp
     * @param sc the sc
     * @param noe_sc the noe_sc
     * @param modID the mod id
     * @param id the id
     */
    public vdw( Vector vecPp,double sc, double noe_sc,int modID,String id)
    {
    	vecPdb=vecPp;
    	vdw_score=sc;   
    	noeRmsd=noe_sc;
    	modelID=modID;
    	_id=id;
    }
    
    /**
     * Instantiates a new vdw.
     * 
     * @param vecPp the vec pp
     * @param sc the sc
     * @param nhRms the nh rms
     * @param chRms the ch rms
     * @param enID the en id
     * @param modID the mod id
     */
    public vdw( Vector vecPp,double sc,double nhRms, double chRms, int enID, int modID)
    {
    	vecPdb=vecPp;
    	vdw_score=sc;   
    	nhRmsd=nhRms;
    	chRmsd=chRms;
    	ensembleID=enID;
    	modelID=modID;
    }
    
    /**
     * Instantiates a new vdw.
     * 
     * @param id2 the id2
     * @param coord2 the coord2
     */
    public vdw(String id2, double[] coord2){
	_id = id2;
	System.arraycopy(coord2, 0, _coord, 0, 3);
    }
    
    /**
     * Gets the pDB.
     * 
     * @return the pDB
     */
    public Vector getPDB(){
    	return vecPdb;
        }
    
    /**
     * Gets the vdw score.
     * 
     * @return the vdw score
     */
    public double getVdwScore(){
    	return vdw_score;
        }
    
    /**
     * Gets the noe rmsd.
     * 
     * @return the noe rmsd
     */
    public double getNoeRmsd(){
    	return noeRmsd;
        }
    
    /**
     * Gets the ensemble id.
     * 
     * @return the ensemble id
     */
    public double getEnsembleID(){
    	return ensembleID;
        }
    
    /**
     * Gets the model id.
     * 
     * @return the model id
     */
    public double getModelID(){
    	return modelID;
        }
    
    /**
     * Gets the string id.
     * 
     * @return the string id
     */
    public String getStringID(){
    	return _id;
        }
    
    /**
     * Gets the nh rmsd.
     * 
     * @return the nh rmsd
     */
    public double getNhRmsd(){
    	return nhRmsd;
        }
    
    /**
     * Gets the ch rmsd.
     * 
     * @return the ch rmsd
     */
    public double getChRmsd(){
    	return chRmsd;
        }
    
    
    /**
     * Gets the iD.
     * 
     * @return the iD
     */
    public String getID(){
	return _id;
    }

    /**
     * Gets the coord.
     * 
     * @return the coord
     */
    public double[] getCoord(){
	return _coord;
    }

    /**
     * Sets the iD.
     * 
     * @param id2 the new iD
     */
    public void setID(String id2){
	_id = id2;
    }

    /**
     * Sets the coord.
     * 
     * @param coord2 the new coord
     */
    public void setCoord(double[] coord2){
	System.arraycopy(coord2, 0, _coord, 0, 3);
    }
     
    /**
     * Length.
     * 
     * @param v1 the v1
     * 
     * @return the double
     */
    public double length(double[] v1){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	return v1Len;
    }
    
    /**
     * Internuclear vec.
     * 
     * @param n1 the n1
     * @param n2 the n2
     * 
     * @return the double[]
     */
    public double[] internuclearVec(double[] n1, double[] n2){
        return new double[]{n2[0]-n1[0], n2[1]-n1[1], n2[2]-n1[2]};
    }
  
    /**
     * The Class VdwComparator.
     */
    public static class VdwComparator implements Comparator{
    	
	    /* (non-Javadoc)
	     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	     */
	    public int compare(Object o1, Object o2){
    	    vdw n1 = (vdw)o1;
    	    vdw n2 = (vdw)o2;
    	    double d1 = n1.getVdwScore();
    	    double d2 = n2.getVdwScore();
    	    if (d1 <d2)
    		return -1;
    	    else if (d1 >d2)
    		return 1;
    	    else return 0;
    	}
        }
    
    /**
     * The Class VdwNoeComparator.
     */
    public static class VdwNoeComparator implements Comparator{
    	
	    /* (non-Javadoc)
	     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	     */
	    public int compare(Object o1, Object o2){
    	    vdw n1 = (vdw)o1;
    	    vdw n2 = (vdw)o2;
    	    double d1 = n1.getNoeRmsd();
    	    double d2 = n2.getNoeRmsd();
    	    if (d1 <d2)
    		return -1;
    	    else if (d1 >d2)
    		return 1;
    	    else return 0;
    	}
        }
    
    /**
     * To convert the PDB coordinates into a flat structure for speed.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector convert2VDW(Vector pdbVec){
	int i, j;
	Pdb pp = new Pdb();
	String atom, resid, name ="";
	Vector atomVec = new Vector();
	Cartesian cc = new Cartesian();
	int no;
	Vector vdwVec = new Vector();
       	for (i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    atomVec = pp.getAtomVec();
	    resid = pp.getResidue();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		name = resid+no+"_"+atom;
		vdwVec.add(new vdw(name, cc.getXYZ()) );
	    }
	}
	return vdwVec;
    }

    /**
     * To convert the PDB coordinates into a flat structure for speed.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector convert2VDW(Pdb[] pdbVec){
	int i, j;
	Pdb pp = new Pdb();
	String resid;
	Vector atomVec = new Vector();
	Cartesian cc = new Cartesian();
	int no;
	Vector vdwVec = new Vector();
	String name ="";
       	for (i=0; i<pdbVec.length; i++){
	    pp = pdbVec[i];
	    no = pp.getResidueNo();
	    atomVec = pp.getAtomVec();
	    resid = pp.getResidue();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		name = resid+no+"_"+cc.getAtom();
		vdwVec.add(new vdw(name, cc.getXYZ()));
	    }
	}
	return vdwVec;
    }
    
    /**
     * Compute the vdw energy.
     * 
     * @param vdwVec coordinate object in a flat form
     * @param vdwValue for returning the computed vdwEnergy
     * @param isHelix is a boolean for treating specifically the H-bond for a helix
     * 
     * @return true, if compute vdw
     */
    public boolean computeVDW (Vector vdwVec, double[] vdwValue, boolean isHelix){
	String id1, id2, name1, name2;
	int index = -1;
	double[] coord1 = new double[3];
	double[] coord2 = new double[3];
	vdw vander1 = new vdw();
	vdw vander2 = new vdw();
	int i, j, k;
	double rms = 0.0;
	double distance = 0.0;
	double rmsOne = 0.0;
	int no1, no2; //residue No.
	String resid = "";
	boolean isHbond = false;
	for (i=0; i<vdwVec.size(); i++){
	    vander1 = (vdw)vdwVec.elementAt(i);
	    id1 = vander1.getID();
	    resid = id1.substring(0, 3);

	    index = id1.indexOf("_");
	    no1 = Integer.parseInt((id1.substring(3, index)).trim()); 
	    if (index > -1)
		name1 = id1.substring(index+1);
	    else return false;
	    coord1 = vander1.getCoord();
	    for (j=i+1; j<vdwVec.size(); j++){
		vander2 = (vdw)vdwVec.elementAt(j);
		id2 = vander2.getID();
		index = id2.indexOf("_");
		no2 = Integer.parseInt((id2.substring(3, index)).trim()); 
		if (index > -1)
		    name2 = id2.substring(index+1);
		else return false;
		coord2 = vander2.getCoord();
		distance = length(internuclearVec(coord2, coord1));
		isHbond = false;
		if(isHelix && name1.equals("O") && name2.equals("H") && (no2 - no1) == 4)
		    isHbond = true;

		if(! isBonded(no1, no2, name1, name2)) { //only for polyAlanine
		    rmsOne = checkVDW(name1, name2, distance, isHbond);
		    rms += Math.sqrt(rmsOne);
		}
	    }
	}
	vdwValue[0] = rms;
	return true;
    }


    /**
     * Compute the vdw energy.
     * 
     * @param vdwVec coordinate object in a flat form
     * @param vdwValue for returning the computed vdwEnergy
     * @param vdwLevel if the vdwEnergy > vdwLevel and printVDWViolation
     * is true, print out the violation for checking
     * @param printVDWViolation whether to print the vdw violation
     * 
     * @return true, if compute vdw
     */
    public boolean computeVDW (Vector vdwVec, double[] vdwValue, double vdwLevel, boolean printVDWViolation){
	String id1, id2, name1, name2;
	int index = -1;
	double[] coord1 = new double[3];
	double[] coord2 = new double[3];
	vdw vander1 = new vdw();
	vdw vander2 = new vdw();
	int i, j;
	double rms = 0.0;
	double distance = 0.0;
	double rmsOne = 0.0;
	String resid ="";
	int no1, no2; //residue No.
	for (i=0; i<vdwVec.size(); i++){
	    vander1 = (vdw)vdwVec.elementAt(i);
	    id1 = vander1.getID();
	    resid = id1.substring(0, 3);

	    index = id1.indexOf("_");
	    no1 = Integer.parseInt((id1.substring(3, index)).trim()); 
	    if (index > -1)
		name1 = id1.substring(index+1);
	    else return false;
	    coord1 = vander1.getCoord();
	    for (j=i+1; j<vdwVec.size(); j++){
		vander2 = (vdw)vdwVec.elementAt(j);
		id2 = vander2.getID();
		index = id2.indexOf("_");
		no2 = Integer.parseInt((id2.substring(3, index)).trim()); 
		if (index > -1)
		    name2 = id2.substring(index+1);
		else return false;
		coord2 = vander2.getCoord();
		distance = length(internuclearVec(coord2, coord1));
  		if(!isBonded(resid, no1, no2, name1, name2)) {
		    rmsOne = checkVDW(name1, name2, distance, false);
 		    if (rmsOne > vdwLevel && printVDWViolation)

			System.out.println(id1+"  "+id2+"  "+Math.sqrt(rmsOne)+"  "+distance);

		    rms += Math.sqrt(rmsOne);
		}
	    }
	}
	vdwValue[0] = rms;
	return true;
    }

    /**
     * Compute the vdw energy.
     * 
     * @param vdwVec coordinate object in a flat form
     * @param vdwValue for returning the computed vdwEnergy
     * @param vdwLevel if the vdwEnergy > vdwLevel and printVDWViolation
     * is true, print out the violation for checking
     * @param printVDWViolation whether to print the vdw violation
     * @param isHelix the is helix
     * 
     * @return true, if compute vdw
     */
    public boolean computeVDW (Vector vdwVec, double[] vdwValue, double vdwLevel, 
			       boolean printVDWViolation, boolean isHelix){
	String id1, id2, name1, name2;
	int index = -1;
	double[] coord1 = new double[3];
	double[] coord2 = new double[3];
	vdw vander1 = new vdw();
	vdw vander2 = new vdw();
	int i, j;
	double rms = 0.0;
	double distance = 0.0;
	double rmsOne = 0.0;
	String resid ="";
	int no1, no2; //residue No.
	boolean isHbond = false;
	for (i=0; i<vdwVec.size(); i++){
	    vander1 = (vdw)vdwVec.elementAt(i);
	    id1 = vander1.getID();
	   	    
	    resid = id1.substring(0, 3);

	    index = id1.indexOf("_");
	    no1 = Integer.parseInt((id1.substring(3, index)).trim()); 
	    if (index > -1)
		name1 = id1.substring(index+1);
	    else return false;
	    coord1 = vander1.getCoord();
	    for (j=i+1; j<vdwVec.size(); j++){
		vander2 = (vdw)vdwVec.elementAt(j);
		id2 = vander2.getID();
				
		
		index = id2.indexOf("_");
		no2 = Integer.parseInt((id2.substring(3, index)).trim()); 
		if (index > -1)
		    name2 = id2.substring(index+1);
		else return false;
		coord2 = vander2.getCoord();
		distance = length(internuclearVec(coord2, coord1));
		
				
		isHbond = false;
  		if(! isBonded(resid, no1, no2, name1, name2)) {
		    if(isHelix && name1.equals("O") && name2.equals("H") && (no2 - no1) == 4)
			isHbond = true;
		    rmsOne = checkVDW(name1, name2, distance, isHbond);
 		    if (rmsOne > vdwLevel && printVDWViolation)

			System.out.println(id1+"  "+id2+"  "+Math.sqrt(rmsOne)+"   "+distance);
		    rms += Math.sqrt(rmsOne);
		}
	    }
	}
	vdwValue[0] = rms;
	return true;
    }
    
    /**
     * check the steric clashes, if the number of steric clashes is larger than some threshold,
     * ruturn true, else return false.
     * 
     * @param vdwVec coordinate object in a flat form
     * @param vdwValue for returning the computed vdwEnergy
     * @param vdwLevel if the vdwEnergy > vdwLevel and printVDWViolation
     * is true, print out the violation for checking
     * @param printVDWViolation whether to print the vdw violation
     * @param isHelix whether it is helix
     * @param stericThreshold the distance threshold for checking steric clash (default:0.6A)
     * @param numThreshold the number threshold for reporting steric clashes(default:1)
     * @param resClash store the residue indexes of steric clash, true if the residue envolves in steric clash
     * @param isPrint the is print
     * 
     * @return the int
     */
    public int OutputStericClashResidues (Vector vdwVec, double[] vdwValue, double vdwLevel, 
			       boolean printVDWViolation, boolean isHelix, double stericThreshold,
			       int numThreshold, boolean[] resClash,boolean isPrint)
    {
    	int nClashRes=0;
		String id1, id2, name1, name2;
		int index = -1;
		double[] coord1 = new double[3];
		double[] coord2 = new double[3];
		vdw vander1 = new vdw();
		vdw vander2 = new vdw();
		int i, j;
		double rms = 0.0;
		double distance = 0.0;
		double rmsOne = 0.0;
		String resid ="";
		int no1, no2; //residue No.
		boolean isHbond = false;
		int numClashes=0;
		for (i=0;i<resClash.length;i++)
		{
			resClash[i]=false;
		}
	
		for (i=0; i<vdwVec.size(); i++)
		{
		    vander1 = (vdw)vdwVec.elementAt(i);
		    id1 = vander1.getID();
		   	    
		    resid = id1.substring(0, 3);
	
		    index = id1.indexOf("_");
		    no1 = Integer.parseInt((id1.substring(3, index)).trim()); 
		    if (index > -1)
			name1 = id1.substring(index+1);
		    else continue;
		    coord1 = vander1.getCoord();
		    for (j=i+1; j<vdwVec.size(); j++)
		    {
				vander2 = (vdw)vdwVec.elementAt(j);
				id2 = vander2.getID();				
				numClashes=0;
				
				index = id2.indexOf("_");
				no2 = Integer.parseInt((id2.substring(3, index)).trim()); 
				if (index > -1)
				    name2 = id2.substring(index+1);
				else continue;
				coord2 = vander2.getCoord();
				distance = length(internuclearVec(coord2, coord1));
				//distance = distance+0.5;//we add 0.5 penatuation //////
						
				isHbond = false;
		  		if(! isBonded(resid, no1, no2, name1, name2)) 
		  		{
				    if(isHelix && name1.equals("O") && name2.equals("H") && (no2 - no1) == 4)
					isHbond = true;
				    rmsOne = checkVDW(name1, name2, distance, isHbond);
				    
				    double rmsd=Math.sqrt(rmsOne);
				    if(rmsd>stericThreshold)
				    	numClashes+=1;
				    
				    if (numClashes>=numThreshold)
				    {
				    	resClash[no1]=true;
				    	resClash[no2]=true;				    	
				    }
		 		    if (rmsOne > vdwLevel && printVDWViolation)
		            //  		    if (rmsOne > 0.1)
					System.out.println(id1+"  "+id2+"  "+Math.sqrt(rmsOne)+"   "+distance);
				    rms += Math.sqrt(rmsOne);
				}
		    }
		}
		vdwValue[0] = rms;
		
		
		for (i=0;i<resClash.length;i++)
		{
			if(resClash[i])
				nClashRes++;			
		}
		String strOut="";
		if(isPrint)
		{
			
			for (i=0;i<resClash.length;i++)
			{
				if(resClash[i])
					strOut=strOut+" , "+ i;			
			}
			
			System.out.println("Clash residues: "+ strOut);
		}//if(isPrint)
		
		
		return nClashRes;
    }

   
    /**
     * check the steric clashes, if the number of steric clashes is larger than some threshold,
     * ruturn true, else return false.
     * 
     * @param vdwVec coordinate object in a flat form
     * @param vdwValue for returning the computed vdwEnergy
     * @param vdwLevel if the vdwEnergy > vdwLevel and printVDWViolation
     * is true, print out the violation for checking
     * @param printVDWViolation whether to print the vdw violation
     * @param isHelix whether it is helix
     * @param stericThreshold the distance threshold for checking steric clash (default:0.6A)
     * @param numThreshold the number threshold for reporting steric clashes(default:1)
     * @param clashIndex store the residue indexes of steric clash
     * 
     * @return true, if check steric clash
     */
    public boolean checkStericClash (Vector vdwVec, double[] vdwValue, double vdwLevel, 
			       boolean printVDWViolation, boolean isHelix, double stericThreshold,
			       int numThreshold, int[] clashIndex)
    {
	String id1, id2, name1, name2;
	int index = -1;
	double[] coord1 = new double[3];
	double[] coord2 = new double[3];
	vdw vander1 = new vdw();
	vdw vander2 = new vdw();
	int i, j;
	double rms = 0.0;
	double distance = 0.0;
	double rmsOne = 0.0;
	String resid ="";
	int no1, no2; //residue No.
	boolean isHbond = false;
	int numClashes=0;
	
	for (i=0; i<vdwVec.size(); i++){
	    vander1 = (vdw)vdwVec.elementAt(i);
	    id1 = vander1.getID();
	   	    
	    resid = id1.substring(0, 3);

	    index = id1.indexOf("_");
	    no1 = Integer.parseInt((id1.substring(3, index)).trim()); 
	    if (index > -1)
		name1 = id1.substring(index+1);
	    else continue;
	    coord1 = vander1.getCoord();
	    for (j=i+1; j<vdwVec.size(); j++){
		vander2 = (vdw)vdwVec.elementAt(j);
		id2 = vander2.getID();				
		
		index = id2.indexOf("_");
		no2 = Integer.parseInt((id2.substring(3, index)).trim()); 
		if (index > -1)
		    name2 = id2.substring(index+1);
		else continue;
		coord2 = vander2.getCoord();
		distance = length(internuclearVec(coord2, coord1));
				
		isHbond = false;
  		if(! isBonded(resid, no1, no2, name1, name2)) {
		    if(isHelix && name1.equals("O") && name2.equals("H") && (no2 - no1) == 4)
			isHbond = true;
		    rmsOne = checkVDW(name1, name2, distance, isHbond);
		    
		    double rmsd=Math.sqrt(rmsOne);
		    if(rmsd>stericThreshold)
		    	numClashes+=1;
		    
		    if (numClashes>=numThreshold)
		    {
		    	clashIndex[0]=no1;
		    	clashIndex[1]=no2;
		    	return true;
		    }
 		    if (rmsOne > vdwLevel && printVDWViolation)

			System.out.println(id1+"  "+id2+"  "+Math.sqrt(rmsOne)+"   "+distance);
		    rms += Math.sqrt(rmsOne);
		}
	    }
	}
	vdwValue[0] = rms;
	
	clashIndex[0]=0;
	clashIndex[1]=0;
	return false;
    }

    /**
     * count the number of steric clash, given the threshold of penetration,
     * ruturn true, else return false.
     * 
     * @param vdwVec coordinate object in a flat form
     * @param vdwValue for returning the computed vdwEnergy
     * @param vdwLevel if the vdwEnergy > vdwLevel and printVDWViolation
     * is true, print out the violation for checking. Not used here.
     * @param printVDWViolation whether to print the vdw violation. Not used here.
     * @param isHelix whether it is helix
     * @param stericThreshold the distance threshold for checking steric clash (default:0.6A)
     * 
     * @return the int
     */
    public int countStericClash (Vector vdwVec, double[] vdwValue, double vdwLevel, 
			       boolean printVDWViolation, boolean isHelix, double stericThreshold)
    {
	String id1, id2, name1, name2;
	int index = -1;
	double[] coord1 = new double[3];
	double[] coord2 = new double[3];
	vdw vander1 = new vdw();
	vdw vander2 = new vdw();
	int i, j;
	double rms = 0.0;
	double distance = 0.0;
	double rmsOne = 0.0;
	String resid ="";
	int no1, no2; //residue No.
	boolean isHbond = false;
	int numClashes=0;
	
	for (i=0; i<vdwVec.size(); i++)
	{
	    vander1 = (vdw)vdwVec.elementAt(i);
	    id1 = vander1.getID();
	   	    
	    resid = id1.substring(0, 3);

	    index = id1.indexOf("_");
	    no1 = Integer.parseInt((id1.substring(3, index)).trim()); 
	    if (index > -1)
		name1 = id1.substring(index+1);
	    else continue;
	    coord1 = vander1.getCoord();
	    for (j=i+1; j<vdwVec.size(); j++){
		vander2 = (vdw)vdwVec.elementAt(j);
		id2 = vander2.getID();				
		
		index = id2.indexOf("_");
		no2 = Integer.parseInt((id2.substring(3, index)).trim()); 
		if (index > -1)
		    name2 = id2.substring(index+1);
		else continue;
		coord2 = vander2.getCoord();
		distance = length(internuclearVec(coord2, coord1));
				
		isHbond = false;
  		if(! isBonded(resid, no1, no2, name1, name2)) 
  		{
		    if(isHelix && name1.equals("O") && (name2.equals("H")|| name2.equals("HN"))&& (no2 - no1) == 4)
		    	isHbond = true;
		    if(isHelix && name2.equals("O") && (name1.equals("H")|| name1.equals("HN"))&& (no1 - no2) == 4)
		    	isHbond = true;
		    
		    rmsOne = checkVDW(name1, name2, distance, isHbond);
		    
		    double rmsd=Math.sqrt(rmsOne);
		    if(rmsd>stericThreshold)
		    	numClashes+=1;		    
		    
 		    if (rmsOne > vdwLevel && printVDWViolation)

			System.out.println(id1+"  "+id2+"  "+Math.sqrt(rmsOne)+"   "+distance);
		    rms += Math.sqrt(rmsOne);
		}
	    }
	}
	vdwValue[0] = rms;

	return numClashes;
    }
    
    /**
     * count the number of steric clash between two structures, given the threshold of penetration,
     * ruturn true, else return false.
     * 
     * @param vdwVec coordinate object in a flat form
     * @param vdwValue for returning the computed vdwEnergy
     * @param vdwLevel if the vdwEnergy > vdwLevel and printVDWViolation
     * is true, print out the violation for checking. Not used here.
     * @param printVDWViolation whether to print the vdw violation. Not used here.
     * @param isHelix whether it is helix
     * @param stericThreshold the distance threshold for checking steric clash (default:0.6A)
     * @param vdwVec2 the vdw vec2
     * 
     * @return the int
     */
    public int countStericClashBetweenTwo (Vector vdwVec,Vector vdwVec2, double[] vdwValue, double vdwLevel, 
			       boolean printVDWViolation, boolean isHelix, double stericThreshold)
    {
	String id1, id2, name1, name2;
	int index = -1;
	double[] coord1 = new double[3];
	double[] coord2 = new double[3];
	vdw vander1 = new vdw();
	vdw vander2 = new vdw();
	int i, j;
	double rms = 0.0;
	double distance = 0.0;
	double rmsOne = 0.0;
	String resid ="";
	int no1, no2; //residue No.
	boolean isHbond = false;
	int numClashes=0;
	
	for (i=0; i<vdwVec.size(); i++)
	{
	    vander1 = (vdw)vdwVec.elementAt(i);
	    id1 = vander1.getID();
	   	    
	    resid = id1.substring(0, 3);

	    index = id1.indexOf("_");
	    no1 = Integer.parseInt((id1.substring(3, index)).trim()); 
	    if (index > -1)
		name1 = id1.substring(index+1);
	    else continue;
	    coord1 = vander1.getCoord();
	    for (j=i+1; j<vdwVec2.size(); j++)
	    {
			vander2 = (vdw)vdwVec2.elementAt(j);
			id2 = vander2.getID();				
			
			index = id2.indexOf("_");
			no2 = Integer.parseInt((id2.substring(3, index)).trim()); 
			if (index > -1)
			    name2 = id2.substring(index+1);
			else continue;
			coord2 = vander2.getCoord();
			distance = length(internuclearVec(coord2, coord1));
				
			isHbond = false;
	  		if(! isBonded(resid, no1, no2, name1, name2)) 
	  		{
			    if(isHelix && name1.equals("O") && (name2.equals("H")|| name2.equals("HN"))&& (no2 - no1) == 4)
			    	isHbond = true;
			    if(isHelix && name2.equals("O") && (name1.equals("H")|| name1.equals("HN"))&& (no1 - no2) == 4)
			    	isHbond = true;
			    
			    rmsOne = checkVDW(name1, name2, distance, isHbond);
			    
			    double rmsd=Math.sqrt(rmsOne);
			    if(rmsd>stericThreshold)
			    	numClashes+=1;		    
			    
	 		    if (rmsOne > vdwLevel && printVDWViolation)

				System.out.println(id1+"  "+id2+"  "+Math.sqrt(rmsOne)+"   "+distance);
			    rms += Math.sqrt(rmsOne);
			}
	    }
	}
	vdwValue[0] = rms;

	return numClashes;
    }

    /**
     * Only for polyalanine.
     * 
     * @param no1 the no1
     * @param no2 the no2
     * @param atom1 the atom1
     * @param atom2 the atom2
     * 
     * @return true, if checks if is bonded
     */
    public boolean isBonded(int no1, int no2, String atom1, String atom2){
	if (no1 == no2){
	    if (atom1.equals("H")){
		if(atom2.equals("N"))
		    return true;
		else return false;
	    }else if(atom1.equals("N")){
		if(atom2.equals("H") || atom2.equals("CA") )
		    return true;
		else return false;
	    }else if(atom1.equals("CA")){
		if(atom2.equals("N")|| atom2.equals("C")|| atom2.equals("HA")|| atom2.equals("CB") )
		    return true;
		else return false;
	    }else if(atom1.equals("C")){
		if(atom2.equals("CA") || atom2.equals("O"))
		    return true;
		else return false;
	    }else if(atom1.equals("HA")){
		if(atom2.equals("CA"))
		    return true;
		else return false;
	    }else if(atom1.equals("CB")){
		if(atom2.equals("CA")||atom2.equals("1HB") || atom2.equals("2HB") || atom2.equals("3HB"))
		    return true;
		else return false;
	    }else if( atom1.indexOf("HB") > -1 && atom2.indexOf("HB") > -1)
		return true;
	}else if ( (no2 - no1) == 1){ //next residue (peptide plane)
	    if(atom1.equals("C")){
		if(atom2.equals("N"))
		    return true;
		else return false;
	    }
	}else if ((no1 - no2 ) == 1){ //previous residue (peptide plane)
	    if(atom1.equals("N")){
		if(atom2.equals("C"))
		    return true;
		else return false;
	    }
	}
	return false;
    }

    /**
     * Checks if is bonded.
     * 
     * @param resid the resid
     * @param no1 the no1
     * @param no2 the no2
     * @param atom1 the atom1
     * @param atom2 the atom2
     * 
     * @return true, if is bonded
     */
    public boolean isBonded(String resid, int no1, int no2, String atom1, String atom2){
	if (no1 == no2){
	    if(Const.bondsMap.containsKey(resid)){
		Vector bondsVec = (Vector)Const.bondsMap.get(resid);

 		if(Collections.binarySearch(bondsVec, atom1+"_"+atom2) > -1)
 		    return true;
 	    }else  
 	    	System.out.println("no such residue "+resid);
	}else if ((no2 - no1 ) == 1){ //next residue (peptide plane)

 	    if(atom1.equals("C")){
 		if(atom2.equals("N"))
 		    return true;
 		else return false;
 	    }
	}else if ((no1 - no2 ) == 1){ //previous residue (peptide plane)

	    if(atom1.equals("N")){
		if(atom2.equals("C"))
		    return true;
		else return false;
	    }
	}
	return false;
    }

    /**
     * Check vdw.
     * 
     * @param atom1 the atom1
     * @param atom2 the atom2
     * @param distance the distance
     * @param isHbond the is hbond
     * 
     * @return the double
     */
    public double checkVDW(String atom1, String atom2, double distance, boolean isHbond){
	int i, j, k;
	double rms = 0.0;
	if( atom1.indexOf("H") > -1 && atom2.indexOf("H") > -1 ){ 
	    //though the argine sidechain has name "NH1, NH2" for N but that will not cause problems.
	    //the distance between them is:> 2.2
	    if(distance > Const.vdwH2H)
		rms = 0.0;
	    else
		rms = (Const.vdwH2H-distance) * (Const.vdwH2H-distance);
	}else if (atom1.indexOf("C") > -1 && atom2.indexOf("C") > -1 ){
	    if(distance > Const.vdwC2C)
		rms = 0.0;
	    else
		rms = (Const.vdwC2C-distance) * (Const.vdwC2C-distance);
	}else if (atom1.indexOf("N") > -1 && atom2.indexOf("N") > -1 ){
	    if(distance > Const.vdwN2N)
		rms = 0.0;
	    else
		rms = (Const.vdwN2N-distance) * (Const.vdwN2N-distance);
	}else if( atom1.indexOf("O") > -1 && atom2.indexOf("O") > -1 ){
	    if(distance > Const.vdwO2O)
		rms = 0.0;
	    else{
		rms = (Const.vdwO2O-distance) * (Const.vdwO2O-distance);

	    }
	}else if( atom1.indexOf("S") > -1 && atom2.indexOf("S") > -1 ){
	    if(distance > Const.vdwS2S)
		rms = 0.0;
	    else
		rms = (Const.vdwS2S-distance) * (Const.vdwS2S-distance);
	}
	else if ((atom1.indexOf("H") > -1 && atom2.indexOf("O") > -1) ||
		  (atom1.indexOf("O") > -1 && atom2.indexOf("H") > -1))
	{
	    if(isHbond)
	    {
			if(distance > Const.hBondH2O)
			    rms = 0.0;
			else 
				rms = (Const.hBondH2O-distance) * (Const.hBondH2O-distance);
		 }else
		 {
		if(distance > Const.vdwH2O)
		    rms = 0.0;
		else rms = (Const.vdwH2O-distance) * (Const.vdwH2O-distance);
	    }
	}else if ((atom1.indexOf("H") > -1 && atom2.indexOf("C") > -1) ||
		  (atom1.indexOf("C") > -1 && atom2.indexOf("H") > -1)) {
	    if(distance > Const.vdwH2C)
		rms = 0.0;
	    else
		rms = (Const.vdwH2C-distance) * (Const.vdwH2C-distance);
	}else if ((atom1.indexOf("H") > -1 && atom2.indexOf("N") > -1) ||
		  (atom1.indexOf("N") > -1 && atom2.indexOf("H") > -1)) {
	    //though the argine sidechain has name "NH1, NH2" for N but that will not cause problems.
	    //the distance between them is:> 2.2
	    if(distance > Const.vdwH2N)
		rms = 0.0;
	    else
		rms = (Const.vdwH2N-distance) * (Const.vdwH2N-distance);
	}else if ((atom1.indexOf("H") > -1 && atom2.indexOf("S") > -1) ||
		  (atom1.indexOf("S") > -1 && atom2.indexOf("H") > -1)) {
	    if(distance > Const.vdwH2S)
		rms = 0.0;
	    else
		rms = (Const.vdwH2S-distance) * (Const.vdwH2S-distance);
	}else if ((atom1.indexOf("C") > -1 && atom2.indexOf("O") > -1) ||
		  (atom1.indexOf("O") > -1 && atom2.indexOf("C") > -1)) {
	    if(distance > Const.vdwC2O)
		rms = 0.0;
	    else
		rms = (Const.vdwC2O-distance) * (Const.vdwC2O-distance);
	}else if ((atom1.indexOf("N") > -1 && atom2.indexOf("O") > -1) ||
		  (atom1.indexOf("O") > -1 && atom2.indexOf("N") > -1)) {
	    if(distance > Const.vdwN2O)
		rms = 0.0;
	    else
		rms = (Const.vdwN2O-distance) * (Const.vdwN2O-distance);
	}else if ((atom1.indexOf("N") > -1 && atom2.indexOf("C") > -1) ||
		  (atom1.indexOf("C") > -1 && atom2.indexOf("N") > -1)) {
	    if(distance > Const.vdwN2C)
		rms = 0.0;
	    else
		rms = (Const.vdwN2C-distance) * (Const.vdwN2C-distance);
	}else if ((atom1.indexOf("C") > -1 && atom2.indexOf("S") > -1) ||
		  (atom1.indexOf("S") > -1 && atom2.indexOf("C") > -1)) {
	    if(distance > Const.vdwC2S)
		rms = 0.0;
	    else
		rms = (Const.vdwC2S-distance) * (Const.vdwC2S-distance);
	}else if ((atom1.indexOf("N") > -1 && atom2.indexOf("S") > -1) ||
		  (atom1.indexOf("S") > -1 && atom2.indexOf("N") > -1)) {
	    if(distance > Const.vdwN2S)
		rms = 0.0;
	    else
		rms = (Const.vdwN2S-distance) * (Const.vdwN2S-distance);
	}else if ((atom1.indexOf("O") > -1 && atom2.indexOf("S") > -1) ||
		  (atom1.indexOf("S") > -1 && atom2.indexOf("O") > -1)) {
	    if(distance > Const.vdwO2S)
		rms = 0.0;
	    else
		rms = (Const.vdwO2S-distance) * (Const.vdwO2S-distance);
 	}else if (atom1.indexOf("Q") > -1 || atom2.indexOf("Q") > - 1 ){
 		rms = 0.0;
	}else System.out.println("NO such atoms in checkVDW  "+atom1+"   "+atom2);
	return rms;
    }
    
    

 
    /**
     * merge all structures in previous ensemble into one enemble (with the same pdb name)
     * and cluster them according to given resolution.
     * 
     * @param src current input directory
     * @param strPreEnsemb the common name in the old ensembles
     * Note: the prvious ensemble ID and structure ID are assumely delimited by "000"
     * @param strNewEnsemb the new name of the clutered ensembles
     * @param resolution the resolution for the clustering step.
     * @param maxStr max number of structures in previous each ensemble
     * @param maxEnsemSize max number of in previous ensembles
     * @param strDelimit the delimited string between enesmble and structure IDs
     * @param srcOut the src out
     * 
     * @throws JampackException the jampack exception
     */
	public void mergeAndCluster(String src, String srcOut, String strPreEnsemb,
			int maxStr, int maxEnsemSize, double resolution, String strNewEnsemb,String strDelimit) throws JampackException
	{
		long startTime = System.currentTimeMillis();		
		Pdb  pp = new Pdb();    	    	
    	Vector vecPdb=new Vector();
    	Vector vecVdw=new Vector();
    	Vector vecEnsemPdbNew=new Vector();    	
    	Model md=new Model(); 	
    	
    	////////////////////////////////////////////
    	//1. we thin the ensemble by only output the representive structures
    	///////////////////////////////////////////
    	int counter=0;  	
    	Vector vecEnsemAfterThin=new Vector();
    	Vector vecRefPdb=new Vector();
    	for (int t=0;t<maxEnsemSize;t++)
    	{    			
	    	for (int j=0;j<=maxStr;j++)
	    	{   		
	    		String strEnsemPdb="";
	    		if (maxEnsemSize<=1)
	    			strEnsemPdb=src+strPreEnsemb+j+".pdb";    	
	    		else 
	    			strEnsemPdb=src+strPreEnsemb+t+strDelimit+j+".pdb";    	
	    		
		    	
		    	if(Const.isDebug)
		    		System.out.println("input: "+strEnsemPdb);
		    	
		    	File f = new File(strEnsemPdb);    		 
	    		if (!f.exists())
	    		{
	    			if(Const.isDebug)
	    				 System.out.println("Couldn't find file "+ strEnsemPdb+ " for clustering, continue...");
	    			 continue;
	    		}
	    		Vector vecEnsemble=pp.readPdbAndNoeHarmFromEnsemble(strEnsemPdb);
		    	if (vecEnsemble==null)
		    		continue;   	
		    		    		
		    	for(int i=0;i<vecEnsemble.size();i++)
		    	{	    		
		    		vecVdw=new Vector();
		    		vdw Vdw=(vdw)vecEnsemble.elementAt(i);
		    		vecPdb=Vdw.getPDB();
		    		if (vecPdb.size()<1)
		    			continue;
		    		
		    		double noeRms=Vdw.getNoeRmsd();
		    		Vector vecTempPdbBB=pp.OutputBackbone(vecPdb);	    	  
		    		
		    		//consider the structure clutering
		    		if( vecEnsemAfterThin.size()==1)
		    		{
		    			Vector vecTempPdb=(Vector)vecEnsemAfterThin.elementAt(0);
		    			vecRefPdb.addAll(vecTempPdb);
		    		}
		    		Vector vecPdbRotate=new Vector();
		    		if( vecEnsemAfterThin.size()>1)
		    			vecPdbRotate=md.BackonbeRotation(vecPdb,vecRefPdb);	  
		    		else
		    			vecPdbRotate=vecPdb;
		    		
		    		boolean isInPreClusters=pp.isInPdbClusters(vecEnsemAfterThin,vecPdbRotate,resolution);///////////
		    		if(isInPreClusters)
		    			continue;
		    		vecEnsemAfterThin.add(vecPdbRotate);    		
		    		
		        	String fileName =srcOut+ strNewEnsemb+counter+".pdb";
			    	
			    	//output all top structures into one pdb file
			    	try
			    	{
			    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
			    		out.println("REMARK  "+ "   the structure index: " + counter);			    	   
		        	    pp.printToFile(vecPdb,fileName, out);				    	 	         	
		        	    out.println("TER");
		        	         	
		        	    out.println("END");  
		        	    out.close();
		    		
		        	    counter+=1;
					}catch (FileNotFoundException e)
					{
						System.out.println("File not found: " + fileName);
					}catch (IOException e)
					{
					   System.out.println("IOException: the stack trace is:");
					   e.printStackTrace();
					}
		    	    
		    	}//for(int i=0;i<vecEnsemble.size();i++)
	    	
    		}//for (int j=0;j<=maxStr;j++)
    	}//for (int t=0;t<maxEnsemble;t++)

		long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Time= "+ totalTime +" minutes");
    			
	}
    
    /**
     * similar to "mergeAndCluster", but cluster all structures in one
     * directory as an ensemble
     * merge all structures in previous ensemble into one enemble (with the same pdb name)
     * and cluster them according to given resolution.
     * 
     * @param src the input directory of all structures in the ensemble
     * @param strNewEnsemb the new name of the clutered ensembles
     * @param resolution the resolution for the clustering step.
     * @param srcOut the src out
     * 
     * @throws JampackException the jampack exception
     */
	public void mergeAndClusterAll(String src, String srcOut, 
			 double resolution, String strNewEnsemb) throws JampackException
	{
		long startTime = System.currentTimeMillis();		
		Pdb  pp = new Pdb();    	    	
    	Vector vecPdb=new Vector();
    	Vector vecVdw=new Vector();
    	Vector vecEnsemPdbNew=new Vector();    	
    	Model md=new Model(); 	
    	String userDir = System.getProperty("user.dir");////
    	////////////////////////////////////////////
    	//1. we thin the ensemble by only output the representive structures
    	///////////////////////////////////////////
    	File myDir = new File(src); 
    	String[] contents = myDir.list();//select all structures
    	if (contents == null) 
    		System.out.println(myDir.getName() + " is not a directory");   	
    	
    	int counter=0;  	
    	Vector vecEnsemAfterThin=new Vector();
    	Vector vecRefPdb=new Vector();
    	//for (int t=0;t<maxEnsemSize;t++)
    	for (int t=0; t<contents.length;t++) 
    	{    
    		String filename = contents[t]; 
	    	//for (int j=0;j<=maxStr;j++)
	    	//{   		
	    	String strEnsemPdb="";
	    	strEnsemPdb=src+filename;	    
		    	
	    	if(Const.isDebug)
	    		System.out.println("input: "+strEnsemPdb);
	    	
	    	File f = new File(strEnsemPdb);    		 
    		if (!f.exists())
    		{
    			if(Const.isDebug)
    				 System.out.println("Couldn't find file "+ strEnsemPdb+ " for clustering, continue...");
    			 continue;
    		}
    		Vector vecEnsemble=pp.readPdbAndNoeHarmFromEnsemble(strEnsemPdb);
	    	if (vecEnsemble==null)
	    		continue;   	
	    		    		
	    	for(int i=0;i<vecEnsemble.size();i++)
	    	{	    		
	    		vecVdw=new Vector();
	    		vdw Vdw=(vdw)vecEnsemble.elementAt(i);
	    		vecPdb=Vdw.getPDB();
	    		if (vecPdb.size()<1)
	    			continue;
	    		
	    		double noeRms=Vdw.getNoeRmsd();
	    		Vector vecTempPdbBB=pp.OutputBackbone(vecPdb);	    	  
	    		
	    		
	        	
	    		//consider the structure clutering
	    		if( vecEnsemAfterThin.size()==1)
	    		{
	    			Vector vecTempPdb=(Vector)vecEnsemAfterThin.elementAt(0);
	    			vecRefPdb.addAll(vecTempPdb);
	    		}
	    		Vector vecPdbRotate=new Vector();
	    		if( vecEnsemAfterThin.size()>1)
	    			vecPdbRotate=md.BackonbeRotation(vecPdb,vecRefPdb);	  
	    		else
	    			vecPdbRotate=vecPdb;
	    		
	    		//////////////////////////////////////////
	    		//check the clustering
	    		//boolean isInPreClusters=pp.isInPdbClusters(vecEnsemAfterThin,vecPdbRotate,resolution*2);//////resolution
	    		////////////////////////////////
	    		boolean isInPreClusters=false;
	    		//read already clustered ensemble
	    		vecEnsemAfterThin=new Vector();	    		
	        	String outPackPath=userDir+strNewEnsemb;
	        	int idPath=outPackPath.lastIndexOf('/');
	        	String outPackDir=outPackPath.substring(0, idPath+1);
	        	File DirNewEnsem = new File(outPackDir);
	        	String[] contentsNew = DirNewEnsem.list();//select all structures
	        	for (int x=0; x<contentsNew.length;x++) 
	        	{    
	        		String filenameNew = contentsNew[x]; 
	        		String strEnsemPdbNew="";
	        		strEnsemPdbNew=outPackDir+filenameNew;	 		    	
	    	    	   	
	    	    	File f_new = new File(strEnsemPdbNew);    		 
	        		if (!f_new.exists())
	        		{
	        			if(Const.isDebug)
	        				 System.out.println("Couldn't find file "+ strEnsemPdbNew+ " for clustering, continue...");
	        			 continue;
	        		}
	        		Vector vecEnsembleNew=pp.readPdbAndNoeHarmFromEnsemble(strEnsemPdbNew);
	    	    	if (vecEnsembleNew==null)
	    	    		continue;  
	    	    	vdw VdwNew=(vdw)vecEnsembleNew.elementAt(0);
	    	    	Vector vecPdbNew =VdwNew.getPDB();
	    	    	//vecEnsemAfterThin.add(vecPdbNew);
	    	    	
	    	    	double rmsd=pp.compRmsdBetwTwoPdbs(vecPdbNew,vecPdbRotate,true);
	    	    	double resolutionNew=resolution*2;
	        		if(rmsd< resolutionNew)
	        		{
	        			isInPreClusters=true;    	
	        			break;
	        		}
	        	}//for (int x=0; x<contents.length;x++) 
	        	////////////////////////////////
	    		    		
	    		if(isInPreClusters)
	    			continue;
	    		//vecEnsemAfterThin.add(vecPdbRotate);    		
	    		//////////////////////////////////////////
	    		
	        	String fileName =userDir+ strNewEnsemb+counter+".pdb";		    	
		    	//output all top structures into one pdb file
		    	try
		    	{
		    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
		    		out.println("REMARK  "+ "   the structure index: " + counter);
		    	  
	        	    pp.printToFile(vecPdb,fileName, out);			    	      	
	        	    out.println("TER");	        	          	
	        	    out.println("END");  
	        	    out.close();
	    		
	        	    counter+=1;
				}catch (FileNotFoundException e)
				{
					System.out.println("File not found: " + fileName);
				}catch (IOException e)
				{
				   System.out.println("IOException: the stack trace is:");
				   e.printStackTrace();
				}
	    	    
	    	}//for(int i=0;i<vecEnsemble.size();i++)    	
    	}//for (int t=0; t<contents.length;t++) 

		long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes    
    			
	}
	
	//public static void main(String[] argv) throws JampackException	
	//add the rotamers to backbones, and refine the rotamers to avoid steric clashes
	//note: be careful about the atom naming scheme	
	/**
	 * The main method.
	 * 
	 * @param argv the arguments
	 * 
	 * @throws JampackException the jampack exception
	 */
	public static void main(String[] argv) throws JampackException
	{
		long startTime = System.currentTimeMillis();
		int chosenNum=25;//size of ensemble
		Pdb  pp = new Pdb();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";		
		
		////////////////////////////////////////////////
		Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	String rotSrc=  src+ "rotasamp-small/"; 
    	//String src   = "../inputFiles/"; //The directory has the input files
    
		String paramFile=src+"parameters.input";
    	Vector<Map<String, String>> paraVec = asg.ParamReader(paramFile);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	double noeBound=0.0;
    	String strSeq="", strBB="", strReson="", strNoesy2D="", strHnNoesy3D="", 
    		strHaNoesy3D="",strManualAsg="", strSSES="";
    	for (int i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		if(paraMap.containsKey("NOEBOUND"))
    			noeBound  =  Double.parseDouble((String)paraMap.get("NOEBOUND"));
    		
    		//get the input file name
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		strSeq=strSeq.toLowerCase();
    		if(paraMap.containsKey("BACKBONE"))
    			strBB  =  (String)paraMap.get("BACKBONE");
    		//strBB=strBB.toLowerCase();
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		strNoesy2D=strNoesy2D.toLowerCase();
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		strHnNoesy3D=strHnNoesy3D.toLowerCase();
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		strHaNoesy3D=strHaNoesy3D.toLowerCase();
    		if(paraMap.containsKey("MANUAL-ASSIGNMENT"))
    			strManualAsg  =  (String)paraMap.get("MANUAL-ASSIGNMENT");
    		strManualAsg=strManualAsg.toLowerCase();
    		
    		if(paraMap.containsKey("SSES"))
    			strSSES  =  (String)paraMap.get("SSES");
    		//strSSES=strSSES.toLowerCase();
    		
    	}   	
    	   	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	//String seqFile = src+"cbf.seq";
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the backbone structure 
    	// (input file name: backbone.pdb)
    	// Note: the backbone must be connected. 
    	//-------------------------------------
    	//String pdbFile = src+"backbone.pdb"; 
    	String pdbFile = src+strBB;
    	//Vector pdbVecBB = pp.readPdb(pdbFile);

    	//update backbone atom names according to protein sequence input
    	//residue name in previous backbone may be all in "ALA".     		
    	//Vector pdbVec3 = pp.residueNameUpdate(vecSeq, pdbVecBB);
    	
    	// this is final backbone pdb used for hosting rotamers
    	//Vector pdbVecBBN = pp.nameConvert(pdbVec3);
    	//Collections.sort(pdbVecBBN, new Pdb.PdbComparator());
    	    	
    	//-------------------------------------
    	// (1.3) Read the resonance list 
    	// 
    	// in the "resonance.prot" format
    	//-------------------------------------
    	   	
    	//**this is for ubq assignment table assigned by Lincong**//
    	 /*	String assignFileUbq = src+"assignByMe.txt";
    	Vector assignVecUbq  = asg.asgReader3(assignFileUbq); */
    	
    	//following is for the general case
    	H1CS h1CS=new H1CS();
    	//String assignFile = src+ "resonance.prot";    
    	String assignFile = src+ strReson; //"resonance.prot"; 
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector assignVec=h1CS.h1CSReader(assignFile,vecSeq);  //different input format
    	//Vector assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	
    	//all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());
    	    	
    	//-------------------------------------
    	// (1.4) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector hnNoeVec=new Vector();//for storing 3d n15 NOESY peaks
    	Vector cnoeVec=new Vector();//for storing 3d c13 noesy peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		//hnNoeVec = noesy.NoesyReader(strNoeFile); 
    		hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile);
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		//cnoeVec=noesy.NoesyReader(strNoeFile);
    		cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	Vector vecNoesy=new Vector();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	
    	//intensity calibration
    	Vector vecNewNoesy=noesy.SetCalibration(vecNoesy);    	

		
		
    	String manualAsgFile = src+ strManualAsg; 
    	Noe noe_temp=new Noe();
    	Vector vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");
		
    	int numFiles=50;//number of files
    	Vector vecPdb=new Vector();
    	Vector vecVdw=new Vector();
    	Vector vecEnsemPdbNew=new Vector();
    	Vector vecVdwScoreNew=new Vector();
    	Vector vecVdwScore=new Vector();
    	Model md=new Model();
    	
    	String strNOEScore="[ ";
    	
    	////////////////////////////////////////////
    	//1. we thin the ensemble by only output the representive structures
    	///////////////////////////////////////////
    	int counter=0;  	
    	Vector vecEnsemAfterThin=new Vector();
    	Vector vecRefPdb=new Vector();
    	for (int j=13;j<=441;j++)
    	{
    		
    		
    		String strEnsemPdb=src+"FF2FinalNew"+j+".pdb";
	    	Vector vecEnsemble=pp.readPdbAndNoeHarmFromEnsemble(strEnsemPdb);
	    	System.out.println("input: "+strEnsemPdb);
	    	
	    	if (vecEnsemble==null)
	    		continue;
	    	
	    	double[] vdwScore=new double[vecEnsemble.size()];
	    	vdw vander = new vdw();
	    	Vector vdwVec = new Vector();
	    	double[] vdwValue = new double[1];
	    	boolean hasVDW = false;
	    	double vdwLevel = 0.05;
	    	boolean printVDWViolation = false;    	
	    	
	    		
	    	for(int i=0;i<vecEnsemble.size();i++)
	    	{
	    		
	    		vecVdw=new Vector();
	    		vdw Vdw=(vdw)vecEnsemble.elementAt(i);
	    		vecPdb=Vdw.getPDB();
	    		double noeRms=Vdw.getNoeRmsd();
	    		Vector vecTempPdbBB=pp.OutputBackbone(vecPdb);
	    		
	    		double [] noeRmsd=new double[2];
	    		double [] noeHarmScore=new double[2];
	    		int [] numConflicts=new int[2];
	    		/////////////////////////////////////
	    		//compute the NOE satisfition score:
	    		
	    	
	    		
	    		////////////////////////////////////////////////
	    		//replace non-gly and non-pro residues with alanine:
	    		//String rotSrc=  src+ "rotasamp-small/";    	
	    		boolean[] resIndex=new boolean[vecTempPdbBB.size()];
		    	for (int kk=0;kk<resIndex.length;kk++)
		    		resIndex[kk]=false;
		    	//String rotSrc=  src+ "rotasamp-small/";  
			    Vector  vecTempPdbALA=pp.AlaninizeStructure(vecTempPdbBB,resIndex,rotSrc);
    		
	    		Vector vecTempPdbRot=pp.RotamSelectAndStructure(2*h1Err, 2*nErr, 2*c13Err,vecTempPdbBB,assignVec,rotSrc,vecNewNoesy,noeBound, 1,0.0);
	    		
	    	
	    		
	    		////////////////////////////////////////
	    		// refine the rotamers to avoid steric clash
	    		boolean [] resClash=new boolean[140];
	    		vdwVec = vander.convert2VDW(vecTempPdbRot);
	        	vander.OutputStericClashResidues(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.4, 1, resClash,true);
	        	int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.8);
	        	Vector vecTemptt=new Vector();
	        	
	        	if(numClashes>0)
	        	{
		        	int firstClash=0;
		        	for (i=0;i<resClash.length;i++)
		        	{
		        		if (resClash[i])
		        		{
		        			firstClash=i;
		        			break;
		        		}
		        	}
		        	//Vector vecTemptt=pp.RefineRotamerSelectionSearch(vecPdb,firstClash,rotSrc,0);
		        	vecTemptt=pp.RefineRotamerSelectionSearch(vecTempPdbRot,firstClash,rotSrc,0);
	        	}
	        	else
	        	{
	        		vecTemptt=vecTempPdbRot;
	        	}
	    		
	    		////////////////////////////////////////   		
	    		
	    	
	        	String fileName =src+ "FF2FinalWSc"+j+".pdb";
		    	
		    	//output all top structures into one pdb file
		    	try
		    	{
		    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		
	    	   // vdwScore[i]=vdwValue[0];  
	    	    
	    	    //vecVdwScore.add(new vdw(vecPdb,vdwValue[0], noeRms) );
	    	   
	    	    out.println("MODEL"+i);///to file
        		//out.println("REMARK  "+ "   VDW energy: " + vdwValue[0]);///to file
        		//out.println("REMARK :   NOE RMSD=:" + noeRms);///to file
        		//out.println("REMARK :   NOE HARM=:" + noeRms);///to file
        		strNOEScore=strNOEScore+numConflicts[0]+ " , ";
        		
        		System.out.println("current ensemble ID: "+ j+ " noe Harm score: "+ noeRmsd[0] + " numbr of conflicts: "+ numConflicts[0]);
    			pp.printToFile(vecTemptt,fileName, out);
		    	//System.out.println("TER");
	         	//System.out.println("END");   
	         	
	             out.println("TER");
	         	out.println("ENDMDL"); 	         	
	         	out.println("END");  
	    		out.close();
	    		
	    		counter+=1;
			}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
	    	    
	    	}//for(int i=0;i<vecEnsemblePdb.size();i++)
    	
	    	
    	
    	}//for (int j=0;j<10;j++)
    	
    	//////////////////////////////////////////////////
    			
    	System.out.println("The NOE satisfication score is: ");
    	System.out.println("strNOEScore= " +  strNOEScore);
    	
		long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Time= "+ totalTime +" minutes");
    			
	}//public static void main(String[] argv) throws JampackException

	/**
	 * Measure vd w4 ensemble.
	 */
	public void measureVDW4Ensemble() 
	{
		long startTime = System.currentTimeMillis();
		int chosenNum=25;//size of ensemble
		
		Pdb  pp = new Pdb();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	int numFiles=50;//number of files
    	Vector vecPdb=new Vector();
    	Vector vecVdw=new Vector();
    	Vector vecEnsemPdbNew=new Vector();
    	Vector vecVdwScoreNew=new Vector();
    	Vector vecVdwScore=new Vector();
    	
    	////////////////////////////////////////////
    	//1. we thin the ensemble by only output the representive structures
    	///////////////////////////////////////////
    	
    	for (int j=0;j<=300;j++)
    	{
    		Vector vecEnsemAfterThin=new Vector();
    		String strEnsemPdb=src+"packing"+j+".pdb";
	    	Vector vecEnsemble=pp.readPdbAndNoeRmsdFromEnsemble(strEnsemPdb);
	    	System.out.println("input: "+strEnsemPdb);
	    	
	    	double[] vdwScore=new double[vecEnsemble.size()];
	    	vdw vander = new vdw();
	    	Vector vdwVec = new Vector();
	    	double[] vdwValue = new double[1];
	    	boolean hasVDW = false;
	    	double vdwLevel = 0.05;
	    	boolean printVDWViolation = false;
	    	
	    	String fileName =src+ "packingNew"+j+".pdb";
	    	
	    	//output all top structures into one pdb file
	    	try
	    	{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		
	    	for(int i=0;i<vecEnsemble.size();i++)
	    	{
	    		vecVdw=new Vector();
	    		vdw Vdw=(vdw)vecEnsemble.elementAt(i);
	    		vecPdb=Vdw.getPDB();
	    		double noeRms=Vdw.getNoeRmsd();	    		
	    	
	    		vdwVec = vander.convert2VDW(vecPdb);
	        	hasVDW = vander.computeVDW(vdwVec, vdwValue, vdwLevel, printVDWViolation, false);  
	    	    vdwScore[i]=vdwValue[0];  
	    	    
	    	    //vecVdwScore.add(new vdw(vecPdb,vdwValue[0], noeRms) );
	    	   
	    	    out.println("MODEL"+i);///to file
        		out.println("REMARK  "+ "   VDW energy: " + vdwValue[0]);///to file
        		out.println("REMARK :   NOE RMSD=:" + noeRms);///to file
        		   			
    			pp.printToFile(vecPdb,fileName, out);
		    	//System.out.println("TER");
	         	//System.out.println("END");   
	         	
	            out.println("TER");
	         	out.println("ENDMDL");    			
	    	    
	    	    
	    	}//for(int i=0;i<vecEnsemblePdb.size();i++)
    	
	    	out.println("END");  
    		out.close();
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
	    	
    	}//for (int j=0;j<10;j++)
    	   	
		long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Time= "+ totalTime +" minutes");
    			
	}//public static void main(String[] argv) throws JampackException



}

   
