package NascaGUI;

/*
NASCA NOE Assignment and Side-Chain Assignment  Software Version 1.0
Copyright (C) 2009-2011 Bruce Donald Lab, Duke University

NASCA is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the Free
Software Foundation, either version 3 of the License, or (at your option) any
later version.

NASCA is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details.

You should have received a copy of the GNU Lesser General Public License
along with this library; if not, see:
    <http://www.gnu.org/licenses/>.

There are additional restrictions imposed on the use and distribution of this
open-source code, including: (A) this header must be included in any
modification or extension of the code; (B) you are required to cite our
papers in any publications that use this code. The citation for the various
different modules of our software, together with a complete list of
requirements and restrictions are found in the document license.pdf enclosed
with this distribution.

Contact Info:
    Bruce R. Donald
    Duke University
    Department of Computer Science
    Levine Science Research Center (LSRC)
    Durham, NC 27708-0129
    USA
    email: www.cs.duke.edu/brd/

<signature of Bruce Donald>, 01 December, 2009
Bruce R. Donald, Professor of Computer Science and Biochemistry
*/
import java.io.File;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JMenuItem;
import javax.swing.JTextArea;
import javax.swing.SwingWorker;

import rdcPanda.Assign;
import rdcPanda.BackNoe;
import rdcPanda.H1CS;
import rdcPanda.Noe;
import rdcPanda.Noesy;
import rdcPanda.Pdb;
import rdcPanda.Peak;
import Jampack.JampackException;

import Nasca.AStar;
import Nasca.GraphNode;
import Nasca.NascaNew;
import Nasca.ProtonLabel;

public class RunNascaForGUI extends SwingWorker<Assignment, Void>{
	
	private JTextArea jTextAreaOutput;
	private JButton jButtonSCResonanceAsg;
    private JButton jButtonSaveNoeAsg;
    private JMenuItem jMenuSaveNoeAsg;
    private JMenuItem jMenuSaveResonanceAsg;
    private JButton jButtonRunNasca;
    private JButton jButtonStopNasca;
    
    private long startTime = System.currentTimeMillis(); 
   // private Assignment allAssignments;
    
	private RunNascaForGUI() {}
	    
	public RunNascaForGUI(JTextArea jTextArea,JButton jButtonSCResonanceAsgTemp,JButton jButtonSaveNoeAsgTemp,
			JMenuItem jMenuSaveResonanceAsgTemp,JMenuItem jMenuSaveNoeAsgTemp,JButton jButtonRunNascaTemp,JButton jButtonStopNascaTemp) {
	        this.jTextAreaOutput =  jTextArea;     
	        this.jButtonSCResonanceAsg=jButtonSCResonanceAsgTemp;
	        this.jButtonSaveNoeAsg=jButtonSaveNoeAsgTemp;
	        this.jMenuSaveResonanceAsg=jMenuSaveResonanceAsgTemp;
	        this.jMenuSaveNoeAsg=jMenuSaveNoeAsgTemp;	
	        this.jButtonRunNasca=jButtonRunNascaTemp;
	        this.jButtonStopNasca=jButtonStopNascaTemp;
	} 
	
	
	@Override
	protected void done() {
		Assignment assignment = null;
       
		if(isCancelled()){
			Global.status=0;
			
			if(Global.status==-1)
        		printMessage(Global.stopMessage);
        	else if(Global.status==0)
        		printMessage(Global.stopByUserMessage);
        	else
        		printMessage("Error: Unknown situations...");
			
			this.jButtonRunNasca.setEnabled(true);
        	return;
    	}
		
		
        try {
        	 assignment = get();
        } catch (Exception ignore) {
            ignore.printStackTrace();
            this.jButtonRunNasca.setEnabled(true);
            Global.status=-1;
            printMessage(Global.stopMessage);
            return;
        }
        
        if(assignment==null){
        	//printMessage("Error: Resonance assignments not computed...");
        	if(Global.status==-1)
        		printMessage(Global.stopMessage);
        	else if(Global.status==0)
        		printMessage(Global.stopByUserMessage);
        	else
        		printMessage("Error: Unknown situations...");
        	
        	return;
        }
        	
        Global.finalAssignments=assignment;
        
        printMessage(Global.successMessage);
        
        //print the sc resonance to Text area
        printMessage("Here are the resonance assignments in BMRB format (including both backbone and side-chain resonance assignments):");
        printMessage("");
        printAllResonanceAssignmentToTextAreaInBMRB(assignment.getResonAsgs());

        printMessage("");
        printMessage("");
        
        printMessage("Here are the NOE assignments in XPLOR format:");
        printMessage("");
        printAllNOEAssignmentsToTextArea(assignment.getNoeAsgs());
        
        this.jButtonRunNasca.setEnabled(true);
        //this.jButtonStopNasca.setEnabled(false);
        
        jButtonSCResonanceAsg.setEnabled(true);
        jButtonSaveNoeAsg.setEnabled(true);
        jMenuSaveResonanceAsg.setEnabled(true);
        jMenuSaveNoeAsg.setEnabled(true);
        
        printMessage(Global.successMessage);
        long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	printMessage("Total time for running NASCA is: "+ String.format("%.3f", totalTime) +" minutes");
        
	}
	 
	@Override
	protected Assignment doInBackground() throws Exception {
		startTime = System.currentTimeMillis(); 
		Assignment assignment=performNasca();
		
		this.jButtonRunNasca.setEnabled(true);
		return assignment;
	}

	/** 
     *Implement the NOE assignment (NASCA) algorithm without using the side-chain resonance information.
     @param src location of the input file
     @param strOut location of the output file
     @param strInput  input file name
     @return void
    */         
	private Assignment performNasca()throws JampackException
	{
		Assignment assignment=new Assignment();//for storing the resonance and NOE assignments
		
    	//boolean isDebug=false;//whether outputing the debugging information    
    	Peak pk = new Peak();    	   	
    	Assign asg = new Assign();
    	//long startTime = System.currentTimeMillis();    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	
    	double haErr  = Global.haErr;
    	double h1Err  = Global.h1Err;
    	double c13Err = Global.c13Err;
    	double hnErr  = Global.hnErr;
    	double nErr   = Global.nErr;
        	
    	/*String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="", strResNameScheme="",strIsCheckLongAA="",
    		strHaNoesy3D="", strResFormat="",strBB="",stroutName="",strNoesyFormat="",strOutOrFormat="",
    		strIsCheck="",strRefPdb="",strRefNameSchem="",strIsWholeStr="",strPdbNameScheme="",str4DNoesy="",
    		stroutResonName="",strOutResonFormat="";
    	int nIsoriginalUp=0; 
    	double noeLimit=0.0,metCor=0.0;*/
    	
    	// -------------------------------------
    	// (1.1) Read the backbone structure     	
    	//-------------------------------------    	
    	String pdbFile = Global.strBBPdbFile;
    	boolean fileExists = (new File(pdbFile)).exists();    	
    	if(!fileExists){
    		String errMessage="Error: Backbone PDB file not found...";
    		Global.status=-1;
    		printMessage(errMessage);
        	return null;
    	}   	
        
    	Vector<Pdb> pdbVecBB = pp.readPdb(pdbFile);
    	if(pdbVecBB==null || pdbVecBB.size()==0){
    		String errMessage="Error: Backbone PDB file not found...";
    		printMessage(errMessage);
    		Global.status=-1;
        	return null;
    	}
    		
    	printMessage("Read backbone PDB coordinates successfully from: "+ pdbFile);
    	
    	
    	//we assume that pdb file has atom names according to protein sequence input   	   	
    	String strPdbNameScheme=pp.obtainPdbAtomNameScheme(pdbVecBB);
    	
    	Vector<Pdb> pdbVecNewSec=new Vector<Pdb>();
		if(strPdbNameScheme.equalsIgnoreCase("PDB-OLD"))
			pdbVecNewSec=pp.nameConvertOrder(pdbVecBB); 
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-NEW"))
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecBB);//need to double check
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-OLD"))
		{
			Vector<Pdb> pdbVecSSE22=pp.nameConvertOrder(pdbVecBB); 
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecSSE22);//need to double check
		}
		else
			pdbVecNewSec.addAll(pdbVecBB);  
    	
		Vector<Pdb> vecStructureAll=new Vector<Pdb>();
		vecStructureAll.addAll(pdbVecNewSec);
		
		Vector<Pdb> vecBBTemp=pp.OutputBackbone(pdbVecNewSec);
		pdbVecNewSec=new Vector<Pdb>();
		pdbVecNewSec.addAll(vecBBTemp);    	
		
		
    	//-------------------------------------
    	// (1.2) Read the protein sequence
    	// 
    	//-------------------------------------    	
    	//String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector<Assign> vecSeq=pp.getSeqFromPdb(pdbVecBB);//asg.ReaderSeq(seqFile);    		   	
    	
    	//-------------------------------------
    	// (1.3) Read the resonance list 
    	//     	
    	//following is for the general case
    	H1CS h1CS=new H1CS();    	
    	String assignFile = Global.strBBCSFile;
    	fileExists = (new File(assignFile)).exists();
    	    		
    	if(!fileExists){
    		String errMessage="Error: Backbone chemical shift file not found...";
    		printMessage(errMessage);
    		Global.status=-1;
    		return null;
    	}   	
    	    	
    	String strResFormat=h1CS.obtainChemicalShiftFileFormat(Global.strBBCSFile);    	
    	
    	Vector<H1CS> assignVec=new Vector<H1CS>();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("Unknown format of the resonance file...");
    		Global.status=-1;
    		return null;
    	}
    	if(assignVec==null || assignVec.size()==0){
    		String errMessage="Error: Backbone chemical shift file not found...";
    		Global.status=-1;
    		printMessage(errMessage);
    		return null;
    	}
    	printMessage("Read backbone chemical shifts successfully from: "+ assignFile);
    	
    	//all sorted proton resonances
    	Vector<Peak> allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());     	
    	//-------------------------------------
    	// (1.4) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	Vector<Noesy> vecNoesy=new Vector<Noesy>();
    	
    	String strAliNoeFile = "", strAroNoeFile = "",strN15NoeFile = ""; 
    	Vector<Noesy> hnNoeVec=new Vector<Noesy>();//for storing 3d n15 NOESY peaks
    	Vector<Noesy> aliNoeVec=new Vector<Noesy>();//for storing 3d c13 ali noesy peaks
    	Vector<Noesy> aroNoeVec=new Vector<Noesy>();//for storing 3d c13 aro noesy peaks
    	//Vector<Noesy> Noe4DVec=new Vector<Noesy>();//for storing 4d NOESY peaks
    	String strNoesyFormat="XEASY";
    	
    	//check whether at least one NOESY peak file exists
    	strAliNoeFile=Global.strAliNoesyFile;
    	strAroNoeFile=Global.strAroNoesyFile;
    	strN15NoeFile=Global.strN15NoesyFile;
    	boolean aliFileExist = (new File(strAliNoeFile)).exists();
    	
    	boolean aroFileExist = (new File(strAroNoeFile)).exists();
    	
    	boolean n15FileExist = (new File(strN15NoeFile)).exists();
    	
    	if(! (aliFileExist || aroFileExist || n15FileExist) ){
    		printMessage("Error: NASCA requires at least one NOESY file as input...");
    		Global.status=-1;
    		return null;
    	}
    	
    	//read c13 aliphatic NOESY data    	
    	strAliNoeFile=strAliNoeFile.trim();
    	int indexOfDot=strAliNoeFile.lastIndexOf(".");
    	if(indexOfDot>-1){
    		String extName=strAliNoeFile.substring(indexOfDot+1,strAliNoeFile.length());
    		if(extName.equalsIgnoreCase("XPK"))
    			strNoesyFormat="NMRVIEW ";
    		else if(extName.equalsIgnoreCase("PEAKS"))
    			strNoesyFormat="XEASY";
    		else
    			strNoesyFormat="UNKNOWN";
    		
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			aliNoeVec = noesy.NoesyReader(strAliNoeFile);
    		else
    			aliNoeVec = noesy.NoesyReaderNMRView(strAliNoeFile);      
    		
    		vecNoesy.addAll(aliNoeVec);
    	}//if(indexOfDot>-1)    	
    	if(aliFileExist)
    		printMessage("Read C13 aliphatic NOESY cross peaks successfully from: "+ strAliNoeFile);
    	else
    		printMessage("C13 aliphatic NOESY cross peaks are not provided...");
    	
    	//read c13 aromatic NOESY data    	
    	strAroNoeFile=strAroNoeFile.trim();
    	indexOfDot=strAroNoeFile.lastIndexOf(".");
    	if(indexOfDot>-1){
    		String extName=strAroNoeFile.substring(indexOfDot+1,strAroNoeFile.length());
    		if(extName.equalsIgnoreCase("XPK"))
    			strNoesyFormat="NMRVIEW ";
    		else if(extName.equalsIgnoreCase("PEAKS"))
    			strNoesyFormat="XEASY";
    		else
    			strNoesyFormat="UNKNOWN";
    		
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			aroNoeVec = noesy.NoesyReader(strAroNoeFile);
    		else
    			aroNoeVec = noesy.NoesyReaderNMRView(strAroNoeFile);      
    		
    		vecNoesy.addAll(aroNoeVec);
    	}//if(indexOfDot>-1)
    	if(aroFileExist)
    		printMessage("Read C13 aromatic NOESY cross peaks successfully from: "+ strAroNoeFile);
    	else
    		printMessage("C13 aromatic NOESY cross peaks are not provided...");
    	
    	//read N15 NOESY data    	
    	strN15NoeFile=strN15NoeFile.trim();
    	indexOfDot=strN15NoeFile.lastIndexOf(".");
    	if(indexOfDot>-1){
    		String extName=strN15NoeFile.substring(indexOfDot+1,strN15NoeFile.length());
    		if(extName.equalsIgnoreCase("XPK"))
    			strNoesyFormat="NMRVIEW ";
    		else if(extName.equalsIgnoreCase("PEAKS"))
    			strNoesyFormat="XEASY";
    		else
    			strNoesyFormat="UNKNOWN";
    		
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			hnNoeVec = noesy.NoesyReader(strN15NoeFile);
    		else
    			hnNoeVec = noesy.NoesyReaderNMRView(strN15NoeFile);      
    		
    		vecNoesy.addAll(hnNoeVec);
    	}//if(indexOfDot>-1)
    	if(n15FileExist)
    		printMessage("Read N15 NOESY cross peaks successfully from: "+ strN15NoeFile);
    	else
    		printMessage("N15 NOESY cross peaks are not provided...");
    	    	
    	if(vecNoesy==null || vecNoesy.size()==0){
    		String errMessage="Error: NOESY cross peaks are not provided...";
    		printMessage(errMessage);
    		Global.status=-1;
    		return null;
    	}
    	
    	////////////////////////////////////    	    	
    	//intensity calibration    	
    	Vector<Noesy> vecNewNoesy=new Vector<Noesy>();
    	if(vecNoesy.size()>0)
    		vecNewNoesy=noesy.SetCalibration(vecNoesy); ////  	
    	  
    	double[] constant=new double[1];
    	Vector<Pdb> vecSSE=new Vector<Pdb>();//not used in the following function:
    	noesy.SetCalibrationN15(hnNoeVec,vecSSE,assignVec,constant);//change from vecNoesy to hnNoeVec
    	////////////////////////////////////  	
    	
		/////////////////////////////////////////////
    	/// 2. Implement the NASCA algorithm
    	//
    	/////////////////////////////////////////////
    	if(isCancelled()){
    		//printMessage(Global.stopByUserMessage);
    		Global.status=0;
    		return null;
    	}
    	
    	NascaNew nasca=new NascaNew();
    	
		ProtonLabel protonLabel=new ProtonLabel();
		GraphNode graphNode=new GraphNode();
		
		String userDir = System.getProperty("user.dir");////
    	String srcRot=userDir+"/system/rot-lib/";//path of rotamer library    	
    	
    	//------------------------------------------
    	//2.1 construct the set of proton labels (which consists of proton coordinates, rotamer index and proton name):
    	//------------------------------------------
    	//construct the label set (here we also call the structure graph), which includes
    	//the set of proton labels after placing rotamers on the RDC-defined backbone
      	Vector<ProtonLabel> vecLabels=protonLabel.ConstructProtonLabelSet(pdbVecNewSec,srcRot,Global.noeDistLimit);
    	
      	//------------------------------------------
    	//2.2 construct the NOESY graph:
    	//------------------------------------------    	
      	
      	if(isCancelled()){
    		//printMessage(Global.stopByUserMessage);
      		Global.status=0;
    		return null;
    	}
      	
    	Vector<H1CS> vecAsgNew=new Vector<H1CS>();		
		vecAsgNew=h1CS.CSNameConvertFromBMRBNewToPDBNew(assignVec);//assume that the default naming scheme of chemical shift is BMRB
		assignVec=new Vector<H1CS>();
		assignVec.addAll(vecAsgNew);
    	Vector<H1CS> vecBBAsg=nasca.ExtractBackboneAsignment(assignVec);		
    	
		//construct initial NOESY graph using the assigned backbone resonances
		Vector<GraphNode> vecResonNodeSet=graphNode.ConstructInitResonGraphFromAsgedList(vecBBAsg,null);
		
		//add the unassigned side-chain resonance nodes from NOESY peak list
		for(i=0;i<vecNewNoesy.size();i++){
			Noesy ne=(Noesy)vecNewNoesy.elementAt(i);
			double csH1=ne.getH1();
			double csHeavy=ne.getHeavy();
			//double csH2=ne.getH2();
			Vector<GraphNode> vecResGraphTemp=new Vector<GraphNode>();
			vecResGraphTemp=graphNode.AddResonanceNode(vecResonNodeSet,csH1,csHeavy,0.02, 0.1, 0.1);//error windows: by half
			
			vecResonNodeSet=new Vector<GraphNode>();
			vecResonNodeSet.addAll(vecResGraphTemp);
		}//for(i=0;i<vecNewNoesy.size();i++){
		
		//group each resonance cluster:
		for(i=0;i<vecResonNodeSet.size();i++){
			GraphNode node=(GraphNode)vecResonNodeSet.elementAt(i);
			Vector<GraphNode> vecAdj=node.getAdjVec();
			int counter=1;
			double csProtonSum=node.getCSProton();
			double csHeavySum=node.getCSHeavy();
			for(int k=0;k<vecAdj.size();k++)///////double check this step
			{
				GraphNode ndAdj=(GraphNode)vecAdj.elementAt(k);
				csProtonSum=csProtonSum+ndAdj.getCSProton();
				csHeavySum=csHeavySum+ndAdj.getCSHeavy();
				counter++;
			}
			csProtonSum=csProtonSum/counter;
			csHeavySum=csHeavySum/counter;
			node.setProtonCS(csProtonSum);
			node.setHeavyCS(csHeavySum);
			node.EmptyAdjSet();
		}		
		//set up the edge connections for resonances:
		Vector<GraphNode> vecResonGraph=graphNode.ConstructResonGraph(vecResonNodeSet,vecNewNoesy,0.04,0.04,0.1,0.1);//0.04,0.03,0.2,0.2
		
		//set up symmetry set
		graphNode.ConstructResonGraphSymmetry(vecResonNodeSet,vecNewNoesy,0.03,0.03,0.1,0.1);//0.04,0.03,0.2,0.2
			
		//read BMRB statistical information:
		String srcBMRB=userDir+"/system/BMRB_CS.txt";    	
    	Vector<H1CS> vecBMRB=h1CS.ReadBMRBSatistics(srcBMRB);
    	Vector<H1CS> vecBMRBNew=new Vector<H1CS>();		
    	vecBMRBNew=h1CS.CSNameConvertFromBMRBNewToPDBNew(vecBMRB);//assume that the default naming scheme of cs is BMRB
    	
    	//--------------------------------------------------------
    	//2.3 pruning infeasible side-chain resonance assignments
    	//    using BMRB outliers and backbone chemical shifts.
    	//--------------------------------------------------------   
    	int [][]Map;//mapping matrix, for storing the mappings between NOESY graph and proton positions
    	double [][] MapScore;
    	Map=new int[vecResonGraph.size()][vecLabels.size()];
    	MapScore=new double[vecResonGraph.size()][vecLabels.size()];
    	
    	if(isCancelled()){
    		//printMessage(Global.stopByUserMessage);
    		Global.status=0;
    		return null;
    	}
    	
    	//initially set up all possible mappings
    	nasca.Initialization(vecResonGraph,vecLabels,Map,MapScore,vecBBAsg);
    	
    	if(isCancelled()){
    		//printMessage(Global.stopByUserMessage);
    		Global.status=0;
    		return null;
    	}
    	
    	/////////////////////////
    	//pruning side-chain resonance assignments that fall outside the BMRB intervals
    	nasca.RefineMappingBasedBMRBOutliers(vecResonGraph,vecLabels,Map,vecBMRBNew);
    	
    	//update the mapping set
    	nasca.UpdateMappingSet(vecResonGraph,vecLabels,Map);
    	nasca.UpdateIsRestrained(vecResonGraph,vecLabels, Map);    	
    	
    	if(isCancelled()){
    		Global.status=0;
    		//printMessage(Global.stopByUserMessage);
    		return null;
    	}
    	
    	//Refinement based on the bb resonance assignments:
        //Note: This function was implemented in the early stage as a pre-processing of the 
    	//side-chain resonance assignments. Now we 
    	//still keep it, since in practice it might speed up the pruning process. 
    	//In principle, the assignments pruned from this function can be also 
    	//pruned by the DEE assignment approach.
    	nasca.RefineMappingBasedRestrained(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);//////
    	
    	//Refinement based on previously assigned HG resonances:
        //Note: This function was implemented in the early stage as a pre-processing of the 
    	//side-chain resonance assignments. Now we 
    	//still keep it, since in practice it helps reduce the running time. 
    	//In principle, the assignments pruned from this function can be also 
    	//pruned by the DEE assignment approach.
    	nasca.RefineMappingForHG(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);    //////   	
    	nasca.UpdateMappingSet(vecResonGraph,vecLabels,Map);
     	
    	if(isCancelled()){
    		//printMessage(Global.stopByUserMessage);
    		Global.status=0;
    		return null;
    	}
    	
     	//-------------------------------------------
     	//2.4 Use the DEE assignment algorithm to prune side-chain 
     	//    resonance assignments that are provably not part of 
     	//    the optimal solution.
    	printMessage("Now performing the DEE pruning...");
    	int numDeePruning=RefineMappingDeeCutForGUI(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);//new     
    	if(numDeePruning==-1 && Global.status==0){
    		
    		return null;
    	}
    	
    	
    	printMessage("DEE pruning completed. Total number of mappings pruned by DEE-CUT is: "+numDeePruning);
    	nasca.UpdateMappingSet(vecResonGraph,vecLabels,Map);     	
       	
        //print out all remaining assignments, after pruning. 
     	//This information is for debugging.
    	nasca.PrintAllPossibleMappings(vecResonGraph,vecLabels,Map);
       	
       	//output the statistical summary of the remaining assignments after pruning.
       	//This information is for debugging.
    	////nasca.StatisticsAllSCAssignment(vecResonGraph,vecLabels, Map,assignVec ); //not used for release
    	
    	//update the mapping set
    	nasca.UpdateMappingSet(vecResonGraph,vecLabels,Map);
    	H1CS h1cs=new H1CS();
    	
    	if(isCancelled()){
    		Global.status=0;
    		//printMessage(Global.stopByUserMessage);
    		return null;
    	}
    	//----------------------------------------------------
    	//2.5 Use the A* search algorithm to find the optimal 
    	//    side-chain resonance assignments.    	
    	//----------------------------------------------------
    	AStar astar=new AStar();
    	
    	printMessage("Now performing the A* search...");
    	Vector<H1CS> vecAsgNewT0=AStarSearchForSCAsgsForGUI(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy,vecBMRBNew);
    	if(vecAsgNewT0==null && Global.status==0)
    		return null;
    	
    	Vector<H1CS> vecAsgNewT=h1cs.DeleteRepeatAsg(vecAsgNewT0);//delete repeated assignments.
    	printMessage("A* search completed.");
    	
    	assignment.setResonAsgs(vecAsgNewT);
       	
    
    	
    	//print out the resonance assignments in different formats:
    	//This information will be useful for debugging.
    	//not used for realease
    	/*h1cs.PrintAllAssignment(vecAsgNewT);  //In BMRB format    	
    	h1cs.PrintAllAssignmentCyana(vecAsgNewT); //in CYANA format
    	
    	//////////////////////////////
    	//print out the resonance assignments into files:
    	String stroutResonName="";//need to be changed here
    	String strOutResonFormat="CYANA";
    	String outResFileName=userDir+stroutResonName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outResFileName)));
    		
    		if(strOutResonFormat.equalsIgnoreCase("CYANA"))        	
    			h1cs.PrintAllAssignmentToFileCyana(vecAsgNewT,out);        	
        	else        	
        		h1cs.PrintAllAssignmentToFileBMRB(vecAsgNewT,out);      	
	       
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ outResFileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + outResFileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}

    	///////////////////////////////////////   	
    	//compare with the reference resonance assignment list:
    	//this information is for debugging, and evaluating the performance.
    	h1cs.CompareSCAsgsWRefAsgs(vecAsgNewT,assignVec,vecBBAsg);    	
    	
    	//compute the completeness of the reference resonance assignment table:
    	double counter=0.0;
    	for(int t=0;t<vecLabels.size();t++)
    	{
    		ProtonLabel node=(ProtonLabel)vecLabels.elementAt(t);
    		String ndAtom=node.getAtomName();
    		int ndNo=node.getResNo();
    		
	    	for (int k=0;k<assignVec.size();k++)
	    	{	    		
	    		H1CS h1csT=(H1CS)assignVec.elementAt(k);	    		
	    		int refNo=h1csT.getResidueNo();
	    		String atom=h1csT.getAtomName();
	    		String subNdAtom=ndAtom;
	    		if(ndAtom.length()>atom.length())
	    			subNdAtom=ndAtom.substring(0,atom.length());
	    		
	    		if( !(atom.substring(0,1).equalsIgnoreCase("H")) )
	    			continue;
	    		if(ndNo==refNo && subNdAtom.equalsIgnoreCase(atom))
	    		{
	    			counter=counter+1;    	
	    			break;
	    		}
	    	}//end for (int k=0;k<assignVec.size();k++)
    	}//end for(int t=0;t<vecGraph.size();t++)
    	System.out.println("Completeness of reference assignment table is: "+(double) (counter/vecLabels.size()));
    
    	//compute the completeness of the our assignment table.
    	counter=0;
    	for(int k=0;k<vecAsgNewT.size();k++)
    	{
    		H1CS h1csT=(H1CS)vecAsgNewT.elementAt(k);
    		String atom=h1csT.getAtomName();
    		if(atom.substring(0,1).equalsIgnoreCase("H"))
    			counter=counter+1;     		
    	}
    	System.out.println("Completeness of our assignment table is: "+ (double)(counter/vecLabels.size()));
            
    	*/
    	
    	//--------------------------------------------
    	//2.6 NOE assignment process
    	//
    	//--------------------------------------------
    	Vector<Noe> vecAmbgAsg=new Vector<Noe>();    	
    	//all sorted proton resonances
    	allH1Vec=new Vector<Peak>();
    	allH1Vec = pk.allProtonSorted(vecAsgNewT);
    	Collections.sort(allH1Vec, new Peak.csComparator()); 
    	
    	if(isCancelled()){
    		//printMessage(Global.stopByUserMessage);
    		Global.status=0;
    		return null;
    	}
    	
    	//compute the initial ambiguous NOE assignments. 
    	//In principle we could use the NOESY graph to generate the set of 
    	//initial ambiugous assignments. But here we re-use the previously-written function.
    	//The result would be the same. 
    	vecAmbgAsg=asg.InitAmbiAssignment(nErr,c13Err,hnErr,haErr,h1Err,vecAsgNewT,allH1Vec,vecNewNoesy );        	
    	
    	//calibration for all initial NOE assignments
    	Vector<Noe> vecAmbgAsgTemp=noesy.SetCalibrationCyana(vecAmbgAsg,vecNewNoesy,constant[0]);    	
    	vecAmbgAsg=new Vector<Noe>();
    	vecAmbgAsg.addAll(vecAmbgAsgTemp);
     	 	
    	
     	//delete symmetric NOEs (which are redundent constraints):
     	Vector<Noe> vecNoeRefine2=new Vector<Noe>();     	
     	for(i=0;i<vecAmbgAsg.size();i++)
     	{
     		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
     		int noA=noe.getResidueNoA();
     		int noB=noe.getResidueNoB();
     		String atomA=noe.getAtomA();
     		String atomB=noe.getAtomB();
     		if(noA==noB && atomA.equalsIgnoreCase(atomB))
     			continue;
     		
     		boolean isIn=true;
     		String strOutOrFormat="0"; ///////////
     		if(strOutOrFormat.equalsIgnoreCase("0"))
     		{
	     		for (j=0;j<vecNoeRefine2.size();j++)
	     		{
	     			Noe noeT=(Noe)vecNoeRefine2.elementAt(j);
	     			int noAT=noeT.getResidueNoA();
	     			int noBT=noeT.getResidueNoB();
	     			String atomAT=noeT.getAtomA();
	     			String atomBT=noeT.getAtomB();
	     			if(noA==noAT && atomA.equalsIgnoreCase(atomAT) && noB==noBT && atomB.equalsIgnoreCase(atomBT))
	     				isIn=false;
	     			if(noA==noBT && atomA.equalsIgnoreCase(atomBT) && noB==noAT && atomB.equalsIgnoreCase(atomAT))
	     				isIn=false;
	     		}//for (j=0;j<vecNoeRefine2.size();j++)
     		}//if(strOutOrFormat.equalsIgnoreCase("0"))
     		if(	isIn)
     			vecNoeRefine2.add(noe);
     	}
     	
     	vecAmbgAsg=new Vector<Noe>();     	
     	vecAmbgAsg.addAll(vecNoeRefine2);
     	
     	//place the rotamers onto the RDC-defined backbone. 
     	Vector<Pdb> vecRotStructure=pp.RotamSelectAndStructure(hnErr, nErr, c13Err,pdbVecNewSec,vecAsgNewT,srcRot,vecNewNoesy,
    			4.5, 1,constant[0]);
     	
     	if(isCancelled()){
    		//printMessage(Global.stopByUserMessage);
    		Global.status=0;
    		return null;
    	}
     	
     	//We prune ambiguous NOE assignments based on the structural information. 
     	//Note: The NOE pruning process is a little different from what described in the paper
     	//but in the principle the results should be similar, since the structural information
     	//combined from RDC-defined backbone and rotamer library is used in a similar way.
     	Vector<Noe> vecRefinedNoes=new Vector<Noe>();    	
     	int nIsoriginalUp=0; 
    	double noeLimit=5.0;//metCor=0.0;
    	vecRefinedNoes=asg.PrunAmbgNoeAsg(vecRotStructure,vecAmbgAsg,noeLimit,0.0,nIsoriginalUp);
    	
     	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
     	
    	//sort the NOE table
     	String strOutOrFormat="0";
    	if(!strOutOrFormat.equalsIgnoreCase("1"))
    	{
	    	for(i=0;i<vecAmbgAsg.size();i++)
	    	{
	    		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
	    		int firstResNo=noe.getResidueNoA();
	    		int secondResNo=noe.getResidueNoB();
	    		String firstResName=noe.getResidueA();
	    		String secondResName=noe.getResidueB();
	    		String firstAtomName=noe.getAtomA();///
	    		String secondAtomName=noe.getAtomB();     		
	    		
	    		
	    		if(secondResNo<firstResNo)
	    		{
	    			noe.setResNoA(secondResNo);
	    			noe.setResNoB(firstResNo);
	    			noe.setAtom1Name(secondAtomName);
	    			noe.setAtom2Name(firstAtomName);
	    			noe.setResNameA(secondResName);
	    			noe.setResNameB(firstResName);
	    		}
	    	}//for(i=0;i<vecManAsgNew.size();i++)
    	
	    	Collections.sort(vecAmbgAsg, new Noe.noeComparatorC());
    	}    	      	
     	
    	String strIsCheckLongAA="0";
    	if(strIsCheckLongAA.equalsIgnoreCase("1"))
    	{    		
    		Vector<Noe> vecTemp=new Vector<Noe>();
    		for(i=0;i<vecRefinedNoes.size();i++)
        	{
        		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
        		int firstResNo=noe.getResidueNoA();
        		int secondResNo=noe.getResidueNoB();
        		String firstResName=noe.getResidueA();
        		String secondResName=noe.getResidueB();
        		String atomA=noe.getAtomA();
        		String atomB=noe.getAtomB();        		
        		String subAtomA=atomA;
        		String subAtomB=atomB;
        		if (atomA.length()>=2)
        			subAtomA=atomA.substring(0,2);
        		if(atomB.length()>=2) 
        			subAtomB=atomB.substring(0,2);
        			
        		if(Math.abs(firstResNo-secondResNo)<=4 )
        		{
        			vecTemp.add(noe);
        			continue;
        		}
        	
        		boolean isKeep=true;
        		if ( pk.isCharged(firstResName) && pk.isHydrophobic(secondResName) )
        			isKeep=false;
        		if ( pk.isCharged(secondResName) && pk.isHydrophobic(firstResName) )
        			isKeep=false;
        		if( (subAtomB.equalsIgnoreCase("HN") || subAtomB.equalsIgnoreCase("H")||subAtomB.equalsIgnoreCase("HA")||subAtomB.equalsIgnoreCase("HB")) &&
        				(subAtomA.equalsIgnoreCase("HN")||subAtomA.equalsIgnoreCase("H")||subAtomA.equalsIgnoreCase("HA")|| subAtomA.equalsIgnoreCase("HB")   ))
        			isKeep=true;
        		
        		if(isKeep)
        			vecTemp.add(noe);
        	}//for(i=0;i<vecManAsgNew.size();i++)
    		
    		vecRefinedNoes=new Vector<Noe>();
    		vecRefinedNoes.addAll(vecTemp);
    	}//if(strIsCheckLongAA.equalsIgnoreCase("1"))    	
    	
    	//sort the NOE restraints, and updated the atom names for metyl groups:
    	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
        Noe ne=new Noe();
        Vector<Noe> vecNoeNewT=ne.NameConvertFromPDBNewToPDBNew(vecRefinedNoes);
        vecRefinedNoes=new Vector<Noe>();
        vecRefinedNoes.addAll(vecNoeNewT);             
       
        assignment.setNoeAsgs(vecRefinedNoes);
        
     	//compute the numbers of unique and multiple assignments:
     /*	vecNoeRefine2=new Vector();
     	Vector vecNoeUniq=new Vector();
     	for(i=0;i<vecRefinedNoes.size();i++)
     	{
     		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
     		int pkID=noe.getPeakID();
     		boolean isUniq=true;
     		for(j=0;j<vecRefinedNoes.size();j++)
     		{
     			if (i==j)
     				continue;
     			Noe noeT=(Noe)vecRefinedNoes.elementAt(j);
     			int pkIDT=noeT.getPeakID();
     			if(pkID==pkIDT)
     				isUniq=false;
     		}
     		if(isUniq)
     			vecNoeUniq.add(noe);
     	}
     	*/
    	//compare the NOE assignments with the reference structure and output the summary.
     	/*String strIsCheck="0";
     	
     	String strRefPdb="";/////////////need to be changed here
     	String strRefNameSchem="PDB-OLD";////need to be changed here
     	String src ="";
     	double[] number=new double[4];
     	if(strIsCheck.equalsIgnoreCase("1"))
     	{
     		Noe noe_temp=new Noe();
     		
     		String pdbFileXray = src+strRefPdb;
     		     		
        	Vector vecTempPdbBB=pp.readPdb(pdbFileXray);
        	Vector vecTempPdbBB2=new Vector();        	
        	
        	if(strRefNameSchem.equalsIgnoreCase("PDB-OLD")) 	
        		vecTempPdbBB2 = pp.nameConvertOrder(vecTempPdbBB);
        	else if(strRefNameSchem.equalsIgnoreCase("BMRB-NEW"))
        		vecTempPdbBB2=pp.nameConvert2PDBNew(vecTempPdbBB);
    		else if(strRefNameSchem.equalsIgnoreCase("BMRB-OLD"))
    		{
    			Vector pdbVecSSE22=pp.nameConvertOrder(vecTempPdbBB); 
    			vecTempPdbBB2=pp.nameConvert2PDBNew(pdbVecSSE22);
    		}
        	else
        		vecTempPdbBB2.addAll(vecTempPdbBB);
        	System.out.println("============================================ ");
        	System.out.println("====Comparisons with reference structure===== ");
        	noe_temp.CompareAsgToRefStr(vecRefinedNoes,vecTempPdbBB2,6.0,0.0,number);
        	//System.out.println("Total number of manual NOEs = "+ vecManAsg.size());
        	System.out.println("Number of correct NOE assignments in reference structure=  "+ number[0] );
        	System.out.println("Number of wrong NOE assignments in reference structure=  "+ number[1] );    	
        	System.out.println("Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
        	System.out.println("Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
        	
        	System.out.println("============================================ ");
     	}//if(strIsCheck.equalsIgnoreCase("1"))
     	/////////////////////////////////////////////
     	*/
     	
    	//Output the final NOE assignments in standard xplor format, and statistical summary    	  
     	/*String stroutName="";///////////need to be changed here
     	String userDirTemp = System.getProperty("user.dir");
     	if(strOutOrFormat.equalsIgnoreCase("0"))
    	{
	     	String fileName=userDirTemp+stroutName;    	
	    	
	    	String xplorNoeStr="";
	    	double[] disUpper=new double[1];
	    	try{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		out.println("!REMARK: Total number of NOESY cross peaks is "+ (vecNoesy.size()));    		
	    		out.println("!REMARK: Total number of NOEs is "+vecRefinedNoes.size());
	    		out.println("!REMARK: the number of unique NOE assignments is "+vecNoeUniq.size());
	    		out.println("!REMARK: the number of multiple NOE assignments is "+(vecRefinedNoes.size()-vecNoeUniq.size()) );
	    		    		
	    		out.println("!REMARK: Number of correct assignments in reference structure=  "+ number[0] );
	        	out.println("!REMARK: Number of wrong assignments in reference structure=  "+ number[1] );    	
	        	out.println("!REMARK: Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
	        	out.println("!REMARK: Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
	    		out.println("");
	    		out.println("");
	    		
		        for (i=0;i<vecRefinedNoes.size();i++)		       	
		    	{
		    		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
		    		int resNo1=noe.getResidueNoA();
		    		int resNo2=noe.getResidueNoB();
		    		String res1=noe.getResidueA();
		    		String res2=noe.getResidueB();
		    		String atom1=noe.getAtomA();
		    		String atom2=noe.getAtomB();
		    		double distUpper=noe.getUpper();
		    		
		    		//xplor format:	    		
					xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
					if(isDebug)
						System.out.println(xplorNoeStr); 
					out.println(xplorNoeStr);
		    	}//for (i=0;i<vecRefinedNoes.size();i++)
		    	out.close();
		    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
	    	}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
    	}
     	else if(strOutOrFormat.equalsIgnoreCase("1")) //we output the NOEs in OR format:
    	{
    		Noe noeTemp=new Noe();    		
    		Vector vecMultiAsg=noeTemp.ConvertSingleToMultiAsg(vecRefinedNoes);
    		String fileName=userDirTemp+stroutName;  
    		noeTemp.OutputMultiNOEAsg(vecMultiAsg,fileName);    		
    	}
     	*/
     	  	    	
    	///////////////////////////////////////
     	/*long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	printMessage("Total time for running NASCA is: "+ totalTime +" minutes");
    	    	*/
        
    	return assignment;
	}
	
	//check the current status of the program, and print out corresponding messages:
	private void checkProgramStatus(){
		if(Global.status==-1){ //the program should be terminated due to the problem with the input files
			
		}
	}
	 
	private void printMessage(String message){
		//if(!message.substring(message.length()-1, message.length()).equalsIgnoreCase("."))
			//message=message+".";
		System.out.println(message);
    	jTextAreaOutput.append(message+"\n");
    	//jTextAreaOutput.selectAll();        	
    	jTextAreaOutput.setCaretPosition(jTextAreaOutput.getDocument().getLength());
	}
	
    /**
     * print all resonance assignments to Text Area, in BMRB format.
     * 
     * @param vecH1CS resonance assignment list
     * @param out the out
     * 
     * @return void
     */
    public void printAllResonanceAssignmentToTextAreaInBMRB(Vector<H1CS> vecH1CS)
    {
    	//out.println("!The exact assignments are:");
    	int preNo=-99;
    	int counter=1;
    	for(int i=0;i<vecH1CS.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH1CS.elementAt(i);
    		int resNo=h1cs.getResidueNo();
    		String res=h1cs.getResidueType();
    		String atom=h1cs.getAtomName();
    		double cs_h=h1cs.getH1CS();
    		if(cs_h<-200.0)
    			continue;
    		String strCS= String.format("%.3f", cs_h);
    		printMessage(counter+ "		"+ resNo+ "		"+res+"		"+atom+"	"+atom.substring(0,1)+"  "+ strCS+  "  "+ 0.0+ "		"+1);
    		
    		counter++;
    	}//for(int i=0;i<vecH1CS.size();i++)   	
    }
    
    public void printAllNOEAssignmentsToTextArea(Vector<Noe> vecNoes){
    	Peak pk=new Peak();
        for (int i=0;i<vecNoes.size();i++)		       	
    	{
    		Noe noe=(Noe)vecNoes.elementAt(i);
    		int resNo1=noe.getResidueNoA();
    		int resNo2=noe.getResidueNoB();
    		String res1=noe.getResidueA();
    		String res2=noe.getResidueB();
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		double distUpper=noe.getUpper();
    		
    		//xplor format:	    		
			String xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
			
			printMessage(xplorNoeStr);
    	}//for (i=0;i<vecRefinedNoes.size();i++)
    }
    
	/**Apply the A* search algorithm to find the optimal side-chain resonances.
     * 
     * Modified from Astar.AStarSearchForSCAsgs() after adding a few lines for addressing the process interruption.
     @param vecResonGraph resonance graph 
     @param vecHPositions structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public Vector<H1CS> AStarSearchForSCAsgsForGUI(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecHPositions, int [][] Map,double [][] MapScore, 
			Vector<Noesy> vecNewNoesy, Vector<H1CS> vecBMRBNew)
	{			
		AStar astar=new AStar();
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		//Noesy noesy=new Noesy();
		NascaNew nasca=new NascaNew();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;
				
		while(true)
		{
			boolean isAllAsg=true;
			double maxScore=-99999.9;
			int maxID=-1;
			int maxMapID=-1;
			
			if(isCancelled()){
	    		printMessage(Global.stopByUserMessage);
	    		return null;
	    	}
			
			for(int j=0;j<vecHPositions.size();j++)
			{    						
				ProtonLabel label=(ProtonLabel)vecHPositions.elementAt(j);
				String res=label.getResName();
				String atom=label.getAtomName();
				String heavName=pk.GetHeavyAtomFromProton(res,atom);
				
				
				int resNo=label.getResNo();				
				Vector<GraphNode> vecMapSet=label.getMappingSet();
				Vector<ProtonLabel> vecAdj=label.getAdjVec();
				if(label.getIsAssigned())
					continue;
				
				if(isCancelled()){
		    		printMessage(Global.stopByUserMessage);
		    		return null;
		    	}
				
				for(int k=0;k<vecMapSet.size();k++)
				{
					GraphNode nodeMap=(GraphNode)vecMapSet.elementAt(k);					
					if(nodeMap.asgedId>=0 )//////////
					{
						ProtonLabel nodeTemp=(ProtonLabel)vecHPositions.elementAt(nodeMap.asgedId);
						if(nodeTemp.stereoID!=label.getID())
							continue;
						else if(nodeMap.vecStereoSymSet.size()>0)
							continue;							////
					}        ///////////////////////
					Vector<H1CS> vecOneSCAsg=new Vector<H1CS> ();
					Vector<BackNoe> vecBackNoe=new Vector<BackNoe>();
					Vector<Noesy> vecNoesyTwo=new Vector<Noesy>();	
					Vector<Noesy> vecNoesy=nodeMap.vecAdjNOESY;
					vecNoesyTwo.addAll(vecNoesy);
					double cs_proton=nodeMap.getCSProton();
					double cs_heavy=nodeMap.getCSHeavy();
					
					if(label.stereoID>=0)
					{
						ProtonLabel nodeStereo=(ProtonLabel)vecHPositions.elementAt(label.stereoID);
					
						if(nodeStereo.asgedId>=0)
						{
							GraphNode ndStereoReson=(GraphNode)vecResonGraph.elementAt(nodeStereo.asgedId);
							if(Math.abs(ndStereoReson.getCSHeavy()-cs_heavy) >0.15)//0.15
								continue;
						}
					}//if(node2.stereoID>=0)
					vecOneSCAsg.add(new H1CS(resNo,res,atom,cs_proton));			
					vecOneSCAsg.add(new H1CS(resNo,res,heavName,cs_heavy));		
					
					for(int t=0;t<vecAdj.size();t++)
					{
						ProtonLabel nodeAdj=(ProtonLabel)vecAdj.elementAt(t);
						if(!(nodeAdj.getIsAssigned() && (nodeAdj.asgedId>=0)  ))
							continue;
						
						GraphNode nodeAdjReson=(GraphNode)vecResonGraph.elementAt(nodeAdj.asgedId);
						vecNoesyTwo.addAll(nodeAdjReson.vecAdjNOESY);
						
						int resNoSec=nodeAdj.getResNo();
						String resSec=nodeAdj.getResName();
						String atomSec=nodeAdj.getAtomName();
						String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
						double cs_protonSec=nodeAdjReson.getCSProton();
						double cs_heavySec=nodeAdjReson.getCSHeavy();
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
						double distUp=6.0;
						if(nasca.isInCloseAdj(nodeAdj,label))
							distUp=2.7;
						vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,distUp,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
						vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,distUp,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
					
					}//for(int t=0;t<vecAdj.size();t++)	
					
					Vector<BackNoe> vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
					//Vector vecNoesyTwoNew=noesy.DeleteRepeatedNoesy3D(vecNoesyTwo);
					int [] numPeaks=new int[1];
					double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
					dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
					Vector<H1CS> vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
					double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);
					
					//compute the actual cost, i.e. from first to current step:
					double gScore=dbScore+asgScore;//*0.5*vecBackNoeNew.size();//current score.		
					
					//////////////////////////////
					//compute the esitmated cost, i.e. from current step to the end:
					
					double hScore=astar.ComputeAStarEstimatedCost(vecResonGraph,vecHPositions,vecBMRBNew,j);
					
					double fScore=gScore+hScore;
					//////////////////////////////
					
					if(fScore>maxScore)
					{
						maxScore=fScore;
						maxID=j;
						maxMapID=k;
					}
				}//for(int k=0;k<vecMapSet.size();k++)
				isAllAsg=false;
				
			}//for(int j=0;j<vecGraph.size();j++)			
			if(isAllAsg || maxID<=0)
				break;				
			
			//update the most confident node:
			System.out.print("Current expansion node: ID="+maxID+", ");
			
			ProtonLabel label=(ProtonLabel)vecHPositions.elementAt(maxID);
			Vector<GraphNode> vecMapSet=label.getMappingSet();
			GraphNode nodeMap=(GraphNode)vecMapSet.elementAt(maxMapID);
			label.asgedId=nodeMap.getID();
			nodeMap.asgedId=label.getID();
			label.setIsAssigned(true);
			nodeMap.setIsAssigned(true);
			System.out.println(label.getResNo()+label.getResName()+" - "+label.getAtomName()+": " + nodeMap.getCSProton()+" , "+ nodeMap.getCSHeavy());
			
			//try to assign the stereo proton node:
			///int stereoProtonID=label.stereoID;
			if(label.stereoID>0)
			{
				ProtonLabel ndStereoProton=(ProtonLabel)vecHPositions.elementAt(label.stereoID);
				Vector<GraphNode> vecStereoReson=nodeMap.vecStereoSymSet;
				if(vecStereoReson.size()>0)
				{
					if(vecStereoReson.size()==1)
					{
						GraphNode ndStereoReson=(GraphNode)vecStereoReson.elementAt(0);
						ndStereoProton.asgedId=ndStereoReson.getID();
						ndStereoReson.asgedId=ndStereoProton.getID();
						ndStereoProton.setIsAssigned(true);
						ndStereoReson.setIsAssigned(true);
					}
					else
					{
						Vector<GraphNode> vecStereoProtonMap=ndStereoProton.getMappingSet();
						vecStereoProtonMap=new Vector<GraphNode>();
						vecStereoProtonMap.addAll(vecMapSet);
					}
				}
			}//if(node2.stereoID>0)
			
		}//while(true)		
			
		Vector<H1CS> vecAsg=new Vector<H1CS>();
		for(int j=0;j<vecHPositions.size();j++)
		{    						
			ProtonLabel node2=(ProtonLabel)vecHPositions.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			String heavName=pk.GetHeavyAtomFromProton(resName,atomName);
			int resNo=node2.getResNo();
			
			int resonID=node2.asgedId;
			if(resonID<0)
				continue;
			GraphNode nodeResonAsg=(GraphNode)vecResonGraph.elementAt(resonID);
			double cs_proton=nodeResonAsg.getCSProton();
			double cs_heavy=nodeResonAsg.getCSHeavy();
			vecAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
			vecAsg.add(new H1CS(resNo,resName,heavName,cs_heavy));
		}//for(int j=0;j<vecGraph.size();j++)
		
		return vecAsg;
	}
	/**Use the DEE algorithm to prune possible side-chain resonance assignments.
	 * Modified from NascaNew.RefineMappingDeeCut() function by adding a few lines for canceling the process.
    @param vecResonGraph resonance graph 
    @param vecHPositions proton labels   
    @param Map mapping matrix between two graphs  
    @param MapScore mapping score matrix
    @param vecNewNoesy noesy cross peak list
    @param vecBMRBNew BMRB information
    @param
    @return 
    */
	public int RefineMappingDeeCutForGUI(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecHPositions, int [][] Map,double [][] MapScore, 
			Vector<Noesy> vecNewNoesy, Vector<H1CS> vecBMRBNew)	{			
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		//Noesy noesy=new Noesy();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;				
		int counter=0;//count how many are pruned, for the purpose of debugging.
		
		for(int j=0;j<vecHPositions.size();j++)	{    		
			
			if(isCancelled()){
	    		//printMessage(Global.stopByUserMessage);
				Global.status=0;
				
	    		return -1;
	    	}
			
		    //	System.out.println("current proton node ID : "+ j);	
			ProtonLabel nodeProton=(ProtonLabel)vecHPositions.elementAt(j);
			if(nodeProton.getIsAssigned())
				continue;
			String res=nodeProton.getResName();
			String atom=nodeProton.getAtomName();
			String heavyName=pk.GetHeavyAtomFromProton(res,atom);
			
			////////////////////
			//new added:
			if(heavyName.equalsIgnoreCase("CB"))//////
				continue;
			/////////////////////
			
			int resNo=nodeProton.getResNo();				
			Vector<GraphNode> vecMapSet=nodeProton.getMappingSet();
			
			Vector<ProtonLabel> vecAdj=nodeProton.getAdjVec();
			if(vecMapSet.size()<1 || vecAdj.size()<1)//rarely happen
				continue;
			
			if(nodeProton.getIsAssigned())
				continue;
			for(int k=0;k<vecMapSet.size();k++)	{
				
				if(isCancelled()){
		    		//printMessage(Global.stopByUserMessage);
					Global.status=0;					
		    		return -1;		    		
		    	}
				//System.out.println("   map ID: "+ k);	
				
				GraphNode ndMapCur=(GraphNode)vecMapSet.elementAt(k);
				boolean isPruned=false;
				double csCur_proton=ndMapCur.getCSProton();
				double csCur_heavy=ndMapCur.getCSHeavy();
				
			//	if(ndMapCur.getID()==255  && nodeProton.getID()==52)
					//System.out.println("stop here, for debugging...");
						
				
				//compute the best score for current assignment
				//compute the homo score:
				Vector<H1CS> vecOneSCAsg=new Vector<H1CS>();//for computing homo score.
				vecOneSCAsg.add(new H1CS(resNo,res,atom,csCur_proton));
				vecOneSCAsg.add(new H1CS(resNo,res,heavyName,csCur_heavy));
	   			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
	       		double totalCurScore=asgScore;
				//compute the best pair score for current assignment
				for(int t=0;t<vecAdj.size();t++){
					ProtonLabel ndCurAdj=(ProtonLabel)vecAdj.elementAt(t);
					if(ndCurAdj.getIsAssigned())
						continue;
					Vector<GraphNode> vecNdCurAdjMap=ndCurAdj.getMappingSet();
					if(vecNdCurAdjMap.size()<1)//rarely happen
						continue;
					
					int resNoAdj=ndCurAdj.getResNo();
					String resAdj=ndCurAdj.getResName();
					String atomAdj=ndCurAdj.getAtomName();
					String heavyNameAdj=pk.GetHeavyAtomFromProton(resAdj,atomAdj);					
					int maxAdjID=-1;					
					double maxScore=-9999.9;
					
					Vector<BackNoe> vecBackNoe=new Vector<BackNoe>();//for computing pairwirse 
					for(int a=0;a<vecNdCurAdjMap.size();a++){
						GraphNode ndAdjMap=(GraphNode)vecNdCurAdjMap.elementAt(a);
						Vector<Noesy> vecNoesyTwo=new Vector<Noesy>();	
						vecNoesyTwo.addAll(ndMapCur.vecAdjNOESY);
						vecNoesyTwo.addAll(ndAdjMap.vecAdjNOESY);
						
						double csAdjMapProton=ndAdjMap.getCSProton();
						double csAdjMapHeavy=ndAdjMap.getCSHeavy();
						vecBackNoe.add(new BackNoe(csCur_proton,csCur_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
						vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csCur_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
						
						Vector<BackNoe> vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
						int [] numPeaks=new int[1];
						double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
						dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
						if(dbScore>maxScore){
							maxScore=dbScore;
							maxAdjID=a;
						}
					}//for(int a=0;a<vecNdCurAdjMap.size();a++)
					totalCurScore=totalCurScore+maxScore;					
				}//for(int t=0;t<vecAdj.size();t++)
				
				//compute the worse score for other assignments
				for(int h=0;h<vecMapSet.size();h++)	{
					if(h==k)
						continue;				
					GraphNode ndMapCurOther=(GraphNode)vecMapSet.elementAt(h);					
					double csOther_proton=ndMapCurOther.getCSProton();
					double csOther_heavy=ndMapCurOther.getCSHeavy();
					//compute the homo score:
					Vector<H1CS> vecOneSCAsgOther=new Vector<H1CS>();//for computing homo score.
					vecOneSCAsgOther.add(new H1CS(resNo,res,atom,csOther_proton));
					vecOneSCAsgOther.add(new H1CS(resNo,res,heavyName,csOther_heavy));
	    			double asgOtherScore=h1cs.BMRBSatisticsScore(vecOneSCAsgOther,vecBMRBNew);
	        		double totalOtherScore=asgOtherScore;
	        		
	        		int minAdjID=-1;
	        		//compute the worse pairwise score for other nodes:
	        		for(int t=0;t<vecAdj.size();t++){
	        			ProtonLabel ndCurAdj=(ProtonLabel)vecAdj.elementAt(t);
						if(ndCurAdj.getIsAssigned())
							continue;
						Vector<GraphNode> vecNdCurAdjMap=ndCurAdj.getMappingSet();
						if(vecNdCurAdjMap.size()<1)//rarely happen
							continue;
						
						int resNoAdj=ndCurAdj.getResNo();
						String resAdj=ndCurAdj.getResName();
						String atomAdj=ndCurAdj.getAtomName();
						String heavyNameAdj=pk.GetHeavyAtomFromProton(resAdj,atomAdj);					
						minAdjID=-1;	
						
						double minScore=999999.9;
						Vector<BackNoe> vecBackNoe=new Vector<BackNoe>();//for computing pairwirse 
						for(int a=0;a<vecNdCurAdjMap.size();a++){
							GraphNode ndAdjMap=(GraphNode)vecNdCurAdjMap.elementAt(a);
							Vector<Noesy> vecNoesyTwo=new Vector<Noesy>();	
							vecNoesyTwo.addAll(ndMapCurOther.vecAdjNOESY);
							vecNoesyTwo.addAll(ndAdjMap.vecAdjNOESY);
							
							double csAdjMapProton=ndAdjMap.getCSProton();
							double csAdjMapHeavy=ndAdjMap.getCSHeavy();
							vecBackNoe.add(new BackNoe(csCur_proton,csCur_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
							vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csCur_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
							
							Vector<BackNoe> vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
							int [] numPeaks=new int[1];
							double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
							dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
							if(dbScore<minScore){
								minScore=dbScore;
								minAdjID=a;
							}
						}//for(int a=0;a<vecNdCurAdjMap.size();a++)
						totalOtherScore=totalOtherScore+minScore;			
					}//for(int t=0;t<vecAdj.size();t++)
	        		
	        		//check whether the DEE criterion is satisfied:
	        		if(totalCurScore<totalOtherScore){
	        			isPruned=true;
	        			break;
	        		}
	        		
				}//for(int h=0;h<vecMapSet.size();h++)
				
				//remove the mapping if it is pruned:
				if(isPruned){
					Map[ndMapCur.getID()][nodeProton.getID()]=0;
					counter++;
				}
			
			}//for(int k=0;k<vecMapSet.size();k++)
						
		}//for(int j=0;j<vecGraph.size();j++)			
	
		return counter;
		//System.out.println("Total number of mappings pruned by DEE-CUT is: "+ counter);	
	
	}
	
}
