package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Noesy.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

  
import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 *  
 *   This class provides data structures and functions for processing NOESY spectra. 
 *   Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
 * 
 */
public class Noesy{
    
	private int ID;
    /** The residue no. */
    private int    residueNo; 
    
    /** The residue name. */
    private String residue;
    
    /** The chemical shift of the first proton. */
    private double dbCSH1; //proton connected to heava atom, such as Ha, HN
    
    /** The chemical shift of the heavy atom attached to the first proton. */
    private double dbCSHeavy;
    
    /** The chemical shift of the second proton. */
    private double dbCSH2;//the proton in space
    
    /** The chemical shift of the first heavy atom, in 4D. */
    private double dbCSHeavy1;//for 4D;
    
    /** The chemical shift of the second heavy atom, in 4D. */
    private double dbCSHeavy2;//for 4D
    
    /** The peak intensity. */
    private double intensity;
    
    /** The uncertainty of noesy peak. */
    private double uncertainty=0.0;
    
    /** The upper distance calibrated from intensity. */
    private double upperDist=0.0;
    
    /** The bin index, used for the Xplor distance calibration scheme. */
    private int binIndex=0;//1 for 1.8-2.7A; 2 for 1.8-3.3A; 3 for 1.8-5.0A; 4 for 1.8-6.0A.
    /*From Xplor-NIH email list:
     * John Kuszewski:
	 * In xplor-nih's marvin code, I just divide the peaks into distance  
	 bins, based on their relative intensities.
	 Specifically, the most intense 20% of peaks get distance bounds of  
	 1.8 - 2.7 A.
	 The next most intense 30% of peaks get bounds of 1.8 - 3.3 A.
	 The next most intense 30% of peaks get bounds of 1.8 - 5.0 A.
	 The least-intense 20% of peaks get bounds of 1.8 - 6.0 A.

	*/
    /**
     * Instantiates a new noesy.
     */
    public Noesy(){
	residueNo = 0;
	residue = null;
	dbCSH1 = 0.0; 
	dbCSHeavy = 0.0;
	dbCSH2 =0.0;
	intensity = 0.0;
	binIndex=0;
    }		
    
    /**
     * Instantiates a new noesy.
     * 
     * @param no the no
     */
    public Noesy(int no){
	residueNo = no;
	residue = null;
	dbCSH1 = 0.0; 
	dbCSHeavy = 0.0;
	dbCSH2 = 0.0;
	intensity = 0.0;
	binIndex=0;
    }		
    
    /**
     * Instantiates a new noesy.
     * 
     * @param no the no
     * @param resid the resid
     * @param H1 the h1
     * @param Heavy the heavy
     * @param H2 the h2
     * @param inten the inten
     */
    public Noesy(int no,String resid, double H1, double Heavy, double H2, double inten)
    {
    	residueNo = no;
		residue = resid;
		dbCSH1 = H1; 
		dbCSHeavy = Heavy;
        dbCSH2 = H2;
        intensity = inten;	
        binIndex=0;
    }	
    
    /**
     * Instantiates a new noesy.
     * 
     * @param no the no
     * @param resid the resid
     * @param H1 the h1
     * @param Heavy the heavy
     * @param H2 the h2
     * @param inten the inten
     * @param uncn the uncn
     */
    public Noesy(int no,String resid, double H1, double Heavy, double H2, double inten,double uncn)
    {
    	residueNo = no;
    	residue = resid;
    	dbCSH1 = H1; 
    	dbCSHeavy = Heavy;
    	dbCSH2 = H2;
    	intensity = inten;
    	uncertainty= uncn;
    	binIndex=0;
        }	
   
    //for 4D
    /**
     * Instantiates a new noesy.
     * 
     * @param no the no
     * @param resid the resid
     * @param H1 the h1
     * @param Heavy1 the heavy1
     * @param H2 the h2
     * @param Heavy2 the heavy2
     * @param inten the inten
     * @param uncn the uncn
     */
    public Noesy(int no,String resid, double H1, double Heavy1, double H2,double Heavy2,  double inten,double uncn)
    {
    	residueNo = no;
    	residue = resid;
    	dbCSH1 = H1; 
    	dbCSHeavy1 = Heavy1;
    	dbCSH2 = H2;
    	dbCSHeavy2 = Heavy2;
    	intensity = inten;
    	uncertainty= uncn;
    	binIndex=0;
        }	
    
    public Noesy(int id,int no,String resid, double H1, double Heavy1, double H2,double Heavy2,  double inten,double uncn)
    {
    	ID=id;
    	residueNo = no;
    	residue = resid;
    	dbCSH1 = H1; 
    	dbCSHeavy1 = Heavy1;
    	dbCSH2 = H2;
    	dbCSHeavy2 = Heavy2;
    	intensity = inten;
    	uncertainty= uncn;
    	binIndex=0;
        }	
    public Noesy(int id,double H1, double Heavy, double H2,  double inten,double uncn)
    {
    	ID=id;    	
    	dbCSH1 = H1; 
    	dbCSHeavy = Heavy;
    	dbCSH2 = H2;
    	intensity = inten;
    	uncertainty= uncn;
    	binIndex=0;
        }	
  //getting the values	
    /**
   * Gets the residue no.
   * 
   * @return the residue no
   */
  public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the residue type.
     * 
     * @return the residue type
     */
    public String getResidueType(){
	return residue;
    }	
    
    /**
     * Gets the h1.
     * 
     * @return the h1
     */
    public double getH1(){
	return dbCSH1;
    }	
    
    public int getID(){
    	return ID;
    }
    /**
     * Gets the heavy.
     * 
     * @return the heavy
     */
    public double getHeavy(){
	return dbCSHeavy;
    }	
    
    /**
     * Gets the heavy1.
     * 
     * @return the heavy1
     */
    public double getHeavy1(){
    	return dbCSHeavy1;
        }	
    
    /**
     * Gets the heavy2.
     * 
     * @return the heavy2
     */
    public double getHeavy2()
    {
    	return dbCSHeavy2;
       }
    
    /**
     * Gets the h2.
     * 
     * @return the h2
     */
    public double getH2(){
	return dbCSH2;
    }	
    
    /**
     * Gets the intensity.
     * 
     * @return the intensity
     */
    public double getIntensity(){
	return intensity;
    }
    
    /**
     * Gets the uncertainty.
     * 
     * @return the uncertainty
     */
    public double getUncertainty(){
	return uncertainty;
    }	
    
    /**
     * Gets the upper dist.
     * 
     * @return the upper dist
     */
    public double getUpperDist(){
    	return upperDist;
        }	
    
    /**
     * Gets the bin index.
     * 
     * @return the bin index
     */
    public int getBinIndex(){
    	return binIndex;
        }	

    public void setID(int id){
    	ID=id;
    }
    //set the values	
    /**
     * Sets the residue no.
     * 
     * @param R the new residue no
     */
    public void setResidueNo(int R){
	residueNo = R;
    }	
    
    /**
     * Sets the residue type.
     * 
     * @param R the new residue type
     */
    public void setResidueType(String R){
	residue = R;
    }	
    
    /**
     * Sets the upper dist.
     * 
     * @param dist the new upper dist
     */
    public void setUpperDist(double dist)
    {
    	upperDist = dist;
    }	
    
    /**
     * Sets the bin index.
     * 
     * @param i the new bin index
     */
    public void setBinIndex(int i)
    {
    	binIndex = i;
    }	

    /**
     * The Class resNoComparator.
     */
    public static class resNoComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Noesy n1 = (Noesy)o1;
	    Noesy n2 = (Noesy)o2;
	    double d1 = n1.getResidueNo();
	    double d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }
    
    /**
     * The Class IntensityComparator.
     */
    public static class IntensityComparator implements Comparator{
    	
	    /* (non-Javadoc)
	     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	     */
	    public int compare(Object o1, Object o2){
    		Noesy n1 = (Noesy)o1;
    		Noesy n2 = (Noesy)o2;
    	    double d1 = n1.getIntensity();
    	    double d2 = n2.getIntensity();
    	    if (d1 < d2)
    		return -1;
    	    else if (d1 > d2)
    		return 1;
    	    else return 0;
    	}
        }

    /**
     * The Class h1Comparator.
     */
    public static class h1Comparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
		Noesy n1 = (Noesy)o1;
		Noesy n2 = (Noesy)o2;
	    double d1 = n1.getH1();
	    double d2 = n2.getH1();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class h2Comparator.
     */
    public static class h2Comparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
		Noesy n1 = (Noesy)o1;
		Noesy n2 = (Noesy)o2;
	    double d1 = n1.getH2();
	    double d2 = n2.getH2();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class heavyComparator.
     */
    public static class heavyComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
		Noesy n1 = (Noesy)o1;
		Noesy n2 = (Noesy)o2;
	    double d1 = n1.getHeavy();
	    double d2 = n2.getHeavy();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }
    
    /**
     * calibration of intensity: using xplor-NIH method
     * The required file format is:.
     * 
     * @param vecNoesy before volume calibration
     * 
     * @return after calibraiton
     */
    public Vector<Noesy> SetCalibration(Vector<Noesy> vecNoesy)
    {
    	int i,j;
    	double volume=0.0;
    	double maxInten=0.0;
    	Noesy noesy;
    	double inten;
    	
    	Collections.sort(vecNoesy, new Noesy.IntensityComparator()); 
    	
    	//remove the diagonal peaks
    	Vector vecNoesyN=new Vector();
    	
    	for (i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy2=(Noesy)vecNoesy.elementAt(i);
    		double cs_h1=noesy2.getH1();
    		double cs_h2=noesy2.getH2();
    		//double cs_heavy=noesy2.getHeavy();
    		    		
    		if (Math.abs(cs_h1-cs_h2)>0.02)
    			vecNoesyN.add(noesy2);
    		
    		
    	}//for (i=0;i< vecNoesy.size();i++)
    	
    	int totalNum=vecNoesyN.size();	
    	
    	int index1=(int) Math.floor(totalNum*0.2);
    	int index2=(int) Math.floor(totalNum*0.5);
    	int index3=(int) Math.floor(totalNum*0.8);
    	Noesy noesy1=(Noesy)vecNoesy.elementAt(index1);
    	Noesy noesy2=(Noesy)vecNoesy.elementAt(index2);
    	Noesy noesy3=(Noesy)vecNoesy.elementAt(index3);
    	/*John Kuszewski:
    	 * In xplor-nih's marvin code, I just divide the peaks into distance  
    	 bins, based on their relative intensities.
    	 Specifically, the most intense 20% of peaks get distance bounds of  
    	 1.8 - 2.7 A.
    	 The next most intense 30% of peaks get bounds of 1.8 - 3.3 A.
    	 The next most intense 30% of peaks get bounds of 1.8 - 5.0 A.
    	 The least-intense 20% of peaks get bounds of 1.8 - 6.0 A.

		Dan Garrett provided this method to me.*/
    	
    	double bin1=noesy1.getIntensity();//the less 20% intense
    	double bin2=noesy2.getIntensity();
    	double bin3=noesy3.getIntensity();
    	    	
    	Vector vecNewNoesy=new Vector();
    	for (i=0; i< vecNoesy.size();i++)
    	{
    		noesy=(Noesy)vecNoesy.elementAt(i);    		
    		    		
    		inten=noesy.getIntensity();
    		if (inten>= bin3)
    		{
    			noesy.setUpperDist(2.7);
    			noesy.setBinIndex(1);
    		}
    		if( (inten< bin3) && (inten>=bin2) )
    		{
    			noesy.setUpperDist(3.3);
    			noesy.setBinIndex(2);
    		}
    		if( (inten<bin2)  && (inten>= bin1))
    		{
    			noesy.setUpperDist(5.0);
    			noesy.setBinIndex(3);
    		}
    		if(inten< bin1 )
    		{
    			noesy.setUpperDist(6.0);
    			noesy.setBinIndex(4);    			
    		}
    		
    		vecNewNoesy.add(noesy); 		
    		
    	}
    	
    	return vecNewNoesy;
    }
    
    /**
     * calibration of intensity: using -3
     * The required file format is:.
     * 
     * @param vecNoesy before volume calibration
     * 
     * @return after calibraiton
     */
    public Vector SetCalibration2(Vector vecNoesy)
    {
    	int i,j;
    	double volume=0.0;
    	double maxInten=0.0;
    	Noesy noesy;
    	double inten;
    	double exponent=(double)(1.0/3.0);
    	
    	Collections.sort(vecNoesy, new Noesy.IntensityComparator()); 
    	
    	//remove the diagonal peaks
    	Vector vecNoesyN=new Vector();
    
    	for (i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy2=(Noesy)vecNoesy.elementAt(i);
    		double cs_h1=noesy2.getH1();
    		double cs_h2=noesy2.getH2();
    		if(maxInten<noesy2.getIntensity())
    			maxInten=noesy2.getIntensity();
    		   		
    		if (Math.abs(cs_h1-cs_h2)>0.02)
    			vecNoesyN.add(noesy2);
    		
    		
    	}//for (i=0;i< vecNoesy.size();i++)
    	Collections.sort(vecNoesy, new Noesy.IntensityComparator()); 
    	int totalNum=vecNoesyN.size();	
    	

    	
    	int index1=(int) Math.floor(totalNum*0.2);
    	int index2=(int) Math.floor(totalNum*0.5);
    	int index3=(int) Math.floor(totalNum*0.8);
    	Noesy noesy1=(Noesy)vecNoesy.elementAt(index1);
    	Noesy noesy2=(Noesy)vecNoesy.elementAt(index2);
    	Noesy noesy3=(Noesy)vecNoesy.elementAt(index3);
    	
    	
    	Noesy noesy0=(Noesy)vecNoesy.elementAt(vecNoesy.size()-1);
    	
    	double constant=(double)( noesy3.getIntensity()*Math.pow(2.7,3 ) );
    	    	
    	exponent=0-exponent;
    	Vector vecNewNoesy=new Vector();
    	for (i=0; i< vecNoesy.size();i++)
    	{
    		noesy=(Noesy)vecNoesy.elementAt(i);
    		inten=noesy.getIntensity();
    		
    		inten=inten/constant;
    		
    		double up_dist=Math.pow(inten, exponent);
    		up_dist=up_dist+0.3*up_dist;
    		
    		if(up_dist>6)
    			up_dist=6.0;
    		if(up_dist<2.7)
    			up_dist=2.7;
    		
    		noesy.setUpperDist(up_dist);
    		vecNewNoesy.add(noesy); 		
    		
    	}
    	
    	return vecNewNoesy;
    }
    
    /**
     * the following rules are used to calibrate the constant k
     * 1. distance between HA-HB in ALA is less than 2.5A (unique assignment, peak intensity need to be divided by 3)
     * 2. Methyl-methyl group NOEs (unque NOE asisgnment, divide by 3*3)
     * Note: unique assignment, not close to diagonal line
     * 
     * @param vecNoesy before volume calibration
     * @param vecAsg resonance assignment list
     * 
     * @return after calibraiton
     */
    public Vector SetCalibrationC13(Vector vecNoesy,Vector vecAsg)
    {
    	Peak pk=new Peak();
    	Assign asg=new Assign();
    	Pdb pp=new Pdb();
    	Collections.sort(vecNoesy, new Noesy.IntensityComparator());     	
    	// all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(vecAsg);
    	Collections.sort(allH1Vec, new Peak.csComparator());
    	String userDir = System.getProperty("user.dir");////
    	String srcRot=userDir+"/system/rot-lib/";
    	
    	Vector vecUniqueAsg=asg.InitAmbiAssignmentUnique(0.2,0.2,0.04,0.04,0.03,vecAsg,allH1Vec,vecNoesy );
    	Vector vecNewNoe=new Vector();
    	for(int i=0;i<vecUniqueAsg.size();i++)
    	{
    		Noe noe=(Noe)vecUniqueAsg.elementAt(i);
    		int firstNo=noe.getResidueNoA();
    		int secondNo=noe.getResidueNoB();
    		if(!(firstNo==secondNo))
    			continue;
    		
    		
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		double csH1=pk.getCSFromList(vecAsg,firstNo,atom1);
    		double csH2=pk.getCSFromList(vecAsg,secondNo,atom2);
    		if(Math.abs(csH1-csH2)<0.06)//prune those near diagonal line
    			continue; 
    		
    		String res=noe.getResidueA();
    		String rotamFile=srcRot+ res.toLowerCase()+ ".pdb"; 
    		Vector vecPdbRotam=pp.readRotamerPdb(rotamFile);
    		double maxDist=-999.9;
    		double minDist=999.9;
    		boolean isLargeDist=false;
    		for(int j=0;j<vecPdbRotam.size();j++)
    		{
    			Pdb pdb=(Pdb)vecPdbRotam.elementAt(j);
    			pdb.setResidueNo(firstNo);
    			double[] dist=new double[2]; 
    			Vector vecPdb=new Vector();
    			vecPdb.add(pdb);
    			pk.checkNoeNew(vecPdb, firstNo, atom1, secondNo,atom2, 6.0, 0.0, dist);
    			
    			if(maxDist>0 && (Math.abs(maxDist-dist[0])>1.0) )
    				isLargeDist=true;
    			if(maxDist<dist[0])
    				maxDist=dist[0];
    			if(minDist<dist[0])
    				minDist=dist[0];
    			
    		}//for(int j=0;j<vecPdbRotam.size();j++)
    		if(isLargeDist)
    			continue;    
    		//if(maxDist>)
    		if(minDist>100)
    			continue;
    		noe.setDistUpper(minDist);
    		vecNewNoe.add(noe);    		
    	}//for(int i=0;i<vecUniqueAsg.size();i++)
    	
    	double counter=0.0;
    	double k_sum=0.0;
    	for(int i=0;i<vecNewNoe.size();i++)
    	{
    		Noe noe=(Noe)vecNewNoe.elementAt(i);
    		int pkID=noe.getPeakID();
    		Noesy noesy=(Noesy)vecNoesy.elementAt(pkID);
    		String res1=noe.getResidueA();
    		String atom1=noe.getAtomA();
    		String res2=noe.getResidueB();
    		String atom2=noe.getAtomB();
    		double intensity=noesy.getIntensity();
    		double distance=noe.getUpper();
    		double divMethyl=1.0;
    		
    		if(res1.equalsIgnoreCase("ALA") && atom1.equalsIgnoreCase("HB"))
        		divMethyl=divMethyl*3;
        	if(res1.equalsIgnoreCase("ILE") && (atom1.equalsIgnoreCase("HG2") || atom1.equalsIgnoreCase("HD1") ) )
        		divMethyl=divMethyl*3;
        	if(res1.equalsIgnoreCase("LEU") && (atom1.equalsIgnoreCase("HD1") || atom1.equalsIgnoreCase("HD2") ) )
        		divMethyl=divMethyl*3;
        	if(res1.equalsIgnoreCase("LEU") && atom1.equalsIgnoreCase("HD")  )
        		divMethyl=divMethyl*9;
        	if(res1.equalsIgnoreCase("VAL") && (atom1.equalsIgnoreCase("HG1") || atom1.equalsIgnoreCase("HG2") ) )
        		divMethyl=divMethyl*3;
        	if(res1.equalsIgnoreCase("VAL") && atom1.equalsIgnoreCase("HG")  )
        		divMethyl=divMethyl*9;        	
        	if(res1.equalsIgnoreCase("THR") && atom1.equalsIgnoreCase("HG2"))
        		divMethyl=divMethyl*3;
        	if(res1.equalsIgnoreCase("MET") && atom1.equalsIgnoreCase("HE"))
        		divMethyl=divMethyl*3;
        	if(res1.equalsIgnoreCase("PHE") && (atom1.equalsIgnoreCase("HE") || atom1.equalsIgnoreCase("HE1") 
        			|| atom1.equalsIgnoreCase("HD")||atom1.equalsIgnoreCase("HD1")))
        		divMethyl=divMethyl*2;
    		
    		//for methyl group protons
        	if(res2.equalsIgnoreCase("ALA") && atom2.equalsIgnoreCase("HB"))
        		divMethyl=divMethyl*3;
        	if(res2.equalsIgnoreCase("ILE") && (atom2.equalsIgnoreCase("HG2") || atom2.equalsIgnoreCase("HD1") ) )
        		divMethyl=divMethyl*3;
        	if(res2.equalsIgnoreCase("LEU") && (atom2.equalsIgnoreCase("HD1") || atom2.equalsIgnoreCase("HD2") ) )
        		divMethyl=divMethyl*3;
        	if(res2.equalsIgnoreCase("LEU") && atom2.equalsIgnoreCase("HD")  )
        		divMethyl=divMethyl*9;
        	if(res2.equalsIgnoreCase("VAL") && (atom2.equalsIgnoreCase("HG1") || atom2.equalsIgnoreCase("HG2") ) )
        		divMethyl=divMethyl*3;
        	if(res2.equalsIgnoreCase("VAL") && atom2.equalsIgnoreCase("HG")  )
        		divMethyl=divMethyl*9;        	
        	if(res2.equalsIgnoreCase("THR") && atom2.equalsIgnoreCase("HG2"))
        		divMethyl=divMethyl*3;
        	if(res2.equalsIgnoreCase("MET") && atom2.equalsIgnoreCase("HE"))
        		divMethyl=divMethyl*3;
        	if(res2.equalsIgnoreCase("PHE") && (atom2.equalsIgnoreCase("HE") || atom2.equalsIgnoreCase("HE1") 
        			|| atom2.equalsIgnoreCase("HD")||atom2.equalsIgnoreCase("HD1")))
        		divMethyl=divMethyl*2;
        	//intensity=(double)(intensity/ divMethyl);
        	
        	counter=counter+1;
			k_sum=k_sum+(double)(intensity/Math.pow(distance,-6));
    	}//for(int i=0;i<vecNewNoe.size();i++)
    	
    	//compute the constant k
    	double constant=(double)(k_sum/counter);
    	
    	Vector vecNewNoesy=new Vector();
       //distance calibration for every peak:
    	for (int i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy=(Noesy)vecNoesy.elementAt(i);
    		double intensity=noesy.getIntensity();
    		double temp=(double)(constant/intensity);
    		double distance=Math.pow(temp, (double)(1.0/6.0));
    		noesy.setUpperDist(distance);   
    		vecNewNoesy.add(noesy);
    	}//for (i=0;i< vecNoesy.size();i++)    	
    	return vecNewNoesy;
    }
    
    /**
     * calibrate one NOE peak, using a calibration strategy similar to CYANA.
     * 
     * @param vecNoe initial NOE assignments
     * @param vecNoesy noesy peak list
     * @param constant constant from sse backbone
     * 
     * @return after calibraiton
     */
    public Vector<Noe> SetCalibrationCyana(Vector<Noe> vecNoe,Vector<Noesy> vecNoesy,double constant)
    {    	
    	double constB=(double)(constant/(2.4*2.4));
    	double constC=(double)(constB/3.0);
    	
    	for(int i=0;i<vecNoe.size();i++)
    	{
    		double exponent=0.0;
    		double constK=constant;
    		Noe noe=(Noe)vecNoe.elementAt(i);
    		int no1=noe.getResidueNoA();
    		int no2=noe.getResidueNoB();
    		String res1=noe.getResidueA();
    		String res2=noe.getResidueB();
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		int pkID=noe.getPeakID();
    		Noesy noesy=(Noesy)vecNoesy.elementAt(pkID);
    		double intensity=noesy.getIntensity(); 		
    		
    		int nType=0;
    		
    		String subAtom1=atom1;
    		String subAtom2=atom2;
    		if(atom1.length()>=2)
    			subAtom1=atom1.substring(0,2);
    		if(atom2.length()>=2)
    			subAtom2=atom2.substring(0,2);
    		
    		//for type 1: 
    		if( (subAtom1.equalsIgnoreCase("H")|| subAtom1.equalsIgnoreCase("HN") || subAtom1.equalsIgnoreCase("HA") ||subAtom1.equalsIgnoreCase("HB")) &&
    				(subAtom2.equalsIgnoreCase("H")||subAtom2.equalsIgnoreCase("HN")||subAtom2.equalsIgnoreCase("HB") ) )
    		{
    			nType=1;
    			if( subAtom1.equalsIgnoreCase("HB") && subAtom2.equalsIgnoreCase("HB") )
    				nType=0;
    			if(res1.equalsIgnoreCase("ALA") && subAtom1.equalsIgnoreCase("HB") )
    				nType=0;
    			if(res2.equalsIgnoreCase("ALA") && subAtom2.equalsIgnoreCase("HB") )
    				nType=0;
    			if(Math.abs(no1-no2)>4)
    				nType=0;
    		}
    		
    		//for type 3: 
    		if(res1.equalsIgnoreCase("ALA") && subAtom1.equalsIgnoreCase("HB"))
    			nType=3;
    		if(res1.equalsIgnoreCase("LEU") && subAtom1.equalsIgnoreCase("HD") )
    			nType=3;
    		if(res1.equalsIgnoreCase("LEU") && subAtom1.equalsIgnoreCase("HD") )
    			nType=3;
    		if(res1.equalsIgnoreCase("MET") && subAtom1.equalsIgnoreCase("HE") )
    			nType=3;
    		if(res1.equalsIgnoreCase("VAL") && subAtom1.equalsIgnoreCase("HG") )
    			nType=3;
    		if(res1.equalsIgnoreCase("THR") && atom1.equalsIgnoreCase("HG2") )
    			nType=3;
    		if(res1.equalsIgnoreCase("ILE") && ( atom1.equalsIgnoreCase("HD1") || atom1.equalsIgnoreCase("HG2") ) )
    			nType=3;
    		
    		if(res2.equalsIgnoreCase("ALA") && subAtom2.equalsIgnoreCase("HB"))
    			nType=3;
    		if(res2.equalsIgnoreCase("LEU") && subAtom2.equalsIgnoreCase("HD") )
    			nType=3;
    		if(res2.equalsIgnoreCase("LEU") && subAtom2.equalsIgnoreCase("HD") )
    			nType=3;
    		if(res2.equalsIgnoreCase("MET") && subAtom2.equalsIgnoreCase("HE") )
    			nType=3;
    		if(res2.equalsIgnoreCase("VAL") && subAtom2.equalsIgnoreCase("HG") )
    			nType=3;
    		if(res2.equalsIgnoreCase("THR") && atom2.equalsIgnoreCase("HG2") )
    			nType=3;
    		if(res2.equalsIgnoreCase("ILE") && ( atom2.equalsIgnoreCase("HD1") || atom2.equalsIgnoreCase("HG2") ) )
    			nType=3;
    		
    		//for type 2:
    		if(nType==0)
    			nType=2;
    		
    		if(nType==1)
    		{
    			exponent=6.0;
    			constK=constant;
    		}
    		else if(nType==2)
    		{
    			exponent=4.0;
    			constK=constB;
    		}
    		else if (nType==3)
    		{
    			exponent=4.0;
    			constK=constC;
    		}
    		double temp=(double)(constK/intensity);
    		double distance=Math.pow(temp, (double)(1.0/exponent));
    		if(distance <2.4)
    			distance=2.4;
    		if(distance>6.0)
    			distance=6.0;
    		noe.setDistUpper(distance);
    		
    	}//for(int i=0;i<vecNoe.size();i++)
    	Vector vecNoeNew=new Vector();
    	vecNoeNew.addAll(vecNoe);
    	return vecNoeNew;
    }
    
    /**
     * calibrate one NOE peak, using a calibration strategy similar to CYANA.
     * 
     * @param constant constant from sse backbone
     * @param bkNoe the bk noe
     * @param noesy the noesy
     * 
     * @return distance after calibraiton
     */
    public double SetCalibrationOneCyana(BackNoe bkNoe,Noesy noesy,double constant)
    {    	
    	double constB=(double)(constant/(2.4*2.4));
    	double constC=(double)(constB/3.0);
		double exponent=0.0;
		double constK=constant;
		
		int no1=bkNoe.getFirstResNo();
		int no2=bkNoe.getSecondResNo();
		String res1=bkNoe.getFirstResName();
		String res2=bkNoe.getSecondResName();
		String atom1=bkNoe.getH1Name();
		String atom2=bkNoe.getH2Name();		
		double intensity=noesy.getIntensity(); 		
    	double dividNum=1.0;
		int nType=0;
		
		String subAtom1=atom1;
		String subAtom2=atom2;
		if(atom1.length()>=2)
			subAtom1=atom1.substring(0,2);
		if(atom2.length()>=2)
			subAtom2=atom2.substring(0,2);
		
		//for type 1: 
		if( (subAtom1.equalsIgnoreCase("H")|| subAtom1.equalsIgnoreCase("HN") || subAtom1.equalsIgnoreCase("HA") ||subAtom1.equalsIgnoreCase("HB")) &&
				(subAtom2.equalsIgnoreCase("H")||subAtom2.equalsIgnoreCase("HN")||subAtom2.equalsIgnoreCase("HB") ) )
		{
			nType=1;
			if( subAtom1.equalsIgnoreCase("HB") && subAtom2.equalsIgnoreCase("HB") )
				nType=0;
			if(res1.equalsIgnoreCase("ALA") && subAtom1.equalsIgnoreCase("HB") )
				nType=0;
			if(res2.equalsIgnoreCase("ALA") && subAtom2.equalsIgnoreCase("HB") )
				nType=0;
			if(Math.abs(no1-no2)>4)
				nType=0;
		}
		
		//for type 3: 
		if(res1.equalsIgnoreCase("ALA") && subAtom1.equalsIgnoreCase("HB"))
		{	nType=3;dividNum=dividNum*3;}
		else if(res1.equalsIgnoreCase("LEU") && atom1.equalsIgnoreCase("HD") )
		{	nType=3;dividNum=dividNum*9;}	
		else if(res1.equalsIgnoreCase("LEU") && ( atom1.equalsIgnoreCase("HD1") || atom1.equalsIgnoreCase("HD2")))
		{	nType=3;dividNum=dividNum*9.0;}
		else if(res1.equalsIgnoreCase("MET") && subAtom1.equalsIgnoreCase("HE") )
		{	nType=3;dividNum=dividNum*3;}
		else if(res1.equalsIgnoreCase("VAL") && atom1.equalsIgnoreCase("HG") )
		{	nType=3;dividNum=dividNum*9;}
		else if(res1.equalsIgnoreCase("VAL") && ( atom1.equalsIgnoreCase("HG1") || atom1.equalsIgnoreCase("HG2")))
		{	nType=3;dividNum=dividNum*9;}
		else if(res1.equalsIgnoreCase("THR") && atom1.equalsIgnoreCase("HG2") )
		{	nType=3;dividNum=dividNum*3;}
		else if(res1.equalsIgnoreCase("ILE") && ( atom1.equalsIgnoreCase("HD1") || atom1.equalsIgnoreCase("HG2") ) )
		{	nType=3;dividNum=dividNum*3;}
		else if((res1.equalsIgnoreCase("CYS")||res1.equalsIgnoreCase("ASP")||res1.equalsIgnoreCase("GLU")
				||res1.equalsIgnoreCase("PHE")|| res1.equalsIgnoreCase("HIS") ||res1.equalsIgnoreCase("LYS")
				||res1.equalsIgnoreCase("LEU") || res1.equalsIgnoreCase("MET")|| res1.equalsIgnoreCase("ASN")
				||res1.equalsIgnoreCase("PRO")||res1.equalsIgnoreCase("GLN")||res1.equalsIgnoreCase("ARG")||
				res1.equalsIgnoreCase("SER")||res1.equalsIgnoreCase("TRP")||res1.equalsIgnoreCase("TYR"))&& subAtom1.equalsIgnoreCase("HB") )
		{
			dividNum=dividNum*2;
		}
		else if(res1.equalsIgnoreCase("TYR") && (subAtom1.equalsIgnoreCase("HD") || subAtom1.equalsIgnoreCase("HE")))
			dividNum=dividNum*2;
	
		if(res2.equalsIgnoreCase("ALA") && subAtom2.equalsIgnoreCase("HB"))
		{	nType=3;dividNum=dividNum*3;}
		else if(res2.equalsIgnoreCase("LEU") && atom2.equalsIgnoreCase("HD") )
		{	nType=3;dividNum=dividNum*9.0;}		
		else if(res2.equalsIgnoreCase("LEU") && ( atom2.equalsIgnoreCase("HD1") || atom2.equalsIgnoreCase("HD2")) )
		{	nType=3;dividNum=dividNum*9.0;}	
		else if(res2.equalsIgnoreCase("MET") && subAtom2.equalsIgnoreCase("HE") )
		{	nType=3;dividNum=dividNum*3;}
		else if(res2.equalsIgnoreCase("VAL") && atom2.equalsIgnoreCase("HG") )
		{	nType=3;dividNum=dividNum*9;}
		else if(res2.equalsIgnoreCase("VAL") && (atom2.equalsIgnoreCase("HG1") || atom2.equalsIgnoreCase("HG2")))
		{	nType=3;dividNum=dividNum*9;}
		else if(res2.equalsIgnoreCase("THR") && atom2.equalsIgnoreCase("HG2") )
		{	nType=3;dividNum=dividNum*3;}
		else if(res2.equalsIgnoreCase("ILE") && ( atom2.equalsIgnoreCase("HD1") || atom2.equalsIgnoreCase("HG2") ) )
		{	nType=3;dividNum=dividNum*3;}
		else if((res2.equalsIgnoreCase("CYS")||res2.equalsIgnoreCase("ASP")||res2.equalsIgnoreCase("GLU")
				||res2.equalsIgnoreCase("PHE")|| res2.equalsIgnoreCase("HIS") ||res2.equalsIgnoreCase("LYS")
				||res2.equalsIgnoreCase("LEU") || res2.equalsIgnoreCase("MET")|| res2.equalsIgnoreCase("ASN")
				||res2.equalsIgnoreCase("PRO")||res2.equalsIgnoreCase("GLN")||res2.equalsIgnoreCase("ARG")||
				res2.equalsIgnoreCase("SER")||res2.equalsIgnoreCase("TRP")||res2.equalsIgnoreCase("TYR"))&& subAtom2.equalsIgnoreCase("HB") )
		{
			dividNum=dividNum*2;
		}
		else if(res2.equalsIgnoreCase("TYR") && (subAtom2.equalsIgnoreCase("HD") || subAtom2.equalsIgnoreCase("HE")))
			dividNum=dividNum*2;
		
		//for type 2:
		if(nType==0)
			nType=2;
		
		if(nType==1)
		{
			exponent=6.0;
			constK=constant;
		}
		else if(nType==2)
		{
			exponent=4.0;
			constK=constB;
		}
		else if (nType==3)
		{
			exponent=4.0;
			constK=constC;
			intensity=(double)(intensity/dividNum);
		}
		double temp=(double)(constK/intensity);
		double distance=Math.pow(temp, (double)(1.0/exponent));
		if(distance <2.4)
			distance=2.4;
		if(distance>6.0)
			distance=6.0;
		return distance;    	
    }
    
    /**
     * incompleted..., this function can be delested...
     * the following rules are used to calibrate the constant k
     * 1. distance between HA-HB in ALA is less than 2.5A (unique assignmen
     * 
     * 
     * Note: unique assignment, not close to diagonal line
     * 
     * @param vecNoesy before volume calibration
     * @param vecSeq sse backbone
     * @param vecAsg resonance assignment list
     * 
     * @return after calibraiton
     */
    public Vector SetCalibrationC13Sec(Vector vecNoesy,Vector vecSeq,Vector vecAsg)
    {
    	double errH1=0.02,errHeavy=0.1, errH2=0.02;
    	Peak pk=new Peak();
    	Assign asg=new Assign();
    	Pdb pp=new Pdb();
    	Collections.sort(vecNoesy, new Noesy.IntensityComparator()); 
    	double counter=0.0;
    	double k_sum=0.0;
    	for(int i=0;i<vecSeq.size();i++)
    	{
    		Assign asign=(Assign)vecSeq.elementAt(i);
    		String res=asign.getResidueType();
    		int no=asign.getResidueNo();
    		
    		if(res.equalsIgnoreCase("ALA"))
    		{
    			double csHB=pk.getCSFromList(vecAsg,no,"HB");
    			double csCB=pk.getCSFromList(vecAsg,no,"CB");
    			double csHA=pk.getCSFromList(vecAsg,no,"HA"); 
    			//for the strongest peak
    			double maxInten=-999.9;
    			double noeH1Save=-999.9,noeH2Save=-999.9,noeHeavySave=-999.9;
    			boolean isFound=false;
    			for(int j=0;j<vecNoesy.size();j++)
    			{
    				Noesy noesy=(Noesy)vecNoesy.elementAt(j);
    				double noeH1=noesy.getH1();
        			double noeH2=noesy.getH2();
        			double noeHeavy=noesy.getHeavy();
        			double csInten=noesy.getIntensity();
        			
        			if(Math.abs(csHB-noeH1)<errH1 && Math.abs(noeHeavy-csCB)<errHeavy && Math.abs(csHA-noeH2)<errH2)
        			{
        				isFound=true;
        				if(maxInten<csInten)
        				{
        					maxInten=csInten;
        					noeH1Save=noeH1;
        					noeH2Save=noeH2;
        					noeHeavySave=noeHeavy;
        				}
        			}//if(Math.abs(csHN-noeH1)<errH1 && Math.abs(noeHeavy-csN)<errHeavy )    
    			}//for(int j=0;j<vecNoesy.size();j++)
    			if(!isFound)
    				continue;
    			counter=counter+1;
    			maxInten=(double)(maxInten/3.0);
    			k_sum=k_sum+(double)(maxInten/Math.pow(2.7,-6));
    		}//if(res.equalsIgnoreCase("ALA"))
    		
    		if(res.equalsIgnoreCase("LEU"))
    		{
    			double csHG=pk.getCSFromList(vecAsg,no,"HG");
    			double csCG=pk.getCSFromList(vecAsg,no,"CG");
    			double csHD1=pk.getCSFromList(vecAsg,no,"HD1"); 
    			double csHD2=pk.getCSFromList(vecAsg,no,"HD2");
    			//for the strongest peak
    			double maxInten=-999.9;
    			double noeH1Save=-999.9,noeH2Save=-999.9,noeHeavySave=-999.9;
    			boolean isFound=false;
    			for(int j=0;j<vecNoesy.size();j++)
    			{
    				Noesy noesy=(Noesy)vecNoesy.elementAt(j);
    				double noeH1=noesy.getH1();
        			double noeH2=noesy.getH2();
        			double noeHeavy=noesy.getHeavy();
        			double csInten=noesy.getIntensity();
        			
        			if(Math.abs(csHG-noeH1)<errH1 && Math.abs(noeHeavy-csCG)<errHeavy && Math.abs(csHD1-noeH2)<errH2)
        			{
        				isFound=true;
        				if(maxInten<csInten)
        				{
        					maxInten=csInten;
        					noeH1Save=noeH1;
        					noeH2Save=noeH2;
        					noeHeavySave=noeHeavy;
        				}
        			}//if(Math.abs(csHN-noeH1)<errH1 && Math.abs(noeHeavy-csN)<errHeavy )    
    			}//for(int j=0;j<vecNoesy.size();j++)
    			if(!isFound)
    				continue;
    			counter=counter+1;
    			maxInten=(double)(maxInten/3.0);
    			k_sum=k_sum+(double)(maxInten/Math.pow(2.7,-6));
    		}//if(res.equalsIgnoreCase("LEU"))
    		
    	}//for(int i=0;i<vecSeq.size();i++)
       	//compute the constant k
    	double constant=(double)(k_sum/counter);
    	
    	Vector vecNewNoesy=new Vector();
       //distance calibration for every peak:
    	for (int i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy=(Noesy)vecNoesy.elementAt(i);
    		double intensity=noesy.getIntensity();
    		double temp=(double)(constant/intensity);
    		double distance=Math.pow(temp, (double)(1.0/6.0));
    		if(distance <2.4)
    			distance=2.4;
    		if(distance>6.0)
    			distance=6.0;
    		noesy.setUpperDist(distance);   
    		vecNewNoesy.add(noesy);
    	}//for (i=0;i< vecNoesy.size();i++)    	
    	return vecNewNoesy;
    }
    
    /**
     * we use the HN(i)-HA(i-1) in SSE regions to calibrate distance from peak
     * intensity (dist<2.5A).
     * Function: I=kd^(-6)
     * 
     * @param vecNoesy before volume calibration
     * @param vecSSEBB sse backbone
     * @param vecAsg resonance assignment list
     * @param constantK the constant k
     * 
     * @return after calibraiton
     */
    public Vector<Noesy> SetCalibrationN15(Vector<Noesy> vecNoesy, Vector<Pdb> vecSSEBB,Vector<H1CS> vecAsg,double[] constantK)
    {
    	Vector vecNewNoesy=new Vector();
    	double errH1=0.03,errHeavy=0.1, errH2=0.015;
    	double volume=0.0;
    	
    	double inten;
    
    	Collections.sort(vecNoesy, new Noesy.IntensityComparator());     	
    	Peak pk =new Peak();
    	double counter=0.0;
    	double k_sum=0.0;

    	for(int i=0;i<vecSSEBB.size();i++)
    	{
    		Pdb pdb=(Pdb)vecSSEBB.elementAt(i);
    		int resNo=pdb.getResidueNo();
    		double csHN=pk.getCSFromList(vecAsg,resNo,"H");
    		double csN=pk.getCSFromList(vecAsg,resNo,"N");
    		
    		//get the chemical shift of HA in proceeding residue    		
    		boolean isPreHA=false;
    		for(int j=0;j<vecSSEBB.size();j++)
    		{
    			if(i==j)
    				continue;
    			Pdb pdbPre=(Pdb)vecSSEBB.elementAt(j);
    			int resNoPre=pdbPre.getResidueNo();
    			if(resNoPre==(resNo-1))
    				isPreHA=true;
    		}//for(int j=0;j<vecSSEBB.size();j++)
    		if(!isPreHA)
    			continue;
    		double csPreHA=pk.getCSFromList(vecAsg,resNo-1,"HA");    		    		
    		
    		//we don't use ref peaks near water line
    		if(Math.abs(csPreHA-4.78)<0.04 || Math.abs(csHN-4.78)<0.04)
    			continue;
    		
    		//get the strongest NOE cross peak within error window
    		double maxInten=-999.9;    
    		double noeH1Save=-999.9, noeH2Save=-999.9,noeHeavySave=-999.9;
    		for(int j=0;j<vecNoesy.size();j++)
    		{
    			Noesy noesy=(Noesy)vecNoesy.elementAt(j);
    			double noeH1=noesy.getH1();
    			double noeH2=noesy.getH2();
    			double noeHeavy=noesy.getHeavy();
    			double csInten=noesy.getIntensity();
    			
    			if(Math.abs(csHN-noeH1)<errH1 && Math.abs(noeHeavy-csN)<errHeavy )
    			{
    				if(maxInten<csInten)
    				{
    					maxInten=csInten;
    					noeH1Save=noeH1;
    					noeH2Save=noeH2;
    					noeHeavySave=noeHeavy;
    				}
    			}//if(Math.abs(csHN-noeH1)<errH1 && Math.abs(noeHeavy-csN)<errHeavy )    			
    		}//for(int j=0;j<vecNoesy.size();j++)
    		
    		//check whether HA in preceeding residue also cross the strongest peak
    		if(Math.abs(noeH2Save-csPreHA)<errH2)
    		{
    			counter=counter+1;
    			k_sum=k_sum+(double)(maxInten/Math.pow(2.7,-6));
    		}//if(Math.abs(noeH2Save-csPreHA)<errH2)   		
    		
    	}//for(i=0;i<vecSSEBB.size();i++)
    	
    	double constant=0.0;
    	
    	
    	double count=0.0;
    	double average=0.0;////for testing....
       //distance calibration for every peak:
    	for (int i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy=(Noesy)vecNoesy.elementAt(i);
    		double intensity=noesy.getIntensity();
    		
    		count=count+1.0;
    		average=average+intensity;//for testing...
    		double temp=(double)(constant/intensity);
    		double distance=Math.pow(temp, (double)(1.0/6.0));
    		if(distance <2.4)
    			distance=2.4;
    		if(distance>6.0)
    			distance=6.0;
    		noesy.setUpperDist(distance);   
    		vecNewNoesy.add(noesy);
    	}//for (i=0;i< vecNoesy.size();i++)   
    	
    	average=(double)(average/count);
    	constantK[0]=(double)(average/Math.pow(3.4,-6));
    	
    	return vecNewNoesy;
    }
    
    /**
     * delete repeated NOE peaks.
     * 
     * @param vecOldNoesy the vec old noesy
     * 
     * @return the vector
     */
    public Vector<Noesy> DeleteRepeatedNoesy3D(Vector<Noesy> vecOldNoesy)
    {
    	Vector vecNoesyNew=new Vector();
    	for(int i=0;i<vecOldNoesy.size();i++)
    	{
    		Noesy noesy=(Noesy)vecOldNoesy.elementAt(i);
    		double csH1=noesy.getH1();
    		double csH2=noesy.getH2();
    		double csHeavy=noesy.getHeavy();
    		boolean isInPre=false;
    		for(int j=0;j<vecNoesyNew.size();j++)
    		{
    			Noesy noesyNew=(Noesy)vecNoesyNew.elementAt(j);
    			double csH1New=noesyNew.getH1();
    			double csH2New=noesyNew.getH2();
    			double csHeavyNew=noesyNew.getHeavy();
    			if(Math.abs(csH1-csH1New)<0.001 && Math.abs(csH2-csH2New)<0.001 && Math.abs(csHeavy-csHeavyNew)<0.001 )
    				isInPre=true;
    		}//for(int j=0;j<vecNoesyNew.size();j++)
    		if(!isInPre)
    			vecNoesyNew.add(noesy);
    	}//for(int i=0;i<vecOldNoesy.size();i++)
    	return vecNoesyNew;
    }

   /**
    * Read NOESY peaks in NMRView format
    * The required file format is:.
    * 
    * @param filename the name of the file
    * 
    * @return a vector of HnNOE object
    */
    public Vector<Noesy> NoesyReaderNMRView(String filename)
    {
	Vector inputs = new Vector();
	double csH1 = 0.0, csHeavy = 0.0, csH2 = 0.0, intensity = 0.0, uncertainty=0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	String strTemp="";
	double csID=0;
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	String aaType = "", ss ="";
	int counter=0;
	int    index = -1;
	try
	{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	   
	    stop:
	    while(true) 
	    {
	    	index = ss.trim().indexOf("#");
	    	if (index==0)
	    	{
	    		if ((ss = in.readLine()) == null)
		    		break stop;
	    		continue;	    		
	    	}
	    	
	    	st = new StringTokenizer(ss);
	    	if (st.hasMoreTokens())
	    		csID = new Double(st.nextToken()).doubleValue();
	    	else 
	    	{
	    		if ((ss = in.readLine()) == null)
		    		break stop;
	    		continue;
	    	}
	    	
	    	//delimit {}	    	
	    	boolean isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    	
	    	if (st.hasMoreTokens())
	    		csH1 = new Double(st.nextToken()).doubleValue();	
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    	
	    
	    	if (st.hasMoreTokens())
	    		csH2 = new Double(st.nextToken()).doubleValue();
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    	
	    	
	    	if (st.hasMoreTokens())
	    		csHeavy = new Double(st.nextToken()).doubleValue();
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    		        	
	    	if (st.hasMoreTokens())
	    	{
	    		intensity = new Double(st.nextToken()).doubleValue();
	    	
	    	}
	    	if (st.hasMoreTokens())
	    		uncertainty = new Double(st.nextToken()).doubleValue();
	    	
	    	
	    	counter=counter+1;
	    	inputs.add(new Noesy(0, "", csH1, csHeavy, csH2, intensity,uncertainty));
	    	if ((ss = in.readLine()) == null)
	    		break stop;
	    	}
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	// System.out.println("The total number of HN-NOE frequency triples is: " + counter);
	return inputs;
    }
 
    /**
     * Read 4D NOESY peaks in NMRView format
     * The required file format is:.
     * 
     * @param filename the name of the file
     * 
     * @return a vector of HnNOE object
     */
    public Vector<Noesy> NoesyReaderNMRView4D(String filename)
    {
	Vector inputs = new Vector();
	double csH1 = 0.0, csHeavy = 0.0, csH2 = 0.0, intensity = 0.0, uncertainty=0.0;
	double csHeavy1=0.0, csHeavy2=0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	String strTemp="";
	double csID=0;
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	String aaType = "", ss ="";
	int counter=0;
	int    index = -1;
	try
	{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    
	    stop:
	    while(true) 
	    {
	    	index = ss.trim().indexOf("#");
	    	if (index==0)
	    	{
	    		if ((ss = in.readLine()) == null)
		    		break stop;
	    		continue;	    		
	    	}
	    	
	    	st = new StringTokenizer(ss);
	    	if (st.hasMoreTokens())
	    		csID = new Double(st.nextToken()).doubleValue();
	    	else 
	    	{
	    		if ((ss = in.readLine()) == null)
		    		break stop;
	    		continue;
	    	}
	    	//delimit {}	    	
	    	boolean isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    	
	    	
	    	
	    	if (st.hasMoreTokens())
	    		csHeavy2 = new Double(st.nextToken()).doubleValue();	//cs-heavy 2
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	

	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)	    	
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)
	    	
	    	
	    	if (st.hasMoreTokens())
	    		csH1 = new Double(st.nextToken()).doubleValue(); //cs h1
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)	    	
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)

	    	if (st.hasMoreTokens())
	    		csHeavy1= new Double(st.nextToken()).doubleValue(); //cs Heavy1
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)	    	
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)    	
	    	
	    	if (st.hasMoreTokens())
	    		csH2= new Double(st.nextToken()).doubleValue(); //cs H2
	    	
	    	//the order here is important, need to refer to the input format    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	//delimit {}	    	
	    	isRead=true;
	    	while(isRead)
	    	{
	    		if (st.hasMoreTokens())
		    		strTemp = new String(st.nextToken());
		    	for(int k=0;k<strTemp.length();k++)
		    	{
		    		String strOne=strTemp.substring(k,k+1);
		    		if ( strOne.equalsIgnoreCase("}") )
		    		{
		    			isRead=false;
		    			break;		    			
		    		}
		    	}
	    	}//while(isRead)

	    	
	    	if (st.hasMoreTokens())
	    	{
	    		intensity = new Double(st.nextToken()).doubleValue();
	    		
	    	}
	    	if (st.hasMoreTokens())
	    		uncertainty = new Double(st.nextToken()).doubleValue();
	    	    	
	    
	    	counter=counter+1;
	    	inputs.add(new Noesy(0, "", csH1, csHeavy1, csH2, csHeavy2,intensity,uncertainty));
	    	
	    	if ((ss = in.readLine()) == null)
	    		break stop;
	    	}
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	 System.out.println("The total number of HN-NOE frequency triples is: " + counter);
	return inputs;
    }
    
    /**
     * Read NOESY peaks in xeasy format
     * The required file format is:.
     * 
     * @param filename the name of the file
     * 
     * @return a vector of HnNOE object
     */
    public Vector<Noesy> NoesyReader(String filename)
    {
	Vector inputs = new Vector();
	double csH1 = 0.0, csHeavy = 0.0, csH2 = 0.0, intensity = 0.0, uncertainty=0.0;
	Vector csVec = new Vector();
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	String strTemp="";
	double csID=0;
	int index1=0, index2=0;
	String order="";
	double value=0.0;
	int no =0;
	String aaType = "", ss ="";
	int counter=0;
	int    index = -1;
	try
	{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	   
	    stop:
	    while(true) 
	    {
	    	index = ss.trim().indexOf("#");
	    	if (index==0)
	    	{
	    		if ((ss = in.readLine()) == null)
		    		break stop;
	    		continue;	    		
	    	}
	    	
	    	st = new StringTokenizer(ss);
	    	if (st.hasMoreTokens())
	    		csID = new Double(st.nextToken()).doubleValue();
	    	if (st.hasMoreTokens())
	    		csHeavy = new Double(st.nextToken()).doubleValue();
	    	//the order here is important, need to refer to the input format    	
	    	if (st.hasMoreTokens())
	    		csH2 = new Double(st.nextToken()).doubleValue();
	    	
	    	if (st.hasMoreTokens())
	    		csH1 = new Double(st.nextToken()).doubleValue();	
	    	
	    	
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	if (st.hasMoreTokens())
	    		strTemp = new String(st.nextToken());
	    	
	    	if (st.hasMoreTokens())
	    	{
	    		intensity = new Double(st.nextToken()).doubleValue();
	    		
	    	}
	    	if (st.hasMoreTokens())
	    		uncertainty = new Double(st.nextToken()).doubleValue();
	    	
	    	counter=counter+1;
	    	inputs.add(new Noesy(0, "", csH1, csHeavy, csH2, intensity,uncertainty));
	    	if ((ss = in.readLine()) == null)
	    		break stop;
	    	}
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	// System.out.println("The total number of HN-NOE frequency triples is: " + counter);
	return inputs;
    }
     
    /**
     * Extract all the HnNoe objects with proton CS or N15 CS in the
     * range of [a, b] from an array of NOE peaks.
     * 
     * @param nucType the type of nucleus: H1 or N15
     * @param a   the lower limit of the range
     * @param b   the up limit of the range
     * @param pkVec an array of NOE peaks.
     * 
     * @return the vector
     */
    public Vector rangeSearch(String nucType, double a, double b, Vector pkVec){
	Vector outVec = new Vector();
	double csValue = 0.0;
	for (int i=0; i< pkVec.size(); i++){
	    if (nucType.equals("H1"))
		csValue = ((HnNoe)pkVec.elementAt(i)).getHN();
	    else if (nucType.equals("N15"))
		csValue = ((HnNoe)pkVec.elementAt(i)).getN15();
	    if (csValue >= a && csValue <= b)
		outVec.add(pkVec.elementAt(i));
	}
	return outVec;
    }

  /**
   * a method to merge NOE peaks since some NOE peaks are picked with
   * a lightly different CS more than once.
   * 
   * @param pkVec the pk vec
   * @param epsH the eps h
   * @param epsN the eps n
   * 
   * @return the vector
   */
    public Vector mergeHnNoe(Vector pkVec, double epsH, double epsN){
	double csHN = 0.0;
	double csN15 = 0.0;
	double csHNTmp = 0.0;
	double csN15Tmp = 0.0;
	Vector outVec = new Vector();
	Vector allVec = new Vector();
	double csValue = 0.0;
	HnNoe hncc = new HnNoe();
	int i = 0, j = 0;
	Collections.sort(pkVec, new HnNoe.n15Comparator());
	int count = 0, cnt1 = 0, cnt2 = 0;
	int N = pkVec.size();
	while (!pkVec.isEmpty()){
	    hncc = (HnNoe)pkVec.elementAt(0);
	    csHN  = hncc.getHN();
	    csN15 = hncc.getN15();
	    outVec.add(hncc);
	    cnt1 = 0;
	    for (i=1; i< pkVec.size(); i++){
		hncc = (HnNoe)pkVec.elementAt(i);
		csHNTmp = hncc.getHN();
		csN15Tmp = hncc.getN15();
		if ( Math.abs(csHNTmp - csHN) < epsH && Math.abs(csN15Tmp - csN15) < epsN){
		    outVec.add(hncc);
		    cnt1++;
		}
	    }
	    allVec.add(outVec);
	    outVec = new Vector();
	    cnt2 = 0;
	    for (j=0; j< pkVec.size();){
		hncc = (HnNoe)pkVec.elementAt(j);
		csHNTmp = hncc.getHN();
		csN15Tmp = hncc.getN15();
		if ( Math.abs(csHNTmp - csHN) < epsH && Math.abs(csN15Tmp - csN15) < epsN){
		    pkVec.removeElementAt(j);
		    cnt2++;
		}else j++;
	    }	   
	    pkVec.trimToSize();
	    N = pkVec.size();

	}
	return allVec;
    }
    
    /**
     * print NOESY peak list to file, in XEASY format.
     * 
     * @param vecH1CS resonance assignment list
     * @param out the out
     * 
     * @return void
     */
    public void PrintAllNoesyToFileXEASY(Vector<Noesy> vecNoesy,PrintWriter out) {
    	//out.println("!The exact assignments are:");
    	int preNo=-99;    	
    	for(int i=0;i<vecNoesy.size();i++) 	{
    		Noesy noesy=(Noesy)vecNoesy.elementAt(i);
    		double csHeavy=noesy.getHeavy();
    		double csH2=noesy.getH2();
    		double csH1=noesy.getH1();
    		double csIntensity=noesy.getIntensity();
    		double csUncertain=noesy.getUncertainty();
    		
    		String strCSHeavy=String.format("%.3f", csHeavy);
    		String strCSH2=String.format("%.3f", csH2);
    		String strCSH1=String.format("%.3f", csH1);
    		String strIntensity=String.format("%.3f", csIntensity);
    		String strUncertain=String.format("%.3f", csUncertain);
    		

    		out.println(i+ "  "+ strCSHeavy+ "   "+strCSH2+"   "+strCSH1+"  "+ "1   U" +"      "+strIntensity+"   " +strUncertain
    				+ "		"+"a   0    0 0 0   0");     		
    	}//for(int i=0;i<vecH1CS.size();i++)
    	
    }
    
    /**
     * print NOESY peak list to file, in NMRView format.
     * 
     * @param vecH1CS resonance assignment list
     * @param out the out
     * 
     * @return void
     */
    public void PrintAllNoesyToFileNMRView(Vector<Noesy> vecNoesy,PrintWriter out) {
    	//out.println("!The exact assignments are:");
    	int preNo=-99;    	
    	for(int i=0;i<vecNoesy.size();i++) 	{
    		Noesy noesy=(Noesy)vecNoesy.elementAt(i);
    		double csHeavy=noesy.getHeavy();
    		double csH2=noesy.getH2();
    		double csH1=noesy.getH1();
    		double csIntensity=noesy.getIntensity();
    		double csUncertain=noesy.getUncertainty();
    		
    		String strCSHeavy=String.format("%.3f", csHeavy);
    		String strCSH2=String.format("%.3f", csH2);
    		String strCSH1=String.format("%.3f", csH1);
    		String strIntensity=String.format("%.3f", csIntensity);
    		String strUncertain=String.format("%.3f", csUncertain);
    		

    		out.println(i+ "  "+"{}"+ strCSH1+"  0.00  0.00  ++  0.0  {}  {}  "+strCSH2+"  0.00  0.00  ++  0.0  {}  {}  "+  
    				strCSHeavy+ "  0.00  0.00  ++  0.0  {}  "+strIntensity+ "   "+ strUncertain+ "  0 {} 0 ");
    				
    				
    	}//for(int i=0;i<vecH1CS.size();i++)
    	
    }

}
