package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Pdb.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

 
import java.io. *;
import java.util. *;
import java.text.NumberFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.FieldPosition;

import Jampack.JampackException;

// TODO: Auto-generated Javadoc
/** * 
 *  
 *   This class provides data structures and functions for processing and generating PDB files and extracting
 *   backbone angles etc.
 *   Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
 */
public class Pdb{
    
    /** The residue no. */
    private int residueNo;
    
    /** The residue name. */
    private String residue;
    
    /** The atom vec. */
    private Vector atomVec = new Vector(); //all the atoms belong to the residue
    
    /** The ratio for storing the probability of each rotamer. */
    private double ratio;//
    
    /**
     * Instantiates a new pdb.
     */
    public Pdb(){
	residueNo = 0;
	residue = null;
	atomVec = null;
	ratio=0.0;
    }
    
    /**
     * Instantiates a new pdb.
     * 
     * @param No the no
     */
    public Pdb(int No){
	residueNo = No;
	atomVec = new Vector();
	residue = "";ratio=0.0;
    }
    
    /**
     * Instantiates a new pdb.
     * 
     * @param No the no
     * @param res the res
     * @param aa the aa
     */
    public Pdb(int No,  String res, Vector aa){
	residueNo = No;
	atomVec = aa;
	residue = res;ratio=0.0;
    }
    
    /**
     * Instantiates a new pdb.
     * 
     * @param No the no
     * @param res the res
     * @param aa the aa
     * @param ra the ra
     */
    public Pdb(int No,  String res, Vector aa,double ra)
    {
    	residueNo = No;
    	atomVec = aa;
    	residue = res;
    	ratio=ra;
        }
    
    /**
     * Instantiates a new pdb.
     * 
     * @param pp the pp
     */
    public Pdb(Pdb pp)
    {
    	residueNo = pp.getResidueNo();
    	//atomVec =pp.getAtomVec();
    	residue = pp.getResidue();
    	ratio=pp.getRotamerRatio();
    	Vector<Cartesian> vecAtom=pp.getAtomVec();
    	for(int i=0;i<vecAtom.size();i++){
    		Cartesian cart=vecAtom.elementAt(i);
    		Cartesian carNew=new Cartesian(cart);
    		atomVec.add(carNew);
    	}
    	
    }
    //getting the values	
    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the atom vec.
     * 
     * @return the atom vec
     */
    public Vector<Cartesian> getAtomVec(){
	return atomVec;
    }	
    
    /**
     * Gets the residue.
     * 
     * @return the residue
     */
    public String getResidue(){
	return residue;
    }	
    
    /**
     * Gets the rotamer ratio.
     * 
     * @return the rotamer ratio
     */
    public double getRotamerRatio(){
    	return ratio;
        }	
    //Set method
    /**
     * Sets the residue no.
     * 
     * @param no the new residue no
     */
    public void setResidueNo(int no){
	residueNo = no;
    }	
    
    /**
     * Sets the residue.
     * 
     * @param res the new residue
     */
    public void setResidue(String res){
	residue = res;
    }	
    
    /**
     * Sets the atom vec.
     * 
     * @param aVec the new atom vec
     */
    public void setAtomVec(Vector aVec){
	atomVec = aVec;
    }	
    
    /**
     * Sets the rotamer ratio.
     * 
     * @param ra the new rotamer ratio
     */
    public void setRotamerRatio(double ra){
    	ratio = ra;
        }	
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
	String desc = String.valueOf(residueNo) + "  "+ residue+"\n";
	String str="";
	for (int i=0; i<atomVec.size(); i++)
	    str += ((Cartesian) atomVec.elementAt(i)).toString()+"\n";
	return (desc+str);	
    }

    /**
     * Prints the array.
     * 
     * @param n1 the n1
     */
    public void printArray(double[] n1){
	for (int i=0; i<n1.length; i++)
	    System.out.print(n1[i]+"  ");
	System.out.println();
    }

    //A comparator for sorting the PDB object based on residue number
    /**
     * The Class PdbComparator.
     */
    public static class PdbComparator implements Comparator<Object>{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Pdb n1 = (Pdb)o1;
	    Pdb n2 = (Pdb)o2;
	    int d1 = n1.getResidueNo();
	    int d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    } 

    /**
     * Length.
     * 
     * @param v1 the v1
     * 
     * @return the double
     */
    public double length(double[] v1){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	return v1Len;
    }
    
    /**
     * cal the directioal cosines of the vector vec.
     * 
     * @param vec the vec
     * 
     * @return the double[]
     */
    public double[] dirCos(double [] vec) {
	double len = 0.0;
	double [] dirs = new double[vec.length];
	for (int i=0; i<vec.length; i++)
	    len += vec[i] * vec[i];
	for (int j=0; j<vec.length; j++)
	    dirs[j] = vec[j] / Math.sqrt(len);
	return dirs;
    }

    /**
     * calculate the angles between two vectors v1 and v2
     * The returned angle is the [0,Pi] region.
     * 
     * @param v1 the v1
     * @param v2 the v2
     * 
     * @return the double
     */
    public double interAngle(double[] v1, double[] v2){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	double v2Len = Math.sqrt(v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2]);
	double c = (v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2]) / (v1Len * v2Len);
	return Math.acos(c);  //acos [0, PI] so should be OK
    }

 /**
  * Compute the internuclear vector between two atoms.
  * 
  * @param n1 the coordinate for atom 1
  * @param n2 the coordinate for atom 2
  * 
  * @return a vector from n1->n2
  */
    public double[] internuclearVec(double[] n1, double[] n2){
        return new double[]{n2[0]-n1[0], n2[1]-n1[1], n2[2]-n1[2]};
    }
    
    /**
     * Reading Rotamer file in PDB Format!!!.
     * 
     * @param pdbFile the pdb file
     * 
     * @return a vector of Pdb object
     */
    public Vector<Pdb> readRotamerPdb(String pdbFile)
    {	
	String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="";
	int Atoms_No, residueNo = 0, noTmp = 0,index=0;
	double x_in,y_in,z_in;
	Vector pdbVec = new Vector();
	Vector atoms = new Vector();
	String form="";
	double rotaRatio=1.0;
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
	    ss = in.readLine();	    
	    if (ss.length() > 66)
	        pdbID = ss.substring(62, 66).trim();
	    stop:
	    while(true) 
	    {
			if (ss.length() > 5)
			    keywords = ss.substring(0, 6).trim();
			else keywords = ss.trim();
			index=ss.indexOf("ratio");
			if(index>0)
				rotaRatio=new Double(ss.substring(index+7, ss.length()).trim()).doubleValue();
			while(keywords.equals("ATOM") )	
			{
			    if ( ss.length() > 54)
			    {
			    	form = ss.substring(16,17).trim(); //We choose only one of it
			    	
					a1_type = ss.substring(12, 16).trim();
					r_type = ss.substring(17, 20).trim();
					residueNo = Integer.parseInt(ss.substring(23, 26).trim());
					x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
					y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
					z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
					atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));				
					
					if ((ss = in.readLine()) == null)
					    break stop;		
					
					
					
					if (ss.length() > 5)
					    str = (ss.substring(0,6)).trim();
					else 
						continue stop;
				
					if (  !(str.equals("ATOM")) )
					{
					    if(r_type.length() == 3)
					    {
							pdbVec.add(new Pdb(residueNo, r_type, atoms,rotaRatio));
							atoms = new  Vector();	
					    }
					    index=ss.indexOf("ratio");
						if(index>0)
							rotaRatio=new Double(ss.substring(index+7, ss.length()).trim()).doubleValue();
					    continue stop;
					}
					
					noTmp = Integer.parseInt(ss.substring(23, 26).trim());
					while (noTmp == residueNo)
					{
					    a1_type = ss.substring(12, 16).trim(); 
					    r_type = ss.substring(17, 20).trim();
					    residueNo = Integer.parseInt(ss.substring(23, 26).trim());
					    x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
					    y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
					    z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
					    atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
					    if ((ss = in.readLine()) == null)
						break stop;	
					   					   
					    if (ss.length() > 5)
						str = (ss.substring(0,6)).trim();
					    else continue stop;
				   
					    if (!(str.equals("ATOM")) )
					    {
							if(r_type.length() == 3)
							{
							    pdbVec.add(new Pdb(residueNo, r_type, atoms,rotaRatio));
							    atoms = new  Vector();		
							}
							index=ss.indexOf("ratio");
							if(index>0)
								rotaRatio=new Double(ss.substring(index+7, ss.length()).trim()).doubleValue();
								
							continue stop;
					    }
	
					    noTmp = Integer.parseInt(ss.substring(23, 26).trim());
					}//while (noTmp == residueNo)
					if(r_type.length() == 3)
					{
					    pdbVec.add(new Pdb(residueNo, r_type, atoms,rotaRatio));
					    atoms = new  Vector();
					}
					index=ss.indexOf("ratio");
					if(index>0)
						rotaRatio=new Double(ss.substring(index+7, ss.length()).trim()).doubleValue();
					keywords = str;		
			    }//if ( ss.length() > 54)
			}//while(keywords.equals("ATOM") )	
		
			if ((ss = in.readLine()) == null)
			{
			    pdbVec.add(new Pdb(residueNo, r_type, atoms,rotaRatio));
			    break stop;
			}
	    }//while(true) 
	    in.close();
	}catch (FileNotFoundException e) 
	{
	    System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(pdbVec,new PdbComparator() );
	return pdbVec;
    }
    
    /**
     * Reading PDB file in PDB Format!!!
     * the magic number for substring is based on format for pdb file.
     * 
     * @param pdbFile the pdb file
     * 
     * @return a vector of Pdb object
     * Comment: There is always some irregularity in PDB files so when reading
     * a real PDB file you must be cautious and ready to check the output from the reader
     * However, in our case we simply need coordinates which is much easier.
     */
    public Vector<Pdb> readPdb(String pdbFile){	
	String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="";
	int Atoms_No, residueNo = 0, noTmp = 0;
	double x_in,y_in,z_in;
	Vector pdbVec = new Vector();
	Vector atoms = new Vector();
	String form="";
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
	    ss = in.readLine();
	 
	    if (ss.length() > 66)
	        pdbID = ss.substring(62, 66).trim();
	    stop:
	    while(true) {
		if (ss.length() > 5)
		    keywords = ss.substring(0, 6).trim();
		else keywords = ss.trim();
		while(keywords.equals("ATOM") || keywords.equals("HETATM") )	{
		    if ( ss.length() > 54){		    
			form = ss.substring(16,17).trim(); //We choose only one of it
			if (form.equals("B")){  //Some Pdb file like 1CBN has A and B 
			    if ((ss = in.readLine()) == null)
				break stop;							
			    if (ss.length() > 54)
				str = (ss.substring(0,6)).trim();
			    else continue stop;

			    while (str.equals("ANISOU")){
				if ((ss = in.readLine()) == null)
				    break stop;		

				if (ss.length() > 54)
				    str = (ss.substring(0,6)).trim();
				else continue stop;
			    }
			    if (  !(str.equals("ATOM") || keywords.equals("HETATM") ) ){
				if(r_type.length() == 3){
					if(r_type.equalsIgnoreCase("HID") || r_type.equalsIgnoreCase("HIS") )
				    	r_type="HIS";
				    pdbVec.add(new Pdb(residueNo, r_type, atoms));
				    atoms = new  Vector();	
				}
				continue stop;
			    }
			}
			a1_type = ss.substring(12, 16).trim();
			r_type = ss.substring(17, 20).trim();
			residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			
			atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			if ((ss = in.readLine()) == null)
			    break stop;		
			if (ss.length() > 54)
			    str = (ss.substring(0,6)).trim();
			else continue stop;
			while (str.equals("ANISOU")){
			    if ((ss = in.readLine()) == null)
				break stop;	

			    if (ss.length() > 54)
				str = (ss.substring(0,6)).trim();
			    else continue stop;
			}
			if (  !(str.equals("ATOM") || keywords.equals("HETATM") ) ){
			    if(r_type.length() == 3){
			    	if(r_type.equalsIgnoreCase("HID") || r_type.equalsIgnoreCase("HIS") )
				    	r_type="HIS";
				pdbVec.add(new Pdb(residueNo, r_type, atoms));
				atoms = new  Vector();	
			    }
			    continue stop;
			}
			form = ss.substring(16,17).trim();
			if (form.equals("B")){
			    if ((ss = in.readLine()) == null)
				break stop;							
			    if (ss.length() > 54)
				str = (ss.substring(0,6)).trim();
			    else continue stop;

			    while (str.equals("ANISOU")){
				if ((ss = in.readLine()) == null)
				    break stop;		

				if (ss.length() > 54)
				    str = (ss.substring(0,6)).trim();
				else continue stop;
			    }
			    if (  !(str.equals("ATOM") || keywords.equals("HETATM") ) ){
				if(r_type.length() == 3){
					if(r_type.equalsIgnoreCase("HID") || r_type.equalsIgnoreCase("HIS") )
				    	r_type="HIS";
				    pdbVec.add(new Pdb(residueNo, r_type, atoms));
				    atoms = new  Vector();		
				}
				continue stop;
			    }
			}
			noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			while (noTmp == residueNo){
			    a1_type = ss.substring(12, 16).trim(); 
			    r_type = ss.substring(17, 20).trim();
			    residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    if ((ss = in.readLine()) == null)
				break stop;							
			    if (ss.length() > 54)
				str = (ss.substring(0,6)).trim();
			    else continue stop;

			    while (str.equals("ANISOU")){
				if ((ss = in.readLine()) == null)
				    break stop;		

				if (ss.length() > 54)
				    str = (ss.substring(0,6)).trim();
				else continue stop;
			    }

			    if (!(str.equals("ATOM") || keywords.equals("HETATM") ) ){
				if(r_type.length() == 3){
					if(r_type.equalsIgnoreCase("HID") || r_type.equalsIgnoreCase("HIS") )
				    	r_type="HIS";
				    pdbVec.add(new Pdb(residueNo, r_type, atoms));
				    atoms = new  Vector();		
				}
				continue stop;
			    }
			    form = ss.substring(16,17).trim();
			    if (form.equals("B")){
				if ((ss = in.readLine()) == null)
				    break stop;							
				if (ss.length() > 54)
				    str = (ss.substring(0,6)).trim();
				else continue stop;

				while (str.equals("ANISOU")){
				    if ((ss = in.readLine()) == null)
					break stop;		

				    if (ss.length() > 54)
					str = (ss.substring(0,6)).trim();
				    else continue stop;
				}
				if (  !(str.equals("ATOM") || keywords.equals("HETATM") ) ){
				    if(r_type.length() == 3){
				    	if(r_type.equalsIgnoreCase("HID") || r_type.equalsIgnoreCase("HIS") )
					    	r_type="HIS";
				    	pdbVec.add(new Pdb(residueNo, r_type, atoms));
				    	atoms = new  Vector();		
				    }
				    continue stop;
				}
			    }
			    noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			}
			if(r_type.length() == 3){
				if(r_type.equalsIgnoreCase("HID") || r_type.equalsIgnoreCase("HIS") )
			    	r_type="HIS";
			    pdbVec.add(new Pdb(residueNo, r_type, atoms));			    
			    atoms = new  Vector();
			}
			keywords = str;		
		    }
		}
		if ((ss = in.readLine()) == null){
			if(r_type.equalsIgnoreCase("HID") || r_type.equalsIgnoreCase("HIS") )
		    	r_type="HIS";
			
		    pdbVec.add(new Pdb(residueNo, r_type, atoms));
		    break stop;
		}
	    }
	    in.close();
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(pdbVec,new PdbComparator() );
	return pdbVec;
    }      
   
    /**
     * * compute the NOE satisfaction scores for all structure in an ensemble.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doCalNOEScores(String src, String strOut, String strInput)throws JampackException
    {	
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////    	
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	String strSeq="",strLocationEnsem="",strEnsemName="", strNOETable="",strNoeFormat="";
    	String strDelimitedTmp="";//not used yet
    	int maxEnsemID=0;
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    		
    	 	
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("LOCATIONENSEMBLE"))
    			strLocationEnsem  =  (String)paraMap.get("LOCATIONENSEMBLE");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");        	
    		
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");
    		
    		
    	}      
    	///////////////////////////////////////////////////////////////
    	String seqFile=src+strSeq;
    	Vector vecSeq=asg.ReaderSeq(seqFile);    
    	
    	String manualAsgFile = src+ strNOETable;
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");//////////////////0.0
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//    	read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		vecManAsg=noe_temp.noeReader(manualAsgFile,0.0);  	  ///need to be careful here  		
    		
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	vecManAsg=new Vector();
    	vecManAsg.addAll(vecManAsgNew);
    	
    	Vector vecPdb=new Vector();
    	
    	String userDir = System.getProperty("user.dir");////
		String strInputFile=userDir+strLocationEnsem;
		File myDir = new File(strInputFile);
    	String[] contents = myDir.list();//select all structures
    	if (contents == null) 
    		System.out.println(myDir.getName() + " is not a directory");
    	    
    	String strNoeScore="";
    	for (int t=0; t<contents.length;t++) 
    	{    
    		String filename = contents[t]; 
    		vecPdb=new Vector();
	    	String strEnsemPdb="";
	    	strEnsemPdb=strInputFile+filename;	    
	    	vecPdb=pp.readPdb(strEnsemPdb);
       		
    		double [] noeRmsd=new double[2];
    		double [] noeHarmScore=new double[2];
    		int [] numConflicts=new int[2];
    		
    		
    		
    		double [] vdw_sc =new double[1];
    		double [] noe_sc =new double[1];
	    	
    		noe_temp.noeSatisfiedScoreFF2(vecManAsg,vecPdb, noeRmsd,noeHarmScore,numConflicts);
    		
    		System.out.println("the noe score of " + filename+ " th pdb is: "+noeHarmScore[0] );
	    	
    		strNoeScore=strNoeScore+" , "+ noeHarmScore[0];   		
	    	
    	}//for (int j=0;j<=103;j++)
    	
    	System.out.println("============================");
    	System.out.println("Here is the NOE score : "+strNoeScore );
    	
    	System.out.println("============================");  	
    	
    	
    	
    	long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for computing NOE scores is:  "+ totalTime +" minutes");  	
    	
    }
   
    //read vdw and noe scores from xplor pdbs
    /**
     * Main1.
     * 
     * @param argv the argv
     */
    public static void main1(String[] argv) 
	{	    
    	Pdb  pp = new Pdb();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/xplor/";
    	int counter=0;
    	Vector vecPdb=new Vector();
    	Vector vecVdw=new Vector();
    	vdw vv=new vdw();
    	
    	String strVdw="";
    	String strNoeRmsd="";
    	System.out.println("vdw=: ");
    	
    	for (int j=0;j<=441;j++)
    	{
    		if (j==18 || j==235 || j==351 || j==387)
    			continue;
    		
    		//System.out.println("j= "+ j);
    		
    		
    		String strEnsemPdb=src+"ff2_complete_"+j+".pdb";
    		Vector vecEnsemble=new Vector();
    		double [] vdw_sc =new double[1];
    		double [] noe_sc =new double[1];
	    	vecEnsemble=pp.readPdbAndParaXplor(strEnsemPdb,vdw_sc, noe_sc);
	    	
	    	strVdw=strVdw+" , "+ vdw_sc[0];
    		strNoeRmsd=strNoeRmsd+" , "+ noe_sc[0];
    		
    		if(vdw_sc[0]< 1283 && noe_sc[0]< 5753)
    			System.out.println("found one id: "+ j);
    		vecVdw.add(new vdw(vecPdb, vdw_sc[0],noe_sc[0],j));
	    	
    	}//for (int j=0;j<=103;j++)
    	System.out.println("here is the strVdw :"+strVdw );
    	System.out.println("============================");
    	System.out.println("here is the NOE rmsd :"+strNoeRmsd );
    	
    	System.out.println("============================");
    	
    	Collections.sort(vecVdw, new vdw.VdwComparator() );
    	for (int i=0;i<50;i++)
    	{
    		vdw vdw_temp=(vdw)vecVdw.elementAt(i) ;
    		double modID=vdw_temp.getModelID();
    		System.out.println("the i-th "+i + ": vdw is: " +modID );
    	}
    	
    	
    	System.out.println("============================");
    	Vector vecVdwNoe=new Vector();    	
    	vecVdwNoe.addAll(vecVdw);    	
    	Collections.sort(vecVdwNoe, new vdw.VdwNoeComparator() );
    	for (int i=0;i<50;i++)
    	{
    		vdw vdw_temp=(vdw)vecVdwNoe.elementAt(i) ;
    		double modID=vdw_temp.getModelID();
    		System.out.println("the i-th "+i + ": noe is: " +modID );
    	}
    		
    }
    //calculate the NOE satisfaction score
    /**
     * Main00.
     * 
     * @param argv the argv
     */
    public static void main00(String[] argv) 
	{	    
    	Pdb  pp = new Pdb();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/xplor/";
    	int counter=0;
    	Vector vecPdb=new Vector();
    	Vector vecVdw=new Vector();
    	vdw vv=new vdw();
    	
    	
    	double[] vdwValue = new double[1];
    	boolean hasVDW = false;
    	double vdwLevel = 0.05;
    	boolean printVDWViolation = false;
    	vdw vander = new vdw();
    	Vector vdwVec = new Vector();
    	
    	String strVdw="";
    	String strNoeRmsd="";
    	System.out.println("vdw=: ");
    	String manualAsgFile=userDir+"/inputFiles/ff2_noe_xplor.tbl";
    	Assign asg = new Assign();
    	String seqFile =userDir+"/inputFiles/ff2.seq";    	
    	Vector vecSeq=asg.ReaderSeq(seqFile);    
    	
    	Noe noe_temp=new Noe();
    	
        Vector vecManAsg=noe_temp.noeReader2(manualAsgFile,0);
    	
    	Vector vecManAsgNew= noe_temp.ConvertXplorAsgToUplFormatFF2(vecManAsg,vecSeq);
    	vecManAsg=new Vector();
    	vecManAsg.addAll(vecManAsgNew);
    	
    	
    	
    	for (int j=0;j<=441;j++)
    	{
    		if (j==18 || j==235 || j==351 || j==387)
    			continue;
    		    		
    		double [] noeRmsd=new double[2];
    		double [] noeHarmScore=new double[2];
    		int [] numConflicts=new int[2];
    		
    		vecPdb=new Vector();
    		String strEnsemPdb=src+"ff2_complete_"+j+".pdb";
    		Vector vecEnsemble=new Vector();
    		double [] vdw_sc =new double[1];
    		double [] noe_sc =new double[1];
	    	
    		vecPdb=pp.readPdb(strEnsemPdb);
    		noe_temp.noeSatisfiedScoreFF2(vecManAsg,vecPdb, noeRmsd,noeHarmScore,numConflicts);
    		if(noeHarmScore[0]<10)
    		System.out.println("the noe score of " + j+ " th pdb is: "+noeHarmScore[0] );
	    
    		strNoeRmsd=strNoeRmsd+" , "+ noeHarmScore[0];
    		
    		//compute the number of soft steric clash
    		vdwVec=new Vector();
    		vdwVec = vander.convert2VDW(vecPdb);
    		
	    	
    	}//for (int j=0;j<=103;j++)
    	System.out.println("here is the strVdw :"+strVdw );
    	System.out.println("============================");
    	System.out.println("here is the NOE rmsd :"+strNoeRmsd );
    	
    	System.out.println("============================");
    	
    	Collections.sort(vecVdw, new vdw.VdwComparator() );
    	for (int i=0;i<50;i++)
    	{
    		vdw vdw_temp=(vdw)vecVdw.elementAt(i) ;
    		double modID=vdw_temp.getModelID();
    		System.out.println("the i-th "+i + ": vdw is: " +modID );
    	}
    	
    	
    	System.out.println("============================");
    	Vector vecVdwNoe=new Vector();    	
    	vecVdwNoe.addAll(vecVdw);    	
    	Collections.sort(vecVdwNoe, new vdw.VdwNoeComparator() );
    	for (int i=0;i<50;i++)
    	{
    		vdw vdw_temp=(vdw)vecVdwNoe.elementAt(i) ;
    		double modID=vdw_temp.getModelID();
    		System.out.println("the i-th "+i + ": noe is: " +modID );
    	}
    		
    }
        
    //merge two pdbs into a single PDB (enesemble) 
    /**
     * The main method.
     * 
     * @param argv the arguments
     */
    public static void main(String[] argv) 
	{	    
    	Pdb  pp = new Pdb();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/test/";
    	String srcOut=userDir+"/inputFiles/";
    	int counter=0;
    	Vector vecPdb=new Vector();
    	Vector vecVdw=new Vector();
    	
    	String strLoops=src+"2E71-One-New.pdb";
    	Vector vecLoops=new Vector();
    	vecLoops=pp.readPdb(strLoops);
    	
    	//change mly and mse to lys and met respectively
    	Vector vecTempNew=new Vector();
    	/*for(int t=0;t<vecLoops.size();t++)
    	{
    		Pdb pdbTemp=(Pdb) vecLoops.elementAt(t);
    		String res=pdbTemp.getResidue();
    		int resno=pdbTemp.getResidueNo();
    		
    		Vector vecAtomOrg=pdbTemp.getAtomVec();
    		if(res.equalsIgnoreCase("MLY"))
    		{
    			Vector vecAtomNew=new Vector();
    			for(int x=0;x<vecAtomOrg.size();x++)
    			{
    				Cartesian cc=(Cartesian) vecAtomOrg.elementAt(x);
    				String atomOrg=cc.getAtom();
    				if(atomOrg.equalsIgnoreCase("CH1") || atomOrg.equalsIgnoreCase("CH2"))
    					continue;
    				vecAtomNew.add(cc);   				
    			}//for(int x=0;x<vecAtomOrg.size();x++)
    			
    			vecTempNew.add(new Pdb(resno, "LYS", vecAtomNew));
    			continue;
    		}//if(res.equalsIgnoreCase("MLY"))
    		
    		if(res.equalsIgnoreCase("MSE"))
    		{
    			Vector vecAtomNew=new Vector();
    			for(int x=0;x<vecAtomOrg.size();x++)
    			{
    				Cartesian cc=(Cartesian) vecAtomOrg.elementAt(x);
    				String atomOrg=cc.getAtom();
    				if(atomOrg.equalsIgnoreCase("SE"))
    					cc.setAtom("SD");
    				vecAtomNew.add(cc);   				
    			}//for(int x=0;x<vecAtomOrg.size();x++)
    			
    			vecTempNew.add(new Pdb(resno, "MET", vecAtomNew));
    			continue;
    		}//if(res.equalsIgnoreCase("MLY"))
    		vecTempNew.add(pdbTemp);
    	}//for(int t=0;t<vecLoops.size();t++)
    	*/
    /*	for(int t=0;t<vecLoops.size();t++)
    	{
    		Pdb pdbTemp=(Pdb) vecLoops.elementAt(t);
    		String res=pdbTemp.getResidue();
    		int resno=pdbTemp.getResidueNo();
    		
    		pdbTemp.setResidueNo(resno-720);
    		vecTempNew.add(pdbTemp);
    		
    	}
    	Collections.sort(vecTempNew, new Pdb.PdbComparator());
    	
    	pp.print(vecTempNew);
    	
    	Vector pdbVecNewSec=pp.nameConvert2PDBNew(vecLoops);//need to double check
    	
    	
    	
    	
    	pp.print(pdbVecNewSec);
    	*/
    	///////////////////////////////////////
    	Vector vecPdbAll=new Vector ();
    	for (int j=1;j<=1;j++)
    	{    		
    		
	    	String strEnsemPdb=src+"average.pdb";//ff2_wps_"+j+".pdb";
	    	/*Vector vecEnsemble=pp.readPdb(strEnsemPdb);
	    	System.out.println("input: "+strEnsemPdb);
	    	Vector vecNewPdb=pp.nameConvert4SmallRotLibStr(vecEnsemble);
	    	vecPdbAll.add(vecNewPdb);*/
	    	Vector vecEnsemble=pp.readPdbFromEnsemble(strEnsemPdb);
	    	System.out.println("input: "+strEnsemPdb);
	    	for(int t=0;t<vecEnsemble.size();t++){
	    		Vector vecPdbTemp=(Vector)vecEnsemble.elementAt(t);
	    		Vector vecPdbTempNew=pp.nameConvertOrder(vecPdbTemp);
	    		Vector vecNewPdb=pp.nameConvert4SmallRotLibStr(vecPdbTempNew);
		    	vecPdbAll.add(vecNewPdb);
	    	}
	    	
    	}
	    	//Collections.sort(vecEnsemble, new Pdb.PdbComparator());
	    	
	    	
	    	String fileName =srcOut+ "ff2_all_templates"+".pdb";
	    	try
	    	{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		
	    		//out.println("REMARK :  ensemble ID=:" + counter);///to file
		        	for(int k=0;k< vecPdbAll.size();k++){
		        		out.println("REMARK model "+ (k+1));
		        		Vector vecOne=(Vector)vecPdbAll.elementAt(k);
		        		//pp.printToFileCyanaFormat(vecOne,fileName, out);
		        		pp.printToFile(vecOne,fileName, out);
		        		out.println("TER");
		        	}
		    	
				    
		    	//out.println("TER");
			  
		    	
			    out.println("END");  
		    	out.close();
		    	
			}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
				System.out.println("IOException: the stack trace is:");
				e.printStackTrace();
			}
			 			
			
	    	
    
    	
    }
    
    /**
     * Reading PDB file and other parameters in PDB Format from ensemble!!!.
     * 
     * @param pdbFile the pdb file
     * 
     * @return a vector of Pdb object ensemble
     */
    public Vector readPdbAndNoeRmsdFromEnsemble(String pdbFile)
    {	
	String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="",gap="";
	int Atoms_No, residueNo = 0, noTmp = 0;
	double x_in,y_in,z_in;
	Vector vecEnsemble=new Vector();///////
	Vector pdbVec = new Vector();
	Vector atoms = new Vector();
	String form="";
	
	
	double allRmsd=0.0;	
	double nhRmsd=0.0;	
	double chRmsd=0.0;
	int ensembleID=0;
	int modelID=0;
	double noeRmsd=0.0;
	StringTokenizer st = new StringTokenizer("");		    			
	
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
	    ss = in.readLine();	    
	    if (ss.length() > 66)
	        pdbID = ss.substring(62, 66).trim();
	    stop:
	    while(true) 
	    {
			if (ss.length() > 5)
			{
			    keywords = ss.substring(0, 6).trim();
			    gap=ss.substring(0,5).trim();
			    if (gap.equalsIgnoreCase("MODEL"))
	    		{
	    			 
	    			if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			pdbVec=new Vector();
	    	    		
	    		}
			}
			else 
			{
				keywords = ss.trim();
				gap=ss.trim();
			}
			
			while(keywords.equals("ATOM") )	
			{
			    if ( ss.length() > 54)
			    {
			    	form = ss.substring(16,17).trim(); //We choose only one of it
			    	if (form.equals("B"))
			    	{  //Some Pdb file like 1CBN has A and B 
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    				//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();	
			    			}
			    			continue stop;
			    		}
			    	}
			    	a1_type = ss.substring(12, 16).trim();
			    	r_type = ss.substring(17, 20).trim();
			    	residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    	x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    	y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    	z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    	atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    	if ((ss = in.readLine()) == null)
			    		break stop;		
			    	if (ss.length() > 54)
			    		str = (ss.substring(0,6)).trim();
			    	else continue stop;
			    	while (str.equals("ANISOU"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;	
			    		//  			    System.out.println(" Sick Line "+ss);
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    	}
			    	if (  !(str.equals("ATOM")) )
			    	{
			    		if(r_type.length() == 3)
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    			atoms = new  Vector();	
			    		}
			    		continue stop;
			    	}
			    	form = ss.substring(16,17).trim();
			    	if (form.equals("B"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    	}
			    	noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	while (noTmp == residueNo)
			    	{
			    		a1_type = ss.substring(12, 16).trim(); 
			    		r_type = ss.substring(17, 20).trim();
			    		residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    		x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    		y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    		z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    		atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else 
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
	    					atoms = new  Vector();	
			    			continue stop;
			    		}
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}

			    		if (!(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    		form = ss.substring(16,17).trim();
			    		if (form.equals("B"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;							
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    			//  				System.out.println(" Sick Line "+ss);
			    			while (str.equals("ANISOU"))
			    			{
			    				if ((ss = in.readLine()) == null)
			    					break stop;		
			    				//  				    System.out.println(" Sick Line "+ss);
			    				if (ss.length() > 54)
			    					str = (ss.substring(0,6)).trim();
			    				else continue stop;
			    			}
			    			if (  !(str.equals("ATOM")) )
			    			{
			    				if(r_type.length() == 3)
			    				{
			    					pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    					atoms = new  Vector();		
			    				}
			    				continue stop;
			    			}
			    		}
			    		noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	}
			    	if(r_type.length() == 3)
			    	{
			    		pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    		atoms = new  Vector();
			    	}
			    	keywords = str;		
			    	
			    	///added by zeng, for reading ensemble
			    	if (ss.length()>=5)
			    	{
			    		gap=ss.substring(0,5).trim();
			    	
			    		if (gap.equalsIgnoreCase("MODEL"))
			    		{
			    			 
			    			if(pdbVec.size()>0)
				    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
			    			pdbVec=new Vector();
			    		
			    		}
			    	}//if (ss.length()>=5)
			    	
			    	
			    }
			}
			if ((ss = in.readLine()) == null)
			{
				
				vecEnsemble.add(new vdw(pdbVec,0.0,noeRmsd) );
				break stop;
			}
			///added by zeng, for reading ensemble
			if (ss.length()>=5)
	    	{
		    	gap=ss.substring(0,5).trim();
		    	if (gap.equalsIgnoreCase("MODEL"))
		    	{
		    		if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,0.0,noeRmsd) );
		    			//vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			
		    		pdbVec=new Vector();
		    		
		    		//	read parameters from remark
		    		/*String strTemp1= in.readLine();//1st remark
	    			int stest=strTemp1.length();
	    			int dotindex=strTemp1.indexOf(".");
	    			ensembleID=Integer.parseInt(strTemp1.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			String strTemp2= in.readLine();//2nd remark
	    			dotindex=strTemp2.indexOf(".");
	    			modelID=Integer.parseInt(strTemp2.substring(28,dotindex));
	    			String strTemp3= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(strTemp3.substring(40,strTemp3.length()).trim());
	    			String strTemp4= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(strTemp4.substring(23,strTemp4.length()).trim());
	    			String strTemp5= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(strTemp5.substring(25,strTemp5.length()).trim());
	    				*/
	    			/////ss= in.readLine();//1st remark	    			
	    			/*int dotindex=ss.indexOf(".");
	    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			ss= in.readLine();//2nd remark
	    			dotindex=ss.indexOf(".");
	    			modelID=Integer.parseInt(ss.substring(28,dotindex));
	    			ss= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
	    			ss= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
	    				*/
	    			/////noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );
		    	}
	    	}
	    }
		}catch (FileNotFoundException e) 
		{
			System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) 
     	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
     	}
     	Collections.sort(pdbVec,new PdbComparator() );
     	//return pdbVec;
     	return vecEnsemble;
    }

    /**
     * Reading PDB file and other parameters in PDB Format from ensemble!!!.
     * 
     * @param pdbFile the pdb file
     * 
     * @return a vector of Pdb object ensemble
     */
    public Vector readPdbAndNoeHarmFromEnsemble(String pdbFile)
    {	
	String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="",gap="";
	int Atoms_No, residueNo = 0, noTmp = 0;
	double x_in,y_in,z_in;
	Vector vecEnsemble=new Vector();///////
	Vector pdbVec = new Vector();
	Vector atoms = new Vector();
	String form="";
	
	
	double allRmsd=0.0;	
	double nhRmsd=0.0;	
	double chRmsd=0.0;
	int ensembleID=0;
	int modelID=0;
	double noeRmsd=0.0;
	StringTokenizer st = new StringTokenizer("");		    			
	
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
	    ss = in.readLine();
	    //	    System.out.println("ss :"+ss);
	    if(ss==null)
	    	return null;
	    if (ss.length() > 66)
	    	pdbID = ss.substring(62, 66).trim();
	    stop:
	    while(true) 
	    {
			if (ss.length() > 5)
			{
			    keywords = ss.substring(0, 6).trim();
			    gap=ss.substring(0,5).trim();
			    if (gap.equalsIgnoreCase("MODEL"))
	    		{
	    			 
	    			if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			pdbVec=new Vector();    	    	
	    				    		
	    		}
			}
			else 
			{
				keywords = ss.trim();
				gap=ss.trim();
			}
			
			while(keywords.equals("ATOM") )	
			{
			    if ( ss.length() > 54)
			    {
			    	form = ss.substring(16,17).trim(); //We choose only one of it
			    	if (form.equals("B"))
			    	{  //Some Pdb file like 1CBN has A and B 
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    				//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();	
			    			}
			    			continue stop;
			    		}
			    	}
			    	a1_type = ss.substring(12, 16).trim();
			    	r_type = ss.substring(17, 20).trim();
			    	residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    	x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    	y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    	z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    	atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    	if ((ss = in.readLine()) == null)
			    		break stop;		
			    	if (ss.length() > 54)
			    		str = (ss.substring(0,6)).trim();
			    	else continue stop;
			    	while (str.equals("ANISOU"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;	
			    		//  			    System.out.println(" Sick Line "+ss);
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    	}
			    	if (  !(str.equals("ATOM")) )
			    	{
			    		if(r_type.length() == 3)
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    			atoms = new  Vector();	
			    		}
			    		continue stop;
			    	}
			    	form = ss.substring(16,17).trim();
			    	if (form.equals("B"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    	}
			    	noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	while (noTmp == residueNo)
			    	{
			    		a1_type = ss.substring(12, 16).trim(); 
			    		r_type = ss.substring(17, 20).trim();
			    		residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    		x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    		y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    		z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    		atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else 
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
	    					atoms = new  Vector();	
			    			continue stop;
			    		}
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}

			    		if (!(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    		form = ss.substring(16,17).trim();
			    		if (form.equals("B"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;							
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    			//  				System.out.println(" Sick Line "+ss);
			    			while (str.equals("ANISOU"))
			    			{
			    				if ((ss = in.readLine()) == null)
			    					break stop;		
			    				//  				    System.out.println(" Sick Line "+ss);
			    				if (ss.length() > 54)
			    					str = (ss.substring(0,6)).trim();
			    				else continue stop;
			    			}
			    			if (  !(str.equals("ATOM")) )
			    			{
			    				if(r_type.length() == 3)
			    				{
			    					pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    					atoms = new  Vector();		
			    				}
			    				continue stop;
			    			}
			    		}
			    		noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	}
			    	if(r_type.length() == 3)
			    	{
			    		pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    		atoms = new  Vector();
			    	}
			    	keywords = str;		
			    	
			    	///added by zeng, for reading ensemble
			    	if (ss.length()>=5)
			    	{
			    		gap=ss.substring(0,5).trim();
			    	
			    		if (gap.equalsIgnoreCase("MODEL"))
			    		{
			    			 
			    			if(pdbVec.size()>0)
				    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
			    			pdbVec=new Vector();
			    			
			    			////ss= in.readLine();//1st remark	    			
			    			/*int dotindex=ss.indexOf(".");
			    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
			    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
			    			ss= in.readLine();//2nd remark
			    			dotindex=ss.indexOf(".");
			    			modelID=Integer.parseInt(ss.substring(28,dotindex));
			    			ss= in.readLine();//3rd remark
			    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
			    			ss= in.readLine();//4th remark
			    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
			    			ss= in.readLine();//5th remark
			    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
			    				*/
			    			/////noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );
			    			////ss= in.readLine();//2nd remark	
			    			
			    		}
			    	}//if (ss.length()>=5)
			    	
			    	
			    }
			}
			if ((ss = in.readLine()) == null)
			{
				//pdbVec.add(new Pdb(residueNo, r_type, atoms));
				//vecEnsemble.add(pdbVec);
				//vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
				vecEnsemble.add(new vdw(pdbVec,0.0,noeRmsd) );
				break stop;
			}
			///added by zeng, for reading ensemble
			if (ss.length()>=5)
	    	{
		    	gap=ss.substring(0,5).trim();
		    	if (gap.equalsIgnoreCase("MODEL"))
		    	{
		    		if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,0.0,noeRmsd) );
		    			//vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			
		    		pdbVec=new Vector();
		    		
		    		//	read parameters from remark
		    		/*String strTemp1= in.readLine();//1st remark
	    			int stest=strTemp1.length();
	    			int dotindex=strTemp1.indexOf(".");
	    			ensembleID=Integer.parseInt(strTemp1.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			String strTemp2= in.readLine();//2nd remark
	    			dotindex=strTemp2.indexOf(".");
	    			modelID=Integer.parseInt(strTemp2.substring(28,dotindex));
	    			String strTemp3= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(strTemp3.substring(40,strTemp3.length()).trim());
	    			String strTemp4= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(strTemp4.substring(23,strTemp4.length()).trim());
	    			String strTemp5= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(strTemp5.substring(25,strTemp5.length()).trim());
	    				*/
	    			////ss= in.readLine();//1st remark	    			
	    			/*int dotindex=ss.indexOf(".");
	    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			ss= in.readLine();//2nd remark
	    			dotindex=ss.indexOf(".");
	    			modelID=Integer.parseInt(ss.substring(28,dotindex));
	    			ss= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
	    			ss= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
	    				*/
	    			////noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );
	    			////ss= in.readLine();//2nd remark	
	    			
		    	}
	    	}
	    }
	    in.close();
	    
	    
	    
	    
		}catch (FileNotFoundException e) 
		{
			System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) 
     	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
     	}
     	Collections.sort(pdbVec,new PdbComparator() );
     	//return pdbVec;
     	return vecEnsemble;
    }
    
    //read vdw and noe scores from xplor Pdb
    /**
     * Read pdb and para xplor.
     * 
     * @param pdbFile the pdb file
     * @param vdwSc the vdw sc
     * @param noeSc the noe sc
     * 
     * @return the vector
     */
    public Vector readPdbAndParaXplor(String pdbFile, double[] vdwSc, double [] noeSc)
    {	
	String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="",gap="";
	int Atoms_No, residueNo = 0, noTmp = 0;
	double x_in,y_in,z_in;
	Vector vecEnsemble=new Vector();///////
	Vector pdbVec = new Vector();
	Vector atoms = new Vector();
	String form="";
	
	
	double allRmsd=0.0;	
	double nhRmsd=0.0;	
	double chRmsd=0.0;
	int ensembleID=0;
	int modelID=0;
	double noeRmsd=0.0;
	StringTokenizer st = new StringTokenizer("");		    			
	
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
	    ss = in.readLine();
	  
	    if (ss.length() > 66)
	        pdbID = ss.substring(62, 66).trim();
	    stop:
	    while(true) 
	    {
			if (ss.length() > 5)
			{
			    keywords = ss.substring(0, 6).trim();
			    gap=ss.substring(0,6).trim();
			    if (gap.equalsIgnoreCase("REMARK"))
	    		{
	    			 
	    			if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			pdbVec=new Vector();
	    			
	    			//read parameters from remark
	    			/*st = new StringTokenizer(ss);
	    			if (st.hasMoreTokens())
	    				st.nextToken();
	    			if (st.hasMoreTokens())
	    				st.nextToken();*/
	    			
	    			ss= in.readLine();//2nd remark	    	
	    			ss= in.readLine();//3rd remark	    	
	    			ss= in.readLine();//4th remark
	    			ss= in.readLine();//5th remark
	    			vdwSc[0]=Double.parseDouble(ss.substring(19,ss.length()).trim());
	    			
	    			ss= in.readLine();//6th remark
	    			
	    			ss= in.readLine();//7th remark
	    			
	    			noeSc[0]=Double.parseDouble(ss.substring(24,ss.length()).trim());
	    			return null;
	    			
	    			/*int dotindex=ss.indexOf(".");
	    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			ss= in.readLine();//2nd remark
	    			dotindex=ss.indexOf(".");
	    			modelID=Integer.parseInt(ss.substring(28,dotindex));
	    			ss= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
	    			ss= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
	    			*/
	    				    		
	    		}
			}
			else 
			{
				keywords = ss.trim();
				gap=ss.trim();
			}
			
			while(keywords.equals("ATOM") )	
			{
			    if ( ss.length() > 54)
			    {
			    	form = ss.substring(16,17).trim(); //We choose only one of it
			    	if (form.equals("B"))
			    	{  //Some Pdb file like 1CBN has A and B 
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    				//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();	
			    			}
			    			continue stop;
			    		}
			    	}
			    	a1_type = ss.substring(12, 16).trim();
			    	r_type = ss.substring(17, 20).trim();
			    	residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    	x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    	y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    	z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    	atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    	if ((ss = in.readLine()) == null)
			    		break stop;		
			    	if (ss.length() > 54)
			    		str = (ss.substring(0,6)).trim();
			    	else continue stop;
			    	while (str.equals("ANISOU"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;	
			    		
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    	}
			    	if (  !(str.equals("ATOM")) )
			    	{
			    		if(r_type.length() == 3)
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    			atoms = new  Vector();	
			    		}
			    		continue stop;
			    	}
			    	form = ss.substring(16,17).trim();
			    	if (form.equals("B"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    	}
			    	noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	while (noTmp == residueNo)
			    	{
			    		a1_type = ss.substring(12, 16).trim(); 
			    		r_type = ss.substring(17, 20).trim();
			    		residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    		x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    		y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    		z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    		atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else 
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
	    					atoms = new  Vector();	
			    			continue stop;
			    		}
			    		
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}

			    		if (!(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    		form = ss.substring(16,17).trim();
			    		if (form.equals("B"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;							
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    			
			    			while (str.equals("ANISOU"))
			    			{
			    				if ((ss = in.readLine()) == null)
			    					break stop;		
			    				
			    				if (ss.length() > 54)
			    					str = (ss.substring(0,6)).trim();
			    				else continue stop;
			    			}
			    			if (  !(str.equals("ATOM")) )
			    			{
			    				if(r_type.length() == 3)
			    				{
			    					pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    					atoms = new  Vector();		
			    				}
			    				continue stop;
			    			}
			    		}
			    		noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	}
			    	if(r_type.length() == 3)
			    	{
			    		pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    		atoms = new  Vector();
			    	}
			    	keywords = str;		
			    	
			    	///added by zeng, for reading ensemble
			    	if (ss.length()>=5)
			    	{
			    		gap=ss.substring(0,5).trim();
			    	
			    		if (gap.equalsIgnoreCase("MODEL"))
			    		{
			    			 
			    			if(pdbVec.size()>0)
				    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
			    			pdbVec=new Vector();
			    			
			    			ss= in.readLine();//1st remark	    			
			    			/*int dotindex=ss.indexOf(".");
			    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
			    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
			    			ss= in.readLine();//2nd remark
			    			dotindex=ss.indexOf(".");
			    			modelID=Integer.parseInt(ss.substring(28,dotindex));
			    			ss= in.readLine();//3rd remark
			    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
			    			ss= in.readLine();//4th remark
			    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
			    			ss= in.readLine();//5th remark
			    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
			    				*/
			    			noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );
			    		}
			    	}//if (ss.length()>=5)
			    	
			    	
			    }
			}
			if ((ss = in.readLine()) == null)
			{				
				vecEnsemble.add(new vdw(pdbVec,0.0,noeRmsd) );
				break stop;
			}
			///added by zeng, for reading ensemble
			if (ss.length()>=5)
	    	{
		    	gap=ss.substring(0,5).trim();
		    	if (gap.equalsIgnoreCase("MODEL"))
		    	{
		    		if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,0.0,noeRmsd) );
		    			//vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			
		    		pdbVec=new Vector();
		    		
		    		//read parameters from remark
		    		/*String strTemp1= in.readLine();//1st remark
	    			int stest=strTemp1.length();
	    			int dotindex=strTemp1.indexOf(".");
	    			ensembleID=Integer.parseInt(strTemp1.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			String strTemp2= in.readLine();//2nd remark
	    			dotindex=strTemp2.indexOf(".");
	    			modelID=Integer.parseInt(strTemp2.substring(28,dotindex));
	    			String strTemp3= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(strTemp3.substring(40,strTemp3.length()).trim());
	    			String strTemp4= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(strTemp4.substring(23,strTemp4.length()).trim());
	    			String strTemp5= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(strTemp5.substring(25,strTemp5.length()).trim());
	    				*/
	    			ss= in.readLine();//1st remark	    			
	    			/*int dotindex=ss.indexOf(".");
	    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			ss= in.readLine();//2nd remark
	    			dotindex=ss.indexOf(".");
	    			modelID=Integer.parseInt(ss.substring(28,dotindex));
	    			ss= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
	    			ss= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
	    				*/
	    			noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );
		    	}
	    	}
	    }
		}catch (FileNotFoundException e) 
		{
			System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) 
     	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
     	}
     	Collections.sort(pdbVec,new PdbComparator() );
     	//return pdbVec;
     	return vecEnsemble;
    }
    //read the energy from xplor Pdb, 
    /**
     * Read pdb energy.
     * 
     * @param pdbFile the pdb file
     * @param scEnerg the sc energ
     * @param rowPos the row pos
     * @param colPos the col pos
     */
    public void readPdbEnergy(String pdbFile, double[] scEnerg, int rowPos, int colPos)
    {	
		String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="",gap="";
		int Atoms_No, residueNo = 0, noTmp = 0;
		double x_in,y_in,z_in;
		double allRmsd=0.0;	
		double nhRmsd=0.0;	
		double chRmsd=0.0;
		int ensembleID=0;
		int modelID=0;
		double noeRmsd=0.0;
		StringTokenizer st = new StringTokenizer("");		    			
	
		try
		{
		    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
		    ss = in.readLine();
		    //	    System.out.println("ss :"+ss);
		    if (ss.length() > 66)
		        pdbID = ss.substring(62, 66).trim();
		    stop:
		    while(true) 
		    {
				if (ss.length() > 5)
				{
				    keywords = ss.substring(0, 6).trim();
				    gap=ss.substring(0,6).trim();
				    if (gap.equalsIgnoreCase("REMARK"))
		    		{
		    			if(rowPos==0)
		    			{
		    				scEnerg[0]=Double.parseDouble(ss.substring(colPos,ss.length()).trim());		    			
			    			return;
		    			} 			
		    		
		    			for (int t=0;t<rowPos;t++)
		    				ss= in.readLine();//2nd remark	
		    			
		    			//int idLimit=ss.indexOf(',');
		    			int idLimit=0;
		    			if((ss.indexOf('.') >ss.indexOf(',')) && (ss.indexOf(',')!=-1) )
		    				idLimit=ss.indexOf(',');
		    			else 
		    				idLimit=ss.indexOf('.')+2;
		    			//scEnerg[0]=Double.parseDouble(ss.substring(colPos,ss.length()).trim());
		    			String strTemp=ss.substring(colPos,idLimit).trim();
		    			scEnerg[0]=Double.parseDouble(strTemp);	
		    			return;    			
		    		}//if (gap.equalsIgnoreCase("REMARK"))
				}//if (ss.length() > 5)    }
			}
		}
	    catch (FileNotFoundException e) 
		{
			System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) 
     	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
     	}    	
     	return;
    }


    
    
    /**
     * Reading PDB file and other parameters in PDB Format from ensemble!!!.
     * 
     * @param pdbFile the pdb file
     * 
     * @return a vector of Pdb object ensemble
     */
    public Vector readPdbAndParametersFromEnsemble(String pdbFile)
    {	
	String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="",gap="";
	int Atoms_No, residueNo = 0, noTmp = 0;
	double x_in,y_in,z_in;
	Vector vecEnsemble=new Vector();///////
	Vector pdbVec = new Vector();
	Vector atoms = new Vector();
	String form="";
	
	
	double allRmsd=0.0;	
	double nhRmsd=0.0;	
	double chRmsd=0.0;
	int ensembleID=0;
	int modelID=0;
	double noeRmsd=0.0;
	StringTokenizer st = new StringTokenizer("");		    			
	
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
	    ss = in.readLine();
	    //	    System.out.println("ss :"+ss);
	    if (ss.length() > 66)
	        pdbID = ss.substring(62, 66).trim();
	    stop:
	    while(true) 
	    {
			if (ss.length() > 5)
			{
			    keywords = ss.substring(0, 6).trim();
			    gap=ss.substring(0,5).trim();
			    if (gap.equalsIgnoreCase("MODEL"))
	    		{
	    			 
	    			if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			pdbVec=new Vector();
	    			
	    			//read parameters from remark
	    			/*st = new StringTokenizer(ss);
	    			if (st.hasMoreTokens())
	    				st.nextToken();
	    			if (st.hasMoreTokens())
	    				st.nextToken();*/
	    			
	    			ss= in.readLine();//1st remark	    			
	    			/*int dotindex=ss.indexOf(".");
	    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			ss= in.readLine();//2nd remark
	    			dotindex=ss.indexOf(".");
	    			modelID=Integer.parseInt(ss.substring(28,dotindex));
	    			ss= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
	    			ss= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
	    			*/
	    			allRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//2nd remark
	    			noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );		    		
	    		}
			}
			else 
			{
				keywords = ss.trim();
				gap=ss.trim();
			}
			
			while(keywords.equals("ATOM") )	
			{
			    if ( ss.length() > 54)
			    {
			    	form = ss.substring(16,17).trim(); //We choose only one of it
			    	if (form.equals("B"))
			    	{  //Some Pdb file like 1CBN has A and B 
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    				//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();	
			    			}
			    			continue stop;
			    		}
			    	}
			    	a1_type = ss.substring(12, 16).trim();
			    	r_type = ss.substring(17, 20).trim();
			    	residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    	x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    	y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    	z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    	atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    	if ((ss = in.readLine()) == null)
			    		break stop;		
			    	if (ss.length() > 54)
			    		str = (ss.substring(0,6)).trim();
			    	else continue stop;
			    	while (str.equals("ANISOU"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;	
			    		//  			    System.out.println(" Sick Line "+ss);
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    	}
			    	if (  !(str.equals("ATOM")) )
			    	{
			    		if(r_type.length() == 3)
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    			atoms = new  Vector();	
			    		}
			    		continue stop;
			    	}
			    	form = ss.substring(16,17).trim();
			    	if (form.equals("B"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    	}
			    	noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	while (noTmp == residueNo)
			    	{
			    		a1_type = ss.substring(12, 16).trim(); 
			    		r_type = ss.substring(17, 20).trim();
			    		residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    		x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    		y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    		z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    		atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else 
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
	    					atoms = new  Vector();	
			    			continue stop;
			    		}
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}

			    		if (!(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    		form = ss.substring(16,17).trim();
			    		if (form.equals("B"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;							
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    			//  				System.out.println(" Sick Line "+ss);
			    			while (str.equals("ANISOU"))
			    			{
			    				if ((ss = in.readLine()) == null)
			    					break stop;		
			    				//  				    System.out.println(" Sick Line "+ss);
			    				if (ss.length() > 54)
			    					str = (ss.substring(0,6)).trim();
			    				else continue stop;
			    			}
			    			if (  !(str.equals("ATOM")) )
			    			{
			    				if(r_type.length() == 3)
			    				{
			    					pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    					atoms = new  Vector();		
			    				}
			    				continue stop;
			    			}
			    		}
			    		noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	}
			    	if(r_type.length() == 3)
			    	{
			    		pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    		atoms = new  Vector();
			    	}
			    	keywords = str;		
			    	
			    	///added by zeng, for reading ensemble
			    	if (ss.length()>=5)
			    	{
			    		gap=ss.substring(0,5).trim();
			    	
			    		if (gap.equalsIgnoreCase("MODEL"))
			    		{
			    			 
			    			if(pdbVec.size()>0)
				    			vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
			    			pdbVec=new Vector();
			    			
			    			ss= in.readLine();//1st remark	    			
			    			/*int dotindex=ss.indexOf(".");
			    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
			    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
			    			ss= in.readLine();//2nd remark
			    			dotindex=ss.indexOf(".");
			    			modelID=Integer.parseInt(ss.substring(28,dotindex));
			    			ss= in.readLine();//3rd remark
			    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
			    			ss= in.readLine();//4th remark
			    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
			    			ss= in.readLine();//5th remark
			    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
			    				*/
			    			allRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
			    			ss= in.readLine();//2nd remark
			    			noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );
			    		}
			    	}//if (ss.length()>=5)
			    	
			    	
			    }
			}
			if ((ss = in.readLine()) == null)
			{
				//pdbVec.add(new Pdb(residueNo, r_type, atoms));
				//vecEnsemble.add(pdbVec);
				//vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
				vecEnsemble.add(new vdw(pdbVec,allRmsd,noeRmsd) );
				break stop;
			}
			///added by zeng, for reading ensemble
			if (ss.length()>=5)
	    	{
		    	gap=ss.substring(0,5).trim();
		    	if (gap.equalsIgnoreCase("MODEL"))
		    	{
		    		if(pdbVec.size()>0)
		    			vecEnsemble.add(new vdw(pdbVec,allRmsd,noeRmsd) );
		    			//vecEnsemble.add(new vdw(pdbVec,allRmsd,nhRmsd,chRmsd,ensembleID,modelID) );
	    			
		    		pdbVec=new Vector();
		    		
		    		//	read parameters from remark
		    		/*String strTemp1= in.readLine();//1st remark
	    			int stest=strTemp1.length();
	    			int dotindex=strTemp1.indexOf(".");
	    			ensembleID=Integer.parseInt(strTemp1.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			String strTemp2= in.readLine();//2nd remark
	    			dotindex=strTemp2.indexOf(".");
	    			modelID=Integer.parseInt(strTemp2.substring(28,dotindex));
	    			String strTemp3= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(strTemp3.substring(40,strTemp3.length()).trim());
	    			String strTemp4= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(strTemp4.substring(23,strTemp4.length()).trim());
	    			String strTemp5= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(strTemp5.substring(25,strTemp5.length()).trim());
	    				*/
	    			ss= in.readLine();//1st remark	    			
	    			/*int dotindex=ss.indexOf(".");
	    			ensembleID=Integer.parseInt(ss.substring(31,dotindex));			    			
	    			//ensembleID=Integer.parseInt(strTemp1.substring(31,strTemp1.length()));			    			
	    			ss= in.readLine();//2nd remark
	    			dotindex=ss.indexOf(".");
	    			modelID=Integer.parseInt(ss.substring(28,dotindex));
	    			ss= in.readLine();//3rd remark
	    			allRmsd=Double.parseDouble(ss.substring(40,ss.length()).trim());
	    			ss= in.readLine();//4th remark
	    			nhRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//5th remark
	    			chRmsd=Double.parseDouble(ss.substring(25,ss.length()).trim());
	    				*/
	    			allRmsd=Double.parseDouble(ss.substring(23,ss.length()).trim());
	    			ss= in.readLine();//2nd remark
	    			noeRmsd=Double.parseDouble(ss.substring(21,ss.length()).trim() );
		    	}
	    	}
	    }
		}catch (FileNotFoundException e) 
		{
			System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) 
     	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
     	}
     	Collections.sort(pdbVec,new PdbComparator() );
     	//return pdbVec;
     	return vecEnsemble;
    }

    /**
     * Reading PDB file in PDB Format from ensemble!!!
     * the magic number for substring is based on format for pdb file.
     * 
     * @param pdbFile the pdb file
     * 
     * @return a vector of Pdb object ensemble
     * Comment: There is always some irregularity in PDB files so when reading
     * a real PDB file you must be cautious and ready to check the output from the reader
     * However, in our case we simply need coordinates which is much easier.
     */
    public Vector readPdbFromEnsemble(String pdbFile)
    {	
	String ss="", str="", a1_type, a2_type, r_type="", keywords, pdbID="",gap="";
	int Atoms_No, residueNo = 0, noTmp = 0;
	double x_in,y_in,z_in;
	Vector vecEnsemble=new Vector();
	Vector pdbVec = new Vector();
	Vector atoms = new Vector();
	String form="";
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));			
	    ss = in.readLine();
	    //	    System.out.println("ss :"+ss);
	    if (ss.length() > 66)
	        pdbID = ss.substring(62, 66).trim();
	    stop:
	    while(true) 
	    {
			if (ss.length() > 5)
			{
			    keywords = ss.substring(0, 6).trim();
			    gap=ss.substring(0,5).trim();
			}
			else 
			{
				keywords = ss.trim();
				gap=ss.trim();
			}
			
			while(keywords.equals("ATOM") )	
			{
			    if ( ss.length() > 54)
			    {
			    	form = ss.substring(16,17).trim(); //We choose only one of it
			    	if (form.equals("B"))
			    	{  //Some Pdb file like 1CBN has A and B 
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		//  			    System.out.println(" Sick Line "+ss);
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    				//  				System.out.println(" Sick Line "+ss);
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();	
			    			}
			    			continue stop;
			    		}
			    	}
			    	a1_type = ss.substring(12, 16).trim();
			    	r_type = ss.substring(17, 20).trim();
			    	residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    	x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    	y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    	z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    	atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    	if ((ss = in.readLine()) == null)
			    		break stop;		
			    	if (ss.length() > 54)
			    		str = (ss.substring(0,6)).trim();
			    	else continue stop;
			    	while (str.equals("ANISOU"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;	
			    		//  			    System.out.println(" Sick Line "+ss);
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    	}
			    	if (  !(str.equals("ATOM")) )
			    	{
			    		if(r_type.length() == 3)
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    			atoms = new  Vector();	
			    		}
			    		continue stop;
			    	}
			    	form = ss.substring(16,17).trim();
			    	if (form.equals("B"))
			    	{
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else continue stop;
			    		
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}
			    		if (  !(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    	}
			    	noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	while (noTmp == residueNo)
			    	{
			    		a1_type = ss.substring(12, 16).trim(); 
			    		r_type = ss.substring(17, 20).trim();
			    		residueNo = Integer.parseInt(ss.substring(23, 26).trim());
			    		x_in = new Double(ss.substring(30, 38).trim()).doubleValue();
			    		y_in = new Double(ss.substring(38, 46).trim()).doubleValue();
			    		z_in = new Double(ss.substring(46, 54).trim()).doubleValue();
			    		atoms.add(new Cartesian(a1_type, x_in, y_in, z_in));
			    		if ((ss = in.readLine()) == null)
			    			break stop;							
			    		if (ss.length() > 54)
			    			str = (ss.substring(0,6)).trim();
			    		else 
			    		{
			    			pdbVec.add(new Pdb(residueNo, r_type, atoms));
	    					atoms = new  Vector();	
			    			continue stop;
			    		}
			    		
			    		while (str.equals("ANISOU"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;		
			    			
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    		}

			    		if (!(str.equals("ATOM")) )
			    		{
			    			if(r_type.length() == 3)
			    			{
			    				pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    				atoms = new  Vector();		
			    			}
			    			continue stop;
			    		}
			    		form = ss.substring(16,17).trim();
			    		if (form.equals("B"))
			    		{
			    			if ((ss = in.readLine()) == null)
			    				break stop;							
			    			if (ss.length() > 54)
			    				str = (ss.substring(0,6)).trim();
			    			else continue stop;
			    			
			    			while (str.equals("ANISOU"))
			    			{
			    				if ((ss = in.readLine()) == null)
			    					break stop;		
			    				
			    				if (ss.length() > 54)
			    					str = (ss.substring(0,6)).trim();
			    				else continue stop;
			    			}
			    			if (  !(str.equals("ATOM")) )
			    			{
			    				if(r_type.length() == 3)
			    				{
			    					pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    					atoms = new  Vector();		
			    				}
			    				continue stop;
			    			}
			    		}
			    		noTmp = Integer.parseInt(ss.substring(23, 26).trim());
			    	}
			    	if(r_type.length() == 3)
			    	{
			    		pdbVec.add(new Pdb(residueNo, r_type, atoms));
			    		atoms = new  Vector();
			    	}
			    	keywords = str;		
			    	
			    	///added by zeng, for reading ensemble
			    	if (ss.length()>=5)
			    	{
			    		gap=ss.substring(0,5).trim();
			    	
			    		if (gap.equalsIgnoreCase("MODEL"))
			    		{
			    			if(pdbVec.size()>0)
				    			vecEnsemble.add(pdbVec);
			    			pdbVec=new Vector();
			    		}
			    	}//if (ss.length()>=5)
			    	
			    	
			    }
			}
			if ((ss = in.readLine()) == null)
			{
				//pdbVec.add(new Pdb(residueNo, r_type, atoms));
				vecEnsemble.add(pdbVec);
				break stop;
			}
			///added by zeng, for reading ensemble
			if (ss.length()>=5)
	    	{
		    	gap=ss.substring(0,5).trim();
		    	if (gap.equalsIgnoreCase("MODEL"))
		    	{
		    		if(pdbVec.size()>0)
		    			vecEnsemble.add(pdbVec);
		    		pdbVec=new Vector();
		    	}
	    	}
	    }
		}catch (FileNotFoundException e) 
		{
			System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) 
     	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
     	}
     	Collections.sort(pdbVec,new PdbComparator() );
     	//return pdbVec;
     	return vecEnsemble;
    }


    /**
     * compute new PDB coordinates after the rotation by matrix R.
     * 
     * @param pdbVec original PDB coordinates
     * @param R      the rotation matrix
     * 
     * @return       a new vector of PDB object
     */
    public Vector<Pdb> newPdb (final Vector<Pdb> pdbVec, final Matrix R){
	Cartesian atom = null;
	String atomName="", resid;
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    for (int j=0; j<atoms.size(); j++){
		atom = (Cartesian) atoms.elementAt(j);
		atomName = atom.getAtom();
		coord[0] = atom.getX();
		coord[1] = atom.getY();
		coord[2] = atom.getZ();
		coordN = R.times(coord);
		atomVec.add(new Cartesian(atomName, coordN[0], coordN[1], coordN[2]));
	    }
	    pdbVecN.add(new Pdb(no, resid, atomVec));
	    atomVec = new Vector();
	}
	return pdbVecN;
    }

    /**
     * compute new PDB coordinates after the translation.
     * 
     * @param pdbVec  original PDB coordinates
     * @param vT      the center in which the PDB is being tranlated to
     * 
     * @return        a new vector of PDB object
     */
    public Vector<Pdb> newPdbByTranslation (final Vector<Pdb> pdbVec, double[] vT){
	Cartesian atom = null;
	String atomName ="", resid = "";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    for (int j=0; j<atoms.size(); j++){
		atom = (Cartesian) atoms.elementAt(j);
		atomName = atom.getAtom();
		coord[0] = atom.getX();
		coord[1] = atom.getY();
		coord[2] = atom.getZ();
		coordN =  new double[] {coord[0] + vT[0], coord[1] + vT[1], coord[2] + vT[2]}; //new coordinates
		atomVec.add(new Cartesian(atomName, coordN[0], coordN[1], coordN[2]));
	    }
	    pdbVecN.add(new Pdb(no, resid, atomVec));
	    atomVec = new Vector();
	}
	return pdbVecN;
    }

    /**
     * Method for computing dihedral angles between two planes defined
     * by three vectors. The middle plane is shared by two planes: the
     * intersection of the two planes.  The first two vectors (n2CO1,
     * n2CA) define the first plane. The second two vectors (n2CA,
     * ca2CO) define the second plane. The names of vectors are for
     * computing backbone (\phi, \psi) angles but the program should
     * be able to compute any dihedral angle. Please note that if the
     * first vector is from CO(i-1) ->N(i) the computed angle will
     * differ from that computed from the direction N(i)->CO(i-1) by
     * 180 degree. In other words, the method is defined  with
     * the 1st and 3rd vectors point away from the shared vector (the
     * 2nd vector).
     * 
     * @param n2CO1 the N(i)->CO(i-1) vector
     * @param n2CA the N(i)->CA(i) vector
     * @param ca2CO the CA(i)->CO(i) vector
     * 
     * @return the dihedral angle
     */  
    public double dihedralAngle (double[] n2CO1, double[] n2CA, double[] ca2CO){
	double [] nToCO1 = dirCos(n2CO1);
	double [] nToCA = dirCos(n2CA);
	double [] caToCO = dirCos(ca2CO);
	// Cal the rotate matrix: First cal the polar angle phi and theta of the 
	// intermediate vector(N->CA) from it's directional cosines
	double theta_nToCa = Math.acos(nToCA[2]); //acos [0,PI) so should be OK
	double phi_nToCa = 0.0;
	double x = nToCA[0];
	double y = nToCA[1];
	if (theta_nToCa != 0.0){
	    if (x > 0) 
		phi_nToCa = Math.atan(y/x);
	    else if (x < 0)
		phi_nToCa = Math.atan(y/x) + Math.PI;
	    else {
		if (y > 0)
		    phi_nToCa = Math.PI / 2.0;
		else phi_nToCa = 3*Math.PI / 2.0;
	    }
	}
	else  phi_nToCa = 0.0;
	final int m = 3; //The dimensions of the Rotation Matrix
	final int n = 3;
	//construct the rotation matrix A needed to bring N-CA vector onto the +y-axis
	double [][] A = new double[3][3];
	A[0][0] = Math.sin(phi_nToCa);
	A[0][1] = -Math.cos(phi_nToCa);
	A[0][2] = 0.0;
	A[1][0] = Math.sin(theta_nToCa) * Math.cos(phi_nToCa);
	A[1][1] = Math.sin(theta_nToCa) * Math.sin(phi_nToCa);
	A[1][2] = Math.cos(theta_nToCa);
	A[2][0] = -Math.cos(theta_nToCa) * Math.cos(phi_nToCa);
	A[2][1] = -Math.cos(theta_nToCa) * Math.sin(phi_nToCa);
	A[2][2] = Math.sin(theta_nToCa);
	
	Matrix Rn2ca = new Matrix(A);
	//Rotate the vectors N(i)->CO(i-1) and CA(i)->CO(i) 
	double [] n2Co1 = Rn2ca.times(nToCO1);
	double [] ca2Co = Rn2ca.times(caToCO);
	double [] n2Ca = Rn2ca.times(nToCA);
	double x1 = ca2Co[0];
	double y1 = ca2Co[1];
	double z1 = ca2Co[2];  //variables for easy recognization
	double x2 = n2Co1[0];
	double y2 = n2Co1[1];
	double z2 = n2Co1[2];
	//For standard geometry gamma1 and gamma2 can NOT be zero
	if ( (1.0 - y1*y1)  <=  0.0 || (1.0 - y2*y2)  <=  0.0){
	    System.out.println("Error in dihedral angle computation");
	    System.exit(1);
	}	    
	double sin_gamma1 = Math.sqrt(1-y1*y1);
	double sin_gamma2 = Math.sqrt(1-y2*y2);
	double cosPhi = (z1*z2+x1*x2) / (sin_gamma1 * sin_gamma2);
	double sinPhi = (x1*z2-z1*x2) / (sin_gamma1 * sin_gamma2);
	if (sinPhi > 0)
	    return (Math.acos(cosPhi));
	else if (sinPhi < 0)
	    return (-Math.acos(cosPhi));
	else {
	    if (cosPhi > 0)
		return 0.0;
	    else return Math.PI;
	}
    }

    /**
     * Compute Phi and Psi directly from three backbone vectors (not
     * NH vector).
     * 
     * @param n2CO1  N->CO vector
     * @param n2CA   N->CA vector
     * @param ca2CO  CA->CO vector
     * 
     * @return phipsi angles
     */  
    public double PhiPsiCalPDB(double[] n2CO1, double[] n2CA, double[] ca2CO){
	double [] nToCO1 = dirCos(n2CO1);
	double [] nToCA = dirCos(n2CA);
	double [] caToCO = dirCos(ca2CO);
	// Cal the rotate matrix: First cal the polar angle phi and theta of the 
	// intermediate vector(N->CA) from it's directional cosines
	double theta_nToCa = Math.acos(nToCA[2]); //acos [0,PI) so should be OK
	double phi_nToCa = 0.0;
	double x = nToCA[0];
	double y = nToCA[1];
	if (theta_nToCa != 0.0){
	    if (x > 0) 
		phi_nToCa = Math.atan(y/x);
	    else if (x < 0)
		phi_nToCa = Math.atan(y/x) + Math.PI;
	    else {
		if (y > 0)
		    phi_nToCa = Math.PI / 2.0;
		else phi_nToCa = 3*Math.PI / 2.0;
	    }
	} else  phi_nToCa = 0.0;
	final int m = 3; //The dimensions of the Rotation Matrix
	final int n = 3;
	final double numMin = 1.0E-10;
	//construct the rotation matrix A needed to bring N-CA vector onto the +y-axis
	double [][] A = new double[3][3];
	A[0][0] = Math.sin(phi_nToCa);
	A[0][1] = -Math.cos(phi_nToCa);
	A[0][2] = 0.0;
	A[1][0] = Math.sin(theta_nToCa) * Math.cos(phi_nToCa);
	A[1][1] = Math.sin(theta_nToCa) * Math.sin(phi_nToCa);
	A[1][2] = Math.cos(theta_nToCa);
	A[2][0] = -Math.cos(theta_nToCa) * Math.cos(phi_nToCa);
	A[2][1] = -Math.cos(theta_nToCa) * Math.sin(phi_nToCa);
	A[2][2] = Math.sin(theta_nToCa);
	
	Matrix Rn2ca = new Matrix(A);
	//Rotate the vectors N(i)->CO(i-1) and CA(i)->CO(i) 
	double [] n2Co1 = Rn2ca.times(nToCO1);
	double [] ca2Co = Rn2ca.times(caToCO);
	double [] n2Ca  = Rn2ca.times(nToCA);
	double x1 = ca2Co[0];
	double y1 = ca2Co[1];
	double z1 = ca2Co[2];  
	double x2 = n2Co1[0];
	double y2 = n2Co1[1];
	double z2 = n2Co1[2];
	//For standard geometry gamma1 and gamma2 can NOT be zero
	double sin_gamma1 = Math.sqrt(1 - y1*y1);
	double sin_gamma2 = Math.sqrt(1 - y2*y2);
	double cosPhi = (z1*z2+x1*x2) / (sin_gamma1 * sin_gamma2);
	
	///////////////////////////
	//if(cosPhi>1.000 && ( Math.abs(cosPhi - 1.0) < numMin) )//new added
	//	cosPhi=1.000;
	//if(cosPhi<-1.000 && ( Math.abs(cosPhi + 1.0) < numMin) )//new added
	//	cosPhi=-1.000;
	
	double sinPhi = (x1*z2-z1*x2) / (sin_gamma1 * sin_gamma2);
	
	
	if (sinPhi > 0){
	    if ( (sinPhi > 1.0) && ( Math.abs(sinPhi - 1.0) < numMin) )
		return 0.0;
	    else if (  (sinPhi > 1.0) && (Math.abs(sinPhi - 1.0) >= numMin)){
		System.out.println("Math Error in PhiPsiCalPDB method");
		return 0.0;
	    }
	    else return (Math.acos(cosPhi));
	}else if (sinPhi < 0){
	    if ( (sinPhi < -1.0) && (Math.abs(sinPhi + 1.0) < numMin) )
		return Math.PI;
	    else if ( (sinPhi < -1.0) &&  (Math.abs(sinPhi + 1.0) >= numMin) ){
		System.out.println("Math Error in PhiPsiCalPDB method");
		return 0.0;
	    }else return (-Math.acos(cosPhi));
	}else {
	    if (cosPhi > 0)
		return 0.0;
	    else return Math.PI;
	}
    }

    /**
     * Compute all the backbone Phi and Psi angles from PDB from the
     * backbone backbone bond vectors between heavy atoms.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return a vector of PhiPsi object
     */  
    public Vector PhiPsiTotalPdb(final Vector pdbVec)
    {
	double [] amide1 = new double[3];
	double [] amide2 = new double[3];
	double [] co1 = new double[3];
	double [] co2 = new double[3];
	double [] ca1 = new double[3];
	double [] ca2 = new double[3];

	int no1 = 0, no2 = 0, no3 = 0, i=0, j=0, k=0;
	String resid1="", resid2="", resid3="";
	Cartesian cc = new Cartesian();
	String atom = "";
	//First Residue
	Pdb pp = (Pdb)pdbVec.elementAt(i); 
	no1 = pp.getResidueNo();
	resid1 = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	//the Atoms N, CA and CO of residue i
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("C"))
		co1 = cc.getXYZ();
	}
	// Beginning With Second Residue
	double [] nToCO1Vec = new double[3];
	double [] co1ToNVec = new double[3];
	Vector phiPsiVec = new Vector(pdbVec.size());
	double[] phiArr = new double[pdbVec.size()];
	double[] psiArr = new double[pdbVec.size()];
	for (int m=0; m<phiPsiVec.size(); m++){
	    phiArr[m] = 999.0;         //For labelling
	    psiArr[m] = 999.0;
	}
	phiArr[i] = -128.342053 * Const.cst;// 0.0; //Assign the phi of the first Residue as 0.0
	//Cal the intraresidue CA->N and CA->CO vector
	double [] caToNVec  =  internuclearVec(ca1, amide1);
	double [] caToCOVec = internuclearVec(ca1, co1);
	i++;       //The second residue
	pp  = (Pdb)pdbVec.elementAt(i);
	no2 = pp.getResidueNo();
	resid2  = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){ //The atoms N, CA and C of residue i+1
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2 = cc.getXYZ();
	    else if (atom.equals("C"))
		co2 = cc.getXYZ();
	}
	// Cal the CO(i-1)->N(i) Vector
	if ( no2 == (no1+1) ){
	    co1ToNVec =  internuclearVec(co1, amide2);
	    psiArr[i-1] = PhiPsiCalPDB(caToNVec, caToCOVec, co1ToNVec); //Cal the 1st Psi
	}
	phiPsiVec.addElement(new PhiPsi(no1, resid1, phiArr[i-1], psiArr[i-1]));
	//The intraResidue N(i)->CA(i) vector
	double [] nToCAVec = internuclearVec(amide2, ca2);
	//The intraResidue CA(i)->CO(i) vector
	caToCOVec = internuclearVec(ca2, co2);
	if (no2 == (no1 + 1)){  // if consecutive, compute the interResidue N(i)->CO(i-1) vector
	    nToCO1Vec =  internuclearVec(amide2, co1);
	    phiArr[i] = PhiPsiCalPDB(nToCO1Vec, nToCAVec, caToCOVec); //calculate phi
	}
        i++; //Begin with the third residue we go into the While loop
	while (i < pdbVec.size()){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no3 = pp.getResidueNo();
	    resid3 = pp.getResidue();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide1 = cc.getXYZ();
		else if (atom.equals("CA"))
		    ca1 = cc.getXYZ();
		else if (atom.equals("C"))
		    co1 = cc.getXYZ();
	    }
	    //Cal the Vector CO(i)->N(i+1), co2 from previous residue
	    if ( no3 == (no2+1) ){
		co1ToNVec =  internuclearVec(co2, amide1);
		for (k=0; k<nToCAVec.length; k++) //reverse the direction of the N(i)->CA(i)
		    caToNVec[k] = -nToCAVec[k];
		psiArr[i-1] = PhiPsiCalPDB(caToNVec, caToCOVec, co1ToNVec);
	    }
	    phiPsiVec.addElement(new PhiPsi(no2,resid2,phiArr[i-1], psiArr[i-1]));
	    //Cal the Vector N(i)->CA(i)    		
	    nToCAVec  = internuclearVec(amide1, ca1);
	    //Cal the Vector CA(i)->CO(i)    		
	    caToCOVec = internuclearVec(ca1, co1);
	    if ( no3 == (no2+1) ){
		for (k=0; k<co1ToNVec.length; k++) //reverse the direction of CO(i)->N(i+1)
		    nToCO1Vec[k] = -co1ToNVec[k];
		phiArr[i] = PhiPsiCalPDB(nToCO1Vec, nToCAVec, caToCOVec);
	    }
	    for (k=0; k<co1.length; k++)  //update co2
		co2[k] = co1[k];
	    i++;      
	    no2 = no3;   //Updating the residueNO and ResidueType
	    resid2 = resid3;
	}
	psiArr[i-1] = Const.psiAveHelix;// 0.0;  //The last Psi is assigned to 0.0
	phiPsiVec.addElement(new PhiPsi(no2,resid2,phiArr[i-1],psiArr[i-1]));
// 	System.out.println((no2-1)+"  "+(phiArr[i-2]/cst)+"  "+(psiArr[i-2]/cst));
	return phiPsiVec;
    }
 
    /**
     * Extract backbone atoms from a PDB file.
     * 
     * @param pdbVec the original Pdb vector
     * 
     * @return a Pdb file with only protein backbone atoms
     */
    public Vector extractBackbone(final Vector pdbVec){
	Cartesian cc = null;
	String atom="", resid;
	Vector atomVec = new Vector();
	Vector atomVecN = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atomVec = pp.getAtomVec();
	    for (int j=0; j<atomVec.size(); j++){
		cc = (Cartesian) atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N") )
		    atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		else if (atom.equals("H"))
		    atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		else if (atom.equals("CA"))
		    atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		else if (atom.equals("HA"))
		    atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		else if (atom.equals("C"))
		    atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		else if (atom.equals("O"))
		    atomVecN.add(new Cartesian(atom, cc.getXYZ()));
	    }
	    pdbVecN.add(new Pdb(no, resid, atomVecN));
	    atomVecN = new Vector();
	}
	return pdbVecN;
    }

    /**
     * Extract backbone atoms from a PDB file.
     * 
     * @param pdbVec the original Pdb vector
     * @param residueFirst the residue first
     * @param residueLast the residue last
     * @param firstCoords the first coords
     * @param lastCoords the last coords
     * 
     * @return a Pdb file with only protein backbone atoms
     */
    public Vector extractFragment(final Vector pdbVec, int residueFirst, int residueLast,
				  double [] firstCoords, double [] lastCoords){
	int no;
	Pdb pp = new Pdb();
	Vector pdbVecN = new Vector();
	String atom="", resid;
	Vector atomVec = new Vector();
	Vector atomVecN = new Vector();
	Cartesian cc = new Cartesian();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    if ( no == residueFirst){
		atomVec = pp.getAtomVec();
		for (int j=0; j<atomVec.size(); j++){
		    cc = (Cartesian) atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N") )
			System.arraycopy(cc.getXYZ(), 0, firstCoords, 0, 3);
		}
	    }
	    if ( no == residueLast){
		atomVec = pp.getAtomVec();
		for (int j=0; j<atomVec.size(); j++){
		    cc = (Cartesian) atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N") )
			atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		    else if (atom.equals("HN")) //only for DYANA
			atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		    else if (atom.equals("CA")){
			atomVecN.add(new Cartesian(atom, cc.getXYZ()));
			System.arraycopy(cc.getXYZ(), 0, lastCoords, 0, 3);
		    }
		}
		pdbVecN.add(new Pdb(no, resid, atomVecN));
	    }
	    if (no > (residueFirst - 1) && no < residueLast)
		pdbVecN.add(pp);
	}
	Vector pdbBB = extractBackbone(pdbVecN);
	return pdbBB;
    }


  /**
   * Extract backbone atoms from a PDB file.
   * 
   * @param pdbVec the original Pdb vector
   * @param residueFirst the residue first
   * @param residueLast the residue last
   * 
   * @return a Pdb file with only protein backbone atoms
   */
    public Vector extractFragment(final Vector pdbVec, int residueFirst, int residueLast){
	int no;
	Pdb pp = new Pdb();
	Vector pdbVecN = new Vector();
	String atom="", resid;
	Vector atomVec = new Vector();
	Vector atomVecN = new Vector();
	Cartesian cc = new Cartesian();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    if ( no == residueFirst){
		atomVec = pp.getAtomVec();
		for (int j=0; j<atomVec.size(); j++){
		    cc = (Cartesian) atomVec.elementAt(j);
		    atom = cc.getAtom();
		}
	    }
	    if ( no == residueLast){
		atomVec = pp.getAtomVec();
		for (int j=0; j<atomVec.size(); j++){
		    cc = (Cartesian) atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N") )
			atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		    else if (atom.equals("HN") || atom.equals("H")) //HN only for DYANA
			atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		    else if (atom.equals("CA")){
			atomVecN.add(new Cartesian(atom, cc.getXYZ()));
		    }
		}
		pdbVecN.add(new Pdb(no, resid, atomVecN));
	    }
	    if (no > (residueFirst - 1) && no < residueLast)
		pdbVecN.add(pp);
	}
	Vector pdbBB = extractBackbone(pdbVecN);
	return pdbBB;
    }
  
    
    
    /**
     * Convert the old PDB namaclature to new one
     * zeng changed the code,.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector<Pdb> nameConvert2PDBNew(final Vector<Pdb> pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++)
	{
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    if (resid.equalsIgnoreCase("ALA"))
	    {
			for (int j=0; j<atoms.size(); j++)
			{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if ( atomName.equalsIgnoreCase("H")  )
					atomName = "HN";
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG"))
	    {
			for (int j=0; j<atoms.size(); j++)
			{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if ( atomName.equalsIgnoreCase("H")  )
					atomName = "HN";
			    if ( atomName.equalsIgnoreCase("HB2")  )
				atomName = "HB1";
			    else if (atomName.equalsIgnoreCase("HB3"))
				atomName = "HB2";
			    if (atomName.equalsIgnoreCase("HG2") )
				atomName = "HG1";
			    else if ( atomName.equalsIgnoreCase("HG3"))
				atomName = "HG2";
			    if ( atomName.equalsIgnoreCase("HD2"))
				atomName = "HD1";
			    else if (atomName.equalsIgnoreCase("HD3"))
				atomName = "HD2";
			   
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("HIS") || resid.equalsIgnoreCase("SER")
		     || resid.equalsIgnoreCase("TRP") || resid.equalsIgnoreCase("CYS"))
	    {
			for (int j=0; j<atoms.size(); j++)
			{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if ( atomName.equalsIgnoreCase("H")  )
					atomName = "HN";
			    if ( atomName.equalsIgnoreCase("HB2"))
				atomName = "HB1";
			    else if (atomName.equalsIgnoreCase("HB3"))
				atomName = "HB2";
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") )
	    {
			for (int j=0; j<atoms.size(); j++){
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if ( atomName.equalsIgnoreCase("H")  )
					atomName = "HN";
			    if (atomName.equalsIgnoreCase("HB2"))
				atomName = "HB1";
			    else if ( atomName.equalsIgnoreCase("HB3"))
				atomName = "HB2";
			    if ( atomName.equalsIgnoreCase("HG2"))
				atomName = "HG1";
			    else if ( atomName.equalsIgnoreCase("HG3"))
				atomName = "HG2";
			     coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if ( atomName.equalsIgnoreCase("HB2"))
			atomName = "HB1";
		    else if ( atomName.equalsIgnoreCase("HB3"))
			atomName = "HB2";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if ( atomName.equalsIgnoreCase("HB2"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("HB3"))
			atomName = "HB2";
		    if (atomName.equalsIgnoreCase("HG2"))
			atomName = "HG1";
		    else if ( atomName.equalsIgnoreCase("HG3"))
			atomName = "HG2";
		   
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if ( atomName.equalsIgnoreCase("HA2"))
			atomName = "HA1";
		    else if ( atomName.equalsIgnoreCase("HA3"))
			atomName = "HA2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("HG12"))
			atomName = "HG11";
		    else if ( atomName.equalsIgnoreCase("HG13"))
			atomName = "HG12";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if ( atomName.equalsIgnoreCase("HB2"))
			atomName = "HB1";
		    else if ( atomName.equalsIgnoreCase("HB3"))
			atomName = "HB2";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("HB2"))
			atomName = "HB1";
		    else if ( atomName.equalsIgnoreCase("HB3"))
			atomName = "HB2";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("HB2"))
			atomName = "HB1";
		    else if ( atomName.equalsIgnoreCase("HB3"))
			atomName = "HB2";
		    if (atomName.equalsIgnoreCase("HG2") )
			atomName = "HG1";
		    else if (atomName.equalsIgnoreCase("HG3"))
			atomName = "HG2";
		    if (atomName.equalsIgnoreCase("HD2"))
			atomName = "HD1";
		    else if ( atomName.equalsIgnoreCase("HD3"))
			atomName = "HD2";
		    if ( atomName.equalsIgnoreCase("HE2"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("HE3"))
			atomName = "HE2";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("HB2"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("HB3"))
			atomName = "HB2";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if ( atomName.equalsIgnoreCase("H")  )
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("HB2"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("HB3"))
			atomName = "HB2";
		    if (atomName.equalsIgnoreCase("HG2"))
			atomName = "HG1";
		    else if (atomName.equalsIgnoreCase("HG3"))
			atomName = "HG2";
		    if (atomName.equalsIgnoreCase("HD2"))
			atomName = "HD1";
		    else if ( atomName.equalsIgnoreCase("HD3"))
			atomName = "HD2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR"))
	    {
			for (int j=0; j<atoms.size(); j++){
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if ( atomName.equalsIgnoreCase("H")  )
					atomName = "HN";
			  
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }
	    else
	    {
	    	for (int j=0; j<atoms.size(); j++)
	    	{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();			    
			    if ( atomName.equalsIgnoreCase("H")  )
					atomName = "HN";
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();	    	
	    }
	}
	return pdbVecN;
    }
    
    /**
     * Convert the old PDB namaclature to new one
     * zeng changed the code, such that all methel group protons become one proxy proton.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector nameConvert4SmallRotLibCS(final Vector pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    if (resid.equalsIgnoreCase("ALA")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1") )
			atomName = "HB";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
			atomName = "HB";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1")  )
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")  || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") ||atomName.equalsIgnoreCase("HG1") )
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG")  || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HH1"))
			atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
			atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
			atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
			atomName = "HH22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("HIS") || resid.equalsIgnoreCase("SER")
		     || resid.equalsIgnoreCase("TRP") || resid.equalsIgnoreCase("CYS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE")||atomName.equalsIgnoreCase("HE1") )
			atomName = "HE";///not quite sure about this, need to double check with some expert
		    else if (atomName.equalsIgnoreCase("2HE")|| atomName.equalsIgnoreCase("HE2"))
			atomName = "HE";///not quite sure about this, need to double check with some expert
		    else if (atomName.equalsIgnoreCase("3HE")||atomName.equalsIgnoreCase("HE3"))
			atomName = "HE";///not quite sure about this, need to double check with some expert
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HA") || atomName.equalsIgnoreCase("HA1"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("2HA")|| atomName.equalsIgnoreCase("HA2"))
			atomName = "HA3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG1")|| atomName.equalsIgnoreCase("HG11"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("2HG1")|| atomName.equalsIgnoreCase("HG12"))
			atomName = "HG13";
		    if (atomName.equalsIgnoreCase("1HG2")|| atomName.equalsIgnoreCase("HG21"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG2")||atomName.equalsIgnoreCase("HG22"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG2")||atomName.equalsIgnoreCase("HG23"))
			atomName = "HG2";
		    if (atomName.equalsIgnoreCase("1HD1") || atomName.equalsIgnoreCase("HD11") )
			atomName = "HD1";//changed by zeng
		    else if (atomName.equalsIgnoreCase("2HD1")|| atomName.equalsIgnoreCase("HD12") )
			atomName = "HD1";//changed by zeng
		    else if (atomName.equalsIgnoreCase("3HD1")||atomName.equalsIgnoreCase("HD13") )
			atomName = "HD1";//changed by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2")|| atomName.equalsIgnoreCase("HD21"))
			atomName = "HD2";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD2")|| atomName.equalsIgnoreCase("HD22"))
			atomName = "HD2";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD2")|| atomName.equalsIgnoreCase("HD23"))
			atomName = "HD2";//by zeng
		    if (atomName.equalsIgnoreCase("1HD1")|| atomName.equalsIgnoreCase("HD11"))
			atomName = "HD1";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD1")|| atomName.equalsIgnoreCase("HD12"))
			atomName = "HD1";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD1")|| atomName.equalsIgnoreCase("HD13"))
			atomName = "HD1";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2")|| atomName.equalsIgnoreCase("HG21"))
			atomName = "HG2";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG2")||atomName.equalsIgnoreCase("HG22"))
			atomName = "HG2";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG2")|| atomName.equalsIgnoreCase("HG23"))
			atomName = "HG2";//by zeng
		    if (atomName.equalsIgnoreCase("1HG1")||atomName.equalsIgnoreCase("HG11") )
			atomName = "HG1";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG1")|| atomName.equalsIgnoreCase("HG12"))
			atomName = "HG1";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG1")|| atomName.equalsIgnoreCase("HG13"))
			atomName = "HG1";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") ||atomName.equalsIgnoreCase("HG1") )
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD")|| atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE") || atomName.equalsIgnoreCase("HE1"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("2HE")||atomName.equalsIgnoreCase("HE2"))
			atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ")||atomName.equalsIgnoreCase("HZ1") )
			atomName = "HZ";
		    else if (atomName.equalsIgnoreCase("2HZ")||atomName.equalsIgnoreCase("HZ2"))
			atomName = "HZ";
		    else if (atomName.equalsIgnoreCase("3HZ")||atomName.equalsIgnoreCase("HZ3"))
			atomName = "HZ";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG")||atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG")||atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") ||atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG") )
			atomName = "HG1";
		    if (atomName.equalsIgnoreCase("1HG2")||atomName.equalsIgnoreCase("HG21") )
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG2")|| atomName.equalsIgnoreCase("HG22"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG2")||atomName.equalsIgnoreCase("HG23"))
			atomName = "HG2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }
	}
	return pdbVecN;
    }
 
    /**
     * Convert the old PDB namaclature to new one
     * for computing the structure rmsd. each proton has different id
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector nameConvert4SmallRotLibStr(final Vector pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    
	   // if (no==1 || no==62)
	    	//continue;
	    
	    
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	   
	    
	    if (resid.equalsIgnoreCase("ALA")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		   
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1") )
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1")  )
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")  || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") ||atomName.equalsIgnoreCase("HG1") )
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG")  || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HH1"))
			atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
			atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
			atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
			atomName = "HH22";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("HIS") || resid.equalsIgnoreCase("SER")
		     || resid.equalsIgnoreCase("TRP") || resid.equalsIgnoreCase("CYS")){
			for (int j=0; j<atoms.size(); j++){
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
				atomName = "HB2";
			    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
				atomName = "HB3";
			    coord = atom.getXYZ();
			    if(atomName.equalsIgnoreCase("HN"))
			    	atomName="H";
			    if(atomName.equalsIgnoreCase("HE2") || atomName.equalsIgnoreCase("HD1"))
			    	continue;
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE")||atomName.equalsIgnoreCase("HE1") )
			atomName = "HE1";///not quite sure about this, need to double check with some expert
		    else if (atomName.equalsIgnoreCase("2HE")|| atomName.equalsIgnoreCase("HE2"))
			atomName = "HE2";///not quite sure about this, need to double check with some expert
		    else if (atomName.equalsIgnoreCase("3HE")||atomName.equalsIgnoreCase("HE3"))
			atomName = "HE3";///not quite sure about this, need to double check with some expert
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HA") || atomName.equalsIgnoreCase("HA1"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("2HA")|| atomName.equalsIgnoreCase("HA2"))
			atomName = "HA3";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG1")|| atomName.equalsIgnoreCase("HG11"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("2HG1")|| atomName.equalsIgnoreCase("HG12"))
			atomName = "HG13";
		    if (atomName.equalsIgnoreCase("1HG2")|| atomName.equalsIgnoreCase("HG21"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2")||atomName.equalsIgnoreCase("HG22"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2")||atomName.equalsIgnoreCase("HG23"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HD1") || atomName.equalsIgnoreCase("HD11") )
			atomName = "HD11";//changed by zeng
		    else if (atomName.equalsIgnoreCase("2HD1")|| atomName.equalsIgnoreCase("HD12") )
			atomName = "HD12";//changed by zeng
		    else if (atomName.equalsIgnoreCase("3HD1")||atomName.equalsIgnoreCase("HD13") )
			atomName = "HD13";//changed by zeng
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2")|| atomName.equalsIgnoreCase("HD21"))
			atomName = "HD21";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD2")|| atomName.equalsIgnoreCase("HD22"))
			atomName = "HD22";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD2")|| atomName.equalsIgnoreCase("HD23"))
			atomName = "HD23";//by zeng
		    if (atomName.equalsIgnoreCase("1HD1")|| atomName.equalsIgnoreCase("HD11"))
			atomName = "HD11";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD1")|| atomName.equalsIgnoreCase("HD12"))
			atomName = "HD12";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD1")|| atomName.equalsIgnoreCase("HD13"))
			atomName = "HD13";//by zeng
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2")|| atomName.equalsIgnoreCase("HG21"))
			atomName = "HG21";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG2")||atomName.equalsIgnoreCase("HG22"))
			atomName = "HG22";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG2")|| atomName.equalsIgnoreCase("HG23"))
			atomName = "HG23";//by zeng
		    if (atomName.equalsIgnoreCase("1HG1")||atomName.equalsIgnoreCase("HG11") )
			atomName = "HG11";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG1")|| atomName.equalsIgnoreCase("HG12"))
			atomName = "HG12";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG1")|| atomName.equalsIgnoreCase("HG13"))
			atomName = "HG13";//by zeng
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") ||atomName.equalsIgnoreCase("HG1") )
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD")|| atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE") || atomName.equalsIgnoreCase("HE1"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("2HE")||atomName.equalsIgnoreCase("HE2"))
			atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ")||atomName.equalsIgnoreCase("HZ1") )
			atomName = "HZ1";
		    else if (atomName.equalsIgnoreCase("2HZ")||atomName.equalsIgnoreCase("HZ2"))
			atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ")||atomName.equalsIgnoreCase("HZ3"))
			atomName = "HZ3";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG")||atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG")||atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") ||atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG") )
			atomName = "HG1";
		    if (atomName.equalsIgnoreCase("1HG2")||atomName.equalsIgnoreCase("HG21") )
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2")|| atomName.equalsIgnoreCase("HG22"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2")||atomName.equalsIgnoreCase("HG23"))
			atomName = "HG23";
		    coord = atom.getXYZ();
		    if(atomName.equalsIgnoreCase("HN"))
		    	atomName="H";
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}	
		
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }
	    else{
	    	for (int j=0; j<atoms.size(); j++){
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    
			    coord = atom.getXYZ();
			    if(atomName.equalsIgnoreCase("HN"))
			    	atomName="H";
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}	
			
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    	
	    }
	}
	return pdbVecN;
    }
 
    /**
     * Convert the old PDB namaclature to new one
     * zeng changed the code, such that all methel group protons become one proxy proton.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector nameConvertCS(final Vector pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    if (resid.equalsIgnoreCase("ALA")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB";
		    else if (atomName.equalsIgnoreCase("3HB"))
			atomName = "HB";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HH1"))
			atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
			atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
			atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
			atomName = "HH22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("HIS") || resid.equalsIgnoreCase("SER")
		     || resid.equalsIgnoreCase("TRP") || resid.equalsIgnoreCase("CYS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("3HE"))
			atomName = "HE3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HA"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("2HA"))
			atomName = "HA3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG13";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG2";
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD1";//changed by zeng
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD1";//changed by zeng
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD1";//changed by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD2";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD2";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD2"))
			atomName = "HD2";//by zeng
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD1";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD1";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD1";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG2";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG2";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG2";//by zeng
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG1";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG1";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG1"))
			atomName = "HG1";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ"))
			atomName = "HZ1";
		    else if (atomName.equalsIgnoreCase("2HZ"))
			atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ"))
			atomName = "HZ3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG1";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }
	}
	return pdbVecN;
    }
  
    /**
     * Convert the old PDB namaclature to new one
     * zeng changed the code, such that all methel group protons become one proxy proton.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector nameConvertStr(final Vector pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    if (resid.equalsIgnoreCase("ALA")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HH1"))
			atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
			atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
			atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
			atomName = "HH22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("HIS") || resid.equalsIgnoreCase("SER")
		     || resid.equalsIgnoreCase("TRP") || resid.equalsIgnoreCase("CYS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("3HE"))
			atomName = "HE3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HA"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("2HA"))
			atomName = "HA3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG13";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD11";//changed by zeng
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD12";//changed by zeng
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD13";//changed by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD2"))
			atomName = "HD23";//by zeng
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD11";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD12";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD13";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG21";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";//by zeng
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG11";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG12";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG1"))
			atomName = "HG13";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ"))
			atomName = "HZ1";
		    else if (atomName.equalsIgnoreCase("2HZ"))
			atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ"))
			atomName = "HZ3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG1";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }
	}
	return pdbVecN;
    }
    
    /**
     * In this function, only the order of atom name is changed, and H-> HN
     * 
     * 
     * eg 2HB=>HB2, 3HB=>HB3.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector<Pdb> nameConvertOrder(final Vector<Pdb> pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    if (resid.equalsIgnoreCase("ALA")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB")||atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2") )
			atomName = "HB2";
		    if (atomName.equalsIgnoreCase("3HB")||atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1")  )
			atomName = "HG1";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG") || atomName.equalsIgnoreCase("HG3"))
				atomName = "HG3";
		    if (atomName.equalsIgnoreCase("3HD") ||  atomName.equalsIgnoreCase("HD3"))
			atomName = "HD3";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
				atomName = "HD1";
		    if (atomName.equalsIgnoreCase("1HH1"))
			atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
			atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
			atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
			atomName = "HH22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if ( resid.equalsIgnoreCase("TRP") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1") )
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    else if (atomName.equalsIgnoreCase("3HE") || atomName.equalsIgnoreCase("HE3"))
				atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HE") || atomName.equalsIgnoreCase("HE1"))
				atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HZ") || atomName.equalsIgnoreCase("HZ2"))
				atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ") || atomName.equalsIgnoreCase("HZ3"))
				atomName = "HZ3";
		    else if (atomName.equalsIgnoreCase("HD"))
				atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("HH"))
				atomName = "HH2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    } else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("SER")
		      || resid.equalsIgnoreCase("CYS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1") )
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    else if (atomName.equalsIgnoreCase("2HD") )
				atomName = "HD2";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG1";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG") || atomName.equalsIgnoreCase("HG3"))
				atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("3HE"))
			atomName = "HE3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB")|| atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG")||atomName.equalsIgnoreCase("HG1") )
			atomName = "HG1"; 
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG") || atomName.equalsIgnoreCase("HG3"))
				atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HA") || atomName.equalsIgnoreCase("HA1"))
			atomName = "HA1";
		    else if (atomName.equalsIgnoreCase("2HA") || atomName.equalsIgnoreCase("HA2"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("3HA") || atomName.equalsIgnoreCase("HA3"))
				atomName = "HA3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HG1") || atomName.equalsIgnoreCase("HG11"))
			atomName = "HG11";
		    else if (atomName.equalsIgnoreCase("2HG1") || atomName.equalsIgnoreCase("HG12"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("3HG1") || atomName.equalsIgnoreCase("HG13"))
				atomName = "HG13";
		    if (atomName.equalsIgnoreCase("1HG2") || atomName.equalsIgnoreCase("HG21"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD11";
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD12";
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD13";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") ||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") ||atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    else if (atomName.equalsIgnoreCase("3HD2"))
			atomName = "HD23";
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD11";
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD12";
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD13";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG11";
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("3HG1"))
			atomName = "HG13";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG1";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG") || atomName.equalsIgnoreCase("HG3"))
				atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("3HD") || atomName.equalsIgnoreCase("HD3"))
				atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE")|| atomName.equalsIgnoreCase("HE1"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE")|| atomName.equalsIgnoreCase("HE2"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("3HE")|| atomName.equalsIgnoreCase("HE3"))
				atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ"))
			atomName = "HZ1";
		    else if (atomName.equalsIgnoreCase("2HZ"))
			atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ"))
			atomName = "HZ3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("H"))
				atomName = "HN";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
				atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG")|| atomName.equalsIgnoreCase("HG1"))
			atomName = "HG1";
		    else if (atomName.equalsIgnoreCase("2HG")|| atomName.equalsIgnoreCase("HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG")|| atomName.equalsIgnoreCase("HG3"))
				atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD")|| atomName.equalsIgnoreCase("HD2"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("3HD")|| atomName.equalsIgnoreCase("HD3"))
				atomName = "HD3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR"))
	    {
			for (int j=0; j<atoms.size(); j++)
			{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if (atomName.equalsIgnoreCase("H"))
					atomName = "HN";
			    if (atomName.equalsIgnoreCase("1HG"))
				atomName = "HG1";
			    if (atomName.equalsIgnoreCase("1HG2"))
				atomName = "HG21";
			    else if (atomName.equalsIgnoreCase("2HG2"))
				atomName = "HG22";
			    else if (atomName.equalsIgnoreCase("3HG2"))
				atomName = "HG23";
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }
	    else
	    {
	    	for (int j=0; j<atoms.size(); j++)
			{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if (atomName.equalsIgnoreCase("H"))
					atomName = "HN";
			    
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }
	}
	return pdbVecN;
    }
    
    /**
     * Lincong's original function:
     * Convert the old PDB namaclature to new one
     * bugs fixed...
     * be careful of the original format.
     * it may have some problem if converting from new format to new format
     * eg HB2=>HB1, HB3=>HB3
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector nameConvert(final Vector pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    if (resid.equalsIgnoreCase("ALA")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB")||atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2") )
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1")  )
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") ||  atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HH1"))
			atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
			atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
			atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
			atomName = "HH22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if ( resid.equalsIgnoreCase("TRP") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1") )
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    else if (atomName.equalsIgnoreCase("3HE") || atomName.equalsIgnoreCase("HE3"))
				atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HE") || atomName.equalsIgnoreCase("HE1"))
				atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HZ") || atomName.equalsIgnoreCase("HZ2"))
				atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ") || atomName.equalsIgnoreCase("HZ3"))
				atomName = "HZ3";
		    else if (atomName.equalsIgnoreCase("HD"))
				atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("HH"))
				atomName = "HH2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    } else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("SER")
		      || resid.equalsIgnoreCase("CYS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1") )
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    else if (atomName.equalsIgnoreCase("2HD") )
				atomName = "HD2";
		    
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("3HE"))
			atomName = "HE3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG")||atomName.equalsIgnoreCase("HG1") )
			atomName = "HG2"; 
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HA") || atomName.equalsIgnoreCase("HA1"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("2HA") || atomName.equalsIgnoreCase("HA2"))
			atomName = "HA3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HG1") || atomName.equalsIgnoreCase("HG11"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("2HG1") || atomName.equalsIgnoreCase("HG12"))
			atomName = "HG13";
		    if (atomName.equalsIgnoreCase("1HG2") || atomName.equalsIgnoreCase("HG21"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD11";
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD12";
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD13";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") ||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    else if (atomName.equalsIgnoreCase("3HD2"))
			atomName = "HD23";
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD11";
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD12";
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD13";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG11";
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("3HG1"))
			atomName = "HG13";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE")|| atomName.equalsIgnoreCase("HE1"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("2HE")|| atomName.equalsIgnoreCase("HE2"))
			atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ"))
			atomName = "HZ1";
		    else if (atomName.equalsIgnoreCase("2HZ"))
			atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ"))
			atomName = "HZ3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("HN"))
				atomName = "H";
		    if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG")|| atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG")|| atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD")|| atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR"))
	    {
			for (int j=0; j<atoms.size(); j++)
			{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if (atomName.equalsIgnoreCase("HN"))
					atomName = "H";
			    if (atomName.equalsIgnoreCase("1HG"))
				atomName = "HG1";
			    if (atomName.equalsIgnoreCase("1HG2"))
				atomName = "HG21";
			    else if (atomName.equalsIgnoreCase("2HG2"))
				atomName = "HG22";
			    else if (atomName.equalsIgnoreCase("3HG2"))
				atomName = "HG23";
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }
	    else
	    {
	    	for (int j=0; j<atoms.size(); j++)
			{
			    atom = (Cartesian) atoms.elementAt(j);
			    atomName = atom.getAtom();
			    if (atomName.equalsIgnoreCase("HN"))
					atomName = "H";
			    
			    coord = atom.getXYZ();
			    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
			}
			pdbVecN.add(new Pdb(no, resid, atomVec));
			atomVec = new Vector();
	    }
	}
	return pdbVecN;
    }
    
    /**
     * Convert the PDB-NEW namaclature to BMRB-NEW namaclature:
     * e.g. HB1,HB2 or 1HB, 2HB-->HB2,HB3.
     * return new atom name
     * 
     * @param resid the resid
     * @param atomNameInput the atom name input
     * 
     * @return the string
     */
    public String  NameFromPdbToBMRB(String resid, String atomNameInput)
    {
    	String atomName=atomNameInput;
    	//String newAtomName=atomName;   	
    	if(atomName.equalsIgnoreCase("HN"))
    		atomName="H";
    	if (resid.equalsIgnoreCase("ALA"))
    	{    			
		   if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1") )
		   atomName = "HB1";
		   else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
		   atomName = "HB2";
		   else if (atomName.equalsIgnoreCase("3HB") || atomName.equalsIgnoreCase("HB3"))
		   atomName = "HB3";
    	}    			
    	else if(resid.equalsIgnoreCase("ARG"))
    	{
    		if (atomName.equalsIgnoreCase("1HB") || atomName.equalsIgnoreCase("HB1")  )
    			atomName = "HB2";
    		else if (atomName.equalsIgnoreCase("2HB")  || atomName.equalsIgnoreCase("HB2"))
    			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") ||atomName.equalsIgnoreCase("HG1") )
		    	atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG")  || atomName.equalsIgnoreCase("HG2"))
		    	atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
		    	atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD") || atomName.equalsIgnoreCase("HD2"))
		    	atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HH1"))
		    	atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
		    	atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
		    	atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
		    	atomName = "HH22";
    			
    	}    		
    	else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
    			     || resid.equalsIgnoreCase("HIS") || resid.equalsIgnoreCase("SER")
    			     || resid.equalsIgnoreCase("TRP") || resid.equalsIgnoreCase("CYS"))
    	{ 			
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
    	}    			 
    	else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") )
    	{
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE")||atomName.equalsIgnoreCase("HE1") )
			atomName = "HE1";///not quite sure about this, need to double check with some expert
		    else if (atomName.equalsIgnoreCase("2HE")|| atomName.equalsIgnoreCase("HE2"))
			atomName = "HE2";///not quite sure about this, need to double check with some expert
		    else if (atomName.equalsIgnoreCase("3HE")||atomName.equalsIgnoreCase("HE3"))
			atomName = "HE3";///not quite sure about this, need to double check with some expert

    	}    		   
    	else if(resid.equalsIgnoreCase("ASN"))
    	{
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
    	}    	
    	else if(resid.equalsIgnoreCase("GLN"))
    	{
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") || atomName.equalsIgnoreCase("HG1"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
    	}    	
    	else if(resid.equalsIgnoreCase("GLY") )
    	{  			
		    if (atomName.equalsIgnoreCase("1HA") || atomName.equalsIgnoreCase("HA1"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("2HA")|| atomName.equalsIgnoreCase("HA2"))
			atomName = "HA3";
    	}
    	else if (resid.equalsIgnoreCase("ILE"))
    	{   			
		    if (atomName.equalsIgnoreCase("1HG1")|| atomName.equalsIgnoreCase("HG11"))
			atomName = "HG12";
		    else if (atomName.equalsIgnoreCase("2HG1")|| atomName.equalsIgnoreCase("HG12"))
			atomName = "HG13";
		    if (atomName.equalsIgnoreCase("1HG2")|| atomName.equalsIgnoreCase("HG21"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2")||atomName.equalsIgnoreCase("HG22"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2")||atomName.equalsIgnoreCase("HG23"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HD1") || atomName.equalsIgnoreCase("HD11") )
			atomName = "HD11";//changed by zeng
		    else if (atomName.equalsIgnoreCase("2HD1")|| atomName.equalsIgnoreCase("HD12") )
			atomName = "HD12";//changed by zeng
		    else if (atomName.equalsIgnoreCase("3HD1")||atomName.equalsIgnoreCase("HD13") )
			atomName = "HD13";//changed by zeng
    	}
    	else if (resid.equalsIgnoreCase("LEU") )
    	{	
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB") || atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2")|| atomName.equalsIgnoreCase("HD21"))
			atomName = "HD21";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD2")|| atomName.equalsIgnoreCase("HD22"))
			atomName = "HD22";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD2")|| atomName.equalsIgnoreCase("HD23"))
			atomName = "HD23";//by zeng
		    if (atomName.equalsIgnoreCase("1HD1")|| atomName.equalsIgnoreCase("HD11"))
			atomName = "HD11";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD1")|| atomName.equalsIgnoreCase("HD12"))
			atomName = "HD12";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD1")|| atomName.equalsIgnoreCase("HD13"))
			atomName = "HD13";//by zeng
    	}    	
    	else if (resid.equalsIgnoreCase("VAL"))
    	{
    		if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2")|| atomName.equalsIgnoreCase("HG21"))
			atomName = "HG21";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG2")||atomName.equalsIgnoreCase("HG22"))
			atomName = "HG22";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG2")|| atomName.equalsIgnoreCase("HG23"))
			atomName = "HG23";//by zeng
		    if (atomName.equalsIgnoreCase("1HG1")||atomName.equalsIgnoreCase("HG11") )
			atomName = "HG11";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG1")|| atomName.equalsIgnoreCase("HG12"))
			atomName = "HG12";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG1")|| atomName.equalsIgnoreCase("HG13"))
			atomName = "HG13";//by zeng
    	}    		
    	else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") )
    	{
		    if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")|| atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG") ||atomName.equalsIgnoreCase("HG1") )
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG") || atomName.equalsIgnoreCase("HG2"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD") || atomName.equalsIgnoreCase("HD1"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD")|| atomName.equalsIgnoreCase("HD2"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE") || atomName.equalsIgnoreCase("HE1"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("2HE")||atomName.equalsIgnoreCase("HE2"))
			atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ")||atomName.equalsIgnoreCase("HZ1") )
			atomName = "HZ1";
		    else if (atomName.equalsIgnoreCase("2HZ")||atomName.equalsIgnoreCase("HZ2"))
			atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ")||atomName.equalsIgnoreCase("HZ3"))
			atomName = "HZ3";
    	}    
    	else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
    			     || resid.equalsIgnoreCase("HIS"))
    	{
    		if (atomName.equalsIgnoreCase("1HB")|| atomName.equalsIgnoreCase("HB1"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB")||atomName.equalsIgnoreCase("HB2"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
    	}    			
    	else  if (resid.equalsIgnoreCase("THR"))
    	{
		    if (atomName.equalsIgnoreCase("1HG") )
			atomName = "HG1";
		    if (atomName.equalsIgnoreCase("1HG2")||atomName.equalsIgnoreCase("HG21") )
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2")|| atomName.equalsIgnoreCase("HG22"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2")||atomName.equalsIgnoreCase("HG23"))
			atomName = "HG23";
    	}    	
    	
    	return atomName;
    }
    
    /**
     * Convert the BMRB-NEW namaclature to PDB-NEW namaclature
     * return new atom name.
     * 
     * @param resName the res name
     * @param atomName the atom name
     * 
     * @return the string
     */
    public String  NameFromBMRBToPDB(String resName, String atomName)
    {
    	String newAtomName=atomName;
    	if(resName.equalsIgnoreCase("ALA") && atomName.equalsIgnoreCase("HB"))
    		newAtomName = "HB";    		
    	else if(resName.equalsIgnoreCase("ILE") && atomName.equalsIgnoreCase("HG2"))
    	{
    		newAtomName = "HG2";
    	}
    	else if(resName.equalsIgnoreCase("ILE") && atomName.equalsIgnoreCase("HD1"))
    	{
    		newAtomName = "HD1";
    		
    	}
    	else if(resName.equalsIgnoreCase("LEU") && atomName.equalsIgnoreCase("HD1"))
    	{
    		newAtomName = "HD1";
    	}
    	else if(resName.equalsIgnoreCase("LEU") && atomName.equalsIgnoreCase("HD2"))
    	{
    		newAtomName = "HD2";
    	}
    	else if(resName.equalsIgnoreCase("VAL") && atomName.equalsIgnoreCase("HG1"))
    	{
    		newAtomName = "HG1";
    	}else if(resName.equalsIgnoreCase("VAL") && atomName.equalsIgnoreCase("HG2"))
    	{
    		newAtomName = "HG2";
    	}else if(resName.equalsIgnoreCase("THR") && atomName.equalsIgnoreCase("HG2"))
    	{
    		newAtomName = "HG2";
    	}else if(resName.equalsIgnoreCase("MET") && atomName.equalsIgnoreCase("HE"))
    	{
    		newAtomName = "HE";
    	}
    	else
    	{ 	
    		//change back to old naming scheme for xplor
    		if (resName.equalsIgnoreCase("ARG")||resName.equalsIgnoreCase("PRO")) 
    		{
    			if(atomName.equalsIgnoreCase("HB2"))
    				newAtomName="HB1";
    			if(atomName.equalsIgnoreCase("HB3"))
    				newAtomName="HB2";
    			if(atomName.equalsIgnoreCase("HG2"))
    				newAtomName="HG1";
    			if(atomName.equalsIgnoreCase("HG3"))
    				newAtomName="HG2";
    			if(atomName.equalsIgnoreCase("HD2"))
    				newAtomName="HD1";
    			if(atomName.equalsIgnoreCase("HD3"))
    				newAtomName="HD2";
    		}
    		if (resName.equalsIgnoreCase("ASP")||resName.equalsIgnoreCase("CYS")||
    			resName.equalsIgnoreCase("HIS")||resName.equalsIgnoreCase("SER")||
	    		resName.equalsIgnoreCase("TRP")||resName.equalsIgnoreCase("ASN")||
	    		resName.equalsIgnoreCase("LEU")||resName.equalsIgnoreCase("VAL")||
	    		resName.equalsIgnoreCase("TYR")||resName.equalsIgnoreCase("PHE")) 
    		{
    			if(atomName.equalsIgnoreCase("HB2"))
    				newAtomName="HB1";
    			if(atomName.equalsIgnoreCase("HB3"))
    				newAtomName="HB2";
    		}
    		if (resName.equalsIgnoreCase("GLU")||resName.equalsIgnoreCase("MET")||resName.equalsIgnoreCase("GLN")) 
    		{
    			if(atomName.equalsIgnoreCase("HB2"))
    				newAtomName="HB1";
    			if(atomName.equalsIgnoreCase("HB3"))
    				newAtomName="HB2";
    			if(atomName.equalsIgnoreCase("HG2"))
    				newAtomName="HG1";
    			if(atomName.equalsIgnoreCase("HG3"))
    				newAtomName="HG2";	    	
    		}
    		if (resName.equalsIgnoreCase("GLY")) 
    		{
    			if(atomName.equalsIgnoreCase("HA2"))
    				newAtomName="HA1";
    			if(atomName.equalsIgnoreCase("HA3"))
    				newAtomName="HA2";   	
    		}
    		if (resName.equalsIgnoreCase("ILE")) 
    		{
    			if(atomName.equalsIgnoreCase("HG12"))
    				newAtomName="HG11";
    			if(atomName.equalsIgnoreCase("HG13"))
    				newAtomName="HG12";   	
    		}
    		if (resName.equalsIgnoreCase("LYS")||resName.equalsIgnoreCase("PRO")) 
    		{
    			if(atomName.equalsIgnoreCase("HB2"))
    				newAtomName="HB1";
    			if(atomName.equalsIgnoreCase("HB3"))
    				newAtomName="HB2";
    			if(atomName.equalsIgnoreCase("HG2"))
    				newAtomName="HG1";
    			if(atomName.equalsIgnoreCase("HG3"))
    				newAtomName="HG2";
    			if(atomName.equalsIgnoreCase("HD2"))
    				newAtomName="HD1";
    			if(atomName.equalsIgnoreCase("HD3"))
    				newAtomName="HD2";
    			if(atomName.equalsIgnoreCase("HE2"))
    				newAtomName="HE1";
    			if(atomName.equalsIgnoreCase("HE3"))
    				newAtomName="HE2";    	
    		}	        
    		if (atomName.equalsIgnoreCase("H"))
    			newAtomName="HN";    	
	    	
    	}
    	return newAtomName;
    	
    }
    
    /**
     * Convert the PDB-NEW namaclature to PDB-NEW namaclature
     * change the methel groups
     * return new atom name.
     * 
     * @param resName the res name
     * @param atomName the atom name
     * 
     * @return the string
     */
    public String  NameFromPDBToPDB(String resName, String atomName)
    {
    	String newAtomName=atomName;
    	if(resName.equalsIgnoreCase("ALA") && atomName.equalsIgnoreCase("HB"))
    		newAtomName = "HB#";    		
    	else if(resName.equalsIgnoreCase("ILE") && atomName.equalsIgnoreCase("HG2"))
    	{
    		newAtomName = "HG2#";
    	}
    	else if(resName.equalsIgnoreCase("ILE") && atomName.equalsIgnoreCase("HD1"))
    	{
    		newAtomName = "HD1#";
    		
    	}
    	else if(resName.equalsIgnoreCase("LEU") && atomName.equalsIgnoreCase("HD1"))
    	{
    		newAtomName = "HD1#";
    	}
    	else if(resName.equalsIgnoreCase("LEU") && atomName.equalsIgnoreCase("HD2"))
    	{
    		newAtomName = "HD2#";
    	}
    	else if(resName.equalsIgnoreCase("VAL") && atomName.equalsIgnoreCase("HG1"))
    	{
    		newAtomName = "HG1#";
    	}else if(resName.equalsIgnoreCase("VAL") && atomName.equalsIgnoreCase("HG2"))
    	{
    		newAtomName = "HG2#";
    	}else if(resName.equalsIgnoreCase("THR") && atomName.equalsIgnoreCase("HG2"))
    	{
    		newAtomName = "HG2#";
    	}
    	else if(resName.equalsIgnoreCase("MET") && atomName.equalsIgnoreCase("HE"))
    	{
    		newAtomName = "HE#";
    	}
    	else if(resName.equalsIgnoreCase("LYS") && atomName.equalsIgnoreCase("HB"))
    	{
    		newAtomName = "HB#";
    	}
    	else if(resName.equalsIgnoreCase("LYS") && atomName.equalsIgnoreCase("HE"))
    	{
    		newAtomName = "HB#";
    	}
    	else if(resName.equalsIgnoreCase("MET") && atomName.equalsIgnoreCase("HB"))
    	{
    		newAtomName = "HE#";
    	}
    	if (atomName.equalsIgnoreCase("H"))
    		newAtomName="HN";    	
	    	
    	
    	return newAtomName;
    	
    }
       
    	
    
    /**
     * Convert the old PDB namaclature to new one.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the vector
     */
    public Vector nameConvert22(final Vector pdbVec){
	Cartesian atom = null;
	String atomName="", resid;
	String atomName2="";
	Vector atoms = new Vector();
	Vector atomVec = new Vector();
	Vector pdbVecN = new Vector();
	int no;
	double [] coord = new double[3];
	double [] coordN = new double[3];
	Pdb pp = new Pdb();
	for (int i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    if (resid.equalsIgnoreCase("ALA")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB1";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("3HB"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ARG")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HH1"))
			atomName = "HH11";
		    else if (atomName.equalsIgnoreCase("2HH1"))
			atomName = "HH12";
		    if (atomName.equalsIgnoreCase("1HH2"))
			atomName = "HH21";
		    else if (atomName.equalsIgnoreCase("2HH2"))
			atomName = "HH22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASP") || resid.equalsIgnoreCase("CYS") 
		     || resid.equalsIgnoreCase("HIS") || resid.equalsIgnoreCase("SER")
		     || resid.equalsIgnoreCase("TRP") || resid.equalsIgnoreCase("CYS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLU") || resid.equalsIgnoreCase("MET") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    else if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("3HE"))
			atomName = "HE3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("ASN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD21";
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLN")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HE2"))
			atomName = "HE21";
		    else if (atomName.equalsIgnoreCase("2HE2"))
			atomName = "HE22";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("GLY") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HA"))
			atomName = "HA2";
		    else if (atomName.equalsIgnoreCase("2HA"))
			atomName = "HA3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("ILE")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG11";
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG12";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG21";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG22";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG23";
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD11";//changed by zeng
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD12";//changed by zeng
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD13";//changed by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("LEU") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD2"))
			atomName = "HD2";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD2"))
			atomName = "HD2";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD2"))
			atomName = "HD2";//by zeng
		    if (atomName.equalsIgnoreCase("1HD1"))
			atomName = "HD1";//by zeng
		    else if (atomName.equalsIgnoreCase("2HD1"))
			atomName = "HD1";//by zeng
		    else if (atomName.equalsIgnoreCase("3HD1"))
			atomName = "HD1";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("VAL")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG2";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG2";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG2";//by zeng
		    if (atomName.equalsIgnoreCase("1HG1"))
			atomName = "HG1";//by zeng
		    else if (atomName.equalsIgnoreCase("2HG1"))
			atomName = "HG1";//by zeng
		    else if (atomName.equalsIgnoreCase("3HG1"))
			atomName = "HG1";//by zeng
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("LYS") || resid.equalsIgnoreCase("PRO") ){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE2";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE3";
		    else if (atomName.equalsIgnoreCase("1HZ"))
			atomName = "HZ1";
		    else if (atomName.equalsIgnoreCase("2HZ"))
			atomName = "HZ2";
		    else if (atomName.equalsIgnoreCase("3HZ"))
			atomName = "HZ3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PHE") || resid.equalsIgnoreCase("TYR")
		     || resid.equalsIgnoreCase("HIS")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD1";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD2";
		    if (atomName.equalsIgnoreCase("1HE"))
			atomName = "HE1";
		    else if (atomName.equalsIgnoreCase("2HE"))
			atomName = "HE2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else if(resid.equalsIgnoreCase("PRO")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HB"))
			atomName = "HB2";
		    else if (atomName.equalsIgnoreCase("2HB"))
			atomName = "HB3";
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG"))
			atomName = "HG3";
		    if (atomName.equalsIgnoreCase("1HD"))
			atomName = "HD2";
		    else if (atomName.equalsIgnoreCase("2HD"))
			atomName = "HD3";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }else  if (resid.equalsIgnoreCase("THR")){
		for (int j=0; j<atoms.size(); j++){
		    atom = (Cartesian) atoms.elementAt(j);
		    atomName = atom.getAtom();
		    if (atomName.equalsIgnoreCase("1HG"))
			atomName = "HG1";
		    if (atomName.equalsIgnoreCase("1HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("2HG2"))
			atomName = "HG2";
		    else if (atomName.equalsIgnoreCase("3HG2"))
			atomName = "HG2";
		    coord = atom.getXYZ();
		    atomVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
		}
		pdbVecN.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
	    }
	}
	return pdbVecN;
    }
    
    /**
     * A temporary method for converting the residue name in the
     * RDC-derived PDB file, which are "ALA", to real Residue name, also
     * to build the conformation of proline and glycine residue.
     * 
     * @param pdbVec1 a pdb object with correct residue name,
     * which can be replaced with a protein sequence file
     * @param pdbVec2 a pdb object with all the residue name as "ALA"
     * 
     * @return a new Pdb object with the residue name changed and
     * conformation for Gly and Proline side chain.
     */
    public Vector residueNameUpdate_from_structure(Vector pdbVec1, Vector pdbVec2){
	Collections.sort(pdbVec2, new Pdb.PdbComparator());
	String atom  ="", resid;
	String atomName2 ="";
	Vector atoms   = new Vector();
	Vector atomVec = new Vector();
	Vector atomVecN = new Vector();
	Vector pdbVecN = new Vector();
	Matrix mat = new Matrix(3, 3);
	int no;

	double [] n  = new double[3];
	double [] nh = new double[3];
	double [] ca = new double[3];
	double [] ha = new double[3];
	double [] ha2 = new double[3];
	double [] cb = new double[3];
	double [] co = new double[3];
	double [] o  = new double[3];
	double [] nToNhVec = new double[3];
	double [] nToCaVec = new double[3];
	double [] ca2haVec = new double[3];
	Pdb pp = new Pdb();
	Pdb pp2 = new Pdb();
	int index = -1;
	Cartesian cc = new Cartesian();
	double proPhi = -65.0 * Const.cst;
	Matrix r2yInv = mat.rotationMat(proPhi, "-y"); 
	Matrix rg = new Matrix(3, 3);
	Matrix rgInv = new Matrix(3, 3);
	PhiPsi ff = new PhiPsi();
	Matrix matT = new Matrix(3, 3);
	sp3Carbon sp3 = new sp3Carbon();
	double[][] coordsCB = new double[3][3];
	double[][] coordsCG = new double[3][3];
	double[][] coordsCD = new double[3][3];
	int i, j;
	for (i=0; i<pdbVec1.size(); i++){
	    pp = (Pdb)pdbVec1.elementAt(i);
	    no = pp.getResidueNo();
	    resid = (pp.getResidue()).trim();
	    atoms = pp.getAtomVec();
	    index = Collections.binarySearch(pdbVec2, new Pdb(no), new Pdb.PdbComparator());
	    if (index > -1){
		pp2 = (Pdb)pdbVec2.elementAt(index);
		pp2.setResidue(resid);
		if (resid.equals("PRO")){
// 		    System.out.println(pp);
		    atomVec = pp2.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian) atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("CA"))
			    ca = cc.getXYZ();
			if (atom.equals("HA"))
			    ha = cc.getXYZ();
			else if(atom.equals("N"))
			    n = cc.getXYZ();
			else if(atom.equals("H"))
			    nh = cc.getXYZ();
			else if(atom.equals("C"))
			    co = cc.getXYZ();
			else if(atom.equals("CB"))
			    cb = cc.getXYZ();
			else if(atom.equals("O"))
			    o  = cc.getXYZ();
		    }
		    atomVecN = new Vector();
		    atomVecN.add(new Cartesian("N", n));
		    atomVecN.add(new Cartesian("CA", ca));
		    atomVecN.add(new Cartesian("HA", ha));
		    atomVecN.add(new Cartesian("CB", cb));
		    atomVecN.add(new Cartesian("C", co));
		    atomVecN.add(new Cartesian("O", o));
		    nToCaVec = internuclearVec(n, ca);
		    nToNhVec = internuclearVec(n, nh);
		    rg = ff.RgCal(nToNhVec, nToCaVec);
		    rgInv = rg.transpose();
		    matT = rgInv.times(Const.r1x9yInv.times(r2yInv.times(Const.rCBInv)));

// 		    System.out.println(" CB back ");
		    double [] coordCb = new double[]{0.0, Const.dCa2Cb, 0.0};
		    double [] caToCbVec = matT.times(coordCb);
		    coordCb   = ff.addCoords(ca, caToCbVec);
		    Matrix [] tetraCbMat = sp3.tetrahedral(coordCb, matT, Const.proCB);

		    Matrix matCG = tetraCbMat[1];
		    double [][] tetraCG = tetraCbMat[0].getArray(); 
		    atomVecN.add(new Cartesian("CG",  tetraCG[0]));
		    atomVecN.add(new Cartesian("1HB", tetraCG[1]));
		    atomVecN.add(new Cartesian("2HB", tetraCG[2]));
		    double [] coordCG = tetraCG[0];
		    Matrix [] tetraCgMat = sp3.tetrahedral(coordCG, matCG, Const.proCG);

		    Matrix matCD = tetraCgMat[1];
		    double[][] tetraCD = tetraCgMat[0].getArray(); 
		    atomVecN.add(new Cartesian("CD",  tetraCD[0]));
		    atomVecN.add(new Cartesian("1HG", tetraCD[1]));
		    atomVecN.add(new Cartesian("2HG", tetraCD[2]));
		    Matrix [] tetraCdMat = sp3.tetrahedral(tetraCD[0], matCD, Const.proCD);
		    double[][] tetraCD2 = tetraCdMat[0].getArray(); 
		    atomVecN.add(new Cartesian("1HD", tetraCD2[1]));
		    atomVecN.add(new Cartesian("2HD", tetraCD2[2]));
		    pdbVecN.add(new Pdb(no, resid, atomVecN));
		    atomVecN = new Vector();
		}else if(resid.equals("GLY")){
		    atomVec = pp2.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian) atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("CA"))
			    ca = cc.getXYZ();
			else if (atom.equals("HA"))
			    ha = cc.getXYZ();
			else if(atom.equals("N"))
			    n = cc.getXYZ();
			else if(atom.equals("H"))
			    nh = cc.getXYZ();
			else if(atom.equals("C"))
			    co = cc.getXYZ();
			else if(atom.equals("CB"))
			    cb = cc.getXYZ();
			else if(atom.equals("O"))
			    o  = cc.getXYZ();
		    }
		    atomVecN = new Vector();
		    atomVecN.add(new Cartesian("N", n));
		    atomVecN.add(new Cartesian("H", nh));
		    atomVecN.add(new Cartesian("CA", ca));
		    atomVecN.add(new Cartesian("1HA", ha));
		    ca2haVec = internuclearVec(ca, cb);
		    ha2 = new double[]{ca2haVec[0] * Const.dCA2HA / Const.dCA2CB, ca2haVec[1] * Const.dCA2HA / Const.dCA2CB, ca2haVec[2] * Const.dCA2HA / Const.dCA2CB};
		    ha2 = ff.addCoords(ca, ha2);
		    atomVecN.add(new Cartesian("2HA", ha2));
		    atomVecN.add(new Cartesian("C", co));
		    atomVecN.add(new Cartesian("O", o));
		    pdbVecN.add(new Pdb(no, resid, atomVecN));
		    atomVecN = new Vector();
		}
		else  pdbVecN.add(pp2);
	    }
	}
	return pdbVecN;
    }

    /**
     * get the residue name from protein sequence.
     * 
     * @param seqVec vector of residues, namely protein sequence
     * @param resNo residue no
     * 
     * @return  residue name
     */
    public String residueNameFromSeq(Vector seqVec, int resNo)
    {
    	for (int i=0; i<seqVec.size(); i++)
    	{
    		Assign asg = (Assign)seqVec.elementAt(i);
    		int no = asg.getResidueNo();
    		String resid = (asg.getResidueType()).trim();
    		if(resNo==no)
    			return resid;
    	}
    	return null;
    	
    }
    
    /**
     * work for bb only?
     * A temporary method for converting the residue name in the
     * RDC-derived PDB file, which are "ALA", to real Residue name, also
     * to build the conformation of proline and glycine residue.
     * 
     * @param seqVec vector of residues, namely protein sequence
     * @param pdbVec2 a pdb object with all the residue name as "ALA"
     * 
     * @return a new Pdb object with the residue name changed and
     * conformation for Gly and Proline side chain.
     */
    public Vector<Pdb> residueNameUpdate(Vector<Assign> seqVec, Vector<Pdb> pdbVec2)
    {
		Collections.sort(pdbVec2, new Pdb.PdbComparator());
		String atom  ="", resid;
		String atomName2 ="";
		Vector atoms   = new Vector();
		Vector atomVec = new Vector();
		Vector atomVecN = new Vector();
		Vector pdbVecN = new Vector();
		Matrix mat = new Matrix(3, 3);
		int no;

		double [] n  = new double[3];
		double [] nh = new double[3];
		double [] ca = new double[3];
		double [] ha = new double[3];
		double [] ha2 = new double[3];
		double [] cb = new double[3];
		double [] co = new double[3];
		double [] o  = new double[3];
		double [] nToNhVec = new double[3];
		double [] nToCaVec = new double[3];
		double [] ca2haVec = new double[3];
		Assign asg = new Assign();
		Pdb pp2 = new Pdb();
		int index = -1;
		Cartesian cc = new Cartesian();
		double proPhi = -65.0 * Const.cst;
		Matrix r2yInv = mat.rotationMat(proPhi, "-y"); 
		Matrix rg = new Matrix(3, 3);
		Matrix rgInv = new Matrix(3, 3);
		PhiPsi ff = new PhiPsi();
		Matrix matT = new Matrix(3, 3);
		sp3Carbon sp3 = new sp3Carbon();
		double[][] coordsCB = new double[3][3];
		double[][] coordsCG = new double[3][3];
		double[][] coordsCD = new double[3][3];
		int i, j;
		for (i=0; i<seqVec.size(); i++)
		{
		    asg = (Assign)seqVec.elementAt(i);
		    no = asg.getResidueNo();
		    resid = (asg.getResidueType()).trim();
		    //atoms = pp.getAtomVec();
		    index = Collections.binarySearch(pdbVec2, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1){
			pp2 = (Pdb)pdbVec2.elementAt(index);
			pp2.setResidue(resid);
			if (resid.equals("PRO") )
			{
				// 		    System.out.println(pp);
			    atomVec = pp2.getAtomVec();
			    for (j=0; j<atomVec.size(); j++)
			    {
					cc = (Cartesian) atomVec.elementAt(j);
					atom = cc.getAtom();
					if (atom.equals("CA"))
					    ca = cc.getXYZ();
					if (atom.equals("HA"))
					    ha = cc.getXYZ();
					else if(atom.equals("N"))
					    n = cc.getXYZ();
					else if(atom.equals("H"))
					    nh = cc.getXYZ();
					else if(atom.equals("C"))
					    co = cc.getXYZ();
					else if(atom.equals("CB"))
					    cb = cc.getXYZ();
					else if(atom.equals("O"))
					    o  = cc.getXYZ();
			    }//for (j=0; j<atomVec.size(); j++)
			    atomVecN = new Vector();
			    atomVecN.add(new Cartesian("N", n));
			    atomVecN.add(new Cartesian("CA", ca));
			    atomVecN.add(new Cartesian("HA", ha));
			    atomVecN.add(new Cartesian("CB", cb));
			    atomVecN.add(new Cartesian("C", co));
			    atomVecN.add(new Cartesian("O", o));
			    nToCaVec = internuclearVec(n, ca);
			    nToNhVec = internuclearVec(n, nh);
			    rg = ff.RgCal(nToNhVec, nToCaVec);
			    rgInv = rg.transpose();
			    matT = rgInv.times(Const.r1x9yInv.times(r2yInv.times(Const.rCBInv)));

			    // 		    System.out.println(" CB back ");
			    double [] coordCb = new double[]{0.0, Const.dCa2Cb, 0.0};
			    double [] caToCbVec = matT.times(coordCb);
			    coordCb   = ff.addCoords(ca, caToCbVec);
			    Matrix [] tetraCbMat = sp3.tetrahedral(coordCb, matT, Const.proCB);
	
			    Matrix matCG = tetraCbMat[1];
			    double [][] tetraCG = tetraCbMat[0].getArray(); 
			    atomVecN.add(new Cartesian("CG",  tetraCG[0]));
			    atomVecN.add(new Cartesian("1HB", tetraCG[1]));
			    atomVecN.add(new Cartesian("2HB", tetraCG[2]));
			    double [] coordCG = tetraCG[0];
			    Matrix [] tetraCgMat = sp3.tetrahedral(coordCG, matCG, Const.proCG);
	
			    Matrix matCD = tetraCgMat[1];
			    double[][] tetraCD = tetraCgMat[0].getArray(); 
			    atomVecN.add(new Cartesian("CD",  tetraCD[0]));
			    atomVecN.add(new Cartesian("1HG", tetraCD[1]));
			    atomVecN.add(new Cartesian("2HG", tetraCD[2]));
			    Matrix [] tetraCdMat = sp3.tetrahedral(tetraCD[0], matCD, Const.proCD);
			    double[][] tetraCD2 = tetraCdMat[0].getArray(); 
			    atomVecN.add(new Cartesian("1HD", tetraCD2[1]));
			    atomVecN.add(new Cartesian("2HD", tetraCD2[2]));
			    pdbVecN.add(new Pdb(no, resid, atomVecN));
			    atomVecN = new Vector();
			}else if(resid.equals("GLY"))
			{
		    atomVec = pp2.getAtomVec();
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian) atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("CA"))
			    ca = cc.getXYZ();
			else if (atom.equals("HA")|| atom.equals("HA1")||atom.equals("1HA"))
			    ha = cc.getXYZ();
			else if(atom.equals("N"))
			    n = cc.getXYZ();
			else if(atom.equals("H") || atom.equals("HN"))
			    nh = cc.getXYZ();
			else if(atom.equals("C"))
			    co = cc.getXYZ();
			else if(atom.equals("CB"))
			    cb = cc.getXYZ();
			else if(atom.equals("O"))
			    o  = cc.getXYZ();
		    }
		    atomVecN = new Vector();
		    atomVecN.add(new Cartesian("N", n));
		    atomVecN.add(new Cartesian("HN", nh));//changed from "H" to "HN"
		    atomVecN.add(new Cartesian("CA", ca));
		    atomVecN.add(new Cartesian("HA1", ha));
		    ca2haVec = internuclearVec(ca, cb);
		    ha2 = new double[]{ca2haVec[0] * Const.dCA2HA / Const.dCA2CB, ca2haVec[1] * Const.dCA2HA / Const.dCA2CB, ca2haVec[2] * Const.dCA2HA / Const.dCA2CB};
		    ha2 = ff.addCoords(ca, ha2);
		    atomVecN.add(new Cartesian("HA2", ha2));
		    atomVecN.add(new Cartesian("C", co));
		    atomVecN.add(new Cartesian("O", o));
		    pdbVecN.add(new Pdb(no, resid, atomVecN));
		    atomVecN = new Vector();
		}
		else  pdbVecN.add(pp2);
	    }
	}
	return pdbVecN;
    }
    
    /**
     * change residue name according to sequnce, without changing structure.
     * 
     * @param seqVec vector of residues, namely protein sequence
     * @param pdbVec2 a pdb object with all the residue name as "ALA"
     * 
     * @return a new Pdb object with the residue name changed and
     * conformation for Gly and Proline side chain.
     */
    public Vector<Pdb> residueNameUpdateNoStr(Vector<Assign> seqVec, Vector<Pdb> pdbVec2)
    {
		Collections.sort(pdbVec2, new Pdb.PdbComparator());
		String atom  ="", resid;
		String atomName2 ="";
		Vector atoms   = new Vector();
		Vector atomVec = new Vector();
		Vector atomVecN = new Vector();
		Vector pdbVecN = new Vector();
		Matrix mat = new Matrix(3, 3);
		int no;

		double [] n  = new double[3];
		double [] nh = new double[3];
		double [] ca = new double[3];
		double [] ha = new double[3];
		double [] ha2 = new double[3];
		double [] cb = new double[3];
		double [] co = new double[3];
		double [] o  = new double[3];
		double [] nToNhVec = new double[3];
		double [] nToCaVec = new double[3];
		double [] ca2haVec = new double[3];
		Assign asg = new Assign();
		Pdb pp2 = new Pdb();
		int index = -1;
		Cartesian cc = new Cartesian();
		double proPhi = -65.0 * Const.cst;
		Matrix r2yInv = mat.rotationMat(proPhi, "-y"); 
		Matrix rg = new Matrix(3, 3);
		Matrix rgInv = new Matrix(3, 3);
		PhiPsi ff = new PhiPsi();
		Matrix matT = new Matrix(3, 3);
		sp3Carbon sp3 = new sp3Carbon();
		double[][] coordsCB = new double[3][3];
		double[][] coordsCG = new double[3][3];
		double[][] coordsCD = new double[3][3];
		int i, j;
		for (i=0; i<seqVec.size(); i++)
		{
		    asg = (Assign)seqVec.elementAt(i);
		    no = asg.getResidueNo();
		    resid = (asg.getResidueType()).trim();
		    //atoms = pp.getAtomVec();
		    index = Collections.binarySearch(pdbVec2, new Pdb(no), new Pdb.PdbComparator());
		    if (index > -1){
			pp2 = (Pdb)pdbVec2.elementAt(index);
			pp2.setResidue(resid);
				 
			pdbVecN.add(pp2);
	    }
	}
	return pdbVecN;
    }
    
   
    /**
     * tell whether a structure is within the clusters of previous ensemble.
     * 
     * @param vecEnsemblePdb ensemble of structures to
     * @param vecPdb the pdb structure
     * @param rmsThreshold rmsd thresold to one structure in the ensemble
     * 
     * @return rmsd
     */
    public boolean isInPdbClusters(Vector vecEnsemblePdb, Vector vecPdb,  double rmsThreshold)
    {
    	boolean isInClusters=false;
    	for(int i=0;i< vecEnsemblePdb.size();i++)
    	{
    		Vector vecPdbTemp=(Vector)vecEnsemblePdb.elementAt(i);
    		double rmsd=compRmsdBetwTwoPdbs(vecPdbTemp,vecPdb,true);
    		if(rmsd< rmsThreshold)
    			isInClusters=true;    		
    	}//for(int i=0;i< vecEnsemblePdb.size();i++)
    	return isInClusters;
    }
    
    /**
     * tell whether a point is within the clusters of previous ensemble.
     * 
     * @param vecPoints ensemble of point to
     * @param cc the obj point
     * @param rmsThreshold rmsd thresold to one structure in the ensemble
     * 
     * @return rmsd
     */
    public boolean isInPointClusters(Vector vecPoints, Cartesian cc,  double rmsThreshold)
    {
    	boolean isInClusters=false;
    	Cartesian cc0=new Cartesian();
    	for(int i=0;i< vecPoints.size();i++)
    	{
    		Cartesian ccTemp=(Cartesian)vecPoints.elementAt(i);
    		double rmsd=cc.measureDist(ccTemp,cc);
    		if(rmsd< rmsThreshold)
    			isInClusters=true;    		
    	}//for(int i=0;i< vecEnsemblePdb.size();i++)
    	return isInClusters;
    }
    
    /**
     * HANA algorithm for the NOE assignment
     * and output the NOE table.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doReadPdbFragments(String src, String strOut, String strInput)throws JampackException
    {	    	 
    	boolean isDebug=true;
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// Read the input files
    	
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);   	 	
    	
    	String strInPdb="",strOutStructureName="",strStartRes="",strEndRes="";
    	
    	
    	for (i=0;i<paraVec.size();i++)
    	{    		
    		Map paraMap = paraVec.elementAt(i);    		
    		//get the input file name   
    		if(paraMap.containsKey("INPDBNAME"))
    			strInPdb =  (String)paraMap.get("INPDBNAME");
    		if(paraMap.containsKey("OUTSTRUCTURENAME"))
    			strOutStructureName  =  (String)paraMap.get("OUTSTRUCTURENAME");
    		if(paraMap.containsKey("STARTRESIDUES"))
    			strStartRes  =  (String)paraMap.get("STARTRESIDUES");  
    		if(paraMap.containsKey("ENDRESIDUES"))
    			strEndRes  =  (String)paraMap.get("ENDRESIDUES");     
    	}       	   	
       	//read input pdb
    	String userDirTemp = System.getProperty("user.dir");////
    	String pdbFile = userDirTemp+strInPdb;    	
    	Vector vecInPdb = pp.readPdb(pdbFile);
    	Vector pdbVecNew=new Vector();    	
    	
    	//read the fragements given the start and end residues NOs:
    	Vector vecStartRes=asg.getOrder(strStartRes);   
    	Vector vecEndRes=asg.getOrder(strEndRes); 
    	for(j=0;j<vecStartRes.size();j++)
		{
    		String strStartNo=(String) vecStartRes.elementAt(j);
    		int startResNo= Integer.parseInt(strStartNo);		
    		String strEndNo=(String) vecEndRes.elementAt(j);
    		int endResNo= Integer.parseInt(strEndNo);
    		for (i=0;i<vecInPdb.size();i++)
    		{
    			Pdb pdb=(Pdb)vecInPdb.elementAt(i);
    			int res_no=pdb.getResidueNo();
    			if( (res_no>=startResNo) && (res_no<=endResNo) )
    				pdbVecNew.add(pdb);
    		}//for (i=0;i<vecInPdb.size();i++)
    		
		}//for(j=0;j<vecStartRes.size();j++)

    	Collections.sort(pdbVecNew, new Pdb.PdbComparator());
    	
    	//output structures:
    	String fileName =userDirTemp+strOutStructureName;
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		out.println("REMARK :  extracted fragments of " + strInPdb);///to file 	
	        		   			
	    	pp.printToFile(pdbVecNew,fileName, out);

		    out.println("END");  
	    	out.close();
	    	
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
			System.out.println("IOException: the stack trace is:");
			e.printStackTrace();
		}            
		
       	long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Total time for extracting structure fragments:  "+ totalTime +" minutes");
    	
    }

    /**
     * Compute rmsd between two pdb structures with the same length and in the same coordinate system.
     * 
     * @param vecPdb1 the first structure
     * @param vecPdb2 the second structure
     * @param isHeavy whether compute rmsd between heavy atoms or all atoms
     * 
     * @return rmsd
     */
    public double compRmsdBetwTwoPdbs(Vector vecPdb1, Vector vecPdb2,  boolean isHeavy)
    {
    	int i,j;
		double sum=0.0;
		Vector atomVec1=new Vector();
    	Vector atomVec2=new Vector();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	String atomName1="";
    	String atomName2="";
    	int resNo1, resNo2;
    	
    	for (i=0;i<vecPdb1.size();i++)
    	{
    		Pdb pdb1=(Pdb)vecPdb1.elementAt(i);
    		resNo1=pdb1.getResidueNo();
    		atomVec1=pdb1.getAtomVec();
    		for (int k=0;k< atomVec1.size(); k++)
    		{    			
    		
    			ccA = (Cartesian) atomVec1.elementAt(k);
    			atomName1=ccA.getAtom();
    			coordA=ccA.getXYZ();
    		
    			if ( !(atomName1.equalsIgnoreCase("C") || atomName1.equalsIgnoreCase("CA")||atomName1.equalsIgnoreCase("N")||atomName1.equalsIgnoreCase("O"))   )
    				continue;
    			
	    		for (j=0;j<vecPdb2.size();j++)
	    		{
	    			Pdb pdb2=(Pdb)vecPdb2.elementAt(j);
	    			atomVec2=pdb2.getAtomVec();
	    			resNo2=pdb2.getResidueNo();
	    			
	    			for (int h=0;h<atomVec2.size();h++)
	        		{
	        			ccB=(Cartesian)atomVec2.elementAt(h);
	        			atomName2=ccB.getAtom();
	        			if ( !(atomName2.equalsIgnoreCase("C") || atomName2.equalsIgnoreCase("CA")||atomName2.equalsIgnoreCase("N")||atomName2.equalsIgnoreCase("O"))   )
	        				continue;
	        				        			
	        			if (atomName1.equalsIgnoreCase(atomName2) && (resNo2==resNo1))
	        			{
	        				coordB=ccB.getXYZ();
	        				double distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
	    						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
	    						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
	        				sum=sum+distance*distance;
	        				
	        			}
	        		}//for (int h=0;h<atomVec2.size();h++)	    			
	    			
	    		}//for (j=0;j<vecPdb2.size();j++)
    		}//for (int k=0;k< atomVec1.size(); k++)
    	}//for (i=0;i<vecPdb1.size();i++)
    	double rms=Math.sqrt(sum/(4*vecPdb1.size()));//here should be *4 because four atoms are consider in each residue...
    	//will be fixed next...
    	
    	return rms;

    }

    
    /**
     * Compute rmsd between a rotamer and corresponding side-chain in x-ray structure.
     * 
     * @param vecPdbXray xray backbone structure, each proton uses specific name, eg hg21...
     * @param vecRotam rotamer vector that keeps the same naming scheme as xray backbone
     * @param isHeavy whether compute rmsd between heavy atoms or all atoms
     * @param resNo the res no
     * @param rmsd the rmsd
     * 
     * @return rmsd
     */
    public boolean compRmsdRotXrayBB(Vector vecPdbXray, Vector vecRotam, int resNo,double [] rmsd, boolean isHeavy)
    {
    	int i,j;
		int ind = Collections.binarySearch(vecPdbXray, new Pdb(resNo), new Pdb.PdbComparator());
		if (ind < 0 )
		    return false;
	
	//The following is for working with X-ray structure without protons.
// 	else if(residueB.equalsIgnoreCase("ALA") && nucleusB.equalsIgnoreCase("HB"))
// 	    nucleusB = "HB";
// 	else if(residueB.equalsIgnoreCase("ILE") && nucleusB.equalsIgnoreCase("HG2"))
// 	    nucleusB = "CG2";
// 	else if(residueB.equalsIgnoreCase("ILE") && nucleusB.equalsIgnoreCase("HD1"))
// 	    nucleusB = "CD1";
// 	else if(residueB.equalsIgnoreCase("LEU") && nucleusB.equalsIgnoreCase("HD1"))
// 	    nucleusB = "CD1";
// 	else if(residueB.equalsIgnoreCase("LEU") && nucleusB.equalsIgnoreCase("HD2"))
// 	    nucleusB = "CD2";
// 	else if(residueB.equalsIgnoreCase("VAL") && nucleusB.equalsIgnoreCase("HG1"))
// 	    nucleusB = "CG1";
// 	else if(residueB.equalsIgnoreCase("VAL") && nucleusB.equalsIgnoreCase("HG2"))
// 	    nucleusB = "CG2";
// 	else if(residueB.equalsIgnoreCase("THR") && nucleusB.equalsIgnoreCase("HG2"))
// 	    nucleusB = "CG2";
		Pdb pdb=new Pdb();
		
		Pdb pdbA = (Pdb)vecPdbXray.elementAt(ind);
		
		Vector pdbAVec=new Vector();
		pdbAVec.add(pdbA);
	
		String atom = "";
		Vector atomVecA = pdbA.getAtomVec();
		
		Vector atomVec=new Vector();
		Cartesian cc = new Cartesian();
		double [] amide = new double[3];
		double [] amide1 = new double[3];
		double [] amide2 = new double[3];
		//double [] nh = new double[3];
		double [] ha = new double[3];
		double [] ca1 = new double[3];
		double [] ca2 = new double[3];
		double [] trans_vec = new double[3];

		PhiPsi ff = new PhiPsi();
		
		//find the pdb of rotamers at the 1st position
		Matrix rgA = Matrix.identity(3,3);
		
		for ( j=0; j<atomVecA.size(); j++)
		{ 
		    cc = (Cartesian)atomVecA.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
			amide = cc.getXYZ();
		    else if (atom.equals("CA"))
		    	ca1=cc.getXYZ();
		    else if (atom.equals("HA")||atom.equals("HA2") )
			ha = cc.getXYZ();		    
		}
			
		double [] CaToHaVecStr =  internuclearVec(ca1,  ha);
		double [] CaToNVecStr =  internuclearVec(ca1,  amide);//////
		Matrix rg = Matrix.identity(3,3);
		rg = ff.RgCal(CaToHaVecStr, CaToNVecStr);		
		//rgInv = rg.transpose();
		Vector pdbANew=pdb.newPdb(pdbAVec,rg);//the new coordinate of str after being rotated
		
		//new coords of xray backbone
		Vector pdbVecNew1=pdb.newPdb(vecPdbXray,rg);//the whole structure after the frist rotation
	
		Pdb pdbTemp = (Pdb)pdbANew.elementAt(0);
		atomVec=pdbTemp.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("CA"))
			ca1 = cc.getXYZ();		    
		} 
			
		Pdb pdbRotA=(Pdb)vecRotam.elementAt(0);
		atomVec=pdbRotA.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
		    	amide = cc.getXYZ();
		    else if (atom.equals("CA"))
		    	ca2=cc.getXYZ();
		    else if (atom.equals("HA")||atom.equals("HA2") )
		    	ha = cc.getXYZ();    
		}
		double [] CaToHaVecRot =  internuclearVec(ca2,  ha);
		double [] CaToNVecRot =  internuclearVec(ca2,  amide);//////
		rgA = ff.RgCal(CaToHaVecRot, CaToNVecRot);
		
		Vector rotamVecNewA_temp=pdb.newPdb(vecRotam,rgA);
		
		pdbTemp = (Pdb)rotamVecNewA_temp.elementAt(0);
		atomVec=pdbTemp.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("CA"))
			ca2 = cc.getXYZ();		    
		} 		
		
		//the translation vector after putting into the same frame
		trans_vec= internuclearVec(ca2,ca1);				
		
		//new coords of rotamers
		Vector<Pdb> rotamVecNewA=pdb.newPdbByTranslation(rotamVecNewA_temp,trans_vec);
		Pdb pdb1=rotamVecNewA.elementAt(0);
		Pdb pdb2=(Pdb)pdbVecNew1.elementAt(ind);
		
		//rmsd[0]=compResRmsdPairWise(pdb1,pdb2);
		if(isHeavy==true)
			rmsd[0]=compResRmsd(pdb1,pdb2);
		else
			rmsd[0]=compResRmsdPairWise(pdb1,pdb2);
		return true;

    }
    
    /**
     * Compute rmsd between two rotamers only for non-proton atoms, assume they are in the same coordinate system.
     * 
     * @param pdb1 rotmaer 1
     * @param pdb2 rotmaer 2
     * 
     * @return rmsd
     */
    public double compResRmsd(Pdb pdb1, Pdb pdb2)
    {
    	double rmsd=0.0;
    	int i,j;
    	double distance=0.0;
    	Vector atomVec1=new Vector();
    	Vector atomVec2=new Vector();
    	atomVec1=pdb1.getAtomVec();
    	atomVec2=pdb2.getAtomVec();
    	String resName=pdb1.getResidue();
    	
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	String atomName1="";
    	String atomName2="";
    	int counter=0;
    	for (i=0;i<atomVec1.size();i++)
    	{
    		ccA = (Cartesian) atomVec1.elementAt(i);
    		atomName1=ccA.getAtom();
    		if (atomName1.equalsIgnoreCase("HN"))
    			atomName1="H";
    		coordA=ccA.getXYZ();
    		
    		//the difference is here
    		if (atomName1.substring(0,1).equalsIgnoreCase("H"))
    			continue;
    		if (atomName1.equalsIgnoreCase("C")|| atomName1.equalsIgnoreCase("O"))
    			continue;
    		//if(atomName1.substring(0,1).equalsIgnoreCase("O")||atomName1.substring(0,1).equalsIgnoreCase("S"))
    			//continue;
    		//if(atomName1.substring(0,1).equalsIgnoreCase("N"))
    		//	continue;
    		//if((resName.equalsIgnoreCase("ASP") || resName.equalsIgnoreCase("ASN")) && atomName1.equalsIgnoreCase("CG"))
    		//	continue;
    		//if((resName.equalsIgnoreCase("GLU")|| resName.equalsIgnoreCase("GLN")) && atomName1.equalsIgnoreCase("CD"))
    		//	continue;
    		
    		
    		
    		//for debugging
    		/*if (atomName1.equalsIgnoreCase("NZ")||atomName1.equalsIgnoreCase("CE"))
    		{
    			System.out.print("over here...");
    			continue;
    		}*/
    		
    		for (j=0;j<atomVec2.size();j++)
    		{
    			ccB=(Cartesian)atomVec2.elementAt(j);
    			atomName2=ccB.getAtom();
    			if (atomName1.equalsIgnoreCase(atomName2))
    			{
    				coordB=ccB.getXYZ();
    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
    				rmsd=rmsd+distance*distance;
    				counter=counter+1;
    			}
    		}
    	}
    	
    	return Math.sqrt(rmsd/counter);
    	
    	
    }
    
    /**
     * Compute rmsd between two rotamers (for all atoms), assume they are in the same coordinate system.
     * 
     * @param pdb1 rotmaer 1
     * @param pdb2 rotmaer 2
     * 
     * @return rmsd
     */
    public double compResRmsdPairWise(Pdb pdb1, Pdb pdb2)
    {
    	double rmsd=0.0;
    	int i,j;
    	double distance=0.0;
    	Vector atomVec1=new Vector();
    	Vector atomVec2=new Vector();
    	atomVec1=pdb1.getAtomVec();
    	atomVec2=pdb2.getAtomVec();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	String atomName1="";
    	String atomName2="";
    	int counter=0;
    	for (i=0;i<atomVec1.size();i++)
    	{
    		ccA = (Cartesian) atomVec1.elementAt(i);
    		atomName1=ccA.getAtom();
    		if (atomName1.equalsIgnoreCase("HN"))
    			atomName1="H";
    		if (atomName1.equalsIgnoreCase("HN")||atomName1.equalsIgnoreCase("H"))
    			continue;
    		if (atomName1.equalsIgnoreCase("C")|| atomName1.equalsIgnoreCase("O"))
    			continue;
    		coordA=ccA.getXYZ();
    		for (j=0;j<atomVec2.size();j++)
    		{
    			ccB=(Cartesian)atomVec2.elementAt(j);
    			atomName2=ccB.getAtom();
    			if (atomName1.equalsIgnoreCase(atomName2))
    			{
    				coordB=ccB.getXYZ();
    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
    				rmsd=rmsd+distance*distance;
    				counter=counter+1;
    			}
    		}
    	}
    	
    	return Math.sqrt(rmsd/counter);
    	
    	
    }
    
    /**
     * Compute all distance patterns for two secondary structure elements, whose
     * coordinates are fixed.
     * 
     * @param pdbVecA the first sse
     * @param pdbVecB  the secondary sse
     * @param DistBound  the bound to compute noe distance beween two protons
     * 
     * @return the Hdist vector that stores all pairs of NOE distances (including from sc to sc)
     */
    public Vector compDistPatternForSSEPacking(final Vector pdbVecA, final Vector pdbVecB, final double DistBound)
    {       	
    	Vector vecHdist=new Vector();
    	int i,j,k,h;
    	Pdb pdbA =new Pdb();
    	Pdb pdbB =new Pdb();
    	Vector atomVecA=new Vector();
    	Vector atomVecB=new Vector();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	int resNoA, resNoB;
    	String resNameA="", resNameB="", nucleusA="", nucleusB="",atomA="",atomB="";
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	double distance=0.0;
    	
    	for (i=0;i<pdbVecA.size();i++)
    	{
    		pdbA = (Pdb)pdbVecA.elementAt(i);
    		atomVecA=pdbA.getAtomVec();
    		resNoA=pdbA.getResidueNo();
    		resNameA=pdbA.getResidue();
    		
    		for (j=0; j<atomVecA.size();j++)
    		{
    			ccA=(Cartesian)atomVecA.elementAt(j);
    			atomA=ccA.getAtom();
    			nucleusA=atomA.substring(0,1);
    			coordA=ccA.getXYZ();
    			if(atomA.equalsIgnoreCase("HN"))
    				atomA="H";
    			if(nucleusA.equalsIgnoreCase("H"))
    			{
    				for (k=0;k<pdbVecB.size();k++)
    				{
    					pdbB = (Pdb)pdbVecB.elementAt(k);
    		    		atomVecB=pdbB.getAtomVec();
    		    		resNoB=pdbB.getResidueNo();
    		    		resNameB=pdbB.getResidue();
    		    		
    		    		for (h=0; h<atomVecB.size();h++)
    		    		{
    		    			ccB=(Cartesian)atomVecB.elementAt(h);
    		    			atomB=ccB.getAtom();
    		    			nucleusB=atomB.substring(0,1);
    		    			coordB=ccB.getXYZ();
    		    			if(atomB.equalsIgnoreCase("HN"))
    		    				atomB="H";
    		    			if(nucleusB.equalsIgnoreCase("H"))
    		    			{
    		    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
    		    						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
    		    						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
    		    				/*
    		    				//new changes here
    		    				if( (resName.equalsIgnoreCase("ALA") && atomA.substring(0,2).equalsIgnoreCase("HB"))||
    		    						 (resName.equalsIgnoreCase("ILE") && atomA.substring(0,2).equalsIgnoreCase("HG2"))||
    		    						 (resName.equalsIgnoreCase("ILE") && atomA.substring(0,2).equalsIgnoreCase("HD1"))||
    		    						 (resName.equalsIgnoreCase("LEU") && atomA.substring(0,2).equalsIgnoreCase("HD1"))||
    		    						 (resName.equalsIgnoreCase("LEU") && atomA.substring(0,2).equalsIgnoreCase("HD2"))||
    		    						 (resName.equalsIgnoreCase("VAL") && atomA.substring(0,2).equalsIgnoreCase("HG1"))||
    		    						 (resName.equalsIgnoreCase("VAL") && atomA.substring(0,2).equalsIgnoreCase("HG2"))||
    		    						 (resName.equalsIgnoreCase("THR") && atomA.substring(0,2).equalsIgnoreCase("HG2")) )
    		    					dist_bound=DistBound+Const.methylCorrection;
    		    				else
    		    					dist_bound=DistBound;    	 */   	
    		    				if (distance < DistBound)
    		    				{
    		    					vecHdist.add(new Hdist(ccA,resNoA,resNameA,atomA, ccB,resNoB,resNameB,atomB,distance));
    		    					//System.out.println(atomA+ "----" + pdbB.getResidue()+pdbB.getResidueNo()+"-"+
    		    							//atomB + " "+"distance="+ distance );    		    						
    		    				}
    		    			}//if(nucleusB.equalsIgnoreCase("H"))    		    			
    		    			
    		    		}//for (h=0; h=atomVecB.size();h++)
    		    		
    				}//for (k=0;k<pdbVecB.size();k++)   			
    				
    			}//if(nucleusA.equalsIgnoreCase("H"))
    			
    		}//for (j=0; j=atomVecA.size();i++)    		
    		
    	}//for (i=0;i<pdbVecA.size();i++)
    	
    	//compute noe patterns for pdbVecB to pdbVecA 
    	for (i=0;i<pdbVecB.size();i++)
    	{
    		pdbA = (Pdb)pdbVecB.elementAt(i);
    		atomVecA=pdbA.getAtomVec();
    		resNoA=pdbA.getResidueNo();
    		resNameA=pdbA.getResidue();
    		
    		for (j=0; j<atomVecA.size();j++)
    		{
    			ccA=(Cartesian)atomVecA.elementAt(j);
    			atomA=ccA.getAtom();
    			nucleusA=atomA.substring(0,1);
    			coordA=ccA.getXYZ();
    			if(atomA.equalsIgnoreCase("HN"))
    				atomA="H";
    			if(nucleusA.equalsIgnoreCase("H"))
    			{
    				for (k=0;k<pdbVecA.size();k++)
    				{
    					pdbB = (Pdb)pdbVecA.elementAt(k);
    		    		atomVecB=pdbB.getAtomVec();
    		    		resNoB=pdbB.getResidueNo();
    		    		resNameB=pdbB.getResidue();
    		    		
    		    		for (h=0; h<atomVecB.size();h++)
    		    		{
    		    			ccB=(Cartesian)atomVecB.elementAt(h);
    		    			atomB=ccB.getAtom();
    		    			nucleusB=atomB.substring(0,1);
    		    			coordB=ccB.getXYZ();
    		    			if(atomB.equalsIgnoreCase("HN"))
    		    				atomB="H";
    		    			if(nucleusB.equalsIgnoreCase("H"))
    		    			{
    		    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
    		    						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
    		    						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
    		    				/*
    		    				//new changes here
    		    				if( (resName.equalsIgnoreCase("ALA") && atomA.substring(0,2).equalsIgnoreCase("HB"))||
    		    						 (resName.equalsIgnoreCase("ILE") && atomA.substring(0,2).equalsIgnoreCase("HG2"))||
    		    						 (resName.equalsIgnoreCase("ILE") && atomA.substring(0,2).equalsIgnoreCase("HD1"))||
    		    						 (resName.equalsIgnoreCase("LEU") && atomA.substring(0,2).equalsIgnoreCase("HD1"))||
    		    						 (resName.equalsIgnoreCase("LEU") && atomA.substring(0,2).equalsIgnoreCase("HD2"))||
    		    						 (resName.equalsIgnoreCase("VAL") && atomA.substring(0,2).equalsIgnoreCase("HG1"))||
    		    						 (resName.equalsIgnoreCase("VAL") && atomA.substring(0,2).equalsIgnoreCase("HG2"))||
    		    						 (resName.equalsIgnoreCase("THR") && atomA.substring(0,2).equalsIgnoreCase("HG2")) )
    		    					dist_bound=DistBound+Const.methylCorrection;
    		    				else
    		    					dist_bound=DistBound;    	 */   	
    		    				if (distance < DistBound)
    		    				{
    		    					vecHdist.add(new Hdist(ccA,resNoA,resNameA,atomA, ccB,resNoB,resNameB,atomB,distance));
    		    					//System.out.println(atomA+ "----" + pdbB.getResidue()+pdbB.getResidueNo()+"-"+
    		    							//atomB + " "+"distance="+ distance );    		    						
    		    				}
    		    			}//if(nucleusB.equalsIgnoreCase("H"))    		    			
    		    			
    		    		}//for (h=0; h=atomVecB.size();h++)
    		    		
    				}//for (k=0;k<pdbVecB.size();k++)   			
    				
    			}//if(nucleusA.equalsIgnoreCase("H"))
    			
    		}//for (j=0; j=atomVecA.size();i++)
    		
    		
    	}//for (i=0;i<pdbVecA.size();i++)   	
    	
    	return vecHdist;
			
    }
      
    /**
     * Compute all distance patterns for a pair of rotamers, distances are from each proton in the rotamer
     * to backbone protons including HA,HN, HB
     * rotamers are in the same coordinate system,includes both from A->B and from B->A.
     * 
     * @param DistBound  the bound to compute noe distance beween two protons
     * @param resNoA the res no a
     * @param resNoB the res no b
     * @param pdbA the pdb a
     * @param pdbB the pdb b
     * 
     * @return the Hdist vector that stores all pairs of NOE distances from rotamer proton to backbone protons
     */
    public Vector compDistPatternFromPairRotams(int resNoA, int resNoB,final Pdb pdbA, final Pdb pdbB, double DistBound)
    {    	
    	Vector atomVecA = pdbA.getAtomVec();
    	Vector atomVecB=pdbB.getAtomVec();
    	String firstResName=pdbA.getResidue();
    	String secondResName=pdbB.getResidue();
    	Vector vecHdist=new Vector();
    	
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	String atomA="",atomB="";
    	String nucleusA="", nucleusB="";
    	double distance=0.0;
    	
    	//1.NOE distance from A to B
    	for (int i = 0; i < atomVecA.size(); i++)
    	{
    		ccA = (Cartesian) atomVecA.elementAt(i);
    	    atomA = ccA.getAtom();
    	    nucleusA=atomA.substring(0,1);      
    	    
    	    if (nucleusA.equalsIgnoreCase("H") )
    	    {    	    	
    	    	coordA = ccA.getXYZ();//coordinates of the first atom   	    		
    	    		
	    		for (int k=0; k< atomVecB.size();k++)
	    		{
	    			ccB=(Cartesian) atomVecB.elementAt(k);
	    			atomB=ccB.getAtom();
	    			nucleusB=atomB.substring(0,1);// .substring(0,2);
	    			coordB=ccB.getXYZ();
    	    			
    	    		if (nucleusB.equalsIgnoreCase("H") )
    	    		{
    	    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
    	    						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
    	    						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));    	    									
    	    					
    	    				if (distance < DistBound)
    	    					vecHdist.add(new Hdist(ccA,resNoA,firstResName,atomA, ccB,resNoB,secondResName,atomB,distance));
     	    		}
    	    		
    	    		
    	    	}//for (int k=0; k< atomVecB.size();k++)  	    	
    	    	
    	    }//if (nucleusA.equalsIgnoreCase("H"))
    	}//for (int i = 0; i < atomVecA.size(); i++)
    	
    	//2.NOE distance from B to A
    	for (int i = 0; i < atomVecB.size(); i++)
    	{
    		ccB = (Cartesian) atomVecB.elementAt(i);
    	    atomB = ccB.getAtom();
    	    nucleusB=atomB.substring(0,1);      
    	    
    	    if (nucleusB.equalsIgnoreCase("H") )
    	    {    	    	
    	    	coordB = ccA.getXYZ();//coordinates of the first atom   	    		
    	    		
	    		for (int k=0; k< atomVecA.size();k++)
	    		{
	    			ccA=(Cartesian) atomVecA.elementAt(k);
	    			atomA=ccA.getAtom();
	    			nucleusA=atomA.substring(0,1);
	    			coordA=ccA.getXYZ();
    	    			
    	    		if (nucleusA.equalsIgnoreCase("H") )
    	    		{
    	    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
    	    						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
    	    						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));    	    									
    	    					
    	    				if (distance < DistBound)
    	    					vecHdist.add(new Hdist(ccB,resNoB,secondResName,atomB,ccA,resNoA,firstResName,atomA, distance));
     	    		}   	    		
    	    		
    	    	}//for (int k=0; k< atomVecB.size();k++)  	    	
    	    	
    	    }//if (nucleusA.equalsIgnoreCase("H"))
    	}//for (int i = 0; i < atomVecA.size(); i++)
    	return vecHdist;
    }
    
    /**
     * Compute all distance patterns for a rotamer, distances are from each proton in the rotamer
     * to backbone protons including HA,HN, HB.
     * 
     * @param rotamVec the rotamer vector in pdb format, note: only *one* rotamer
     * @param resNo residue No of the rotamer
     * @param pdbVec  the pdb vector of the whole protein backbone
     * @param DistBound  the bound to compute noe distance beween two protons
     * 
     * @return the Hdist vector that stores all pairs of NOE distances from rotamer proton to backbone protons
     */
    public Vector<Hdist> compDistPatternForRotam(final Vector rotamVec, int resNo, final Vector pdbVec, final double DistBound)
    {    	
    	//we first translate the the rotamer to be consistent with the protein coordinate system
    	Vector pdbVecN=pdbVec;    	
    	Vector pdbRotamNewA=rotamVec;
    	int noA=resNo;
    	int i,j,k;
    	
    	int ind1 = Collections.binarySearch(pdbVecN, new Pdb(noA), new Pdb.PdbComparator());
    	
    	Pdb pdbA = (Pdb)pdbVecN.elementAt(ind1);
    	Vector pdbAVec=new Vector();
    	pdbAVec.add(pdbA);
    	Pdb pdb=new Pdb();
    	PhiPsi ff = new PhiPsi();
    	
    	String atom = "";
    	Vector atomVecA = pdbA.getAtomVec();
    	String resName=pdbA.getResidue();
    		
    	Vector atomVec=new Vector();
    	Cartesian cc = new Cartesian();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	double [] amide = new double[3];
    	double [] amide1 = new double[3];
    	double [] amide2 = new double[3];
    	double [] nh = new double[3];
    	//double [] nh = new double[3];
    	double [] ha = new double[3];
    	double [] ca1 = new double[3];
    	double [] ca_temp = new double[3];
    	double [] ca2 = new double[3];
    	double [] trans_vec = new double[3];
    	
    	Matrix rgA = Matrix.identity(3,3);
    	
    	for ( j=0; j<atomVecA.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVecA.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide = cc.getXYZ();
    	    else if (atom.equals("CA"))
    	    	ca1=cc.getXYZ();
    	    else if (atom.equals("HA")||atom.equals("HA2")||atom.equals("HA1")||atom.equals("HA3"))
    		ha = cc.getXYZ();		    
    	}
    		
    	double [] CaToHaVecStr =  pdb.internuclearVec(ca1,  ha);
    	double [] CaToNVecStr =  pdb.internuclearVec(ca1, amide);//////
    	Matrix rg = Matrix.identity(3,3);
    	rg = ff.RgCal(CaToHaVecStr, CaToNVecStr);	
    	
    	Vector pdbANew=pdb.newPdb(pdbAVec,rg);//the new coordinate of str after being rotated
    	    	
    	//can be used for the next calcualtion
    	Vector pdbVecNew1=pdb.newPdb(pdbVecN,rg);//the whole structure after the frist rotation

    	Pdb pdbTemp = (Pdb)pdbANew.elementAt(0);
    	atomVec=pdbTemp.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("CA"))
    	    	ca_temp = cc.getXYZ();	//for translation	    
    	} 
    	
    	Pdb pdbRotA=(Pdb)pdbRotamNewA.elementAt(0);
    	atomVec=pdbRotA.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide = cc.getXYZ();
    	    else if (atom.equals("CA"))
    	    	ca2=cc.getXYZ();
    	    else if (atom.equals("HA") ||atom.equals("HA2"))
    		ha = cc.getXYZ();    
    	}
    	double [] CaToHaVecRot =  pdb.internuclearVec(ca2,  ha);
    	double [] CaToNVecRot =  pdb.internuclearVec(ca2, amide);//////
    	rgA = ff.RgCal(CaToHaVecRot, CaToNVecRot);
    	
    	Vector rotamVecNewA_temp=pdb.newPdb(pdbRotamNewA,rgA);
    	
    	pdbTemp = (Pdb)rotamVecNewA_temp.elementAt(0);
    	atomVec=pdbTemp.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("CA"))
    		ca2 = cc.getXYZ();		    
    	} 		
    	
    	//the translation vector after putting into the same frame
    	trans_vec= pdb.internuclearVec(ca2,ca_temp);				
    	
    	Vector<Pdb> rotamVecNewA=pdb.newPdbByTranslation(rotamVecNewA_temp,trans_vec);

    	//  here we only have one rotamer in the vector...
    	pdbA=(Pdb)rotamVecNewA.elementAt(0);
    	
    	Pdb pp=new Pdb();
    
    	pdbVecN=pdbVecNew1;
    	  
    	////report rmsd between original residue and rotmaer 
    	//using x-ray backbone
    	ind1 = Collections.binarySearch(pdbVecN, new Pdb(noA), new Pdb.PdbComparator());
    	Pdb pdb222 = (Pdb)pdbVecN.elementAt(ind1);
    	double rmsd=pdb.compResRmsd(pdbA,pdb222);///only heavy atoms
    	     	
        atomVecA = pdbA.getAtomVec();
        int firstResNo=resNo; //pdbA.getResidueNo();
        String firstResName=pdbA.getResidue();
        int secondResNo=0;
        String secondResName="";
        
    	String atomA = "";
    	String atomB = "";
    	Pdb pdbB;
    	Vector atomVecB;
    	    	
        cc = new Cartesian();
    	String nucleusA="";
    	String nucleusB="";
    	double distance=0.0;
    	
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	
       	Vector vecHdist=new Vector();
    	
    	
    	for (i = 0; i < atomVecA.size(); i++)
    	{
    		ccA = (Cartesian) atomVecA.elementAt(i);
    	    atomA = ccA.getAtom();
    	    nucleusA=atomA.substring(0,1);
    	    
    	    ////new for only considerring protons connected to carbon
    	    Peak pk=new Peak();
    	    String HeavyA="";
    	    if (nucleusA.equalsIgnoreCase("H"))
    	    	HeavyA=pk.GetHeavyAtomFromProton(resName,atomA);
    	    
    	    
    	    if (atomA.equalsIgnoreCase("HN"))
    	    	atomA="H";
    	       
    	    
    	    if (atomA.equalsIgnoreCase("HN")|| atomA.equalsIgnoreCase("HA")||atomA.equalsIgnoreCase("H")
    	    		||atomA.equalsIgnoreCase("HA1")||atomA.equalsIgnoreCase("HA2"))
    	    	continue; //the backbone protons in the rotamer is not useful for positioning
    	   	
    	    if (nucleusA.equalsIgnoreCase("H") )
    	    {
    	    	
    	    	coordA = ccA.getXYZ();//coordinates of the first atom
    	    	for (j=0;j<pdbVecN.size();j++)
    	    	{
    	    		pdbB = (Pdb)pdbVecN.elementAt(j);
    	    		atomVecB=pdbB.getAtomVec();
    	    		secondResNo=pdbB.getResidueNo();
    	    		secondResName=pdbB.getResidue();
    	    		for (k=0; k< atomVecB.size();k++)
    	    		{
    	    			ccB=(Cartesian) atomVecB.elementAt(k);
    	    			atomB=ccB.getAtom();
    	    			nucleusB=atomB.substring(0,1);// .substring(0,2);
    	    			coordB=ccB.getXYZ();
    	    			if(atomB.equalsIgnoreCase("HN"))
    	    				atomB="H";
    	    			if ( (resNo==secondResNo && atomB.substring(0,1).equalsIgnoreCase("H")) || (  atomB.equalsIgnoreCase("H")||atomB.equalsIgnoreCase("HN")||atomB.equalsIgnoreCase("HA") ) )
    	    			{
    	    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
    	    						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
    	    						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
    	    				//double dist_bound=0.0;
    	    				/*
    	    				//new changes here
    	    				if( (resName.equalsIgnoreCase("ALA") && atomA.substring(0,2).equalsIgnoreCase("HB"))||
    	    						 (resName.equalsIgnoreCase("ILE") && atomA.substring(0,2).equalsIgnoreCase("HG2"))||
    	    						 (resName.equalsIgnoreCase("ILE") && atomA.substring(0,2).equalsIgnoreCase("HD1"))||
    	    						 (resName.equalsIgnoreCase("LEU") && atomA.substring(0,2).equalsIgnoreCase("HD1"))||
    	    						 (resName.equalsIgnoreCase("LEU") && atomA.substring(0,2).equalsIgnoreCase("HD2"))||
    	    						 (resName.equalsIgnoreCase("VAL") && atomA.substring(0,2).equalsIgnoreCase("HG1"))||
    	    						 (resName.equalsIgnoreCase("VAL") && atomA.substring(0,2).equalsIgnoreCase("HG2"))||
    	    						 (resName.equalsIgnoreCase("THR") && atomA.substring(0,2).equalsIgnoreCase("HG2")) )
    	    					dist_bound=DistBound+Const.methylCorrection;
    	    				else
    	    					dist_bound=DistBound;    	 */   					
    	    					
    	    				if (distance < DistBound)
    	    				{
    	    					vecHdist.add(new Hdist(ccA,firstResNo,firstResName,atomA, ccB,secondResNo,secondResName,atomB,distance));
    	    					//added on 9/28/08:
    	    					vecHdist.add(new Hdist(ccB,secondResNo,secondResName,atomB,ccA,firstResNo,firstResName,atomA,distance));
    	    					
    	    					//System.out.println(atomA+ "----" + pdbB.getResidue()+pdbB.getResidueNo()+"-"+
    	    							//atomB + " "+"distance="+ distance );
    	    						
    	    				}
    	    			}
    	    		}
    	    		
    	    		
    	    	}//for (j=0;j<pdbVecN.size();j++)    	    	
    	    	
    	    }//if (nucleusA.equalsIgnoreCase("H"))
    	}
    	
    	return vecHdist;
    	
    }
   
    /**
     * Compute all distance patterns for a rotamer, distances are from each proton in the rotamer
     * to backbone protons including HA,HN, HB.
     * 
     * @param rotamVec the rotamer vector in pdb format, note: only *one* rotamer
     * @param resNo residue No of the rotamer
     * @param pdbVec  the pdb vector of the whole protein backbone
     * @param DistBound  the bound to compute noe distance beween two protons
     * 
     * @return the Hdist vector that stores all pairs of NOE distances from rotamer proton to backbone protons
     */
    public Vector compDistPatternForRotam_old(final Vector rotamVec, int resNo, final Vector pdbVec, final double DistBound)
    {
    	
    	//we first translate the the rotamer to be consistent with the protein coordinate system
    	Vector pdbVecN=pdbVec;    	
    	Vector pdbRotamNewA=rotamVec;
    	int noA=resNo;
    	int i,j,k;
    	
    	int ind1 = Collections.binarySearch(pdbVecN, new Pdb(noA), new Pdb.PdbComparator());
    	
    	Pdb pdbA = (Pdb)pdbVecN.elementAt(ind1);
    	Vector pdbAVec=new Vector();
    	pdbAVec.add(pdbA);
    	Pdb pdb=new Pdb();
    	PhiPsi ff = new PhiPsi();
    	
    	String atom = "";
    	Vector atomVecA = pdbA.getAtomVec();	
    		
    	Vector atomVec=new Vector();
    	Cartesian cc = new Cartesian();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	double [] amide = new double[3];
    	double [] amide1 = new double[3];
    	double [] amide2 = new double[3];
    	double [] nh = new double[3];
    	double [] ca1 = new double[3];
    	double [] ca2 = new double[3];
    	double [] trans_vec = new double[3];
    	
    	Matrix rgA = Matrix.identity(3,3);
    	
    	for ( j=0; j<atomVecA.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVecA.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide = cc.getXYZ();
    	    else if (atom.equals("CA"))
    	    	ca1=cc.getXYZ();
    	    else if (atom.equals("H") || atom.equals("NH"))
    		nh = cc.getXYZ();		    
    	}
    		
    	double [] nToNHVecStr =  pdb.internuclearVec(amide,  nh);
    	double [] nToCAVecStr =  pdb.internuclearVec(amide,  ca1);//////
    	Matrix rg = Matrix.identity(3,3);
    	rg = ff.RgCal(nToNHVecStr, nToCAVecStr);	
    	
    	Vector pdbANew=pdb.newPdb(pdbAVec,rg);//the new coordinate of str after being rotated
    	
    	
    	//can be used for the next calcualtion
    	Vector pdbVecNew1=pdb.newPdb(pdbVecN,rg);//the whole structure after the frist rotation

    	Pdb pdbTemp = (Pdb)pdbANew.elementAt(0);
    	atomVec=pdbTemp.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide1 = cc.getXYZ();		    
    	} 

    	
    	Pdb pdbRotA=(Pdb)pdbRotamNewA.elementAt(0);
    	atomVec=pdbRotA.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide = cc.getXYZ();
    	    else if (atom.equals("CA"))
    	    	ca2=cc.getXYZ();
    	    else if (atom.equals("H") || atom.equals("NH"))
    		nh = cc.getXYZ();    
    	}
    	double [] nToNHVecRot =  pdb.internuclearVec(amide,  nh);
    	double [] nToCAVecRot =  pdb.internuclearVec(amide,  ca2);//////
    	rgA = ff.RgCal(nToNHVecRot, nToCAVecRot);
    	
    	Vector rotamVecNewA_temp=pdb.newPdb(pdbRotamNewA,rgA);
    	
    	pdbTemp = (Pdb)rotamVecNewA_temp.elementAt(0);
    	atomVec=pdbTemp.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide2 = cc.getXYZ();		    
    	} 		
    	
    	//the translation vector after putting into the same frame
    	trans_vec= pdb.internuclearVec(amide2,amide1);				
    	
    	Vector<Pdb> rotamVecNewA=pdb.newPdbByTranslation(rotamVecNewA_temp,trans_vec);

//    	we can change the index here for different rotamer positions
    	pdbA=(Pdb)rotamVecNewA.elementAt(0);//////////////////////be careful here...
    	    	
    	pdbVecN=pdbVecNew1;
    	   
    	////report rmsd between original residue and rotmaer 
    	ind1 = Collections.binarySearch(pdbVecN, new Pdb(noA), new Pdb.PdbComparator());
    	Pdb pdb222 = (Pdb)pdbVecN.elementAt(ind1);
    	double rmsd=pdb.compResRmsd(pdbA,pdb222);
    	System.out.print("The rmsd between original residue and rotmaer is : "+ rmsd);
    	
    	
        atomVecA = pdbA.getAtomVec();
        int firstResNo=pdbA.getResidueNo();
        String firstResName=pdbA.getResidue();
        int secondResNo=0;
        String secondResName="";
        
    	String atomA = "";
    	String atomB = "";
    	Pdb pdbB;
    	Vector atomVecB;
    	    	
        cc = new Cartesian();
    	String nucleusA="";
    	String nucleusB="";
    	double distance=0.0;
    	
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	
    	System.out.println(" The following shows HNs that intereact with all protons from: " +firstResName+"-"+firstResNo);
    	
    	Vector vecHdist=new Vector();
    	
    	
    	for (i = 0; i < atomVecA.size(); i++)
    	{
    		ccA = (Cartesian) atomVecA.elementAt(i);
    	    atomA = ccA.getAtom();
    	    nucleusA=atomA.substring(0,1);
    	    
    	    if (nucleusA.equalsIgnoreCase("H"))
    	    {
    	    	System.out.println("========================");
    	    	System.out.println("The proton in "+ firstResName+"-"+firstResNo +" is: "+ atomA);
    	    	
    	    	coordA = ccA.getXYZ();//coordinates of the first atom
    	    	for (j=0;j<pdbVecN.size();j++)
    	    	{
    	    		pdbB = (Pdb)pdbVecN.elementAt(j);
    	    		atomVecB=pdbB.getAtomVec();
    	    		secondResNo=pdbB.getResidueNo();
    	    		secondResName=pdbB.getResidue();
    	    		
    	    		for (k=0; k< atomVecB.size();k++)
    	    		{
    	    			ccB=(Cartesian) atomVecB.elementAt(k);
    	    			atomB=ccB.getAtom();
    	    			nucleusB=atomB.substring(0,1);// .substring(0,2);
    	    			coordB=ccB.getXYZ();
    	    			if (atomB.equalsIgnoreCase("H")||atomB.equalsIgnoreCase("HN")||atomB.equalsIgnoreCase("HA") )
    	    			{
    	    				distance= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
    	    						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
    	    						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
    	    				if (distance < DistBound)
    	    				{
    	    					vecHdist.add(new Hdist(ccA,firstResNo,firstResName,atomA, ccB,secondResNo,secondResName,atomB,distance));
    	    					System.out.println(" The H1 is:" + pdbB.getResidue()+pdbB.getResidueNo()+"-"+
    	    							atomB + " "+"distance="+ distance );
    	    				}
    	    			}
    	    		}
    	    		
    	    		
    	    	}//for (j=0;j<pdbVecN.size();j++)    	    	
    	    	
    	    }//if (nucleusA.equalsIgnoreCase("H"))
    	}
    	
    	return vecHdist;
    	
    }
   

    

    
    
    /**
     * Compute the NOE distance between two nuclei (A, B) in a structure specified by pdbVec.
     * 
     * @param pdbVec the PDB structure
     * @param noA     the residue NO of nucleus A
     * @param residueA  the residue type of nucleus A
     * @param nucleusA  the name of the nucleus of nucleus A
     * @param noB     the residue NO of nucleus A
     * @param residueB  the residue type of nucleus A
     * @param nucleusB  the name of the nucleus of nucleus A
     * @param distance  save the back-computed NOE distance
     * 
     * @return true if such a distance can be computed in the structure, false if there is no
     * such nuclei in the structure.
     */
    public boolean measureDisXray (final Vector pdbVec, int noA, String residueA, String nucleusA, int noB, 
				   String residueB, String nucleusB, double[] distance){
	int ind1 = Collections.binarySearch(pdbVec, new Pdb(noA), new Pdb.PdbComparator());
	int ind2 = Collections.binarySearch(pdbVec, new Pdb(noB), new Pdb.PdbComparator());
	if (ind1 < 0 || ind2 < 0)
	    return false;
	if (nucleusA.equalsIgnoreCase("HN")) //We need to convert all the CS name to PDB name
	    nucleusA = "H";
	if (nucleusB.equalsIgnoreCase("HN"))
	    nucleusB = "H";
	else if(residueB.equalsIgnoreCase("ALA") && nucleusB.equalsIgnoreCase("HB"))
	    nucleusB = "CB";
	else if(residueB.equalsIgnoreCase("ILE") && nucleusB.equalsIgnoreCase("HG2"))
	    nucleusB = "CG2";
	else if(residueB.equalsIgnoreCase("ILE") && nucleusB.equalsIgnoreCase("HD1"))
	    nucleusB = "CD1";
	else if(residueB.equalsIgnoreCase("LEU") && nucleusB.equalsIgnoreCase("HD1"))
	    nucleusB = "CD1";
	else if(residueB.equalsIgnoreCase("LEU") && nucleusB.equalsIgnoreCase("HD2"))
	    nucleusB = "CD2";
	else if(residueB.equalsIgnoreCase("VAL") && nucleusB.equalsIgnoreCase("HG1"))
	    nucleusB = "CG1";
	else if(residueB.equalsIgnoreCase("VAL") && nucleusB.equalsIgnoreCase("HG2"))
	    nucleusB = "CG2";
	else if(residueB.equalsIgnoreCase("THR") && nucleusB.equalsIgnoreCase("HG2"))
	    nucleusB = "CG2";

	Pdb pdbA = (Pdb)pdbVec.elementAt(ind1);
	Pdb pdbB = (Pdb)pdbVec.elementAt(ind2);
	String atom = "";
	Vector atomVecA = pdbA.getAtomVec();
	Vector atomVecB = pdbB.getAtomVec();
	Cartesian cc = new Cartesian();
	double [] coordA = {0.0, 0.0, 0.0};
	double [] coordB = {0.0, 0.0, 0.0};
	int i = 0;
	boolean found = false;
	for (i = 0; i < atomVecA.size(); i++){
	    cc = (Cartesian) atomVecA.elementAt(i);
	    atom = cc.getAtom();
	    if (atom.equalsIgnoreCase(nucleusA)){
		coordA = cc.getXYZ();
		found = true;
	    }
	}
	if ( !found ){

	    return found;
	}
	found = false;
	for (i = 0; i < atomVecB.size(); i++){
	    cc = (Cartesian) atomVecB.elementAt(i);
	    atom = cc.getAtom();
	    if (atom.equalsIgnoreCase(nucleusB)){
		coordB = cc.getXYZ();
		found = true;
	    }
	}
	if ( !found ){

	    return found;
	}
	distance[0] =  Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
	    + (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
	    + (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
	return true;
    }

    /**
     * measure the packing distances between two sses
     * all rotamers are put into the sse backbones
     * The number themselves are set rather arbitrariry. Reaaly need
     * search through PDB to compute them.
     * 
     * @param noA     the residue NO of nucleus A
     * @param nucleusA  the name of the nucleus of residue A
     * @param noB     the residue NO of nucleus A
     * @param nucleusB  the name of the nucleus of residue B
     * @param lower lower bound of NOE constraint
     * @param upper upper bound of NOE constraint
     * @param distance  save distance deviation
     * @param pdbVecA the pdb vec a
     * @param pdbVecB the pdb vec b
     * @param disRMS the dis rms
     * 
     * @return true if such a distance can be computed in the structure, false if there is no
     * such nuclei in the structure.
     */
    public boolean measurePackDisAllRotamers(final Vector pdbVecA, final Vector pdbVecB, int noA, String nucleusA, int noB, 
			     String nucleusB, double lower, double upper, double[] disRMS, double[] distance)
    {
	int ind1 = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator());
	int ind2 = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator());
	if (ind1 < 0 || ind2 < 0)
	    return false;
	
	if (nucleusA.equalsIgnoreCase("H")) //We need to convert all the CS name to PDB name
	    nucleusA = "HN";
	if (nucleusB.equalsIgnoreCase("H"))
	    nucleusB = "HN";	
	

	Pdb pdbA = (Pdb)pdbVecA.elementAt(ind1);
	Pdb pdbB = (Pdb)pdbVecB.elementAt(ind2);
	String atomA = "", atomB="";
	Vector atomVecA = pdbA.getAtomVec();
	Vector atomVecB = pdbB.getAtomVec();
	Cartesian cc = new Cartesian();
	double [] coordA = {0.0, 0.0, 0.0};
	double [] coordB = {0.0, 0.0, 0.0};
	int i = 0;
	boolean foundA = false, foundB = false, found=false;
	Map fromCBMap = new TreeMap();
	Residue resid = new Residue();
	double disDifMin = 1000.0;
	double distanceMin=10000.0;
	double disDif=0.0;
	
	for (i = 0; i < atomVecA.size(); i++)
	{ //after backbone structure being computed, this always return true
	    cc = (Cartesian) atomVecA.elementAt(i);
	    atomA = cc.getAtom();
	    foundA = false;
	    
	    
	    if (atomA.equalsIgnoreCase(nucleusA))
	    {
	    	coordA = cc.getXYZ();
	    	foundA = true;
	    }	    
	    if(nucleusA.length()>=2)
	    {
		    if(nucleusA.substring(0,1).equalsIgnoreCase("Q") && (!nucleusA.substring(1,2).equalsIgnoreCase("Q")) && (atomA.length()>=2) )
		    {
		    	String temp="H"+nucleusA.substring(1,nucleusA.length());
		    	if(atomA.length()>=temp.length())
		    	{
			    	if(atomA.substring(0,temp.length()).equalsIgnoreCase(temp))
			    	{
			    		foundA= true;	
			    		coordA = cc.getXYZ();
			    	}//if(atomA.substring(0,temp.length()).equalsIgnoreCase(temp))
		    	}//if(atomA.length()>=temp.length())
		    }  //if(nucleusA.substring(0,1).equalsIgnoreCase("Q") && (!nucleusA.substring(1,2).equalsIgnoreCase("Q")) && (atomA.length()>=2) )
		    
		    if(nucleusA.substring(0,2).equalsIgnoreCase("QQ") && (atomA.length()>=2) )
		    {
		    	String temp="H"+nucleusA.substring(2,nucleusA.length());
		    	if (atomA.length()>=temp.length())
		    	{
			    	if(atomA.substring(0,temp.length()).equalsIgnoreCase(temp))
			    	{
			    		foundA= true;	
			    		coordA = cc.getXYZ();
			    	}//if(atomA.substring(0,temp.length()).equalsIgnoreCase(temp))
		    	}//if (atomA.length()>=temp.length())
		    }  //if(nucleusA.substring(0,2).equalsIgnoreCase("QQ") && (atomA.length()>=2) )
	    }//if(nucleusA.length()>=2)
	    
	    if ( !foundA )
		    continue;
	    
		foundB = false;
		double disFromCB = 0.0;
		double dis2CB = 0.0;
		for (int j = 0; j < atomVecB.size(); j++)
		{
			disDif=0.0;
		    cc = (Cartesian) atomVecB.elementAt(j);
		    atomB = cc.getAtom();
		    if (atomB.equalsIgnoreCase(nucleusB))
		    {
		    	foundB = true;
		    	coordB = cc.getXYZ();
		    	double dist_temp= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
		    	if (dist_temp > lower && dist_temp < upper)
				    disDif = 0.0;
				else if(dist_temp <= lower )
					disDif = lower - dist_temp;
				else if(dist_temp >= upper)
					disDif = dist_temp - upper;
		    	
		    	if(disDif < disDifMin )
		    	{
		    		disDifMin=disDif;	
		    		distanceMin=dist_temp;
		    	//	isSatisfied=true;
		    	}		    	
		    }//if (atomB.equalsIgnoreCase(nucleusB))
		   if(nucleusB.length()>=2)
		   {
			    if(nucleusB.substring(0,1).equalsIgnoreCase("Q") && (!nucleusB.substring(1,2).equalsIgnoreCase("Q")) && (atomB.length()>=2) )
			    {
			    	String temp="H"+nucleusB.substring(1,nucleusB.length());
			    	if(atomB.length()>=temp.length())
			    	{
				    	if(atomB.substring(0,temp.length() ).equalsIgnoreCase(temp))
				    	{
				    		foundB = true;	
				    		coordB = cc.getXYZ();
				    		
				    		double dist_temp= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
									+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
									+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
					    	if (dist_temp > lower && dist_temp < upper)
							    disDif = 0.0;
							else if(dist_temp <= lower )
								disDif = lower - dist_temp;
							else if(dist_temp >= upper)
								disDif = dist_temp - upper;
					    	
					    	if(disDif < disDifMin )
					    	{
					    		disDifMin=disDif;	
					    		distanceMin=dist_temp;
					    		//isSatisfied=true;
					    	}
				    	}//if(atomB.substring(0,temp.length() ).equalsIgnoreCase(temp))
			    	}//if(atomB.length()>=temp.length())
			    }  //if(nucleusB.substring(0,1).equalsIgnoreCase("Q") && (!nucleusB.substring(1,2).equalsIgnoreCase("Q")) && (atomB.length()>=2) )
				   
			    if(nucleusB.substring(0,2).equalsIgnoreCase("QQ") && (atomB.length()>=2) )
			    {
			    	String temp="H"+nucleusB.substring(2,nucleusB.length());
			    	if(atomB.length()>= temp.length() )
			    	{
				    	if(atomB.substring(0,temp.length() ).equalsIgnoreCase(temp) )
				    	{
				    		foundB = true;	
				    		coordB = cc.getXYZ();
				    		//System.out.println("atomA="+atomA+  " atomA= "+ atomB);
				    		double dist_temp= Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
									+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
									+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
					    	if (dist_temp > lower && dist_temp < upper)
							    disDif = 0.0;
							else if(dist_temp <= lower )
								disDif = lower - dist_temp;
							else if(dist_temp >= upper)
								disDif = dist_temp - upper;
					    	
					    	if(disDif < disDifMin )
					    	{
					    		disDifMin=disDif;	
					    		distanceMin=dist_temp;
					    		//isSatisfied=true;
					    	}		
				    	}//if(atomB.substring(0,temp.length() ).equalsIgnoreCase(temp) )
			    	}//if(atomB.length()>= temp.length() )
			    }  //if(nucleusB.substring(0,2).equalsIgnoreCase("QQ") && (atomB.length()>=2) )
		   }//if(nucleusB.length()>=2)
		}//for (i = 0; i < atomVecB.size(); i++)
		
	 	if ( foundA & foundB )
	 	    found=true;

	}//for (i = 0; i < atomVecA.size(); i++)	
	disRMS[0]=disDifMin;
	distance[0]=distanceMin;
	
	//debugging: 
	if(!found)
		System.out.println("debugging: Not found atom name");

	return found;
    }
 
    /**
     * Compute the NOE distance between two nuclei (A, B) in a
     * structure specified by pdbVec There are two numbers: 15 Degree
     * and 45 Degree for correct the most possible orientation changing.
     * The number themselves are set rather arbitrariry. Reaaly need
     * search through PDB to compute them.
     * 
     * @param pdbVec the PDB structure
     * @param noA     the residue NO of nucleus A
     * @param residueA  the residue type of nucleus A
     * @param nucleusA  the name of the nucleus of residue A
     * @param noB     the residue NO of nucleus A
     * @param residueB  the residue type of nucleus B
     * @param nucleusB  the name of the nucleus of residue B
     * @param distance  save the back-computed NOE distance
     * 
     * @return true if such a distance can be computed in the structure, false if there is no
     * such nuclei in the structure.
     */
    public boolean measureDis(final Vector pdbVec, int noA, String residueA, String nucleusA, int noB, 
			      String residueB, String nucleusB, double[] distance){
	int ind1 = Collections.binarySearch(pdbVec, new Pdb(noA), new Pdb.PdbComparator());
	int ind2 = Collections.binarySearch(pdbVec, new Pdb(noB), new Pdb.PdbComparator());
	if (ind1 < 0 || ind2 < 0)
	    return false;
	if (nucleusA.equalsIgnoreCase("HN")) //We need to convert all the CS name to PDB name
	    nucleusA = "H";
	if (nucleusB.equalsIgnoreCase("HN"))
	    nucleusB = "H";
	
	Pdb pdbA = (Pdb)pdbVec.elementAt(ind1);
	Pdb pdbB = (Pdb)pdbVec.elementAt(ind2);
	String atom = "";
	Vector atomVecA = pdbA.getAtomVec();
	Vector atomVecB = pdbB.getAtomVec();
	Cartesian cc = new Cartesian();
	double [] coordA = {0.0, 0.0, 0.0};
	double [] coordB = {0.0, 0.0, 0.0};
	int i = 0;
	boolean found = false;
	Map fromCBMap = new TreeMap();
	Residue resid = new Residue();
	for (i = 0; i < atomVecA.size(); i++){ //after backbone structure being computed, this always return true
	    cc = (Cartesian) atomVecA.elementAt(i);
	    atom = cc.getAtom();
	    if (atom.equalsIgnoreCase(nucleusA)){
		coordA = cc.getXYZ();
		found = true;
	    }
	}
	if ( !found ){

	    return found;
	}
	found = false;
	double disFromCB = 0.0;
	double dis2CB = 0.0;
	for (i = 0; i < atomVecB.size(); i++){
	    cc = (Cartesian) atomVecB.elementAt(i);
	    atom = cc.getAtom();
	    if (atom.equalsIgnoreCase(nucleusB)){
		coordB = cc.getXYZ();
		distance[0] = Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
					+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
					+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));

		return true;
	    }else if ( (!residueB.equals("PRO")) && atom.equals("CB")) {
		coordB = cc.getXYZ();
		dis2CB = Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
				   + (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
				   + (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
		fromCBMap = resid.fromCB(residueB);
		if (fromCBMap.get(nucleusB) != null){
		    disFromCB = ((Double)fromCBMap.get(nucleusB)).doubleValue();
		    if (noA == noB)
			distance[0] = dis2CB - disFromCB * Math.cos(45.0 * Const.cst);
		    else  distance[0] = dis2CB - disFromCB * Math.cos(15.0 * Const.cst);

		    found = true;
		}else System.out.println(residueB+" "+nucleusB+" Can NOT be found");
	    }
	}
 	if ( !found ){

 	    return found;
 	}
	return true;
    }

    /**
     * Compute a structure, given the backbone and rotamers at all residue positions.
     * 
     * @param pdbBBVec the PDB backbone structure
     * @param asgVec    resonace list
     * @param rotSrc  the path of rotmaer library
     * @param hnNoeVec  hn Noe vector
     * @param cNoeVec ha noe vector
     * @param pdbVecXray  x-ray structure, including both backbone and side-chains
     * @param distBound  the distance bound for back computing distance pattern for each rotamer
     * @param pickNum  the best number of rotamers picked for computation
     * 
     * @return overall rmsd.
     */
    public double compOverAllRmsdBetweenRotXray(Vector pdbBBVec, Vector asgVec, String rotSrc, 
    		Vector hnNoeVec, Vector cNoeVec,Vector pdbVecXray,double distBound, int pickNum)
    {
    	int i,j;
    	Pdb pdb,pdb_temp;
    	Pdb pp=new Pdb();
    	String resName;
    	String rotamFile="";
    	Vector pdbRotam=new Vector();
    	Vector pdbRotamNewStr=new Vector();
    	Vector pdbRotamNewCS=new Vector();
    	Vector pdbRotamNewRef=new Vector();
    	int resNo;
    	double [] rmsd=new double[pdbBBVec.size()];
    	RotaPattern rp=new RotaPattern();
    	int counter=0;
    	double sum=0.0;
    	for (i=0;i<pdbBBVec.size();i++)
    	{
    		rmsd[i]=0.0;
    		pdb=(Pdb)pdbBBVec.elementAt(i);
    		resName=pdb.getResidue();
    		resNo=pdb.getResidueNo();
    		if (resNo==1||resNo==72)
    			continue;
    		
    		if (resNo==55)
    		{
    			System.out.println("we stop hreee....");
    			
    		}
    		Vector pdbRotVec=new Vector();
    		
    		if( resName.equalsIgnoreCase("GLY"))
    		{
    			
    			int ind1 = Collections.binarySearch(pdbBBVec, new Pdb(resNo), new Pdb.PdbComparator());
    			Pdb pdbA = (Pdb)pdbBBVec.elementAt(ind1);    			
    			pdbRotVec.add(pdbA);
    			pdbRotamNewStr=pdbRotVec;
    			
    		}
    		else
    		{
    			rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
    			pdbRotam=pp.readPdb(rotamFile);
    			pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
        		pdbRotamNewCS=pp.nameConvert4SmallRotLibCS(pdbRotam);
        		pdbRotVec=pdbRotamNewCS;//important here...	    		
    		}   		
    		
    		
    		
    		int indRef = Collections.binarySearch(pdbBBVec, new Pdb(resNo), new Pdb.PdbComparator());
    		int [] index=new int[2];
    		if (!(indRef<0))
        		pdbRotamNewRef=rp.RefineRotLib(pdbRotVec,pdbBBVec,asgVec, hnNoeVec, cNoeVec,resNo,distBound, 1,index,pdbVecXray);
    	
    		
    		
    		double[] rms=new double[1];
    		pdb_temp=(Pdb)pdbRotamNewStr.elementAt(index[0]);
    		Vector pdbRotamNewStr2=new Vector();
    		pdbRotamNewStr2.add(pdb_temp);
    		boolean isHeavy=true; //true;
    		boolean hasRmsd=pp.compRmsdRotXrayBB(pdbVecXray,pdbRotamNewStr2,resNo, rms,isHeavy);
    		if(hasRmsd)
    		{
    			if( ((double)(index[1])/pdbRotVec.size())>=0.0)
    			{
    				System.out.print("average number of peaks is: "+ index[1]+ "\n");
    				System.out.print("ratio between number of peaks and number of rotamers is: "+ ((double)(index[1])/pdbRotVec.size())+"\n");
    				
    				System.out.print("resiudue No="+resNo+" res name="+ resName+" rmsd="+rms[0]+"\n");
    				System.out.println("====================================="); 
    				rmsd[i]=rms[0];
    				sum=sum+rms[0]*rms[0];
    				counter=counter+1;  
    			}
    		}//if(hasRmsd) 
    		
    		
    	}//for (i=0;i<pdbBBVec.size();i++)
    	
    	System.out.print("\n" + "overall rmsd="+ Math.sqrt(sum/counter)+ " total num="+counter +"\n");
    	return Math.sqrt(sum/counter);
    }
    
    /**
     * Checks if is in atom vec.
     * 
     * @param vecAtom the vec atom
     * @param atomName the atom name
     * 
     * @return true, if is in atom vec
     */
    public boolean isInAtomVec(Vector vecAtom, String atomName)
    {
    	boolean isIn=false;
    	for(int j=0;j<vecAtom.size();j++)
		{
    		Cartesian cc = (Cartesian) vecAtom.elementAt(j);			
			String strTemp = cc.getAtom();
			if(strTemp.equalsIgnoreCase(atomName))
			{
				isIn=true;   		
				break;
			}
		}
    	return isIn;
    	
    }
    
    /**
     * output the backbone of a pdb structure.
     * 
     * @param vecPdb pdb structure, new naming scheme, eg HB2,HB3...
     * 
     * @return the backbone pdb
     */
    public Vector<Pdb> OutputBackbone(Vector<Pdb> vecPdb)
    {
    	int i,j;
    	Vector vecNewBB=new Vector();
    	Pdb pdb=new Pdb();
    	Vector atomVec = new Vector();
    	
    	//Vector atom;
    	Cartesian cc;
    	double [] coord=new double[3];
    	int resNo;
    	String resName;
    	for (i=0;i<vecPdb.size();i++)
    	{
    		pdb=(Pdb)vecPdb.elementAt(i);
    		atomVec=pdb.getAtomVec();
    		resNo=pdb.getResidueNo();
    		resName=pdb.getResidue();
    		Vector atomBBVec=new Vector();
    		for(j=0;j<atomVec.size();j++)
    		{
    			cc = (Cartesian) atomVec.elementAt(j);
    			coord = cc.getXYZ();
    			String atomName = cc.getAtom();
    			boolean isAlreadyIn=false;
    		    if (atomName.equalsIgnoreCase("H")||atomName.equalsIgnoreCase("HN")||
    		    		atomName.equalsIgnoreCase("HA")||atomName.equalsIgnoreCase("N")||
    		    		atomName.equalsIgnoreCase("CA")||atomName.equalsIgnoreCase("O")||
    		    		atomName.equalsIgnoreCase("C")||atomName.equalsIgnoreCase("CB")||
    		    		atomName.equalsIgnoreCase("HB"))
    		    {
    		    	if(isInAtomVec(atomBBVec,atomName) )
    		    		continue;
    		    	atomBBVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));    		    	
    		    	  		    	
    		    }
    		    if(atomName.length()>2 && atomName.substring(0,2).equalsIgnoreCase("HA"))
    		    {
    		    	if(isInAtomVec(atomBBVec,atomName) )
    		    		continue;
    		    	if(atomName.substring(0,2).equalsIgnoreCase("HA"))
    		    		atomBBVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
    		    }
    		    	
    		    if(isInAtomVec(atomBBVec,atomName) )
		    		continue;
    		    if(atomName.length()>2 && atomName.substring(0,2).equalsIgnoreCase("HB"))
    		    	atomBBVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
    		}//for(j=0;j<atomVec.size();j++)
    		vecNewBB.add(new Pdb(resNo, resName,atomBBVec)); 
    	}//for (i=0;i<vecPdb.size();i++)
    	return vecNewBB;
    	
    }
    
    /**
     * output the backbone of a pdb structure.
     * 
     * @param vecPdb pdb structure, new naming scheme, eg HB2,HB3...
     * 
     * @return the backbone pdb
     */
    public Vector OutputBackboneNoCB(Vector vecPdb)
    {
    	int i,j;
    	Vector vecNewBB=new Vector();
    	Pdb pdb=new Pdb();
    	Vector atomVec = new Vector();
    	
    	//Vector atom;
    	Cartesian cc;
    	double [] coord=new double[3];
    	int resNo;
    	String resName;
    	for (i=0;i<vecPdb.size();i++)
    	{
    		pdb=(Pdb)vecPdb.elementAt(i);
    		atomVec=pdb.getAtomVec();
    		resNo=pdb.getResidueNo();
    		resName=pdb.getResidue();
    		Vector atomBBVec=new Vector();
    		for(j=0;j<atomVec.size();j++)
    		{
    			cc = (Cartesian) atomVec.elementAt(j);
    			coord = cc.getXYZ();
    			String atomName = cc.getAtom();
    		    if (atomName.equalsIgnoreCase("H")||atomName.equalsIgnoreCase("HN")||
    		    		atomName.equalsIgnoreCase("HA")||atomName.equalsIgnoreCase("N")||
    		    		atomName.equalsIgnoreCase("CA")||atomName.equalsIgnoreCase("O")||
    		    		atomName.equalsIgnoreCase("C"))
    		    {
    		    	atomBBVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));    		    	
    		    	  		    	
    		    }
    		    if(atomName.length()>2 && atomName.substring(0,2).equalsIgnoreCase("HA"))
    		    {
    		    	if(atomName.substring(0,2).equalsIgnoreCase("HA"))
    		    		atomBBVec.add(new Cartesian(atomName, coord[0], coord[1], coord[2]));
    		    }
    		    	
    		    
    		    
    		}//for(j=0;j<atomVec.size();j++)
    		vecNewBB.add(new Pdb(resNo, resName,atomBBVec)); 
    	}//for (i=0;i<vecPdb.size();i++)
    	return vecNewBB;
    	
    }
    
    /**
     * plug rotamer into backbone.
     * 
     * @param rotamVec the rotamer vector in pdb format, note: only *one* rotamer
     * @param resNo residue No of the rotamer
     * @param pdbVecN the pdb vec n
     * 
     * @return the new structure after pluging rotamer into backbone
     */
    public Vector<Pdb> CombinRotBB(final Vector<Pdb> rotamVec, final Vector<Pdb> pdbVecN, int resNo)
    {    	
    	Pdb pp=new Pdb();
    
    	//we first translate the the rotamer to be consistent with the protein coordinate system 	
    	Vector pdbRotamNewA=rotamVec;
    	int noA=resNo;
    	int i,j,k;
    	
    	int ind1 = Collections.binarySearch(pdbVecN, new Pdb(noA), new Pdb.PdbComparator());
    	
    	Pdb pdbA = (Pdb)pdbVecN.elementAt(ind1);
    	Vector pdbAVec=new Vector();
    	pdbAVec.add(pdbA);
    	Pdb pdb=new Pdb();
    	PhiPsi ff = new PhiPsi();
    	
    	String atom = "";
    	Vector atomVecA = pdbA.getAtomVec();
    	
    	//String resName=pdbA.getResidue();////
    	//we let the residue name be from rotamers rather from the structure. 
    	//since we assume that residue name should be the consistent with its side-chain
        Pdb pdbTempRot=(Pdb)rotamVec.elementAt(0);
        String resName=pdbTempRot.getResidue();
    		
    	Vector atomVec=new Vector();
    	Cartesian cc = new Cartesian();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	double [] amide = new double[3];
    	double [] amide1 = new double[3];
    	double [] amide2 = new double[3];
    	double [] nh = new double[3];
    	//double [] nh = new double[3];
    	double [] ha = new double[3];
    	double [] ca1 = new double[3];
    	double [] ca_temp = new double[3];
    	double [] ca2 = new double[3];
    	double [] trans_vec = new double[3];
    	
    	Matrix rgA = Matrix.identity(3,3);
    	
    	for ( j=0; j<atomVecA.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVecA.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide = cc.getXYZ();
    	    else if (atom.equals("CA"))
    	    	ca1=cc.getXYZ();
    	    else if (atom.equals("HA")||atom.equals("HA2"))
    		ha = cc.getXYZ();		    
    	}
    		
    	double [] CaToHaVecStr =  pdb.internuclearVec(ca1,  ha);
    	double [] CaToNVecStr =  pdb.internuclearVec(ca1, amide);//////
    	Matrix rg = Matrix.identity(3,3);
    	rg = ff.RgCal(CaToHaVecStr, CaToNVecStr);	
    	
    	Vector pdbANew=pdb.newPdb(pdbAVec,rg);//the new coordinate of str after being rotated
    	
    	//can be used for the next calcualtion
    	Vector pdbVecNew1=pdb.newPdb(pdbVecN,rg);//the whole structure after the frist rotation

    	
    	Pdb pdbTemp = (Pdb)pdbANew.elementAt(0);
    	atomVec=pdbTemp.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("CA"))
    	    	ca_temp = cc.getXYZ();	//for translation	    
    	} 
    	
    	Pdb pdbRotA=(Pdb)pdbRotamNewA.elementAt(0);
    	atomVec=pdbRotA.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("N"))
    		amide = cc.getXYZ();
    	    else if (atom.equals("CA"))
    	    	ca2=cc.getXYZ();
    	    else if (atom.equals("HA") ||atom.equals("HA2"))
    		ha = cc.getXYZ();    
    	}
    	double [] CaToHaVecRot =  pdb.internuclearVec(ca2,  ha);
    	double [] CaToNVecRot =  pdb.internuclearVec(ca2, amide);//////
    	rgA = ff.RgCal(CaToHaVecRot, CaToNVecRot);
    	
    	Vector rotamVecNewA_temp=pdb.newPdb(pdbRotamNewA,rgA);
    	
    	pdbTemp = (Pdb)rotamVecNewA_temp.elementAt(0);
    	atomVec=pdbTemp.getAtomVec();
    	for ( j=0; j<atomVec.size(); j++)
    	{ 
    	    cc = (Cartesian)atomVec.elementAt(j);
    	    atom = cc.getAtom();
    	    if (atom.equals("CA"))
    		ca2 = cc.getXYZ();		    
    	} 		
    	
    	//the translation vector after putting into the same frame
    	trans_vec= pdb.internuclearVec(ca2,ca_temp);				
    	
    	//rotamer pdb in the same coordinate system
    	Vector<Pdb> rotamVecNewA=pdb.newPdbByTranslation(rotamVecNewA_temp,trans_vec);
  
    	//combine bb atoms from pdbANew and other atoms from  rotamVecNewA
    	Vector vecNewPdb=new Vector();
    	Vector vecAtom4Res=new Vector();
    	for (i=0;i<pdbANew.size();i++)
    	{
    		Pdb pdbCombin=(Pdb)pdbANew.elementAt(i);
    		Vector atomVecCombin=pdbCombin.getAtomVec();
    		for (j=0;j<atomVecCombin.size();j++)
    		{
    			Cartesian ccCombin=(Cartesian)atomVecCombin.elementAt(j);
    			String strComb=ccCombin.getAtom();
    			if(strComb.equalsIgnoreCase("HN")|| strComb.equalsIgnoreCase("H")||
    					strComb.equalsIgnoreCase("N")||strComb.equalsIgnoreCase("CA")||
    					strComb.equalsIgnoreCase("HA")||strComb.equalsIgnoreCase("C")||
    					strComb.equalsIgnoreCase("O"))
    				vecAtom4Res.add(ccCombin);
    			
    			/*if(strComb.length()>2)
    			{
    				if (strComb.substring(0,2).equalsIgnoreCase("HA"))
    					vecAtom4Res.add(ccCombin);
    			}*/
    				
    			
    		}//for (j=0;j<atomVecCombin.size();j++)
       	}//for (i=0;i<pdbANew.size();i++)
    	
    	for(i=0;i<rotamVecNewA.size();i++)
    	{
    		Pdb pdbCombin=(Pdb)rotamVecNewA.elementAt(i);
    		Vector atomVecCombin=pdbCombin.getAtomVec();
    		for (j=0;j<atomVecCombin.size();j++)
    		{
    			Cartesian ccCombin=(Cartesian)atomVecCombin.elementAt(j);
    			String strComb=ccCombin.getAtom();
    			if(strComb.equalsIgnoreCase("HN")|| strComb.equalsIgnoreCase("H")||
    					strComb.equalsIgnoreCase("N")||strComb.equalsIgnoreCase("CA")||
    					strComb.equalsIgnoreCase("HA")||strComb.equalsIgnoreCase("C")||
    					strComb.equalsIgnoreCase("O"))
    				continue;
    			/*if(strComb.length()>2)
    			{
    				if (strComb.substring(0,2).equalsIgnoreCase("HA"))
    					continue;
    			}*///for gly...
    			vecAtom4Res.add(ccCombin );
    		}//for (j=0;j<atomVecCombin.size();j++)
    		
    	}//for(i=0;i<rotamVecNewA.size();i++)    	
    	
    	//combine other residues in the backbone
    	int ind2 = Collections.binarySearch(pdbVecNew1, new Pdb(resNo), new Pdb.PdbComparator());
    	for (i=0;i<pdbVecNew1.size();i++)
    	{
    		Pdb pdbTemp2=(Pdb)pdbVecNew1.elementAt(i);
    		if ((i==ind2)  )
    			vecNewPdb.add(new Pdb(resNo,resName,vecAtom4Res ));
    		else
    			vecNewPdb.add(pdbTemp2);
    		
    	}//for (i=1;i<pdbVecN.size();i++)
    	
    	return vecNewPdb;
    	
    }
    
    /**
     * put all rotamers at each redidue.
     * 
     * @param pdbBBVec the PDB backbone structure
     * @param rotSrc  the path of rotmaer library
     * 
     * @return final structure.
     */
    public Vector<Pdb> AllRotamersStructure(Vector pdbBBVec, String rotSrc)
    {
    	int i,j;
    	Pdb pdb;
    	String resName;
    	int resNo;
    	String rotamFile;
    	Vector pdbRotam=new Vector();
    	Vector pdbRotamNewStr=new Vector();
    	Vector pdbRotamNewCS=new Vector();
    	Vector pdbRotamNewRef=new Vector();
    	Pdb pp=new Pdb();
    	RotaPattern rp=new RotaPattern();
    	Vector vecGrowPdb=new Vector();
    	vecGrowPdb=pdbBBVec;//initial structure to plug in rotamers
    	
    	for (j=0;j<pdbBBVec.size();j++)
    	{ 	
        	pdb=(Pdb)vecGrowPdb.elementAt(j);
        	resName=pdb.getResidue();
        	resNo=pdb.getResidueNo();
        	Vector pdbRotVec=new Vector();
        	Vector vecSelectedRot=new Vector();	
        	pdbRotamNewStr=new Vector();
        	if( resName.equalsIgnoreCase("GLY11"))//this step is never reached...
        	{
        		int ind1 = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
        		Pdb pdbA = (Pdb)vecGrowPdb.elementAt(ind1);    			
        		pdbRotVec.add(pdbA);  
        		vecSelectedRot.addAll(pdbRotVec);
        	}
        	else
        	{
        		rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
        		pdbRotam=pp.readPdb(rotamFile);
        		//atom names are consistet with structure, namely HB1,HB2...
        		pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
        		//atom names are consistent with chemical shifts, eg, HB for HB#
            	pdbRotamNewCS=pp.nameConvert4SmallRotLibCS(pdbRotam);
            	pdbRotVec=pdbRotamNewCS;	    		
        	}       		
        	       	
        	//Pdb pdb_temp=(Pdb)pdbRotamNewStr.elementAt(index[0]);
        	vecSelectedRot.addAll(pdbRotam);///changed here
        	//update the backbone structure, including previous fixed rotamer
        	Vector vecTemp=pp.CombinRotBB(vecSelectedRot,vecGrowPdb,resNo);
    			//pp.print(vecTemp);
        	vecGrowPdb=new Vector();        	
    		vecGrowPdb=vecTemp;
    		
    	}//for (j=0;j<pdbBBVec.size();j++)
    	
    	return vecGrowPdb;    	
    }
          

    
    /**
     * Compute a structure, given the backbone and rotamers at all residue positions.
     * 
     * @param csErrH csErrN csErrCA error window size at each dimension
     * @param pdbBBVec the PDB backbone structure
     * @param asgVec    resonace list
     * @param rotSrc  the path of rotmaer library
     * @param NoesyVec  Noe vector including both ha and hn noes
     * @param distBound  the distance bound for back computing distance pattern for each rotamer
     * @param pickNum  the best number of rotamers picked for computation
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * @param constant the constant
     * 
     * @return final structure. and Updated asgVec.
     * Note: the structure will be rotated for installing rotamers.
     * So the strucgture need to be rotated back outside this function.
     */
    public Vector<Pdb> RotamSelectAndStructure(double csErrH, double csErrN, double csErrCA,
    		Vector<Pdb> pdbBBVec, Vector asgVec, String rotSrc, 
    		Vector<Noesy> NoesyVec,double distBound, int pickNum,double constant)
    {
    	int i,j;
    	Pdb pdb;
    	String resName;
    	int resNo;
    	String rotamFile;
    	Vector pdbRotam=new Vector();
    	Vector pdbRotamNewStr=new Vector();
    	Vector pdbRotamNewCS=new Vector();
    	Vector pdbRotamNewRef=new Vector();
    	Pdb pp=new Pdb();
    	RotaPattern rp=new RotaPattern();
    	Vector vecGrowPdb=new Vector();
    	vecGrowPdb=pdbBBVec;//initial structure to plug in rotamers
    	
    	for (j=0;j<pdbBBVec.size();j++)
    	{ 	
        	pdb=(Pdb)vecGrowPdb.elementAt(j);
        	resName=pdb.getResidue();
        	resNo=pdb.getResidueNo();
        	Vector pdbRotVec=new Vector();      	
        	      
        	
        		rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
        		pdbRotam=pp.readRotamerPdb(rotamFile);
        		
        		//atom names are consistet with structure, namely HB1,HB2...
        		pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
        
            	pdbRotVec=pdbRotam;//changed for the ff2 case//////////
     
        		
        	int indRef = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
        	int [] index=new int[2];
        	
        	
         
        	if (!(indRef<0))    
        	{
            	pdbRotamNewRef=rp.RefineRotLibNew(csErrH,csErrN, csErrCA,pdbRotVec,vecGrowPdb,asgVec, NoesyVec,resNo,distBound, pickNum,pdbRotVec,constant);
        		//pdbRotamNewRef=rp.RefineRotLibNewUsingBins(csErrH,csErrN, csErrCA,pdbRotVec,vecGrowPdb,asgVec, NoesyVec,resNo,distBound, pickNum,pdbRotVec);
        	
        	}
        	/*if(resName.equalsIgnoreCase("GLY"))
        	{
        		pdbRotamNewRef=new Vector();
        		pdbRotamNewRef.addAll(pdbRotVec);
        	}  */       	
        	
        	//here we need to be careful when bb is not accurate enough 
        	//so that index[0]=-1;
        	if (index[0]<0.0)
        		index[0]=0;
        	
        	Vector vecSelectedRot=new Vector();
        	Pdb pdb_temp=(Pdb)pdbRotamNewStr.elementAt(index[0]);
        
        	vecSelectedRot.add(pdb_temp);
        	
          	
        	//update the backbone structure, including previous fixed rotamer
        	//Vector vecTemp=pp.CombinRotBB(vecSelectedRot,vecGrowPdb,resNo);
        	Vector vecTemp=pp.CombinRotBB(pdbRotamNewRef,vecGrowPdb,resNo);
    			
    		vecGrowPdb=vecTemp;
    		
    	}//for (j=0;j<pdbBBVec.size();j++)
    	
    	return vecGrowPdb;    	
    }
 
    /**
     * replace all non-NOE residues in pdb with alaline.
     * 
     * @param vecPdb original (bb) pdb structrue
     * @param resIndex whether the residue index is non-ala, the value is true if yes,
     * otherwise false
     * @param rotSrc the rot src
     * 
     * @return the updated structure.
     * Note: the structure will be rotated for installing rotamers.
     * So the strucgture need to be rotated back outside this function.
     * 
     * @throws JampackException the jampack exception
     */
   public Vector AlaninizeStructure(Vector vecPdb, boolean[] resIndex, String rotSrc)throws JampackException
   {    
	   Pdb pp=new Pdb();
	   Vector pdbRotam=new Vector();
	   String resName="ALA";
	   String rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
	   pdbRotam=pp.readPdb(rotamFile);
		//atom names are consistet with structure, namely HB1,HB2...
	   Vector pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
	   
	   Vector pdbRotamPro=new Vector();
	   String resNamePro="PRO";
	   String rotamFilePro=rotSrc+ resNamePro.toLowerCase()+ ".pdb"; 
	   pdbRotamPro=pp.readPdb(rotamFilePro);
		//atom names are consistet with structure, namely HB1,HB2...
	   Vector pdbRotamNewStrProAll=pp.nameConvert4SmallRotLibStr(pdbRotamPro);
	   Pdb pp_temp=new Pdb();
	   pp_temp=(Pdb)pdbRotamNewStrProAll.elementAt(0);
	   Vector pdbRotamNewStrPro=new Vector();
	   pdbRotamNewStrPro.add(pp_temp);
	   
	   Vector vecGrowPdb=new Vector();
	   vecGrowPdb.addAll(vecPdb);
	   for(int i=0; i<vecPdb.size();i++)
	   {
		   Pdb pdb=(Pdb)vecPdb.elementAt(i);
		   int res_no=pdb.getResidueNo();
		   String res=pdb.getResidue();
		   if( (resIndex[i]==false) && (res.equalsIgnoreCase("PRO") ) ) 
		   {
			   Vector vecTemp=pp.CombinRotBB(pdbRotamNewStrPro,vecGrowPdb,res_no);
			   vecGrowPdb=new Vector();
			   vecGrowPdb.addAll(vecTemp);
			   continue;
		   }
		   
		   if( (resIndex[i]==false) && !(res.equalsIgnoreCase("PRO") || res.equalsIgnoreCase("GLY") ) )
		   {
			   Vector vecTemp=pp.CombinRotBB(pdbRotamNewStr,vecGrowPdb,res_no);
			   vecGrowPdb=new Vector();
			   vecGrowPdb.addAll(vecTemp);
		   }//if(resIndex[i]==false)
			   
	   }//for(int i=0; i<vecPdb.size();i++)
	   
	   //rotate back, since the rotation has been changed
	   Model md=new Model();
	   Vector vecGrowPdbNew=new Vector();
	   vecGrowPdbNew=  md.BackonbeRotation(vecGrowPdb,vecPdb); 
	   return vecGrowPdbNew;  
   }
    
    /**
     * replace all non-NOE residues in pdb with alaline, according to the residue names.
     * 
     * @param vecPdb original (bb) pdb structrue
     * @param rotSrc the rot src
     * 
     * @return the updated structure.
     * Note: the structure will be rotated for installing rotamers.
     * So the strucgture need to be rotated back outside this function.
     * 
     * @throws JampackException the jampack exception
     */
   public Vector AlaninizeStructureResName(Vector vecPdb, String rotSrc)throws JampackException
   {    
	   Pdb pp=new Pdb();
	   Vector pdbRotam=new Vector();
	   String resName="ALA";
	   String rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
	   pdbRotam=pp.readPdb(rotamFile);
		//atom names are consistet with structure, namely HB1,HB2...
	   Vector pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
	   
	   //for gly
	   Vector pdbRotamGly=new Vector();
	   String resNameGLY="GLYTEMP";
	   String rotamFileGly=rotSrc+ resNameGLY.toLowerCase()+ ".pdb"; 
	   pdbRotamGly=pp.readPdb(rotamFileGly);
		//atom names are consistet with structure, namely HB1,HB2...
	   //Vector pdbRotamGlyNew=pp.nameConvert4SmallRotLibStr(pdbRotamGly);
	   
	   //for pro
	   Vector pdbRotamPro=new Vector();
	   String resNamePro="PRO";
	   String rotamFilePro=rotSrc+ resNamePro.toLowerCase()+ ".pdb"; 
	   pdbRotamPro=pp.readPdb(rotamFilePro);   
		//atom names are consistet with structure, namely HB1,HB2...
	   Vector pdbRotamNewStrProAll=pp.nameConvert4SmallRotLibStr(pdbRotamPro);
	   Pdb pp_temp=new Pdb();
	   pp_temp=(Pdb)pdbRotamNewStrProAll.elementAt(0);
	   Vector pdbRotamNewStrPro=new Vector();
	   pdbRotamNewStrPro.add(pp_temp);
	   
	   Vector vecGrowPdb=new Vector();
	   vecGrowPdb.addAll(vecPdb);
	   for(int i=0; i<vecPdb.size();i++)
	   {
		   Pdb pdb=(Pdb)vecPdb.elementAt(i);
		   int res_no=pdb.getResidueNo();
		   String res=pdb.getResidue();
		   if( (res.equalsIgnoreCase("PRO") ) ) 
		   {
			   Vector vecTemp=pp.CombinRotBB(pdbRotamNewStrPro,vecGrowPdb,res_no);
			   vecGrowPdb=new Vector();
			   vecGrowPdb.addAll(vecTemp);
			   continue;
		   }
		   
		   if( (res.equalsIgnoreCase("GLY") ) ) 
		   {
			   Vector vecTemp=pp.CombinRotBB(pdbRotamGly,vecGrowPdb,res_no);
			   vecGrowPdb=new Vector();
			   vecGrowPdb.addAll(vecTemp);
			   continue;
		   }
		   
		   if(  !(res.equalsIgnoreCase("PRO") || res.equalsIgnoreCase("GLY") ) )
		   {
			   Vector vecTemp=pp.CombinRotBB(pdbRotamNewStr,vecGrowPdb,res_no);
			   vecGrowPdb=new Vector();
			   vecGrowPdb.addAll(vecTemp);
		   }//if(resIndex[i]==false)
			   
	   }//for(int i=0; i<vecPdb.size();i++)
	   
	   //rotate back, since the rotation has been changed
	   Model md=new Model();
	   Vector vecGrowPdbNew=new Vector();
	   vecGrowPdbNew=  md.BackonbeRotation(vecGrowPdb,vecPdb); 
	   return vecGrowPdbNew;  
   }
    
    /**
     * Rotam select and structure no clash.
     * 
     * @param csErrH csErrN csErrCA error window size at each dimension
     * @param pdbBBVec the PDB backbone structure
     * @param asgVec    resonace list
     * @param rotSrc  the path of rotmaer library
     * @param NoesyVec  Noe vector including both ha and hn noes
     * @param distBound  the distance bound for back computing distance pattern for each rotamer
     * @param resIndex residue index to be analized
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * @param pickNum the pick num
     * 
     * @return vector (ensemble) of structures.
     * Note: the structure will be rotated for installing rotamers.
     * So the strucgture need to be rotated back outside this function.
     * 
     * @throws JampackException the jampack exception
     */
    public Vector RotamSelectAndStructureNoClash(double csErrH, double csErrN, double csErrCA,
    		Vector pdbBBVec, Vector asgVec, String rotSrc, 
    		Vector NoesyVec,double distBound,int pickNum,boolean[] resIndex)throws JampackException
    {
    	int i,j;
    	Pdb pdb;
    	String resName;
    	int resNo;
    	String rotamFile;
    	Vector pdbRotam=new Vector();
    	Vector pdbRotamNewStr=new Vector();
    	Vector pdbRotamNewCS=new Vector();
    	Vector pdbRotamNewRef=new Vector();
    	Pdb pp=new Pdb();
    	RotaPattern rp=new RotaPattern();
    	Vector vecGrowPdb=new Vector();
    
    	
//    	note: need to be fixed...
    	//initial rotamer selection for the structure:
    	//for each residue, choose the best rotamer based on NOE patterns:
    	vecGrowPdb= RotamSelectAndStructure(csErrH,csErrN,csErrCA,pdbBBVec,asgVec,rotSrc, 
    	    		NoesyVec,distBound, pickNum,0.0);  //////////  	
    	   	
    	Vector vecPdbSseRot2=new Vector();	
		vecPdbSseRot2=pp.AlaninizeStructure(vecGrowPdb,resIndex,rotSrc);
		vecGrowPdb=new Vector();
		vecGrowPdb.addAll(vecPdbSseRot2);		
		
    	Vector vecRot4AllResidues=new Vector();
    	int [] rotID=new int[pdbBBVec.size()];
    	for (j=0;j<pdbBBVec.size();j++)
    	{ 	
    		rotID[j]=0;//initially we choose the first rotamer
        	pdb=(Pdb)vecGrowPdb.elementAt(j);
        	resName=pdb.getResidue();
        	resNo=pdb.getResidueNo();
        	Vector pdbRotVec=new Vector();
        		
        	if( resName.equalsIgnoreCase("GLY"))
        	{
        		int ind1 = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
        		Pdb pdbA = (Pdb)vecGrowPdb.elementAt(ind1);    			
        		pdbRotVec.add(pdbA);    
        		pdbRotamNewStr.addAll(pdbRotVec);
        	}
        	else
        	{
        		rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
        		pdbRotam=pp.readPdb(rotamFile);
        		//atom names are consistet with structure, namely HB1,HB2...
        		pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
        		//atom names are consistent with chemical shifts, eg, HB for HB#
            	pdbRotamNewCS=pp.nameConvert4SmallRotLibCS(pdbRotam);
            	pdbRotVec=pdbRotamNewCS;//important here...	    		
        	}   		
        		
        	int indRef = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
        	int [] index=new int[2];
        		
        	//if (!(indRef<0))
            	//pdbRotamNewRef=rp.RefineRotLib(csErrH,csErrN, csErrCA,pdbRotVec,vecGrowPdb,asgVec, NoesyVec,resNo,distBound, pickNum,index);
//        	note: need to be fixed...
        	if (!(indRef<0))
            	pdbRotamNewRef=rp.RefineRotLibNew(csErrH,csErrN, csErrCA,pdbRotVec,vecGrowPdb,asgVec, NoesyVec,resNo,distBound, pdbRotVec.size(),pdbRotamNewStr,0.0);
  
        	vecRot4AllResidues.add(pdbRotamNewRef);        	
    		
    	}//for (j=0;j<pdbBBVec.size();j++)
    	
    	//refine the rotamer selection to avoid steric clash:
    	Vector vecRefineGrowPdb=new Vector();
    	vecRefineGrowPdb=RefineRotamerSelection(vecGrowPdb,vecRot4AllResidues, rotID);
    	return vecRefineGrowPdb;    	
    
   
    	 	
    	
    }
  
    /**
     * refine the rotamer selection until no steric clash happens.
     * 
     * @param vecGrowPdb previous structure after placing rotmaers into the backbone
     * @param vecRot4AllResidues vector of rotamers at each residue
     * @param rotID store the rotamer that is current in the residue
     * 
     * @return refined structure.
     */
    public Vector RefineRotamerSelection(Vector vecGrowPdb,Vector vecRot4AllResidues,
    		int [] rotID)
    {  
    	Vector vecRefinePdb=new Vector();
    	
    	vdw vander = new vdw();
    	Vector vdwVec = new Vector();
    	double[] vdwValue = new double[1];
    	boolean hasVDW = false;
    	double vdwLevel = 0.05;
    	boolean printVDWViolation = false;    
		vdwVec = vander.convert2VDW(vecGrowPdb);
		int[] clashResNo=new int[2];
		boolean isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
				true, 0.8, 1, clashResNo); 
    	
		if (!isStericClash)
			return vecGrowPdb;	
		
    	int resNo1=clashResNo[0];
    	int resNo2=clashResNo[1];
    	//int resNoFixed=0, resNoUnFixed=0;
		int index1 = Collections.binarySearch(vecGrowPdb, new Pdb(resNo1), new Pdb.PdbComparator());
		int index2 = Collections.binarySearch(vecGrowPdb, new Pdb(resNo2), new Pdb.PdbComparator());
    	
		Vector vecRot1=(Vector)vecRot4AllResidues.elementAt(index1);
	    int rotNum1=vecRot1.size();
		Vector vecRot2=(Vector)vecRot4AllResidues.elementAt(index2);
		int rotNum2=vecRot2.size();
		
					
		Pdb pp=new Pdb();
		boolean isRefined=false;
		for(int i=0; i<vecRot1.size();i++ )
		{
			//pp.print(vecRot1);
			Pdb pdbRot=(Pdb)vecRot1.elementAt(i);
			Vector vecPdbRot=new Vector();
			vecPdbRot.add(pdbRot);
			Vector vecTemp1=pp.CombinRotBB(vecPdbRot,vecGrowPdb,resNo1);			
			
			boolean isBreak=false;
			for(int j=0;j<vecRot2.size();j++)
			{
				Pdb pdbRot2=(Pdb)vecRot2.elementAt(j);
				Vector vecPdbRot2 =new Vector ();
				vecPdbRot2.add(pdbRot2);
				Vector vecTemp2=pp.CombinRotBB(vecPdbRot2,vecTemp1,resNo2);
				
				vdwVec = vander.convert2VDW(vecTemp2);
				isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
						true, 0.8, 1, clashResNo); 
				if( !( (clashResNo[0]== resNo1 && clashResNo[1]==resNo2) || (clashResNo[0]== resNo2 && clashResNo[1]==resNo1)) )
			    //if( (clashResNo[0]!= resNo2) && (clashResNo[1]!=resNo2) && (clashResNo[0]!= resNo1) && (clashResNo[1]==resNo1) )
				{
					System.out.println("We have refined the roatmer position at residue: "+ resNo1 + " and residue: "+ resNo2);
					isRefined=true;
					vecGrowPdb=new Vector();
					vecGrowPdb=vecTemp2;
					isBreak=true;
					break;
				}	
				
			}//for(int i=0;i<vecRot2.size();i++)
			
			if(isBreak)
				break;
			
		}//for(int i=0; i<vecRot1.size();i++ )
		
	
	    if(!isRefined)
	    	System.out.println("We counldn't refine the rotamer selection at residue:"+ resNo1 + " and residue: "+resNo2 );
		
		//pp.print(vecTemp);	    
	    vecRefinePdb=RefineRotamerSelection(vecGrowPdb,vecRot4AllResidues,rotID);
	    
    	return vecRefinePdb;
    }

    /**
     * refine the rotamer selection until no steric clash happens,
     * using Gaussian random sampling at each rotamer position.
     * 
     * @param vecGrowPdb previous structure after placing rotmaers into the backbone
     * @param vecRot4AllResidues vector of rotamers at each residue
     * @param rotID store the rotamer that is current in the residue
     * 
     * @return refined structure.
     */
    public Vector RefineRotamerSelectionRandom(Vector vecGrowPdb,Vector vecRot4AllResidues,
    		int [] rotID)
    {  
    	Vector vecRefinePdb=new Vector();
    	
    	vdw vander = new vdw();
    	Vector vdwVec = new Vector();
    	double[] vdwValue = new double[1];
    	boolean hasVDW = false;
    	double vdwLevel = 0.05;
    	boolean printVDWViolation = false;    
		vdwVec = vander.convert2VDW(vecGrowPdb);
		int[] clashResNo=new int[2];
		boolean isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
				true, 0.25, 1, clashResNo); 
    	
		if (!isStericClash)
			return vecGrowPdb;	
		
    	int resNo1=clashResNo[0];
    	int resNo2=clashResNo[1];
    	
		int index1 = Collections.binarySearch(vecGrowPdb, new Pdb(resNo1), new Pdb.PdbComparator());
		int index2 = Collections.binarySearch(vecGrowPdb, new Pdb(resNo2), new Pdb.PdbComparator());
    	
		Vector vecRot1=(Vector)vecRot4AllResidues.elementAt(index1);
	    int rotNum1=vecRot1.size();
		Vector vecRot2=(Vector)vecRot4AllResidues.elementAt(index2);
		int rotNum2=vecRot2.size();
		
			
		Random generator = new Random( );
		
		Pdb pp=new Pdb();
		boolean isRefined=false;
		//for(int i=0; i<vecRot1.size();i++ )
		{
			
			double tet=generator.nextGaussian();
			double temp1=vecRot1.size()*generator.nextGaussian();
			if(temp1<0)
				temp1=-temp1;
			int i=(int)Math.min(Math.floor(temp1), vecRot1.size()-1);			
			
			Pdb pdbRot=(Pdb)vecRot1.elementAt(i);
			Vector vecPdbRot=new Vector();
			vecPdbRot.add(pdbRot);
			Vector vecTemp1=pp.CombinRotBB(vecPdbRot,vecGrowPdb,resNo1);			
			
			boolean isBreak=false;
			//for(int j=0;j<vecRot2.size();j++)			
			{
				//int j= generator.nextInt( vecRot2.size());
				double temp2=vecRot2.size()*generator.nextGaussian();
				if(temp2<0)
					temp2=-temp2;
				int j=(int)Math.min(Math.floor(temp2), vecRot2.size()-1);
				//int j=(int)Math.floor(vecRot2.size()*generator.nextGaussian());
				
				Pdb pdbRot2=(Pdb)vecRot2.elementAt(j);
				Vector vecPdbRot2 =new Vector ();
				vecPdbRot2.add(pdbRot2);
				Vector vecTemp2=pp.CombinRotBB(vecPdbRot2,vecTemp1,resNo2);
				
				vdwVec = vander.convert2VDW(vecTemp2);
				isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
						true, 0.25, 1, clashResNo); 
				if( !( (clashResNo[0]== resNo1 && clashResNo[1]==resNo2) || (clashResNo[0]== resNo2 && clashResNo[1]==resNo1)) )
			   {
					System.out.println("We have refined the roatmer position at residue: "+ resNo1 + " and residue: "+ resNo2);
					isRefined=true;
					vecGrowPdb=new Vector();
					vecGrowPdb.addAll(vecTemp2);
					isBreak=true;
					//break;
				}	
				
			}//for(int i=0;i<vecRot2.size();i++)
			
			
			
		}//for(int i=0; i<vecRot1.size();i++ )
		
	
	    if(!isRefined)
	    	System.out.println("We counldn't refine the rotamer selection at residue:"+ resNo1 + " and residue: "+resNo2 );
		
		//pp.print(vecTemp);	    
	    vecRefinePdb=RefineRotamerSelectionRandom(vecGrowPdb,vecRot4AllResidues,rotID);
	    
    	return vecRefinePdb;
    }

    /**
     * wait...
     * 
     * @param vecGrowPdb previous structure after placing rotmaers into the backbone
     * @param resNo the res no
     * @param rotSrc the rot src
     * @param depth the depth
     * 
     * @return refined structure.
     */
    public Vector RefineRotamerSelectionSearch(Vector vecGrowPdb,int resNo,String rotSrc,int depth)
    {  
    	
    	Random generator = new Random( );
    	Pdb pp=new Pdb();
    	Vector vecRefinePdb=new Vector();
    	
    	vdw vander = new vdw();
    	Vector vdwVec = new Vector();
    	double[] vdwValue = new double[1];
    	boolean hasVDW = false;
    	double vdwLevel = 0.05;
    	boolean printVDWViolation = false;    
 
    	int ind = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
    	Pdb pdb = (Pdb)vecGrowPdb.elementAt(ind);    
    	
    	String resName=pdb.getResidue();
    	
    	Vector pdbRotVec=new Vector ();
    	Vector pdbRotamNewStr=new Vector();
    	
    	
    	if( resName.equalsIgnoreCase("GLY"))
    	{
    		int ind1 = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
    		Pdb pdbA = (Pdb)vecGrowPdb.elementAt(ind1);    			
    		pdbRotVec.add(pdbA);    
    		pdbRotamNewStr.addAll(pdbRotVec);
    	}
    	else
    	{
    		String rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
    		Vector pdbRotam=pp.readPdb(rotamFile);
    		//atom names are consistet with structure, namely HB1,HB2...
    		//changed here for ff2
    			
    		pdbRotamNewStr=pdbRotam;//pp.nameConvert4SmallRotLibStr(pdbRotam);  
    	}   		
	
    	int minClashes=99999;
    	int minID=0;
		double minRmsd=99999.9;
	    for (int i=0;i<pdbRotamNewStr.size();i++)
	    {
	    	Pdb ppRot=(Pdb)pdbRotamNewStr.elementAt(i);
	    	Vector vecRotTemp=new Vector();
	    	vecRotTemp.add(ppRot);
	    	Vector vecTemp=new Vector();
	    	vecTemp=pp.CombinRotBB(vecRotTemp,vecGrowPdb,resNo);
	    	vdwVec=new Vector();
	    	vdwVec = vander.convert2VDW(vecTemp);
	    	int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.4);
        	if(minClashes>numClashes)
        	{
        		minClashes=numClashes;
        		minID=i;
        	}
	    	
	    	
	    }//for (int i=0;i<pdbRotamNewStr.size();i++)
	    
	    //debug
	    //output status:
	    if (Const.isDebug)
	    	System.out.println("depth: "+ depth+ " current reside id: "+resNo+ " rotamer ID: "+minID);
	    
	    Pdb ppRotNew=(Pdb)pdbRotamNewStr.elementAt(minID);
    	Vector vecRotTempNew=new Vector();
    	vecRotTempNew.add(ppRotNew);
    	Vector vecTempNew=new Vector();
    	vecTempNew=pp.CombinRotBB(vecRotTempNew,vecGrowPdb,resNo);
    	vecGrowPdb=new Vector();
    	
    	vecGrowPdb.addAll(vecTempNew);
    	
    	boolean [] resClash=new boolean[140];
    	
    	int numRes=vander.OutputStericClashResidues(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.4, 1, resClash,true);
    	
    	if(numRes<=4 || depth>=100)//////changed heree
    	{
    		return vecGrowPdb;
    	}
    	int rnumber = generator.nextInt(resClash.length);
    	int nextResNo=-1;
    	for (int i=resNo+1;i<resClash.length;i++)
	    {
	    	if(resClash[i]==true)
	    	{
	    		nextResNo=i;
	    		break;
	    	}
	    }
    	if(nextResNo<0)
    	{
    		int firstClash=0;
        	for (int i=0;i<resClash.length;i++)
        	{
        		if (resClash[i])
        		{
        			firstClash=i;
        			break;
        		}
        	}
    		nextResNo=firstClash;
    	}
    	
		return 	RefineRotamerSelectionSearch(vecGrowPdb,nextResNo,rotSrc,depth+1);
    }

    
   
    /**
     * refine the rotamer selection until no steric clash happens,
     * by enumerating all possible rotamers in the order of NOE matching scores.
     * 
     * @param vecGrowPdb previous structure after placing rotmaers into the backbone
     * @param vecRot4AllResidues vector of rotamers at each residue
     * @param vecSavePdb vector to save returing pdbs
     * @param resInd the res ind
     * @param resNoeIndex the res noe index
     * 
     * @return refined structure.
     */
    public boolean RefineRotamerSelectionRecur(Vector vecGrowPdb,Vector vecRot4AllResidues,
    		int resInd, Vector vecSavePdb,boolean[] resNoeIndex)
    {  
    	Vector vecRefinePdb=new Vector();
    	
    	vdw vander = new vdw();
    	Vector vdwVec = new Vector();
    	double[] vdwValue = new double[1];
    	boolean hasVDW = false;
    	double vdwLevel = 0.05;
    	boolean printVDWViolation = false;    
    	int resIndex=resInd;
		int[] clashResNo=new int[2];   		
		
		if(resNoeIndex[resInd]!=true)
		{
			int index_temp=-1;
			for (int j=resInd+1;j<resNoeIndex.length;j++)
			{				
				if(resNoeIndex[j]==true)
				{
					index_temp=j;
					resIndex=j;
					break;
				}
			}
			
			if (index_temp==-1)
				return false;
		}
    	
		Vector vecRot1=(Vector)vecRot4AllResidues.elementAt(resIndex);
	    int rotNum1=vecRot1.size();
		
		
	    Pdb pp_temp=(Pdb)vecGrowPdb.elementAt(resIndex);
	    int resNo1=pp_temp.getResidueNo();
	    
		Pdb pp=new Pdb();
		boolean isRefined=false;
		for(int i=0; i<vecRot1.size();i++ )
		{			
			Pdb pdbRot=(Pdb)vecRot1.elementAt(i);
			
			Vector vecPdbRot=new Vector();
			vecPdbRot.add(pdbRot);
			Vector vecTemp1=pp.CombinRotBB(vecPdbRot,vecGrowPdb,resNo1);				
			vecGrowPdb=new Vector();
			vecGrowPdb.addAll(vecTemp1);
			
			if (resIndex>=vecGrowPdb.size()-1)
			{
				vdwVec = vander.convert2VDW(vecGrowPdb);
				boolean isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
						true, 0.5, 1, clashResNo); 
				if (!isStericClash)
				{					
					vecSavePdb.add(vecGrowPdb);
					return true;
				}
				else return false;
			}				
			int index_temp=-1;
			for (int j=resIndex+1;j<resNoeIndex.length;j++)
			{
				if(resNoeIndex[j]==true)
				{
					index_temp=j;
					break;
				}
			}
			if (index_temp>=0)
				RefineRotamerSelectionRecur(vecGrowPdb,vecRot4AllResidues,index_temp,vecSavePdb,resNoeIndex);	
			else
			{
				vdwVec = vander.convert2VDW(vecGrowPdb);
				boolean isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
						true, 0.5, 1, clashResNo); 
				if (!isStericClash)
				{					
					vecSavePdb.add(vecGrowPdb);
					return true;
				}
				else return false;			
			}
		}//for(int i=0; i<vecRot1.size();i++ )
	
	    return true;
    	
    }

    
    
    /**
     * assemble two sses with some phi psi deviations in the break point,
     * the assembled structure has minimum rdc rmsd and vdw energy.
     * 
     * @param vecSSEPdb1 the first sse
     * @param vecSSEPdb2 the second sse
     * @param rdcVecNh a vector of NH RDC values
     * @param rdcVecCh a vector of CH RDC values
     * @param Szz the three diagonalized Saupe elements
     * @param Syy the three diagonalized Saupe elements
     * @param brkResNo the brk res no
     * 
     * @return new structure after being assembled.
     * 
     * @throws JampackException the jampack exception
     */
    public Vector AssembleTwoSSEs(Vector vecSSEPdb1, Vector vecSSEPdb2, int brkResNo,
    		final Vector<Dipolar> rdcVecNh, final Vector<Dipolar> rdcVecCh,double Syy, double Szz)throws JampackException
    {
    	double Sxx=-Syy-Szz;
    	double nhRms=0;
    	double chRms=0;
    	Vector vecStrPdb=new Vector();
    	Vector vecPhiPsiSSE1=PhiPsiTotalPdb(vecSSEPdb1); 
    	Vector vecPhiPsiSSE2=PhiPsiTotalPdb(vecSSEPdb2); 
    	double deviation=5.0;
    	double resolution=1.0;
    	ModelRdc mdc=new ModelRdc();
 	    PdbRdc pdc=new PdbRdc();
 	    Model md=new Model();
    	
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
	   	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	 
    	Pdb pdb11=(Pdb)vecSSEPdb1.elementAt(0);
    	Pdb pdb12=(Pdb)vecSSEPdb1.elementAt(vecSSEPdb1.size()-1);
    	int resNo11=pdb11.getResidueNo();
    	int resNo12=pdb12.getResidueNo();
    	
    	Vector atomVec = pdb11.getAtomVec();
	    for (int j=0; j<atomVec.size(); j++)
	    {
	    	Cartesian cc = (Cartesian) atomVec.elementAt(j);
	    	String atom = cc.getAtom();
	    	if (atom.equals("CA"))
	    		ca1 = cc.getXYZ();
	    	if (atom.equals("N"))
	    		n1 = cc.getXYZ();
	    	if (atom.equals("H")|| atom.equals("HN"))
	    		nh1 = cc.getXYZ();
	    }
	   
    	Pdb pdb22=(Pdb)vecSSEPdb2.elementAt(vecSSEPdb2.size()-1);
    	int resNo22=pdb22.getResidueNo();    	
    	
    	Vector vecPhiPsiAll=new Vector();
    	Vector vecPhiPsiFirst=new Vector();
    	double[] phiFirst = new double[vecPhiPsiSSE1.size()]; 
    	double[] psiFirst=new double [vecPhiPsiSSE1.size()]; 
    	for (int i=0; i<vecPhiPsiSSE1.size();i++)
    	{
    		PhiPsi phipsi=(PhiPsi)vecPhiPsiSSE1.elementAt(i);
    		int resNo=phipsi.getResidueNo();
    		double phi_temp=phipsi.getPhi();
    		double psi_temp=phipsi.getPsi();
    		phiFirst[i]=phi_temp;
    		psiFirst[i]=psi_temp;
    		if( (resNo>= resNo11) && (resNo< brkResNo) )
    			vecPhiPsiFirst.add(phipsi);    		
    	}
    	
    	Vector vecPhiPsiSec=new Vector();
    	for (int i=0; i<vecPhiPsiSSE2.size();i++)
    	{
    		PhiPsi phipsi=(PhiPsi)vecPhiPsiSSE2.elementAt(i);
    		int resNo=phipsi.getResidueNo();
    		if( (resNo> brkResNo) && (resNo<= resNo22) )
    			vecPhiPsiSec.add(phipsi);    		
    	}    	
   
    	Vector phiPsiVec_temp=new Vector();
    	for (int i=resNo11; i<=resNo12; i++)  
    		phiPsiVec_temp.add(new PhiPsi(i, "ALA", Const.phiAveHelix, Const.psiAveHelix));
    		
    	double [] n1_temp  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
    	double [] nh1_temp = {0.0, 0.0, -Const.dN2H}; 
        double [] ca1_temp = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};

         	
    	int brkIndex= Collections.binarySearch(vecSSEPdb1, new Pdb(brkResNo), new Pdb.PdbComparator());
 	    if(brkIndex<0)
 	    {
 	    	System.out.println("We couldnot find the residue Index");
 	    	return null; 	    	
 	    }
 	    
 	    PhiPsi phipsiBrk=(PhiPsi)vecPhiPsiSSE1.elementAt(brkIndex);
 	    String brkResName=phipsiBrk.getResidue();
 	    
 	    double phiBrk=phipsiBrk.getPhi();
 	    double psiBrk=phipsiBrk.getPsi();
 	    double phiBrkDegree=phiBrk*180.0/Math.PI; //change it to the degree format
 	    double psiBrkDegree=psiBrk*180/Math.PI;	   
 	   
 	    vdw vander = new vdw();
 	    Vector vdwVec = new Vector();
 	    double[] vdwValue = new double[1];
 	    boolean hasVDW = false;
 	    double vdwLevel = 0.05;
 	    boolean printVDWViolation = false;       	
 	    
 	    double minScore=99999.9;
 	    double [][] eMat    = {{1.0, 0.0, 0.0},  {0.0, 1.0, 0.0},{0.0, 0.0, 1.0}}; 
 	    Matrix A = new Matrix(eMat);
 	    double[] sRmsdCH= new double[1];
 		int[] sizeCH=new int[1];
 		double[] sRmsdNH=new double[1];
 		int[] sizeNH=new int[1];
 		Vector vecPdbBest=new Vector();
 		
 		for (int i= (int )(phiBrkDegree-deviation);i<=Math.ceil(phiBrkDegree+deviation);i++)
		{
			double phiValue=i*resolution* Math.PI / 180.0;
			System.out.println("i="+i+" phiValue="+phiValue);
			for (int k=(int) (psiBrkDegree-deviation);k<=psiBrkDegree+deviation;k++)
			{
				double psiValue=k*resolution* Math.PI / 180.0;
				System.out.println("k="+k+" phiValue="+psiValue);
				vecPhiPsiAll=new Vector();
				vecPhiPsiAll.addAll(vecPhiPsiFirst);
				vecPhiPsiAll.add(new PhiPsi(brkResNo, brkResName, phiValue, psiValue)); 
				vecPhiPsiAll.addAll(vecPhiPsiSec);
				
				vecStrPdb=new Vector();
				vecStrPdb=mdc.modelBuild(vecPhiPsiAll, n1_temp, nh1_temp, ca1_temp);
				Collections.sort(vecStrPdb, new Pdb.PdbComparator());
				Vector vecStrPdbNew=new Vector();
				vecStrPdbNew=md.BackonbeRotation(vecStrPdb,vecSSEPdb1);
				
				Collections.sort(vecStrPdbNew, new Pdb.PdbComparator());
				
				vdwVec = vander.convert2VDW(vecStrPdbNew);
	        	hasVDW = vander.computeVDW(vdwVec, vdwValue, vdwLevel, printVDWViolation, false); 
	        	
	        	//calculate the rdc rmsd:
	        	double[] rdc1Cal  = new double[vecStrPdbNew.size()];
	        	double[] rdc2Cal  = new double[vecStrPdbNew.size()];
	        	nhRms = pdc.BackCalNH(vecStrPdbNew, rdcVecNh, "N", "H",  A, Sxx, Syy, Szz, Const.nhRatio, rdc1Cal,sRmsdNH, sizeNH);
	        	chRms = pdc.BackCal(vecStrPdbNew, rdcVecCh, "CA", "HA", A, Sxx, Syy, Szz, Const.cahaRatio, rdc2Cal,sRmsdCH, sizeCH );
				double sc=nhRms+chRms;//vdwValue[0]+
				
				if(minScore>sc)
				{
					System.out.println("score=: "+ sc+ "  vdw= "+ vdwValue[0]+ " nhRms="+nhRms+ " chRms="+chRms);
					vecPdbBest=new Vector();
					vecPdbBest.addAll(vecStrPdbNew);
					minScore=sc;
				}
			}//for (int k=(int) (psiBrkDegree-deviation);k<=psiBrkDegree+deviation;k++)
		}//for (int i= (int )(phiBrkDegree-deviation);i<=Math.ceil(phiBrkDegree+deviation);i++)
				
 		return vecPdbBest;	   	
    }
    
    
    /**
     * Compute a structure, given the backbone and rotamers at all residue positions.
     * 
     * @param csErrH csErrN csErrCA error window size at each dimension
     * @param seqVec vector of protein sequence
     * @param vecPdbBbSSE the secondary structure backbone structure solved from RDC-EXACT
     * @param asgVec    resonace list
     * @param rotSrc  the path of rotmaer library
     * @param NoesyVec  Noe vector including both ha and hn noes
     * @param distBound  the distance bound for back computing distance pattern for each rotamer
     * @param firstResNo  the first residue no of the loop
     * @param secondResNo  the first residue no of the loop
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * 
     * @return final structure.
     */
    public Vector LoopComputFromNOEPatterns(double csErrH, double csErrN, double csErrCA,
    		Vector seqVec, Vector vecPdbBbSSE, Vector asgVec, String rotSrc, 
    		Vector NoesyVec,double distBound, int firstResNo, int secondResNo)
    {
    	int i,j,k,h;
    	double resolution=4.0;
   
	    double phiValue,psiPreValue;
	    Vector phiPsiVec;
	    ModelRdc mdc = new ModelRdc();
	    Vector pdbLoopVec=new Vector();
	    Vector vecGrowPdb=new Vector();
    	vecGrowPdb=vecPdbBbSSE;//initial sse structure
    	Pdb pp=new Pdb();
    	Vector pdbRotamNewRef=new Vector();
    	RotaPattern rp=new RotaPattern();
    	//get phi value in the previous residue
    	Vector vecPhiPsiSSE=PhiPsiTotalPdb(vecPdbBbSSE);
    	double prePhi=0.0;
    	for(i=0;i<vecPhiPsiSSE.size();i++)
    	{
    		PhiPsi phipsi=(PhiPsi)vecPhiPsiSSE.elementAt(i);
    		if(phipsi.getResidueNo()==(firstResNo-1))
    			prePhi=phipsi.getPhi();    		
    	}
    	for(i=firstResNo;i<secondResNo+1;i++)
    	{
    		//compute the coordinates of the first peptide
    		int index = Collections.binarySearch(vecGrowPdb, new Pdb(i-1), new Pdb.PdbComparator());
     	    if(index<0)
     	    {
     	    	System.out.println("We couldnot find the residue Index");
     	    	return null; 	    	
     	    }
    		Pdb pdbFirst = (Pdb)vecGrowPdb.elementAt(index);
     	    Vector atomVec = new Vector();
     	    Cartesian cc = new Cartesian();
     	    double [] n1 = {0.0, 0.0, 0.0};       
     		double [] nh1 = {0.0, 0.0, 0.0};  
     	    double [] ca1 = {0.0, 0.0, 0.0};  
     	    atomVec = pdbFirst.getAtomVec();
    	    for (j=0; j<atomVec.size(); j++)
    	    {
    	    	cc = (Cartesian) atomVec.elementAt(j);
    	    	String atom = cc.getAtom();
    	    	if (atom.equals("CA"))
    	    		ca1 = cc.getXYZ();
    	    	if (atom.equals("N"))
    	    		n1 = cc.getXYZ();
    	    	if (atom.equals("H")|| atom.equals("HN"))
    	    		nh1 = cc.getXYZ();
    	    }
    	        		
    	    //pdb that doens't include current residue
    	    Vector vecGrowPdbNew=new Vector();
    	    for(int t=0;t<vecGrowPdb.size();t++)
    	    {
    	    	Pdb pdbTemp=(Pdb)vecGrowPdb.elementAt(t);
    	    	int resNoTemp=pdbTemp.getResidueNo();
    	    	if( !(resNoTemp>=(i-1)&& resNoTemp<=secondResNo) )
    	    		vecGrowPdbNew.add(pdbTemp);   	    	
    	    }
    	    
    	    //find the phi-psi angles that have max score
    		double maxScore=-999999.9;
    		double phiMax=0.0;
    		double psiPreMax=0.0;
    		phiValue=0.0;
    		psiPreValue=0.0;
    		phiPsiVec=new Vector();
    		String residPre=residueNameFromSeq(seqVec, i-1);
    		String residCur=residueNameFromSeq(seqVec, i);
    		Vector vecTemp;//template pdb
    		for (j=0;j<360/resolution;j++)
    		{
    			psiPreValue=j*resolution* Math.PI / 180.0;
    			System.out.println("j="+j+" psiPreValue="+psiPreValue);
    			for (k=0;k<360/resolution;k++)
    			{
    				phiValue=k*resolution* Math.PI / 180.0;
    				System.out.println("k="+k+" phiValue="+phiValue);
    				
    				pdbLoopVec=new Vector();
    				phiPsiVec=new Vector();
    				phiPsiVec.add(new PhiPsi(i-1, residPre, prePhi, psiPreValue)); 
    				phiPsiVec.add(new PhiPsi(i, residPre, phiValue, 0.0));
    				pdbLoopVec = mdc.modelBuild(phiPsiVec, n1, nh1, ca1);
    				vecTemp=new Vector();
    				vecTemp.addAll(vecGrowPdbNew);//template pdb
    				vecTemp.addAll(pdbLoopVec);
    				Collections.sort(vecTemp, new Pdb.PdbComparator());
    				
    				//using the NOE patterns
    				Pdb pdb=(Pdb)pdbLoopVec.elementAt(1);
    	        	String resName=pdb.getResidue();
    	        	int resNo=pdb.getResidueNo();
    	        	Vector pdbRotVec=new Vector();
    	        		
    	        	if( resName.equalsIgnoreCase("GLY"))
    	        	{
    	        		int ind1 = Collections.binarySearch(vecTemp, new Pdb(resNo), new Pdb.PdbComparator());
    	        		Pdb pdbA = (Pdb)vecTemp.elementAt(ind1);    			
    	        		pdbRotVec.add(pdbA);        			
    	        	}
    	        	else
    	        	{
    	        		String rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
    	        		Vector pdbRotam=pp.readPdb(rotamFile);
    	        		//atom names are consistet with structure, namely HB1,HB2...
    	        		Vector pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
    	        		//atom names are consistent with chemical shifts, eg, HB for HB#
    	            	Vector pdbRotamNewCS=pp.nameConvert4SmallRotLibCS(pdbRotam);
    	            	pdbRotVec=pdbRotamNewCS;//important here...	    		
    	        	}   		
    	        		
    	        	//int indRef = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
    	        	double [] index_temp=new double[2];    	        		
    	        	pdbRotamNewRef=rp.RefineRotLib(csErrH,csErrN, csErrCA,pdbRotVec,vecTemp,asgVec, NoesyVec,resNo,distBound, 1,index_temp);
    	        	if(maxScore<index_temp[1])
    	        	{
    	        		maxScore=index_temp[1];
    	        		phiMax=phiValue;
    	        		psiPreMax=psiPreValue;
    	        	}
    				
    			}//for (k=0;k<360/resolution;k++)    			
    		}//for (j=0;j<360/resolution;j++)
    		
    		System.out.println("i-1="+(i-1)+" psiPreMax="+psiPreMax+" i="+i+ " phiMax="+phiMax);
    		vecGrowPdb=new Vector();
    		vecGrowPdb.addAll(vecGrowPdbNew);
    		pdbLoopVec=new Vector();
			phiPsiVec=new Vector();
			phiPsiVec.add(new PhiPsi(i-1, residPre, prePhi, psiPreMax)); 
			phiPsiVec.add(new PhiPsi(i, residCur, phiMax, 0.0)); 
			pdbLoopVec = mdc.modelBuild(phiPsiVec, n1, nh1, ca1);
			vecGrowPdb.addAll(pdbLoopVec);
			Collections.sort(vecGrowPdb, new Pdb.PdbComparator());   	
			prePhi=phiMax;
    		
    	}//for(i=firstResNo;i<secondResNo+1;i++)
    	
    	return vecGrowPdb;    	
    }
     	
    
    /**
     * Compute the NOE distance between two nuclei (A, B) in a
     * structure specified by pdbVec There are two numbers: 15 Degree
     * and 45 Degree for correct the most possible orientation changing.
     * The number themselves are set rather arbitrariry. Reaaly need
     * search through PDB to compute them.
     * 
     * @param pdbVec the PDB structure
     * @param noA     the residue NO of nucleus A
     * @param residueA  the residue type of nucleus A
     * @param nucleusA  the name of the nucleus of residue A
     * @param noB     the residue NO of nucleus A
     * @param residueB  the residue type of nucleus B
     * @param nucleusB  the name of the nucleus of residue B
     * @param distance  save the back-computed NOE distance
     * @param rotamVecA the rotam vec a
     * @param rotamVecB the rotam vec b
     * 
     * @return true if such a distance can be computed in the structure, false if there is no
     * such nuclei in the structure.
     */
    public boolean measureDisFromRotam(final Vector pdbVec, final Vector rotamVecA, final Vector rotamVecB,
    		int noA, String residueA, String nucleusA, int noB, 
			      String residueB, String nucleusB, double[] distance)
    {
    	int i,j;
		int ind1 = Collections.binarySearch(pdbVec, new Pdb(noA), new Pdb.PdbComparator());
		int ind2 = Collections.binarySearch(pdbVec, new Pdb(noB), new Pdb.PdbComparator());
		if (ind1 < 0 || ind2 < 0)
		    return false;
		if (nucleusA.equalsIgnoreCase("HN")) //We need to convert all the CS name to PDB name
		    nucleusA = "H";
		if (nucleusB.equalsIgnoreCase("HN"))
		    nucleusB = "H";
	
		Pdb pdb=new Pdb();
		
		Pdb pdbA = (Pdb)pdbVec.elementAt(ind1);
		
		Vector pdbAVec=new Vector();
		pdbAVec.add(pdbA);
	
		String atom = "";
		Vector atomVecA = pdbA.getAtomVec();
		
		Vector atomVec=new Vector();
		Cartesian cc = new Cartesian();
		double [] amide = new double[3];
		double [] amide1 = new double[3];
		double [] amide2 = new double[3];
		//double [] nh = new double[3];
		double [] ha = new double[3];
		double [] ca1 = new double[3];
		double [] ca2 = new double[3];
		double [] trans_vec = new double[3];

		PhiPsi ff = new PhiPsi();
		
		//find the pdb of rotamers at the 1st position
		Matrix rgA = Matrix.identity(3,3);
		
		for ( j=0; j<atomVecA.size(); j++)
		{ 
		    cc = (Cartesian)atomVecA.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
			amide = cc.getXYZ();
		    else if (atom.equals("CA"))
		    	ca1=cc.getXYZ();
		    else if (atom.equals("HA")||atom.equals("HA2") )
			ha = cc.getXYZ();		    
		}
			
		double [] CaToHaVecStr =  internuclearVec(ca1,  ha);
		
		double [] CaToNVecStr =  internuclearVec(ca1,  amide);
		Matrix rg = Matrix.identity(3,3);
		rg = ff.RgCal(CaToHaVecStr, CaToNVecStr);		
	
		Vector pdbANew=pdb.newPdb(pdbAVec,rg);//the new coordinate of str after being rotated
		
		Vector pdbVecNew1=pdb.newPdb(pdbVec,rg);//the whole structure after the frist rotation
	
		Pdb pdbTemp = (Pdb)pdbANew.elementAt(0);
		atomVec=pdbTemp.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("CA"))
			ca1 = cc.getXYZ();		    
		} 
	
		
		Pdb pdbRotA=(Pdb)rotamVecA.elementAt(0);
		atomVec=pdbRotA.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
		    	amide = cc.getXYZ();
		    else if (atom.equals("CA"))
		    	ca2=cc.getXYZ();
		    else if (atom.equals("HA")||atom.equals("HA2") )
		    	ha = cc.getXYZ();    
		}
		double [] CaToHaVecRot =  internuclearVec(ca2,  ha);
		double [] CaToNVecRot =  internuclearVec(ca2,  amide);//////
		rgA = ff.RgCal(CaToHaVecRot, CaToNVecRot);
		
		Vector rotamVecNewA_temp=pdb.newPdb(rotamVecA,rgA);
		
		pdbTemp = (Pdb)rotamVecNewA_temp.elementAt(0);
		atomVec=pdbTemp.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("CA"))
			ca2 = cc.getXYZ();		    
		} 		
		
		//the translation vector after putting into the same frame
		trans_vec= internuclearVec(ca2,ca1);				
		
		Vector<Pdb> rotamVecNewA=pdb.newPdbByTranslation(rotamVecNewA_temp,trans_vec);
	
		///////////////////////////////////////////////////////////////////////
		//find the pdb of the rotamers at the 2nd position
		Pdb pdbB = (Pdb)pdbVecNew1.elementAt(ind2);
		Vector pdbBVec=new Vector();
		pdbBVec.add(pdbB);
		Vector atomVecB = pdbB.getAtomVec();
		
		Matrix rgB = Matrix.identity(3,3);
		
		for ( j=0; j<atomVecB.size(); j++)
		{ 
		    cc = (Cartesian)atomVecB.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
		    	amide = cc.getXYZ();
		    else if (atom.equals("CA"))
		    	ca1=cc.getXYZ();
		    else if (atom.equals("HA")||atom.equals("HA2") )
		    	ha = cc.getXYZ();	    
		}
	
		CaToHaVecStr =  internuclearVec(ca1,  ha);
		CaToNVecStr =  internuclearVec(ca1,  amide);///
		
		rg = ff.RgCal(CaToHaVecStr, CaToNVecStr);
		Vector pdbBNew=pdb.newPdb(pdbBVec,rg);		
		
		pdbTemp = (Pdb)pdbBNew.elementAt(0);/////
		atomVec=pdbTemp.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("CA"))
			ca1 = cc.getXYZ();
		} 		
		//////////////////////////////////////////////
		//here we also need to rotate the structure residue at A position
		Vector rotamVecNewA2=pdb.newPdb(rotamVecNewA,rg);
		Vector pdbVecNew2=pdb.newPdb(pdbVecNew1,rg);//the whole structure after the frist rotation
		
		Pdb pdbRotB=(Pdb)rotamVecB.elementAt(0);
		atomVec=pdbRotB.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
			amide = cc.getXYZ();
		    else if (atom.equals("CA"))
		    	ca2=cc.getXYZ();
		    else if (atom.equals("HA") ||atom.equals("HA2"))
			ha = cc.getXYZ();    
		}
		CaToHaVecRot =  internuclearVec(ca2,  ha);
		CaToNVecRot =  internuclearVec(ca2,  amide);
		rgB = ff.RgCal(CaToHaVecRot, CaToNVecRot);
		Vector rotamVecNewB_temp=pdb.newPdb(rotamVecB,rgB);
		
		pdbTemp = (Pdb)rotamVecNewB_temp.elementAt(0);
		atomVec=pdbTemp.getAtomVec();
		for ( j=0; j<atomVec.size(); j++)
		{ 
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("CA"))
			ca2 = cc.getXYZ();		   
		} 	
		trans_vec= internuclearVec(ca2,ca1);
		Vector<Pdb> rotamVecNewB=pdb.newPdbByTranslation(rotamVecNewB_temp,trans_vec);
		
		
		//compute the distance between rotamers at position1 and position2
		double minDis=999.9;
		double disTemp=0.0;
		Pdb pdbAA,pdbBB;
		int i1,j1;
		double [] coordA = {0.0, 0.0, 0.0};
		double [] coordB = {0.0, 0.0, 0.0};
		boolean foundA = false;
		boolean foundB = false;
		
		for (i=0;i<rotamVecNewA2.size();i++)
		{
			pdbAA=(Pdb)rotamVecNewA2.elementAt(i);
			atomVecA=pdbAA.getAtomVec();
			for (i1=0; i1<atomVecA.size();i1++)
			{
				cc = (Cartesian) atomVecA.elementAt(i1);
			    atom = cc.getAtom();
			    if (atom.equalsIgnoreCase("HN")) //We need to convert all the CS name to PDB name
			    	atom = "H";
			    
			    if (atom.equalsIgnoreCase(nucleusA))
			    {
					coordA = cc.getXYZ();
					foundA = true;
			    }
			}//for (i1=0; i1<atomVecA.size();i1++)
			
			for (j=0; j<rotamVecNewB.size();j++)
			{
				pdbBB=rotamVecNewB.elementAt(j);
				atomVecB=pdbBB.getAtomVec();
				for (j1=0;j1<atomVecB.size();j1++)
				{
					cc = (Cartesian) atomVecB.elementAt(j1);
					atom = cc.getAtom();
					if (atom.equalsIgnoreCase("HN")) //We need to convert all the CS name to PDB name
					    atom = "H";
					if (atom.equalsIgnoreCase(nucleusB))
					{
						coordB = cc.getXYZ();
						foundB = true;
						
						//in case there are three same protons in methyl group
						disTemp = Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
								+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
								+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
						if (minDis>disTemp)
							minDis=disTemp;
					}
				}//for (j1=0;j1<atomVecB.size();j1++)
				
				disTemp = Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
						+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
						+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
				if (minDis>disTemp)
					minDis=disTemp;
				
			}//for (j=0; j<rotamVecNewB.size();j++)
				
			
		}//for (i=0;i<rotamVecNewA.size();i++)
			
		
		distance[0]=minDis;
		if ( foundA && foundB )
		{
			 //  	    System.out.println("Can NOT find "+nucleusB);
			 	    return true;
		}
		return false;
		
	}

    /**
     * A method to compute all the angles from ultrahigh resolution PDBs.
     * 
     * @param pdbLists the pdb lists
     */
    public void statsOfSixAngles(String pdbLists){
	String str, ss;
  	String src = "/net/drizzle/wlincong/pdbLocal/pdb/ultraHPdb/";
	PhiPsi ff = new PhiPsi();
	Pdb pp = new Pdb();

      	Vector pdbVec = new Vector();
  	Vector fileVec = new Vector();
  	Vector phiPsiRdc = new Vector();
	Vector angles = new Vector();
       	Vector allAngles = new Vector();
	try{
	    BufferedReader in =
            	new BufferedReader( new FileReader(pdbLists));
	    while( (ss = in.readLine()) != null) {
		StringTokenizer st = new StringTokenizer(ss);
		while (st.hasMoreTokens())
		    fileVec.add(st.nextToken());
	    }
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + pdbLists);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	double theta1 = 0.0, theta3 = 0.0, theta5 = 0.0, theta6 = 0.0, theta7 = 0.0, theta9 = 0.0;

	for (int i=0; i<fileVec.size(); i++){
	    pdbVec =  pp.readPdb(src+(String)fileVec.elementAt(i));
	    angles = sixAngles(pdbVec);
	    allAngles.addAll(angles);
	}
	double theta = 0.0;
	final int sampleSize = allAngles.size();
	System.out.println(sampleSize);
	double sum = 0.0;
	int N = 0;
	for (int j=0; j<allAngles.size(); j++){
	    theta = ((Double)allAngles.elementAt(j)).doubleValue();
   	    if ( theta < 2.5){

  		sum = sum + theta;
  		N++;
  	    }
 	}
  	System.out.println("Theta = "+ (sum *180.0/ (N*Math.PI) ));

    }

    /**
     * We can also derived the minimum and maximam using analytic expression but
     * using rotamer as appeared in PDB may have certain advantage.
     * 
     * @param pdbVec is an array of PDB object
     * @param residue the type of residue
     * @param hAtom the h atom
     * 
     * @return the vector
     */
    public Vector radiiFromCb(Vector pdbVec, String residue, String hAtom){
	int no = 0, i = 0, j = 0;
	String resid = "";
	Cartesian cc = new Cartesian();
	String atom = "";
	Pdb pp = new Pdb();
	double [] cb = new double[3];
	double [] h1 = new double[3];
	double [] d1 = new double[3];
	Vector disVec = new Vector();
	double len = 0.0;
	boolean hasCB = false;
	for (i =0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = pp.getResidue();
	    atomVec = pp.getAtomVec();
	    if (resid.equals(residue)){
		for (j=0; j<atomVec.size(); j++){
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("CB")){
			cb = cc.getXYZ();
			hasCB = true;
		    }
		}
		if(hasCB){
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.indexOf(hAtom) > - 1){
			    h1 = cc.getXYZ();
			    d1 = internuclearVec(cb, h1);
			    len = length(d1);
			    disVec.add(new Double(len));
			}
		    }
		}
	    }
	}
	return disVec;
    }

    /**
     * Compute the distance from the nucleu NH to other nucleu.
     * We can also derived the minimum and maximam using analytic expression but
     * using rotamer as appeared in PDB may have certain advantage
     * 
     * @param pdbVec is an array of PDB object
     * @param residue the type of residue
     * @param hAtom the h atom
     * 
     * @return the vector
     */
    public Vector radiiFromNH(Vector pdbVec, String residue, String hAtom){
	int no = 0, i = 0, j = 0;
	String resid = "";
	Cartesian cc = new Cartesian();
	String atom = "";
	Pdb pp = new Pdb();
	double [] cb = new double[3];
	double [] h1 = new double[3];
	double [] d1 = new double[3];
	Vector disVec = new Vector();
	double len = 0.0;
	boolean hasCB = false;
	for (i =0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    resid = pp.getResidue();
	    atomVec = pp.getAtomVec();
	    if (resid.equals(residue)){
		for (j=0; j<atomVec.size(); j++){
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("H")){
			cb = cc.getXYZ();
			hasCB = true;
		    }
		}
		if(hasCB){
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.indexOf(hAtom) > - 1){
			    h1 = cc.getXYZ();
			    d1 = internuclearVec(cb, h1);
			    len = length(d1);

			    disVec.add(new Double(len));
			}
		    }
		}
	    }
	}
	return disVec;
    }

   /**
    * pdbVec is a Vector of Pdb Objects e.g. that from the pdbRead method
    * return the all the OmegaNH angle of the peptide plane as a vector
    * The program may have bugs if some seq no are missing!!!!
    * We want to compute all the relavant angles. Six plus angles for HA and CO, CACB vectors
    * 
    * @param pdbVec the pdb vec
    * 
    * @return the vector
    */
    public Vector sixAngles(final Vector pdbVec){
	double [] coI = new double[3];//For the very 1st residue
	double [] amide1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh1 = new double[3];
	double [] nh2 = new double[3];
	double [] ha1 = new double[3]; //For HA and O
	double [] ha2 = new double[3]; //For HA and O
	double [] o1 = new double[3];
	double [] o2 = new double[3];
	double [] co1 = new double[3];
	double [] co2 = new double[3];
	double [] ca1 = new double[3];
	double [] ca2 = new double[3];

	double [] nToCO1Vec = new double[3];
	double [] co1ToNVec = new double[3];
	double [] caToCOVec = new double[3];
	double [] coToCAVec = new double[3];
	double [] nToCAVec = new double[3];
	double [] caToNVec = new double[3];
	double [] nToNHVec = new  double[3];
	double [] nToNHVec2 = new  double[3];
	double [] caToHAVec = new double[3];  
	double [] coToOVec = new double[3];
	double dCO2O = 0.0, angleNCAHA, dCA2HA, angleOCON, omega1;
	int no1 = 0, no2 = 0, no3 = 0, i = 0, j = 0, k = 0;
	String resid1 = "", resid2 = "", resid3="";
	Cartesian cc = new Cartesian();
	String atom = "";
	double theta1 = Math.PI/6.0, theta3 = -Math.PI/9.0, theta5, theta6, theta7, theta9=0.0;
	Matrix rg = new Matrix(3,3);
	final double deg = 180.0/Math.PI;
	Vector phiPsiVec = new Vector();
	Vector theta1Vec = new Vector();
	Vector theta3Vec = new Vector();
	Vector theta5Vec = new Vector();
	Vector theta6Vec = new Vector();
	Vector theta7Vec = new Vector();
	Vector theta9Vec = new Vector();
	Vector angleNCAHAVec = new Vector();
	Vector angleOCONVec = new Vector();
	Vector omegaVec = new Vector();
	Vector angleNCACOHAVec = new Vector();
	double angleNCACOHA = 0.0;
	double[] ppp = new double[4];
	//1st Residue of the entire sequence
	Pdb pp = (Pdb)pdbVec.elementAt(i); 
	no1 = pp.getResidueNo();
	resid1 = pp.getResidue();

	Vector atomVec = pp.getAtomVec();
	//the Atoms N, CA and CO of residue i
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if(atom.equals("C"))
		coI = cc.getXYZ();
	}
	i++;       	// Beginning With Second Residue
	pp = (Pdb)pdbVec.elementAt(i);
	no2 = pp.getResidueNo();
	resid2 = pp.getResidue();
	atomVec = pp.getAtomVec();
	if ( ! (resid2.equals("PRO"))){
	    for (j=0; j<atomVec.size(); j++){ 
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("N"))
		    amide1 = cc.getXYZ();
		else if (atom.equals("H"))
		    nh1 = cc.getXYZ();
		else if (atom.equals("CA"))
 		    ca1 = cc.getXYZ();
		else if (atom.equals("C"))
		    co1 = cc.getXYZ();
		else if (atom.equals("HA"))
		    ha1 = cc.getXYZ();
		else if (atom.equals("O"))
		    o1 = cc.getXYZ();
	    }
	    //intraResidue N(i)->NH(i) of the 2nd residue
	    nToNHVec = internuclearVec(amide1, nh1);
	    //N(i)->CA(i) vector
	    nToCAVec = internuclearVec(amide1, ca1);

	    caToHAVec = internuclearVec(ca1, ha1); 
	    angleNCAHA = Math.PI - interAngle(nToCAVec, caToHAVec);
	    angleNCAHAVec.add(new Double(angleNCAHA));
	    dCA2HA = length(caToHAVec);

	    coToOVec = internuclearVec(o1, co1);
	    dCO2O = length(coToOVec);

	    theta1 = interAngle(nToNHVec, nToCAVec) - Math.PI/2.0;
	    theta1Vec.addElement(new Double(theta1));
	    //CA(i)->CO(i) vector
	    caToCOVec = internuclearVec(ca1, co1);
	    coToCAVec = internuclearVec(co1, ca1);
	    theta3 = interAngle(nToCAVec, caToCOVec) - Math.PI/2.0;
	    theta3Vec.addElement(new Double(theta3));

	    angleNCACOHA = PhiPsiCalPDB(caToCOVec, nToCAVec, caToHAVec);
	    angleNCACOHAVec.add(new Double(angleNCACOHA));

	    if ( no2 == (no1+1) ){
		//Cal the interResidue CO(i-1)->N(i) between the 1st and the 2nd 
		co1ToNVec = internuclearVec(coI, amide1);
		//CO(i-1)->N(i), N(i)->CA(i), N(i)->NH(i)
		theta9 = PhiPsiCalPDB(nToNHVec, nToCAVec, co1ToNVec);
		theta9Vec.addElement(new Double(theta9));
	    }
	}else{ //the 2nd residue is a Proline
	    for (j=0; j<atomVec.size(); j++){ //The atoms N, CA and C of residue i+1
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
	        if (atom.equals("C"))
		    co1 = cc.getXYZ();
	    }
	}
        i++; //Begin with the third residue we go into the While loop
	while (i < pdbVec.size()){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no3 = pp.getResidueNo();
	    resid3 = pp.getResidue();

	    atomVec = pp.getAtomVec();
	    if ( ! (resid3.equals("PRO"))){
		for (j=0; j<atomVec.size(); j++){
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("N"))
			amide1 = cc.getXYZ();
		    else if (atom.equals("H"))
			nh1 = cc.getXYZ();
		    else if (atom.equals("CA"))
			ca1 = cc.getXYZ();
		    else if (atom.equals("C"))
			co1 = cc.getXYZ();
		    else if (atom.equals("HA"))
			ha1 = cc.getXYZ();
		    else if (atom.equals("O"))
			o1 = cc.getXYZ();
		}
		//Cal the intraResidue N(i+1)->NH(i+1) 
		nToNHVec2 = internuclearVec(amide2, nh2);
//  		//N(i+1)->CA(i+1) vector
		nToCAVec = internuclearVec(amide2, ca2);
		caToHAVec = internuclearVec(ca2, ha2); 
		angleNCAHA = Math.PI - interAngle(nToCAVec, caToHAVec);
		angleNCAHAVec.add(new Double(angleNCAHA));
		dCA2HA = length(caToHAVec);

		if ( no3 == (no2+1) ){
		    //Cal the interResidue CO(i)->N(i+1) between the 1st and the 2nd 
		    co1ToNVec =  internuclearVec(co1, amide2);
		    angleOCON = Math.PI - interAngle(coToOVec, co1ToNVec);
		    angleOCONVec.add(new Double(angleOCON));


		    //The angle between CO(i)->CA(i) and CO(i)->N(i+1)
		    theta5 = Math.PI / 2.0 - interAngle(caToCOVec, co1ToNVec);
		    omega1 =  PhiPsiCalPDB(co1ToNVec, caToCOVec, coToOVec);
		    omegaVec.add(new Double(omega1));

		    coToOVec = internuclearVec(o2, co2);   //The third residue
		    dCO2O = length(coToOVec);


		    theta6 = PhiPsiCalPDB(coToCAVec, co1ToNVec, nToNHVec2);
		    theta7 = Math.PI / 2.0 - interAngle(co1ToNVec, nToNHVec2);
		    theta5Vec.addElement(new Double(theta5));
		    theta6Vec.addElement(new Double(theta6));
		    theta7Vec.addElement(new Double(theta7));
		    //CO(i-1)->N(i), N(i)->CA(i), N(i)->NH(i)
		    if ( !resid2.equals("PRO") ){
			theta9Vec.addElement(new Double(theta9));
		    }
		    theta9 = PhiPsiCalPDB(nToNHVec2, nToCAVec, co1ToNVec);
		}	
		for (k=0; k<nToNHVec2.length; k++)  //update co2
		    nToNHVec[k] = nToNHVec2[k];
		theta1 = interAngle(nToNHVec, nToCAVec) - Math.PI/2.0;
		theta1Vec.addElement(new Double(theta1));
		//update to CA(i+1)->CO(i+1)
		caToCOVec = internuclearVec(ca2, co2);
		coToCAVec = internuclearVec(co2, ca2);
		theta3 = interAngle(nToCAVec, caToCOVec) - Math.PI/2.0;
		theta3Vec.addElement(new Double(theta3));
		angleNCACOHA = PhiPsiCalPDB(caToCOVec, nToCAVec, caToHAVec);
		angleNCACOHAVec.add(new Double(angleNCACOHA));
	    }else{
		for (j=0; j<atomVec.size(); j++){ //The atoms N, CA and C of residue i+1
		    cc = (Cartesian)atomVec.elementAt(j);
		    atom = cc.getAtom();
		    if (atom.equals("C"))
			co2 = cc.getXYZ();
		}
	    }
	    no2 = no3;   //Updating the residueNO and ResidueType for checking
	    resid2 = resid3;
	    for (k=0; k<co2.length; k++)  //update co2
		co1[k] = co2[k];
	    i++;      
	}
	return angleNCACOHAVec;

    }
	
    /**
     * A method for printing the coordinates in PDB format
     * the magic Number comes from PDB file format.
     * 
     * @param pdbVec a vector of Pdb object
     * @param fileName the file name
     * @param out the out
     */
    public void printToFile(Vector pdbVec,String fileName,PrintWriter out) {
	final int d = 3;              //number of digits after the decimal
	int w = 0;
	DecimalFormat format = new DecimalFormat();
	format.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.US));
	format.setMinimumIntegerDigits(1);
	format.setMaximumFractionDigits(d);
	format.setMinimumFractionDigits(d);
	format.setGroupingUsed(false);
      	Vector atomVec = new Vector();
	Cartesian cc = new Cartesian();
	Pdb pp = new Pdb();
	String noS;
	int curNo,preNo;
	String resid, atom;
	double x,y,z;
	String sX, sY, sZ;
	int padding = 0;
	String atomStr;
	int i, j, k;
	int atomNo=0;
	String str="";
	Pdb ppTem=(Pdb)pdbVec.elementAt(0);
	curNo=ppTem.getResidueNo();
	preNo=ppTem.getResidueNo();
	
 	for (i=0; i<pdbVec.size(); i++)
 	{
	    pp = (Pdb)pdbVec.elementAt(i);
	    curNo=pp.getResidueNo();
	    noS = String.valueOf(pp.getResidueNo());
	    resid = pp.getResidue();
	    atomVec = pp.getAtomVec();
	    
	    ////////////////////////////////
	    ///if (Math.abs(curNo-preNo)>2)
	    ///	out.println("TER");
	    ///////////////////////////////
	    
	    for (j=0; j<atomVec.size(); j++)
	    {
			cc = (Cartesian) atomVec.elementAt(j);
			atom = cc.getAtom();
			
			//////////////////////
			//for ff2 here:
			//if(atom.equalsIgnoreCase("H"))
			//	atom="HN";
			//////////////////////
			
			//temp
			if(atom.substring(0,1).equalsIgnoreCase("Q"))
				continue;
			
			x = cc.getX();
			y = cc.getY();
			z = cc.getZ();
			sX = format.format(x);
			sY = format.format(y);
			sZ = format.format(z);
			w = 39-31;
			padding = w - sX.length(); 
			for (k = 0; k < padding; k++)
			    sX = " "+sX;
			padding = w - sY.length(); 
			for (k = 0; k < padding; k++)
			    sY = " "+sY;
			padding = w - sZ.length(); 
			for (k = 0; k < padding; k++)
			    sZ = " "+sZ;
      
			str += "ATOM  ";
			w = 12 - 7 ;
	 		atomNo++;
			atomStr = String.valueOf(atomNo);
			padding = w - atomStr.length(); 
			for (k = 0; k < padding; k++)
			    atomStr = " "+atomStr;
			str += atomStr+" "; //add 2 empty space
			
			w = 17 - 13; 
			if (Character.isDigit(atom.charAt(0))|| atom.length()>=4)
			{
			    padding = w - atom.length();
			    for (k = 0; k < padding; k++)
			    	atom += " "; 
			}
			else 
			{
			    atom = " "+ atom;
			    padding =  w - atom.length();
			    for (k = 0; k < padding; k++)
			    	atom += " "; 
			}
			str += atom+" "; //add 1 empty space
	
			str += resid +"  "; //add 2 empty space, if has chainID go to here
	
			w = 27 - 23 ;
			padding = w - noS.length(); // At _least_ 1 space
			for (k = 0; k < padding; k++)
			    noS = " "+noS;
			str += noS+"    "; //add 4 empty space, if has iCode go to here
	
			str += sX+sY+sZ+" ";
			str += " 1.00"+" ";  //Occupancy number
			str += " 0.00"+" ";  //tempaerature Factor
			padding = 80 - str.length(); //finally make the length 80. Some junk stuff go to here
			for (k = 0; k < padding; k++)
				str += " "; 
			
			out.println(str);
			noS = String.valueOf(pp.getResidueNo());
			str="";
	    }
	    preNo=curNo;
 	}
    }
    /**
     * A method for printing the coordinates in PDB format
     * the magic Number comes from PDB file format.
     * 
     * @param pdbVec a vector of Pdb object
     * @param fileName the file name
     * @param out the out
     */
    public void printToFileCyanaFormat(Vector pdbVec,String fileName,PrintWriter out) {
	final int d = 3;              //number of digits after the decimal
	int w = 0;
	DecimalFormat format = new DecimalFormat();
	format.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.US));
	format.setMinimumIntegerDigits(1);
	format.setMaximumFractionDigits(d);
	format.setMinimumFractionDigits(d);
	format.setGroupingUsed(false);
      	Vector atomVec = new Vector();
	Cartesian cc = new Cartesian();
	Pdb pp = new Pdb();
	String noS;
	int curNo,preNo;
	String resid, atom;
	double x,y,z;
	String sX, sY, sZ;
	int padding = 0;
	String atomStr;
	int i, j, k;
	int atomNo=0;
	String str="";
	Pdb ppTem=(Pdb)pdbVec.elementAt(0);
	curNo=ppTem.getResidueNo();
	preNo=ppTem.getResidueNo();
	
 	for (i=0; i<pdbVec.size(); i++)
 	{
	    pp = (Pdb)pdbVec.elementAt(i);
	    curNo=pp.getResidueNo();
	    noS = String.valueOf(pp.getResidueNo());
	    resid = pp.getResidue();
	    atomVec = pp.getAtomVec();
	    
	    ////////////////////////////////
	    ///if (Math.abs(curNo-preNo)>2)
	    ///	out.println("TER");
	    ///////////////////////////////
	    
	    for (j=0; j<atomVec.size(); j++)
	    {
			cc = (Cartesian) atomVec.elementAt(j);
			atom = cc.getAtom();
			
			//////////////////////
			//for ff2 here:
			//if(atom.equalsIgnoreCase("H"))
			//	atom="HN";
			//////////////////////
			
			//temp
			if(atom.substring(0,1).equalsIgnoreCase("Q"))
				continue;
			
			x = cc.getX();
			y = cc.getY();
			z = cc.getZ();
			sX = format.format(x);
			sY = format.format(y);
			sZ = format.format(z);
			w = 39-31;
			padding = w - sX.length(); 
			for (k = 0; k < padding; k++)
			    sX = " "+sX;
			padding = w - sY.length(); 
			for (k = 0; k < padding; k++)
			    sY = " "+sY;
			padding = w - sZ.length(); 
			for (k = 0; k < padding; k++)
			    sZ = " "+sZ;
      
			///str += "ATOM  ";
			w = 5 - 0 ;
	 		atomNo++;
			atomStr = String.valueOf(atomNo);
			padding = w - atomStr.length(); 
			for (k = 0; k < padding; k++)
			    atomStr = " "+atomStr;
			str += atomStr; //add 2 empty space
			
			//if(atom.equalsIgnoreCase("HG12"))
			//	System.out.println("sotp here...");
			w = 18 - 13; 
			if (Character.isDigit(atom.charAt(0))|| atom.length()>=4)
			{ atom = " "+ atom;
			    padding = w - atom.length();
			    for (k = 0; k < padding; k++)
			    	atom += " "; 
			}
			else 
			{
			    atom = " "+ atom;
			    padding =  w - atom.length();
			    for (k = 0; k < padding; k++)
			    	atom += " "; 
			}
			str += atom+""; //add 1 empty space
	
			
	
			w = 27 - 23 ;
			padding = w - noS.length(); // At _least_ 1 space
			for (k = 0; k < padding; k++)
			    noS = " "+noS;
			str +="   " +noS+" "; //add 4 empty space, if has iCode go to here
	
			str += resid ; //add 2 empty space, if has chainID go to here
			
			str += "     "+sX+"   "+sY+"   "+sZ+" ";
			///str += " 1.00"+" ";  //Occupancy number
			///str += " 0.00"+" ";  //tempaerature Factor
			padding = 80 - str.length(); //finally make the length 80. Some junk stuff go to here
			for (k = 0; k < padding; k++)
				str += " "; 
			
			out.println(str);
			noS = String.valueOf(pp.getResidueNo());
			str="";
	    }
	    preNo=curNo;
 	}
    }
    
    /**
     * A method for printing the coordinates in PDB format
     * the magic Number comes from PDB file format.
     * 
     * @param pdbVec a vector of Pdb object
     */
    public void print (Vector pdbVec) {
	final int d = 3;              //number of digits after the decimal
	int w = 0;
	DecimalFormat format = new DecimalFormat();
	format.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.US));
	format.setMinimumIntegerDigits(1);
	format.setMaximumFractionDigits(d);
	format.setMinimumFractionDigits(d);
	format.setGroupingUsed(false);
      	Vector atomVec = new Vector();
	Cartesian cc = new Cartesian();
	Pdb pp = new Pdb();
	String noS;
	String resid, atom;
	double x,y,z;
	String sX, sY, sZ;
	int padding = 0;
	String atomStr;
	int i, j, k;
	int atomNo=0;
	String str="";
 	for (i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    noS = String.valueOf(pp.getResidueNo());
	    resid = pp.getResidue();
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){
		cc = (Cartesian) atomVec.elementAt(j);
		atom = cc.getAtom();
		x = cc.getX();
		y = cc.getY();
		z = cc.getZ();
		sX = format.format(x);
		sY = format.format(y);
		sZ = format.format(z);
		w = 39-31;
		padding = w - sX.length(); 
		for (k = 0; k < padding; k++)
		    sX = " "+sX;
		padding = w - sY.length(); 
		for (k = 0; k < padding; k++)
		    sY = " "+sY;
		padding = w - sZ.length(); 
		for (k = 0; k < padding; k++)
		    sZ = " "+sZ;
      
		str += "ATOM  ";
		w = 12 - 7 ;
 		atomNo++;
		atomStr = String.valueOf(atomNo);
		padding = w - atomStr.length(); 
		for (k = 0; k < padding; k++)
		    atomStr = " "+atomStr;
		str += atomStr+" "; //add 2 empty space
		
		w = 17 - 13; 
		if (Character.isDigit(atom.charAt(0))|| atom.length()>=4)
		{
		    padding = w - atom.length();
		    for (k = 0; k < padding; k++)
		    	atom += " "; 
		}
		else 
		{
		    atom = " "+ atom;
		    padding =  w - atom.length();
		    for (k = 0; k < padding; k++)
		    	atom += " "; 
		}
		str += atom+" "; //add 1 empty space

		str += resid +"  "; //add 2 empty space, if has chainID go to here

		w = 27 - 23 ;
		padding = w - noS.length(); // At _least_ 1 space
		for (k = 0; k < padding; k++)
		    noS = " "+noS;
		str += noS+"    "; //add 4 empty space, if has iCode go to here

		str += sX+sY+sZ+" ";
		str += " 1.00"+" ";  //Occupancy number
		str += " 0.00"+" ";  //tempaerature Factor
		padding = 80 - str.length(); //finally make the length 80. Some junk stuff go to here
		for (k = 0; k < padding; k++)
			str += " "; 
		System.out.println(str);
		noS = String.valueOf(pp.getResidueNo());
		str="";
	    }
	}
    }

   /**
    * A method to compute all the angles from ultrahigh resolution PDBs.
    * 
    * @param pdbLists the pdb lists
    * @param aaType the aa type
    * @param nucType the nuc type
    */
    public void h1Distance(String pdbLists, String aaType, String nucType){
	String str, ss;
  	String src = "/net/drizzle/wlincong/pdbLocal/pdb/ultraHPdb/";
	Pdb pp = new Pdb();
      	Vector pdbVec = new Vector();
  	Vector fileVec = new Vector();
  	Vector phiPsiRdc = new Vector();
	Vector disVec = new Vector();
       	Vector allDisVec = new Vector();
	try{
	    BufferedReader in =
            	new BufferedReader( new FileReader(pdbLists));
	    while( (ss = in.readLine()) != null) {
		StringTokenizer st = new StringTokenizer(ss);
		while (st.hasMoreTokens())
		    fileVec.add(st.nextToken());
	    }
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + pdbLists);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	for (int i=0; i<fileVec.size(); i++){

	    pdbVec =  pp.readPdb(src+(String)fileVec.elementAt(i));

	    disVec = radiiFromNH(pdbVec, aaType, nucType); //for computing from NH
	    allDisVec.addAll(disVec);
	}
	double theta = 0.0;
	final int sampleSize = allDisVec.size();

	double sum = 0.0;
	int N = 0;
	for (int j=0; j<allDisVec.size(); j++){
	    theta = ((Double)allDisVec.elementAt(j)).doubleValue();

 		System.out.println( theta);
  		sum = sum + theta;
  		N++;

 	}

    }

    //test on ff56 short helix.
    //here we test on helix 42-47 at different orientations of the first peptide planes
 
    /**
     * Main0.
     * 
     * @param argv the argv
     */
    public static void main0(String[] argv)
    {
    	long startTime = System.currentTimeMillis();
    	int i, j;
    	
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	
    	
    	ModelRdc mdc = new ModelRdc();
    	Model mod = new Model();
    			
    	//Read Parameter file
    	String paraFile = src+"parameter.txt";
    	Vector<Map<String, String>> paraVec = mod.parameterReader(paraFile);
    	
//    Read the 2ary identification file
    	IDof2aryStructure id2ary = new IDof2aryStructure();
    	String id2aryFile = src+"secondaryElements.txt";
    	Vector<IDof2aryStructure> idVec = id2ary.IDof2aryStructureReader(id2aryFile);
    	Vector[] orderVec = id2ary.getOrder(id2aryFile);

    	for (i=0; i<idVec.size(); i++){
    	    id2ary = idVec.elementAt(i);
    	    System.out.println(id2ary);
    	}
    	
    	/* Temporary PDB files for pairing strands using H-bonds */
    	Pdb pp = new Pdb();

    	//RDC files for one helix used to compute Saupe elements
    	String rdcNhFile = src+"nhRdc.txt";
        String rdcChFile = src+"chRdc.txt";
      	Dipolar dd  =  new Dipolar();
    	final Vector<Dipolar> nhRdc = dd.rdcRead(rdcNhFile);
    	final Vector<Dipolar> cahaRdc = dd.rdcRead(rdcChFile);    	
    
    	Vector<Vector<Dipolar>> hRdc1Vec = dd.rdcReader(rdcNhFile, idVec, "H"); //read RDCs for helices
    	Vector<Vector<Dipolar>> hRdc2Vec = dd.rdcReader(rdcChFile, idVec, "H");
    	Vector helixRdc1Vec = hRdc1Vec.elementAt(0); //2st helix
    	Vector helixRdc2Vec = hRdc2Vec.elementAt(0);

    	double[] saupe = new double[5]; //for returning Saupe elements
    	Map saupeMap = paraVec.elementAt(0); //The first parameters are for computing Saupe elements
    	int refineCycle = Integer.parseInt((String)saupeMap.get("REFINECYCLE"));
    	int dfsCycle    = Integer.parseInt((String)saupeMap.get("DFSCYCLE"));
    	boolean debugDFS = new Boolean((String)saupeMap.get("DEBUGDFS")).booleanValue();
            boolean printResults = new Boolean((String)saupeMap.get("PRINTRESULTS")).booleanValue();
    	PdbRdc pdr = new PdbRdc();

    	//For Helix: this helix is used to compute Saupe elements for two media (7 parameters)
    	double [] ramaFilter = {Const.phiLowHelix,Const.phiHighHelix,Const.psiLowHelix,Const.psiHighHelix};
    	double phiAve = Const.phiAveHelix;
    	double psiAve = Const.psiAveHelix;

    	double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
    	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};

    	Vector<PhiPsi> phiPsiVec = new Vector<PhiPsi>();
    	for (i=42; i<=47; i++)  
    	    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
        Vector<Pdb> pdbHelix42To47 = mdc.modelBuild(phiPsiVec, n1, nh1, ca1);
    	
    	///////////////////changed here...
    	double Syy=4.376784017029875;//4.369119289462044;  
    	double Szz=  8.196982687769985;//7.784097179042096;  		
    	
    	Map helixMap = paraVec.elementAt(1); //The first parameters are for computing Saupe elements
    	
        double resolution  = new Double((String)helixMap.get("RESOLUTION")).doubleValue();
    	refineCycle = Integer.parseInt((String)helixMap.get("REFINECYCLE"));
    	dfsCycle    = Integer.parseInt((String)helixMap.get("DFSCYCLE"));
    	debugDFS = new Boolean((String)helixMap.get("DEBUGDFS")).booleanValue();
        printResults = new Boolean((String)helixMap.get("PRINTRESULTS")).booleanValue();
        double w4Angles = new Double((String) helixMap.get("WEIGHT4ANGLES")).doubleValue();
        
        //ramaFilter = {Const.phiLowHelix,Const.phiHighHelix,Const.psiLowHelix,Const.psiHighHelix};
        
        phiAve = Const.phiAveHelix;
    	psiAve = Const.psiAveHelix;
    	
    	Pdb temp2 = new Pdb();
    	////////////////////////////////
        //for translation of two parts in the same helix
    	
    	///////////
    	//parameters
    	resolution=1.0;//5.0;
    	refineCycle=6;
    	dfsCycle=931072; 
    	debugDFS=true;
    	printResults=true;
    	w4Angles=0.6;
    	double rdcRmsdThreshold=0.0;
    	
        ///////////
    	/////1. 
    	//print out all possible orientations of helices
    	String strPdbOptimal=src+"H5NEW.pdb";//"IdealHelixOptimalPosition.pdb";   	
    	Vector<Pdb> ppVecOptimal42To47  = pp.readPdb(strPdbOptimal);
    	Vector vecBBTemp=pp.OutputBackbone(ppVecOptimal42To47);
    	mdc.PrintAllByGridSearch(vecBBTemp, helixRdc1Vec, helixRdc2Vec, Syy, Szz, resolution, rdcRmsdThreshold );
    	 
    	System.out.println("noting... ");
    	/////////////////////////////
        	
    	
    	int ensembleIDTemp=5;
    	int nTotalModel=9;
    	Vector vecVdw=new Vector();
    	for(j=0;j<=nTotalModel;j++)
    	{
    		//if (j==37)
    		//	continue;
    		Vector vecEnsemblePdb=new Vector();
	    	String strEnsemPdb=src+"final_ensemble"+j+".pdb";
	    	vecEnsemblePdb=pp.readPdbAndParametersFromEnsemble(strEnsemPdb);
	    	System.out.println("input: "+strEnsemPdb);
	    	vecVdw.addAll(vecEnsemblePdb);
	    	
	    	
    	}//for(j=0;j<=nTotalEnsembleID;j++)
    	
    	
    	Collections.sort(vecVdw, new vdw.VdwComparator());
    	//print out all pdbs in one file:
    	String fileNameTemp =src+ "final_ensemble_all"+".pdb";
    	int chosenNumTemp=vecVdw.size();//size of ensemble
    	Vector vecPdbTemp=new Vector();
    	//output all top structures into one pdb file
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileNameTemp)));
    	
    		for (i=0;i<Math.min(chosenNumTemp,vecVdw.size());i++)
    		{
    			vdw vdw1=new vdw();
        		vdw1=(vdw)vecVdw.elementAt(i);
        		vecPdbTemp=vdw1.getPDB();
        		double sc=vdw1.getVdwScore();
        		System.out.println(i+"th score ="+sc); 
        		
        		out.println("MODEL"+i);///to file
        		out.println("REMARK  "+ "   original ensembleID:" +vdw1.getEnsembleID());///to file
        		out.println("REMARK  "+ "   original modelID:" +  (vdw1.getModelID()));///to file
    			        		
        		out.println("REMARK  "+ "   overall (ch and nh) RDC RMSD:" + sc);///to file
        		out.println("REMARK  "+ "   HN RDC RMSD:" + vdw1.getNhRmsd());///to file
        		out.println("REMARK  "+ "   CaHa RDC RMSD:" + vdw1.getChRmsd());///to file

        		
        		pp.printToFile(vecPdbTemp,fileNameTemp, out);
		    	
	            out.println("TER");
	         	out.println("ENDMDL");       
    			
    		}//for (int i=0; i<vecEnsemPdbNew.size();i++ )
    		out.println("END");  
    		out.close();
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileNameTemp);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		

    	///////////////////////////////////
    	Vector vecPdb=new Vector();
    	resolution=2.0;
    	refineCycle=3;
    	double[] nhRdcRmsd=new double[1];
    	double[] chRdcRmsd=new double[1];
    	Vector vecVdwScore=new Vector();
    	//read ensembles of structures
    	
    	int ensembleID=9;////
    	//for ( j=ensembleID;j<=ensembleID;j++)
    	for ( j=ensembleID;j<=ensembleID;j++)
    	{
	    	String strEnsemPdb=src+"IdealHelix"+j+".pdb";
	    	Vector vecEnsemblePdb=pp.readPdbFromEnsemble(strEnsemPdb);
	    	System.out.println("input: "+strEnsemPdb);
	    	//double[] vdwScore=new double[vecEnsemblePdb.size()];
	    	vecVdwScore=new Vector();
	    	//for( i=0;i<vecEnsemblePdb.size();i++)
	    	for( i=14;i>=9;i--)
	    	{
	    		String fileNameOne =src+ "final_ensemble"+ensembleID+"00"+ i+".pdb";
	    		try
	        	{
	        		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileNameOne)));
	        	
		    		System.out.println(" i= " + i );
		    		vecPdb=new Vector();
		    		vecPdb=(Vector)vecEnsemblePdb.elementAt(i);
		    		
		    		refineCycle=3;
		    		//refine the helix
		    		System.out.println("===============helix1=====================");
		    		
		    		Vector<Pdb> helixPdbVec22 = mdc.refineHelix22NoGridSearch(vecPdb,helixRdc1Vec, helixRdc2Vec,  Syy,  Szz, ramaFilter, 
		       		     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,nhRdcRmsd,chRdcRmsd);
		    		temp2 = new Pdb(); ///added by Zeng
		    		temp2.print(helixPdbVec22);///added by Zeng for debugging
		    		
		    		refineCycle=1;
		    		System.out.println("===============helix2=====================");	           
		    		Vector<Pdb> helixPdbVec2 = mdc.refineHelix22(helixPdbVec22,helixRdc1Vec, helixRdc2Vec,  Syy,  Szz, ramaFilter, 
		       		     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,nhRdcRmsd,chRdcRmsd);
		          
		    		temp2.print(helixPdbVec2);///added by Zeng for debugging
		    	    
		           
		    		System.out.println("===============helix3=====================");
		    		Vector<Pdb> helixPdbVec3 = mdc.refineHelix22(helixPdbVec2,helixRdc1Vec, helixRdc2Vec,  Syy,  Szz, ramaFilter, 
		       		     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,nhRdcRmsd,chRdcRmsd);
		    		temp2.print(helixPdbVec3);///added by Zeng for debugging
		           
		    		vecVdwScore.add(new vdw(helixPdbVec3,nhRdcRmsd[0]+ chRdcRmsd[0],nhRdcRmsd[0],chRdcRmsd[0], j,i ) );
		    	    double temp=nhRdcRmsd[0]+ chRdcRmsd[0];
		           	out.println("MODEL"+i);///to file
		       		out.println("REMARK  "+ "   original ensembleID:" +ensembleID);///to file
		       		out.println("REMARK  "+ "   original modelID:" + i);///to file
		   			        		
		       		out.println("REMARK  "+ "   overall (ch and nh) RDC RMSD:" + temp);///to file
		       		out.println("REMARK  "+ "   HN RDC RMSD:" + nhRdcRmsd[0]);///to file
		       		out.println("REMARK  "+ "   CaHa RDC RMSD:" +chRdcRmsd[0]);///to file
		       		pp.printToFile(helixPdbVec3,fileNameOne, out);			    	  
		         	
		            out.println("TER");
		         	out.println("ENDMDL");     
		         	out.println("END");  
		    		out.close();
				}catch (FileNotFoundException e)
				{
					System.out.println("File not found: " + fileNameOne);
				}catch (IOException e)
				{
				   System.out.println("IOException: the stack trace is:");
				   e.printStackTrace();
				}
				
	    	}//for(int i=0;i<vecEnsemblePdb.size();i++)
	    	
    	}//for ( j=0;j<=11;j++)    	
    	
    	Collections.sort(vecVdwScore, new vdw.VdwComparator());
    	
    	//print out all pdbs in one file:
    	String fileName =src+ "final_ensemble"+ensembleID+".pdb";
    	int chosenNum=vecVdwScore.size();//size of ensemble
    	
    	//output all top structures into one pdb file
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    	
    		for (i=0;i<Math.min(chosenNum,vecVdwScore.size());i++)
    		{
    			vdw vdw1=new vdw();
        		vdw1=(vdw)vecVdwScore.elementAt(i);
        		vecPdb=vdw1.getPDB();
        		double sc=vdw1.getVdwScore();
        		System.out.println(i+"th score ="+sc); 
        		
        		out.println("MODEL"+i);///to file
        		out.println("REMARK  "+ "   original ensembleID:" +vdw1.getEnsembleID());///to file
        		out.println("REMARK  "+ "   original modelID:" + vdw1.getModelID());///to file
    			        		
        		out.println("REMARK  "+ "   overall (ch and nh) RDC RMSD:" + sc);///to file
        		out.println("REMARK  "+ "   HN RDC RMSD:" + vdw1.getNhRmsd());///to file
        		out.println("REMARK  "+ "   CaHa RDC RMSD:" + vdw1.getChRmsd());///to file

        	
        		
        		pp.printToFile(vecPdb,fileName, out);
		    
	            out.println("TER");
	         	out.println("ENDMDL");       
    			
    		}//for (int i=0; i<vecEnsemPdbNew.size();i++ )
    		out.println("END");  
    		out.close();
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		

    
        
     	long endTime = System.currentTimeMillis() ;
     	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
     	System.out.println("Time= "+ totalTime +" minutes");

        }
    
    /**  obtain the pdb atom name scheme for a give PDB file
    * 
    * @param vecPdb the given pdb file
   	Note: the function return the following four pdb naming schemes:
 
	(1) "PDB-NEW": eg HA1, HA2 (Gly);   
	(2) "PDB-OLD": eg 1HA, 2HA (Gly);
	(3) "BMRB-NEW": eg HA2, HA3 (Gly);   
	(4) "BMRB-OLD": eg 2HA, 3HA (Gly).
	For bb (either "H" or "HN" is fine)

    * @return one of the four name schemes.
    */
    public String obtainPdbAtomNameScheme(Vector<Pdb> vecPdb){
    	String nameScheme="PDB-NEW";
    	for(int i=0;i<vecPdb.size();i++){
    		Pdb pdb=vecPdb.elementAt(i);
    		String res=pdb.getResidue();
    		Vector<Cartesian> vecAtom=pdb.getAtomVec();
    		if( !( res.equalsIgnoreCase("ALA") || res.equalsIgnoreCase("GLY")|| res.equalsIgnoreCase("ILE")||
    				res.equalsIgnoreCase("THR") || res.equalsIgnoreCase("VAL") ) ){
    			for(int j=0;j<vecAtom.size();j++){
    				Cartesian cc=vecAtom.elementAt(j);
    				String atom=cc.getAtom();
    				
    				if(atom.equalsIgnoreCase("HB3")){
    					nameScheme="BMRB-NEW";
    					return nameScheme;
    				}
    				if(atom.equalsIgnoreCase("3HB")){
    					nameScheme="BMRB-OLD";
    					return nameScheme;
    				}   				
    			}//for(int j=0;j<vecAtom.size();j++)  			
    		}
    		
    	}//for(int i=0;i<vecPdb.size();i++)
    	
    	for(int i=0;i<vecPdb.size();i++){
    		Pdb pdb=vecPdb.elementAt(i);
    		String res=pdb.getResidue();
    		Vector<Cartesian> vecAtom=pdb.getAtomVec();
    		if( !( res.equalsIgnoreCase("ALA") || res.equalsIgnoreCase("GLY")|| res.equalsIgnoreCase("ILE")||
    				res.equalsIgnoreCase("THR") || res.equalsIgnoreCase("VAL") ) ){
    			for(int j=0;j<vecAtom.size();j++){
    				Cartesian cc=vecAtom.elementAt(j);
    				String atom=cc.getAtom();
    				
    				if(atom.equalsIgnoreCase("2HB")){
    					nameScheme="PDB-OLD";
    					return nameScheme;
    				}    								
    			}//for(int j=0;j<vecAtom.size();j++)  			
    		}
    		
    	}//for(int i=0;i<vecPdb.size();i++)
    	
    	return nameScheme;   	
    }
    
    public Vector<Assign> getSeqFromPdb(Vector<Pdb> vecPdb){
    	Vector<Assign> vecSegTemp=new Vector<Assign>();
    	for(int i=0;i<vecPdb.size();i++){
    		Pdb pdb=vecPdb.elementAt(i);
    		int resNo=pdb.getResidueNo();
    		String res=pdb.getResidue();
    		if(res.equalsIgnoreCase("HIE") || res.equalsIgnoreCase("HID"))
    			res="HIS";
    		if(res.equalsIgnoreCase("CYX"))
    			res="CYS";
    		vecSegTemp.add(new Assign(resNo, res));
    	}
    	return vecSegTemp;
    }
}
