package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	RdcPanda.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.Vector;
import java.io.*;

import Jampack.Inv;
import Jampack.JampackException;
import Jampack.Zmat;


// TODO: Auto-generated Javadoc
/**
 * This class implements the main function of RDC-Panda.
 *  Written by  Jianyang (Michael) Zeng (2005-2009)
* 
*/
public class RdcPanda 
{
	
	/** The path of rot lib location. */
	public String rotLibName="rot-lib";	
	
	//entry: main function
	/**
	 * The main method.
	 * 
	 * @param args the arguments
	 * 
	 * @throws JampackException the jampack exception
	 * @throws InterruptedException the interrupted exception
	 */
	public static void main (String[] args)throws JampackException, InterruptedException
	{
		
		RdcPanda rdcPanda=new RdcPanda();
		rdcPanda.outputProgInfo();
		rdcPanda.parse(args);

	}  // End of main()
	
	//The main function which handles all commands
	/**
	 * Parses the.
	 * 
	 * @param args the args
	 * 
	 * @throws JampackException the jampack exception
	 * @throws InterruptedException the interrupted exception
	 */
	public void parse(String[] args) throws JampackException, InterruptedException
	{
		
		boolean done = false;
		boolean commandLineScript = false;
		boolean firstCommandLine = false;
		byte bytebuff[];
		String s = new String("");  // line being parsed
		
		///////////////////////////////////
		//system configuration:
		
		String userDir = System.getProperty("user.dir");
		
		String strSystemFile=userDir+"/system/system-parameters.txt";
		Assign asgTemp = new Assign();
		Vector<Map<String, String>> paraVec = asgTemp.ParamReader(strSystemFile);
		String strInputPath="";
		for (int i=0;i<paraVec.size();i++)
    	{
    		// get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("INPUTPATH"))
    			strInputPath  =  (String)paraMap.get("INPUTPATH");
    	}
		
		
    	//String src=userDir+"/inputFiles/"; 
    	String src=userDir+strInputPath; 
    	String strOut=userDir+"/outFiles/";
    	
		if (args.length > 0) 
		{
			commandLineScript = true;
			firstCommandLine = true;
		}
		
		// MAIN LOOP
		while (!done) 
		{
			
			bytebuff = new byte[150];
			if (!commandLineScript) 
			{
				System.out.print("RDC-PANDA> ");
				try 
				{
					System.in.read(bytebuff);
				}
				catch ( Exception e )
				{
					System.out.println("ERROR: An error occurred while reading input");
					System.exit(0);
				}
				s = new String(bytebuff);  // create a string from bytebuff
			}
			else if (commandLineScript && !firstCommandLine) 
			{
				// If you were running a command line script and the file is over then quit
				s = new String("quit");
			}
			else if (firstCommandLine)
			{
				s = new String("");
				for(int i=0;i<args.length;i++)
					s = s.concat(args[i] + " ");
				firstCommandLine = false;
			}			
			 
			s = s.trim();  // remove whitespace from beginning and end of line
				
			StringTokenizer st = new StringTokenizer(s," ;\t\n\r\f");
			String firstToken = new String("");
			if (st.hasMoreTokens())
				firstToken = st.nextToken();  // snag a copy of the first token
						
			if (firstToken.length() > 1)
				if (firstToken.substring(0,2).equals("//"))
					continue;
			
			String secondToken = new String("");
			if (st.hasMoreTokens())
				secondToken = st.nextToken(); 
			
			if (firstToken.equalsIgnoreCase("quit") || firstToken.equalsIgnoreCase("q"))
			{				
				done = true;
				return ;
			}
			
			//NOE assignment only based on the chemical shift information
			else if (firstToken.equalsIgnoreCase("NOEAsgFromCS")  )
				if(secondToken.length() >0)
				{
					Assign asg=new Assign();
					asg.NOEAsgFromCS(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("NOEAsgFromCS");
			
			//packing two SSEs using sparse NOE restraints
			else if(firstToken.equalsIgnoreCase("SSEPacking")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doSSEPacking(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("SSEPacking");
			 
			//check the packing symmetry that best fit the noe restraints (namely with minimum NOE RMSD)
			else if(firstToken.equalsIgnoreCase("checkPackSym")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doCheckPackingSym(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("checkPackSym");
			
			//	merge all structures in previous ensemble into one enemble (with the same pdb name)
		     // and cluster them according to given resolution.		     
			else if(firstToken.equalsIgnoreCase("mergeNCluster")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doMergeNCluster(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("mergeNCluster");
			
			//merge all structures in previous ensemble into one enemble (with the same pdb name)
		     // and cluster them according to given resolution.		     
			else if(firstToken.equalsIgnoreCase("mergeNClusterAll")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doMergeNClusterAll(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("mergeNClusterAll");			

			//add side-chains and prune steric clashes:
			else if(firstToken.equalsIgnoreCase("addSideChains")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doAddSideChains(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("addSideChains");
			
			//add random loops to an ensemble of SSE structures
			else if(firstToken.equalsIgnoreCase("addRandomLoops")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doAddRandomLoops(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("addRandomLoops");

			//calculate the NOE satisfaction scores for all structures in an ensemble
			else if(firstToken.equalsIgnoreCase("calNOEScore")  )
				if(secondToken.length() >0)
				{
					Pdb pp=new Pdb();
					pp.doCalNOEScores(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("calNOEScore");
			
			//HANA for NOE assignment:
			else if(firstToken.equalsIgnoreCase("HANA_NOE_Asg")  )
				if(secondToken.length() >0)
				{
					Assign asg=new Assign();
					asg.doHANANOEAsg(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("HANA_NOE_Asg");
			
			//refine NOE assignment, given the backbone:
			else if(firstToken.equalsIgnoreCase("RefineNOEsWBB")  )
				if(secondToken.length() >0)
				{
					Assign asg=new Assign();
					asg.doRefineNOEsWBB(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("RefineNOEsWBB");
			
			//compare two NOE tables:
			else if(firstToken.equalsIgnoreCase("CompareTwoNOETables")  )
				if(secondToken.length() >0)
				{
					Assign asg=new Assign();
					asg.doCompareTwoNOETables(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("CompareTwoNOETables");
			
			//compare two NOE tables:
			else if(firstToken.equalsIgnoreCase("ExtractNOEs")  )
				if(secondToken.length() >0)
				{
					Assign asg=new Assign();
					asg.doExtractNOEs(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("ExtractNOEs");
			
			
			//refine long range NOE assignments:
			else if(firstToken.equalsIgnoreCase("LongNOEsAsgWOStructure")  )
				if(secondToken.length() >0)
				{
					Assign asg=new Assign();
					asg.doLongNOEsAsgWOStructure(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("LongNOEsAsgWOStructure");
			
			//refine long range NOE assignments:
			else if(firstToken.equalsIgnoreCase("LongNOEsAsgWOStructureNew")  )
				if(secondToken.length() >0)
				{
					Assign asg=new Assign();
					asg.doLongNOEsAsgWOStructureNew(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("LongNOEsAsgWOStructureNew");
			
			//	refine energy from xplor pdb:
			else if(firstToken.equalsIgnoreCase("ReadPDBEnergy")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doReadPDBEnergy(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("ReadPDBEnergy");
		
			//refine the half residues in RDC-exact backbone:
			else if(firstToken.equalsIgnoreCase("RefineRDCExact")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doRefineRdcExact(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("RefineRDCExact");

			//refine the middle residues in RDC-exact backbone:
			else if(firstToken.equalsIgnoreCase("RefineRDCExactMiddle")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doRefineRdcExactMiddle(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("RefineRDCExactMiddle");
			
			//do the RDC-EXACT part to calculate bb
			else if(firstToken.equalsIgnoreCase("RdcExactHelix")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doRdcExactHelix(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("RdcExactHelix");

			//			do the RDC-EXACT part to calculate bb
			else if(firstToken.equalsIgnoreCase("RdcExactHelixWOAT")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doRdcExactHelixWOAT(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("RdcExactHelixWOAT");
			
			//do the RDC-EXACT part to calculate beta sheet
			else if(firstToken.equalsIgnoreCase("RdcExactSheet")  )
				if(secondToken.length() >0)
				{
					Model mod=new Model();
					mod.doRdcExactSheet(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("RdcExactHelixWOAT");
			//compute the alignment tensor
			else if(firstToken.equalsIgnoreCase("CalAlignmentTensor")  )
				if(secondToken.length() >0)
				{
					RdcPanda rdcPanda=new RdcPanda();
					rdcPanda.doCalAlignmentTensor(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("CalAlignmentTensor");
			
			//check the NOEs from ensemble of structures
			else if(firstToken.equalsIgnoreCase("CheckNoeByEnsemble")  )
				if(secondToken.length() >0)
				{
					Noe noe=new Noe();
					noe.doCheckNoeByEnsemble(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("CheckNoeByEnsemble");
			
			//	output the NOE statistics
			else if(firstToken.equalsIgnoreCase("NoeStatistics")  )
				if(secondToken.length() >0)
				{
					Noe noe=new Noe();
					noe.doNoeStatistics(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("NoeStatistics");
			
			//NOE assignment without sidechain resonance asg, using NASCA algorithm
			else if(firstToken.equalsIgnoreCase("Nasca")  )
				if(secondToken.length() >0)
				{
					Nasca nasca=new Nasca();
					nasca.doNasca(src,strOut,secondToken);					
				}
				else
					errorTooFewParams("Nasca");
			
			//Run a script file 
			else if(firstToken.equalsIgnoreCase("RunXplor")  )
			{
				RdcPanda rdcPanda=new RdcPanda();
				rdcPanda.doRunXplor(src,strOut,secondToken);
			}
			
			//read structure fragments from a pdb file:
			else if(firstToken.equalsIgnoreCase("ReadPdbFragments")  )
			{
				Pdb pp=new Pdb();
				pp.doReadPdbFragments(src,strOut,secondToken);
			}
			
			//compute the loop fragments:
			else if(firstToken.equalsIgnoreCase("ComputeLoops")  )
			{
				Loops loops=new Loops();
				loops.doCalLoops(src,strOut,secondToken);		
			}
			
			//compute the loop fragments, using the GOAL algorithm:
			else if(firstToken.equalsIgnoreCase("goal")  )
			{
				Goal goal=new Goal();
				goal.doGoalCalLoops(src,strOut,secondToken);		
			}
				
			System.exit(0);//currently we just execute one command
			
		} // End main "while (!done)" loop

	} // End parse function	
	
	 /**
 	 * call Xplor as a subroutine given a script file.
 	 * 
 	 * @param src location of the input file
 	 * @param strOut location of the output file
 	 * @param strInput  input file name
 	 * 
 	 * @return void
 	 * 
 	 * @throws JampackException the jampack exception
 	 * @throws InterruptedException the interrupted exception
 	 */       
    public void doRunXplor(String src, String strOut, String strInput)throws JampackException, InterruptedException
    {	     
    	long startTime = System.currentTimeMillis();
    	
    	String userDir = System.getProperty("user.dir");////
    	String srcXplor=userDir+"/xplor/";
    	
    	System.setProperty("user.dir",	srcXplor);
    	
    	String userDirNew = System.getProperty("user.dir");////
    	System.out.println("Current working directory is:"+userDirNew);//for debugging p
    	
    	//String strRun="csh "+srcXplor+"RunXplor.csh";
    	String strRun="/usr/project/dlab/Users/Software/xplor-NIH-64/xplor-nih-2.16.0/bin/xplor < eta.inp > eta.out";
    	Runtime runtime = Runtime.getRuntime();
    	try
        {    		
    		//execute the xplor scripts:
    		File  direc =new File (userDirNew);
    		Process xplor = runtime.exec(strRun, null, direc );
    		InputStream xplorStdOut = xplor.getInputStream();
    		InputStreamReader isr = new InputStreamReader(xplorStdOut);
    	    BufferedReader br = new BufferedReader(isr);
    	    String line;
    	    String outFileName =srcXplor+"xplorOutResults.txt";
    	    try
	    	{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outFileName)));
	    	    while ((line = br.readLine()) != null)
	    	    {
	    	    	 out.println(line);///to file
	    	         System.out.println(line);//for debugging...
	    	    }
	    	    out.close();
	    	
	        }catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + outFileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
		
    	    xplor.waitFor();
        }
        catch(IOException e)
        {
           System.err.println("Error on exec() method");
           e.printStackTrace();  
        }
       
        long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Time= "+ totalTime +" minutes");
    }
    
    /**
     * Pack SSEs using sparse inter-SSE NOE restraints.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doSSEPacking(String src, String strOut, String strInput)throws JampackException
    {	 
    	int maxStructurePerEnsemble=2000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	
    	String userDirTemp0 = System.getProperty("user.dir");
    	// Read the input files    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	String strSeq="", strIsOne2One="",strSSES="",strOutPackedStr="",strInLocation="",strIsSkipOutEnsemble="",strIsAmbiNOEs="",
    		strNoeFormat="",strNOETable="",strIsSkipSymCheck="",strIsSkipCluster="",strEnseName="",strAveOutPackedStr="";
    	double maxEnsemSize=1000;
    	double resol_cluster=0.0,noeCor=0.0;;
    	int nSymID=0;
    	
    	for (int i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("ISONE2ONE"))
    			strIsOne2One  =  (String)paraMap.get("ISONE2ONE");	    			
    		if(paraMap.containsKey("MAXSTRUCTURES"))
    			maxEnsemSize  =  Double.parseDouble((String)paraMap.get("MAXSTRUCTURES"));
    		if(paraMap.containsKey("SSES"))
    			strSSES  =  (String)paraMap.get("SSES");       		
    		if(paraMap.containsKey("OUTPACKEDSTRUCTURES"))
    			strOutPackedStr  =  (String)paraMap.get("OUTPACKEDSTRUCTURES");
    		if(paraMap.containsKey("OUTAVEPACKEDSTRUCTURES"))
    			strAveOutPackedStr  =  (String)paraMap.get("OUTAVEPACKEDSTRUCTURES");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");
    		if(paraMap.containsKey("ISSKIPSYMCHECK"))
    			strIsSkipSymCheck  =  (String)paraMap.get("ISSKIPSYMCHECK");
    		if(paraMap.containsKey("ISSKIPOUTENSEMBLE"))
    			strIsSkipOutEnsemble  =  (String)paraMap.get("ISSKIPOUTENSEMBLE");   		
    		if(paraMap.containsKey("SYMID"))
    			nSymID  =  Integer.parseInt((String)paraMap.get("SYMID"));   		
    		if(paraMap.containsKey("RESOLCLUSTER"))
    			resol_cluster  =  Double.parseDouble((String)paraMap.get("RESOLCLUSTER"));
    		if(paraMap.containsKey("ISSKIPCLUSTER"))
    			strIsSkipCluster  =  (String)paraMap.get("ISSKIPCLUSTER");   		
    		if(paraMap.containsKey("NOE-CORRECTION"))
    			noeCor  =  Double.parseDouble((String)paraMap.get("NOE-CORRECTION"));
    		if(paraMap.containsKey("INPUTFILELOCATION"))
    			strInLocation  =  (String)paraMap.get("INPUTFILELOCATION");
    		if(paraMap.containsKey("ENSEMBLENAME"))
    			strEnseName  =  (String)paraMap.get("ENSEMBLENAME");    	
    		if(paraMap.containsKey("ISAMBIGUOUSNOES"))
    			strIsAmbiNOEs  =  (String)paraMap.get("ISAMBIGUOUSNOES");   		
    	}      
    	
    	// Read the protein sequence    	
    	String seqFile = src + strSeq;    	
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	// Read the rotamer library path.     	
    	// Use the Xtal' library from Richardson's lab    	
    	String rotSrc=userDirTemp0+"/system/rot-lib/";//  src+ "rotasamp-small/";    	
    	
    	// read the NOE assignment table (sparse inter-SSE NOEs)
    	String manualAsgFile = src+ strNOETable; 
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,noeCor,"PDB-NEW");//////////////////0.0
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	// read from xplor format: (note: xplor format does not include residue names...)
    	else
    	{    		
    		vecManAsg=noe_temp.noeReader(manualAsgFile,noeCor,vecSeq);    		
    		vecManAsgNew= noe_temp.ConvertXplorAsgToUplFormat(vecManAsg,vecSeq,"PDB-NEW");    	
    	}    	
    	  	   
    	//For all-to-one packing:
    	Vector vecPdbAllTemp=new Vector();
    	if(!strIsOne2One.equalsIgnoreCase("1"))
    	{
    		String userDir = System.getProperty("user.dir");
    		String strInputFile=userDir+strInLocation;
    		File myDir = new File(strInputFile);
        	String[] contents = myDir.list();
        	if (contents == null) 
        		System.out.println(myDir.getName() + " is not a directory");
        	
        	//select all structures
        	for (int t=0; t<contents.length;t++) 
        	{    
        		String filename = contents[t]; 
    	    			
    	    	String strEnsemPdb="";
    	    	strEnsemPdb=strInputFile+filename;	    
    	    	Vector vecPdbABB=pp.readPdb(strEnsemPdb);
    	    	if(vecPdbABB.size()>0)
    	    	{
    	    		Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, vecPdbABB);            	
    	    		// this is final backbone pdb used for hosting rotamers
    	    		Vector pdbVecSSE = pp.nameConvertOrder(pdbVecSSE_temp2);         	
    	    		vecPdbAllTemp.addAll(pdbVecSSE);
    	    		break;
    	    	}            	
        	}//end for (int t=0; t<contents.length;t++) 
        	if(vecPdbAllTemp.size()<1)
        	{
        		System.out.println("No input PDB is found for packing....");
        		return;
        	}        	
    	}//if(!strIsOne2One.equalsIgnoreCase("1"))
    	
    	//read the order of sse packing
    	Vector vecOrderSSES=asg.getOrder(strSSES);
    	
    	//read the pdb file of each sse
    	Vector vecPdbSSE=new Vector();
    	for(int i=0; i< vecOrderSSES.size();i++)
    	{
    		//for all-to-one packing
    		if((!strIsOne2One.equalsIgnoreCase("1")) && (i==0))
        	{
    			vecPdbSSE.add(vecPdbAllTemp);
    			continue;    	
        	}
    		
    		String strPdbFile= (String)vecOrderSSES.elementAt(i);
    		strPdbFile=src+strPdbFile+".pdb";
    		Vector pdbVecSSE_temp = pp.readPdb(strPdbFile);

        	//update backbone atom names according to input protein sequence 
        	//residue name in previous backbone may be all in "ALA" (from RDC-EXACT).     		
            Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, pdbVecSSE_temp);
        	
        	// this is final backbone pdb used for hosting rotamers
        	Vector pdbVecSSE = pp.nameConvertOrder(pdbVecSSE_temp2);      	
    		
           Collections.sort(pdbVecSSE, new Pdb.PdbComparator());        	
           vecPdbSSE.add(pdbVecSSE);
    		
    	}//for(i=0; i< vecOrderSSES.size();i++)	
    	
    	
    	// Rotamer placement at each individual SSE.
    	// We place all rotamers onto each SSE.    	
    	Vector vecPdbRotSSE=new Vector();    
    	Model md=new Model();
    	
    	for(int i=0;i<vecPdbSSE.size();i++)
    	{
    		Vector vecPdbSseBb=(Vector)vecPdbSSE.elementAt(i);
    		
    		//be careful of rotation, we can't change the orientation of each sse
    		Vector vecPdbSseRot=pp.AllRotamersStructure(vecPdbSseBb,rotSrc);    		
    		//a function to rotate SSE back to the same coordinate system:
    		Vector vecPdbSseRotNew=md.BackonbeRotation(vecPdbSseRot,vecPdbSseBb); 
    		
    		Collections.sort(vecPdbSseRotNew, new Pdb.PdbComparator());
    		vecPdbRotSSE.add(vecPdbSseRotNew);
    	}
    	    	
    	/////////////////////////////////////////////////
    	//Assemble SSEs based on NOEs 
    	
    	Vector vecPdbA=new Vector();
    	Vector vecPdbB =new Vector();
    	
    	vecPdbA.addAll((Vector)vecPdbRotSSE.elementAt(0));
    	Collections.sort(vecPdbA, new Pdb.PdbComparator());
    	PdbRmsd pdr=new PdbRmsd();
    	double [] max_score=new double[4];
    	Vector[] vecEnsembPack = new Vector[4];
    	
    	PdbRmsd psd = new PdbRmsd();
     	double[] noeRmsd = new double[1];
     	double[] noeRms  = new double[4];
     	boolean debugNOE = true;	
      	Vector[] ubqByRdcVec = new Vector[4];
      	Vector[] vecNewNOE=new Vector[4];
      	
      	Vector vecNoeUpdateOrder=noe_temp.UpdateNOE(vecManAsgNew,vecPdbSSE);
      	     	
    	////check the symmetry of packing SSEs
       	int minSym=0;
      	//skip the symmetry checking
      	if(strIsSkipSymCheck.equalsIgnoreCase("1"))
      		minSym=nSymID;
      	else//do the symmetry checking
      	{
	    	Vector vecPdbATemp=new Vector();
	    	Vector vecPdbBTemp =new Vector();
	    	
	    	vecPdbATemp.addAll((Vector)vecPdbRotSSE.elementAt(0));
	    	Collections.sort(vecPdbATemp, new Pdb.PdbComparator());
	    	vecPdbBTemp.addAll((Vector)vecPdbRotSSE.elementAt(1));
	    	Collections.sort(vecPdbBTemp, new Pdb.PdbComparator());
	    	Vector noeVecTemp =new Vector();
	    	noeVecTemp.addAll((Vector)vecNoeUpdateOrder.elementAt(0));
	    	
	    	double minRms=999999.9;
	     	for (int i = 0; i < 4; i++)
		 	{	  
	     		Vector vecPdbATempNew=new Vector();
	     		if (i==0)
	     			vecPdbATempNew.addAll(vecPdbATemp);
	     		else
	     			vecPdbATempNew = pp.newPdb(vecPdbATemp, Const.mat4ThreeDirs[i - 1]);
	     		psd.calcMinPackingRms(noeVecTemp, vecPdbATempNew, vecPdbBTemp, noeRmsd, debugNOE);
	    	    if(noeRmsd[0]<minRms)
	    	    {
	    	    	minRms=noeRmsd[0];
	    	    	minSym=i;
	    	    }
	    	
		 	}
      	}//else for "do the symmetry checking"
      	
      	if(Const.isDebug)
      		System.out.println("The symmetry ID is: "+ minSym);
    	   	
      	for (int i = 0; i < 4; i++)
     	{
     	    ubqByRdcVec[i] = new Vector<Pdb>();
     	    vecNewNOE[i]=new Vector<Noe>();
     	}
    		    	
    	Vector vecTempBB=new Vector ();   	
    	String strPass="";
    	if(strIsSkipOutEnsemble.equalsIgnoreCase("1"))
    		psd.isSkipOutEnsembe=true;
    	    	
    	//check whether the output directory exists:
    	String userDirTemp = System.getProperty("user.dir");////
    	String outPackPath=userDirTemp+strOutPackedStr;
    	int idPath=outPackPath.lastIndexOf('/');
    	String outPackDir=outPackPath.substring(0, idPath+1);
    	
    	boolean existsAlready = (new File(outPackDir)).exists();
    	if(existsAlready)
        {
        	//if the output directory already exists, we delete the whole directory
        	boolean success = deleteDir(new File(outPackDir));
        	if(!success)
        	{
        		System.out.println("Failed to delete an already existing output directry!");
        		System.exit(0);
        	}
        }
    	
    	//create a new directory
    	boolean exists = (new File(outPackDir)).exists();
        if (!exists) 
        {
            // create the directory if it does not exist
        	boolean success = (new File(outPackDir)).mkdir();
        	if(!success)
        	{
        		System.out.println("Failed to create output directry!");
        		System.exit(0);
        	}
        	
        } 
           	
    	//one-to-one packing
    	if(strIsOne2One.equalsIgnoreCase("1"))
    	{
    		
    		strPass=userDirTemp+strOutPackedStr+"Temp";//strOut+strOutPackedStr+"Temp";//this is temporary file, and will be deleted in the end 
    		for (int k=1;k<vecPdbRotSSE.size();k++)
        	{
    			double[] dbTemp=new double[1];
        		vecPdbB =new Vector();
        		vecPdbB.addAll((Vector)vecPdbRotSSE.elementAt(k));
        		Vector<Pdb> helixVecN = new Vector<Pdb>();
        		
        		Vector noeVec =new Vector();
    	      	noeVec.addAll((Vector)vecNoeUpdateOrder.elementAt(k-1));
    	      	
    	      	Vector noeVecUpdate=new Vector();
    	      	//noeVecUpdate.addAll(noeVec);
    	      	noeVecUpdate=noe_temp.setSameUpperBounds4All(noeVec,Const.noeUpBound4Packing);//////need to test here...
    	      	if (isDebug)
    	      		noe_temp.PrintNOE(noeVecUpdate);
    	 	
    	      	helixVecN=new Vector();
        		if(minSym==0)
        			helixVecN.addAll(vecPdbA);
        		else
        	 	   helixVecN = pp.newPdb(vecPdbA, Const.mat4ThreeDirs[minSym - 1]);
        		
        		if(strIsAmbiNOEs.equalsIgnoreCase("1"))
        			ubqByRdcVec[0] = psd.positionByAmbiNOEAllRotamers(noeVecUpdate, helixVecN, vecPdbB, noeRmsd, debugNOE,vecNewNOE[0],strPass,vecSeq,resol_cluster);
        		else
        			ubqByRdcVec[0] = psd.positionByNOEAllRotamers(noeVecUpdate, helixVecN, vecPdbB, noeRmsd, debugNOE,vecNewNOE[0],strPass,vecSeq,resol_cluster);
        	 	
        		noeRms[0] = noeRmsd[0];
        	 	  
        	 	vecTempBB=pp.OutputBackbone(ubqByRdcVec[0]);
        	 	if(isDebug)
        	 	{
        	 		System.out.println("REMARK here is average packed structure:");
        	 	    pp.print(vecTempBB);
    	         	System.out.println("TER");
    	         	System.out.println("END");    	       
        	 	}    
        	 	
        	 	//output the average structure 
        	 	String outFileName = userDirTemp+ strAveOutPackedStr;
        	 	try
	    		{	    				
	    			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outFileName)));
	    			pp.printToFile(vecTempBB,outFileName, out);			    	
		            out.println("TER");		         	
		         	out.println("END");  
	    			out.close();
	        	}catch (FileNotFoundException e)
	        	{
	        		System.out.println("File not found: " + outFileName);
	        	}catch (IOException e)
	        	{
	        	   System.out.println("IOException: the stack trace is:");
	        	   e.printStackTrace();
	        	}
        	
        	}//for (i=0;i<vecPdbSSE.size();i++)        	
    		
    	}//if(strIsOne2One.equalsIgnoreCase("1"))
    	else //all-to-one packing
    	{                
    		String userDir = System.getProperty("user.dir");////
    		String strInputFile=userDir+strInLocation;
    		File myDir = new File(strInputFile);
        	String[] contents = myDir.list();//select all structures
        	if (contents == null) 
        		System.out.println(myDir.getName() + " is not a directory");
        	for (int t=0; t<contents.length;t++) 
        	{            		
        		String filename = contents[t]; 
    	    			
    	    	String strEnsemPdb="";
    	    	strEnsemPdb=strInputFile+filename;	    		
           		vecPdbA=new Vector();
           		
           		// ensemble and structure IDs are delimted by "000"
        		strPass=userDirTemp+strOutPackedStr+t+"Temp";
        		if(Const.isDebug)
        			System.out.println("**********************input: "+strEnsemPdb);
        		
        		 File f = new File(strEnsemPdb);
        		 
        		 if (!f.exists())
        		 {
        			 if(Const.isDebug)
        				 System.out.println("Couldn't find file "+ strEnsemPdb+ " for packing, continue...");
        			 continue;
        		 }
        		    	
        		Vector vecPdbABB=pp.readPdb(strEnsemPdb);
        		Vector vecPdbATEmp=pp.AllRotamersStructure(vecPdbABB,rotSrc);	    		
        		
        		//a function to rotate SSE back to the same coordinate system:
        		vecPdbA=md.BackonbeRotation(vecPdbATEmp,vecPdbABB);     	    	
        	
	        	for (int k=1;k<vecPdbRotSSE.size();k++)
	        	{
	        		double[] dbTemp=new double[1];
	        		vecPdbB =new Vector();
	        		vecPdbB.addAll((Vector)vecPdbRotSSE.elementAt(k));
	        		Vector<Pdb> helixVecN = new Vector<Pdb>();
	        		
	        		Vector noeVec =new Vector();
	    	      	noeVec.addAll((Vector)vecNoeUpdateOrder.elementAt(k-1));
	    	      	
	    	      	Vector noeVecUpdate=new Vector();
	    	      	noeVecUpdate=noe_temp.setSameUpperBounds4All(noeVec,Const.noeUpBound4Packing);//////need to test here...
	    	      	if (isDebug)
	    	      		noe_temp.PrintNOE(noeVecUpdate);
	    	 	
	    	      	helixVecN=new Vector();
	        		if(minSym==0)
	        			helixVecN.addAll(vecPdbA);
	        		else
	        	 	   helixVecN = pp.newPdb(vecPdbA, Const.mat4ThreeDirs[minSym - 1]);
	        		
	        		if(strIsAmbiNOEs.equalsIgnoreCase("1"))
	        			ubqByRdcVec[0] = psd.positionByAmbiNOEAllRotamers(noeVecUpdate, helixVecN, vecPdbB, noeRmsd, debugNOE,vecNewNOE[0],strPass,vecSeq,resol_cluster);
	        		else
	        			ubqByRdcVec[0] = psd.positionByNOEAllRotamers(noeVecUpdate, helixVecN, vecPdbB, noeRmsd, debugNOE,vecNewNOE[0],strPass,vecSeq,resol_cluster);
	        	 	
	        		noeRms[0] = noeRmsd[0];        	 	  
	        	 	vecTempBB=pp.OutputBackbone(ubqByRdcVec[0]);
	        	 	if(Const.isDebug)
	        	 	{
	        	 		System.out.println("REMARK here is average packed structure:");
	        	 	    pp.print(vecTempBB);
	    	         	System.out.println("TER");
	    	         	System.out.println("END");    	       
	        	 	}      	        	
	        	}//for (i=0;i<vecPdbSSE.size();i++)   	        	 
        	}//for (int t=0; t<contents.length;t++)    		
    	}//else //one-to-all packing
    	       	    	
    	////cluster the temporary packed structure in the ensemble 
    	//here we don't need to do the clustering step again, since it has been done in positionByNOEAllRotamers.
    	if(strIsSkipCluster.equalsIgnoreCase("0"))
    	{
    		int maxTempEnsem=0;
    		String strDelimited="Temp";
    		vdw vv=new vdw();
    		String strPreEnsemb="";
    		if(strIsOne2One.equalsIgnoreCase("1"))
    		{
    			strPreEnsemb=strOutPackedStr+"Temp";
    			maxTempEnsem=1;
    		}
    		else
    		{
    			strPreEnsemb=strOutPackedStr;
    			maxTempEnsem=(int)maxEnsemSize;
    		}
    		vv.mergeAndCluster(strOut, strOut, strPreEnsemb,maxStructurePerEnsemble, maxTempEnsem, resol_cluster,  strOutPackedStr,strDelimited);
    		
    	}//if(strIsSkipCluster.equalsIgnoreCase("0"))    	  	
    	
        long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for packing is:  "+ totalTime +" minutes");  	
    	
    }
    
    /**
     * * merge all structures in previous ensemble into one enemble (with the same pdb name)
     * and cluster them according to given resolution.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doMergeNCluster(String src, String strOut, String strInput)throws JampackException
    {	 
    	String strDelimited="000";
    	int maxStructurePerEnsemble=3000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////    	
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	String strOutPackedStr="",strPreStrName="",strInLocation="";
    	String strDelimitedTmp="";//not used yet
    	int maxEnsemSize=1000;
    	double resol_cluster=0.0;
    	int nSymID=0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    		
    	 			
    		if(paraMap.containsKey("MAXSTRUCTURES"))
    			maxEnsemSize  =   Integer.parseInt((String)paraMap.get("MAXSTRUCTURES"));			
    		
    		if(paraMap.containsKey("OUTPACKEDSTRUCTURES"))
    			strOutPackedStr  =  (String)paraMap.get("OUTPACKEDSTRUCTURES");
    		if(paraMap.containsKey("STRDELIMIT"))
    			strDelimitedTmp  =  (String)paraMap.get("STRDELIMIT");
    		    		
    		if(paraMap.containsKey("SYMID"))
    			nSymID  =  Integer.parseInt((String)paraMap.get("SYMID"));
    		
    		if(paraMap.containsKey("RESOLCLUSTER"))
    			resol_cluster  =  Double.parseDouble((String)paraMap.get("RESOLCLUSTER"));
    		if(paraMap.containsKey("PREENSEMNAME"))
    			strPreStrName  =  (String)paraMap.get("PREENSEMNAME");
    		if(paraMap.containsKey("INPUTFILELOCATION"))
    			strInLocation  =  (String)paraMap.get("INPUTFILELOCATION");
    		
    	}      
    	///////////////////////////////////////////////////////////////
    	//cluster the temporary packed structure in the ensemble
    	//
    	///////////////////////////////////////////////////////////////
    	//do the clustering step:
    	String userDir = System.getProperty("user.dir");////
    	src=userDir+strInLocation;
    	
		vdw vv=new vdw();
		String strPreEnsemb="";
				
		vv.mergeAndCluster(src, strOut, strPreStrName,maxStructurePerEnsemble, maxEnsemSize, resol_cluster,  strOutPackedStr,strDelimited);
  	
        long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for clustering is:  "+ totalTime +" minutes");  	
    	
    }
   
    
    /**
     * * add the side chains and prune steic clahes.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doAddSideChains(String src, String strOut, String strInput)throws JampackException
    {	 
    	String strDelimited="000";
    	int maxStructurePerEnsemble=3000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	Peak pk = new Peak();
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////    	
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	
    	String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="",strNoesyFormat="", strResNameScheme="",
		strHaNoesy3D="", strResFormat="",strNewEnsemb="",strLocBBEnsem="",strIsCheckLocalClash="",strSSEBB="";
    	
    	
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error tolerance for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		//get the input file name   
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("RESFORMAT"))
    			strResFormat  =  (String)paraMap.get("RESFORMAT");
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("RESNAMESCHEME"))
    			strResNameScheme  =  (String)paraMap.get("RESNAMESCHEME");
    		if(paraMap.containsKey("NOESY-FORMAT"))
    			strNoesyFormat  =  (String)paraMap.get("NOESY-FORMAT");   		
    		if(paraMap.containsKey("SSEBB"))
    			strSSEBB  =  (String)paraMap.get("SSEBB");
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		//strNoesy2D=strNoesy2D.toLowerCase();
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		//strHnNoesy3D=strHnNoesy3D.toLowerCase();
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		//strHaNoesy3D=strHaNoesy3D.toLowerCase();
    		if(paraMap.containsKey("LOCATIONBBENSEMBLE"))
    			strLocBBEnsem  =  (String)paraMap.get("LOCATIONBBENSEMBLE");
    	
    		if(paraMap.containsKey("OUTSTRUCTURENAME"))
    			strNewEnsemb  =  (String)paraMap.get("OUTSTRUCTURENAME");
    		if(paraMap.containsKey("ISCHECKLOCALCLASH"))
    			strIsCheckLocalClash  =  (String)paraMap.get("ISCHECKLOCALCLASH");
    				
    	}       	   	
    	    	
       	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------    	
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the resonance list 
    	// 
    	// in the "resonance.prot" format
    	//-------------------------------------	    	
    	//following is for the general case
    	H1CS h1CS=new H1CS();  	
    	String assignFile = src+ strReson; 
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector assignVec=new Vector();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("unknown format of the resonance file...");
    		System.exit(0);
    	}
    	
    	//all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());
    	    	
    	//-------------------------------------
    	// (1.3) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector hnNoeVec=new Vector();//for storing 3d n15 NOESY peaks
    	Vector cnoeVec=new Vector();//for storing 3d c13 noesy peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;    		
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			hnNoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile); 
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;    		
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			cnoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	Vector vecNoesy=new Vector();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	///////////////////////////////
    	//compute the constant for Cyana's distance calibration scheme
    	String pdbSSEFile = src+strSSEBB;
    	Vector vecSSE = pp.readPdb(pdbSSEFile);
    	double[] constant=new double[1];
    	noesy.SetCalibrationN15(hnNoeVec,vecSSE,assignVec,constant);    	
    	///////////////////////////////
    	
    	////////////////////////////////////
    	//remove the diagonal NOESY peaks:
    	Vector vecNoesyNoDia=new Vector();
    	for (i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy2=(Noesy)vecNoesy.elementAt(i);
    		double cs_h1=noesy2.getH1();
    		double cs_h2=noesy2.getH2();
    		    		    		
    		if (Math.abs(cs_h1-cs_h2)>0.02)
    			vecNoesyNoDia.add(noesy2);    		
    		
    	}//for (i=0;i< vecNoesy.size();i++)
    	vecNoesy=new Vector();
    	vecNoesy.addAll(vecNoesyNoDia);
    	
    	Vector vecNewNoesy=new Vector();
    	vecNewNoesy.addAll(vecNoesy);
    	
    	
    	// -------------------------------------
    	// (2.1) Initial ambiguous NOE assignment 
    	// for each noe peak.Assign noes within 
    	// error tolerance in each dimension
    	//-------------------------------------    		
    	Vector vecAmbgAsg=new Vector();
    	
    	vecAmbgAsg=asg.InitAmbiAssignment(nErr,c13Err,hnErr,haErr,h1Err,assignVec,allH1Vec,vecNewNoesy );    	
    	Collections.sort(vecAmbgAsg, new Noe.NoeComparator());   

    	//////////////////////////////
    	Vector vecNewNoes=new Vector();
    	vecNewNoes.addAll(vecAmbgAsg);
    	
    	if(strResNameScheme.equalsIgnoreCase("PDB-NEW"))
    	{
    		Vector vecTemp=new Vector();
    		Noe ee=new Noe();
    		vecTemp=ee.NameConvertFromPDBNewToPDBNew(vecNewNoes);
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecTemp);    		
    	}
    	if(strResNameScheme.equalsIgnoreCase("BMRB-NEW"))
    	{
    		Vector vecTemp=new Vector();
    		Noe ee=new Noe();
    		vecTemp=ee.NameConvertFromBMRBNewToPDBNew(vecNewNoes);
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecTemp);
    		
    		Vector vecAsgNew=new Vector();
    		H1CS h1cs=new H1CS();
    		vecAsgNew=h1cs.CSNameConvertFromBMRBNewToPDBNew(assignVec);
    		assignVec=new Vector();
    		assignVec.addAll(vecAsgNew);
    	}//if(strResNameScheme.equalsIgnoreCase("BMRB-NEW"))
    	vecAmbgAsg=new Vector();
    	vecAmbgAsg.addAll(vecNewNoes);
    	
    	////////////////////////////////////
    	//calibration for all initial NOE assignments,using cyana's scheme
    	Vector vecAmbgAsgTemp=noesy.SetCalibrationCyana(vecNewNoes,vecNewNoesy,constant[0]);
    	
    	vecNewNoes=new Vector();
    	vecNewNoes.addAll(vecAmbgAsgTemp);    	
     	///////////////////////////////////////
    	
    	/////////////////////////////////////////////////////////////
    	//add the sidechains for each bb
    	String userDir = System.getProperty("user.dir");////
    	File myDir = new File(userDir+strLocBBEnsem);
    	String[] contents = myDir.list();
    	if (contents == null) 
    	{
    		System.out.println(myDir.getName() + " is not a directory");
    		System.exit(0);
    	}
    	
    	String rotSrc=userDir+"/system/rot-lib/";//  src+ "rotasamp-small/"; 
    	//check whether the output directory exists:
    	String userDirTemp = System.getProperty("user.dir");////
    	String outPackPath=userDirTemp+strNewEnsemb;
    	int idPath=outPackPath.lastIndexOf('/');
    	String outPackDir=outPackPath.substring(0, idPath+1);
    	
    	boolean existsAlready = (new File(outPackDir)).exists();
    	if(existsAlready)
        {
        	//if the output directory already exists, we delete the whole directory
        	boolean success = deleteDir(new File(outPackDir));
        	if(!success)
        	{
        		System.out.println("Failed to delete an already existing output directry!");
        		System.exit(0);
        	}
        }
    	
    	//create a new directory
    	boolean exists = (new File(outPackDir)).exists();
        if (!exists) 
        {
            // create the directory if it does not exist
        	boolean success = (new File(outPackDir)).mkdir();
        	if(!success)
        	{
        		System.out.println("Failed to create output directry!");
        		System.exit(0);
        	}
        	
        } 
            	
    	int counter=0;  	
    	Vector vecPdb=new Vector();
    	Vector vecEnsemAfterThin=new Vector();
    	Vector vecRefPdb=new Vector();
    	for (int t=0; t<contents.length;t++) 
    	{    
    		String filename = contents[t]; 
	    	  		
	    	String strEnsemPdb="";
	    	strEnsemPdb=userDir+strLocBBEnsem+filename;	    
		    	
	    	if(Const.isDebug)
	    		System.out.println("input: "+strEnsemPdb);
	    	
	    	File f = new File(strEnsemPdb);    		 
    		if (!f.exists())
    		{
    			if(Const.isDebug)
    				 System.out.println("Couldn't find file "+ strEnsemPdb+ " for adding side-chains, continue...");
    			 continue;
    		}   		
    		
	    	Vector vecEnsemble=pp.readPdbAndNoeHarmFromEnsemble(strEnsemPdb);	    	
	    	
	    	if (vecEnsemble==null)
	    		continue;
	    	
	    	double[] vdwScore=new double[vecEnsemble.size()];
	    	vdw vander = new vdw();
	    	Vector vdwVec = new Vector();
	    	double[] vdwValue = new double[1];
	    	boolean hasVDW = false;
	    	double vdwLevel = 0.05;
	    	boolean printVDWViolation = false;       	
	    		
	    	for( i=0;i<vecEnsemble.size();i++)
	    	{    
	    		vdw Vdw=(vdw)vecEnsemble.elementAt(i);
	    		vecPdb=Vdw.getPDB();
	    		if (vecPdb.size()<1)
	    			continue;
	    		double noeRms=Vdw.getNoeRmsd();
	    		Vector vecTempPdbBB=pp.OutputBackbone(vecPdb);
	    		
	    		Vector pdbVecBBNew = pp.residueNameUpdate(vecSeq, vecTempPdbBB);
	    		
	    		double [] noeRmsd=new double[2];
	    		double [] noeHarmScore=new double[2];
	    		int [] numConflicts=new int[2];    	    		
	    		
	    		//place optimal rotamer based on noe pattern score:	    		
	    		Vector vecTempPdbRot=pp.RotamSelectAndStructure(h1Err, nErr, c13Err,pdbVecBBNew,assignVec,rotSrc,vecNewNoesy,4.5, 1,constant[0]);
	    			    			    		
	    		////////////////////////////////////////
	    		// refine the rotamers to avoid steric clash
	    		Vector vecTemptt=new Vector();
	    		if(strIsCheckLocalClash.equalsIgnoreCase("1"))
	    		{
		    		boolean [] resClash=new boolean[140];
		    		vdwVec = vander.convert2VDW(vecTempPdbRot);
		        	vander.OutputStericClashResidues(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.4, 1, resClash,true);
		        	int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.8);
		        	if(numClashes>0 )
		        	{
			        	int firstClash=0;
			        	for (i=0;i<resClash.length;i++)
			        	{
			        		if (resClash[i])
			        		{
			        			firstClash=i;
			        			break;
			        		}
			        	}
			        	vecTemptt=pp.RefineRotamerSelectionSearch(vecTempPdbRot,firstClash,rotSrc,0);
		        	}
		        	else
		        	{
		        		vecTemptt=vecTempPdbRot;
		        	}
	    		}
	    		else
	    			vecTemptt=vecTempPdbRot;
	    		////////////////////////////////////////   	
	    		
	        	String fileName =userDir+strNewEnsemb+counter+".pdb";//strOut+ strNewEnsemb+counter+".pdb";
		    	
		    	//output all top structures into one pdb file
		    	try
		    	{
		    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));	    		
	    	 
	    	  
		    		out.println("REMARK the backbone file name is:" + strEnsemPdb);///to file       		
        		
		    		pp.printToFile(vecTemptt,fileName, out);
		    	  	
		         	out.println("END");  
		    		out.close();
		    		
		    		counter+=1;
				}catch (FileNotFoundException e)
				{
					System.out.println("File not found: " + fileName);
				}catch (IOException e)
				{
				   System.out.println("IOException: the stack trace is:");
				   e.printStackTrace();
				}
	    	    
	    	}//for(int i=0;i<vecEnsemblePdb.size();i++)  		    
    	
    	}//for (int t=0; t<contents.length;t++)
    
    	///////////////////////////////////////////////////////////////////
    	
        long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for adding side-chains and pruning steric clashes is:  "+ totalTime +" minutes");  	
    	
    }
    
    /**
     * * merge all structures in previous ensemble into one enemble (with the same pdb name)
     * and cluster them according to given resolution.
     * similar to "doMergeNCluster", but clustering all structures in one directory
     * as an ensemble
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doMergeNClusterAll(String src, String strOut, String strInput)throws JampackException
    {	 
    	String strDelimited="000";
    	int maxStructurePerEnsemble=3000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////    	
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	String strOutPackedStr="",strPreStrName="",strInLocation="";
    	String strDelimitedTmp="";//not used yet
    	int maxEnsemSize=1000;
    	double resol_cluster=0.0;
    	int nSymID=0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    		
    	 			
    		if(paraMap.containsKey("MAXSTRUCTURES"))
    			maxEnsemSize  =   Integer.parseInt((String)paraMap.get("MAXSTRUCTURES"));			
    		
    		if(paraMap.containsKey("OUTPACKEDSTRUCTURES"))
    			strOutPackedStr  =  (String)paraMap.get("OUTPACKEDSTRUCTURES");    		  		
    		
    		if(paraMap.containsKey("RESOLCLUSTER"))
    			resol_cluster  =  Double.parseDouble((String)paraMap.get("RESOLCLUSTER"));
    	
    		if(paraMap.containsKey("INPUTFILELOCATION"))
    			strInLocation  =  (String)paraMap.get("INPUTFILELOCATION");
    		
    	}      
    	///////////////////////////////////////////////////////////////
    	//cluster the temporary packed structure in the ensemble
    	//
    	///////////////////////////////////////////////////////////////
    	//do the clustering step:
    	String userDir = System.getProperty("user.dir");////
    	src=userDir+strInLocation;
    	
		vdw vv=new vdw();
		String strPreEnsemb="";
				
		//check whether the output directory exists:
    	String userDirTemp = System.getProperty("user.dir");////
    	String outPackPath=userDirTemp+strOutPackedStr;
    	int idPath=outPackPath.lastIndexOf('/');
    	String outPackDir=outPackPath.substring(0, idPath+1);
    	
    	boolean existsAlready = (new File(outPackDir)).exists();
    	if(existsAlready)
        {
        	//if the output directory already exists, we delete the whole directory
        	boolean success = deleteDir(new File(outPackDir));
        	if(!success)
        	{
        		System.out.println("Failed to delete an already existing output directry!");
        		System.exit(0);
        	}
        }
    	
    	//create a new directory
    	boolean exists = (new File(outPackDir)).exists();
        if (!exists) 
        {
            // create the directory if it does not exist
        	boolean success = (new File(outPackDir)).mkdir();
        	if(!success)
        	{
        		System.out.println("Failed to create output directry!");
        		System.exit(0);
        	}
        	
        } 
		
		vv.mergeAndClusterAll(src, strOut, resol_cluster,  strOutPackedStr);
  	
        long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for clustering is:  "+ totalTime +" minutes");  	
    	
    }
    
    /**
     * add random loops to an ensemble of packed SSE structures (i.e. core structures).
     * It is quite simple: just merging two PDB files and not caring about the translations, rotation, etc.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doAddRandomLoops(String src, String strOut, String strInput)throws JampackException
    {	 
    	String strDelimited="000";
    	int maxStructurePerEnsemble=3000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	 // Read the input files
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	String strOutCompleteStr="",strInLocation="",strLoopPDB="";   	
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);       	 			
    		
    		if(paraMap.containsKey("OUTCOMPLETESTRUCTURES"))
    			strOutCompleteStr  =  (String)paraMap.get("OUTCOMPLETESTRUCTURES");      		
    	
    		if(paraMap.containsKey("LOCATIONSSEENSEMBLE"))
    			strInLocation  =  (String)paraMap.get("LOCATIONSSEENSEMBLE");
    		if(paraMap.containsKey("LOOPPDBNAME"))
    			strLoopPDB  =  (String)paraMap.get("LOOPPDBNAME");
    		
    	}      
    	
    	String strLoops=src+strLoopPDB;
    	Vector vecLoops=new Vector();
    	vecLoops=pp.readPdb(strLoops);
    	///////////////////////////////////////////////////////////////
    	
    	String userDir = System.getProperty("user.dir");////
    	File myDir = new File(userDir+strInLocation);
    	String[] contents = myDir.list();
    	if (contents == null) 
    	{
    		System.out.println(myDir.getName() + " is not a directory");
    		System.exit(0);
    	}
    	
    	
      	//check whether the output directory exists:
    	String userDirTemp = System.getProperty("user.dir");////
    	String outPackPath=userDirTemp+strOutCompleteStr;
    	int idPath=outPackPath.lastIndexOf('/');
    	String outPackDir=outPackPath.substring(0, idPath+1);
    	
    	boolean existsAlready = (new File(outPackDir)).exists();
    	if(existsAlready)
        {
        	//if the output directory already exists, we delete the whole directory
        	boolean success = deleteDir(new File(outPackDir));
        	if(!success)
        	{
        		System.out.println("Failed to delete an already existing output directry!");
        		System.exit(0);
        	}
        }
    	
    	//create a new directory
    	boolean exists = (new File(outPackDir)).exists();
        if (!exists) 
        {
            // create the directory if it does not exist
        	boolean success = (new File(outPackDir)).mkdir();
        	if(!success)
        	{
        		System.out.println("Failed to create output directry!");
        		System.exit(0);
        	}
        	
        } 
        
    	int counter=0;  	
    	Vector vecPdb=new Vector();
    	Vector vecEnsemAfterThin=new Vector();
    	Vector vecRefPdb=new Vector();
    	for (int t=0; t<contents.length;t++) 
    	{    
    		Vector vecPdbAll=new Vector ();
    		vecPdbAll.addAll(vecLoops);
    		
    		String filename = contents[t]; 
	    	  		
	    	String strEnsemPdb="";
	    	strEnsemPdb=userDir+strInLocation+filename;	    
		    	
	    	if(Const.isDebug)
	    		System.out.println("input: "+strEnsemPdb);
	    	
	    	File f = new File(strEnsemPdb);    		 
    		if (!f.exists())
    		{
    			if(Const.isDebug)
    				 System.out.println("Couldn't find file "+ strEnsemPdb+ " for adding side-chains, continue...");
    			 continue;
    		}   		
    		
    		Vector vecEnsemble=pp.readPdb(strEnsemPdb);	    	
	    	if (vecEnsemble==null)
	    		continue;
	    	
	    	vecPdbAll.addAll(vecEnsemble);
	    	Collections.sort(vecPdbAll, new Pdb.PdbComparator());
	    	
	        ////////////////////
	    	String fileName =userDir+strOutCompleteStr +counter+".pdb";//
	    	try
	    	{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		out.println("REMARK :  input PDB name=:" + strEnsemPdb);///to file
	    		out.println("REMARK :  ensemble ID=:" + counter);///to file
		        		   			
		    	pp.printToFile(vecPdbAll,fileName, out);
			    out.println("END");  
		    	out.close();
		    	
			}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
				System.out.println("IOException: the stack trace is:");
				e.printStackTrace();
			}
			 			
			counter++;    		    	
	    	
    	}//for (int t=0; t<contents.length;t++) 	
		
        long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for clustering is:  "+ totalTime +" minutes");  	
    	
    }
    
    /**
     * Read the engergy value from xplor PDB,
     * and output the top structures with lowest energies.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doReadPDBEnergy(String src, String strOut, String strInput)throws JampackException
    {	 
    	String strDelimited="000";
    	int maxStructurePerEnsemble=3000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	Vector vecVdw=new Vector();
    	
    	/////////////////////////////////////////////
    	/// Read the input files
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	String strInLocation="",isOut="",strOutName="",isCutOff="";   	
    	int outNumber=0,rowPos=0,colPos=0;
    	double cutOff=0.0;
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);      		   		
    	
    		if(paraMap.containsKey("LOCATIONSSEENSEMBLE"))
    			strInLocation  =  (String)paraMap.get("LOCATIONSSEENSEMBLE");    		
    		if(paraMap.containsKey("OUTPUTNUMBER"))
    			outNumber  =  Integer.parseInt((String)paraMap.get("OUTPUTNUMBER"));
    		if(paraMap.containsKey("ROWPOS"))
    			rowPos  =  Integer.parseInt((String)paraMap.get("ROWPOS"));
    		if(paraMap.containsKey("COLPOS"))
    			colPos  =  Integer.parseInt((String)paraMap.get("COLPOS"));      		
    		if(paraMap.containsKey("ISOUTPUT"))
    			isOut  =  (String)paraMap.get("ISOUTPUT");    
    		if(paraMap.containsKey("ISCUTOFF"))
    			isCutOff  =  (String)paraMap.get("ISCUTOFF");  
    		if(paraMap.containsKey("OUTNAME"))
    			strOutName  =  (String)paraMap.get("OUTNAME"); 
    		if(paraMap.containsKey("CUTOFF"))
    			cutOff = Double.parseDouble((String)paraMap.get("CUTOFF"));
    	}      	
    	
    	///////////////////////////////////////////////////////////////
    	
    	String userDir = System.getProperty("user.dir");
    	File myDir = new File(userDir+strInLocation);
    	String[] contents = myDir.list();
    	if (contents == null) 
    	{
    		System.out.println(myDir.getName() + " is not a directory");
    		System.exit(0);
    	}
    	
    	int counter=0;  	
    	Vector vecPdb=new Vector();
    	Vector vecEnsemAfterThin=new Vector();
    	Vector vecRefPdb=new Vector();
    	for (int t=0; t<contents.length;t++) 
    	{        		
    		String filename = contents[t]; 
	    	  		
	    	String strEnsemPdb="";
	    	strEnsemPdb=userDir+strInLocation+filename;	    
		    	
	    	if(Const.isDebug)
	    		System.out.println("input: "+strEnsemPdb);
	    	
	    	File f = new File(strEnsemPdb);    		 
    		if (!f.exists())
    		{
    			if(Const.isDebug)
    				 System.out.println("Couldn't find file "+ strEnsemPdb+ " for adding side-chains, continue...");
    			 continue;
    		}   		
    		Vector vecEnsemble=pp.readPdb(strEnsemPdb);	    	
	    	if (vecEnsemble==null)
	    		continue;
	    	
    		double [] noeEnerg=new double[1];
    		pp.readPdbEnergy(strEnsemPdb,noeEnerg, rowPos,colPos);
    		if(Const.isDebug)
	    		System.out.println("The energy is: "+noeEnerg[0]);
    		vecVdw.add(new vdw(vecEnsemble, noeEnerg[0],0.0 ,t,filename)); 		
 
    		
    	}//for (int t=0; t<contents.length;t++) 
				
    	Collections.sort(vecVdw, new vdw.VdwComparator() );   	
    	
    	if(isCutOff.equalsIgnoreCase("0"))
    	{
	    	for (i=0;i<Math.min(outNumber,contents.length);i++)
	    	{
	    		vdw vdw_temp=(vdw)vecVdw.elementAt(i) ;
	    		double scEnergy=vdw_temp.getVdwScore();
	    		String strID=vdw_temp.getStringID();
	    		System.out.println("the "+ (i+1) +"-th "+ "energy is : " +strID +" with energy="+ scEnergy );
	    		Vector vecPdbTemp=(Vector)vdw_temp.getPDB();
	    		
	       		//print out the pdb file
	        	if(isOut.equalsIgnoreCase("1"))
	        	{       		
		    		String fileName =userDir+strOutName +(i+1)+".pdb";//strOut+strOutName +i+".pdb";
			    	try
			    	{
			    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
			    		out.println("REMARK :  input PDB ID=:" + strID);///to file
			    		out.println("REMARK energy: " + scEnergy);///to file
				        		   			
				    	pp.printToFile(vecPdbTemp,fileName, out);
						    
				    	//out.println("TER");
	
					    out.println("END");  
				    	out.close();
				    	
					}catch (FileNotFoundException e)
					{
						System.out.println("File not found: " + fileName);
					}catch (IOException e)
					{
						System.out.println("IOException: the stack trace is:");
						e.printStackTrace();
					}            
	        	}//if(isOut.equalsIgnoreCase("1"))		    		
	    	}//for (i=0;i<outNumber;i++)
    	}//if(isCutOff.equalsIgnoreCase("0"))
    	if(isCutOff.equalsIgnoreCase("1"))
    	{
	    	for (i=0;i<vecVdw.size();i++)
	    	{
	    		vdw vdw_temp=(vdw)vecVdw.elementAt(i) ;
	    		double scEnergy=vdw_temp.getVdwScore();
	    		if(scEnergy>=cutOff)
	    			continue;
	    		
	    		String strID=vdw_temp.getStringID();
	    		System.out.println("the "+ (i+1) +"-th "+ "energy is : " +strID +" with energy="+ scEnergy );
	    		Vector vecPdbTemp=(Vector)vdw_temp.getPDB();
	    		
	       		//print out the pdb file
	        	if(isOut.equalsIgnoreCase("1"))
	        	{       		
		    		String fileName =userDir+strOutName +(i+1)+".pdb";//strOut+strOutName +i+".pdb";
			    	try
			    	{
			    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
			    		out.println("REMARK :  input PDB ID=:" + strID);///to file
			    		out.println("REMARK vdw_energy: " + scEnergy);///to file
				        		   			
				    	pp.printToFile(vecPdbTemp,fileName, out);
	
					    out.println("END");  
				    	out.close();
				    	
					}catch (FileNotFoundException e)
					{
						System.out.println("File not found: " + fileName);
					}catch (IOException e)
					{
						System.out.println("IOException: the stack trace is:");
						e.printStackTrace();
					}            
	        	}//if(isOut.equalsIgnoreCase("1"))		    		
	    	}//for (i=0;i<outNumber;i++)
    		
    	}//else
        long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for clustering is:  "+ totalTime +" minutes");  	
    	
    }
    
    /**
     * check the packing symmetry
     * that best fit the noe restraints (namely with minimum NOE RMSD).
     * This function is the same as the begining part of doSSEPacking function.
     * 
     * Note: Currently we consider all possible rotamer states for pruning packing symmetry degenarcies.
     * This might not be the best stragy. It might require more thoughts about this step.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doCheckPackingSym(String src, String strOut, String strInput)throws JampackException
    {	 
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////    	
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	String strSeq="", strSSES="",strOutPackedStr="",
    		strNoeFormat="",strNOETable="";   	
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");        		
    		if(paraMap.containsKey("SSES"))
    			strSSES  =  (String)paraMap.get("SSES");   	    		
    		if(paraMap.containsKey("OUTPACKEDSTRUCTURES"))
    			strOutPackedStr  =  (String)paraMap.get("OUTPACKEDSTRUCTURES");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");
    	}      
    	// -------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------    
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	

    	//    	-------------------------------------
    	// (1.4) Read the rotamer library path 
    	// 
    	// use the small Xtal' library from Richardson's lab
    	//-------------------------------------
    	String rotSrc=  src+ "rotasamp-small/";    	
    	
    	// read the NOE assignment table from cyana
    	String manualAsgFile = src+ strNOETable;//"noe-H123.txt";//strManualAsg; //"resonance.prot";  
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		vecManAsg=noe_temp.noeReader(manualAsgFile,0.0);  	    		
    		vecManAsgNew= noe_temp.ConvertXplorAsgToUplFormat(vecManAsg,vecSeq,"PDB-NEW");    	
    	}

    	// -------------------------------------
    	// (1.5) get the order of secondary 
    	//       structure elements
    	// 
    	//-------------------------------------
    	//read the order of sse packing
    	Vector vecOrderSSES=asg.getOrder(strSSES);
    	
    	//read the pdb file of each sse
    	Vector vecPdbSSE=new Vector();
    	for(i=0; i< vecOrderSSES.size();i++)
    	{
    		String strPdbFile= (String)vecOrderSSES.elementAt(i);
    		strPdbFile=src+strPdbFile+".pdb";
    		Vector pdbVecSSE_temp = pp.readPdb(strPdbFile);

        	//update backbone atom names according to protein sequence input
        	//residue name in previous backbone may be all in "ALA".     		
            Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, pdbVecSSE_temp);
        	
        	// this is final backbone pdb used for hosting rotamers
        	Vector pdbVecSSE = pp.nameConvertOrder(pdbVecSSE_temp2);        	
        	
           Collections.sort(pdbVecSSE, new Pdb.PdbComparator());
        	
           vecPdbSSE.add(pdbVecSSE);
    		
    	}//for(i=0; i< vecOrderSSES.size();i++)	
    	
   
    	/////////////////////////////////////////////////
    	// 2. Rotamer selection at each individual SSE
    	//
    	/////////////////////////////////////////////////    	
    
    	//this structure is in BMRB-NEW format:
    	Vector vecPdbRotSSE=new Vector();
    	//vecPdbRotSSE.addAll(vecPdbSSE);
    	Model md=new Model();
    	
    	for(i=0;i<vecPdbSSE.size();i++)
    	{
    		Vector vecPdbSseBb=(Vector)vecPdbSSE.elementAt(i);
    		
    		//be careful of rotation, we can't change the orientation of each sse
    		Vector vecPdbSseRot=pp.AllRotamersStructure(vecPdbSseBb,rotSrc);	    		
    		
    		//a function to rotate SSE back to the same coordinate system:
    		Vector vecPdbSseRotNew=md.BackonbeRotation(vecPdbSseRot,vecPdbSseBb); 
    		
    		Collections.sort(vecPdbSseRotNew, new Pdb.PdbComparator());
    		vecPdbRotSSE.add(vecPdbSseRotNew);
    	}
    	    	
    	/////////////////////////////////////////////////
    	Vector vecPdbA=new Vector();
    	Vector vecPdbB =new Vector();
    	
    	vecPdbA.addAll((Vector)vecPdbRotSSE.elementAt(0));
    	Collections.sort(vecPdbA, new Pdb.PdbComparator());
    	PdbRmsd pdr=new PdbRmsd();
    	double [] max_score=new double[4];
    	Vector[] vecEnsembPack = new Vector[4];
    	
    	PdbRmsd psd = new PdbRmsd();
     	double[] noeRmsd = new double[1];
     	double[] noeRms  = new double[4];
     	boolean debugNOE = true;	
      	Vector[] ubqByRdcVec = new Vector[4];
      	Vector[] vecNewNOE=new Vector[4];
      	
      	//we need to update the pseudo protons of NOEs
      	Vector vecNoeUpdateOrder=noe_temp.UpdateNOE(vecManAsgNew,vecPdbSSE);
      
      	
      	
      	///////////////////////////////////////////////////
    	//3. check the symmetry of packing helices
      	//
      	///////////////////////////////////////////////////
      	int minSym=0;
      	
    	Vector vecPdbATemp=new Vector();
    	Vector vecPdbBTemp =new Vector();
    	
    	vecPdbATemp.addAll((Vector)vecPdbRotSSE.elementAt(0));
    	Collections.sort(vecPdbATemp, new Pdb.PdbComparator());
    	vecPdbBTemp.addAll((Vector)vecPdbRotSSE.elementAt(1));
    	Collections.sort(vecPdbBTemp, new Pdb.PdbComparator());
    	Vector noeVecTemp =new Vector();
    	noeVecTemp.addAll((Vector)vecNoeUpdateOrder.elementAt(0));
    	
    	double minRms=999999.9;
    	Vector pdbAppxPacked=new Vector();
    	Vector vecPdbSave=new Vector();
     	for (i = 0; i < 4; i++)
	 	{	 
     		pdbAppxPacked=new Vector();
     		Vector vecPdbATempNew=new Vector();
     		if (i==0)
     			vecPdbATempNew.addAll(vecPdbATemp);
     		else
     			vecPdbATempNew = pp.newPdb(vecPdbATemp, Const.mat4ThreeDirs[i - 1]);
     		pdbAppxPacked=psd.calcMinPackingRms(noeVecTemp, vecPdbATempNew, vecPdbBTemp, noeRmsd, debugNOE);
    	    if(noeRmsd[0]<minRms)
    	    {
    	    	vecPdbSave=new Vector();
    	    	vecPdbSave.addAll(pdbAppxPacked);
    	    	minRms=noeRmsd[0];
    	    	minSym=i;
    	    }
    	
	 	}     
     	System.out.println("The symmetry ID is: "+ minSym);
    	
     	///////////////////////////////////////////////////    	  
     	String fileName=strOut+strOutPackedStr+".pdb";
     	//output the save pdb of the packed structure
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		out.println("REMARK :   The symmetry ID= "+minSym);
	    	   	
	    	pp.printToFile(vecPdbSave,fileName, out);        	
            out.println("TER");
         	out.println("END");
         	out.close();
    	}catch (FileNotFoundException e)
    	{
    		System.out.println("File not found: " + fileName);
    	}catch (IOException e)
    	{
    	   System.out.println("IOException: the stack trace is:");
    	   e.printStackTrace();
    	}
    	
    	       	
        long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		if(Const.isDebug)
			System.out.println("The total time for packing is:  "+ totalTime +" minutes");  	
    	
    }
    
    /**
     * calculate the alignment tensor.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doCalAlignmentTensor(String src, String strOut, String strInput)throws JampackException
    {	 
    	boolean isHelix=true;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	Pdb pp=new Pdb();
    	PdbRdc pdr=new PdbRdc();
    	int i, j;    	
    	ModelRdc  mdc= new ModelRdc();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	// Read the input files
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	
    	String strInPdbName="", strChFile="",strNhFile="",
    		strCaCoFile="",strCoNFile="";   	
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    	
    		if(paraMap.containsKey("CHFILE"))
    			strChFile  =  (String)paraMap.get("CHFILE"); 
    		
    		if(paraMap.containsKey("NHFILE"))
    			strNhFile  =  (String)paraMap.get("NHFILE"); 
    		if(paraMap.containsKey("COCAFILE"))
    			strCaCoFile  =  (String)paraMap.get("COCAFILE"); 
    		if(paraMap.containsKey("CONFILE"))
    			strCoNFile  =  (String)paraMap.get("CONFILE");     		
    		if(paraMap.containsKey("INPUTPDBNAME"))
    			strInPdbName  =  (String)paraMap.get("INPUTPDBNAME");     		
    				 		
    	} //for (i=0;i<paraVec.size();i++)    	
    	
    	//read the RDC data:
    	String strChRdcFile=src+strChFile;
    	String strNhRdcFile=src+strNhFile;
    	String strCaCoRdcFile=src+strCaCoFile;
    	String strCoNRdcFile=src+strCoNFile;
    	Dipolar dd  =  new Dipolar();
    	Vector<Dipolar> cacoRdc = new Vector();
		Vector<Dipolar> conRdc = new Vector();
		double scale_ca_to_nh=Math.pow(Const.dCA2HA /Const.dN2H, 3) * (Const.nitrogenGyroRatio /Const.carbonGyroRatio);
		scale_ca_to_nh=1/scale_ca_to_nh;
		
		Vector<Dipolar> nhRdc = dd.rdcRead(strNhRdcFile,1.0);
		Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,1.0);
		//Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,scale_ca_to_nh);		
		
		
		boolean existCaCo = (new File(strCaCoRdcFile)).exists();
		boolean existCoN = (new File(strCoNRdcFile)).exists();
		
		
		if(existCaCo)
			cacoRdc=dd.rdcRead(strCaCoRdcFile,0.19845);
		else
			System.out.println("Warn: The Co-Ca RDC doesn't exist.");
		if(existCoN) 			
			dd.rdcRead(strCoNRdcFile,0.12086);	    	
		else
			System.out.println("Warn: The Co-N RDC doesn't exist.");
    	//read the input pdb 
    	String strInPdb=src+strInPdbName;
    	Vector vecInPdb=new Vector();
    	Vector ppVec=pp.readPdb(strInPdb);
    	vecInPdb=pp.nameConvert(ppVec);
    	
    	double[] saupe = new double[4]; //for returning Saupe elements   	
    	
    	Vector<Pdb> ppVec1 = mdc.refineSaupe3(vecInPdb,  nhRdc, cahaRdc, saupe);
    }
    
    /**
     * refine the half start and end residues of pdb backbone solved from RDC-EXACT.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doRefineRdcExact(String src, String strOut, String strInput)throws JampackException
    {	     	
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	Pdb pp=new Pdb();
    	PdbRdc pdr=new PdbRdc();
    	int i, j;    	
    	ModelRdc  mdc= new ModelRdc();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// Read the input files
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double Syy=0.0,Szz=0.0,dbResolution=0.0,wtCoCA=0.0, wtCoN=0.0;
    	String strInPdbName="", strStartRes="",strEndRes="",strOutPdb="",strChFile="",strNhFile="",
    		strCaCoFile="",strCoNFile="",strTalos="",isRotate="",strIsHelix="";   	
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    	
    		if(paraMap.containsKey("CHFILE"))
    			strChFile  =  (String)paraMap.get("CHFILE"); 
    		if(paraMap.containsKey("TALOS"))
    			strTalos  =  (String)paraMap.get("TALOS"); 
    		if(paraMap.containsKey("ISHELIX"))
    			strIsHelix  =  (String)paraMap.get("ISHELIX"); 
    		
    		
    		if(paraMap.containsKey("ISROTATE"))
    			isRotate  =  (String)paraMap.get("ISROTATE"); 
    		
    		if(paraMap.containsKey("NHFILE"))
    			strNhFile  =  (String)paraMap.get("NHFILE"); 
    		if(paraMap.containsKey("COCAFILE"))
    			strCaCoFile  =  (String)paraMap.get("COCAFILE"); 
    		if(paraMap.containsKey("CONFILE"))
    			strCoNFile  =  (String)paraMap.get("CONFILE");     		
    		if(paraMap.containsKey("INPUTPDBNAME"))
    			strInPdbName  =  (String)paraMap.get("INPUTPDBNAME"); 
    		if(paraMap.containsKey("SYY"))
    			Syy  =  Double.parseDouble((String)paraMap.get("SYY"));
    		if(paraMap.containsKey("SZZ"))
    			Szz  =  Double.parseDouble((String)paraMap.get("SZZ"));    	
    		
    		if(paraMap.containsKey("SZZ"))
    			Szz  =  Double.parseDouble((String)paraMap.get("SZZ"));   
    		if(paraMap.containsKey("WTCOCA"))
    			wtCoCA=  Double.parseDouble((String)paraMap.get("WTCOCA"));   
    		if(paraMap.containsKey("WTCON"))
    			wtCoN=  Double.parseDouble((String)paraMap.get("WTCON")); 
    			
    		if(paraMap.containsKey("STARTRESIDUES"))
    			strStartRes  =  (String)paraMap.get("STARTRESIDUES");  
    		if(paraMap.containsKey("ENDRESIDUES"))
    			strEndRes  =  (String)paraMap.get("ENDRESIDUES");      		
    		if(paraMap.containsKey("OUTPDBNAME"))
    			strOutPdb  =  (String)paraMap.get("OUTPDBNAME");    		
    		if(paraMap.containsKey("RESOLUTION"))
    			dbResolution  =  Double.parseDouble((String)paraMap.get("RESOLUTION"));     		
    	} //for (i=0;i<paraVec.size();i++)
    	
    	boolean isHelix=true;
    	if(strIsHelix.equalsIgnoreCase("1"))
    		isHelix=true;
    	else
    		isHelix=false;
    	
    	//read the talos angle table:
    	String strTalosFile=src+strTalos;
    	PhiPsi phipsi0 =new PhiPsi();
    	Vector vecTalos=phipsi0.talosAngleReader(strTalosFile);
    	
    	//read the RDC data:
    	String strChRdcFile=src+strChFile;
    	String strNhRdcFile=src+strNhFile;
    	String strCaCoRdcFile=src+strCaCoFile;
    	String strCoNRdcFile=src+strCoNFile;
    	Dipolar dd  =  new Dipolar();
		final Vector<Dipolar> nhRdc = dd.rdcRead(strNhRdcFile,1.0);
		final Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,1.0);
		
		boolean existCaCo = (new File(strCaCoRdcFile)).exists();
		boolean existCoN = (new File(strCoNRdcFile)).exists();
		
		
		Vector<Dipolar> cacoRdc =new Vector();
		Vector<Dipolar> conRdc =new Vector();
		 
		if(existCaCo)
			cacoRdc=dd.rdcRead(strCaCoRdcFile,0.19845);
		else
			System.out.println("Warning: The Co-Ca RDC doesn't exist...");
		if(existCoN) 			
			conRdc=dd.rdcRead(strCoNRdcFile,0.12086);	    	
		else
			System.out.println("Warning: The Co-N RDC doesn't exist...");
		
    	//read the input pdb 
    	String strInPdb=src+strInPdbName;
    	Vector vecInPdb=new Vector();
    	Vector ppVec=pp.readPdb(strInPdb);
    	vecInPdb=pp.nameConvert(ppVec);    	
    	
    	//read the start and end residues NOs:
    	Vector vecStartRes=asg.getOrder(strStartRes);   
    	Vector vecEndRes=asg.getOrder(strEndRes); 
    	
    	Vector<Pdb> pdbVecSC=new Vector();
    	pdbVecSC.addAll(ppVec);
    	
    	//rotate the pdb to let it fit the rdc data
    	double[] rdc1Cal = new double[vecInPdb.size()];
    	double[] rdc2Cal = new double[vecInPdb.size()];
    	double[] saupe = new double[4]; //for returning Saupe elements
    	
    	if(isRotate.equalsIgnoreCase("1"))
    	{
	    	Matrix mm = pdr.bestFit(vecInPdb, nhRdc, cahaRdc, rdc1Cal, rdc2Cal, saupe);
	    	Vector<Pdb> pdbVec2 = pp.newPdb(vecInPdb, mm);
	    	pdbVecSC=pp.newPdb(ppVec,mm);
	    	   	
	    	vecInPdb=new Vector();
	    	vecInPdb.addAll(pdbVec2);
    	}
    	
    	//get the sequence of all phi/psi angles:
    	double [][] Inv_temp = {{1.0, 0.0, 0.0},{0.0, 1.0, 0.0},{0.0, 0.0, 1.0}};
       	Matrix Mat  = new Matrix(Inv_temp);	
		
    	Vector vecPhiPsi=pp.PhiPsiTotalPdb(vecInPdb);
    	
    	Vector vecPdbNew=new Vector();
    	
		//refine the end residues:
		for (j=0;j<vecEndRes.size();j++)
		{
			String strNo=(String) vecEndRes.elementAt(j);
			int ResNo= Integer.parseInt(strNo);
							
			double [] coordN  = new double[3];
			double [] coordNH = new double[3];
			double [] coordCA = new double[3]; 
			
			int indA = Collections.binarySearch(vecInPdb, new Pdb(ResNo), new Pdb.PdbComparator() );
			if (indA<0)
				continue;
			
			Pdb pdbA = (Pdb)vecInPdb.elementAt(indA);
			String resId=pdbA.getResidue();
			Vector atomVecSC=new Vector();
			Vector atomVecA = pdbA.getAtomVec();
			
			Pdb pdbAsc=(Pdb)pdbVecSC.elementAt(indA);
			Vector atomVecAsc=pdbAsc.getAtomVec();
			for (int k=0;k<atomVecAsc.size();k++)
			{
				Cartesian cc = (Cartesian)atomVecAsc.elementAt(k);
	    	    String atom = cc.getAtom();
	    	    if(! (atom.equalsIgnoreCase("N") || atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||atom.equalsIgnoreCase("CA")
	    	    		||atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("O")||atom.equalsIgnoreCase("CB")) )
	    	    	atomVecSC.add(cc);
			}
			
			for (int k=0;k<atomVecA.size();k++)
			{
				Cartesian cc = (Cartesian)atomVecA.elementAt(k);
	    	    String atom = cc.getAtom();	    	   
	    	    if (atom.equalsIgnoreCase("N"))
	    	    	coordN = cc.getXYZ();
	    	    else if (atom.equalsIgnoreCase("CA"))
	    	    	coordCA=cc.getXYZ();
	    	    else if (atom.equalsIgnoreCase("HN")||atom.equalsIgnoreCase("H"))
	    	    	coordNH = cc.getXYZ();		    
			}//for (int k=0;k<atomVecA.size();k++)
			
			int ind1 = Collections.binarySearch(vecPhiPsi, new PhiPsi(ResNo), new PhiPsi.PPComparator());
			PhiPsi phipsi = (PhiPsi)vecPhiPsi.elementAt(ind1);
			double dbPhi=phipsi.getPhi();
			//sampling all possible psi angles;
			double minSc=99999.9;
			double psiSave=0.0;
			
			//read the psi range based on Ranmanchandran and talos angles:			
			double psiHigh=0.0; //in the degre unit
			double psiLow=0.0;
			double psiRamHigh=0.0;
			double psiRamLower=0.0;
			if(isHelix)
			{
				psiRamHigh = Const.psiHighHelix/Const.cst;
				psiRamLower  = Const.psiLowHelix/Const.cst;
			}else
			{
				psiRamHigh = Const.psiHighBeta/Const.cst;
				psiRamLower  = Const.psiLowBeta/Const.cst;
			}
			psiHigh=psiRamHigh;
			psiLow=psiRamLower;
			
			for (int h=0; h<vecTalos.size();h++)
			{
				PhiPsi phipsiTemp=(PhiPsi)vecTalos.elementAt(h);
				int resNo_temp=phipsiTemp.getResidueNo();
				if(resNo_temp==ResNo)
				{
					psiHigh=Math.min( phipsiTemp.getPsiUpper(),psiRamHigh);
					psiLow=Math.max( phipsiTemp.getPsiLower(), psiRamLower);
				}
			}
			
			for (int t=0;t<( psiHigh-psiLow )/dbResolution;t++)
			{
				double ranPsi = psiLow*Const.cst + t * dbResolution * Math.PI / 180.0;			 
				
				Pdb pp1 =mdc.coordByResidueFull(dbPhi, ranPsi, coordN, coordNH, coordCA, ResNo, resId, false);
				Vector vecPp1=new Vector();
				vecPp1.add(pp1);
				
				double r_nh=pdr.BackCalOnePdb(vecPp1, nhRdc, "N", "H", Mat, -Syy-Szz, Syy, Szz, Const.nhRatio);
			    double r_caha=pdr.BackCalOnePdb(vecPp1, cahaRdc, "CA", "HA", Mat, -Syy-Szz, Syy, Szz, Const.cahaRatio);
			    double r_caco=pdr.BackCalCACOOnePdb(vecPp1, cacoRdc, "C", "CA", Mat, -Syy-Szz, Syy, Szz, Const.cacoRatio);
				
				double rT = (r_caha/Const.cahaRatio)+(r_nh/Const.nhRatio)+wtCoCA*(r_caco/Const.cacoRatio);//+wtCoN*(r_con/Const.conRatio);
				if(minSc>rT)
				{
					minSc=rT;
					psiSave=ranPsi;
				}
			}//for (int t=0;t<(Const.psiHighHelix-Const.psiLowHelix)/dbResolution;t++)
			
			//add the pdb with the saved psi angle:
			Pdb ppSave=mdc.coordByResidueFull(dbPhi, psiSave, coordN, coordNH, coordCA, ResNo, resId, false);
			Vector atomVec=ppSave.getAtomVec();
			atomVec.addAll(atomVecSC);
			vecPdbNew.add(new Pdb(ResNo, resId, atomVec));
		}//for (j=0;j<vecEndRes.size();j++)
		
		//refine the start residues:
		for(j=0;j<vecStartRes.size();j++)
		{
			Vector atomVecSC=new Vector();
			String strNo=(String) vecStartRes.elementAt(j);
			int ResNo= Integer.parseInt(strNo);						
			
			int indA = Collections.binarySearch(pdbVecSC, new Pdb(ResNo), new Pdb.PdbComparator() );
			if (indA<0)
				continue;
			Pdb pdbA = (Pdb)pdbVecSC.elementAt(indA);
			String resId=pdbA.getResidue();
			Vector atomVecA=pdbA.getAtomVec();    			
			for (int k=0;k<atomVecA.size();k++)
			{
				Cartesian cc = (Cartesian)atomVecA.elementAt(k);
	    	    String atom = cc.getAtom();
	    	    if(! (atom.equalsIgnoreCase("N") || atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||atom.equalsIgnoreCase("CA")
	    	    		||atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("O")||atom.equalsIgnoreCase("CB")) )
	    	    	atomVecSC.add(cc);    	    	   
			}//for (int k=0;k<atomVecA.size();k++)
			
			//n, nh, ca coordinate of NEXT residue:
			double [] coordN  = new double[3];
			double [] coordNH = new double[3];
			double [] coordCA = new double[3]; 
			int indB = Collections.binarySearch(vecInPdb, new Pdb(ResNo+1), new Pdb.PdbComparator() );
			Pdb pdbB = (Pdb)vecInPdb.elementAt(indB);    			
			
			Vector atomVecB = pdbB.getAtomVec();
			for (int k=0;k<atomVecB.size();k++)
			{
				Cartesian cc = (Cartesian)atomVecB.elementAt(k);
	    	    String atom = cc.getAtom();
	    	   
	    	    if (atom.equalsIgnoreCase("N"))
	    	    	coordN = cc.getXYZ();
	    	    else if (atom.equalsIgnoreCase("CA"))
	    	    	coordCA=cc.getXYZ();
	    	    else if (atom.equalsIgnoreCase("HN")||atom.equalsIgnoreCase("H"))
	    	    	coordNH = cc.getXYZ();		    
			}//for (int k=0;k<atomVecA.size();k++)
			
			int ind1 = Collections.binarySearch(vecPhiPsi, new PhiPsi(ResNo), new PhiPsi.PPComparator());
			PhiPsi phipsi = (PhiPsi)vecPhiPsi.elementAt(ind1);
			double dbPsi=phipsi.getPsi();
			//sampling all possible psi angles;
			double minSc=99999.9;
			double phiSave=0.0;    						
			
			//read the psi range based on Ranmanchandran and talos angles:			
			double phiHigh=0.0; //in the degre unit
			double phiLow=0.0;
			double phiRamHigh=0.0;
			double phiRamLower=0.0;
			if(isHelix)
			{
				phiRamHigh = Const.phiHighHelix/Const.cst;
				phiRamLower  = Const.phiLowHelix/Const.cst;
			}else
			{
				phiRamHigh = Const.phiHighBeta/Const.cst;
				phiRamLower  = Const.phiLowBeta/Const.cst;
			}			
			phiHigh=phiRamHigh;
			phiLow=phiRamLower;
			
			for (int h=0; h<vecTalos.size();h++)
			{
				PhiPsi phipsiTemp=(PhiPsi)vecTalos.elementAt(h);
				int resNo_temp=phipsiTemp.getResidueNo();
				if(resNo_temp==ResNo)
				{
					phiHigh=Math.min( phipsiTemp.getPhiUpper(),phiRamHigh);
					phiLow=Math.max( phipsiTemp.getPhiLower(), phiRamLower);
				}
			}
   			for (int t=0;t<( phiHigh-phiLow )/dbResolution;t++)
			{
				double ranPhi = phiLow*Const.cst + t * dbResolution * Math.PI / 180.0;	
		
				Pdb pp1 =mdc.coordByBackward(ranPhi, dbPsi, coordN, coordNH, coordCA, ResNo, resId);
				
				Vector vecPp1=new Vector();
				vecPp1.add(pp1);
				 
					
				double r_nh=pdr.BackCalOnePdb(vecPp1, nhRdc, "N", "H", Mat, -Syy-Szz, Syy, Szz, Const.nhRatio);
			    double r_caha=pdr.BackCalOnePdb(vecPp1, cahaRdc, "CA", "HA", Mat, -Syy-Szz, Syy, Szz, Const.cahaRatio);
			    double r_caco=pdr.BackCalCACOOnePdb(vecPp1, cacoRdc, "C", "CA", Mat, -Syy-Szz, Syy, Szz, Const.cacoRatio);
				
				double rT = (r_caha/Const.cahaRatio)+(r_nh/Const.nhRatio)+wtCoCA*(r_caco/Const.cacoRatio);//+wtCoN*(r_con/Const.conRatio);
				if(minSc>rT)
				{
					minSc=rT;
					phiSave=ranPhi;
				}
			}//for (int t=0;t<(Const.psiHighHelix-Const.psiLowHelix)/dbResolution;t++)
			
			//add the pdb with the saved psi angle:
			Pdb ppSave=mdc.coordByBackward(phiSave, dbPsi, coordN, coordNH, coordCA, ResNo, resId);
			Vector atomVec=ppSave.getAtomVec();
			atomVec.addAll(atomVecSC);
			vecPdbNew.add(new Pdb(ResNo, resId, atomVec));

		}//for(j=0;j<vecStartRes.size();j++)
		
		//add all other residues:
		for(i=0;i<pdbVecSC.size();i++)
    	{
    		boolean isMiddle=true;
    		Pdb ppTemp=(Pdb)pdbVecSC.elementAt(i);
    		int resNoTemp=ppTemp.getResidueNo();
    		for (j=0;j<vecEndRes.size();j++)
    		{
    			String strNo=(String) vecEndRes.elementAt(j);
				int ResNo= Integer.parseInt(strNo);
				if(resNoTemp==ResNo)
					isMiddle=false;    				
    		}
    		for(j=0;j<vecStartRes.size();j++)
    		{
    			Vector atomVecSC=new Vector();
    			String strNo=(String) vecStartRes.elementAt(j);
				int ResNo= Integer.parseInt(strNo);				
				if(resNoTemp==ResNo)
					isMiddle=false;
    		}    		
    		if(isMiddle==true)
    		{
    			vecPdbNew.add(ppTemp);
    		}    	    		
    	}//for(i=0;i<vecInPdb.size();i++)
    	
    	Collections.sort(vecPdbNew, new Pdb.PdbComparator());
    	
    	//write the new pdb file:
    	String fileName =strOut+strOutPdb;
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));    		    		   			
	    	pp.printToFile(vecPdbNew,fileName, out);		    	
	    	out.println("END");  
	    	out.close();
	    	
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
			System.out.println("IOException: the stack trace is:");
			e.printStackTrace();
		}            

    }
    
    /**
     * refine the middle residues of pdb backbone solved from RDC-EXACT.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doRefineRdcExactMiddle(String src, String strOut, String strInput)throws JampackException
    {	 
    	boolean isHelix=true;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	Pdb pp=new Pdb();
    	PdbRdc pdr=new PdbRdc();
    	int i, j;    	
    	ModelRdc  mdc= new ModelRdc();   
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// Read the input files
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double Syy=0.0,Szz=0.0,dbResolution=0.0;
    	String strInPdbName="", strMiddleRes="",strOutPdb="",strChFile="",strNhFile="",
    		strCaCoFile="",strCoNFile="",strTalos="";   	
    	int nEndResNo=0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    	
    		if(paraMap.containsKey("ENDRESIDUE"))
    			nEndResNo  =  Integer.parseInt((String)paraMap.get("ENDRESIDUE"));
    		
    		if(paraMap.containsKey("CHFILE"))
    			strChFile  =  (String)paraMap.get("CHFILE"); 
    		if(paraMap.containsKey("TALOS"))
    			strTalos  =  (String)paraMap.get("TALOS"); 
    		if(paraMap.containsKey("NHFILE"))
    			strNhFile  =  (String)paraMap.get("NHFILE"); 
    		if(paraMap.containsKey("COCAFILE"))
    			strCaCoFile  =  (String)paraMap.get("COCAFILE"); 
    		if(paraMap.containsKey("CONFILE"))
    			strCoNFile  =  (String)paraMap.get("CONFILE");     		
    		if(paraMap.containsKey("INPUTPDBNAME"))
    			strInPdbName  =  (String)paraMap.get("INPUTPDBNAME"); 
    		if(paraMap.containsKey("SYY"))
    			Syy  =  Double.parseDouble((String)paraMap.get("SYY"));
    		if(paraMap.containsKey("SZZ"))
    			Szz  =  Double.parseDouble((String)paraMap.get("SZZ"));    		
    		if(paraMap.containsKey("MIDDLERESIDUES"))
    			strMiddleRes  =  (String)paraMap.get("MIDDLERESIDUES");  
    	     		
    		if(paraMap.containsKey("OUTPDBNAME"))
    			strOutPdb  =  (String)paraMap.get("OUTPDBNAME");    		
    		if(paraMap.containsKey("RESOLUTION"))
    			dbResolution  =  Double.parseDouble((String)paraMap.get("RESOLUTION"));     		
    	} //for (i=0;i<paraVec.size();i++)
    	
    	//read the talos angle table:
    	String strTalosFile=src+strTalos;
    	PhiPsi phipsi0 =new PhiPsi();
    	Vector vecTalos=phipsi0.talosAngleReader(strTalosFile);
    	
    	//read the RDC data:
    	String strChRdcFile=src+strChFile;
    	String strNhRdcFile=src+strNhFile;
    	String strCaCoRdcFile=src+strCaCoFile;
    	String strCoNRdcFile=src+strCoNFile;
    	Dipolar dd  =  new Dipolar();
		final Vector<Dipolar> nhRdc = dd.rdcRead(strNhRdcFile,1.0);
		final Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,2.008);
		final Vector<Dipolar> cacoRdc = dd.rdcRead(strCaCoRdcFile,0.19845);
		final Vector<Dipolar> conRdc = dd.rdcRead(strCoNRdcFile,0.12086);		
    	
    	//read the input pdb 
    	String strInPdb=src+strInPdbName;
    	Vector vecInPdb=new Vector();
    	Vector ppVec=pp.readPdb(strInPdb);
    	Collections.sort(ppVec, new Pdb.PdbComparator());
    	vecInPdb=pp.nameConvert(ppVec);    	   	
    	
    	//read the middle residues NOs:
    	Vector vecMiddleRes=asg.getOrder(strMiddleRes);      
    	
    	//rotate the pdb to let it fit the rdc data
    	double[] rdc1Cal = new double[vecInPdb.size()];
    	double[] rdc2Cal = new double[vecInPdb.size()];
    	double[] saupe = new double[4]; //for returning Saupe elements
    	Matrix mm = pdr.bestFit(vecInPdb, nhRdc, cahaRdc, rdc1Cal, rdc2Cal, saupe);
    	Vector<Pdb> pdbVec2 = pp.newPdb(vecInPdb, mm);
    	Vector<Pdb> pdbVecSC=pp.newPdb(ppVec,mm);
    	
    	
    	vecInPdb=new Vector();
    	vecInPdb.addAll(pdbVec2);
    	
    	//get the sequence of all phi/psi angles:
    	double [][] Inv_temp = {{1.0, 0.0, 0.0},{0.0, 1.0, 0.0},{0.0, 0.0, 1.0}};
       	Matrix Mat  = new Matrix(Inv_temp);	
		
    	Vector vecPhiPsi=pp.PhiPsiTotalPdb(vecInPdb);
    	
    	Vector vecPdbNew=new Vector();
    	
		//refine the phi/psi angles of middle residues:
		for (j=0;j<vecMiddleRes.size();j++)
		{
			String strNo=(String) vecMiddleRes.elementAt(j);
			int ResNo= Integer.parseInt(strNo);
							
			double [] coordN  = new double[3];
			double [] coordNH = new double[3];
			double [] coordCA = new double[3]; 
			
			int indA = Collections.binarySearch(vecInPdb, new Pdb(ResNo), new Pdb.PdbComparator() );
			Pdb pdbA = (Pdb)vecInPdb.elementAt(indA);
			String resId=pdbA.getResidue();
			Vector atomVecSC=new Vector();
			Vector atomVecA = pdbA.getAtomVec();
			
			//add previous pdbs
			for (int k=0;k<indA;k++)
			{
				Pdb pdbTT=(Pdb)vecInPdb.elementAt(k);
				vecPdbNew.add(pdbTT);
			}
			
			//add the residues after end residue (i.e. other SSEs)
			int EndInd = Collections.binarySearch(vecInPdb, new Pdb(nEndResNo), new Pdb.PdbComparator() );
			for (int k=EndInd+1;k<vecInPdb.size();k++)
			{
				Pdb pdbTT=(Pdb)vecInPdb.elementAt(k);
				vecPdbNew.add(pdbTT);
			}
						
			Pdb pdbAsc=(Pdb)pdbVecSC.elementAt(indA);
			Vector atomVecAsc=pdbAsc.getAtomVec();
			for (int k=0;k<atomVecAsc.size();k++)
			{
				Cartesian cc = (Cartesian)atomVecAsc.elementAt(k);
	    	    String atom = cc.getAtom();
	    	    if(! (atom.equalsIgnoreCase("N") || atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||atom.equalsIgnoreCase("CA")
	    	    		||atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("O")||atom.equalsIgnoreCase("CB")) )
	    	    	atomVecSC.add(cc);
			}
			
			for (int k=0;k<atomVecA.size();k++)
			{
				Cartesian cc = (Cartesian)atomVecA.elementAt(k);
	    	    String atom = cc.getAtom();	    	   
	    	    if (atom.equalsIgnoreCase("N"))
	    	    	coordN = cc.getXYZ();
	    	    else if (atom.equalsIgnoreCase("CA"))
	    	    	coordCA=cc.getXYZ();
	    	    else if (atom.equalsIgnoreCase("HN")||atom.equalsIgnoreCase("H"))
	    	    	coordNH = cc.getXYZ();		    
			}//for (int k=0;k<atomVecA.size();k++)
			
			//sampling all possible psi/phi angles;
			double minSc=99999.9;
			double psiSave=0.0;
			double phiSave=0.0;
			
			//read the psi range based on Ranmanchandran and talos angles:			
			double psiHigh=0.0; //in the degre unit
			double psiLow=0.0;
			double psiRamHigh=0.0;
			double psiRamLower=0.0;
			if(isHelix)
			{
				psiRamHigh = Const.psiHighHelix/Const.cst;
				psiRamLower  = Const.psiLowHelix/Const.cst;
			}else
			{
				psiRamHigh = Const.psiHighBeta/Const.cst;
				psiRamLower  = Const.psiLowBeta/Const.cst;
			}
			for (int h=0; h<vecTalos.size();h++)
			{
				PhiPsi phipsiTemp=(PhiPsi)vecTalos.elementAt(h);
				int resNo_temp=phipsiTemp.getResidueNo();
				if(resNo_temp==ResNo)
				{
					psiHigh=Math.min( phipsiTemp.getPsiUpper(),psiRamHigh);
					psiLow=Math.max( phipsiTemp.getPsiLower(), psiRamLower);
				}
			}
			double phiHigh=0.0; //in the degre unit
			double phiLow=0.0;
			double phiRamHigh=0.0;
			double phiRamLower=0.0;
			if(isHelix)
			{
				phiRamHigh = Const.phiHighHelix/Const.cst;
				phiRamLower  = Const.phiLowHelix/Const.cst;
			}else
			{
				phiRamHigh = Const.phiHighBeta/Const.cst;
				phiRamLower  = Const.phiLowBeta/Const.cst;
			}
			for (int h=0; h<vecTalos.size();h++)
			{
				PhiPsi phipsiTemp=(PhiPsi)vecTalos.elementAt(h);
				int resNo_temp=phipsiTemp.getResidueNo();
				if(resNo_temp==ResNo)
				{
					phiHigh=Math.min( phipsiTemp.getPhiUpper(),phiRamHigh);
					phiLow=Math.max( phipsiTemp.getPhiLower(), phiRamLower);
				}
			}
			
			//save the rest of phi/psi angle chain:			
			int ind0 = Collections.binarySearch(vecPhiPsi, new PhiPsi(ResNo), new PhiPsi.PPComparator());
			int indEnd = Collections.binarySearch(vecPhiPsi, new PhiPsi(nEndResNo), new PhiPsi.PPComparator());
			
			
			Vector vecPhiS = new Vector(); 
			Vector VecPsiS = new Vector();  
			for (int tt=ind0+1;tt<=indEnd;tt++)
			{
				PhiPsi phipsiT=(PhiPsi)vecPhiPsi.elementAt(tt);
				vecPhiS.add(phipsiT.getPhi());
				VecPsiS.add(phipsiT.getPsi());
			}		
			
			for (int t=0;t<( psiHigh-psiLow )/dbResolution;t++)
			{
				double ranPsi = psiLow*Const.cst + t * dbResolution * Math.PI / 180.0;			 
				
				for (int w=0;w<(phiHigh-psiLow)/dbResolution;w++  )
				{
					double ranPhi=phiLow*Const.cst +w* dbResolution*Math.PI / 180.0;
					Vector<Double> vecPhiSave=new Vector<Double>();
					Vector<Double> vecPsiSave=new Vector<Double>();
					vecPhiSave.add(ranPhi);
					vecPhiSave.addAll(vecPhiS);
					vecPsiSave.add(ranPsi);
					vecPsiSave.addAll(VecPsiS);			
					
					double[] phiS = new double[vecPhiSave.size()]; 
					double[] psiS = new double[vecPsiSave.size()]; 
					for(int w2=0;w2<vecPhiSave.size();w2++)
						phiS[w2]=(double)vecPhiSave.elementAt(w2);
					for(int w2=0;w2<vecPsiSave.size();w2++)
						psiS[w2]=(double)vecPsiSave.elementAt(w2);
					Vector<Pdb> pdbHelixTemp = new Vector();					
			       	pdbHelixTemp= mdc.modelBuild(phiS, psiS, coordN, coordNH, coordCA, ResNo, false);
			       
			       	//check the steric clash
			    	boolean[] resIndex=new boolean[pdbHelixTemp.size()];		    		
		    		for (int kk=0;kk<resIndex.length;kk++)
		    			resIndex[kk]=false;
		    		Vector vecPdbSseRot2=new Vector();
		    		String rotSrc=  src+ "rotasamp-small/"; 
			       	vecPdbSseRot2=pp.AlaninizeStructure(pdbHelixTemp,resIndex,rotSrc);
			       	vdw vander = new vdw();
			    	Vector vdwVec = new Vector();

			       	double[] vdwValue = new double[1];
			       	double vdwLevel = 0.05;
			       	boolean printVDWViolation = false;
			       	int[] clashResNo=new int[2];
			    	vdwVec = vander.convert2VDW(vecPdbSseRot2);
			    	boolean isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
							true, 0.4, 1, clashResNo); 
			    	
			    	if(isStericClash)
			    	{
			    		System.out.println("here we have one steric clash...");
			    		continue;
			    	}
			       	
			       	double r_nh=pdr.BackCalOnePdb(pdbHelixTemp, nhRdc, "N", "H", Mat, -Syy-Szz, Syy, Szz, Const.nhRatio);
				    double r_caha=pdr.BackCalOnePdb(pdbHelixTemp, cahaRdc, "CA", "HA", Mat, -Syy-Szz, Syy, Szz, Const.cahaRatio);
				    double r_caco=pdr.BackCalCACOOnePdb(pdbHelixTemp, cacoRdc, "C", "CA", Mat, -Syy-Szz, Syy, Szz, Const.cacoRatio);
					double r_con=pdr.BackCalCONOnePdb(pdbHelixTemp, conRdc, "C", "N",  Mat, -Syy-Szz, Syy, Szz, Const.conRatio);
					
					double rT = (r_caha/Const.cahaRatio)+(r_nh/Const.nhRatio)+Const.wtCaCo*(r_caco/Const.cacoRatio);//+Const.wtCoN*(r_con/Const.conRatio);
					if(minSc>rT)
					{
						minSc=rT;
						phiSave=ranPhi;
						psiSave=ranPsi;
					}																		
				}//for (int w=0;w<(phiHigh-psiLow)/dbResolution;w++  )				
			}//for (int t=0;t<(Const.psiHighHelix-Const.psiLowHelix)/dbResolution;t++)
			
			//add the pdb with the saved psi angle:			
			Vector<Double> vecPhiSave=new Vector<Double>();
			Vector<Double> vecPsiSave=new Vector<Double>();
			vecPhiSave.add(phiSave);
			vecPhiSave.addAll(vecPhiS);
			vecPsiSave.add(psiSave);
			vecPsiSave.addAll(VecPsiS);			
			
			double[] phiS = new double[vecPhiSave.size()]; 
			double[] psiS = new double[vecPsiSave.size()]; 
			for(int w2=0;w2<vecPhiSave.size();w2++)
				phiS[w2]=(double)vecPhiSave.elementAt(w2);
			for(int w2=0;w2<vecPsiSave.size();w2++)
				psiS[w2]=(double)vecPsiSave.elementAt(w2);
			Vector<Pdb> pdbHelixTemp = new Vector();					
	       	pdbHelixTemp= mdc.modelBuild(phiS, psiS, coordN, coordNH, coordCA, ResNo, false);
	       	vecPdbNew.addAll(pdbHelixTemp);		
			
		}//for (j=0;j<vecEndRes.size();j++)
		    	
    	Collections.sort(vecPdbNew, new Pdb.PdbComparator());
    	
    	//write the new pdb file:
    	String fileName =strOut+strOutPdb;
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    	pp.printToFile(vecPdbNew,fileName, out);			    	    	
	    	out.println("END");  
	    	out.close();
	    	
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
			System.out.println("IOException: the stack trace is:");
			e.printStackTrace();
		}            

    }
    
    /**
     * RDC-EXACT for two RDCs (CH and NH RDCs), and for long helix
     * Co-Ca and Co-N RDCs are optional.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doRdcExactHelix(String src, String strOut, String strInput)throws JampackException
    {	 
    	int maxStructurePerEnsemble=2000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	ModelRdc mdc=new ModelRdc();
    	
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// Read the input files    	 	
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double Syy=0.0,Szz=0.0, wtCoCA=0.0, wtCoN=0.0;
    	String strResBounds="",strOutPdb="",strChFile="",strNhFile="",
    		strCaCoFile="",strCoNFile="",strTalos="",strPrePdb="",strSeq="";   	
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    	
    		if(paraMap.containsKey("CHFILE"))
    			strChFile  =  (String)paraMap.get("CHFILE"); 
    		if(paraMap.containsKey("TALOS"))
    			strTalos  =  (String)paraMap.get("TALOS"); 
    		if(paraMap.containsKey("NHFILE"))
    			strNhFile  =  (String)paraMap.get("NHFILE"); 
    		if(paraMap.containsKey("COCAFILE"))
    			strCaCoFile  =  (String)paraMap.get("COCAFILE"); 
    		if(paraMap.containsKey("CONFILE"))
    			strCoNFile  =  (String)paraMap.get("CONFILE");     		
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("SYY"))
    			Syy  =  Double.parseDouble((String)paraMap.get("SYY"));
    		if(paraMap.containsKey("SZZ"))
    			Szz  =  Double.parseDouble((String)paraMap.get("SZZ"));   
    		if(paraMap.containsKey("WTCOCA"))
    			wtCoCA=  Double.parseDouble((String)paraMap.get("WTCOCA"));   
    		if(paraMap.containsKey("WTCON"))
    			wtCoN=  Double.parseDouble((String)paraMap.get("WTCON")); 
    		
    		if(paraMap.containsKey("RESBOUNDS"))
    			strResBounds  =  (String)paraMap.get("RESBOUNDS");      		
    		  		
    		if(paraMap.containsKey("OUTPDBNAME"))
    			strOutPdb  =  (String)paraMap.get("OUTPDBNAME");  
    		
    		if(paraMap.containsKey("PREPDBNAME"))
    			strPrePdb  =  (String)paraMap.get("PREPDBNAME");    		
    		
    	} //for (i=0;i<paraVec.size();i++)
    	
    	//read the sequence
    	String seqFile = src + strSeq;    	
    	Vector vecSeq=asg.ReaderSeq(seqFile);   
    	
    	//read the talos angle table:
    	String strTalosFile=src+strTalos;
    	PhiPsi phipsi0 =new PhiPsi();
    	Vector vecTalos=new Vector();
    	boolean existTalos = (new File(strTalosFile)).exists();
    	if(existTalos)
    		vecTalos=phipsi0.talosAngleReader(strTalosFile);
    	else
    		System.out.println("Warning: The TALOS input is not available...");
    	//read the RDC data:
    	String strChRdcFile=src+strChFile;
    	String strNhRdcFile=src+strNhFile;
    	String strCaCoRdcFile=src+strCaCoFile;
    	String strCoNRdcFile=src+strCoNFile;
    	Dipolar dd  =  new Dipolar();
		Vector<Dipolar> nhRdc = dd.rdcRead(strNhRdcFile,1.0);
		
		Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,1.0);		
		
		boolean existCaCo = (new File(strCaCoRdcFile)).exists();
		boolean existCoN = (new File(strCoNRdcFile)).exists();
		
		Vector<Dipolar> cacoRdc = new Vector();
		Vector<Dipolar> conRdc = new Vector();
		if(existCaCo)
			cacoRdc=dd.rdcRead(strCaCoRdcFile,0.19845);
		else
			System.out.println("Warning: The Co-Ca RDC doesn't exist...");
		if(existCoN) 			
			conRdc=dd.rdcRead(strCoNRdcFile,0.12086);	    	
		else
			System.out.println("Warning: The Co-N RDC doesn't exist...");
		
		//read the previous structure 
		String strPdb=src+strPrePdb;
    	Vector vecPrePdb=new Vector();
    	boolean existPreFile = (new File(strPdb)).exists();
    	if(existPreFile)
    	{
    		Vector ppVec=pp.readPdb(strPdb);
    		vecPrePdb=pp.nameConvert(ppVec);
    	}   	
    	else
    		System.out.println("Warning: no previous structure was found...");
    	
    	//read the start and end residues NOs:
    	Vector vecResBounds=asg.getOrder(strResBounds);   
    	String strNo1=(String) vecResBounds.elementAt(0);
    	String strNo2=(String) vecResBounds.elementAt(1);
		int StartResNo= Integer.parseInt(strNo1);
		int EndResNo= Integer.parseInt(strNo2);
		
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
	   	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	   	
	   	double [] ramaFilter = {Const.phiLowHelix,Const.phiHighHelix,Const.psiLowHelix,Const.psiHighHelix};
	   	
	   	///////////
		//parameters
		double resolution=2.0;
		int refineCycle=4;
		int dfsCycle=931072; 
		boolean debugDFS=true;
		boolean printResults=true;
		double w4Angles=0.1;//50;//0.1;
	    ///////////
	   //build the initial structure using ideal geometry
	   	double phiAve = Const.phiAveHelix;
		double psiAve = Const.psiAveHelix;
		Vector phiPsiVec=new Vector();
		for (i=StartResNo; i<=EndResNo; i++)  
		    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
	    Vector pdbVec = mdc.modelBuild(phiPsiVec, n1, nh1, ca1);    	
	    // for the first time, we need to fit the structure into the rdc using grid search:
	       
	    int nRefineCycles=4;
	    Vector<Pdb> helixPdbVec22 =new Vector();
	 	for(int t=0;t<=nRefineCycles;t++)
	 	{
	    
	 		System.out.println("===============helix"+t+"=====================");
	 		refineCycle=2;
		    helixPdbVec22 = new Vector();
		    helixPdbVec22= mdc.refineHelixW4RDCs(vecPrePdb,pdbVec,nhRdc, cahaRdc,cacoRdc, conRdc, Syy,  Szz, ramaFilter, 
				     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos,wtCoCA,wtCoN,StartResNo,EndResNo,vecSeq);
		    
		    Pdb temp2 = new Pdb(); ///added by Zeng
		    temp2.print(helixPdbVec22);///added by Zeng for debugging
		    pdbVec=new Vector();
		    pdbVec.addAll(helixPdbVec22);
	 	}
	 	
    	//write the new pdb file:
    	String fileName =strOut+strOutPdb;
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		//out.println("REMARK :  input PDB ID=:" + strID);///to file
    		//out.println("REMARK vdw_energy: " + scEnergy);///to file	        		   			
	    	pp.printToFile(pdbVec,fileName, out);			    
	    	//out.println("TER");
	    	out.println("END");  
	    	out.close();
	    	
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
			System.out.println("IOException: the stack trace is:");
			e.printStackTrace();
		}            


		long endTime = System.currentTimeMillis() ;
	 	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
	 	System.out.println("Time= "+ totalTime +" minutes");
    }
    
    /**
     * RDC-EXACT without alignment tensor for two RDCs (CH and NH RDCs), and for long helix.
     * Co-Ca and Co-N RDCs are optional.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doRdcExactHelixWOAT(String src, String strOut, String strInput)throws JampackException
    {	 
    	int maxStructurePerEnsemble=2000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	ModelRdc mdc=new ModelRdc();
    	
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	// Read the input files
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double Syy=0.0,Szz=0.0, wtCoCA=0.0, wtCoN=0.0,nRefineCycles=0;;
    	String strResBounds="",strOutPdb="",strChFile="",strNhFile="",strIsNonAT="",
    		strCaCoFile="",strCoNFile="",strTalos="",strPrePdb="",strSeq="";   	
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    	
    		if(paraMap.containsKey("CHFILE"))
    			strChFile  =  (String)paraMap.get("CHFILE"); 
    		if(paraMap.containsKey("TALOS"))
    			strTalos  =  (String)paraMap.get("TALOS"); 
    		
    		if(paraMap.containsKey("ISNONAT"))
    			strIsNonAT  =  (String)paraMap.get("ISNONAT"); 
    		
    		if(paraMap.containsKey("NHFILE"))
    			strNhFile  =  (String)paraMap.get("NHFILE"); 
    		if(paraMap.containsKey("COCAFILE"))
    			strCaCoFile  =  (String)paraMap.get("COCAFILE"); 
    		if(paraMap.containsKey("CONFILE"))
    			strCoNFile  =  (String)paraMap.get("CONFILE");     		
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("SYY"))
    			Syy  =  Double.parseDouble((String)paraMap.get("SYY"));
    		if(paraMap.containsKey("SZZ"))
    			Szz  =  Double.parseDouble((String)paraMap.get("SZZ"));   
    		if(paraMap.containsKey("WTCOCA"))
    			wtCoCA=  Double.parseDouble((String)paraMap.get("WTCOCA"));   
    		if(paraMap.containsKey("WTCON"))
    			wtCoN=  Double.parseDouble((String)paraMap.get("WTCON")); 
    		if(paraMap.containsKey("REFINECYCLES"))
    			nRefineCycles=  Double.parseDouble((String)paraMap.get("REFINECYCLES")); 
    		
    		
    		
    		if(paraMap.containsKey("RESBOUNDS"))
    			strResBounds  =  (String)paraMap.get("RESBOUNDS");      		
    		  		
    		if(paraMap.containsKey("OUTPDBNAME"))
    			strOutPdb  =  (String)paraMap.get("OUTPDBNAME");  
    		
    		if(paraMap.containsKey("PREPDBNAME"))
    			strPrePdb  =  (String)paraMap.get("PREPDBNAME");    		
    		
    	} //for (i=0;i<paraVec.size();i++)
    	
    	//read the sequence
    	String seqFile = src + strSeq;    	
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	
    	//read the talos angle table:
    	String strTalosFile=src+strTalos;
    	PhiPsi phipsi0 =new PhiPsi();
    	Vector vecTalos=new Vector();
    	boolean existTalos = (new File(strTalosFile)).exists();
    	if(existTalos)
    		vecTalos=phipsi0.talosAngleReader(strTalosFile);
    	else
    		System.out.println("Warning: The TALOS input is not available...");
    	
    	//read the RDC data:
    	String strChRdcFile=src+strChFile;
    	String strNhRdcFile=src+strNhFile;
    	String strCaCoRdcFile=src+strCaCoFile;
    	String strCoNRdcFile=src+strCoNFile;
    	Dipolar dd  =  new Dipolar();
		Vector<Dipolar> nhRdc = dd.rdcRead(strNhRdcFile,1.0);
		///Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,2.008);
		Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,1.0);		
		
		boolean existCaCo = (new File(strCaCoRdcFile)).exists();
		boolean existCoN = (new File(strCoNRdcFile)).exists();
		
		Vector<Dipolar> cacoRdc = new Vector();
		Vector<Dipolar> conRdc = new Vector();
		if(existCaCo)
			cacoRdc=dd.rdcRead(strCaCoRdcFile,0.19845);
		else
			System.out.println("Warning: The Co-Ca RDC doesn't exist...");
		if(existCoN) 			
			dd.rdcRead(strCoNRdcFile,0.12086);	    	
		else
			System.out.println("Warning: The Co-N RDC doesn't exist...");
		
		//read the previous structure 
		String strPdb=src+strPrePdb;
    	Vector vecPrePdb=new Vector();
    	boolean existPreFile = (new File(strPdb)).exists();
    	if(existPreFile)
    	{
    		Vector ppVec=pp.readPdb(strPdb);
    		vecPrePdb=pp.nameConvert(ppVec);
    	}   	
    	else
    		System.out.println("Warning: no previous structure was found...");
    	
    	//read the start and end residues NOs:
    	Vector vecResBounds=asg.getOrder(strResBounds);   
    	String strNo1=(String) vecResBounds.elementAt(0);
    	String strNo2=(String) vecResBounds.elementAt(1);
		int StartResNo= Integer.parseInt(strNo1);
		int EndResNo= Integer.parseInt(strNo2);
			
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
	   	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	   	
	   	double [] ramaFilter = {Const.phiLowHelix,Const.phiHighHelix,Const.psiLowHelix,Const.psiHighHelix};
	   	
	   	///////////
		//parameters
		double resolution=2.0;
		int refineCycle=4;
		int dfsCycle=631072; 
		boolean debugDFS=true;
		boolean printResults=true;
		double w4Angles=11;//50;//0.1;
	    ///////////
	   //build the initial structure using ideal geometry
	   	double phiAve = Const.phiAveHelix;
		double psiAve = Const.psiAveHelix;
		Vector phiPsiVec=new Vector();
		for (i=StartResNo; i<=EndResNo; i++)  
		    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
	    Vector pdbVec = mdc.modelBuild(phiPsiVec, n1, nh1, ca1);    	
	    // for the first time, we need to fit the structure into the rdc using grid search:
	       
	    ////////////////////////////////////////////////////////////////
	    if (strIsNonAT.equalsIgnoreCase("1"))
	    {
		    /////////////////////////compute the helix without alignment tensor
		    double[] saupe = new double[5]; //for returning Saupe elements
		    System.out.println("===============helix0=====================");
			 Vector<Pdb> helixPdbVec = mdc.refineSaupe4RDCsWOAT(vecPrePdb,pdbVec,nhRdc, cahaRdc,cacoRdc, conRdc, saupe, ramaFilter, 
				    phiAve, psiAve, refineCycle, dfsCycle, debugDFS, printResults,vecTalos,wtCoCA,wtCoN,StartResNo,EndResNo,vecSeq);
		        	    
		    Pdb temp1 = new Pdb(); 
			temp1.print(helixPdbVec);
			
			Syy = saupe[0];
			Szz = saupe[1];
		 	double rdc1Rmsd = saupe[2];
		 	double rdc2Rmsd = saupe[3];
		 	System.out.println("First  "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
		 	//	 	added by zeng
			System.out.println("-----------------------------------");
			System.out.println("The alignment tensor for helix:");
			System.out.println("Sxx, Syy, Szz=    " +(-Syy-Szz)+"    "   +Syy+"  "+Szz);
			System.out.println("rdc1Rmsd (CH), rdc2Rmsd (NH)=    " +" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
			
		 	System.out.println("Second   (Licong's original version) "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
		 	
		 	System.out.println("Another format of alingment tenosr:  "); //added by zeng
		 	System.out.println("A_a, A_r =   "+ (Szz/2 ) + "   " + (((-Syy-Szz)-Syy)/3));   
		 	System.out.println("-----------------------------------");
		 	System.out.println("====================================");
		 	
		 	
		 	pdbVec=new Vector ();
		 	pdbVec.addAll(helixPdbVec);
		 	///////////////////////////////////////////////////////
	    }
	 	
	    Vector<Pdb> helixPdbVec22 =new Vector();
	 	for(int t=0;t<=nRefineCycles;t++)
	 	{
	    
	 		System.out.println("===============helix"+t+"=====================");
	 		refineCycle=2;
		    helixPdbVec22 = new Vector();
		    helixPdbVec22= mdc.refineHelixW4RDCs(vecPrePdb,pdbVec,nhRdc, cahaRdc,cacoRdc, conRdc, Syy,  Szz, ramaFilter, 
				     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos,wtCoCA,wtCoN,StartResNo,EndResNo,vecSeq);
		    
		    Pdb temp2 = new Pdb(); ///added by Zeng
		    temp2.print(helixPdbVec22);///added by Zeng for debugging
		    pdbVec=new Vector();
		    pdbVec.addAll(helixPdbVec22);
	 	}
	 	
	 	 //write the new pdb file:
    	String fileName =strOut+strOutPdb;
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		//out.println("REMARK :  input PDB ID=:" + strID);///to file
    		//out.println("REMARK vdw_energy: " + scEnergy);///to file	        		   			
	    	pp.printToFile(pdbVec,fileName, out);			    
	    	//out.println("TER");
	    	out.println("END");  
	    	out.close();
	    	
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
			System.out.println("IOException: the stack trace is:");
			e.printStackTrace();
		}            


		long endTime = System.currentTimeMillis() ;
	 	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
	 	System.out.println("Time= "+ totalTime +" minutes");
    }
    
	
	/**
	 * Prints an error message because too few parameters were in line.
	 * 
	 * @param s the s
	 */
	// Prints an error message because too few parameters were in line
	private void errorTooFewParams(String s) 
	{
	
		System.out.println("ERROR: Too few parameters for " + s);
	}
	
	//	Displays the program version and citations
	/**
	 * Output prog info.
	 */
	public void outputProgInfo() {
		
		//System.out.println();		
		System.out.println("*****************************************************************************************");		
		System.out.println("**         ,rMr.,------.,rMr.                RDC-PANDA Version 1.0                     **");		
	    System.out.println("**         (GNP'        `?ND)                Contact Info:                             **");
	    System.out.println("**           P  xMx.  ,yMb  ?                    Bruce Donald                          **");		
  	    System.out.println("**           (  ?X_O  O_XP  )                    Duke University                       **");
  	    System.out.println("**           (      qp      )   RDCs             Department of Computer Science        **");	
   	    System.out.println("**            1  `--'`--'  7                     Levine Science Research Center (LSRC) **");        
  	    System.out.println("**         _____         ,__)                    Durham                                **");
   	    System.out.println("**        (--|__) _.._  _| _,                    NC 27708-0129, USA                    **");	
 	    System.out.println("**          _|   (_|| |(_|(_|                    brd@cs.duke.edu                       **");
 	    System.out.println("**         (                                                                           **");	
 	    System.out.println("**                                                                                     **");    
 	    System.out.println("**          J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. **");  
 	    System.out.println("** 	    \"High-Resolution Protein Structure Determination Starting with a Global    **");
 	    System.out.println("**          Fold Calculated from Exact Solutions to the RDC Equations.\" J. Biom. NMR,  **");       
 	    System.out.println("**          45(3):265-281, 2009.                                                       **");
 		System.out.println("*****************************************************************************************");
		
 	    //old output information:
 	   /* 
 	    System.out.println("*****************************************************************************************");
		System.out.println("**         ,rMr.,------.,rMr.                   RDC-PANDA Version 0.1                  **");
	    System.out.println("**         (GNP'        `?ND)                   By Jianyang (Michael) Zeng             **");
	    System.out.println("**           P  xMx.  ,yMb  ?                   Email:zengjy@cs.duke.edu               **");		
 	    System.out.println("**           (  ?X_O  O_XP  )                                                          **");
 	    System.out.println("**           (      qp      )   RDCs            Bruce Donald's lab                     **"	);	
  	    System.out.println("**            1  `--'`--'  7                    Department of Computer Science, and    **");        
 	    System.out.println("**         _____         ,__)                   Department of Biochemistry,            **");
  	    System.out.println("**        (--|__) _.._  _| _,                   Duke University                        **"	);	
	    System.out.println("**          _|   (_|| |(_|(_|                                                          **");
	    System.out.println("**         (                                                                           **"	);	
	    System.out.println("*****************************************************************************************");
			*/
 	    
		/*		
				
		System.out.println("**        RDC-PANDA Version 1.0                    **");
		System.out.println("**        Department of Computer Science, and       **");
		System.out.println("**        Department of Biochemistry,               **");
		System.out.println("**        Duke University                           **");
		//System.out.println("**                                                  **");
		System.out.println("******************************************************");
		//System.out.println("");*/
	}
	
	 // Deletes all files and subdirectories under dir.
    // Returns true if all deletions were successful.
    // If a deletion fails, the method stops attempting to delete and returns false.
     /**
 	 * Delete a directory.
 	 * 
 	 * @param dir the dir
 	 * 
 	 * @return true, if successful
 	 */
 	public static boolean deleteDir(File dir) {
        if (dir.isDirectory()) {
            String[] children = dir.list();
            for (int i=0; i<children.length; i++) {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success) {
                    return false;
                }
            }
        }
    
        // The directory is now empty so delete it
        return dir.delete();
    }

}