package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Residue.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 *  
    A very simple Residue class with the Notation like Val1, Ala25.
    Written by Lincong Wang (2001-2005).
*/
public class Residue{
    
    /** The residue no. */
    private int residueNo = 0;
    
    /** The residue. */
    private String residue = null;
    
    /** The peak vec. */
    private Vector peakVec = new Vector(); //all the resonance assignment
    
    /** The noe map. */
    private Map noeMap = new TreeMap();  //all the noe vectors, each one is a Map structue
    
    /** The hcch vec. */
    private Vector hcchVec = new Vector(); //all the resonance assignment
//     private Map noeMap = new TreeMap();  //all the noe vectors, each one is a Map structue
    /** The Constant GLY. */
static final double GLY = 63.8; //The numbers were taken from 
    //A table from the paper "JACS 1999, 121, 4031-4039"
    /** The Constant ALA. */
    static final double ALA = 90.1;
    
    /** The Constant VAL. */
    static final double VAL = 139.1;
    
    /** The Constant ILE. */
    static final  double ILE = 164.9;
    
    /** The Constant LEU. */
    static final  double LEU = 164.6;
    
    /** The Constant MET. */
    static final  double MET = 167.7;
    
    /** The Constant PHE. */
    static final  double PHE = 193.5;
    
    /** The Constant TYR. */
    static final  double TYR = 197.1;
    
    /** The Constant TRP. */
    static final  double TRP = 231.7;
    
    /** The Constant SER. */
    static final  double SER = 94.2;
    
    /** The Constant THE. */
    static final  double THE = 120.0;
    
    /** The Constant ASN. */
    static final  double ASN = 127.5;
    
    /** The Constant ASP. */
    static final  double ASP = 117.1;
    
    /** The Constant GLN. */
    static final  double GLN = 149.4;
    
    /** The Constant GLU. */
    static final  double GLU = 140.8;
    
    /** The Constant PRO. */
    static final  double PRO = 123.1;
    
    /** The Constant LYS. */
    static final  double LYS = 170.0;
    //The following numbers for the three residues were estimated by me since there were no data 
    //for them in the table of the JACS paper.
    /** The Constant ARG. */
    static final  double ARG = 186.4;  
    //149.4 + 22 + 15, 22 is for an additional CH2 group and 15 is for NH group.
    /** The Constant HIS. */
    static final  double HIS = 190.0;  //197.1 - 7.1 for 
    
    /** The Constant CYS. */
    static final  double CYS = 96.2;   //94.2 + 2 for the voluem of S compared with SER
    
    /**
     * Instantiates a new residue.
     */
    public  Residue() {
	residue = null;
	residueNo = 0;
	peakVec = null;
	hcchVec = null;
	noeMap = new  TreeMap();
    }
    
    /**
     * A constructor designed specifically for the Comparator class used for the Sort and BinarySearch.
     * 
     * @param no the no
     */
    public Residue(int no) {
	residue = null;
	residueNo = no;
	peakVec = new Vector();
	hcchVec = new Vector();
	noeMap = new  TreeMap(); 
    }
    
    /**
     * Instantiates a new residue.
     * 
     * @param no the no
     * @param type the type
     */
    public  Residue(int no, String type) {
	residue = type;
	residueNo = no;
	peakVec = new Vector();
	hcchVec = new Vector();
	noeMap =  new TreeMap();
    }

    /**
     * Instantiates a new residue.
     * 
     * @param no the no
     * @param type the type
     * @param hcVec the hc vec
     */
    public  Residue (int no,  String type, Vector hcVec) {
	residueNo = no;
	residue = type;
	peakVec = new Vector();
	hcchVec = hcVec;
	noeMap =  new TreeMap();
    }

    /**
     * Instantiates a new residue.
     * 
     * @param no the no
     * @param type the type
     * @param pkVec the pk vec
     * @param hcVec the hc vec
     */
    public  Residue (int no,  String type, Vector pkVec, Vector hcVec) {
	residueNo = no;
	residue = type;
	peakVec = pkVec;
	hcchVec = hcVec;
	noeMap =  new TreeMap();
    }

    /**
     * Instantiates a new residue.
     * 
     * @param no the no
     * @param type the type
     * @param pkVec the pk vec
     * @param nVec the n vec
     */
    public  Residue (int no, String type, Vector pkVec, Map nVec) {
	residueNo = no;
	residue = type;
	peakVec = pkVec;
	noeMap = nVec;
    }

    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo()	{
	return residueNo;
    }
    
    /**
     * Gets the residue type.
     * 
     * @return the residue type
     */
    public String getResidueType() {
	return residue;
    }
    
    /**
     * Gets the peak vec.
     * 
     * @return the peak vec
     */
    public Vector getPeakVec() {
	return peakVec;
    }
    
    /**
     * Gets the hcch vec.
     * 
     * @return the hcch vec
     */
    public Vector getHcchVec() {
	return hcchVec;
    }

    /**
     * Gets the noe map.
     * 
     * @return the noe map
     */
    public Map getNoeMap() {
	return noeMap;
    }

 //set the values	
    /**
  * Sets the residue no.
  * 
  * @param R the new residue no
  */
 public void setResidueNo(int R){
	residueNo = R;
    }	
    
    /**
     * Sets the residue type.
     * 
     * @param R the new residue type
     */
    public void setResidueType(String R){
	residue = R;
    }	
    
    /**
     * Sets the peak vec.
     * 
     * @param vec the new peak vec
     */
    public void setPeakVec(Vector vec){
	peakVec = vec;
    }	
    
    /**
     * Sets the hcch vec.
     * 
     * @param vec the new hcch vec
     */
    public void setHcchVec(Vector vec){
	hcchVec = vec;
    }	

   /**
    * The Class caComparator.
    */
   public static class caComparator implements Comparator{
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Residue n1 = (Residue)o1;
            Residue n2 = (Residue)o2;
	    Peak pk = new Peak();
	    String nucleus ="";
	    Vector pkVec1 =  n1.getPeakVec();
	    Vector pkVec2 =  n2.getPeakVec();
	    int j = 0;
	    double d1 = 0.0, d2 = 0.0;
	    for (j=0; j<pkVec1.size(); j++){
		    pk = (Peak) pkVec1.elementAt(j);
		    nucleus = pk.getNucleus();
		    if (nucleus.equals("CA"))
			d1 = pk.getCS();
	    }
	    for (j=0; j<pkVec2.size(); j++){
		    pk = (Peak) pkVec2.elementAt(j);
		    nucleus = pk.getNucleus();
		    if (nucleus.equals("CA"))
			d2 = pk.getCS();
	    }
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }

    /**
     * The Class cbComparator.
     */
    public static class cbComparator implements Comparator{
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Residue n1 = (Residue)o1;
            Residue n2 = (Residue)o2;
	    Peak pk = new Peak();
	    String nucleus ="";
	    Vector pkVec1 =  n1.getPeakVec();
	    Vector pkVec2 =  n2.getPeakVec();
	    int j = 0;
	    double d1=0.0, d2=0.0;
	    for (j=0; j<pkVec1.size(); j++){
		    pk = (Peak) pkVec1.elementAt(j);
		    nucleus = pk.getNucleus();
		    if (nucleus.equals("CB"))
			d1 = pk.getCS();
	    }
	    for (j=0; j<pkVec2.size(); j++){
		    pk = (Peak) pkVec2.elementAt(j);
		    nucleus = pk.getNucleus();
		    if (nucleus.equals("CB"))
			d2 = pk.getCS();
	    }
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }

//     public String toString(){
// 	String desc = "";
// 	String str = String.valueOf(residueNo) +" "+residue+"\n";
// 	double csValue = 0.0;
// 	for (int i=0; i<peakVec.size(); i++)
// 	    desc += (Peak) peakVec.elementAt(i);
// 	Vectoe noeVec = new Vector();
// 	if (!noeMap.isEmpty()){
// 	    for (Iterator i = noeMap.entrySet().iterator(); i.hasNext(); ){
// 		Map.Entry e = (Map.Entry) i.next();		
// 		noeVec = (Vector)e.getValue();
// 		csValue =  ((Double)e.getValue()).doubleValue();
// 		if( Math.abs(csValue+999.99) > 0.01)
// 		    str += (String)e.getKey()+" = "+ (Double)e.getValue()+"\n";
// 	    }
// 	    desc +=  str+"\n";
// 	    desc += String.valueOf(csHN)+"  "+String.valueOf(csN15)
// 		+"   "+String.valueOf(csH1)+"  "+String.valueOf(intensity);
// 	}else{
// 	    desc = String.valueOf(csHN) +"  "+String.valueOf(csN15)
// 		+"   "+String.valueOf(csH1)+"  "+String.valueOf(intensity);
// 	}
// 	return desc;	
//     }

    /**
 * The atom identifiers are based on the IUPAC-IUB and X-plor
 * For different nomenclature this functions should modified correspondingly
 * input: residue type
 * output: all atoms belong to one amino acid.
 * 
 * @param type the type
 * 
 * @return the map
 */
    public Map nucluesByAA(String type){
	Map mapAtoms = new TreeMap();
	double dummy = -999.99;
	if(type.equalsIgnoreCase("Gly")) {
	    mapAtoms.put("H",  new Double(dummy));
	    mapAtoms.put("N",  new Double(dummy));
	    mapAtoms.put("C",  new Double(dummy));
	    mapAtoms.put("CA",  new Double(dummy));
	    mapAtoms.put("HA",  new Double(dummy));//A dummy for uniformality
	    mapAtoms.put("HA3",  new Double(dummy));
	    mapAtoms.put("HA2",  new Double(dummy));
	    mapAtoms.put("HA#",  new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Ala")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Arg")){
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HG3", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HG#", new Double(dummy));
	    mapAtoms.put("CD", new Double(dummy));
	    mapAtoms.put("HD3", new Double(dummy));
	    mapAtoms.put("HD2", new Double(dummy));
	    mapAtoms.put("HD#", new Double(dummy));
	    mapAtoms.put("HE", new Double(dummy));
	    mapAtoms.put("HH11", new Double(dummy));
	    mapAtoms.put("HH12", new Double(dummy));
	    mapAtoms.put("HH1#", new Double(dummy));
	    mapAtoms.put("HH21", new Double(dummy));
	    mapAtoms.put("HH22", new Double(dummy));
	    mapAtoms.put("HH2#", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Asn")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("ND2", new Double(dummy));
	    mapAtoms.put("HD21", new Double(dummy));
	    mapAtoms.put("HD22", new Double(dummy));
	    mapAtoms.put("HD2#", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Asp"))	{
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Cys"))  {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("HG", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Gln")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HG3", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HG#", new Double(dummy));
	    mapAtoms.put("CD", new Double(dummy));
	    mapAtoms.put("NE2", new Double(dummy));
	    mapAtoms.put("HE21", new Double(dummy));
	    mapAtoms.put("HE22", new Double(dummy));
	    mapAtoms.put("HE2#", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Glu")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HG3", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HG#", new Double(dummy));
	    mapAtoms.put("CD", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "His")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HD1", new Double(dummy));
	    mapAtoms.put("HD2", new Double(dummy));
	    mapAtoms.put("HD#", new Double(dummy));
	    mapAtoms.put("HE1", new Double(dummy));
	    mapAtoms.put("HE2", new Double(dummy));
	    mapAtoms.put("HE#", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Ile")){
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB", new Double(dummy));
	    mapAtoms.put("CG1", new Double(dummy));
	    mapAtoms.put("HG13", new Double(dummy));
	    mapAtoms.put("HG12", new Double(dummy));
	    mapAtoms.put("HG1#", new Double(dummy));
	    mapAtoms.put("CG2", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy)); //no sterospecific assignment is possible
	    mapAtoms.put("CD1", new Double(dummy));
	    mapAtoms.put("HD1", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Leu")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HG", new Double(dummy));
	    mapAtoms.put("CD1", new Double(dummy));
	    mapAtoms.put("HD1", new Double(dummy));
	    mapAtoms.put("CD2", new Double(dummy));
	    mapAtoms.put("HD2", new Double(dummy));
	    mapAtoms.put("HD#", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Lys")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HG3", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HG#", new Double(dummy));
	    mapAtoms.put("CD", new Double(dummy));
	    mapAtoms.put("HD3", new Double(dummy));
	    mapAtoms.put("HD2", new Double(dummy));
	    mapAtoms.put("HD#", new Double(dummy));
	    mapAtoms.put("CE", new Double(dummy));
	    mapAtoms.put("HE3", new Double(dummy));
	    mapAtoms.put("HE2", new Double(dummy));
	    mapAtoms.put("HE#", new Double(dummy));
	    mapAtoms.put("NZ", new Double(dummy));
	    mapAtoms.put("HZ#", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	} else if(type.equalsIgnoreCase("Met")){
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HG3", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HG#", new Double(dummy));
	    mapAtoms.put("CE", new Double(dummy));
	    mapAtoms.put("HE", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Phe")){
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("CD1", new Double(dummy));
	    mapAtoms.put("CD2", new Double(dummy));
	    mapAtoms.put("CE1", new Double(dummy));
	    mapAtoms.put("CE2", new Double(dummy));
	    mapAtoms.put("CZ", new Double(dummy));
	    mapAtoms.put("HD1", new Double(dummy));
	    mapAtoms.put("HD2", new Double(dummy));
	    mapAtoms.put("HD#", new Double(dummy));
	    mapAtoms.put("HE1", new Double(dummy));
	    mapAtoms.put("HE2", new Double(dummy));
	    mapAtoms.put("HE#", new Double(dummy));
	    mapAtoms.put("HZ", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Ser")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("HG", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Thr"))     	{
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB", new Double(dummy));
	    mapAtoms.put("CG2", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    mapAtoms.put("HG1", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Trp")){
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    mapAtoms.put("CD1", new Double(dummy));  
	    mapAtoms.put("CD2", new Double(dummy));  
	    mapAtoms.put("CE2", new Double(dummy));  
	    mapAtoms.put("CE3", new Double(dummy));  
	    mapAtoms.put("CZ2", new Double(dummy));  
	    mapAtoms.put("CZ3", new Double(dummy)); 
	    mapAtoms.put("NE1", new Double(dummy));   
	    mapAtoms.put("HD1", new Double(dummy));  
	    mapAtoms.put("HE1", new Double(dummy));  
	    mapAtoms.put("HE3", new Double(dummy));  
	    mapAtoms.put("HZ2", new Double(dummy));  
	    mapAtoms.put("HZ3", new Double(dummy));  
	    mapAtoms.put("HH2", new Double(dummy));  
	    mapAtoms.put("HQ", new Double(dummy));  
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	} else if(type.equalsIgnoreCase( "Tyr")) {
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("CD1", new Double(dummy));
	    mapAtoms.put("CD2", new Double(dummy));
	    mapAtoms.put("CE1", new Double(dummy));
	    mapAtoms.put("CE2", new Double(dummy));
	    mapAtoms.put("CZ", new Double(dummy));
	    mapAtoms.put("HD1", new Double(dummy));
	    mapAtoms.put("HD2", new Double(dummy));
	    mapAtoms.put("HD#", new Double(dummy));
	    mapAtoms.put("HE1", new Double(dummy));
	    mapAtoms.put("HE2", new Double(dummy));
	    mapAtoms.put("HE#", new Double(dummy));
	    mapAtoms.put("HH", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Val")){//	HG# was changed to HG** for X-plor
	    mapAtoms.put("H", new Double(dummy));
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB", new Double(dummy));
	    mapAtoms.put("CG1", new Double(dummy));
	    mapAtoms.put("HG1", new Double(dummy));
	    mapAtoms.put("CG2", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Pro")) {
	    mapAtoms.put("H", new Double(dummy)); //A dummy for uniformality
	    mapAtoms.put("N", new Double(dummy));
	    mapAtoms.put("C", new Double(dummy));
	    mapAtoms.put("CA", new Double(dummy));
	    mapAtoms.put("HA", new Double(dummy));
	    mapAtoms.put("CB", new Double(dummy));
	    mapAtoms.put("HB3", new Double(dummy));
	    mapAtoms.put("HB2", new Double(dummy));
	    mapAtoms.put("HB#", new Double(dummy));
	    mapAtoms.put("CG", new Double(dummy));
	    mapAtoms.put("HG3", new Double(dummy));
	    mapAtoms.put("HG2", new Double(dummy));
	    mapAtoms.put("HG#", new Double(dummy));
	    mapAtoms.put("CD", new Double(dummy));
	    mapAtoms.put("HD3", new Double(dummy));
	    mapAtoms.put("HD2", new Double(dummy));
	    mapAtoms.put("HD#", new Double(dummy));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	}
	return mapAtoms;
    }

    /**
     * The atom identifiers are based on the IUPAC-IUB and X-plor
     * For different nomenclature this functions should modified correspondingly
     * input: residue type
     * output: all atoms belong to one amino acid.
     * 
     * @param type the type
     * 
     * @return the map
     */
    public Map fromCB(String type){
	Map mapAtoms = new TreeMap();
	double dummy = -999.99;
	if(type.equalsIgnoreCase("Ala")) {
	    mapAtoms.put("HB", new Double(Const.dCA2HA));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Arg")){
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HG3", new Double(Const.cb2HG));
	    mapAtoms.put("HG2", new Double(Const.cb2HG));
	    mapAtoms.put("HG#", new Double(Const.cb2HG));
	    mapAtoms.put("HD3", new Double(Const.cb2HD));
	    mapAtoms.put("HD2", new Double(Const.cb2HD));
	    mapAtoms.put("HE", new Double(Const.cb2HE));
	    mapAtoms.put("HD#", new Double(Const.cb2HD));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Asn")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HD21", new Double(Const.cb2HG));
	    mapAtoms.put("HD22", new Double(Const.cb2HG));
	    mapAtoms.put("HD2#", new Double(Const.cb2HG));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Asp"))	{
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Cys"))  {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Gln")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HG3", new Double(Const.cb2HG));
	    mapAtoms.put("HG2", new Double(Const.cb2HG));
	    mapAtoms.put("HG#", new Double(Const.cb2HG));
	    mapAtoms.put("HE21", new Double(Const.cb2HD));
	    mapAtoms.put("HE22", new Double(Const.cb2HD));
	    mapAtoms.put("HE2#", new Double(Const.cb2HD));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Glu")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HG3", new Double(Const.cb2HG));
	    mapAtoms.put("HG2", new Double(Const.cb2HG));
	    mapAtoms.put("HG#", new Double(Const.cb2HG));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "His")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HD1", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HD2", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HD#", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HE1", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HE2", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HE#", new Double(Const.cb2HEPhe));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Ile")){
	    mapAtoms.put("HB", new Double(Const.dCA2HA));
	    mapAtoms.put("HG13", new Double(Const.cb2HG));
	    mapAtoms.put("HG12", new Double(Const.cb2HG));
	    mapAtoms.put("HG1#", new Double(Const.cb2HG));
	    mapAtoms.put("HG2", new Double(Const.cb2HG)); //no sterospecific assignment is possible
	    mapAtoms.put("HD1", new Double(Const.cb2HD));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Leu")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HG", new Double(Const.cb2HG));
	    mapAtoms.put("HD1", new Double(Const.cb2HD));
	    mapAtoms.put("HD2", new Double(Const.cb2HD));
	    mapAtoms.put("HD#", new Double(Const.cb2HD));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Lys")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HG3", new Double(Const.cb2HG));
	    mapAtoms.put("HG2", new Double(Const.cb2HG));
	    mapAtoms.put("HG#", new Double(Const.cb2HG));
	    mapAtoms.put("HD3", new Double(Const.cb2HD));
	    mapAtoms.put("HD2", new Double(Const.cb2HD));
	    mapAtoms.put("HD#", new Double(Const.cb2HD));
	    mapAtoms.put("HE3", new Double(Const.cb2HE));
	    mapAtoms.put("HE2", new Double(Const.cb2HE));
	    mapAtoms.put("HE#", new Double(Const.cb2HE));
	    return mapAtoms;
	} else if(type.equalsIgnoreCase("Met")){
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HG3", new Double(Const.cb2HG));
	    mapAtoms.put("HG2", new Double(Const.cb2HG));
	    mapAtoms.put("HG#", new Double(Const.cb2HG));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Phe")){
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HD1", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HD2", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HD#", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HE1", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HE2", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HE#", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HZ", new Double(Const.cb2HZPhe));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Ser")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Thr"))     	{
	    mapAtoms.put("HB", new Double(Const.dCA2HA));
	    mapAtoms.put("HG2", new Double(Const.dCA2HA));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase( "Trp")){
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HQ", new Double(dummy)); //for unidentified 
	    mapAtoms.put("CQ", new Double(dummy));  
	    mapAtoms.put("CD1", new Double(dummy));  
	    mapAtoms.put("CD2", new Double(dummy));  
	    mapAtoms.put("CE2", new Double(dummy));  
	    mapAtoms.put("CE3", new Double(dummy));  
	    mapAtoms.put("CZ2", new Double(dummy));  
	    mapAtoms.put("CZ3", new Double(dummy)); 
	    mapAtoms.put("NE1", new Double(dummy));   
	    mapAtoms.put("HD1", new Double(dummy));  
	    mapAtoms.put("HE1", new Double(dummy));  
	    mapAtoms.put("HE3", new Double(dummy));  
	    mapAtoms.put("HZ2", new Double(dummy));  
	    mapAtoms.put("HZ3", new Double(dummy));  
	    mapAtoms.put("HH2", new Double(dummy));  
	    mapAtoms.put("HQ", new Double(dummy));  
	    mapAtoms.put("CQ", new Double(dummy));  
	    return mapAtoms;
	} else if(type.equalsIgnoreCase( "Tyr")) {
	    mapAtoms.put("HB3", new Double(Const.dCA2HA));
	    mapAtoms.put("HB2", new Double(Const.dCA2HA));
	    mapAtoms.put("HB#", new Double(Const.dCA2HA));
	    mapAtoms.put("HD1", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HD2", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HD#", new Double(Const.cb2HDPhe));
	    mapAtoms.put("HE1", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HE2", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HE#", new Double(Const.cb2HEPhe));
	    mapAtoms.put("HZ", new Double(Const.cb2HZPhe));
	    return mapAtoms;
	}else if(type.equalsIgnoreCase("Val")){//	HG# was changed to HG** for X-plor
	    mapAtoms.put("HB", new Double(Const.dCA2HA));
	    mapAtoms.put("HG1", new Double(Const.cb2HG));
	    mapAtoms.put("HG2", new Double(Const.cb2HG));
	    return mapAtoms;
	}
	return mapAtoms;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
	double csValue = 0.0;
	String desc = String.valueOf(residueNo) +" "+ residue+"\n";
	String str = "";
	for (int i = 0; i < peakVec.size(); i++){
	    Peak e = (Peak)peakVec.elementAt(i);
	    str += e.getNucleus()+" = "+ e.getCS()+"\n";
	}
	desc +=  str;
	return desc;	
    }
}
