package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	H1CS.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */



import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 *  
*   This class provides data structures that involve chemical shifts.
*   Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
*/
public class H1CS
{
    
    /** The residue no. */
    private int    residueNo; 
    
    /** The atom name. */
    private String atomName;
    
    /** The residue name. */
    private String residue;
    
    /** The cs h1. */
    private double csH1; 
    
    //for reading BMRB outliers
    /** The cs lower. */
    private double csLower=-99.9;
    
    /** The cs upper. */
    private double csUpper=-99.9;
    
    //for reading BMRB statistical information:
    /** The cs average. */
    private double csAverage=-99.9;
    
    /** The cs sigma. */
    private double csSigma=-99.9;
    
    /** The proton id. */
    public int protonID=-1;//for graph matching algorithm
    
    /** The reson id. */
    public int resonID=-1;
    
    /**
     * Instantiates a new h1 cs.
     */
    public H1CS(){
	residueNo = 0;
	residue = null;
	atomName=null;
	csH1 = 0.0; 
    }		
    
    /**
     * Instantiates a new h1 cs.
     * 
     * @param no the no
     */
    public H1CS(int no)
    {
    	residueNo = no;
    	residue = null;
    	atomName=null;
    	csH1 = 0.0; 
    }		
    
    /**
     * Instantiates a new h1 cs.
     * 
     * @param no the no
     * @param resid the resid
     * @param atomid the atomid
     */
    public H1CS(int no,String resid, String atomid)
    {
    	residueNo = no;
    	residue = resid;
    	atomName=atomid;
    	csH1 = 0.0; 
    }

    /**
     * Instantiates a new h1 cs.
     * 
     * @param no the no
     * @param resid the resid
     * @param atomid the atomid
     * @param cs the cs
     * @param proton_id the proton_id
     * @param reson_id the reson_id
     */
    public H1CS(int no,String resid, String atomid, double cs,int proton_id, int reson_id)
    {
    	residueNo = no;
    	residue = resid;
    	atomName=atomid;
    	csH1 = cs; 
    	protonID=proton_id;
    	resonID=reson_id;
    }	
    
    /**
     * Instantiates a new h1 cs.
     * 
     * @param no the no
     * @param resid the resid
     * @param atomid the atomid
     * @param cs the cs
     */
    public H1CS(int no,String resid, String atomid, double cs)
    {
    	residueNo = no;
    	residue = resid;
    	atomName=atomid;
    	csH1 = cs; 
    }	
    
    /**
     * Instantiates a new h1 cs.
     * 
     * @param no the no
     * @param resid the resid
     * @param atomid the atomid
     * @param csL the cs l
     * @param csU the cs u
     */
    public H1CS(int no,String resid, String atomid, double csL, double csU)
    {
    	residueNo = no;
    	residue = resid;
    	atomName=atomid;
    	csLower = csL; 
    	csUpper=csU;
    }	
    
    /**
     * Instantiates a new h1 cs.
     * 
     * @param no the no
     * @param resid the resid
     * @param atomid the atomid
     * @param csL the cs l
     * @param csU the cs u
     * @param cs_ave the cs_ave
     * @param cs_sigma the cs_sigma
     */
    public H1CS(int no,String resid, String atomid, double csL, double csU,double cs_ave,double cs_sigma)
    {
    	residueNo = no;
    	residue = resid;
    	atomName=atomid;
    	csLower = csL; 
    	csUpper=csU;
    	csAverage=cs_ave;
    	csSigma=cs_sigma;
    }	
  //getting the values	
    /**
   * Gets the residue no.
   * 
   * @return the residue no
   */
  public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the residue type.
     * 
     * @return the residue type
     */
    public String getResidueType(){
	return residue;
    }	
    
    /**
     * Gets the atom name.
     * 
     * @return the atom name
     */
    public String getAtomName(){
    	return atomName;
        }	
    
    /**
     * Gets the h1 cs.
     * 
     * @return the h1 cs
     */
    public double getH1CS(){
	return csH1;
    }	
    
    /**
     * Gets the cS lower.
     * 
     * @return the cS lower
     */
    public double getCSLower(){
    	return csLower;
        }	
    
    /**
     * Gets the cS average.
     * 
     * @return the cS average
     */
    public double getCSAverage(){
    	return csAverage;
        }	
    
    /**
     * Gets the cS sigma.
     * 
     * @return the cS sigma
     */
    public double getCSSigma(){
    	return csSigma;
        }	
    
    /**
     * Gets the cS upper.
     * 
     * @return the cS upper
     */
    public double getCSUpper(){
    	return csUpper;
        }	
    //set the values	
    /**
     * Sets the residue no.
     * 
     * @param R the new residue no
     */
    public void setResidueNo(int R){
	residueNo = R;
    }	
    
    /**
     * Sets the residue type.
     * 
     * @param R the new residue type
     */
    public void setResidueType(String R){
	residue = R;
    }	
    
    /**
     * Sets the atom name.
     * 
     * @param R the new atom name
     */
    public void setAtomName(String R){
    	atomName = R;
        }	

    /**
     * Sets the cS.
     * 
     * @param cs the new cS
     */
    public void setCS(double cs){
    	csH1 = cs;
        }	
   
    /**
     * The Class resNoComparator.
     */
    public static class resNoComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    H1CS n1 = (H1CS)o1;
	    H1CS n2 = (H1CS)o2;
	    double d1 = n1.getResidueNo();
	    double d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class csComparator.
     */
    public static class csComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
		H1CS n1 = (H1CS)o1;
		H1CS n2 = (H1CS)o2;
	    double d1 = n1.getH1CS();
	    double d2 = n2.getH1CS();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }
    
    /**
     * print all resonance assignments to file, in CYANA format.
     * 
     * @param vecH1CS resonance assignment list
     * @param out the out
     * 
     * @return void
     */
    public void PrintAllAssignmentToFileCyana(Vector vecH1CS,PrintWriter out)
    {
    	//System.out.println("The computed resonance assignments in CYANA format are:");
    	int preNo=-99;
    	for(int i=0;i<vecH1CS.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH1CS.elementAt(i);
    		int resNo=h1cs.getResidueNo();
    		String res=h1cs.getResidueType();
    		String atom=h1cs.getAtomName();
    		double cs_h=h1cs.getH1CS();
    		String strCS= String.format("%.3f", cs_h);
    		out.println(i+ "		"+ strCS+  "		"+ 0.0+ "		"+atom+"		"+ resNo);
    		
    		
    	}//for(int i=0;i<vecH1CS.size();i++)
    	
    }
    
    /**
     * print all resonance assignments in CYANA format.
     * 
     * @param vecH1CS resonance assignment list
     * 
     * @return void
     */
    public void PrintAllAssignmentCyana(Vector vecH1CS)
    {
    	System.out.println("The computed resonance assignments in CYANA format are:");
    	int preNo=-99;
    	for(int i=0;i<vecH1CS.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH1CS.elementAt(i);
    		int resNo=h1cs.getResidueNo();
    		String res=h1cs.getResidueType();
    		String atom=h1cs.getAtomName();
    		double cs_h=h1cs.getH1CS();
    		
    		System.out.println(i+ "		"+ cs_h+  "		"+ 0.0+ "		"+atom+"		"+ resNo);
    		
    		
    	}//for(int i=0;i<vecH1CS.size();i++)
    	
    }
    
    /**
     * print all resonance assignments to file, in BMRB format.
     * 
     * @param vecH1CS resonance assignment list
     * @param out the out
     * 
     * @return void
     */
    public void PrintAllAssignmentToFileBMRB(Vector vecH1CS,PrintWriter out)
    {
    	//out.println("!The exact assignments are:");
    	int preNo=-99;
    	int counter=1;
    	for(int i=0;i<vecH1CS.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH1CS.elementAt(i);
    		int resNo=h1cs.getResidueNo();
    		String res=h1cs.getResidueType();
    		String atom=h1cs.getAtomName();
    		double cs_h=h1cs.getH1CS();
    		String strCS= String.format("%.3f", cs_h);
    		out.println(counter+ "		"+ resNo+ "		"+res+"		"+atom+"	"+atom.substring(0,1)+"  "+ strCS+  "  "+ 0.0+ "		"+1);
    		
    		counter++;
    	}//for(int i=0;i<vecH1CS.size();i++)
    	
    }
    
    /**
     * print all resonance assignments to file, in BMRB format.
     * 
     * @param vecH1CS resonance assignment list
     * @param out the out
     * 
     * @return void
     */
    public void PrintAllAssignmentToFileBMRB(Vector vecH1CS,String fileName)
    {    	    	
    	int counter=1;
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));   
	    	for(int i=0;i<vecH1CS.size();i++)
	    	{
	    		H1CS h1cs=(H1CS)vecH1CS.elementAt(i);
	    		int resNo=h1cs.getResidueNo();
	    		String res=h1cs.getResidueType();
	    		String atom=h1cs.getAtomName();
	    		double cs_h=h1cs.getH1CS();
	    		if(cs_h<-200.0)
	    			continue;
	    		String strCS= String.format("%.3f", cs_h);
	    		out.println(counter+ "		"+ resNo+ "		"+res+"		"+atom+"	"+atom.substring(0,1)+"  "+ strCS+  "  "+ 0.0+ "		"+1);
	    		
	    		counter++;
	    	}//for(int i=0;i<vecH1CS.size();i++)
    	
	    	out.close();	    	
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
    	
    }
    
    /**
     * print all resonance assignments.
     * 
     * @param vecH1CS resonance assignment list
     * 
     * @return void
     */
    public void PrintAllAssignment(Vector vecH1CS)
    {
    	System.out.println("The exact assignments are:");
    	int preNo=-99;
    	for(int i=0;i<vecH1CS.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH1CS.elementAt(i);
    		int resNo=h1cs.getResidueNo();
    		String res=h1cs.getResidueType();
    		String atom=h1cs.getAtomName();
    		double cs_h=h1cs.getH1CS();
    		if(resNo==preNo)
    			System.out.println("        "+atom+" : "+cs_h);
    		else
    		{
    			System.out.println(resNo+"-"+res+" : ");
    			System.out.println("        "+atom+" : "+cs_h);
    		}
    		preNo=resNo;
    	}//for(int i=0;i<vecH1CS.size();i++)
    	
    }
    
    /**
     * Read resonance list
     * The required file format is:.
     * 
     * @param filename the name of the file
     * @param vecSeq backbone structure to update the residue name
     * 
     * @return a vector of resonance list
     */
    public Vector<H1CS> h1CSReader(String filename,Vector vecSeq)
    {
    	Vector inputs = new Vector();
    	double  csH1 = 0.0, csSigma=0.0, intensity = 0.0;
    	Vector csVec = new Vector();
    	StringTokenizer st = new StringTokenizer("");
    	String str = "";
    	int index1=0, index2=0;
    	String order="";
    	double value=0.0;
    	int no =0;
    	String aaType = "", ss ="";
    	String atomName="";
    	String resName="";
    	int resNo=0;
    	int csID=0;
    	Noe noe=new Noe();
    	
    	try
    	{
    		BufferedReader in = new BufferedReader(new FileReader(filename));	
    		ss = in.readLine();    		
    		stop:
    			while(true) 
    			{   			    
    				st = new StringTokenizer(ss);
    				if (st.hasMoreTokens())
    					csID =  Integer.parseInt(st.nextToken());
    				if (st.hasMoreTokens())
    				{
    					csH1  = new Double(st.nextToken()).doubleValue();
    					if (csH1>300.0 || csH1<-300.0)
    					{
    						if ((ss = in.readLine()) == null)
    							break stop;     
    						else if(ss.equalsIgnoreCase(""))
    	    				{
    	    					if ((ss = in.readLine()) == null )
    	        				{
    	        					break stop;
    	        				}
    	    				}
    						continue;
    					}
    					
    				}
    				if (st.hasMoreTokens())
    					csSigma  = new Double(st.nextToken()).doubleValue();
    				if (st.hasMoreTokens())
    					atomName = st.nextToken();
    				
    				if (st.hasMoreTokens())
    					resNo =  Integer.parseInt(st.nextToken());
    				
    				if(atomName.substring(0,1).equalsIgnoreCase("Q"))
    				{
    					atomName="H"+atomName.substring(1,atomName.length() );
    				}
    				
    				resName="";
    				for(int i=0;i<vecSeq.size();i++)
    				{
    					Assign assign=(Assign)vecSeq.elementAt(i);
    					if (resNo==assign.getResidueNo())
    						resName=assign.getResidueType();     					
    				}
    				
    				
    				if(!resName.trim().equalsIgnoreCase(""))    					
    					atomName=noe.UpdateNOEAtomName2(resName, atomName);
    				
    				inputs.add(new H1CS(resNo, resName, atomName, csH1));
    				
    				
    				if ((ss = in.readLine()) == null )
    				{
    					break stop;
    				}
    				else if(ss.equalsIgnoreCase(""))
    				{
    					if ((ss = in.readLine()) == null )
        				{
        					break stop;
        				}
    				}
    					
    					
    			}
    	}catch (FileNotFoundException e) {
    		System.out.println("File not found: " + filename);
    	}catch (IOException e) 
    	{
    		System.out.println("IOException: the stack trace is:");
    		e.printStackTrace();
    	}
    	return inputs;
    }
    /**
     * Read resonance list
     * with Q for pseudo atom
     * 
     * @param filename the name of the file
     * @param vecSeq backbone structure to update the residue name
     * 
     * @return a vector of resonance list
     */
    public Vector<H1CS> h1CSReaderWQ(String filename,Vector vecSeq)
    {
    	Vector inputs = new Vector();
    	double  csH1 = 0.0, csSigma=0.0, intensity = 0.0;
    	Vector csVec = new Vector();
    	StringTokenizer st = new StringTokenizer("");
    	String str = "";
    	int index1=0, index2=0;
    	String order="";
    	double value=0.0;
    	int no =0;
    	String aaType = "", ss ="";
    	String atomName="";
    	String resName="";
    	int resNo=0;
    	int csID=0;
    	Noe noe=new Noe();
    	
    	try
    	{
    		BufferedReader in = new BufferedReader(new FileReader(filename));	
    		ss = in.readLine();    		
    		stop:
    			while(true) 
    			{   			    
    				st = new StringTokenizer(ss);
    				if (st.hasMoreTokens())
    					csID =  Integer.parseInt(st.nextToken());
    				if (st.hasMoreTokens())
    				{
    					csH1  = new Double(st.nextToken()).doubleValue();
    					if (csH1>300.0 || csH1<-300.0)
    					{
    						if ((ss = in.readLine()) == null)
    							break stop;     
    						else if(ss.equalsIgnoreCase(""))
    	    				{
    	    					if ((ss = in.readLine()) == null )
    	        				{
    	        					break stop;
    	        				}
    	    				}
    						continue;
    					}
    					
    				}
    				if (st.hasMoreTokens())
    					csSigma  = new Double(st.nextToken()).doubleValue();
    				if (st.hasMoreTokens())
    					atomName = st.nextToken();
    				
    				if (st.hasMoreTokens())
    					resNo =  Integer.parseInt(st.nextToken());
    				
    				//if(atomName.substring(0,1).equalsIgnoreCase("Q"))
    				//{
    				//	atomName="H"+atomName.substring(1,atomName.length() );
    				//}
    				
    				resName="";
    				for(int i=0;i<vecSeq.size();i++)
    				{
    					Assign assign=(Assign)vecSeq.elementAt(i);
    					if (resNo==assign.getResidueNo())
    						resName=assign.getResidueType();     					
    				}
    				
    				atomName=noe.UpdateNOEAtomName2(resName, atomName);
    				inputs.add(new H1CS(resNo, resName, atomName, csH1));
    				
    				
    				if ((ss = in.readLine()) == null )
    				{
    					break stop;
    				}
    				else if(ss.equalsIgnoreCase(""))
    				{
    					if ((ss = in.readLine()) == null )
        				{
        					break stop;
        				}
    				}
    					
    					
    			}
    	}catch (FileNotFoundException e) {
    		System.out.println("File not found: " + filename);
    	}catch (IOException e) 
    	{
    		System.out.println("IOException: the stack trace is:");
    		e.printStackTrace();
    	}
    	return inputs;
    }
    
    /**
     * delete repeated resonance assignments.
     * 
     * @param vecAsg original assignment list
     * 
     * @return new assignment list
     */
    public Vector<H1CS> DeleteRepeatAsg(Vector<H1CS> vecAsg)
	{
    	Vector vecAsgNew=new Vector();
    	for(int i=0;i< vecAsg.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecAsg.elementAt(i);
    		int resNo=h1cs.getResidueNo();
    		String atom=h1cs.getAtomName();
    		boolean isExist=false;
    		
    		for(int j=0; j<vecAsgNew.size();j++)
    		{
    			H1CS h1csNew=(H1CS)vecAsgNew.elementAt(j);
    			int resNoNew=h1csNew.getResidueNo();
    			String atomNew=h1csNew.getAtomName();
    			if(resNo==resNoNew && atom.equalsIgnoreCase(atomNew))
    				isExist=true;
    		}//for(int j=0; j<vecAsg.size();j++)
    		if(!isExist)
    			vecAsgNew.add(h1cs);
    	}//for(int i=0;i< vecAsg.size();i++)
    	return vecAsgNew;
	}
    
	/**
	 * generate all stereospecific assignments
	 * Note: there might exist two frequencies assigning to the same proton.
	 * 
	 * @param vecSrcAsg original assignment list
	 * 
	 * @return new assignment list with additional stereospecific assignments
	 */
	public Vector GenerateStereoAsg(Vector vecSrcAsg)
	{
		for(int k=0;k<vecSrcAsg.size();k++)
		{
			H1CS h1cs=(H1CS)vecSrcAsg.elementAt(k);
			int resRefNo=h1cs.getResidueNo();
			String resRef=h1cs.getResidueType();
			String atomRef=h1cs.getAtomName();
			double csRef=h1cs.getH1CS();
			/////////////
			String subAtomTwo="";
    		if(atomRef.length()>2)
    			subAtomTwo=atomRef.substring(0,2);
    		else
    			subAtomTwo=atomRef;
    		String atomRefStereo="";
			if(subAtomTwo.equalsIgnoreCase("HB"))
    		{
    			if(resRef.equalsIgnoreCase("CYS")||resRef.equalsIgnoreCase("ASP")||resRef.equalsIgnoreCase("GLU")||
    					resRef.equalsIgnoreCase("PHE")||resRef.equalsIgnoreCase("HIS")||resRef.equalsIgnoreCase("LYS")||
    					resRef.equalsIgnoreCase("LEU")||resRef.equalsIgnoreCase("MET")||resRef.equalsIgnoreCase("ASN")||
    					resRef.equalsIgnoreCase("PRO")||resRef.equalsIgnoreCase("GLN")||resRef.equalsIgnoreCase("ARG")||
    					resRef.equalsIgnoreCase("SER")||resRef.equalsIgnoreCase("TRP")||resRef.equalsIgnoreCase("TYR"))
    			{
    				if(atomRef.equalsIgnoreCase("HB1"))
    					atomRefStereo="HB2";
    				else if(atomRef.equalsIgnoreCase("HB2"))
    					atomRefStereo="HB1";
    			}   			
    		}    		
    		if(subAtomTwo.equalsIgnoreCase("HA")&& resRef.equalsIgnoreCase("GLY") )
    		{
    			if(atomRef.equalsIgnoreCase("HA1"))
    				atomRefStereo="HA2";
				else if(atomRef.equalsIgnoreCase("HA2"))
					atomRefStereo="HA1";
    		}
    		if((subAtomTwo.equalsIgnoreCase("HG")||subAtomTwo.equalsIgnoreCase("CG"))  && 
    				resRef.equalsIgnoreCase("VAL")  )//changed...
    		{
    			if(atomRef.equalsIgnoreCase("HG1"))
    				atomRefStereo="HG2";
				else if(atomRef.equalsIgnoreCase("HG2"))
					atomRefStereo="HG1";
				else if(atomRef.equalsIgnoreCase("CG1"))
					atomRefStereo="CG2";
				else if(atomRef.equalsIgnoreCase("CG2"))
					atomRefStereo="CG1";
    		}
    		if((subAtomTwo.equalsIgnoreCase("HD")||subAtomTwo.equalsIgnoreCase("CD"))&& resRef.equalsIgnoreCase("LEU") )
    		{
    			if(atomRef.equalsIgnoreCase("HD1"))
    				atomRefStereo="HD2";
				else if(atomRef.equalsIgnoreCase("HD2"))
					atomRefStereo="HD1";
				else if(atomRef.equalsIgnoreCase("CD1"))
					atomRefStereo="CD2";
				else if(atomRef.equalsIgnoreCase("CD2"))
					atomRefStereo="CD1";
    		}
			/////////////
    		if(subAtomTwo.equalsIgnoreCase("HG")  && (resRef.equalsIgnoreCase("GLN") || resRef.equalsIgnoreCase("GLU")||
    						resRef.equalsIgnoreCase("LYS")||resRef.equalsIgnoreCase("MET") || resRef.equalsIgnoreCase("PRO")||
    						resRef.equalsIgnoreCase("ARG") ) )//changed...
    		{
    			if(atomRef.equalsIgnoreCase("HG1"))
    				atomRefStereo="HG2";
				else if(atomRef.equalsIgnoreCase("HG2"))
					atomRefStereo="HG1";				
    		}
    		if(subAtomTwo.equalsIgnoreCase("HD")  && (resRef.equalsIgnoreCase("PHE") || 
    				resRef.equalsIgnoreCase("LYS")||resRef.equalsIgnoreCase("PRO")||
    				resRef.equalsIgnoreCase("ARG") || resRef.equalsIgnoreCase("TYR")  ) )//changed...
			{
				if(atomRef.equalsIgnoreCase("HD1"))
					atomRefStereo="HD2";
				else if(atomRef.equalsIgnoreCase("HD2"))
					atomRefStereo="HD1";				
			}
    		if(subAtomTwo.equalsIgnoreCase("HE")  && (resRef.equalsIgnoreCase("PHE") || 
    				resRef.equalsIgnoreCase("LYS") || resRef.equalsIgnoreCase("TYR")  ) )//changed...
			{
				if(atomRef.equalsIgnoreCase("HE1"))
					atomRefStereo="HE2";
				else if(atomRef.equalsIgnoreCase("HE2"))
					atomRefStereo="HE1";				
			}
    		if(atomRef.length()>=3)
    		{
    			if(atomRef.substring(0,3).equalsIgnoreCase("HG1") && resRef.equalsIgnoreCase("ILE"))
				{
					if(atomRef.equalsIgnoreCase("HG11"))
						atomRefStereo="HG12";
					else if(atomRef.equalsIgnoreCase("HG12"))
						atomRefStereo="HG11";				
				}
    			if(atomRef.substring(0,3).equalsIgnoreCase("HD2") && resRef.equalsIgnoreCase("ASN"))
				{
					if(atomRef.equalsIgnoreCase("HD21"))
						atomRefStereo="HD22";
					else if(atomRef.equalsIgnoreCase("HD22"))
						atomRefStereo="HD21";				
				}
    			if(atomRef.substring(0,3).equalsIgnoreCase("HE2") && resRef.equalsIgnoreCase("GLN"))
				{
					if(atomRef.equalsIgnoreCase("HE21"))
						atomRefStereo="HE22";
					else if(atomRef.equalsIgnoreCase("HE22"))
						atomRefStereo="HE21";				
				}
    			if(atomRef.substring(0,3).equalsIgnoreCase("HH1") && resRef.equalsIgnoreCase("ARG"))
				{
					if(atomRef.equalsIgnoreCase("HH11"))
						atomRefStereo="HH12";
					else if(atomRef.equalsIgnoreCase("HH12"))
						atomRefStereo="HH11";				
				}
    			if(atomRef.substring(0,3).equalsIgnoreCase("HH2") && resRef.equalsIgnoreCase("ARG"))
				{
					if(atomRef.equalsIgnoreCase("HH21"))
						atomRefStereo="HH22";
					else if(atomRef.equalsIgnoreCase("HH22"))
						atomRefStereo="HH21";				
				}
    		}
    		if(subAtomTwo.equalsIgnoreCase("HH")  && resRef.equalsIgnoreCase("ARG")  )//changed...
			{
				if(atomRef.equalsIgnoreCase("HH1"))
					atomRefStereo="HH2";
				else if(atomRef.equalsIgnoreCase("HH2"))
					atomRefStereo="HH1";				
			}
    		
    		/////////////////
			if( !(atomRef.substring(0,1).equalsIgnoreCase("H") || atomRef.substring(0,1).equalsIgnoreCase("C")||atomRef.substring(0,1).equalsIgnoreCase("N"))   )
				continue;
		}//for(int j=0;j<vecSrcAsg.size();j++)

		return null;
		
	}
	
	/**
	 * compute the statistical information of the sidechain assignment, compared with the reference assignment list.
	 * 
	 * @param vecSrcAsg original assignment list
	 * @param vecRefAsg reference assignment list
	 * @param vecBBAsg the vec bb asg
	 */
	public void CompareSCAsgsWRefAsgsNew(Vector vecSrcAsg,Vector vecRefAsg,Vector vecBBAsg )
	{
		double csErrH=0.05;
		double csErrHeavy=0.5;
		
		Peak pk=new Peak();
		int totalScH=0;
		int missing=0;
		double wrong=0.0;
		double correct=0.0;
		double correctI=0.0;
		double correctII=0.0;
		Assign asg=new Assign();		
		double nTotalAsgs=0.0;//all proton resonance assignments, including both bb and sc
		double nTotalClassI=0.0;//total number of resonances for Class I residues (See definition in JBNMR09)
		double nTotalClassII=0.0;//...Class II residues.
		for(int k=0;k<vecSrcAsg.size();k++)
		{
			H1CS h1cs=(H1CS)vecSrcAsg.elementAt(k);
			String res=h1cs.getResidueType();
			String atom=h1cs.getAtomName();
			if(atom.substring(0, 1).equalsIgnoreCase("H"))
			{
				nTotalAsgs=nTotalAsgs+1;
				if(res.equalsIgnoreCase("VAL")||res.equalsIgnoreCase("ILE")||res.equalsIgnoreCase("LEU")||res.equalsIgnoreCase("MET")
						||res.equalsIgnoreCase("PHE")||res.equalsIgnoreCase("TRP")||res.equalsIgnoreCase("CYS") )
					nTotalClassI=nTotalClassI+1;
				else if( !(res.equalsIgnoreCase("ALA") || res.equalsIgnoreCase("GLY")) )
					nTotalClassII=nTotalClassII+1;
			}
		}
		double nTotalBBAsgs=0.0;//number of bb resonance assignments
		for(int k=0;k<vecBBAsg.size();k++)
		{
			H1CS h1cs=(H1CS)vecBBAsg.elementAt(k);
			String atom=h1cs.getAtomName();
			if(atom.substring(0, 1).equalsIgnoreCase("H"))
				nTotalBBAsgs=nTotalBBAsgs+1;
		}
		
		for(int j=0;j<vecSrcAsg.size();j++)
		{
			H1CS h1csA=(H1CS)vecSrcAsg.elementAt(j);
			int resNoA=h1csA.getResidueNo();
			String resA=h1csA.getResidueType();
			String atomA=h1csA.getAtomName();
			double cs_hA=h1csA.getH1CS();	
			if(!atomA.substring(0,1).equalsIgnoreCase("H"))
				continue;
			
			boolean isFound=false;
			boolean isCorrect=false;
			
			for(int k=0;k<vecRefAsg.size();k++)
			{
				H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
				int resRefNo=h1cs.getResidueNo();
				String resRef=h1cs.getResidueType();
				String atomRef=h1cs.getAtomName();
				double csRef=h1cs.getH1CS();
				/////////////
				String subAtomTwo="";
	    		if(atomRef.length()>2)
	    			subAtomTwo=atomRef.substring(0,2);
	    		else
	    			subAtomTwo=atomRef;
	    		String atomRefStereo="";
				if(subAtomTwo.equalsIgnoreCase("HB"))
	    		{
	    			if(resRef.equalsIgnoreCase("CYS")||resRef.equalsIgnoreCase("ASP")||resRef.equalsIgnoreCase("GLU")||
	    					resRef.equalsIgnoreCase("PHE")||resRef.equalsIgnoreCase("HIS")||resRef.equalsIgnoreCase("LYS")||
	    					resRef.equalsIgnoreCase("LEU")||resRef.equalsIgnoreCase("MET")||resRef.equalsIgnoreCase("ASN")||
	    					resRef.equalsIgnoreCase("PRO")||resRef.equalsIgnoreCase("GLN")||resRef.equalsIgnoreCase("ARG")||
	    					resRef.equalsIgnoreCase("SER")||resRef.equalsIgnoreCase("TRP")||resRef.equalsIgnoreCase("TYR"))
	    			{
	    				if(atomRef.equalsIgnoreCase("HB1"))
	    					atomRefStereo="HB2";
	    				else if(atomRef.equalsIgnoreCase("HB2"))
	    					atomRefStereo="HB1";
	    			}   			
	    		}    		
	    		if(subAtomTwo.equalsIgnoreCase("HA")&& resRef.equalsIgnoreCase("GLY") )
	    		{
	    			if(atomRef.equalsIgnoreCase("HA1"))
	    				atomRefStereo="HA2";
					else if(atomRef.equalsIgnoreCase("HA2"))
						atomRefStereo="HA1";
	    		}
	    		if((subAtomTwo.equalsIgnoreCase("HG")||subAtomTwo.equalsIgnoreCase("CG"))  && 
	    				resRef.equalsIgnoreCase("VAL")  )//changed...
	    		{
	    			if(atomRef.equalsIgnoreCase("HG1"))
	    				atomRefStereo="HG2";
					else if(atomRef.equalsIgnoreCase("HG2"))
						atomRefStereo="HG1";
					else if(atomRef.equalsIgnoreCase("CG1"))
						atomRefStereo="CG2";
					else if(atomRef.equalsIgnoreCase("CG2"))
						atomRefStereo="CG1";
	    		}
	    		if((subAtomTwo.equalsIgnoreCase("HD")||subAtomTwo.equalsIgnoreCase("CD"))&& resRef.equalsIgnoreCase("LEU") )
	    		{
	    			if(atomRef.equalsIgnoreCase("HD1"))
	    				atomRefStereo="HD2";
					else if(atomRef.equalsIgnoreCase("HD2"))
						atomRefStereo="HD1";
					else if(atomRef.equalsIgnoreCase("CD1"))
						atomRefStereo="CD2";
					else if(atomRef.equalsIgnoreCase("CD2"))
						atomRefStereo="CD1";
	    		}
				/////////////
	    		if(subAtomTwo.equalsIgnoreCase("HG")  && (resRef.equalsIgnoreCase("GLN") || resRef.equalsIgnoreCase("GLU")||
	    						resRef.equalsIgnoreCase("LYS")||resRef.equalsIgnoreCase("MET") || resRef.equalsIgnoreCase("PRO")||
	    						resRef.equalsIgnoreCase("ARG") ) )//changed...
	    		{
	    			if(atomRef.equalsIgnoreCase("HG1"))
	    				atomRefStereo="HG2";
					else if(atomRef.equalsIgnoreCase("HG2"))
						atomRefStereo="HG1";				
	    		}
	    		if(subAtomTwo.equalsIgnoreCase("HD")  && (resRef.equalsIgnoreCase("PHE") || 
	    				resRef.equalsIgnoreCase("LYS")||resRef.equalsIgnoreCase("PRO")||
	    				resRef.equalsIgnoreCase("ARG") || resRef.equalsIgnoreCase("TYR")  ) )//changed...
				{
					if(atomRef.equalsIgnoreCase("HD1"))
						atomRefStereo="HD2";
					else if(atomRef.equalsIgnoreCase("HD2"))
						atomRefStereo="HD1";				
				}
	    		if(subAtomTwo.equalsIgnoreCase("HE")  && (resRef.equalsIgnoreCase("PHE") || 
	    				resRef.equalsIgnoreCase("LYS") || resRef.equalsIgnoreCase("TYR")  ) )//changed...
				{
					if(atomRef.equalsIgnoreCase("HE1"))
						atomRefStereo="HE2";
					else if(atomRef.equalsIgnoreCase("HE2"))
						atomRefStereo="HE1";				
				}
	    		if(atomRef.length()>=3)
	    		{
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HG1") && resRef.equalsIgnoreCase("ILE"))
					{
						if(atomRef.equalsIgnoreCase("HG11"))
							atomRefStereo="HG12";
						else if(atomRef.equalsIgnoreCase("HG12"))
							atomRefStereo="HG11";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HD2") && resRef.equalsIgnoreCase("ASN"))
					{
						if(atomRef.equalsIgnoreCase("HD21"))
							atomRefStereo="HD22";
						else if(atomRef.equalsIgnoreCase("HD22"))
							atomRefStereo="HD21";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HE2") && resRef.equalsIgnoreCase("GLN"))
					{
						if(atomRef.equalsIgnoreCase("HE21"))
							atomRefStereo="HE22";
						else if(atomRef.equalsIgnoreCase("HE22"))
							atomRefStereo="HE21";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HH1") && resRef.equalsIgnoreCase("ARG"))
					{
						if(atomRef.equalsIgnoreCase("HH11"))
							atomRefStereo="HH12";
						else if(atomRef.equalsIgnoreCase("HH12"))
							atomRefStereo="HH11";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HH2") && resRef.equalsIgnoreCase("ARG"))
					{
						if(atomRef.equalsIgnoreCase("HH21"))
							atomRefStereo="HH22";
						else if(atomRef.equalsIgnoreCase("HH22"))
							atomRefStereo="HH21";				
					}
	    		}
	    		if(subAtomTwo.equalsIgnoreCase("HH")  && resRef.equalsIgnoreCase("ARG")  )//changed...
				{
					if(atomRef.equalsIgnoreCase("HH1"))
						atomRefStereo="HH2";
					else if(atomRef.equalsIgnoreCase("HH2"))
						atomRefStereo="HH1";				
				}
	    		
	    		/////////////////
				if( !(atomRef.substring(0,1).equalsIgnoreCase("H") || atomRef.substring(0,1).equalsIgnoreCase("C")||atomRef.substring(0,1).equalsIgnoreCase("N"))   )
					continue;
				
				String subAtom=atomA;
				double csErr=0.0;
				if(atomRef.substring(0,1).equalsIgnoreCase("H"))
					csErr=csErrH;
				else
					csErr=csErrHeavy;
				
				if(atomA.length()>atomRef.length())
					subAtom=atomA.substring(0,atomRef.length());
				
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRef) )
					isFound=true;
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRefStereo) )
					isFound=true;
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRef) && Math.abs(csRef-cs_hA)<csErr)
					isCorrect=true;		
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRefStereo) && Math.abs(csRef-cs_hA)<csErr)
					isCorrect=true;	
			}//for(int k=0;k<vecRefAsg.size();k++)
			
			if(!isFound)
			{
				System.out.println("Missing resonance assignment: "+ resNoA +" "+ resA+ " - " +atomA);
				missing++;
			}
			else
			{
				if(!isCorrect)			
				{
					System.out.println("...Wrong resonance assignment: "+ resNoA +" "+ resA+ " - " +atomA);
					wrong=wrong+1;
				}
				else
				{
					correct=correct+1;
					if(resA.equalsIgnoreCase("VAL")||resA.equalsIgnoreCase("ILE")||resA.equalsIgnoreCase("LEU")||resA.equalsIgnoreCase("MET")
							||resA.equalsIgnoreCase("PHE")||resA.equalsIgnoreCase("TRP")||resA.equalsIgnoreCase("CYS") )
						correctI=correctI+1;
					else if( !(resA.equalsIgnoreCase("ALA") || resA.equalsIgnoreCase("GLY")) )
						correctII=correctII+1;
				}
			}
			
			totalScH++;
		}//for(int j=0;j<vecSrcAsg.size();j++)

		System.out.println("Total number of protons (including both bb and sc protons): "+ totalScH);
		System.out.println("Total number of missing sidechain assignments: "+ missing);
		System.out.println("Total number of wrong sidechain assignments: "+ wrong);		
		System.out.println("Total number of correct resonance assignments: "+correct);
		System.out.println("Total number of side chain proton resonance assignments: "+ (nTotalAsgs-nTotalBBAsgs));
		System.out.println("Percentage of correct proton resonance assignments (including both bb and sc resonances): "+(double)( correct/nTotalAsgs ));
		System.out.println("Percentage of correct proton resonance assignments for class I residues: "+(double)( correctI/nTotalClassI ));
		System.out.println("Percentage of correct proton resonance assignments for class II residues: "+(double)( correctII/nTotalClassII ));
	}
	
	/**
	 * compute the statistical information of the sidechain assignment, compared with the reference assignment list.
	 * 
	 * @param vecSrcAsg original assignment list
	 * @param vecRefAsg reference assignment list
	 * @param vecBBAsg the vec bb asg
	 */
	public void CompareSCAsgsWRefAsgs(Vector vecSrcAsg,Vector vecRefAsg,Vector vecBBAsg )
	{
		double csErrH=0.05;
		double csErrHeavy=0.5;
		
		Peak pk=new Peak();
		int totalScH=0;
		int missing=0;
		double wrong=0.0;
		double correct=0.0;
		double correctI=0.0;
		double correctII=0.0;
		Assign asg=new Assign();		
		double nTotalAsgs=0.0;//all proton resonance assignments, including both bb and sc
		double nTotalClassI=0.0;//total number of resonances for Class I residues (See definition in JBNMR09)
		double nTotalClassII=0.0;//...Class II residues.
		for(int k=0;k<vecSrcAsg.size();k++)
		{
			H1CS h1cs=(H1CS)vecSrcAsg.elementAt(k);
			String res=h1cs.getResidueType();
			String atom=h1cs.getAtomName();
			if(atom.substring(0, 1).equalsIgnoreCase("H"))
			{
				nTotalAsgs=nTotalAsgs+1;
				if(res.equalsIgnoreCase("VAL")||res.equalsIgnoreCase("ILE")||res.equalsIgnoreCase("LEU")||res.equalsIgnoreCase("MET")
						||res.equalsIgnoreCase("PHE")||res.equalsIgnoreCase("TRP")||res.equalsIgnoreCase("CYS") )
					nTotalClassI=nTotalClassI+1;
				else if( !(res.equalsIgnoreCase("ALA") || res.equalsIgnoreCase("GLY")) )
					nTotalClassII=nTotalClassII+1;
			}
		}
		double nTotalBBAsgs=0.0;//number of bb resonance assignments
		for(int k=0;k<vecBBAsg.size();k++)
		{
			H1CS h1cs=(H1CS)vecBBAsg.elementAt(k);
			String atom=h1cs.getAtomName();
			if(atom.substring(0, 1).equalsIgnoreCase("H"))
				nTotalBBAsgs=nTotalBBAsgs+1;
		}
		
		for(int j=0;j<vecSrcAsg.size();j++)
		{
			H1CS h1csA=(H1CS)vecSrcAsg.elementAt(j);
			int resNoA=h1csA.getResidueNo();
			String resA=h1csA.getResidueType();
			String atomA=h1csA.getAtomName();
			double cs_hA=h1csA.getH1CS();	
			if(!atomA.substring(0,1).equalsIgnoreCase("H"))
				continue;
			
			boolean isFound=false;
			boolean isCorrect=false;
			
			for(int k=0;k<vecRefAsg.size();k++)
			{
				H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
				int resRefNo=h1cs.getResidueNo();
				String resRef=h1cs.getResidueType();
				String atomRef=h1cs.getAtomName();
				double csRef=h1cs.getH1CS();
				/////////////
				String subAtomTwo="";
	    		if(atomRef.length()>2)
	    			subAtomTwo=atomRef.substring(0,2);
	    		else
	    			subAtomTwo=atomRef;
	    		String atomRefStereo="";
				if(subAtomTwo.equalsIgnoreCase("HB"))
	    		{
	    			if(resRef.equalsIgnoreCase("CYS")||resRef.equalsIgnoreCase("ASP")||resRef.equalsIgnoreCase("GLU")||
	    					resRef.equalsIgnoreCase("PHE")||resRef.equalsIgnoreCase("HIS")||resRef.equalsIgnoreCase("LYS")||
	    					resRef.equalsIgnoreCase("LEU")||resRef.equalsIgnoreCase("MET")||resRef.equalsIgnoreCase("ASN")||
	    					resRef.equalsIgnoreCase("PRO")||resRef.equalsIgnoreCase("GLN")||resRef.equalsIgnoreCase("ARG")||
	    					resRef.equalsIgnoreCase("SER")||resRef.equalsIgnoreCase("TRP")||resRef.equalsIgnoreCase("TYR"))
	    			{
	    				if(atomRef.equalsIgnoreCase("HB1"))
	    					atomRefStereo="HB2";
	    				else if(atomRef.equalsIgnoreCase("HB2"))
	    					atomRefStereo="HB1";
	    			}   			
	    		}    		
	    		if(subAtomTwo.equalsIgnoreCase("HA")&& resRef.equalsIgnoreCase("GLY") )
	    		{
	    			if(atomRef.equalsIgnoreCase("HA1"))
	    				atomRefStereo="HA2";
					else if(atomRef.equalsIgnoreCase("HA2"))
						atomRefStereo="HA1";
	    		}
	    		if((subAtomTwo.equalsIgnoreCase("HG")||subAtomTwo.equalsIgnoreCase("CG"))  && 
	    				resRef.equalsIgnoreCase("VAL")  )//changed...
	    		{
	    			if(atomRef.equalsIgnoreCase("HG1"))
	    				atomRefStereo="HG2";
					else if(atomRef.equalsIgnoreCase("HG2"))
						atomRefStereo="HG1";
					else if(atomRef.equalsIgnoreCase("CG1"))
						atomRefStereo="CG2";
					else if(atomRef.equalsIgnoreCase("CG2"))
						atomRefStereo="CG1";
	    		}
	    		if((subAtomTwo.equalsIgnoreCase("HD")||subAtomTwo.equalsIgnoreCase("CD"))&& resRef.equalsIgnoreCase("LEU") )
	    		{
	    			if(atomRef.equalsIgnoreCase("HD1"))
	    				atomRefStereo="HD2";
					else if(atomRef.equalsIgnoreCase("HD2"))
						atomRefStereo="HD1";
					else if(atomRef.equalsIgnoreCase("CD1"))
						atomRefStereo="CD2";
					else if(atomRef.equalsIgnoreCase("CD2"))
						atomRefStereo="CD1";
	    		}
				/////////////
	    		if(subAtomTwo.equalsIgnoreCase("HG")  && (resRef.equalsIgnoreCase("GLN") || resRef.equalsIgnoreCase("GLU")||
	    						resRef.equalsIgnoreCase("LYS")||resRef.equalsIgnoreCase("MET") || resRef.equalsIgnoreCase("PRO")||
	    						resRef.equalsIgnoreCase("ARG") ) )//changed...
	    		{
	    			if(atomRef.equalsIgnoreCase("HG1"))
	    				atomRefStereo="HG2";
					else if(atomRef.equalsIgnoreCase("HG2"))
						atomRefStereo="HG1";				
	    		}
	    		if(subAtomTwo.equalsIgnoreCase("HD")  && (resRef.equalsIgnoreCase("PHE") || 
	    				resRef.equalsIgnoreCase("LYS")||resRef.equalsIgnoreCase("PRO")||
	    				resRef.equalsIgnoreCase("ARG") || resRef.equalsIgnoreCase("TYR")  ) )//changed...
				{
					if(atomRef.equalsIgnoreCase("HD1"))
						atomRefStereo="HD2";
					else if(atomRef.equalsIgnoreCase("HD2"))
						atomRefStereo="HD1";				
				}
	    		if(subAtomTwo.equalsIgnoreCase("HE")  && (resRef.equalsIgnoreCase("PHE") || 
	    				resRef.equalsIgnoreCase("LYS") || resRef.equalsIgnoreCase("TYR")  ) )//changed...
				{
					if(atomRef.equalsIgnoreCase("HE1"))
						atomRefStereo="HE2";
					else if(atomRef.equalsIgnoreCase("HE2"))
						atomRefStereo="HE1";				
				}
	    		if(atomRef.length()>=3)
	    		{
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HG1") && resRef.equalsIgnoreCase("ILE"))
					{
						if(atomRef.equalsIgnoreCase("HG11"))
							atomRefStereo="HG12";
						else if(atomRef.equalsIgnoreCase("HG12"))
							atomRefStereo="HG11";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HD2") && resRef.equalsIgnoreCase("ASN"))
					{
						if(atomRef.equalsIgnoreCase("HD21"))
							atomRefStereo="HD22";
						else if(atomRef.equalsIgnoreCase("HD22"))
							atomRefStereo="HD21";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HE2") && resRef.equalsIgnoreCase("GLN"))
					{
						if(atomRef.equalsIgnoreCase("HE21"))
							atomRefStereo="HE22";
						else if(atomRef.equalsIgnoreCase("HE22"))
							atomRefStereo="HE21";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HH1") && resRef.equalsIgnoreCase("ARG"))
					{
						if(atomRef.equalsIgnoreCase("HH11"))
							atomRefStereo="HH12";
						else if(atomRef.equalsIgnoreCase("HH12"))
							atomRefStereo="HH11";				
					}
	    			if(atomRef.substring(0,3).equalsIgnoreCase("HH2") && resRef.equalsIgnoreCase("ARG"))
					{
						if(atomRef.equalsIgnoreCase("HH21"))
							atomRefStereo="HH22";
						else if(atomRef.equalsIgnoreCase("HH22"))
							atomRefStereo="HH21";				
					}
	    		}
	    		if(subAtomTwo.equalsIgnoreCase("HH")  && resRef.equalsIgnoreCase("ARG")  )//changed...
				{
					if(atomRef.equalsIgnoreCase("HH1"))
						atomRefStereo="HH2";
					else if(atomRef.equalsIgnoreCase("HH2"))
						atomRefStereo="HH1";				
				}
	    		
	    		/////////////////
				if( !(atomRef.substring(0,1).equalsIgnoreCase("H") || atomRef.substring(0,1).equalsIgnoreCase("C")||atomRef.substring(0,1).equalsIgnoreCase("N"))   )
					continue;
				
				String subAtom=atomA;
				double csErr=0.0;
				if(atomRef.substring(0,1).equalsIgnoreCase("H"))
					csErr=csErrH;
				else
					csErr=csErrHeavy;
				
				if(atomA.length()>atomRef.length())
					subAtom=atomA.substring(0,atomRef.length());
				
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRef) )
					isFound=true;
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRefStereo) )
					isFound=true;
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRef) && Math.abs(csRef-cs_hA)<csErr)
					isCorrect=true;		
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRefStereo) && Math.abs(csRef-cs_hA)<csErr)
					isCorrect=true;	
			}//for(int k=0;k<vecRefAsg.size();k++)
			
			if(!isFound)
			{
				System.out.println("Missing resonance assignment: "+ resNoA +" "+ resA+ " - " +atomA);
				missing++;
			}
			else
			{
				if(!isCorrect)			
				{
					System.out.println("...Wrong resonance assignment: "+ resNoA +" "+ resA+ " - " +atomA);
					wrong=wrong+1;
				}
				else
				{
					correct=correct+1;
					if(resA.equalsIgnoreCase("VAL")||resA.equalsIgnoreCase("ILE")||resA.equalsIgnoreCase("LEU")||resA.equalsIgnoreCase("MET")
							||resA.equalsIgnoreCase("PHE")||resA.equalsIgnoreCase("TRP")||resA.equalsIgnoreCase("CYS") )
						correctI=correctI+1;
					else if( !(resA.equalsIgnoreCase("ALA") || resA.equalsIgnoreCase("GLY")) )
						correctII=correctII+1;
				}
			}
			
			totalScH++;
		}//for(int j=0;j<vecSrcAsg.size();j++)

		System.out.println("Total number of assignments (including both bb and sc protons): "+ totalScH);
		System.out.println("Total number of missing sidechain assignments: "+ missing);
		System.out.println("Total number of wrong sidechain assignments: "+ wrong);		
		System.out.println("Total number of correct resonance assignments: "+correct);
		System.out.println("Total number of side chain proton resonance assignments: "+ (nTotalAsgs-nTotalBBAsgs));
		System.out.println("Percentage of correct proton resonance assignments (including both bb and sc resonances): "+(double)( correct/nTotalAsgs ));
		System.out.println("Percentage of correct proton resonance assignments for class I residues: "+(double)( correctI/nTotalClassI ));
		System.out.println("Percentage of correct proton resonance assignments for class II residues: "+(double)( correctII/nTotalClassII ));
	}
	
	/**
	 * written by LW.
	 * check the order of chemical shifts for protons in one residue
	 * 
	 * @param vecOneResidue the vec one residue
	 * 
	 * @return true, if check csbmrb order
	 */
	public boolean checkCSBMRBOrder(Vector vecOneResidue)
	{
		if(vecOneResidue.size()<1)
			return false;	
		double aveCS_ha=0.0, n_ha=0.0;
		double aveCS_hb=0.0, n_hb=0.0;
		double aveCS_hg=0.0, n_hg=0.0;
		double aveCS_hd=0.0,n_hd=0.0;
		double aveCS_he=0.0, n_he=0.0;
		double aveCS_hg2Ile=0.0, n_hg2Ile=0.0;
		double aveCS_hg1Ile=0.0, n_hg1Ile=0.0;
		double CS_hzPhe=-999.9;//for ring, we need to specify
		double CS_he3Trp=-999.9;
		double CS_hz2Trp=-999.9;
		double CS_hz3Trp=-999.9;
		double CS_hh2Trp=-999.9;
		double CS_he1His=-999.9;
		double CS_hd2His=-999.9;
		H1CS h1cs0=(H1CS)vecOneResidue.elementAt(0);
		String res=h1cs0.getResidueType();
		
		for(int i=0;i< vecOneResidue.size();i++)
		{
			H1CS h1cs=(H1CS)vecOneResidue.elementAt(i);
			String atom=h1cs.getAtomName();
			String subAtomTwo=atom;
			String subAtomThree=atom;
			if(atom.length()>=2)
				subAtomTwo=atom.substring(0,2);
			if(atom.length()>=3)
				subAtomThree=atom.substring(0,3);
			if(subAtomTwo.equalsIgnoreCase("HA"))
			{
				aveCS_ha=aveCS_ha+h1cs.getH1CS();
				n_ha=n_ha+1.0;
			}
			if(subAtomTwo.equalsIgnoreCase("HB"))
			{
				aveCS_hb=aveCS_hb+h1cs.getH1CS();
				n_hb=n_hb+1.0;
			}
			if(subAtomTwo.equalsIgnoreCase("HG"))
			{
				aveCS_hg=aveCS_hg+h1cs.getH1CS();
				n_hg=n_hg+1.0;
			}
			if(subAtomTwo.equalsIgnoreCase("HD"))
			{
				aveCS_hd=aveCS_hd+h1cs.getH1CS();
				n_hd=n_hd+1.0;
			}
			if(subAtomTwo.equalsIgnoreCase("HE"))
			{
				aveCS_he=aveCS_he+h1cs.getH1CS();
				n_he=n_he+1.0;
			}
			if(subAtomThree.equalsIgnoreCase("HG2"))
			{
				aveCS_hg2Ile=aveCS_hg2Ile+h1cs.getH1CS();
				n_hg2Ile=n_hg2Ile+1.0;
			}
			if(subAtomThree.equalsIgnoreCase("HG1"))
			{
				aveCS_hg1Ile=aveCS_hg1Ile+h1cs.getH1CS();
				n_hg1Ile=n_hg1Ile+1.0;
			}
			if(subAtomThree.equalsIgnoreCase("HE1") && res.equalsIgnoreCase("HIS"))
			{
				CS_he1His=h1cs.getH1CS();
			}
			if(subAtomThree.equalsIgnoreCase("HD1") && res.equalsIgnoreCase("HIS"))
			{
				CS_hd2His=h1cs.getH1CS();
			}
		}//for(int i=0;i< vecOneResidue.size();i++)
		if(n_ha>0)
			aveCS_ha=aveCS_ha/n_ha;
		if(n_hb>0)
			aveCS_hb=aveCS_hb/n_hb;
		if(n_hg>0)
			aveCS_hg=aveCS_hg/n_hg;
		if(n_hd>0)
			aveCS_hd=aveCS_hd/n_hd;
		if(n_he>0)
			aveCS_he=aveCS_he/n_he;
		if(n_hg2Ile>0)
			aveCS_hg2Ile=aveCS_hg2Ile/n_hg2Ile;
		if(n_hg1Ile>0)
			aveCS_hg1Ile=aveCS_hg1Ile/n_hg1Ile;
		
		boolean isSuccess=true;
		if(res.equalsIgnoreCase("ALA")||res.equalsIgnoreCase("SER")||res.equalsIgnoreCase("CYS")||
				res.equalsIgnoreCase("ASP") || res.equalsIgnoreCase("ASN")|| res.equalsIgnoreCase("PHE")||
				res.equalsIgnoreCase("TYR")|| res.equalsIgnoreCase("HIS") || res.equalsIgnoreCase("TRP"))
			isSuccess=(aveCS_ha>aveCS_hb);
		if(res.equalsIgnoreCase("VAL")||res.equalsIgnoreCase("THR"))
			isSuccess=(aveCS_ha>aveCS_hb) && (aveCS_hb>aveCS_hg);
		if(res.equalsIgnoreCase("LEU"))
			isSuccess=(aveCS_ha>aveCS_hb) && (aveCS_hb>aveCS_hg) &&(aveCS_hg>aveCS_hd);
		if(res.equalsIgnoreCase("ILE"))
			isSuccess=(aveCS_ha>aveCS_hb) && (aveCS_hb>aveCS_hg1Ile) && (aveCS_hg1Ile>aveCS_hg2Ile) && (aveCS_hg2Ile>aveCS_hd);
		if(res.equalsIgnoreCase("GLN") ||res.equalsIgnoreCase("GLU") ||res.equalsIgnoreCase("MET")  )
			isSuccess=(aveCS_ha>aveCS_hg) && (aveCS_hg>aveCS_hb);
		if(res.equalsIgnoreCase("PRO") || res.equalsIgnoreCase("ARG"))
			isSuccess=(aveCS_ha>aveCS_hd) && (aveCS_hd>aveCS_hb) && (aveCS_hb>aveCS_hg);
		if(res.equalsIgnoreCase("LYS"))
			isSuccess=(aveCS_ha>aveCS_he) && (aveCS_he>aveCS_hb) && (aveCS_hb>aveCS_hd)&& (aveCS_hd>aveCS_hg);
		if(res.equalsIgnoreCase("HIS"))
			isSuccess=isSuccess && (CS_he1His>CS_hd2His);
		if(res.equalsIgnoreCase("TYR"))
			isSuccess=isSuccess && (aveCS_hd>aveCS_he);
		return isSuccess;
	}
	
	/**
	 * This is an old function.
	 * compute the statistical information of the sidechain assignment
	 * 
	 * @param vecSrcAsg original assignment list
	 * @param vecRefAsg reference assignment list
	 */
	public void CompareSCAsgsWRefAsgsOld(Vector vecSrcAsg,Vector vecRefAsg )
	{
		double csErrH=0.04;
		double csErrHeavy=0.4;
		
		Peak pk=new Peak();
		int totalScH=0;
		int wrong=0,correct=0;
		Assign asg=new Assign();		
		
		for(int j=0;j<vecSrcAsg.size();j++)
		{
			H1CS h1csA=(H1CS)vecSrcAsg.elementAt(j);
			int resNoA=h1csA.getResidueNo();
			String resA=h1csA.getResidueType();
			String atomA=h1csA.getAtomName();
			double cs_hA=h1csA.getH1CS();		
					
			boolean isAtomInRefAsg=false;
			
			if(atomA.equalsIgnoreCase("CA") ||atomA.equalsIgnoreCase("N")|| 
					atomA.equalsIgnoreCase("HN") ||atomA.equalsIgnoreCase("H")||
					atomA.equalsIgnoreCase("HA")||atomA.equalsIgnoreCase("HA1")|| atomA.equalsIgnoreCase("HA2")||atomA.equalsIgnoreCase("HA3")||
					atomA.equalsIgnoreCase("C")||atomA.equalsIgnoreCase("CO")||atomA.equalsIgnoreCase("CB"))
				continue;		
			
			//check whether the proton name is in assignment list
			for(int k=0;k<vecRefAsg.size();k++)
			{
				H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
				int resRefNo=h1cs.getResidueNo();
				String atomRef=h1cs.getAtomName();
				String subAtom=atomA;
				if(atomA.length()>atomRef.length())
					subAtom=atomA.substring(0,atomRef.length());
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRef) )
					isAtomInRefAsg=true;								
			}//for(int k=0;k<vecRefAsg.size();k++)			
				
			if(!isAtomInRefAsg)
				continue;
			
			Vector vecRefAsgStereo=asg.SwapStereoAsg(vecRefAsg,resNoA);		
			
			boolean isCorrect=false;
			for(int k=0;k<vecRefAsg.size();k++)
			{
				H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
				int resRefNo=h1cs.getResidueNo();
				String atomRef=h1cs.getAtomName();
				String subAtom=atomA;
				double csErr=0.0;
				if(atomRef.substring(0,1).equalsIgnoreCase("H"))
					csErr=csErrH;
				else
					csErr=csErrHeavy;
				double csRef=h1cs.getH1CS();
				if(atomA.length()>atomRef.length())
					subAtom=atomA.substring(0,atomRef.length());
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRef) && Math.abs(csRef-cs_hA)<csErr)
					isCorrect=true;								
			}//for(int k=0;k<vecRefAsg.size();k++)
			
			for(int k=0;k<vecRefAsgStereo.size();k++)
			{
				H1CS h1cs=(H1CS)vecRefAsgStereo.elementAt(k);
				int resRefNo=h1cs.getResidueNo();
				String atomRef=h1cs.getAtomName();
				String subAtom=atomA;
				double csErr=0.0;
				if(atomRef.substring(0,1).equalsIgnoreCase("H"))
					csErr=csErrH;
				else
					csErr=csErrHeavy;
				double csRef=h1cs.getH1CS();
				if(atomA.length()>atomRef.length())
					subAtom=atomA.substring(0,atomRef.length());
				if(resNoA==resRefNo && subAtom.equalsIgnoreCase(atomRef) && Math.abs(csRef-cs_hA)<csErr)
					isCorrect=true;								
			}//for(int k=0;k<vecRefAsg.size();k++)
			if(isCorrect)
				correct++;
			else
			{
				wrong++;
				System.out.println("Wrong sidechain assignment: "+ resNoA +" "+ resA+ " - " +atomA); 
			}
				
			totalScH++;						
		}//for(int j=0;j<vecSrcAsg.size();j++)
		
		System.out.println("Total number of sidechain atoms: "+ totalScH);
		System.out.println("Total number of wrong sidechain assignments: "+ wrong);
		System.out.println("Total number of correct side chain resonance assignments: "+correct);
		
	}
   
   /**
    * Another method for reading chemical shifts, in BMRB format.
    * 
    * @param filename the name of the file
    * 
    * @return a vector of h1cs object
    */
    public Vector<H1CS> h1CSReader_BMRB(String filename)
    {
    	Vector inputs = new Vector();
    	double csHN = 0.0, csN15 = 0.0, csH1 = 0.0, intensity = 0.0;
    	Vector csVec = new Vector();
    	StringTokenizer st = new StringTokenizer("");
    	String str = "";
    	int index1=0, index2=0;
    	String order="";
    	double value=0.0;
    	int no =0;
    	String aaType = "", ss ="";
    	String atomName="";
    	String resName="";
    	int resNo=0;
    	int csID=0;
    	Noe noe=new Noe();
    	
    	try
    	{
    		BufferedReader in = new BufferedReader(new FileReader(filename));	
    		ss = in.readLine();
    		
    		stop:
    			while(true) 
    			{   			    
    				st = new StringTokenizer(ss);
    				if (st.hasMoreTokens())
    					csID =  Integer.parseInt(st.nextToken());
    					
    				if (st.hasMoreTokens())
    					resNo =  Integer.parseInt(st.nextToken());
    				if (st.hasMoreTokens())
    					resName = st.nextToken();
    				if (st.hasMoreTokens())
    					atomName = st.nextToken();
    				if (st.hasMoreTokens())
    					aaType = st.nextToken();
    				if (st.hasMoreTokens())
    					csH1  = new Double(st.nextToken()).doubleValue();
    				
    				if (csH1>300.0 || csH1<-300.0)
					{
						if ((ss = in.readLine()) == null)
							break stop;     
						else if(ss.equalsIgnoreCase(""))
	    				{
	    					if ((ss = in.readLine()) == null )
	        				{
	        					break stop;
	        				}
	    				}
						continue;
					}
    				
    				atomName=noe.UpdateNOEAtomName2(resName, atomName);//changed here
    				inputs.add(new H1CS(resNo, resName, atomName, csH1));
    				if ((ss = in.readLine()) == null){
    					break stop;
    				}
    				else if(ss.equalsIgnoreCase(""))
    				{
    					if ((ss = in.readLine()) == null )
        				{
        					break stop;
        				}
    				}
    			}
    	}catch (FileNotFoundException e) {
    		System.out.println("File not found: " + filename);
    	}catch (IOException e) 
    	{
    		System.out.println("IOException: the stack trace is:");
    		e.printStackTrace();
    	}
    	return inputs;
    }
  
    /**
     * Another method for reading chemical shifts, in BMRB format.
     * 
     * @param filename the name of the file
     * 
     * @return a vector of h1cs object
     */
     public Vector<H1CS> h1CSReader_CYANA(String filename)
     {
     	Vector inputs = new Vector();
     	double csHN = 0.0, csN15 = 0.0, csH1 = 0.0, intensity = 0.0,csErr=0.0;
     	Vector csVec = new Vector();
     	StringTokenizer st = new StringTokenizer("");
     	String str = "";
     	int index1=0, index2=0;
     	String order="";
     	double value=0.0;
     	int no =0;
     	String aaType = "", ss ="";
     	String atomName="";
     	String resName="";
     	int resNo=0;
     	int csID=0;
     	Noe noe=new Noe();
     	
     	try
     	{
     		BufferedReader in = new BufferedReader(new FileReader(filename));	
     		ss = in.readLine();
     		
     		stop:
     			while(true) 
     			{   			    
     				st = new StringTokenizer(ss);
     				if (st.hasMoreTokens())
     					csID =  Integer.parseInt(st.nextToken());
     					
     				if (st.hasMoreTokens())
     					csH1  = new Double(st.nextToken()).doubleValue();
     				if (st.hasMoreTokens())
     					csErr  = new Double(st.nextToken()).doubleValue();
     				if (st.hasMoreTokens())
     					atomName = st.nextToken();
     				
     				
     				if (st.hasMoreTokens())
     					resNo =  Integer.parseInt(st.nextToken());
     				
     				
     				
     				if (csH1>300.0 || csH1<-300.0)
 					{
 						if ((ss = in.readLine()) == null)
 							break stop;     
 						else if(ss.equalsIgnoreCase(""))
 	    				{
 	    					if ((ss = in.readLine()) == null )
 	        				{
 	        					break stop;
 	        				}
 	    				}
 						continue;
 					}
     				
     				//atomName=noe.UpdateNOEAtomName2(resName, atomName);//changed here
     				inputs.add(new H1CS(resNo, resName, atomName, csH1));
     				if ((ss = in.readLine()) == null){
     					break stop;
     				}
     				else if(ss.equalsIgnoreCase(""))
     				{
     					if ((ss = in.readLine()) == null )
         				{
         					break stop;
         				}
     				}
     			}
     	}catch (FileNotFoundException e) {
     		System.out.println("File not found: " + filename);
     	}catch (IOException e) 
     	{
     		System.out.println("IOException: the stack trace is:");
     		e.printStackTrace();
     	}
     	return inputs;
     }
     
    /**
     * Another method for reading chemical shifts, in TALOS format.
     * 
     * @param filename the name of the file
     * 
     * @return a vector of h1cs object
     */
     public Vector<H1CS> h1CSReader_TALOS(String filename)
     {
     	Vector inputs = new Vector();
     	double csHN = 0.0, csN15 = 0.0, csH1 = 0.0, intensity = 0.0;
     	Vector csVec = new Vector();
     	StringTokenizer st = new StringTokenizer("");
     	String str = "";
     	int index1=0, index2=0;
     	String order="";
     	double value=0.0;
     	int no =0;
     	String aaType = "", ss ="";
     	String atomName="";
     	String resName="";
     	int resNo=0;
     	int csID=0;
     	//Noe noe=new Noe();
     	
     	try
     	{
     		BufferedReader in = new BufferedReader(new FileReader(filename));	
     		ss = in.readLine();
     		
     		stop:
     			while(true) 
     			{   			    
     				st = new StringTokenizer(ss);
     				     
     				String strFirst="";
     				if (st.hasMoreTokens()){
     					strFirst=st.nextToken();
     					if(isIntNumber(strFirst) ){    						
    						resNo= Integer.parseInt(strFirst);   					
    					}//if(isIntNumber(strFirst) )
     					else
     					{
     						if ((ss = in.readLine()) == null )
 	        				{
 	        					break stop;
 	        				}
     						continue;
     					}
     				}
     				
     				    				
     				if (st.hasMoreTokens())
     					resName = st.nextToken();
     				if (st.hasMoreTokens())
     					atomName = st.nextToken();
     				//if (st.hasMoreTokens())
     					//aaType = st.nextToken();
     				if (st.hasMoreTokens())
     					csH1  = new Double(st.nextToken()).doubleValue();
     				
     				if (csH1>300.0 || csH1<-300.0)
 					{
 						if ((ss = in.readLine()) == null)
 							break stop;     
 						else if(ss.equalsIgnoreCase(""))
 	    				{
 	    					if ((ss = in.readLine()) == null )
 	        				{
 	        					break stop;
 	        				}
 	    				}
 						continue;
 					}
     				
     				//atomName=noe.UpdateNOEAtomName2(resName, atomName);//changed here
     				inputs.add(new H1CS(resNo, resName, atomName, csH1));
     				if ((ss = in.readLine()) == null){
     					break stop;
     				}
     				else if(ss.equalsIgnoreCase(""))
     				{
     					if ((ss = in.readLine()) == null )
         				{
         					break stop;
         				}
     				}
     			}
     	}catch (FileNotFoundException e) {
     		System.out.println("File not found: " + filename);
     	}catch (IOException e) 
     	{
     		System.out.println("IOException: the stack trace is:");
     		e.printStackTrace();
     	}
     	return inputs;
     }
     
     public boolean isIntNumber(String num){	 
 		try{	  
 	      Integer.parseInt(num);
 	      } catch(NumberFormatException nfe) {	   
 	      return false;	   
 	      }	   
 	      return true;
       }
     
    /**
     * compute the fitness score compared to BMRB statistical information.
     * 
     * @param vecSrcAsg the vec src asg
     * @param vecBMRB the vec bmrb
     * 
     * @return matching score
     */
    public double BMRBSatisticsScore(Vector vecSrcAsg, Vector vecBMRB)
    {
    	double counter=0.0;
    	double sum_dist=0.0;
    	double sum_score=0.0;
    	for(int i=0;i<vecSrcAsg.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecSrcAsg.elementAt(i);
    		int resNo=h1cs.getResidueNo();
    		String res=h1cs.getResidueType();
    		String atom=h1cs.getAtomName();
    		double csH=h1cs.getH1CS();
    		double cs_ave=0.0;
    		double cs_signma=0.0;
    		
    		boolean isFoundInBMRB=false;
    		for(int j=0;j<vecBMRB.size();j++)
    		{
    			H1CS h1csBMRB=(H1CS)vecBMRB.elementAt(j);
    			String resBMRB=h1csBMRB.getResidueType();
    			String atomBMRB=h1csBMRB.getAtomName();
    			
    			if(res.equalsIgnoreCase(resBMRB) && atom.equalsIgnoreCase(atomBMRB) )
    			{
    				cs_ave=h1csBMRB.getCSAverage();
    				if(atomBMRB.substring(0,1).equalsIgnoreCase("H"))
    					cs_signma=1*h1csBMRB.getCSSigma();
    				else
    					cs_signma=1*h1csBMRB.getCSSigma();
    				isFoundInBMRB=true;
    			}
    			
    		}//for(int j=0;j<vecBMRB.size();j++)
    		if(!isFoundInBMRB)
    		{
    			System.out.println("Not found match in BMRB: "+resNo+res+"-"+atom);
    			continue;
    		}
    		
			double cs_dist=((csH-cs_ave)*(csH-cs_ave))/(cs_signma*cs_signma);
			if(atom.substring(0,1).equalsIgnoreCase("H"))
			{
				sum_dist=sum_dist+cs_dist;
				counter=counter+1;
			}
			else
			{
				sum_dist=sum_dist+1*cs_dist;
				counter=counter+1;
			}
				
    	}//for(int i=0;i<vecSrcAsg.size();i++)
    	double ave_dist=(double)(sum_dist/counter);
    	double score=Math.exp(-0.5*ave_dist);
    	//double score=Math.exp(-0.5*sum_dist);
    	return score;//sum_score/counter;
    }
    
    /**
     * read BMRB statistical information.
     * The required file format is:
     * 
     * @param filename the name of the file
     * 
     * @return a vector of H1CS object
     */
    public Vector<H1CS> ReadBMRBSatistics(String filename)
    {
    	Vector inputs = new Vector();
    	double csHN = 0.0, csN15 = 0.0, csH1 = 0.0, intensity = 0.0;
    	double csL=0.0, csU=0.0, csAve=0.0,csSigma=0.0;
    	Vector csVec = new Vector();
    	StringTokenizer st = new StringTokenizer("");
    	String str = "";
    	int index1=0, index2=0;
    	String order="";
    	double value=0.0;
    	int no =0;
    	String aaType = "", ss ="";
    	String atomName="";
    	String resName="";
    	int resNo=0;
    	int csID=0;
    	
    	try
    	{
    		BufferedReader in = new BufferedReader(new FileReader(filename));	
    		ss = in.readLine();
    		
    		stop:
    			while(true) 
    			{   			    
    				st = new StringTokenizer(ss);
    				
    				if (st.hasMoreTokens())
    					resName = st.nextToken();
    				if (st.hasMoreTokens())
    					atomName = st.nextToken();
    				String subAtomName=atomName;
    				if(atomName.length()>2)
    					subAtomName=atomName.substring(0,2);
    				
    				if (st.hasMoreTokens())
    					st.nextToken();//atom type
    				
    				if (st.hasMoreTokens())
    					st.nextToken();//number of shifts
    				
    				if (st.hasMoreTokens())
    					csL  = new Double(st.nextToken()).doubleValue();
    				if (st.hasMoreTokens())
    					csU  = new Double(st.nextToken()).doubleValue();
    				
    				if (st.hasMoreTokens())
    					csAve  = new Double(st.nextToken()).doubleValue();//Average shift
    				
    				if (st.hasMoreTokens())
    					csSigma  = new Double(st.nextToken()).doubleValue();//Standard deviation
    				   				
    			
    				//if(subAtomName.equalsIgnoreCase("HB"))
    				//	inputs.add(new H1CS(0, resName, atomName, csAve-4*csSigma,csAve+4*csSigma,csAve,csSigma));
    				//else
    					
    				//inputs.add(new H1CS(0, resName, atomName, csAve-2*csSigma,csAve+2*csSigma,csAve,csSigma)); //original one
    				
    				/*if(atomName.substring(0,1).equalsIgnoreCase("H") || atomName.substring(0,1).equalsIgnoreCase("N") 
    						|| atomName.equalsIgnoreCase("CB") || resName.equalsIgnoreCase("PHE") )	
    					inputs.add(new H1CS(0, resName, atomName, csAve-2*csSigma,csAve+2*csSigma,csAve,csSigma));
    				else
    					inputs.add(new H1CS(0, resName, atomName, csAve-csSigma,csAve+csSigma,csAve,csSigma)); 
    				*/
    				
    				if( (atomName.equalsIgnoreCase("CD") || atomName.equalsIgnoreCase("CG") ) &&
    						(resName.equalsIgnoreCase("LYS") || resName.equalsIgnoreCase("ARG") ) )	
    					inputs.add(new H1CS(0, resName, atomName, csAve-csSigma,csAve+csSigma,csAve,csSigma));
    				else
    					inputs.add(new H1CS(0, resName, atomName, csAve-2*csSigma,csAve+2*csSigma,csAve,csSigma)); 
    				
    				if ((ss = in.readLine()) == null){
    					break stop;
    				}
    			}
    	}catch (FileNotFoundException e) {
    		System.out.println("File not found: " + filename);
    	}catch (IOException e) 
    	{
    		System.out.println("IOException: the stack trace is:");
    		e.printStackTrace();
    	}
    	return inputs;
    }
    
    /**
     * Convert naming scheme From BMRBNew To PDBNew,
     * including pseudo protons of methyl groups.
     * 
     * @param vecAsg the vec asg
     * 
     * @return the vector
     */
    public Vector<H1CS> CSNameConvertFromBMRBNewToPDBNew(final Vector<H1CS> vecAsg)
    {
		Cartesian atom = null;
		String atomName="", resid;
		String atomName2="";
		Vector atoms = new Vector();
		Vector atomVec = new Vector();
		
		Pdb pp=new Pdb();
		String res;
		
	
		H1CS h1cs = new H1CS();
		Vector vecAsgOld=new Vector();
		vecAsgOld.addAll(vecAsg);
		
		Vector vecAsgNew=new Vector();
		
		for (int i=0; i<vecAsgOld.size(); i++)
		{
			h1cs = (H1CS)vecAsgOld.elementAt(i);
			res=h1cs.getResidueType();
			atomName=h1cs.getAtomName();
			
			String atomNew=pp.NameFromBMRBToPDB(res,atomName);
			h1cs.setAtomName(atomNew);
			
			vecAsgNew.add(h1cs);
			
		}//for (int i=0; i<vecNoeOld.size(); i++)
	
		return vecAsgNew;
    }
    
    /**
     * Convert naming scheme From PDBNew To BMRBNew,
     * including pseudo protons of methyl groups.
     * 
     * @param vecAsg the vec asg
     * 
     * @return the vector
     */
    public Vector CSNameConvertFromPDBNewToBMRBNew(final Vector vecAsg)
    {
		Cartesian atom = null;
		String atomName="", resid;
		String atomName2="";
		Vector atoms = new Vector();
		Vector atomVec = new Vector();
		
		Pdb pp=new Pdb();
		String res;
		
		H1CS h1cs = new H1CS();
		Vector vecAsgOld=new Vector();
		vecAsgOld.addAll(vecAsg);
		
		Vector vecAsgNew=new Vector();
		
		for (int i=0; i<vecAsgOld.size(); i++)
		{
			h1cs = (H1CS)vecAsgOld.elementAt(i);
			res=h1cs.getResidueType();
			atomName=h1cs.getAtomName();
			
			String atomNew=pp.NameFromPdbToBMRB(res,atomName);
			h1cs.setAtomName(atomNew);
			
			vecAsgNew.add(h1cs);
			
		}//for (int i=0; i<vecNoeOld.size(); i++)
	
		return vecAsgNew;
    }
    
  //convert cs format from PDB-new to BMRB-New:
    /**
   * Main0 for testing.
   * 
   * @param argv the argv
   */
  public static void main0(String[] argv)
    {
    	Assign asg = new Assign();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String assignFile=src+"ff2_cs_different_format.txt";    	
    	
    	String seqFile = src + "ff2.seq";
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile); 
    	H1CS h1CS=new H1CS();
    	Vector assignVec=new Vector();    	
    	assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	Vector assignNew=h1CS.CSNameConvertFromPDBNewToBMRBNew(assignVec);
    	//output the cs in BMRB-New format:
    	int counter=1;
    	System.out.println("The chemical shift in the BMRB-NEW format:");
    	for(int i=0;i<assignNew.size();i++)
    	{
    		H1CS h1csTemp=(H1CS)assignNew.elementAt(i);
    		int no=h1csTemp.getResidueNo();
    		String res=h1csTemp.getResidueType();
    		String atom=h1csTemp.getAtomName();
    		double cs=h1csTemp.getH1CS();
    		
    		System.out.println(counter+"   "+cs+"   "+0.0+"   "+atom+"   "+no);
    		
    		counter++;
    	}//for(int i=0;i<assignVec.size();i++)
    	
    	
    }
  
  public void addH1CS(Vector vecOldAsg, H1CS h1csNew){
	  boolean isIn=false;
	  for(int i=0;i<vecOldAsg.size();i++){
	  		H1CS h1cs=(H1CS)vecOldAsg.elementAt(i);
	  		String atom=h1cs.getAtomName();
	  		int resNo=h1cs.getResidueNo();
	  		if( (resNo==h1csNew.getResidueNo() ) && atom.equalsIgnoreCase(h1csNew.getAtomName() ))
	  			isIn=true;
	  }
	  if(! isIn)
		  vecOldAsg.add(h1csNew);
  }
  //combine different prot files into one file:
  public static void main4(String[] argv)
  {
	  
	  Assign asg = new Assign();
	  Peak pk=new Peak();
	  String userDir = System.getProperty("user.dir");////
	  	String src=userDir+"/inputFiles/";
	  	String assignFile=src+"Cycle1_3_ff2_n15_noe.prot";
	  	String asgFileAli=src+"Cycle1_2_ff2_c13_ali.prot";
	  	String asgFileAro=src+"Cycle1_1_ff2_c13_aro_CT_noe.prot";
	  	
	  	//String seqFile = src+"cbf.seq";
	  	String seqFile = src + "ff2.seq";
	  	//this is the vector of residues (protein sequence)
	  	Vector vecSeq=asg.ReaderSeq(seqFile);    
	  	
	  	String outResFileName =src+ "final_combined.prot";
	  	Vector assignVec=new Vector();
	  	H1CS h1CS=new H1CS();
	  	//assignVec=h1CS.h1CSReader(assignFile,vecSeq);//CYANA format   	
	  	assignVec=h1CS.h1CSReaderWQ(assignFile,vecSeq);  //BMRB format
	  	Nasca nasca=new Nasca();
	  	//Vector vecFinalCSs=nasca.ExtractBackboneAsignment(assignVec);
	  	Vector vecFinalCSs=new Vector();
	  	for(int i=0;i<assignVec.size();i++){
	  		H1CS h1cs=(H1CS)assignVec.elementAt(i);
	  		String atom=h1cs.getAtomName();
	  		if(atom.substring(0,1).equalsIgnoreCase("Q"))
			{
	  			atom="H"+atom.substring(1,atom.length() );
			}
	  		
	  		if(atom.substring(0, 1).equalsIgnoreCase("N"))
	  		{
	  			vecFinalCSs.add(h1cs);	
	  			continue;	  			
	  		}
	  		String res=h1cs.getResidueType();
	  		String atomHeavy=pk.GetHeavyAtomFromProton(res, atom);
	  		if( !atomHeavy.substring(0,1).equalsIgnoreCase("N"))
	  			continue;
	  		vecFinalCSs.add(h1cs);		
	  	}
	  	
	  	
	  	//aliphatic scs
	  	Vector vecAsgAli=h1CS.h1CSReaderWQ(asgFileAli,vecSeq);
	  	//Vector vecAsgAliSCs=nasca.ExtractSideChainAsignment(vecAsgAli);
	  	for(int i=0;i<vecAsgAli.size();i++){
	  		H1CS h1cs=(H1CS)vecAsgAli.elementAt(i);
			String atom=h1cs.getAtomName();
			
			if(atom.substring(0,1).equalsIgnoreCase("Q"))
			{
	  			atom="H"+atom.substring(1,atom.length() );
			}
			if(atom.substring(0, 1).equalsIgnoreCase("C"))
	  		{
	  			vecFinalCSs.add(h1cs);	
	  			continue;	  			
	  		}
			
			String res=h1cs.getResidueType();
			String atomHeavy=pk.GetHeavyAtomFromProton(res, atom);
			if( !atomHeavy.substring(0,1).equalsIgnoreCase("C"))
	  			continue;			
			
			if(res.equalsIgnoreCase("PHE") || res.equalsIgnoreCase("HIS") || 
					res.equalsIgnoreCase("TRP") || res.equalsIgnoreCase("TYR") )
			{
				if(atomHeavy.length()>=2)
				{
					if( !(atomHeavy.substring(0, 2).equalsIgnoreCase("CA") || atomHeavy.substring(0, 2).equalsIgnoreCase("CB")) )
						continue;
				}
			}
				
			vecFinalCSs.add(h1cs);			
	  	}
	  		
	  	//aromatic scs
	  	Vector vecAsgAro=h1CS.h1CSReaderWQ(asgFileAro,vecSeq);
	  	//Vector vecAsgAroSCs=nasca.ExtractSideChainAsignment(vecAsgAro);
	  	for(int i=0;i<vecAsgAro.size();i++){
	  		H1CS h1cs=(H1CS)vecAsgAro.elementAt(i);
			String atom=h1cs.getAtomName();
			if(atom.substring(0,1).equalsIgnoreCase("Q"))
			{
	  			atom="H"+atom.substring(1,atom.length() );
			}
			if(atom.substring(0, 1).equalsIgnoreCase("C"))
	  		{
	  			vecFinalCSs.add(h1cs);	
	  			continue;	  			
	  		}
			
			String res=h1cs.getResidueType();
			String atomHeavy=pk.GetHeavyAtomFromProton(res, atom);
			if( !atomHeavy.substring(0,1).equalsIgnoreCase("C"))
	  			continue;
			if( !(res.equalsIgnoreCase("PHE") || res.equalsIgnoreCase("HIS") || 
					res.equalsIgnoreCase("TRP") || res.equalsIgnoreCase("TYR")) )
				continue;
			else
			{
				if(atomHeavy.length()>=2)
				{
					if( atomHeavy.substring(0, 2).equalsIgnoreCase("CA") || atomHeavy.substring(0, 2).equalsIgnoreCase("CB") )
						continue;
				}
			}
			vecFinalCSs.add(h1cs);			
	  	}
	  	
	  	Collections.sort(vecFinalCSs, new H1CS.resNoComparator());
	  	
	  	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outResFileName)));
    		
    		//if(strOutResonFormat.equalsIgnoreCase("CYANA"))        	
    		//h1CS.PrintAllAssignmentToFileCyana(vecBBReson,out);        	
        	//else        	
           // h1CS.PrintAllAssignmentToFileBMRB(vecFinalCSs,out); //using the same atom naming scheme 
    		h1CS.PrintAllAssignmentToFileCyana(vecFinalCSs,out);  
	       
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ outResFileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + outResFileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
  }
    
    //compare two prot files:
    /**
     * The main method for testing.
     * 
     * @param argv the arguments
     */
    public static void main(String[] argv)
    {
    	Assign asg = new Assign();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String assignFile=src+"ff2.prot";
    	String assignFileOld=src+"ff2temp.prot";
    	
    	String seqFile = src + "ff2.seq";
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile); 
    	H1CS h1CS=new H1CS();
    	Vector assignVec=new Vector();
    	Vector assignVecOld=new Vector();
    	assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	
    	assignVecOld=h1CS.h1CSReader(assignFileOld,vecSeq); 
    	double errH=0.03;
    	double errHeavy=0.3;
    	
    	System.out.println("----------Print out the missing resonances:---------------");
    	for(int i=1;i<assignVec.size();i++)
    	{
    		H1CS h1cs_new=(H1CS)assignVec.elementAt(i);
    		boolean isIn=false;
    		String res_new=h1cs_new.getResidueType();
    		String atom_new=h1cs_new.getAtomName();
    		int no_new=h1cs_new.getResidueNo();   		
    		
    		for(int j=0;j<assignVecOld.size();j++)
    		{
    			H1CS h1cs_old=(H1CS)assignVecOld.elementAt(j);
    			String res_old=h1cs_old.getResidueType();
    			String atom_old=h1cs_old.getAtomName();
    			int no_old=h1cs_old.getResidueNo();
    			if(no_new==no_old && atom_new.equalsIgnoreCase(atom_old))
    			{
    				isIn=true;
    				break;
    			}
    			
    		}//for(int j=0;j<assignVecOld.size();j++)
    		
    		if(!isIn)
    			System.out.println("...Missing..."+no_new+" - "+res_new+" : "+atom_new);
    	}//for(int i=1;i<assignVec.size();i++)
    	
    	System.out.println("----------Print out the different resonances:---------------");
    	for(int i=1;i<assignVec.size();i++)
    	{
    		H1CS h1cs_new=(H1CS)assignVec.elementAt(i);
    		
    		String res_new=h1cs_new.getResidueType();
    		String atom_new=h1cs_new.getAtomName();
    		int no_new=h1cs_new.getResidueNo();   	
    		double cs_new=h1cs_new.getH1CS();
    		double err=0.0;
    		if(atom_new.substring(0,1).equalsIgnoreCase("H"))
    			err=errH;
    		else
    			err=errHeavy;
    		
    		for(int j=0;j<assignVecOld.size();j++)
    		{
    			H1CS h1cs_old=(H1CS)assignVecOld.elementAt(j);
    			String res_old=h1cs_old.getResidueType();
    			String atom_old=h1cs_old.getAtomName();
    			double cs_old=h1cs_old.getH1CS();
    			int no_old=h1cs_old.getResidueNo();
    			if(no_new==no_old && atom_new.equalsIgnoreCase(atom_old) && Math.abs(cs_new-cs_old)>err)
    			{
    				System.out.println("Different..."+no_new+"-"+res_new+"-"+atom_new+"...Old:"+cs_old+" New: "+cs_new);
    				break;
    			}
    			
    		}//for(int j=0;j<assignVecOld.size();j++)
    		
    		//System.out.println("...Missing..."+no_new+" - "+res_new+" : "+atom_new);
    	}//for(int i=1;i<assignVec.size();i++)
    	
    	
    	
    	
    }
    //extract the backbone resonance assignments and output them
    /**
     * Main2 for testing.
     * 
     * @param argv the argv
     */
    public static void main2(String[] argv)
    {
    	Assign asg = new Assign();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String assignFile=src+"eta_bmrb.prot";
    	//String seqFile = src+"cbf.seq";
    	String seqFile = src + "eta.seq";
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    
    	
    	String outResFileName =src+ "eta_bb_bmrb.prot";
    	Vector assignVec=new Vector();
    	H1CS h1CS=new H1CS();
    	//assignVec=h1CS.h1CSReader(assignFile,vecSeq);//CYANA format   	
    	assignVec=h1CS.h1CSReader_BMRB(assignFile); //BMRB format
    	Nasca nasca=new Nasca();
    	Vector vecBBReson=nasca.ExtractBackboneAsignment(assignVec);
    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outResFileName)));
    		
    		//if(strOutResonFormat.equalsIgnoreCase("CYANA"))        	
    		//h1CS.PrintAllAssignmentToFileCyana(vecBBReson,out);        	
        	//else        	
            h1CS.PrintAllAssignmentToFileBMRB(vecBBReson,out); //using the same atom naming scheme     	
	       
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ outResFileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + outResFileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
    	
    	
    }
    
    /** obtain the format of a given chemical shift file:
     * 
     * "CYANA" for the CYANA format, "BMRB" for BMRB format.

     * @return
     */
    public String obtainChemicalShiftFileFormat(String fileName){
    	String fileFormat="BMRB";//default format.
    	
     	
     	double csHN = 0.0, csN15 = 0.0,  intensity = 0.0,csErr=0.0;
     	Vector csVec = new Vector();
     	StringTokenizer st = new StringTokenizer("");
     	String str = "";
     	int index1=0, index2=0;
     	String order="";
     	double value=0.0;
     	int no =0;
     	String aaType = "", ss ="";
     	String atomName="";
     	String resName="";
     	int resNo=0;
     	String csID="",csH1 = "";
     	Noe noe=new Noe();
     	
     	try
     	{
     		BufferedReader in = new BufferedReader(new FileReader(fileName));	
     		ss = in.readLine();
     		
     		stop:
     			while(true) 
     			{   			    
     				st = new StringTokenizer(ss);
     				if (st.hasMoreTokens())     					
     					csID =  st.nextToken();
     					
     				if (st.hasMoreTokens())
     					csH1  = st.nextToken();
     				
     				
     				if (st.hasMoreTokens()){
     					boolean isNum=isNumeric(st.nextToken());
     					if(isNum)
     						fileFormat="CYANA";     					
     					else
     						fileFormat="BMRB"; 
     					return fileFormat;
     				}//if (st.hasMoreTokens())
     					
     				
     				if ((ss = in.readLine()) == null){
     					break stop;
     				}
     				else if(ss.equalsIgnoreCase(""))
     				{
     					if ((ss = in.readLine()) == null )
         				{
         					break stop;
         				}
     				}
     			}
     	}catch (FileNotFoundException e) {
     		System.out.println("File not found: " + fileName);
     	}catch (IOException e) 
     	{
     		System.out.println("IOException: the stack trace is:");
     		e.printStackTrace();
     	}   	
    	return fileFormat;
    }
   
    public static boolean isNumeric(String str)  
    {  
      try  
      {  
        double d = Double.parseDouble(str);  
      }  
      catch(NumberFormatException nfe)  
      {  
        return false;  
      }  
      return true;  
    }

}
