package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Noe.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

   
import java.io. *;
import java.util. *;

import Jampack.JampackException;

// TODO: Auto-generated Javadoc
/** * 
 * 
 *   This class provides data structures and functions for processing NOE assignment.
 *    Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
 * 
 */
public class Noe implements Cloneable{
	
	/** The peak id. */
	private int peakID=0;
    
    /** The residue no a. */
    private int residueNoA;   //the residue  No of one of the two partners of the NOE
    
    /** The residue no b. */
    private int residueNoB;   //the residue  No of Another partner of the NOE
    
    /** The res a. */
    private String resA="";
    
    /** The res b. */
    private String resB="";
    
    /** The atom a. */
    private String atomA;     //the name of Atom A
    
    /** The atom b. */
    private String atomB;     //the name of Atom B
    
    /** The lower bound. */
    private double lower;    //the bound for NOEs
    
    /** The upper bound. */
    private double upper;
    
    /** rotio of correct NOEs in an ensemble of structures. */
    private double ratioEnsem;//
    
    /** all protons in the first position in multiple assignments. */
    private Vector vecH_A=new Vector();// 
    
    /** all second protons in multiple assignments. */
    private Vector vecH_B=new Vector();
   
    /**
     * Constructors.
     */
    public Noe()   {
    	residueNoA = 0;
	residueNoB = 0;
        atomA = null;
        atomB = null;
	lower = 0.0;
	upper = 0.0;
	 peakID=0;
	 ratioEnsem=0.0;
	 vecH_A=new Vector();
	 vecH_B=new Vector();
    }
    
    /**
     * Instantiates a new noe.
     * 
     * @param NoA the no a
     * @param noB the no b
     */
    public Noe(int NoA, int noB) {
    	 peakID=0;
    	residueNoA = NoA;
	residueNoB = noB;
        atomA = null;
        atomB = null;
	lower = 0.0;
	upper = 0.0;
	ratioEnsem=0.0;
   }
   
   /**
    * Instantiates a new noe.
    * 
    * @param Ano the ano
    * @param Bno the bno
    * @param res1 the res1
    * @param res2 the res2
    * @param atom1 the atom1
    * @param atom2 the atom2
    * @param low the low
    * @param up the up
    */
   public Noe(int Ano,  int Bno, String res1, String res2,String atom1, String atom2,  double low, double up)
   { peakID=0;
	   	residueNoA = Ano;
	   	residueNoB = Bno;
	   	resA=res1;
	   	resB=res2;
        this.atomA = atom1;
        this.atomB = atom2;
        lower = low;
        upper = up;ratioEnsem=0.0;
    }
   
   /**
    * Instantiates a new noe.
    * 
    * @param pkID the pk id
    * @param Ano the ano
    * @param Bno the bno
    * @param res1 the res1
    * @param res2 the res2
    * @param atom1 the atom1
    * @param atom2 the atom2
    * @param low the low
    * @param up the up
    */
   public Noe(int pkID, int Ano,  int Bno, String res1, String res2,String atom1, String atom2,  double low, double up)
   {    peakID=pkID;
	   	residueNoA = Ano;
	   	residueNoB = Bno;
	   	resA=res1;
	   	resB=res2;
        this.atomA = atom1;
        this.atomB = atom2;
        lower = low;
        upper = up;ratioEnsem=0.0;
    }
   
   /**
    * Instantiates a new noe.
    * 
    * @param pkID the pk id
    * @param Ano the ano
    * @param Bno the bno
    * @param res1 the res1
    * @param res2 the res2
    * @param atom1 the atom1
    * @param atom2 the atom2
    * @param low the low
    * @param up the up
    * @param r the r
    */
   public Noe(int pkID, int Ano,  int Bno, String res1, String res2,String atom1, String atom2,  
		   double low, double up,double r)
   {    peakID=pkID;
	   	residueNoA = Ano;
	   	residueNoB = Bno;
	   	resA=res1;
	   	resB=res2;
        this.atomA = atom1;
        this.atomB = atom2;
        lower = low;
        upper = up;
        ratioEnsem=r;
    }
   
   /**
    * Instantiates a new noe.
    * 
    * @param Ano the ano
    * @param atom1 the atom1
    * @param Bno the bno
    * @param atom2 the atom2
    * @param low the low
    * @param up the up
    */
   public Noe(int Ano,  String atom1, int Bno,  String atom2,  double low, double up)
   { peakID=0;
	   	residueNoA = Ano;
	   	residueNoB = Bno;
	   	resA="";
	   	resB="";
        this.atomA = atom1;
        this.atomB = atom2;
        lower = low;
        upper = up;ratioEnsem=0.0;
    }
   
   /**
    * Instantiates a new noe.
    * 
    * @param pkID the pk id
    * @param vec1 the vec1
    * @param vec2 the vec2
    * @param low the low
    * @param up the up
    */
   public Noe(int pkID, Vector vec1,  Vector vec2,  double low, double up)
   { 
	   peakID=pkID;
   	   vecH_A.addAll(vec1);
   	   vecH_B.addAll(vec2);
   	   if(vec1.size()>0)
   	   {
   		   H1CS h1cs=(H1CS)vecH_A.elementAt(0);
   		   residueNoA=h1cs.getResidueNo();
   		   resA=h1cs.getResidueType();
   		   atomA=h1cs.getAtomName();   	
   		}
   	   if(vec2.size()>0)
   	   {
   		   H1CS h1cs=(H1CS)vecH_B.elementAt(0);
		   residueNoB=h1cs.getResidueNo();
		   resB=h1cs.getResidueType();
		   atomB=h1cs.getAtomName();   	
   	   }
       lower = low;
       upper = up;ratioEnsem=0.0;
    }
   
   /**
    * Instantiates a new noe.
    * 
    * @param pkID the pk id
    * @param vec1 the vec1
    * @param vec2 the vec2
    * @param noe the noe
    */
   public Noe(int pkID, Vector vec1,  Vector vec2, Noe noe)
   { 
	   peakID=pkID;
   	   vecH_A.addAll(vec1);
   	   vecH_B.addAll(vec2);
   	      	   
       lower =noe.getLower();
       upper = noe.getUpper();
       ratioEnsem=noe.getRatioEnsemb();
    }
   
   /**
    * The Class NoeComparator.
    */
   public static class NoeComparator implements Comparator<Object>
   {
       
       /* (non-Javadoc)
        * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
        */
       public int compare(Object o1, Object o2){
           Noe n1 = (Noe)o1;
           Noe n2 = (Noe)o2;
           int d1 = n1.getResidueNoA(); 
           int d2 = n2.getResidueNoA();
          // String atom1A=n1.getAtomA();
          // String atom2A=n2.getAtomA();
           //String atom1B=n1.getAtomB();
          // String atom2B=n2.getAtomB();
           if (d1 < d2)
               return -1;
           else if (d1 > d2)
               return 1;
           /*else if (d1==d2)//added
           {
        	   if(!atom1A.equalsIgnoreCase(atom2A) )
        		   return atom1A.compareTo(atom2A);   
        	   else
        		   return atom1B.compareTo(atom2B);   
           }*/
           else return 0;
       }
   }
   
    /**
     * Gets the residue no a.
     * 
     * @return the residue no a
     */
    public int getResidueNoA() {
	return residueNoA;
    }
    
    /**
     * Gets the residue no b.
     * 
     * @return the residue no b
     */
    public int getResidueNoB() {
	return residueNoB;
    }
    
    /**
     * Gets the residue a.
     * 
     * @return the residue a
     */
    public String getResidueA() {
    	return resA;
        }
    
    /**
     * Gets the residue b.
     * 
     * @return the residue b
     */
    public String getResidueB() {
    	return resB;
        }
    
    /**
     * Gets the atom a.
     * 
     * @return the atom a
     */
    public String getAtomA()   {
	return atomA;
    }
    
    /**
     * Gets the atom b.
     * 
     * @return the atom b
     */
    public String getAtomB()   {
	return atomB;
    }
    
    /**
     * Gets the ratio ensemb.
     * 
     * @return the ratio ensemb
     */
    public double getRatioEnsemb()   {
    	return ratioEnsem;
        }
    
    /**
     * Gets the range.
     * 
     * @return the range
     */
    public double[] getRange(){
	return new double[]{lower, upper};
    }

    /**
     * Gets the lower.
     * 
     * @return the lower
     */
    public double getLower(){
	return lower;
    }
    
    /**
     * Gets the peak id.
     * 
     * @return the peak id
     */
    public int getPeakID(){
    	return peakID;
        }
    
    /**
     * Gets the upper.
     * 
     * @return the upper
     */
    public double getUpper(){
	return upper;
    }
    
    /**
     * Gets the vec h_ a.
     * 
     * @return the vec h_ a
     */
    public Vector  getVecH_A(){
    	return vecH_A;
        }
    
    /**
     * Gets the vec h_ b.
     * 
     * @return the vec h_ b
     */
    public Vector  getVecH_B(){
    	return vecH_B;
        }
    //  set the values	
    /**
     * Sets the dist upper.
     * 
     * @param up the new dist upper
     */
    public void setDistUpper(double up)
    {
    	upper = up;
    }	
    
    /**
     * Sets the dist lower.
     * 
     * @param lw the new dist lower
     */
    public void setDistLower(double lw)
    {
    	lower = lw;
    }	
    
    /**
     * Sets the ratio ensemb.
     * 
     * @param ratio the new ratio ensemb
     */
    public void setRatioEnsemb(double ratio)   {
    	ratioEnsem=ratio;
        } 
    
    /**
     * Sets the res name a.
     * 
     * @param res1 the new res name a
     */
    public void setResNameA(String res1)
    {
    	resA = res1;
    }	
    
    /**
     * Sets the res name b.
     * 
     * @param res2 the new res name b
     */
    public void setResNameB(String res2)
    {
    	resB = res2;
    }	
    
    /**
     * Sets the atom1 name.
     * 
     * @param atom1 the new atom1 name
     */
    public void setAtom1Name(String atom1)
    {
    	atomA = atom1;
    }	
    
    /**
     * Sets the atom2 name.
     * 
     * @param atom2 the new atom2 name
     */
    public void setAtom2Name(String atom2)
    {
    	atomB = atom2;
    }	
    
    /**
     * Sets the peak id.
     * 
     * @param id the new peak id
     */
    public void setPeakID(int id)
    {
    	peakID = id;
    }	
    
    /**
     * Sets the res no a.
     * 
     * @param n1 the new res no a
     */
    public void setResNoA(int n1)
    {
    	residueNoA = n1;
    }	
    
    /**
     * Sets the res no b.
     * 
     * @param n2 the new res no b
     */
    public void setResNoB(int n2)
    {
    	residueNoB = n2;
    }	
    
    /**
     * Sets the vec h_ a.
     * 
     * @param vec1 the new vec h_ a
     */
    public void setVecH_A(Vector vec1)
    {
    	vecH_A =new Vector();
    	vecH_A.addAll(vec1);    	
    }	
    
    /**
     * Sets the vec h_ b.
     * 
     * @param vec2 the new vec h_ b
     */
    public void setVecH_B(Vector vec2)
    {
    	vecH_B =new Vector();
    	vecH_B.addAll(vec2);    	
    }	
    
    //add the proton into the multiple NOE assignment
    /**
     * Adds the h_ a.
     * 
     * @param h1csA the h1cs a
     */
    public void addH_A(H1CS h1csA)
    {
    	//we need to make sure it is not repreated with previous protons
    	String atomNew=h1csA.getAtomName();
    	int resNoNew=h1csA.getResidueNo();
    	boolean isNew=true;
    	for(int i=0; i<vecH_A.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH_A.elementAt(i);
    		String atomName=h1cs.getAtomName();
    		int resNo=h1cs.getResidueNo();
    		if(resNo==resNoNew && atomName.equalsIgnoreCase(atomNew))
    			isNew=false;
    	}
    	if(isNew)
    		vecH_A.add(h1csA);  		
    	  	
    }	
    //add the proton into the multiple NOE assignment
    /**
     * Adds the h_ b.
     * 
     * @param h1csB the h1cs b
     */
    public void addH_B(H1CS h1csB)
    {
    	//we need to make sure it is not repreated with previous protons
    	String atomNew=h1csB.getAtomName();
    	int resNoNew=h1csB.getResidueNo();
    	boolean isNew=true;
    	for(int i=0; i<vecH_B.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH_B.elementAt(i);
    		String atomName=h1cs.getAtomName();
    		int resNo=h1cs.getResidueNo();
    		if(resNo==resNoNew && atomName.equalsIgnoreCase(atomNew))
    			isNew=false;
    	}
    	if(isNew)
    		vecH_B.add(h1csB);  	   	  	
    }	
    //add the proton into the multiple NOE assignment
    /**
     * Adds the protons.
     * 
     * @param noe the noe
     */
    public void addProtons(Noe noe)
    {
    	//we need to make sure it is not repreated with previous protons
    	int resNo1=noe.getResidueNoA();
    	String atom1=noe.getAtomA();
    	String res1=noe.getResidueA();
    	
    	int resNo2=noe.getResidueNoB();
    	String atom2=noe.getAtomB();
    	String res2=noe.getResidueB();
    	
    	H1CS h1cs1=new H1CS(resNo1,res1,atom1);
    	H1CS h1cs2=new H1CS(resNo2,res2,atom2);
    	boolean isNew=true;
    	for(int i=0; i<vecH_A.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH_A.elementAt(i);
    		String atomName=h1cs.getAtomName();
    		int resNo=h1cs.getResidueNo();
    		if(resNo==resNo1 && atomName.equalsIgnoreCase(atom1))
    			isNew=false;
    	}
    	if(isNew)
    		vecH_A.add(h1cs1);  	    	
    	
        isNew=true;
    	for(int i=0; i<vecH_B.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecH_B.elementAt(i);
    		String atomName=h1cs.getAtomName();
    		int resNo=h1cs.getResidueNo();
    		if(resNo==resNo2 && atomName.equalsIgnoreCase(atom2))
    			isNew=false;
    	}
    	if(isNew)
    		vecH_B.add(h1cs2);  	   	  	
    }	
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString()	{
	String desc = residueNoA + atomA + "__"+residueNoB+atomB+"  "+lower+"---"+upper;
	return desc;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    protected Object clone(){
      	try{
	    Object s = super.clone();     // clone the stack
	    return s;                           // return the clone
      	} catch (CloneNotSupportedException e)	{
	    throw new InternalError();
	}
    }

    /**
     * The Class noeComparatorA.
     */
    public static class noeComparatorA implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Noe n1 = (Noe)o1;
	    Noe n2 = (Noe)o2;
	    int d1 = n1.getResidueNoA();
	    int d2 = n2.getResidueNoA();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class noeComparatorB.
     */
    public static class noeComparatorB implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Noe n1 = (Noe)o1;
	    Noe n2 = (Noe)o2;
	    int d1 = n1.getResidueNoB();
	    int d2 = n2.getResidueNoB();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }
    
    /**
     * The Class noeComparatorC.
     */
    public static class noeComparatorC implements Comparator{
    	
	    /* (non-Javadoc)
	     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	     */
	    public int compare(Object o1, Object o2){
    	    Noe n1 = (Noe)o1;
    	    Noe n2 = (Noe)o2;
    	    int d1 = n1.getResidueNoA();
    	    int d2 = n2.getResidueNoA();
    	    
    	    int dA = n1.getResidueNoB();
    	    int dB = n2.getResidueNoB();
    	    if (d1 < d2)
    		return -1;
    	    else if (d1 > d2)
    		return 1;
    	    else if (d1==d2)
    	    {
    	    	if (dA < dB)
    	    		return -1;
    	    	    else if (dA > dB)
    	    		return 1;
    	    	    else return 0;
    	    }
    	    else return 0;
    	}
        }
    
    /**
     * convert all multiple NOE assignments into single assignment format.
     * 
     * @param vecMultiNoe original multiple NOE assignment vector
     * 
     * @return vector of multiple NOE assignments.
     */       
    public Vector ConvertMultiToSingleAsg(Vector vecMultiNoe)
    {
    	Vector vecSingleAsg=new Vector();
    	for(int i=0;i< vecMultiNoe.size();i++)
    	{
    		Noe noe=(Noe)vecMultiNoe.elementAt(i);
    		int pkID=noe.getPeakID();
    		double distUp=noe.getUpper();
    		double distLower=noe.getLower();
    		double ratio=noe.getRatioEnsemb();
    		Vector vecHa=noe.getVecH_A();
    		Vector vecHb=noe.getVecH_B();
    		
    		for (int j=0; j< vecHa.size();j++)
    		{
    			H1CS h1csA =(H1CS)vecHa.elementAt(j);
    			int resNoA=h1csA.getResidueNo();
    			String resA=h1csA.getResidueType();
    			String atomA=h1csA.getAtomName();
    			
    			for(int k=0;k<vecHb.size();k++)
    			{
    				H1CS h1csB=(H1CS) vecHb.elementAt(k);
    				int resNoB=h1csB.getResidueNo();
    				String resB=h1csB.getResidueType();
    				String atomB=h1csB.getAtomName();
    				vecSingleAsg.add(new Noe(pkID,resNoA,resNoB,resA,resB,atomA,atomB,distLower,distUp,ratio) );
    				
    			}//for(int k=0;k<vecHb.size();k++)
    		}//for (int j=0; j< vecHa.size();j++)  		
    		
    	}//for(int i=0;i< vecMultiNoe.size();i++)
    	
    	return vecSingleAsg;
    }
    
    /**
     * convert all NOE assignments into multiple assignment format.
     * 
     * @param vecSingleNoe original signle NOE assignment vector
     * 
     * @return vector of multiple NOE assignments.
     */       
    public Vector<Noe> ConvertSingleToMultiAsg(Vector<Noe> vecSingleNoe)
    {
    	Vector vecMulti=new Vector();
    	int maxPkID=-999;
    	for (int i=0;i<vecSingleNoe.size();i++)
    	{
    		Noe noe=(Noe)vecSingleNoe.elementAt(i);
    		int pkID=noe.getPeakID();
    		if (pkID>=maxPkID)
    			maxPkID=pkID;	
    	}
    	
    	//we assume that pk ID starts from 0
    	for (int i=-maxPkID;i<=maxPkID;i++)
    	{    		
    		Noe multi_noe=new Noe();
    		boolean isSet=false;
    		for (int j=0;j<vecSingleNoe.size();j++)
    		{
    			Noe noe2=(Noe)vecSingleNoe.elementAt(j);
    			int pkID2=noe2.getPeakID();
    			double upDist=noe2.getUpper();    			
    			double lowDist=noe2.getLower();
    			double ratio=noe2.getRatioEnsemb();
    			String resA=noe2.getResidueA();
    			String resB=noe2.getResidueB();
    			String atomA=noe2.getAtomA();
    			String atomB=noe2.getAtomB();
    			int resNoA=noe2.getResidueNoA();
    			int resNoB=noe2.getResidueNoB();
    			if(pkID2==i)
    			{
    				isSet=true;    									  		
		    		multi_noe.setPeakID(pkID2);
		    		multi_noe.setRatioEnsemb(ratio);
		    		multi_noe.setDistUpper(upDist);
		    		multi_noe.setDistLower(lowDist);   				
		    		multi_noe.addProtons(noe2);		
		    		multi_noe.setResNameA(resA);
		    		multi_noe.setResNameB(resB);
		    		multi_noe.setAtom1Name(atomA);
		    		multi_noe.setAtom2Name(atomB);
		    		multi_noe.setResNoA(resNoA);
		    		multi_noe.setResNoB(resNoB);		    		
    			}    				
    		}//for (int j=0;j<vecSingleNoe.size();j++)
    		if(isSet)
    			vecMulti.add(multi_noe);
    	}//for (int i=0;i<=maxPkID;i++)
    	Collections.sort(vecMulti, new noeComparatorC());
    	return vecMulti;    	
    }
    
    /**
     * output the noe assignment in multple format.
     * 
     * @param vecMultiNoe multiple NOE assignment
     * @param strOutName output path name
     */
    public void OutputMultiNOEAsg(Vector vecMultiNoe, String strOutName)
    {
    	String xplorNoeStr="";
    	double[] disUpper=new double[1];
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(strOutName)));
    		out.println("!REMARK: Total number of cross peaks is "+vecMultiNoe.size());
    		out.println("");
    		out.println("");
    		
	       	for (int i=0;i<vecMultiNoe.size();i++)
	    	{
	    		Noe noe=(Noe)vecMultiNoe.elementAt(i);
	    		int pkID=noe.getPeakID();
	    		double upLimit=noe.getUpper();
	    		double lowerLimit=noe.getLower();
	    		out.println("! peak ID : " + pkID);
	    		Vector vecHA=noe.getVecH_A();
	    		Vector vecHB=noe.getVecH_B();
	    		String str = "assign (";
	    		boolean isFirst=true;
	    		for (int k=0;k<vecHA.size();k++)
	    		{
	    			H1CS h1cs=(H1CS)vecHA.elementAt(k);
	    			int no1=h1cs.getResidueNo();
	    			String atom1=h1cs.getAtomName();
	    			if(isFirst)
	    			{
	    				str=str+"( resid "+String.valueOf(no1)+" and name "+atom1+" )";
	    				if ( k == (vecHA.size()-1) )
	    					str=str+")";
	    				else
	    					str=str+" or";
	    				out.println(str);
	    				isFirst=false;
	    			}
	    			else
	    			{  
	    				str = "         ( resid "+String.valueOf(no1)+" and name "+atom1+" )";
	    				if ( k == (vecHA.size()-1) )
	    					str=str+")";
	    				else
	    					str=str+" or";
	    				out.println(str);
	    			}
	    		}//for (int k=0;k<vecHA.size();k++)
	    		
	    		isFirst=true;	    		
	    		str=  "                                        (";
	    		for(int k=0;k<vecHB.size();k++)
	    		{
	    			H1CS h1cs=(H1CS)vecHB.elementAt(k);
	    			int no1=h1cs.getResidueNo();
	    			String atom1=h1cs.getAtomName();
	    			if(isFirst)
	    			{	    				
	    				str=str+"( resid "+String.valueOf(no1)+" and name "+atom1+" )";
	    				if ( k == (vecHB.size()-1) )
	    					str=str+")      "+String.valueOf(upLimit) +"   "+String.valueOf(upLimit-lowerLimit)+"  "+String.valueOf(0.0); 
	    				else 
	    					str=str+" or";
	    				out.println(str);
	    				isFirst=false;
	    			}
	    			else
	    			{  	    				
	    				str = "                                          ( resid "+String.valueOf(no1)+" and name "+atom1+" )";
	    				if ( k == (vecHB.size()-1) )
	    					str=str+")      " +String.valueOf(upLimit) +"   "+String.valueOf(upLimit-lowerLimit)+"  "+String.valueOf(0.0); 
	    				else 
	    					str=str+" or";
	    				out.println(str);
	    			}	    			
	    		}//for(int k=0;k<vecHB.size();k++) 		
	    		
	    	}//	for (int i=0;i<vecMultiNoe.size();i++)
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ strOutName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + strOutName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		
    }
    		
    /**
     * Compare assignment results with manual assignments.
     * 
     * @param vecPdbRef referernce pdb structure
     * @param number number[0]: number of correct assignments, number[1]: number of wrong assignments
     * @param vecNoe the vec noe
     * @param noeLimit the noe limit
     * @param methylCorrection the methyl correction
     */
    public void CompareAsgToRefStr(Vector vecNoe,Vector vecPdbRef, double noeLimit, double methylCorrection, double[] number)
    { 	      	   
	    	int i,j;
	    	Noe noe=new Noe();
	    	int firstResNo,secondResNo;
	    	String firstResName, secondResName;
	    	String firstAtomName, secondAtomName; 
	    	Peak pk=new Peak();
	    	double dist_from_inten=0.0;
    	    	
	    	boolean isFound=false;
	    	double [] noeDist = new double[2];
	    	Vector vecRefinedNoe=new Vector();
	    	int numCorrect=0;
	    	int numWrong=0;
	    	int numCorrectLong=0;
	    	int numWrongLong=0;
	    	
	    	for(i=0;i<vecNoe.size();i++)
	    	{
	    		noe=(Noe)vecNoe.elementAt(i);
	    		firstResNo=noe.getResidueNoA();
	    		secondResNo=noe.getResidueNoB();
	    		firstResName=noe.getResidueA();
	    		secondResName=noe.getResidueB();
	    		firstAtomName=noe.getAtomA();///
	    		secondAtomName=noe.getAtomB();
	    		dist_from_inten=noe.getUpper();
	    		isFound=pk.checkNoeNew(vecPdbRef,firstResNo,firstAtomName,secondResNo,secondAtomName,
	    				dist_from_inten,2.5,noeDist);	//use methyl correction 1.5 for paper
	    			    		
	    		if(isFound)
	    		{
	    			numCorrect=numCorrect+1;
	    			if(Math.abs(firstResNo-secondResNo)>4)
	    				numCorrectLong=numCorrectLong+1;	
	    		}
	    		else
	    		{
	    			numWrong=numWrong+1;
	    			if(Math.abs(firstResNo-secondResNo)>4)
	    				numWrongLong=numWrongLong+1;	    			
	    		}
	    			
	    		
	    	}
	    	number[0]=  numCorrect;
	    	number[1]=numWrong;
	    	number[2]=numCorrectLong;
	    	number[3]=numWrongLong;
	
    }
    
    /**
     * correct the NOE upper distance for an methyl NOE
     * we assume that the pseudo methyl atoms of all old NOEs are labeled with "#" or "##".
     * 
     * @param vecNoeOld original NOEs before correction
     * @param metCorrection metyl correction
     * 
     * @return new NOE table after correction
     * Note: created May 29,08.
     * 
     * @throws JampackException the jampack exception
     */
    
    public Vector NoeMethylCorrection(Vector vecNoeOld, double metCorrection)throws JampackException
    {     	
    	Vector vecNoeNew=new Vector();
    	for(int i=0;i<vecNoeOld.size();i++)
    	{
    		Noe noe=(Noe)vecNoeOld.elementAt(i);
    		int firstResNo=noe.getResidueNoA();
    		int secondResNo=noe.getResidueNoB();
    		String firstResName=noe.getResidueA();
    		String secondResName=noe.getResidueB();
    		String firstAtomName=noe.getAtomA();///
    		String secondAtomName=noe.getAtomB();
    		double dist_from_inten=noe.getUpper();
    		
    		if(firstAtomName.substring(firstAtomName.length()-1, firstAtomName.length()).equalsIgnoreCase("#") ||
    				secondAtomName.substring(secondAtomName.length()-1, secondAtomName.length()).equalsIgnoreCase("#")	)
    			noe.setDistUpper(Math.min(dist_from_inten+metCorrection,6.0));
    		vecNoeNew.add(noe);   		
    	}
    	return vecNoeNew;
    	
    }
    
    /**
     * check the NOE table by an ensemble of structures, and output the statistical information.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * Note: created May 29,08.
     * 
     * @throws JampackException the jampack exception
     */
    
    public void doCheckNoeByEnsemble(String src, String strOut, String strInput)throws JampackException
    {
    	Assign asg = new Assign();
    	int i, j;    
    	boolean isDebug=Const.isDebug;
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
     	String userDir = System.getProperty("user.dir");////
		
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////  
    	String strSeq="",strInLocation="",strPdbNameScheme="",strNoeFormat="",strIsCheckLongNeighbor="",
    		strNoeNameScheme="",strInNoeTable="",strOutNoeTable="",strIsOutVioNOEs="",strIsCheckLongAA="",
    		strIsMultiAsg="",strIsOutMultiAsg="";
    	double metCor=0.0,noeCutOff=0.0,noeLimit=0.0;
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("INPUTPDBLOCATION"))
    			strInLocation  =  (String)paraMap.get("INPUTPDBLOCATION");
    		if(paraMap.containsKey("PDBNAMESCHEME"))
    			strPdbNameScheme  =  (String)paraMap.get("PDBNAMESCHEME");	
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");    		
    		if(paraMap.containsKey("NOEATOMNAMINGSCHEME"))
    			strNoeNameScheme  =  (String)paraMap.get("NOEATOMNAMINGSCHEME");    		
    		if(paraMap.containsKey("INPUTNOETABLE"))
    			strInNoeTable  =  (String)paraMap.get("INPUTNOETABLE");
    		if(paraMap.containsKey("METHYL-CORRECTION"))
    			metCor  =  Double.parseDouble((String)paraMap.get("METHYL-CORRECTION"));    		
    		if(paraMap.containsKey("NOECUTOFF"))
    			noeCutOff  =  Double.parseDouble((String)paraMap.get("NOECUTOFF"));
    		if(paraMap.containsKey("OUTNOETABLE"))
    			strOutNoeTable  =  (String)paraMap.get("OUTNOETABLE");   		
    		if(paraMap.containsKey("NOELIMIT"))
    			noeLimit  =  Double.parseDouble((String)paraMap.get("NOELIMIT"));
    		if(paraMap.containsKey("ISOUTVIONOES"))
    			strIsOutVioNOEs  =  (String)paraMap.get("ISOUTVIONOES");    
    		if(paraMap.containsKey("ISCHECKLONGNEIGHBOR"))
    			strIsCheckLongNeighbor =  (String)paraMap.get("ISCHECKLONGNEIGHBOR");    		
    		if(paraMap.containsKey("ISCHECKLONGAA"))
    			strIsCheckLongAA =  (String)paraMap.get("ISCHECKLONGAA"); 	
    		if(paraMap.containsKey("ISMULTIASSIGNMENT"))
    			strIsMultiAsg =  (String)paraMap.get("ISMULTIASSIGNMENT"); 	
    		if(paraMap.containsKey("ISOUTMULTIASGFORMAT"))
    			strIsOutMultiAsg =  (String)paraMap.get("ISOUTMULTIASGFORMAT");
    	}//for (i=0;i<paraVec.size();i++)
    	
    	// -------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    
    	
     	//read the NOE table:
    	String manualAsgFile = userDir+ strInNoeTable;  //src+strInNoeTable;
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	
    	//for the NOE format, we convert all formats into XPLOR with PDB-NEW naming scheme, i.e. HD1#, HG##
    	// read from cyana (namely upl) format:
    	
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,strNoeNameScheme);//need to double check about atom name
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//    	read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		//we assume that all noes use PDB-NEW scheme
    		if(strIsMultiAsg.equalsIgnoreCase("1"))
    		{
    			vecManAsg=noe_temp.noeReaderMultiAsg(manualAsgFile,vecSeq);
    			Vector vecTemp=noe_temp.ConvertMultiToSingleAsg(vecManAsg);
    			vecManAsg=new Vector();
    			vecManAsg.addAll(vecTemp);
    		}
    		else
    			vecManAsg=noe_temp.noeReader(manualAsgFile,0.0,vecSeq);  	//xplor format   
    			
    			
    		if(strNoeNameScheme.equalsIgnoreCase("BMRB-NEW"))
    			vecManAsg=noe_temp.ConvertNoeFromBMRMToPDB(vecManAsg);
    		vecManAsgNew.addAll(vecManAsg);
    		   	
    	}//need to check this (in packing):    	
    	
    	///////////////////////////////////////////////
    	//remove the diagonal NOEs
    	Vector vecTempDia=new Vector();
    	for(i=0;i<vecManAsgNew.size();i++)
    	{
    		Noe noe=(Noe)vecManAsgNew.elementAt(i);
    		int no1=noe.getResidueNoA();
    		int no2=noe.getResidueNoB();
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		if(no1==no2 && atom1.equalsIgnoreCase(atom2))
    			continue;
    		vecTempDia.add(noe);    		
    	}
    	vecManAsgNew=new Vector();
    	vecManAsgNew.addAll(vecTempDia);
    	////////////////////////////////////////////////
    	if(strIsCheckLongAA.equalsIgnoreCase("1"))
    	{
    		Peak pk=new Peak();
    		Vector vecTemp=new Vector();
    		for(i=0;i<vecManAsgNew.size();i++)
        	{
        		Noe noe=(Noe)vecManAsgNew.elementAt(i);
        		int firstResNo=noe.getResidueNoA();
        		int secondResNo=noe.getResidueNoB();
        		String firstResName=noe.getResidueA();
        		String secondResName=noe.getResidueB();
        		String atomA=noe.getAtomA();
        		String atomB=noe.getAtomB();        		
        		String subAtomA=atomA;
        		String subAtomB=atomB;
        		if (atomA.length()>=2)
        			subAtomA=atomA.substring(0,2);
        		if(atomB.length()>=2) 
        			subAtomB=atomB.substring(0,2);
        			
        		if(Math.abs(firstResNo-secondResNo)<=4 )
        		{
        			vecTemp.add(noe);
        			continue;
        		}
        		        		
        		boolean isKeep=true;
        		if ( pk.isCharged(firstResName) && pk.isHydrophobic(secondResName) )
        			isKeep=false;
        		if ( pk.isCharged(secondResName) && pk.isHydrophobic(firstResName) )
        			isKeep=false;
        		if( (subAtomB.equalsIgnoreCase("HN") || subAtomB.equalsIgnoreCase("H")||subAtomB.equalsIgnoreCase("HA")||subAtomB.equalsIgnoreCase("HB")) &&
        				(subAtomA.equalsIgnoreCase("HN")||subAtomA.equalsIgnoreCase("H")||subAtomA.equalsIgnoreCase("HA")|| subAtomA.equalsIgnoreCase("HB")   ))
        			isKeep=true;
        		
        		if(isKeep)
        			vecTemp.add(noe);
        	}//for(i=0;i<vecManAsgNew.size();i++)
    		
    		vecManAsgNew=new Vector();
    		vecManAsgNew.addAll(vecTemp);
    	}//if(strIsCheckLongAA.equalsIgnoreCase("1"))
    	//////////////////////////////////////////////////
    	
    	//////////////////////////
    	//temp for debugging:
    //	Noe ne=new Noe();
    	//Vector vecNoeNewCyana=ne.ConvertXplorAsgToUplFormat(vecManAsgNew, vecSeq, "BMRB-NEW");
    	//ne.PrintNOE(vecNoeNewCyana);
    	
    	/////////////////////////
    	
    	
    	
    	
      
		String strInputFile=userDir+strInLocation;
		File myDir = new File(strInputFile);
    	String[] contents = myDir.list();//select all structures
    	if (contents == null) 
    		System.out.println(myDir.getName() + " is not a directory");
   
    	//check each NOE by each structure in the ensemble:
    	Peak pk=new Peak();
    	double [] noeDist = new double[2];
    	boolean isFound=false;
    	for(i=0;i<vecManAsgNew.size();i++)
    	{
    		Noe noe=(Noe)vecManAsgNew.elementAt(i);
    		int firstResNo=noe.getResidueNoA();
    		int secondResNo=noe.getResidueNoB();
    		String firstResName=noe.getResidueA();
    		String secondResName=noe.getResidueB();
    		String firstAtomName=noe.getAtomA();///
    		String secondAtomName=noe.getAtomB();
    		double dist_from_inten=noe.getUpper();
    		double noeUpperPrune=0.0;
    		
    		//read each PDB:
    		int counterPdb=0;
       	 	for (int t=0; t<contents.length;t++) 
        	{    
        		String filename = contents[t]; 
    	    			
    	    	String strEnsemPdb="";
    	    	strEnsemPdb=strInputFile+filename;	    
    	    	
    	    	Vector vecEnsemble=pp.readPdbAndNoeHarmFromEnsemble(strEnsemPdb);
    	    	Vector pdbVecSSE = new Vector();
    	    	
    	    	if(vecEnsemble.size()>0)
    	    	{
    	    		for(int w=0;w<vecEnsemble.size();w++)
    		    	{	   
    	    			Vector vecVdw=new Vector();
    		    		vecVdw=new Vector();
    		    		vdw Vdw=(vdw)vecEnsemble.elementAt(w);
    		    		Vector vecPdb=Vdw.getPDB();
    		    		if (vecPdb.size()<1)
    		    			continue;
    		    		
	    	    		Vector pdbVecSSE_temp2 = pp.residueNameUpdateNoStr(vecSeq, vecPdb); 
	    	    		if(strPdbNameScheme.equalsIgnoreCase("PDB-OLD"))
	    	    			pdbVecSSE=pp.nameConvertOrder(pdbVecSSE_temp2); 
	    	    		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-NEW"))
	    	    			pdbVecSSE=pp.nameConvert2PDBNew(pdbVecSSE_temp2);//need to double check
	    	    		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-OLD"))
	    	    		{
	    	    			Vector pdbVecSSE22=pp.nameConvertOrder(pdbVecSSE_temp2); 
	    	    			pdbVecSSE=pp.nameConvert2PDBNew(pdbVecSSE22);//need to double check
	    	    		}
	    	    		else
	    	    			pdbVecSSE.addAll(pdbVecSSE_temp2);  
	    	    		counterPdb++;
	    	    	
	    	    		if(noeLimit<=0.0)
	    	    			noeUpperPrune=dist_from_inten;
	    	    		else
	    	    			noeUpperPrune=noeLimit;
	    	    		noeUpperPrune=noeUpperPrune+0.5;//add 0.5A extra disance
	    	    		if(noeUpperPrune>6.0)
	    	    			noeUpperPrune=6.0;
	    	    		isFound=pk.checkNoeNew(pdbVecSSE,firstResNo,firstAtomName,secondResNo,secondAtomName,
	    	    				noeUpperPrune,metCor,noeDist);	
	    	    		
	    	    		double dbRatio=noe.getRatioEnsemb()+1.0;
	    	    		if(isFound)
	    	    			noe.setRatioEnsemb(dbRatio);
    		    	}//for(int w=0;i<vecEnsemble.size();w++)
    	    	} //if(vecEnsemble.size()>0)           	
        	}//for (int t=0; t<contents.length;t++) 
        	if(counterPdb<1)
        	{
        		System.out.println("No input PDB is found for checking NOEs....");
        		return;
        	}   
        	double dbRatio=noe.getRatioEnsemb();
        	noe.setRatioEnsemb(dbRatio/counterPdb);   		
    	}//for(i=0;i<vecManAsgNew.size();i++)
    	
    	//output those NOEs with high ratios:
    	Vector vecNoeNew=new Vector();
    	Vector vecDeltNoe=new Vector();
    	for(i=0;i<vecManAsgNew.size();i++)
    	{
    		Noe noe=(Noe)vecManAsgNew.elementAt(i);
    		
    		double dbRatio=noe.getRatioEnsemb();
    		if (dbRatio>=noeCutOff)
    			vecNoeNew.add(noe);   
    		else
    			vecDeltNoe.add(noe);
    			
    	}//for(i=0;i<vecManAsgNew.size();i++)    	
    	
    	///////////////////////////////////
    	//whether check the neighbor information of long-range NOEs.
    	if(strIsCheckLongNeighbor.equalsIgnoreCase("1"))
    	{
    		Vector  vecRefineLongNOE=new Vector();
    		vecRefineLongNOE=pk.CheckLongRangNOE(vecNoeNew,3);
    		vecNoeNew=new Vector();
    		vecNoeNew.addAll(vecRefineLongNOE);
    	}
    	///////////////////////////////////
    	
    	//////////////////////////////////////
    	//sort the NOE table
    	if(!strIsOutMultiAsg.equalsIgnoreCase("1"))
    	{
	    	for(i=0;i<vecNoeNew.size();i++)
	    	{
	    		Noe noe=(Noe)vecNoeNew.elementAt(i);
	    		int firstResNo=noe.getResidueNoA();
	    		int secondResNo=noe.getResidueNoB();
	    		String firstResName=noe.getResidueA();
	    		String secondResName=noe.getResidueB();
	    		String firstAtomName=noe.getAtomA();///
	    		String secondAtomName=noe.getAtomB();    
	    		
	    		
	    		
	    		if(secondResNo<firstResNo)
	    		{
	    			noe.setResNoA(secondResNo);
	    			noe.setResNoB(firstResNo);
	    			noe.setAtom1Name(secondAtomName);
	    			noe.setAtom2Name(firstAtomName);
	    			noe.setResNameA(secondResName);
	    			noe.setResNameB(firstResName);
	    		}
	    	}//for(i=0;i<vecManAsgNew.size();i++)
    	
	    	Collections.sort(vecNoeNew, new noeComparatorC());
    	}
    	//////////////////////////////////////////////////////   	
    	///////////////////////////////////
    	//output the final Pdb
    	String fileName=userDir+strOutNoeTable; ;//strOut+strOutNoeTable;    	
    	
    	if(strIsOutMultiAsg.equalsIgnoreCase("1"))
    	{
    		Noe noeTemp=new Noe();    		
    		Vector vecMultiAsg=noeTemp.ConvertSingleToMultiAsg(vecNoeNew);
    		//String fileName=strOut+stroutName;  
    		noeTemp.OutputMultiNOEAsg(vecMultiAsg,fileName); 
    	}
    	else
    	{
    		String xplorNoeStr="";
        	double[] disUpper=new double[1];
        	try{
        		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
        		out.println("!REMARK: Total number of original NOEs is "+vecManAsgNew.size());
        		out.println("!REMARK: Total number of NOEs is "+vecNoeNew.size());
        		out.println("");
        		out.println("");
        		
    	       	for (i=0;i<vecNoeNew.size();i++)
    	    	{
    	    		Noe noe=(Noe)vecNoeNew.elementAt(i);
    	    		int resNo1=noe.getResidueNoA();
    	    		int resNo2=noe.getResidueNoB();
    	    		String res1=noe.getResidueA();
    	    		String res2=noe.getResidueB();
    	    		String atom1=noe.getAtomA();
    	    		String atom2=noe.getAtomB();
    	    		double distUpper=noe.getUpper();
    	    		double dbRatio=noe.getRatioEnsemb();
    	    		if(isDebug)
    	    			out.println("!ratio: "+dbRatio);
    	    		//xplor format:	    		
    				xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
    				if(isDebug)
    					System.out.println(xplorNoeStr); 
    				out.println(xplorNoeStr);
    	    	}//for (i=0;i<vecRefinedNoes.size();i++)
    	       	out.println("!-------------------------!");
    	       	out.println("! violated NOEs   !!!!!");
    	       	if(isDebug)
    	       		System.out.println("!--------! violated NOEs   !!!!!-----------------!");
    	       	// output those NOEs with ratios less than cutoff:
    	    	if(strIsOutVioNOEs.equalsIgnoreCase("1"))
    	    	{
    	    		for(i=0;i<vecDeltNoe.size();i++)
    	        	{
    	        		Noe noe=(Noe)vecDeltNoe.elementAt(i);
    	        		int resNo1=noe.getResidueNoA();
    		    		int resNo2=noe.getResidueNoB();
    		    		String res1=noe.getResidueA();
    		    		String res2=noe.getResidueB();
    		    		String atom1=noe.getAtomA();
    		    		String atom2=noe.getAtomB();
    		    		double distUpper=noe.getUpper();
    		    		double dbRatio=noe.getRatioEnsemb();
    		    		if(isDebug)
    		    			out.println("!ratio: "+dbRatio);
    		    		//xplor format:	    		
    					xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
    					if(isDebug)
    						System.out.println("!!!  "+xplorNoeStr); 
    					out.println("!!!  "+xplorNoeStr);
    	        	}//for(i=0;i<vecManAsgNew.size();i++)  
    	    	}//if(strIsOutVioNOEs.equalsIgnoreCase("1"))
    	    	
    	    	
    	    	out.close();
    	    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
        	}catch (FileNotFoundException e)
    		{
    			System.out.println("File not found: " + fileName);
    		}catch (IOException e)
    		{
    		   System.out.println("IOException: the stack trace is:");
    		   e.printStackTrace();
    		}
    		
    	}//else
    	
    	System.out.println("!REMARK: Total number of original NOEs is "+vecManAsgNew.size());
		System.out.println("!REMARK: Total number of NOEs is "+vecNoeNew.size());
		
	   long endTime = System.currentTimeMillis() ;
	   double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
	   if(Const.isDebug)
		   System.out.println("The total time for CheckNoeByEnsemble is:  "+ totalTime +" minutes");  	
    }
    
    
    /**
     * output the NOE statistics.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * Note: created May 29,08.
     * 
     * @throws JampackException the jampack exception
     */
    
    public void doNoeStatistics(String src, String strOut, String strInput)throws JampackException
    {
    	Assign asg = new Assign();
    	int i, j;    
    	boolean isDebug=Const.isDebug;
    	Pdb  pp = new Pdb();   
    	long startTime = System.currentTimeMillis();
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////  
    	String strSeq="",strInLocation="",strPdbNameScheme="",strNoeFormat="",strIsCheckLongNeighbor="",
    		strNoeNameScheme="",strInNoeTable="",strOutNoeTable="",strIsOutVioNOEs="",strIsCheckLongAA="",
    		strIsMultiAsg="",strIsOutMultiAsg="";
    	double metCor=0.0,noeCutOff=0.0,noeLimit=0.0;
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("INPUTPDBLOCATION"))
    			strInLocation  =  (String)paraMap.get("INPUTPDBLOCATION");
    	
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");    		
    		if(paraMap.containsKey("NOEATOMNAMINGSCHEME"))
    			strNoeNameScheme  =  (String)paraMap.get("NOEATOMNAMINGSCHEME");    		
    		if(paraMap.containsKey("INPUTNOETABLE"))
    			strInNoeTable  =  (String)paraMap.get("INPUTNOETABLE");
    	
    	
    		if(paraMap.containsKey("ISMULTIASSIGNMENT"))
    			strIsMultiAsg =  (String)paraMap.get("ISMULTIASSIGNMENT"); 	
    		
    	}//for (i=0;i<paraVec.size();i++)
    	
    	// -------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    
    	
     	//read the NOE table:
    	String manualAsgFile = src+ strInNoeTable;//"noe-H123.txt";//strManualAsg; //"resonance.prot";  
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();    	
    	
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,strNoeNameScheme);//need to double check about atom name
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//    	read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		//we assume that all noes use PDB-NEW scheme
    		if(strIsMultiAsg.equalsIgnoreCase("1"))
    		{
    			vecManAsg=noe_temp.noeReaderMultiAsg(manualAsgFile,vecSeq);
    			Vector vecTemp=noe_temp.ConvertMultiToSingleAsg(vecManAsg);
    			vecManAsg=new Vector();
    			vecManAsg.addAll(vecTemp);
    		}
    		else
    			vecManAsg=noe_temp.noeReader(manualAsgFile,0.0,vecSeq);  	//xplor format   
    		
    		if(strNoeNameScheme.equalsIgnoreCase("BMRB-NEW"))
    			vecManAsg=noe_temp.ConvertNoeFromBMRMToPDB(vecManAsg);
    		
    		vecManAsgNew.addAll(vecManAsg);    	
    	}//need to check this (in packing):    	
 
    	int intraNum=0,seqNum=0, medNum=0,longNum=0;      	
    	for(i=0;i<vecManAsgNew.size();i++)
    	{
    		Noe noe=(Noe)vecManAsgNew.elementAt(i);
    		int resNoA=noe.getResidueNoA();
    		int resNoB=noe.getResidueNoB();
    		if(resNoA==resNoB)
    			intraNum++;
    		else if(Math.abs(resNoA-resNoB) == 1)
    			seqNum++;
    		else if(Math.abs(resNoA-resNoB) >=5)
    			longNum++;
    		else
    			medNum++;    		
    	}//for(i=0;i<vecManAsgNew.size();i++)
    	
    	
    	System.out.println("!REMARK: Total number of NOEs is "+vecManAsgNew.size());
		System.out.println("!REMARK: Number of Intra-residue NOEs is "+intraNum);
		System.out.println("!REMARK: Number of Sequential NOEs is "+seqNum);		
		System.out.println("!REMARK: Number of Medium-range NOEs is "+medNum);
		System.out.println("!REMARK: Number of Long-range NOEs is "+longNum);
		
	   long endTime = System.currentTimeMillis() ;
	   double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
	   if(Const.isDebug)
		   System.out.println("The total time for CheckNoeByEnsemble is:  "+ totalTime +" minutes");  	
    }
       
    /**
     * Convert naming scheme From PDBNew To PDBNew,
     * just do the changes of pseudo protons of methyl groups (eg.HB#)
     * 
     * @param vecNoe the vec noe
     * 
     * @return the vector
     */
    public Vector<Noe> NameConvertFromPDBNewToPDBNew(final Vector<Noe> vecNoe)
    {
		Cartesian atom = null;
		String atomName="", resid;
		String atomName2="";
		Vector atoms = new Vector();
		Vector atomVec = new Vector();
		
		Pdb pp=new Pdb();
		String res1, res2;
		String atom1, atom2;
	
		Noe noe = new Noe();
		Vector vecNoeOld=new Vector();
		vecNoeOld.addAll(vecNoe);
		
		Vector vecNoeNew=new Vector();
		
		for (int i=0; i<vecNoeOld.size(); i++)
		{
			noe = (Noe)vecNoeOld.elementAt(i);
			res1=noe.getResidueA();
			res2=noe.getResidueB();
			atom1=noe.getAtomA();
			atom2=noe.getAtomB();
			//String atom1New=pp.NameFromPDBToPDB(res1,atom1);
			//String atom2New=pp.NameFromPDBToPDB(res2,atom2);
			String atom1New=UpdateNOEAtomName(res1,atom1);
			String atom2New=UpdateNOEAtomName(res2,atom2);
			
			
			noe.setAtom1Name(atom1New);
			noe.setAtom2Name(atom2New);
			vecNoeNew.add(noe);
			
		}//for (int i=0; i<vecNoeOld.size(); i++)
	
		return vecNoeNew;
    }
      
    /**
     * Convert naming scheme From BMRBNew To PDBNew,
     * including pseudo protons of methyl groups.
     * 
     * @param vecNoe the vec noe
     * 
     * @return the vector
     */
    public Vector<Noe> NameConvertFromBMRBNewToPDBNew(final Vector<Noe> vecNoe)
    {
		Cartesian atom = null;
		String atomName="", resid;
		String atomName2="";
		Vector atoms = new Vector();
		Vector atomVec = new Vector();
		
		Pdb pp=new Pdb();
		String res1, res2;
		String atom1, atom2;
	
		Noe noe = new Noe();
		Vector vecNoeOld=new Vector();
		vecNoeOld.addAll(vecNoe);
		
		Vector vecNoeNew=new Vector();
		
		for (int i=0; i<vecNoeOld.size(); i++)
		{
			noe = (Noe)vecNoeOld.elementAt(i);
			res1=noe.getResidueA();
			res2=noe.getResidueB();
			atom1=noe.getAtomA();
			atom2=noe.getAtomB();
			String atom1NewT=pp.NameFromBMRBToPDB(res1,atom1);
			String atom1New=pp.NameFromPDBToPDB(res1, atom1NewT);
			
			String atom2NewT=pp.NameFromBMRBToPDB(res2,atom2);
			String atom2New=pp.NameFromPDBToPDB(res2, atom2NewT);
			
			noe.setAtom1Name(atom1New);
			noe.setAtom2Name(atom2New);
			vecNoeNew.add(noe);
			
		}//for (int i=0; i<vecNoeOld.size(); i++)
	
		return vecNoeNew;
    }
   
    /**
     * Compare assignment results with manual assignments.
     * 
     * @param vecAsg final noe resonance assignment
     * @param vecManualAsg manual noe resonance assignment
     * @param number number[0]: number of correct assignments, number[1]: number of wrong assignments
     */
    public void CompareAsgToManualAsg(Vector vecAsg,Vector vecManualAsg, double[] number)
    {
    	int i,j;
    	int resNo1,resNo2;
    	String res1,res2;
    	String atom1,atom2;
    	String atom_temp1,atom_temp2;
    	
    	int manResNo1,manResNo2;
    	String manRes1,manRes2;
    	String manAtom1,manAtom2;   	

    	Pdb pp=new Pdb();
    	Noe noe=new Noe();
    	Noe noeManual=new Noe();
    	
    	int sumCorrect=0;
    	int sumWrong=0;
    	int sumCorrectLong=0;
    	int sumWrongLong=0;
    	boolean isFound=false;
    	for (i=0;i<vecAsg.size();i++)
    	{
    		noe=(Noe)vecAsg.elementAt(i);
    		resNo1=noe.getResidueNoA();
    		resNo2=noe.getResidueNoB();
    		res1=noe.getResidueA();
    		res2=noe.getResidueB();
    		atom_temp1=noe.getAtomA();
    		if (atom_temp1.equalsIgnoreCase("H"))
    			atom_temp1="HN";
    		atom_temp2=noe.getAtomB();
    		if(atom_temp2.equalsIgnoreCase("H"))
    			atom_temp2="HN";
    		isFound=false;
    		atom1=pp.NameFromBMRBToPDB(res1,atom_temp1);
    		atom2=pp.NameFromBMRBToPDB(res2,atom_temp2 );   		
    		double distUpper=noe.getUpper();
    		String subAtom1=atom1.substring(0,2);
    		String subAtom2=atom2.substring(0,2);
    		
    		
    		for(j=0;j<vecManualAsg.size();j++)
    		{
    			noeManual=(Noe)vecManualAsg.elementAt(j);
    			manResNo1=noeManual.getResidueNoA();
    			manResNo2=noeManual.getResidueNoB();
    			manRes1=noeManual.getResidueA();
    			manRes2=noeManual.getResidueB();
    			manAtom1=noeManual.getAtomA();
    			manAtom2=noeManual.getAtomB();
    			if(manAtom1.equalsIgnoreCase("H"))
    				manAtom1="HN";
    			if(manAtom2.equalsIgnoreCase("H"))
    				manAtom2="HN";
    			
    			String subManAtom1="",subManAtom2="";
    			if(manAtom1.length()>=2)
    				subManAtom1=manAtom1.substring(0,2);
    			else
    				subManAtom1=manAtom1;
    			if(manAtom2.length()>=2)
    				subManAtom2=manAtom2.substring(0,2);
    			else 
    				subManAtom2=manAtom2;
    			
    			if (resNo1==manResNo1 && resNo2==manResNo2 && atom1.equalsIgnoreCase(manAtom1) && atom2.equalsIgnoreCase(manAtom2) )
    			{
    				sumCorrect=sumCorrect+1;
    				if(Math.abs(resNo1-resNo2)> 4 )
        				sumCorrectLong=sumCorrectLong+1;
    				isFound=true;
    				break;
    			}  
    			if (resNo1==manResNo2 && resNo2==manResNo1 && atom1.equalsIgnoreCase(manAtom2) && atom2.equalsIgnoreCase(manAtom1) )
    			{
    				sumCorrect=sumCorrect+1;
    				if(Math.abs(resNo1-resNo2)> 4 )
        				sumCorrectLong=sumCorrectLong+1;
    				isFound=true;
    				break;
    			}  
    			
    			//new...
    			if (resNo1==manResNo1 && resNo2==manResNo2 && subAtom1.equalsIgnoreCase(subManAtom1) && subAtom2.equalsIgnoreCase(subManAtom2) )
    			{
    				sumCorrect=sumCorrect+1;
    				if(Math.abs(resNo1-resNo2)> 4 )
        				sumCorrectLong=sumCorrectLong+1;
    				isFound=true;
    				break;
    			}  
    			if (resNo1==manResNo2 && resNo2==manResNo1 && subAtom1.equalsIgnoreCase(subManAtom2) && subAtom2.equalsIgnoreCase(subManAtom1) )
    			{
    				sumCorrect=sumCorrect+1;
    				if(Math.abs(resNo1-resNo2)> 4 )
        				sumCorrectLong=sumCorrectLong+1;
    				isFound=true;
    				break;
    			}  //end of new...
    			
    			String temp1=atom1.substring(0,atom1.length()-1);
    			String temp1Man=manAtom1.substring(0,manAtom1.length()-1);
    			String temp2=atom2.substring(0,atom2.length()-1);
    			String temp2Man=manAtom2.substring(0,manAtom2.length()-1);
    			if (resNo1==manResNo1 && resNo2==manResNo2)
    			{
    				
    				if (manAtom1.substring(manAtom1.length()-1).equalsIgnoreCase("#"))
    					if (temp1.equalsIgnoreCase(temp1Man) && atom2.equalsIgnoreCase(manAtom2))
    					{
    						sumCorrect=sumCorrect+1;
    						if(Math.abs(resNo1-resNo2)> 4 )
    	        				sumCorrectLong=sumCorrectLong+1;
            				isFound=true;
            				break;
    					}
    				if (manAtom2.substring(manAtom2.length()-1).equalsIgnoreCase("#"))
    					if(temp2.equalsIgnoreCase(temp2Man) && atom1.equalsIgnoreCase(manAtom1) )
    					{
    						sumCorrect=sumCorrect+1;
    						if(Math.abs(resNo1-resNo2)> 4 )
    	        				sumCorrectLong=sumCorrectLong+1;
            				isFound=true;
            				break;
    					}
    				if (manAtom1.substring(manAtom1.length()-1).equalsIgnoreCase("#") && manAtom2.substring(manAtom2.length()-1).equalsIgnoreCase("#") )
    					if(temp1.equalsIgnoreCase(temp1Man) && temp2.equalsIgnoreCase(temp2Man))
    					{
    						sumCorrect=sumCorrect+1;
    						if(Math.abs(resNo1-resNo2)> 4 )
    	        				sumCorrectLong=sumCorrectLong+1;
            				isFound=true;
            				break;
    					}
    			}
    			if (resNo1==manResNo2  && resNo2==manResNo1)
    			{
    				if (manAtom1.substring(manAtom1.length()-1).equalsIgnoreCase("#"))
    					if (temp2.equalsIgnoreCase(temp1Man) && atom1.equalsIgnoreCase(manAtom2))
    					{
    						sumCorrect=sumCorrect+1;
    						if(Math.abs(resNo1-resNo2)> 4 )
    	        				sumCorrectLong=sumCorrectLong+1;
            				isFound=true;
            				break;
    					}
    				if (manAtom2.substring(manAtom2.length()-1).equalsIgnoreCase("#"))
    					if(temp1.equalsIgnoreCase(temp2Man) && atom2.equalsIgnoreCase(manAtom1))
    					{
    						sumCorrect=sumCorrect+1;
    						if(Math.abs(resNo1-resNo2)> 4 )
    	        				sumCorrectLong=sumCorrectLong+1;
            				isFound=true;
            				break;
    					}
    				if (manAtom1.substring(manAtom1.length()-1).equalsIgnoreCase("#") && manAtom2.substring(manAtom2.length()-1).equalsIgnoreCase("#"))
    					if(temp1.equalsIgnoreCase(temp2Man) && temp2.equalsIgnoreCase(temp1Man))
    					{
    						sumCorrect=sumCorrect+1;
    						if(Math.abs(resNo1-resNo2)> 4 )
    	        				sumCorrectLong=sumCorrectLong+1;
            				isFound=true;
            				break;
    					}
    			}
    				
    				
    		}//for(j=0;j<vecManualAsg.size();j++)
    		if(isFound==false)
    		{
    			sumWrong=sumWrong+1;   
    			
    			if(Math.abs(resNo1-resNo2)> 4 )
    				sumWrongLong=sumWrongLong+1;
    			
    			//debugging: print out the wront noe assignments
    			Peak pk=new Peak();
    			String xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
    			
    		}//if(isFound==false)
    		
    	}//for (i=0;i<vecAsg.size();i++)
    	
    	number[0]=sumCorrect;
    	number[1]=sumWrong;
    	number[2]=sumCorrectLong;
    	number[3]=sumWrongLong;
    }
    
    /**
     * get the residue name from the sequence vector.
     * 
     * @param vecSeq the vec seq
     * @param resNo the res no
     * 
     * @return residue name
     */
    public String getResNameFromSequence(Vector vecSeq, int resNo)
    {
    	String resName="";
    	for(int i=0;i<vecSeq.size();i++)
    	{
    		Assign asg = (Assign)vecSeq.elementAt(i);
 		    int no = asg.getResidueNo();
 		    String resid = (asg.getResidueType()).trim();
    		if(no==resNo)
    			return resid;
    	}
    	return resName;
    }
    
    /**
     * identify NOE residues for a sse.
     * 
     * @param vecSseSeq the sequence of sse structures
     * @param vecNoeAll the vector all Noe retraints
     * @param sseIndex the index of SSE to be determined
     * 
     * @return resIndex residue indexes that involve in NOE restraints
     */
    public boolean[] identifyNoeResidues(Vector vecSseSeq, Vector vecNoeAll, int sseIndex)
    {
    	Noe noe_temp=new Noe();
    	   	
    	Vector vecSSE=(Vector)vecSseSeq.elementAt(sseIndex); //pdb of the structrue to be determined   	
    	boolean[] resIndex=new boolean[vecSSE.size()];
    	for(int i=0;i<vecSSE.size();i++)
    		resIndex[i]=false;
    	
    	for (int i=0; i<vecSseSeq.size();i++)
    	{
    		if(i==sseIndex)
    			continue;
    		Vector vecSseOther=(Vector)vecSseSeq.elementAt(i);
    		Vector vecSseTemp=new Vector();
    		vecSseTemp.add(vecSSE);
    		vecSseTemp.add(vecSseOther);
    		Vector vecNoeUpdateOrder=noe_temp.UpdateNOE(vecNoeAll,vecSseTemp);
    		Vector noeVec=(Vector)vecNoeUpdateOrder.elementAt(0);
    		for(int j=0;j<noeVec.size();j++ )
    		{
    			Noe noe=(Noe)noeVec.elementAt(j);
    			int resNo=noe.getResidueNoA();
    			
    			int  inRef = Collections.binarySearch(vecSSE, new Pdb(resNo), new Pdb.PdbComparator() );
        		if(inRef>-1)
        			resIndex[inRef]=true;
    			
    		}//for(int j=0;j<noeVec.size();j++ )
    		
    	}//for (int i=0; i<vecSseSeq.size();i++)    	
    	
    	return resIndex;
    }
    
    /**
     * A very simple reader for NOE data, see the file format.
     * 
     * @param noeFile the name of the file
     * 
     * @return a vector of NOE object
     */
    public Vector<Noe> noeReader_old_fromLW(String noeFile){
	Vector<Noe> inputs = new Vector<Noe>();
	int noA, noB;
	String atomA, atomB;
	double noeValue;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(noeFile));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){ //ignore the comments
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    noA = (int)in.nval;
  		    in.nextToken();         
		    atomA = in.sval;
		    in.nextToken();
		    noB = (int)in.nval;
		    in.nextToken();
		    atomB = in.sval;
		    in.nextToken();
 		    lower = in.nval;
		    in.nextToken();
 		    upper = in.nval;
		    inputs.add(new Noe(noA, atomA, noB, atomB, lower, upper));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + noeFile);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }



    /**
     * A very simple reader for NOE data, see the file format.
     * 
     * @param noeFile the name of the file
     * @param upCorrection the up correction
     * 
     * @return a vector of NOE object
     */
    public Vector noeReader(String noeFile,double upCorrection)
    {
    	Vector inputs = new Vector();
    	int noA=0, noB=0;
    	String atomA="", atomB="'";
    	double noeValue;
    	int    index = -1;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	double dist=0.0,minus=0.0, plus=0.0;
    	double lowerDist=0.0, upperDist=0.0;
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	
    	    stop:
    	    while(true) 
    	    {
    	    	index = ss.trim().indexOf("assign");
    	    	if (!(index==0))
    	    	{
    	    		if ((ss = in.readLine()) == null)
    		    		break stop;
    	    		continue;	    		
    	    	}
    	    	
    	    	st = new StringTokenizer(ss);
    	    	st.nextToken();//asssign
    	    	st.nextToken();//(    	    	
    	    	st.nextToken();//resid        	    	
    	    	if (st.hasMoreTokens())
    	    		noA = new Integer(st.nextToken()).intValue();
    	    	st.nextToken();//and 
    	    	st.nextToken();//name          	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomA = new String(st.nextToken());
    	    	st.nextToken();//)
    	    	st.nextToken();//(    	    	
    	    	st.nextToken();//resid     
    	    	if (st.hasMoreTokens())
    	    		noB = new Integer(st.nextToken()).intValue();
    	    	st.nextToken();//and 
    	    	st.nextToken();//name          	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomB = new String(st.nextToken());
    	    	st.nextToken();//)    	    	
    	    	if (st.hasMoreTokens())
    	    		dist = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		minus = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		plus = new Double(st.nextToken()).doubleValue();
    	    	lowerDist=dist-minus;
    	    	upperDist=dist+plus;
    	    	
    	    	if(upCorrection<0)
    	    		upperDist=Const.noeUpBound4Packing;
    	    	else 
    	    		upperDist=upperDist+upCorrection;
    	    	
    	    	inputs.add( new Noe(noA, noB, "" , "", atomA,  atomB, lowerDist, upperDist) );
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    	}
    	    in.close();
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	}
    	
		return inputs;
    }
    
    /**
     * read the NOE assignment in mutiple format.
     * 
     * @param noeFile the name of the file
     * @param vecSeq vector of sequence
     * 
     * @return a vector of NOE object
     * Note: created May 29, 08.
     */
    public Vector noeReaderMultiAsg(String noeFile,Vector vecSeq)
    {
    	Vector inputs = new Vector();
    	int noA=0, noB=0;
    	String atomA="", atomB="'";
    	double noeValue;
    	int    index = -1;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	double dist=0.0,minus=0.0, plus=0.0;
    	double lowerDist=0.0, upperDist=0.0;
    	Noe noeTemp=new Noe();
    	int pkID=-1;
        int unknownID=-1;
        boolean isPreID=false;
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	   
    	    stop:
    	    while(true) 
    	    {
    	    	if(!isPreID)
    	    		pkID=-1;
    	    	index=ss.indexOf("peak ID");    	    	
    	    	if (index>=0)
    	    	{
    	    		pkID = Integer.parseInt(ss.substring(index+10, ss.length()).trim());
    	    		isPreID=true;
    	    	}
    	    	
    	    	index = ss.indexOf("assign");
    	    	if (!(index==0))
    	    	{
    	    		if ((ss = in.readLine()) == null)
    		    		break stop;
    	    		continue;	    		
    	    	}
    	    	isPreID=false;
    	    	Noe noe=new Noe();
    	    	
    	    	Vector vecHA=new Vector();
    	    	Vector vecHB=new Vector();
    	    	//boolean isSecond=false;
    	    	int lastEnd=0;
    	    	while(true)
    	    	{
    	    		int indResID = ss.indexOf("resid");
        	    	int indTemp=ss.indexOf("and");
        	    	
        	    	int lastInd=ss.lastIndexOf("resid");
        	    	int lastIndTemp=ss.lastIndexOf("and");
        	    	
        	    	if(indResID!=lastInd)//we have two 
        	    	{
        	    		int resNoA=Integer.parseInt(ss.substring(indResID+6, indTemp).trim());
        	    		int indEndA=ss.indexOf(")");            	    	
            	    	atomA=ss.substring(indTemp+9, indEndA).trim();
            	    	String resA=noeTemp.getResNameFromSequence(vecSeq,resNoA);
            	    	
            	    	int resNoB=Integer.parseInt(ss.substring(lastInd+6, lastIndTemp).trim());
        	    		int indEndB=ss.lastIndexOf("))");      
        	    		String resB=noeTemp.getResNameFromSequence(vecSeq,resNoB);
        	    		if(indEndB<0 || (indEndB== ss.indexOf("))") ))
        	    			indEndB=ss.lastIndexOf(")");  
            	    	atomB=ss.substring(lastIndTemp+9, indEndB).trim();
            	    	
            	    	vecHA.add(new H1CS(resNoA,resA,atomA));
            	    	vecHB.add(new H1CS(resNoB,resB,atomB) );
            	    	
            	    	if(ss.lastIndexOf("))") >40 )
            	    	{
            	    		lastEnd=2+ss.lastIndexOf("))");
            	    		break;
            	    	}            	    	
        	    	}//if(indResID!=lastInd)
        	    	else 
        	    	{
        	    		if (indResID<40)
        	    		{
	        	    		int resNoA=Integer.parseInt(ss.substring(indResID+6, indTemp).trim());
	        	    		int indEndA=ss.indexOf(")");            	    	
	            	    	atomA=ss.substring(indTemp+9, indEndA).trim();
	            	    	String resA=noeTemp.getResNameFromSequence(vecSeq,resNoA);
	            	    	vecHA.add(new H1CS(resNoA,resA,atomA));
        	    		}//if (indResID<40)
        	    		else
        	    		{
        	    			int resNoB=Integer.parseInt(ss.substring(lastInd+6, lastIndTemp).trim());
            	    		int indEndB=ss.lastIndexOf("))");      
            	    		String resB=noeTemp.getResNameFromSequence(vecSeq,resNoB);
            	    		if(indEndB<0)
            	    			indEndB=ss.lastIndexOf(")");  
                	    	atomB=ss.substring(lastIndTemp+9, indEndB).trim();
                	    	vecHB.add(new H1CS(resNoB,resB,atomB) );
                	    	if(ss.indexOf("))") >40 )
                	    	{
                	    		lastEnd=2+ss.indexOf("))");
                	    		break;
                	    	}            	
        	    		}
        	    	}//else 
        	    	
        	    	if ((ss = in.readLine()) == null)
    		    		break stop;   	    		
    	    	}//while(true)
    	    	
    	    	String sub_ss=ss.substring(lastEnd,ss.length());    	    	
    	    	st = new StringTokenizer(sub_ss);
    	    	
    	    	if (st.hasMoreTokens())
    	    		dist = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		minus = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		plus = new Double(st.nextToken()).doubleValue();
    	    	lowerDist=dist-minus;
    	    	upperDist=dist+plus; 
    	    	
    	    	//if the peak IDs are not given, we label it with decreasing integers.
    	    	if(pkID==-1)
    	    	{
    	    		pkID=unknownID-1;
    	    		unknownID--;
    	    	}
    	    	inputs.add( new Noe(pkID,vecHA,vecHB, lowerDist,upperDist) );
    	    	
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    }//while(true)
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	}
    	
		return inputs;
    }
    
    /**
     * A very simple reader for NOE data, see the file format.
     * update the residue name from sequence.
     * check whether the psudo name "#" or "##" has been updated correctly.
     * 
     * @param noeFile the name of the file
     * @param upCorrection the up correction
     * @param vecSeq vector of sequence
     * 
     * @return a vector of NOE object
     * Note: created May 29, 08.
     */
    public Vector noeReader(String noeFile,double upCorrection,Vector vecSeq)
    {
    	Vector inputs = new Vector();
    	int noA=0, noB=0;
    	String atomA="", atomB="'";
    	double noeValue;
    	int    index = -1;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	double dist=0.0,minus=0.0, plus=0.0;
    	double lowerDist=0.0, upperDist=0.0;
    	Noe noeTemp=new Noe();
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	   
    	    stop:
    	    while(true) 
    	    {
    	    	index = ss.trim().indexOf("assign");
    	    	if (!(index==0))
    	    	{
    	    		if ((ss = in.readLine()) == null)
    		    		break stop;
    	    		continue;	    		
    	    	}
    	    	
    	    	st = new StringTokenizer(ss);
    	    	st.nextToken();//asssign
    	    	st.nextToken();//(    	    	
    	    	st.nextToken();//resid        	    	
    	    	if (st.hasMoreTokens())
    	    		noA = new Integer(st.nextToken()).intValue();
    	    	st.nextToken();//and 
    	    	st.nextToken();//name          	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomA = new String(st.nextToken());
    	    	st.nextToken();//)
    	    	st.nextToken();//(    	    	
    	    	st.nextToken();//resid     
    	    	if (st.hasMoreTokens())
    	    		noB = new Integer(st.nextToken()).intValue();
    	    	st.nextToken();//and 
    	    	st.nextToken();//name          	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomB = new String(st.nextToken());
    	    	st.nextToken();//)    	    	
    	    	if (st.hasMoreTokens())
    	    		dist = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		minus = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		plus = new Double(st.nextToken()).doubleValue();
    	    	lowerDist=dist-minus;
    	    	upperDist=dist+plus;
    	    	
    	    	if(upCorrection<0)
    	    		upperDist=Const.noeUpBound4Packing;
    	    	else 
    	    		upperDist=upperDist+upCorrection;
    	   
    	    	String resA=noeTemp.getResNameFromSequence(vecSeq,noA);
    	    	String resB=noeTemp.getResNameFromSequence(vecSeq,noB);
    	    	String atomA_new=noeTemp.UpdateNOEAtomName(resA,atomA);
    	    	String atomB_new=noeTemp.UpdateNOEAtomName(resB,atomB);
    	    	inputs.add( new Noe(noA, noB, resA , resB, atomA_new,  atomB_new, lowerDist, upperDist) );
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    	}
    	    in.close();
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	}
    	
		return inputs;
    }

    
    /**
     * update the atom name in NOEs:
     * we assume that all noes use PDB-NEW scheme.
     * Strictly use "#" and "##" for psuedo-atoms.
     * return new atom name
     * Note: Created May 29, 2008.
     * 
     * @param resName the res name
     * @param atomName the atom name
     * 
     * @return the string
     */
    public String  UpdateNOEAtomName(String resName, String atomName)
    {
    	String newAtomName=atomName;
    	if(atomName.equalsIgnoreCase("H"))
    	{
    		newAtomName= "HN";
    		return newAtomName;
    	}
    	String userDir = System.getProperty("user.dir");////
    	RdcPanda rdcPan=new RdcPanda();
    	String rotSrc=  userDir+"/system/"+rdcPan.rotLibName+ "/"; 
    	Pdb pp=new Pdb();
    	String rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
    	Vector pdbRotam=pp.readPdb(rotamFile);
    	Pdb ppNode=(Pdb)pdbRotam.elementAt(0);
    	
    	String atomSub="";
    	atomSub=atomName;
    	//pseudo atom name convert:
    	//extract the previous name without "#"
		if(atomName.length()>=2)
		{			
    		if(atomName.substring(atomName.length()-2,atomName.length()).equalsIgnoreCase("##")  )
    			atomSub=atomName.substring(0,atomName.length()-2);    			
    		else if (atomName.substring(atomName.length()-1,atomName.length()).equalsIgnoreCase("#") )
    			atomSub=atomName.substring(0,atomName.length()-1);
		}		

		Vector atomVec = ppNode.getAtomVec();	
		for (int j=0; j<atomVec.size(); j++)
		{
			Cartesian cc = (Cartesian) atomVec.elementAt(j);
			String atomPPName = cc.getAtom();
			String sub_atomPPName=atomPPName;
			if(atomPPName.length()>=atomSub.length() )
				sub_atomPPName=atomPPName.substring(0,atomSub.length());
			if(sub_atomPPName.equalsIgnoreCase(atomSub))
			{
				if( (atomPPName.length()-atomSub.length() ) == 1)
					newAtomName=atomSub+"#";
				if( (atomPPName.length()-atomSub.length() ) == 2)
					newAtomName=atomSub+"##";		
				break;
			}
		}//for (int j=0; j<atomVec.size(); j++)
		return newAtomName;    	
    }
    
    /**
     * update the atom name in NOEs:
     * we assume that all noes use PDB-NEW scheme.
     * convert "#" and "##" of psuedo-atoms into non-"#".
     * return new atom name
     * Note: Created May 29, 2008.
     * 
     * @param resName the res name
     * @param atomName the atom name
     * 
     * @return the string
     */
    public String  UpdateNOEAtomName2(String resName, String atomName)
    {
    	String newAtomName=atomName;
    	if(atomName.equalsIgnoreCase("H"))
    	{
    		newAtomName= "HN";
    		return newAtomName;
    	}
    	String userDir = System.getProperty("user.dir");////
    	RdcPanda rdcPan=new RdcPanda();
    	String rotSrc=  userDir+"/system/"+rdcPan.rotLibName+ "/"; 
    	Pdb pp=new Pdb();
    	String rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
    	Vector pdbRotam=pp.readPdb(rotamFile);
    	Pdb ppNode=(Pdb)pdbRotam.elementAt(0);
    	
    	String atomSub="";
    	atomSub=atomName;
    	//pseudo atom name convert:
    	//extract the previous name without "#"
		if(atomName.length()>=2)
		{			
    		if(atomName.substring(atomName.length()-2,atomName.length()).equalsIgnoreCase("##")  )
    			atomSub=atomName.substring(0,atomName.length()-2);    			
    		else if (atomName.substring(atomName.length()-1,atomName.length()).equalsIgnoreCase("#") )
    			atomSub=atomName.substring(0,atomName.length()-1);
		}		

		Vector atomVec = ppNode.getAtomVec();	
		for (int j=0; j<atomVec.size(); j++)
		{
			Cartesian cc = (Cartesian) atomVec.elementAt(j);
			String atomPPName = cc.getAtom();
			String sub_atomPPName=atomPPName;
			if(atomPPName.length()>=atomSub.length() )
				sub_atomPPName=atomPPName.substring(0,atomSub.length());
			if(sub_atomPPName.equalsIgnoreCase(atomSub))
			{
				if( (atomPPName.length()-atomSub.length() ) == 1)
					newAtomName=atomSub;//+"#";
				if( (atomPPName.length()-atomSub.length() ) == 2)
					newAtomName=atomSub;//+"##";		
				break;
			}
		}//for (int j=0; j<atomVec.size(); j++)
		return newAtomName;    	
    }
    
  
    /**
     * A very simple reader for NOE data, see the file format.
     * deal with the compact format
     * 
     * @param noeFile the name of the file
     * @param upCorrection correction for the nOE upper bound
     * 
     * @return a vector of NOE object
     */
    public Vector noeReader2(String noeFile,double upCorrection)
    {
    	Vector inputs = new Vector();
    	int noA=0, noB=0;
    	String atomA="", atomB="'";
    	double noeValue;
    	int    index = -1;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	double dist=0.0,minus=0.0, plus=0.0;
    	double lowerDist=0.0, upperDist=0.0;
    	double correction=upCorrection;
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	
    	    stop:
    	    while(true) 
    	    {
    	    	index = ss.trim().indexOf("assign");
    	    	if (!(index==0))
    	    	{
    	    		if ((ss = in.readLine()) == null)
    		    		break stop;
    	    		continue;	    		
    	    	}
    	    	
    	    	st = new StringTokenizer(ss);
    	    	st.nextToken();//asssign
    	    		    	
    	    	st.nextToken();//(resid        	    	
    	    	if (st.hasMoreTokens())
    	    		noA = new Integer(st.nextToken()).intValue();
    	    	st.nextToken();//and 
    	    	st.nextToken();//name          	    	    	    	
    	    	if (st.hasMoreTokens())
    	    	{
    	    		String temp=new String(st.nextToken());
    	    		atomA = temp.substring(0,temp.length()-1);/////
    	    	}
    	    	    	
    	    	st.nextToken();//resid     
    	    	if (st.hasMoreTokens())
    	    		noB = new Integer(st.nextToken()).intValue();
    	    	st.nextToken();//and 
    	    	st.nextToken();//name          	    	    	    	
    	    	if (st.hasMoreTokens())
    	    	{
    	    		String temp=new String(st.nextToken());
    	    		atomB = temp.substring(0,temp.length()-1);
    	    	}
    	     	
    	    	if (st.hasMoreTokens())
    	    		dist = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		minus = new Double(st.nextToken()).doubleValue();
    	    	if (st.hasMoreTokens())
    	    		plus = new Double(st.nextToken()).doubleValue();
    	    	
    	    	lowerDist=1.8;//dist-minus;
    	    	upperDist=dist+plus;///////
    	    	    	    	
    	    	if(upCorrection<0)
    	    		upperDist=Const.noeUpBound4Packing;
    	    	else 
    	    		upperDist=upperDist+correction;
    	    	
    	    	inputs.add( new Noe(noA, noB, "" , "", atomA,  atomB, lowerDist, upperDist) );
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    	}
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	}
    	
		return inputs;
    }
       
    /**
     * check whether there is a atom name in noe talbe which
     * is not in pdb.
     * 
     * @param vecNoe the vec noe
     * @param vecPdb the vec pdb
     * 
     * @return true, if all noe proton names are in pdb, false else
     */
    public boolean CheckNoeHInPdb(Vector vecNoe, Vector vecPdb)
    {
    	System.out.println("--------------------------------------");
    	int resNoA=0, resNoB=0;
    	String resA="", resB="";
    	String atomA="",atomB="";  
    	int index = -1;
    	boolean isIn=false;
    	for (int i=0; i<vecNoe.size();i++)
    	{
    		Noe noe=(Noe)vecNoe.elementAt(i);
    		resNoA=noe.getResidueNoA();
    		resA=noe.getResidueA();
    		atomA=noe.getAtomA();
    		resNoB=noe.getResidueNoB();
    		resB=noe.getResidueB();
    		atomB=noe.getAtomB();
    		index = Collections.binarySearch(vecPdb, new Pdb(resNoA), new Pdb.PdbComparator() );
    		if ( index <0 )
    		{
    			isIn=false;
    			System.out.println("No NOE in PDB (resNoA): "+resNoA + " " + atomA + "-" + resNoB+ " "+atomB);
    		}
    		else
    		{
    			isIn=false;
    			Pdb pp = (Pdb)vecPdb.elementAt(index);
    	    	Vector atomVec = pp.getAtomVec();
    	    	for (int j=0; j<atomVec.size(); j++)
    	    	{
    	    		Cartesian cc = (Cartesian)atomVec.elementAt(j);
    			    String atom = cc.getAtom();
    			    if ( atom.equalsIgnoreCase(atomA) )
    			    	isIn=true;   			    	
    	    	}
    	    	if(!isIn)
    	    		System.out.println("No NOE in PDB (atomA name): "+resNoA + " " + atomA + "-" + resNoB+ " "+atomB);
    		}
    		
    		index = Collections.binarySearch(vecPdb, new Pdb(resNoB), new Pdb.PdbComparator() );
    		if ( index <0 )
    		{
    			isIn=false;
    			System.out.println("No NOE in PDB (resNoB): "+resNoA + " " + atomA + "-" + resNoB+ " "+atomB);
    		}
    		else
    		{
    			isIn=false;
    			Pdb pp = (Pdb)vecPdb.elementAt(index);
    	    	Vector atomVec = pp.getAtomVec();
    	    	for (int j=0; j<atomVec.size(); j++)
    	    	{
    	    		Cartesian cc = (Cartesian)atomVec.elementAt(j);
    			    String atom = cc.getAtom();
    			    if ( atom.equalsIgnoreCase(atomB) )
    			    	isIn=true;   			    	
    	    	}
    	    	if(!isIn)
    	    		System.out.println("No NOE in PDB (atomB name): "+resNoA + " " + atomA + "-" + resNoB+ " "+atomB);
    		}
    	}//for (int i=0; i<vecNoe.size();i++)
    	
    	System.out.println("--------------------------------------");
    	
    	return isIn;
    }
    
    /**
     * update proton name from pseudo atoms. notes: may be specific for ff domain
     * 
     * @param resName the res name
     * @param atomName the atom name
     * 
     * @return a vector of updated NOE object
     */
    public Vector UpdatePseudoProton(String resName, String atomName)
    {
    	Vector vecAtomName=new Vector();
    	if ( !(atomName.substring(0,1).equalsIgnoreCase("Q")) )
    	{
    		vecAtomName.add(atomName);
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("ALA")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB1");
    		vecAtomName.add("HB2");
    		vecAtomName.add("HB3");   
    		return vecAtomName;
    	}
    	
    	if(resName.equalsIgnoreCase("ASP")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2");
    		vecAtomName.add("HB3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("CYS")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2");
    		vecAtomName.add("HB3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("GLU")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2");
    		vecAtomName.add("HB3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("GLU")&& atomName.equalsIgnoreCase("QG"))
    	{
    		vecAtomName.add("HG2");
    		vecAtomName.add("HG3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("PHE")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2");
    		vecAtomName.add("HB3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("PHE")&& atomName.equalsIgnoreCase("QD"))
    	{
    		vecAtomName.add("HD1");
    		vecAtomName.add("HD2"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("PHE")&& atomName.equalsIgnoreCase("QE"))
    	{
    		vecAtomName.add("HE1");
    		vecAtomName.add("HE2"); 
    		return vecAtomName;
    	}    	
    	if(resName.equalsIgnoreCase("HIS")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2");
    		vecAtomName.add("HB3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("HIS")&& atomName.equalsIgnoreCase("QD"))
    	{
    		vecAtomName.add("HD1");
    		vecAtomName.add("HD2"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("PHE")&& atomName.equalsIgnoreCase("QE"))
    	{
    		vecAtomName.add("HE1");
    		vecAtomName.add("HE2"); 
    		return vecAtomName;
    	}    	
    	if(resName.equalsIgnoreCase("ILE")&& atomName.equalsIgnoreCase("QG2"))
    	{
    		vecAtomName.add("HG21");
    		vecAtomName.add("HG22"); 
    		vecAtomName.add("HG23"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("ILE")&& atomName.equalsIgnoreCase("QG1"))
    	{
    		vecAtomName.add("HG12"); 
    		vecAtomName.add("HG13"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("ILE")&& atomName.equalsIgnoreCase("QD1"))
    	{
    		vecAtomName.add("HD11");
    		vecAtomName.add("HD12"); 
    		vecAtomName.add("HD13"); 
    		return vecAtomName;
    	}    	
    	if(resName.equalsIgnoreCase("LYS")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("LYS")&& atomName.equalsIgnoreCase("QG"))
    	{
    		vecAtomName.add("HG2"); 
    		vecAtomName.add("HG3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("LYS")&& atomName.equalsIgnoreCase("QD"))
    	{
    		vecAtomName.add("HD2"); 
    		vecAtomName.add("HD3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("LYS")&& atomName.equalsIgnoreCase("QE"))
    	{
    		vecAtomName.add("HE2"); 
    		vecAtomName.add("HE3"); 
    		return vecAtomName;
    	}    	
    	if(resName.equalsIgnoreCase("LEU")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3"); 
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("LEU")&& atomName.equalsIgnoreCase("QD1"))
    	{
    		vecAtomName.add("HD11"); 
    		vecAtomName.add("HD12"); 
    		vecAtomName.add("HD13");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("LEU")&& atomName.equalsIgnoreCase("QD2"))
    	{
    		vecAtomName.add("HD21"); 
    		vecAtomName.add("HD22"); 
    		vecAtomName.add("HD23");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("MET")&& atomName.equalsIgnoreCase("QB"))
    	{
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("MET")&& atomName.equalsIgnoreCase("QG"))
    	{
    		vecAtomName.add("HG2"); 
    		vecAtomName.add("HG3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("MET")&& atomName.equalsIgnoreCase("QE"))
    	{
    		vecAtomName.add("HE1");  	
    		vecAtomName.add("HE2"); 
    		vecAtomName.add("HE3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("ASN")&& atomName.equalsIgnoreCase("QB"))
    	{ 	
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("PRO")&& atomName.equalsIgnoreCase("QB"))
    	{ 	
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("PRO")&& atomName.equalsIgnoreCase("QG"))
    	{ 	
    		vecAtomName.add("HG2"); 
    		vecAtomName.add("HG3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("PRO")&& atomName.equalsIgnoreCase("QD"))
    	{ 	
    		vecAtomName.add("HD2"); 
    		vecAtomName.add("HD3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("GLN")&& atomName.equalsIgnoreCase("QB"))
    	{ 	
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("GLN")&& atomName.equalsIgnoreCase("QG"))
    	{ 	
    		vecAtomName.add("HG2"); 
    		vecAtomName.add("HG3");
    		return vecAtomName;
    	}   	
    	if(resName.equalsIgnoreCase("ARG")&& atomName.equalsIgnoreCase("QD"))
    	{ 	
    		vecAtomName.add("HD2"); 
    		vecAtomName.add("HD3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("ARG")&& atomName.equalsIgnoreCase("QB"))
    	{ 	
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("ARG")&& atomName.equalsIgnoreCase("QG"))
    	{ 	
    		vecAtomName.add("HG2"); 
    		vecAtomName.add("HG3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("SER")&& atomName.equalsIgnoreCase("QB"))
    	{ 	
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("THR")&& atomName.equalsIgnoreCase("QG2"))
    	{ 	
    		vecAtomName.add("HG21"); 
    		vecAtomName.add("HG22");
    		vecAtomName.add("HG23");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("VAL")&& atomName.equalsIgnoreCase("QG2"))
    	{ 	
    		vecAtomName.add("HG21"); 
    		vecAtomName.add("HG22");
    		vecAtomName.add("HG23");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("VAL")&& atomName.equalsIgnoreCase("QG1"))
    	{ 	
    		vecAtomName.add("HG11"); 
    		vecAtomName.add("HG12");
    		vecAtomName.add("HG13");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("TRP")&& atomName.equalsIgnoreCase("QB"))
    	{ 	
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("TYR")&& atomName.equalsIgnoreCase("QB"))
    	{ 	
    		vecAtomName.add("HB2"); 
    		vecAtomName.add("HB3");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("TYR")&& atomName.equalsIgnoreCase("QD"))
    	{ 	
    		vecAtomName.add("HD1"); 
    		vecAtomName.add("HD2");
    		return vecAtomName;
    	}
    	if(resName.equalsIgnoreCase("TYR")&& atomName.equalsIgnoreCase("QE"))
    	{ 	
    		vecAtomName.add("HE1"); 
    		vecAtomName.add("HE2");
    		return vecAtomName;
    	}
    	return vecAtomName;
    	
    }
 
    /**
     * set the same upper bound for all NOEs.
     * 
     * @param vecOldNOE old noe constraints with pseudo protons
     * @param dbUpper NOE upper bound
     * 
     * @return a vector of updated NOE object
     */
    public Vector setSameUpperBounds4All(final Vector vecOldNOE, double dbUpper)
    {
    	Vector vecUpdateNOE=new Vector();
    	vecUpdateNOE.addAll(vecOldNOE);
    	int resNoA=0, resNoB=0;
    	String resA="", resB="";
    	String atomA="",atomB="";
    	double lo=0.0, up=0.0;
    	for (int i=0; i<vecUpdateNOE.size();i++)
    	{
    		Noe noe=(Noe)vecUpdateNOE.elementAt(i);
    		double originalUp=noe.getUpper();
    		double maxUp=Math.max(dbUpper,originalUp);
    		noe.setDistUpper(maxUp);   		
    	}
    	
    	return vecUpdateNOE;
    }	
    	
    /**
     * delete repeated NOEs.
     * 
     * @param vecOldNOE old noe constraints with pseudo protons
     * 
     * @return a vector of updated NOE object
     */
    public Vector<Noe> DeleteRepeatNOEs(final Vector<Noe> vecOldNOE)
    {
    	Vector vecUpdateNOE=new Vector();
    	
    	int resNoA=0, resNoB=0;
    	String resA="", resB="";
    	String atomA="",atomB="";
    	double lo=0.0, up=0.0;
    	for (int i=0; i<vecOldNOE.size();i++)
    	{
    		Noe noe=(Noe)vecOldNOE.elementAt(i);
    		resNoA=noe.getResidueNoA();
    		resA=noe.getResidueA();
    		atomA=noe.getAtomA();
    		resNoB=noe.getResidueNoB();
    		resB=noe.getResidueB();
    		atomB=noe.getAtomB();
    		
    		boolean isRepeated =false;
    		for (int j=0; j< vecUpdateNOE.size();j++)
    		{
    			Noe noe2=(Noe)vecUpdateNOE.elementAt(j);
        		int resNoA2=noe2.getResidueNoA();
        		String resA2=noe2.getResidueA();
        		String atomA2=noe2.getAtomA();
        		int resNoB2=noe2.getResidueNoB();
        		String resB2=noe2.getResidueB();
        		String atomB2=noe2.getAtomB();
        		if((resNoA2==resNoA) && (resNoB2==resNoB) && (atomA2.equalsIgnoreCase(atomA)) && (atomB2.equalsIgnoreCase(atomB)) )
        			isRepeated=true;       		
    		}
    		if (! isRepeated)
    			vecUpdateNOE.add(noe);   		
    		
    	}
    	
    	return vecUpdateNOE;
    }	
    /**
     * A method for printing NOEs to file in xplor format
     * 
     * @param pdbVec a vector of Pdb object
     * @param fileName the file name
     * @param out the out
     */
    public void printNoesToFileInXplor(Vector<Noe> vecNoes,String fileName) {
    	Peak pk=new Peak();
    	String xplorNoeStr="";
    	//double[] disUpper=new double[1];
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));    			
    		//out.println("!REMARK: Total number of NOEs is "+vecNoes.size());   		
    		
	        for (int i=0;i<vecNoes.size();i++)		       	
	    	{
	    		Noe noe=(Noe)vecNoes.elementAt(i);
	    		int resNo1=noe.getResidueNoA();
	    		int resNo2=noe.getResidueNoB();
	    		String res1=noe.getResidueA();
	    		String res2=noe.getResidueB();
	    		String atom1=noe.getAtomA();
	    		String atom2=noe.getAtomB();
	    		double distUpper=noe.getUpper();
	    		
	    		//xplor format:	    		
				xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);				
				out.println(xplorNoeStr);
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	//System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
    	
    }
    /**
     * print noe.
     * 
     * @param vecNOE the vec noe
     * 
     * @return a vector of updated NOE object
     */
    public void PrintNOE(final Vector vecNOE)
    {
    	int resNoA=0, resNoB=0;
    	String resA="", resB="";
    	String atomA="",atomB="";
    	double lo=0.0, up=0.0;
    	for (int i=0; i<vecNOE.size();i++)
    	{
    		Noe noe=(Noe)vecNOE.elementAt(i);
    		resNoA=noe.getResidueNoA();
    		resA=noe.getResidueA();
    		atomA=noe.getAtomA();
    		resNoB=noe.getResidueNoB();
    		resB=noe.getResidueB();
    		atomB=noe.getAtomB();
    		lo=noe.getLower();
    		up=noe.getUpper();
    		System.out.println(resNoA+ " "+ resA+ "  "+ atomA+ "  "+ resNoB+" "+ 
    				resB+"  "+ atomB+"  " + up+ " "+ " #peak    "+ i);
    	}
    
    }
    
    /**
     * update noe constraint, for pseudo atoms.
     * 
     * @param vecOldNOE old noe constraints with pseudo protons
     * @param vecSSEOrder the vec sse order
     * 
     * @return a vector of updated NOE object
     */
    public Vector UpdateNOE(final Vector vecOldNOE, final Vector vecSSEOrder)
    {
    	Vector vecUpdateNOE=new Vector();
    	vecUpdateNOE.addAll(vecOldNOE);
    	int resNoA=0, resNoB=0;
    	String resA="", resB="";
    	String atomA="",atomB="";
    	double lo=0.0, up=0.0;
    	    	
    	//now we divide the whole noe table into different subsets based on packing order
    	Vector vecNOEOrder=new Vector();//for storing order of noe vectors
    	Vector vecPdbA=new Vector();
    	Vector vecPdbB=new Vector();
    	Pdb	pdb1=new Pdb();
		Pdb	pdb2=new Pdb(); 
		
    	vecPdbA.addAll( (Vector)vecSSEOrder.elementAt(0) );
    	Collections.sort(vecPdbA, new Pdb.PdbComparator());
         
    	for(int i=1;i<vecSSEOrder.size();i++)
    	{
    		vecPdbB=new Vector();
    		vecPdbB.addAll( (Vector)vecSSEOrder.elementAt(i));   
    		Vector vecNoe=new Vector();    		
    		
    		Collections.sort(vecPdbB, new Pdb.PdbComparator());
    		
    		pdb1=(Pdb)vecPdbA.elementAt(0);
    		pdb2=(Pdb)vecPdbA.elementAt(vecPdbA.size()-1);
    		int NoA1=pdb1.getResidueNo();
    		int NoA2=pdb2.getResidueNo();
    		
    		pdb1=(Pdb)vecPdbB.elementAt(0);
    		pdb2=(Pdb)vecPdbB.elementAt(vecPdbB.size()-1);
    		int NoB1=pdb1.getResidueNo();
    		int NoB2=pdb2.getResidueNo();
    		
    		for (int j=0;j<vecUpdateNOE.size();j++)
    		{
    			Noe noe=(Noe)vecUpdateNOE.elementAt(j);
    			resNoA=noe.getResidueNoA();
        		resA=noe.getResidueA();
        		atomA=noe.getAtomA();
        		resNoB=noe.getResidueNoB();
        		resB=noe.getResidueB();
        		atomB=noe.getAtomB();
        		lo=noe.getLower();
        		up=noe.getUpper();
        		
        		
        		
        		int indexAInA = Collections.binarySearch(vecPdbA, new Pdb(resNoA), new Pdb.PdbComparator() );
        		int indexBInB = Collections.binarySearch(vecPdbB, new Pdb(resNoB), new Pdb.PdbComparator() );
        		
        		
        		int indexAInB = Collections.binarySearch(vecPdbB, new Pdb(resNoA), new Pdb.PdbComparator() );
        		int indexBInA = Collections.binarySearch(vecPdbA, new Pdb(resNoB), new Pdb.PdbComparator() );
        		   
        		if( (indexAInA>-1) && (indexBInB>-1) )
        			vecNoe.add(new Noe(resNoA,resNoB,resA,resB,atomA, atomB,lo, up));
        		if((indexAInB>-1) && (indexBInA>-1)  )
        			vecNoe.add(new Noe(resNoB,resNoA,resB,resA,atomB, atomA,lo, up));
        		
        		
    		}//for (int j=0;j<vecUpdateNOE.size();j++)
    		
    		Collections.sort(vecNoe, new Noe.NoeComparator());//test here
    		Vector vecNoeNew=DeleteRepeatNOEs(vecNoe);
    		vecNOEOrder.add(vecNoeNew);
    		
    		vecPdbA.addAll(vecPdbB);
    		Collections.sort(vecPdbA, new Pdb.PdbComparator());
    		
    	}//for(int i=0;i<vecSSEOrder.size();i++)
    	
    	return vecNOEOrder;
    }
    
    /**
     * This is an old function.
     * long reange noe read from cyana format.
     * 
     * @param noeFile the name of the file
     * 
     * @return a vector of NOE object
     */
    public Vector LongRangeNoeReaderTestLW(String noeFile)
    {
    	Vector<Noe> inputs = new Vector<Noe>();
    	//int noA, noB;
    	int noA=0, noB=0;
    	String atomA="", atomB="",resA="", resB="";
    	double noeValue;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	double up_exp=0.0;
    	
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	   
    	    stop:
    	    while(true) 
    	    {   	    	
    	    	double correction=0.0;
    	    	double low_correct=0.0;
    	    	st = new StringTokenizer(ss);
    	    	//st.nextToken();//resNo1    	    	     	    	
    	    	if (st.hasMoreTokens())
    	    		noA = new Integer(st.nextToken()).intValue();
    	    	//st.nextToken();//res1 name
    	    	if (st.hasMoreTokens())
    	    		resA = new String(st.nextToken());
    	    	//st.nextToken();//atom1 name         	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomA = new String(st.nextToken());
    	    	  
    	    	
    	    	//st.nextToken();// resno2    	    	  
    	    	if (st.hasMoreTokens())
    	    		noB = new Integer(st.nextToken()).intValue();
    	    	//st.nextToken();//res 2 name
    	    	if (st.hasMoreTokens())
    	    		resB = new String(st.nextToken());
    	    	//st.nextToken();//atom2 name        	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomB = new String(st.nextToken());
    	    	   	    	
    	    	//st.nextToken();// upper distance bound    	    	
    	    	if (st.hasMoreTokens())
    	    		up_exp = new Double(st.nextToken()).doubleValue();
    	    	
    	    	if(atomA.substring(0,1).equalsIgnoreCase("Q"))
      		    {    	   
    	    		if (atomA.substring(0,2).equalsIgnoreCase("QB"))
    	    		{
    	    			atomA="CB" ;//+atomA.substring(1,atomA.length());
    	    			upper = up_exp +3.0;
          		    	lower=Math.max(1.8, upper-1.5);
    	    		}
    	    		else
    	    		{
    	    			atomA="CB" ;//+atomA.substring(1,atomA.length());
          		    	upper=12;
          		    	lower=5;    
    	    		}
      		    	
      		    	 		    	
      		    }
    	    	else 
      		    {
      		    	upper = up_exp +2.0;
      		    	lower=Math.max(1.8, upper-1.5);
      		    }
    	    	if(atomB.substring(0,1).equalsIgnoreCase("Q"))
    		    {
    	    		if(atomB.substring(0,2).equalsIgnoreCase("QB"))
    	    		{
    	    			atomB="CB" ;
    	    			upper = up_exp +3.0;
          		    	lower=Math.max(1.8, upper-1.5);
    	    		}
    	    		else
    	    		{
	    	    		atomB="CB" ;//+atomA.substring(1,atomA.length());
	      		    	//if(atomA.substring(1,2).equalsIgnoreCase("D") )
	      		    	upper=12;
	      		    	lower=5;     	
    	    		}
    		    }
      		    else 
      		    {
      		    	upper = up_exp +2.0;
      		    	lower=Math.max(1.8, upper-1.5);
      		    }
    	    	
    	    
    	    	inputs.add( new Noe(noA, noB, resA , resB, atomA,  atomB, lower,upper));
    	    	
    	    	
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    	}
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	}
    	    	
    	//here we only filter noes among sses for ff domain
    	//debuggin version
    	Vector vecNoe=new Vector();
    	for (int i=0; i< inputs.size();i++)
    	{
    		Noe noe=inputs.elementAt(i);
    		int resno1=noe.getResidueNoA();
    		int resno2=noe.getResidueNoB();
    		
    		boolean isInSSES=true;
    		if( (resno1<6) || (resno1>131)  )
    			isInSSES=false;
    		if( (resno2<6) || (resno2>131)  )
    			isInSSES=false;
    		if( (20<resno1) && (resno1<30)   )
    			isInSSES=false;
    		if( (20<resno2) && (resno2<30)   )
    			isInSSES=false;
    		if( (38<resno1) && (resno1<50)   )
    			isInSSES=false;
    		if( (38<resno2) && (resno2<50)   )
    			isInSSES=false;    		
    		if( (77<resno1) && (resno1<84)   )
    			isInSSES=false;
    		if( (77<resno2) && (resno2<84)   )
    			isInSSES=false;
    		if( (93<resno1) && (resno1<95)   )
    			isInSSES=false;
    		if( (93<resno2) && (resno2<95)   )
    			isInSSES=false;
    		if( (105<resno1) && (resno1<117)   )
    			isInSSES=false;
    		if( (105<resno2) && (resno2<117)   )
    			isInSSES=false;
    		
    		if( (Math.min(resno1,resno2)>=6) && (Math.max(resno1,resno2)<=20)  )
    			isInSSES=false;
    		if( (Math.min(resno1,resno2)>=30) && (Math.max(resno1,resno2)<=38)  )
    			isInSSES=false;
    		if( (Math.min(resno1,resno2)>=50) && (Math.max(resno1,resno2)<=77)  )
    			isInSSES=false;
    		if( (Math.min(resno1,resno2)>=84) && (Math.max(resno1,resno2)<=93)  )
    			isInSSES=false;
    		if( (Math.min(resno1,resno2)>=95) && (Math.max(resno1,resno2)<=105)  )
    			isInSSES=false;
    		if( (Math.min(resno1,resno2)>=117) && (Math.max(resno1,resno2)<=131)  )
    			isInSSES=false;
    		
    		if (isInSSES)
    			vecNoe.add(noe);
    		
    	}//for (int i=0; i< inputs.size();i++)
    	
    	
    	//return inputs;
    	return vecNoe;

    }
    
    /**
     * Xplor noe statement.
     * 
     * @param no1 the no1
     * @param resid1 the resid1
     * @param nucleus1 the nucleus1
     * @param no2 the no2
     * @param resid the resid
     * @param nucleus2 the nucleus2
     * @param lower the lower
     * @param upper the upper
     * 
     * @return the string
     */
    public String xplorNoeStatement(int no1,String resid1,String nucleus1, int no2, String resid, 
    		String nucleus2, double lower, double upper)
    {
    	double lowerLimit = 1.80;
    	double middle   = 0.0;
    	double upLimit = 5.50;
    	double methylCorrection = 3.00;
    	double strongBound = 2.70;
    	double middleBound = 3.50;
    	double methylBound = 7.00;
    	double methylMin = 4.00;
    	double [] noeLimits = new double[3];
    	boolean isMethyl=false;
    	
    	/////////////////////////////////////////
    	if(resid1.equalsIgnoreCase("ALA") && nucleus1.equalsIgnoreCase("HB"))
    	{
    		nucleus1 = "HB#";
    		
    	}
    	else if(resid1.equalsIgnoreCase("ILE") && nucleus1.equalsIgnoreCase("HG2"))
    	{
    		nucleus1 = "HG2#";
    		
    	}
    	else if(resid1.equalsIgnoreCase("ILE") && nucleus1.equalsIgnoreCase("HD1"))
    	{
    		nucleus1 = "HD1#";
    		
    	}
    	else if(resid1.equalsIgnoreCase("LEU") && nucleus1.equalsIgnoreCase("HD1"))
    	{
    		nucleus1 = "HD1#";
    		
    	}
    	else if(resid1.equalsIgnoreCase("LEU") && nucleus1.equalsIgnoreCase("HD2"))
    	{
    		nucleus1 = "HD2#";
    		
    	}
    	else if(resid1.equalsIgnoreCase("VAL") && nucleus1.equalsIgnoreCase("HG1"))
    	{
    		nucleus1 = "HG1#";
    		
    	}else if(resid1.equalsIgnoreCase("VAL") && nucleus1.equalsIgnoreCase("HG2"))
    	{
    		nucleus1 = "HG2#";
    	
    	}else if(resid1.equalsIgnoreCase("THR") && nucleus1.equalsIgnoreCase("HG2"))
    	{
    		nucleus1 = "HG2#";
    		
    	}else
    	{ 
    		//change back to old naming scheme for xplor
    		if (resid1.equalsIgnoreCase("ARG")||resid1.equalsIgnoreCase("PRO")) 
    		{
    			if(nucleus1.equalsIgnoreCase("HB2"))
    				nucleus1="HB1";
    			if(nucleus1.equalsIgnoreCase("HB3"))
    				nucleus1="HB2";
    			if(nucleus1.equalsIgnoreCase("HG2"))
    				nucleus1="HG1";
    			if(nucleus1.equalsIgnoreCase("HG3"))
    				nucleus1="HG2";
    			if(nucleus1.equalsIgnoreCase("HD2"))
    				nucleus1="HD1";
    			if(nucleus1.equalsIgnoreCase("HD3"))
    				nucleus1="HD2";
    		}
    		if (resid1.equalsIgnoreCase("ASP")||resid1.equalsIgnoreCase("CYS")||
	    		resid1.equalsIgnoreCase("HIS")||resid1.equalsIgnoreCase("SER")||
	    		resid1.equalsIgnoreCase("TRP")||resid1.equalsIgnoreCase("ASN")||
	    		resid1.equalsIgnoreCase("LEU")||resid1.equalsIgnoreCase("VAL")||
	    		resid1.equalsIgnoreCase("TYR")||resid1.equalsIgnoreCase("PHE")) 
    		{
    			if(nucleus1.equalsIgnoreCase("HB2"))
    				nucleus1="HB1";
    			if(nucleus1.equalsIgnoreCase("HB3"))
    				nucleus1="HB2";
    		}
    		if (resid1.equalsIgnoreCase("GLU")||resid1.equalsIgnoreCase("MET")||resid1.equalsIgnoreCase("GLN")) 
    		{
    			if(nucleus1.equalsIgnoreCase("HB2"))
    				nucleus1="HB1";
    			if(nucleus1.equalsIgnoreCase("HB3"))
    				nucleus1="HB2";
    			if(nucleus1.equalsIgnoreCase("HG2"))
    				nucleus1="HG1";
    			if(nucleus1.equalsIgnoreCase("HG3"))
    				nucleus1="HG2";	    	
    		}
    		if (resid1.equalsIgnoreCase("GLY")) 
    		{
    			if(nucleus1.equalsIgnoreCase("HA2"))
    				nucleus1="HA1";
    			if(nucleus1.equalsIgnoreCase("HA3"))
    				nucleus1="HA2";   	
    		}
    		if (resid1.equalsIgnoreCase("ILE")) 
    		{
    			if(nucleus1.equalsIgnoreCase("HG12"))
    				nucleus1="HG11";
    			if(nucleus1.equalsIgnoreCase("HG13"))
    				nucleus1="HG12";   	
    		}
    		if (resid1.equalsIgnoreCase("LYS")||resid1.equalsIgnoreCase("PRO")) 
    		{
    			if(nucleus1.equalsIgnoreCase("HB2"))
    				nucleus1="HB1";
    			if(nucleus1.equalsIgnoreCase("HB3"))
    				nucleus1="HB2";
    			if(nucleus1.equalsIgnoreCase("HG2"))
    				nucleus1="HG1";
    			if(nucleus1.equalsIgnoreCase("HG3"))
    				nucleus1="HG2";
    			if(nucleus1.equalsIgnoreCase("HD2"))
    				nucleus1="HD1";
    			if(nucleus1.equalsIgnoreCase("HD3"))
    				nucleus1="HD2";
    			if(nucleus1.equalsIgnoreCase("HE2"))
    				nucleus1="HE1";
    			if(nucleus1.equalsIgnoreCase("HE3"))
    				nucleus1="HE2";    	
    		}	        
    		if (nucleus1.equalsIgnoreCase("H"))
    			nucleus1="HN";    	
	    
    		
    	}
    	
    	/////////////////////because we consider noe from side-chains to side-chains
    	if(resid.equalsIgnoreCase("ALA") && nucleus2.equalsIgnoreCase("HB"))
    	{
    		nucleus2 = "HB#";
    		
    	}
    	else if(resid.equalsIgnoreCase("ILE") && nucleus2.equalsIgnoreCase("HG2"))
    	{
    		nucleus2 = "HG2#";
    		
    	}
    	else if(resid.equalsIgnoreCase("ILE") && nucleus2.equalsIgnoreCase("HD1"))
    	{
    		nucleus2 = "HD1#";
    		
    	}
    	else if(resid.equalsIgnoreCase("LEU") && nucleus2.equalsIgnoreCase("HD1"))
    	{
    		nucleus2 = "HD1#";
    		
    	}
    	else if(resid.equalsIgnoreCase("LEU") && nucleus2.equalsIgnoreCase("HD2"))
    	{
    		nucleus2 = "HD2#";
    		
    	}
    	else if(resid.equalsIgnoreCase("VAL") && nucleus2.equalsIgnoreCase("HG1"))
    	{
    		nucleus2 = "HG1#";
    		
    	}else if(resid.equalsIgnoreCase("VAL") && nucleus2.equalsIgnoreCase("HG2"))
    	{
    		nucleus2 = "HG2#";
    		//noeLimits = noeLimit (dis);
    	}else if(resid.equalsIgnoreCase("THR") && nucleus2.equalsIgnoreCase("HG2"))
    	{
    		nucleus2 = "HG2#";
    		
    	}else
    	{
    		//change back to old naming scheme for xplor
    		if (resid.equalsIgnoreCase("ARG")||resid.equalsIgnoreCase("PRO")) 
    		{
    			if(nucleus2.equalsIgnoreCase("HB2"))
    				nucleus2="HB1";
    			if(nucleus2.equalsIgnoreCase("HB3"))
    				nucleus2="HB2";
    			if(nucleus2.equalsIgnoreCase("HG2"))
    				nucleus2="HG1";
    			if(nucleus2.equalsIgnoreCase("HG3"))
    				nucleus2="HG2";
    			if(nucleus2.equalsIgnoreCase("HD2"))
    				nucleus2="HD1";
    			if(nucleus2.equalsIgnoreCase("HD3"))
    				nucleus2="HD2";
    		}
    		if (resid.equalsIgnoreCase("ASP")||resid.equalsIgnoreCase("CYS")||
	    		resid.equalsIgnoreCase("HIS")||resid.equalsIgnoreCase("SER")||
	    		resid.equalsIgnoreCase("TRP")||resid.equalsIgnoreCase("ASN")||
	    		resid.equalsIgnoreCase("LEU")||resid.equalsIgnoreCase("VAL")||
	    		resid.equalsIgnoreCase("TYR")||resid.equalsIgnoreCase("PHE")) 
    		{
    			if(nucleus2.equalsIgnoreCase("HB2"))
    				nucleus2="HB1";
    			if(nucleus2.equalsIgnoreCase("HB3"))
    				nucleus2="HB2";
    		}
    		if (resid.equalsIgnoreCase("GLU")||resid.equalsIgnoreCase("MET")||resid.equalsIgnoreCase("GLN")) 
    		{
    			if(nucleus2.equalsIgnoreCase("HB2"))
    				nucleus2="HB1";
    			if(nucleus2.equalsIgnoreCase("HB3"))
    				nucleus2="HB2";
    			if(nucleus2.equalsIgnoreCase("HG2"))
    				nucleus2="HG1";
    			if(nucleus2.equalsIgnoreCase("HG3"))
    				nucleus2="HG2";	    	
    		}
    		if (resid.equalsIgnoreCase("GLY")) 
    		{
    			if(nucleus2.equalsIgnoreCase("HA2"))
    				nucleus2="HA1";
    			if(nucleus2.equalsIgnoreCase("HA3"))
    				nucleus2="HA2";   	
    		}
    		if (resid.equalsIgnoreCase("ILE")) 
    		{
    			if(nucleus2.equalsIgnoreCase("HG12"))
    				nucleus2="HG11";
    			if(nucleus2.equalsIgnoreCase("HG13"))
    				nucleus2="HG12";   	
    		}
    		if (resid.equalsIgnoreCase("LYS")||resid.equalsIgnoreCase("PRO")) 
    		{
    			if(nucleus2.equalsIgnoreCase("HB2"))
    				nucleus2="HB1";
    			if(nucleus2.equalsIgnoreCase("HB3"))
    				nucleus2="HB2";
    			if(nucleus2.equalsIgnoreCase("HG2"))
    				nucleus2="HG1";
    			if(nucleus2.equalsIgnoreCase("HG3"))
    				nucleus2="HG2";
    			if(nucleus2.equalsIgnoreCase("HD2"))
    				nucleus2="HD1";
    			if(nucleus2.equalsIgnoreCase("HD3"))
    				nucleus2="HD2";
    			if(nucleus2.equalsIgnoreCase("HE2"))
    				nucleus2="HE1";
    			if(nucleus2.equalsIgnoreCase("HE3"))
    				nucleus2="HE2";    	
    		}	        
    		if (nucleus2.equalsIgnoreCase("H"))
    			nucleus2="HN";    	
	    
    		
    	}
    	
    	//////////////////////////////////////////////////
    	lowerLimit = lower;
    	middle = 0.0;
    	upLimit = upper -lower;
    	
    	String str = "assign (resid     "+String.valueOf(no1)+"  and name       "+nucleus1
    		+")   (resid     "+String.valueOf(no2)+" and name     "+nucleus2+")      "
    		+String.valueOf(lowerLimit) +"   "+String.valueOf(middle)+"  "+String.valueOf(upLimit); 
    	//disUp[0]=upLimit;
	
    	return str;
    }

    //convert noe from cyana  format to xplor format    
    /**
     * The main method.
     * 
     * @param argv the arguments
     */
    public static void main111(String[] argv)
    {
    	Assign asg=new Assign();
    	Noe noe_temp=new Noe();
    	
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String noeCyanaFile=src+"final.upl";
    	Noe noe=new Noe();
    	String fileXplorName =src+ "ff2_noe_xplor.tbl";
    	String seqFile=src+"ff2.seq";
    	Vector vecSeq=asg.ReaderSeq(seqFile);    
    	
    	Vector vecManAsg=noe_temp.noeReader(fileXplorName,0.0,vecSeq);  	//xplor format   
    	
    	Noe ne=new Noe();
    	Vector vecNoeNewCyana=ne.ConvertXplorAsgToUplFormat(vecManAsg, vecSeq, "BMRB-NEW");
    	ne.PrintNOE(vecNoeNewCyana);
    	
    	
    }
    //convert noe from cyana  format to xplor format    
    /**
     * The main method.
     * 
     * @param argv the arguments
     */
    public static void main(String[] argv)
    {
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String noeFile=src+"final.upl";
    	Noe noe=new Noe();
    	String fileName =src+ "ff2_noe_xplor.tbl";
    	
    	//out put noe constraint file
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		
    		System.out.println("! here is the noe assignment in xplor format:");
    		out.println("! here is the noe assignment in xplor format:");
    	
    		noe.ConvertCyanNoeAsgToXplor(noeFile,out);
    		out.close();
    	}catch (FileNotFoundException e)
    	{
    		System.out.println("File not found: " + fileName);
    	}catch (IOException e)
    	{
    	   System.out.println("IOException: the stack trace is:");
    	   e.printStackTrace();
    	}
    	
    }
    //generate the diheral angle constraints from bb solved from rdc-exact
    /**
     * The main method.
     * 
     * @param argv the arguments
     */
    public static void main11(String[] argv)
    {
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String noeFile=src+"final.upl";
    	Noe noe=new Noe();
    	String fileName =src+ "ff2_noe_xplor.tbl";
    	
    	//out put noe constraint file
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		
    		System.out.println("! here is the noe assignment in xplor format:");
    		out.println("! here is the noe assignment in xplor format:");
    	
    		noe.ConvertCyanNoeAsgToXplor(noeFile,out);
    		out.close();
    	}catch (FileNotFoundException e)
    	{
    		System.out.println("File not found: " + fileName);
    	}catch (IOException e)
    	{
    	   System.out.println("IOException: the stack trace is:");
    	   e.printStackTrace();
    	}
    	
    
    	//output the dihedral angle constraint file
    	Pdb pp=new Pdb();
    	String pdbFile = src+"FF2FinalPacking5.pdb"; 
    	Vector vecTempPdbBB=pp.readPdb(pdbFile);
    	Vector vecTempPdbBB2 = pp.nameConvert(vecTempPdbBB);
    	
    	Vector vecPhiPsi=pp.PhiPsiTotalPdb(vecTempPdbBB2);
    	String fileName2 =src+ "ff2_dihe_xplor.tbl";
    	String strTemp1="", strTemp2="";
    	try
    	{
    		PrintWriter out2 = new PrintWriter(new BufferedWriter(new FileWriter(fileName2)));   		
    		
    		out2.println("! here is the dihedral angle constraint in xplor format:");
    		System.out.println("! here is the dihedral angle constraint in xplor format:");
    		
    		for (int i=0;i<vecPhiPsi.size();i++)
        	{
        		PhiPsi phipsi=(PhiPsi)vecPhiPsi.elementAt(i);
        		int resNo=phipsi.getResidueNo();
        		double phiValue=(180.0/Math.PI)*(phipsi.getPhi());
        		double psiValue=(180.0/Math.PI)*(phipsi.getPsi());
        		
        	    String strPhi1 = "assign (resid     "+String.valueOf(resNo-1)+"  and name       "+"C"
           			+")   (resid     "+String.valueOf(resNo)+" and name     "+"N"+")      ";         		
        		String strPhi2 = "       (resid     "+String.valueOf(resNo)+"  and name       "+"CA"
       				+")   (resid     "+String.valueOf(resNo)+" and name     "+"C"+")      "
       				+String.valueOf(1.0) +"   "+String.valueOf(phiValue)+"  "+String.valueOf(10.0)
       				+"  "+String.valueOf(2); 
        		System.out.println(strPhi1);
        		System.out.println(strPhi2);
        		out2.println(strPhi1);
        		out2.println(strPhi2);
        		
        		     	   
        	    String strPsi1 = "assign (resid     "+String.valueOf(resNo)+"  and name       "+"N"
          			+")   (resid     "+String.valueOf(resNo)+" and name     "+"CA"+")      ";         		
       		
        		String strPsi2 = "       (resid     "+String.valueOf(resNo)+"  and name       "+"C"
      				+")   (resid     "+String.valueOf(resNo+1)+" and name     "+"N"+")      "
      				+String.valueOf(1.0) +"   "+String.valueOf(psiValue)+"  "+String.valueOf(10.0)
      				+"  "+String.valueOf(2);    
        		System.out.println(strPsi1);
        		System.out.println(strPsi2);
        		out2.println(strPsi1);
        		out2.println(strPsi2);  
        		
        		System.out.println();
        		out2.println();
        	}//for (int i=0;i<vecPhiPsi.size();i++)
    		
    	
    		noe.ConvertCyanNoeAsgToXplor(noeFile,out2);
    		out2.close();
    	}catch (FileNotFoundException e)
    	{
    		System.out.println("File not found: " + fileName2);
    	}catch (IOException e)
    	{
    	   System.out.println("IOException: the stack trace is:");
    	   e.printStackTrace();
    	}    	
    	
    }
    
    /**
     * convert genaral noe into noes between backbones.
     * 
     * @param vecNoeAll the vec noe all
     * @param vecSeq the vec seq
     * 
     * @return a vector of noe in the upl format (bmrb-new)
     */
    public Vector ConvertAllNoeToBBNoe(Vector vecNoeAll, Vector vecSeq)
    {
    	Vector vecNoeBB=new Vector();
    	//Vector vecNoeTemp=new Vector();
    	String atom_A="", atom_B="";
    	String res_A="", res_B="";
    	int resNoA=0,resNoB=0;
    	double low_dist=0.0, upper_dist=0.0;
    	
    	Pdb pdb=new Pdb();
    	double up_correction=2.5;
    	for(int i=0;i<vecNoeAll.size();i++)
    	{
    		double correctionA=0.0;
    		double correctionB=0.0;
    		
    		Noe noe=(Noe)vecNoeAll.elementAt(i);
    		atom_A=noe.getAtomA();
    		atom_B=noe.getAtomB();
    		res_A=noe.getResidueA();//should be "";
    		res_B=noe.getResidueB();//should be "";
    		resNoA=noe.getResidueNoA();
    		resNoB=noe.getResidueNoB();
    		low_dist=noe.getLower();
    		upper_dist=noe.getUpper();
    		
    		
    		
    		res_A=getResNameFromSequence(vecSeq,resNoA);
    		res_B=getResNameFromSequence(vecSeq,resNoB);
    		
    		//convert from pdb-new format to bmrb-new format, e.g. HB1,HB2-->HB2,HB3.
    		String atom_A_new=pdb.NameFromPdbToBMRB(res_A,atom_A);
    		String atom_B_new=pdb.NameFromPdbToBMRB(res_B,atom_B);  		
    		
    		//pseudo atom name convert:
    		if(atom_A_new.length()>=2)
    		{
	    		if(atom_A_new.substring(0,2).equalsIgnoreCase("QQ")  )
	    			atom_A_new="QQ"+atom_A_new.substring(1,atom_A_new.length()-2);
	    		else if (atom_A_new.substring(atom_A_new.length()-1,atom_A_new.length()).equalsIgnoreCase("#") )
	    			atom_A_new="Q"+atom_A_new.substring(1,atom_A_new.length()-1);
    		}
    		if(atom_B_new.length()>=2)
    		{
	    		if(atom_B_new.substring(atom_B_new.length()-2,atom_B_new.length()).equalsIgnoreCase("##")  )
	    			atom_B_new="QQ"+atom_B_new.substring(1,atom_B_new.length()-2);
	    		else if (atom_B_new.substring(atom_B_new.length()-1,atom_B_new.length()).equalsIgnoreCase("#") )
	    			atom_B_new="Q"+atom_B_new.substring(1,atom_B_new.length()-1);
    		}
   		
    		vecNoeBB.add(new Noe(resNoA,resNoB,res_A,res_B,atom_A_new,atom_B_new,low_dist,upper_dist));
    	} 	
    	    	    	
    	return vecNoeBB;
       	
    }
    
    /**
     * convert from the xplor noe format to cyna upl format.
     * Since the xplor is in pdb-new format, we need to change it into
     * BMRB-new format.
     * Also, since the xplor format doesn't include the residue name, we also
     * need to add residue name from the sequence information.
     * 
     * @param nameScheme naming scheme, defult "PDB-NEW"; if using "BMRB-NEW", change to "PDB-NEW"
     * @param vecNoeXplor the vec noe xplor
     * @param vecSeq the vec seq
     * 
     * @return a vector of noe in the upl format (bmrb-new)
     */
    public Vector ConvertXplorAsgToUplFormat(Vector vecNoeXplor, Vector vecSeq,String nameScheme)
    {
    	Vector vecNoeUpl=new Vector();
    	
    	String atom_A="", atom_B="";
    	String res_A="", res_B="";
    	int resNoA=0,resNoB=0;
    	double low_dist=0.0, upper_dist=0.0;
    	
    	Pdb pdb=new Pdb();
    	double up_correction=2.0;
    	for(int i=0;i<vecNoeXplor.size();i++)
    	{
    		Noe noe=(Noe)vecNoeXplor.elementAt(i);
    		atom_A=noe.getAtomA();
    		atom_B=noe.getAtomB();
    		res_A=noe.getResidueA();//should be "";
    		res_B=noe.getResidueB();//should be "";
    		resNoA=noe.getResidueNoA();
    		resNoB=noe.getResidueNoB();
    		low_dist=noe.getLower();
    		upper_dist=noe.getUpper();
    		res_A=getResNameFromSequence(vecSeq,resNoA);
    		res_B=getResNameFromSequence(vecSeq,resNoB);
    		
    		String atom_A_new=atom_A;
    		String atom_B_new=atom_B;
    		//convert from pdb-new format to bmrb-new format, e.g. HB1,HB2-->HB2,HB3.
    		if(nameScheme.equalsIgnoreCase("BMRB-NEW"))
    		{
    		    atom_A_new=pdb.NameFromPdbToBMRB(res_A,atom_A);
    		    atom_B_new=pdb.NameFromPdbToBMRB(res_B,atom_B);  	
    		}
    		
    		//pseudo atom name convert:
    		if(atom_A_new.length()>=2)
    		{
	    		if(atom_A_new.substring(atom_A_new.length()-2,atom_A_new.length()).equalsIgnoreCase("##")  )
	    			atom_A_new="QQ"+atom_A_new.substring(1,atom_A_new.length()-2);
	    		else if (atom_A_new.substring(atom_A_new.length()-1,atom_A_new.length()).equalsIgnoreCase("#") )
	    			atom_A_new="Q"+atom_A_new.substring(1,atom_A_new.length()-1);
    		}
    		if(atom_B_new.length()>=2)
    		{
	    		if(atom_B_new.substring(atom_B_new.length()-2,atom_B_new.length()).equalsIgnoreCase("##")  )
	    			atom_B_new="QQ"+atom_B_new.substring(1,atom_B_new.length()-2);
	    		else if (atom_B_new.substring(atom_B_new.length()-1,atom_B_new.length()).equalsIgnoreCase("#") )
	    			atom_B_new="Q"+atom_B_new.substring(1,atom_B_new.length()-1);
    		}
   		
    		vecNoeUpl.add(new Noe(resNoA,resNoB,res_A,res_B,atom_A_new,atom_B_new,low_dist,upper_dist));
    	} 	
    	    	    	
    	return vecNoeUpl;
    	
    }
    
    /**
     * convert from the xplor noe format to cyna upl format.
     * Since the xplor is in pdb-new format, we need to change it into
     * BMRB-new format.
     * Also, since the xplor format doesn't include the residue name, we also
     * need to add residue name from the sequence information.
     * 
     * @param vecNoeXplor the vec noe xplor
     * @param vecSeq the vec seq
     * 
     * @return a vector of noe in the upl format (bmrb-new)
     */
    public Vector ConvertXplorAsgToUplFormatFF2(Vector vecNoeXplor, Vector vecSeq)
    {
    	Vector vecNoeUpl=new Vector();
    	
    	String atom_A="", atom_B="";
    	String res_A="", res_B="";
    	int resNoA=0,resNoB=0;
    	double low_dist=0.0, upper_dist=0.0;
    	
    	Pdb pdb=new Pdb();
    	double up_correction=2.0;
    	for(int i=0;i<vecNoeXplor.size();i++)
    	{
    		Noe noe=(Noe)vecNoeXplor.elementAt(i);
    		atom_A=noe.getAtomA();
    		atom_B=noe.getAtomB();
    		res_A=noe.getResidueA();//should be "";
    		res_B=noe.getResidueB();//should be "";
    		resNoA=noe.getResidueNoA();
    		resNoB=noe.getResidueNoB();
    		low_dist=noe.getLower();
    		upper_dist=noe.getUpper();
    		res_A=getResNameFromSequence(vecSeq,resNoA);
    		res_B=getResNameFromSequence(vecSeq,resNoB);
    		
    		//convert from pdb-new format to bmrb-new format, e.g. HB1,HB2-->HB2,HB3.
    		
    		String atom_A_new=pdb.NameFromPdbToBMRB(res_A,atom_A);
    		String atom_B_new=pdb.NameFromPdbToBMRB(res_B,atom_B);  	
    		
    		//pseudo atom name convert:
    		if(atom_A_new.length()>=2)
    		{
	    		if(atom_A_new.substring(atom_A_new.length()-2,atom_A_new.length()).equalsIgnoreCase("##")  )
	    			atom_A_new="QQ"+atom_A_new.substring(1,atom_A_new.length()-2);
	    		else if (atom_A_new.substring(atom_A_new.length()-1,atom_A_new.length()).equalsIgnoreCase("#") )
	    			atom_A_new="Q"+atom_A_new.substring(1,atom_A_new.length()-1);
    		}
    		if(atom_B_new.length()>=2)
    		{
	    		if(atom_B_new.substring(atom_B_new.length()-2,atom_B_new.length()).equalsIgnoreCase("##")  )
	    			atom_B_new="QQ"+atom_B_new.substring(1,atom_B_new.length()-2);
	    		else if (atom_B_new.substring(atom_B_new.length()-1,atom_B_new.length()).equalsIgnoreCase("#") )
	    			atom_B_new="Q"+atom_B_new.substring(1,atom_B_new.length()-1);
    		}
   		
    		vecNoeUpl.add(new Noe(resNoA,resNoB,res_A,res_B,atom_A,atom_B,low_dist,6.0));
    	} 	
    	    	    	
    	return vecNoeUpl;
    	
    }
    
    /**
     * long reange noe read from cyana format.
     * 
     * @param noeFile the name of the file
     * @param out the out
     * 
     * @return a vector of NOE object
     */
    public void ConvertCyanNoeAsgToXplor(String noeFile,PrintWriter out)
    {
    	Vector<Noe> inputs = new Vector<Noe>();
    	
    	int noA=0, noB=0;
    	String atomA="", atomB="",resA="", resB="";
    	double noeValue;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	   
    	    stop:
    	    while(true) 
    	    {   	    	
    	    	double correction=1.8;
    	    	double low_correct=0.0;
    	    	st = new StringTokenizer(ss);
    	    	  	     	    	
    	    	if (st.hasMoreTokens())
    	    		noA = new Integer(st.nextToken()).intValue();
    	    
    	    	if (st.hasMoreTokens())
    	    		resA = new String(st.nextToken());
    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomA = new String(st.nextToken());
    	    	  
    	    	if(atomA.substring(0,1).equalsIgnoreCase("Q"))
      		    {
      		    	atomA="H"+atomA.substring(1,atomA.length())+"#";
      		    	
      		    	if (atomA.substring(1,2).equalsIgnoreCase("Q"))
      		    		atomA="H"+ atomA.substring(2,atomA.length())+"#";      		    	
      		    }
    	    	
    	    	   	  
    	    	if (st.hasMoreTokens())
    	    		noB = new Integer(st.nextToken()).intValue();
    	    	
    	    	if (st.hasMoreTokens())
    	    		resB = new String(st.nextToken());
    	    	 	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomB = new String(st.nextToken());
    	    	if(atomB.substring(0,1).equalsIgnoreCase("Q"))
    		    {
    		    	atomB="H"+atomB.substring(1,atomB.length()) + "#";
    		    	if(atomB.substring(1,2).equalsIgnoreCase("Q"))
    		    		atomB="H"+atomB.substring(2,atomB.length()) + "#";   		    	
    		    }
    	    	
    	    		    	
    	    	if (st.hasMoreTokens())
    	    		upper = new Double(st.nextToken()).doubleValue();
    	    	if(upper> 1.0000)
    	    	{
    	    		upper=upper+correction;
    	    	
    	    		lower=1.8;   	    	
    	    
	    	    	//output the noe into xplor format
	    	    	//we don't change the atom naming scheme
	    	    	String xplorNoeStr="";
	    	    	Peak pk=new Peak();
	    	    	
	    	    	
	    	    	xplorNoeStr = xplorNoeStatement(noA,resA, atomA, noB, resB, atomB, lower,upper);
	    	    
	    	    	System.out.println(xplorNoeStr);
	    	    	out.println(xplorNoeStr);   	    	
    	    	}
    	    	
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    	}
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	}
    	
    	

    }
    
    /**
     * calculate the NOE satisfiation score.
     * 
     * @param noeVec the noe vec
     * @param vecPdb the vec pdb
     * @param noeRmsd the noe rmsd
     * @param noeHarmScore the noe harm score
     * @param numConflicts the num conflicts
     * 
     * @return true, if noe satisfied score
     */
    public boolean noeSatisfiedScore(Vector noeVec,Vector vecPdb, double [] noeRmsd, double [] noeHarmScore, int [] numConflicts)
    {
    	
    	double [][] noeDistance = new double[noeVec.size()][2];	
    	for (int i=0; i<noeVec.size(); i++)
			noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		Pdb pp=new Pdb();
		double disRms = 0.0;
	    double  noeHarm=0.0;
	    int counter=0;
	    for (int i = 0; i < noeVec.size(); i++)
	    {
	    	int noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
	    	String atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
	    	
	    	int noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
	 	    String atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
	 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
			
	 	    double [] dis=new double[1];
	 	    double [] distance=new double[1];
	 	 	pp.measurePackDisAllRotamers(vecPdb,vecPdb,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
	 	 	
	 	 	if(dis[0]>100)
	 	 		counter++;
	 	 	
	 	 	disRms += dis[0] * dis[0];
	 	 	noeHarm+=dis[0] * dis[0];
	    }
	    noeHarm=noeHarm/noeVec.size();		    
	    disRms = Math.sqrt(disRms / noeVec.size());
	    
	    noeRmsd[0]=disRms;
	    noeHarmScore[0]=noeHarm;
	    numConflicts[0]=counter;
    	return true;
    	
    }
    
    /**
     * Special for ff2
     * // only consider noes in between two SSEs
     * calculate the NOE satisfiation score.
     * 
     * @param noeVec the noe vec
     * @param vecPdb the vec pdb
     * @param noeRmsd the noe rmsd
     * @param noeHarmScore the noe harm score
     * @param numConflicts the num conflicts
     * 
     * @return true, if noe satisfied score f f2
     */
    public boolean noeSatisfiedScoreFF2(Vector noeVec,Vector vecPdb, double [] noeRmsd, double [] noeHarmScore, int [] numConflicts)
    {
    	
    	double [][] noeDistance = new double[noeVec.size()][2];	
    	for (int i=0; i<noeVec.size(); i++)
			noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		Pdb pp=new Pdb();
		double disRms = 0.0;
	    double  noeHarm=0.0;
	    double noe_sat=0.0;
	    int counter=0;
	    int count_temp=0;
	    for (int i = 0; i < noeVec.size(); i++)
	    {
	    	int noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
	    	String atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
	    	
	    	int noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
	 	    String atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
	 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
	 	    noeDistance[i][1]=6.0;
	 	   
	 	   
	 	  
	 	   count_temp++;
	 	   
	 	    double [] dis=new double[1];
	 	    double [] distance=new double[1];
	 	 	pp.measurePackDisAllRotamers(vecPdb,vecPdb,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
	 	 
	 	 	if(dis[0]>100)
	 	 		counter++;
	 	 	if (dis[0]<0)
	 	 		System.out.println("error...");
	 	 
	 	 	
	 	 	noe_sat=noe_sat+dis[0];
	 	 	disRms += dis[0] * dis[0];
	 	 	noeHarm+=dis[0] * dis[0];
	    }
	    noeHarm=noeHarm/noeVec.size();		    
	    disRms = Math.sqrt(disRms / noeVec.size());
	    
	    noeRmsd[0]=disRms;
	    //noeHarmScore[0]=noeHarm;
	    noeHarmScore[0]=noe_sat/count_temp;
	    
	    numConflicts[0]=counter;
    	return true;
    	
    }
    
    /**
     * convert an NOE set from BMRB-NEW to PDB-NEW format.
     * 
     * @param vecNoeBMRB original NOE vector
     * 
     * @return a vector of new NOE set in PDB-NEW format
     */
    public Vector ConvertNoeFromBMRMToPDB(Vector vecNoeBMRB)
    {
    	Vector vecNOENew=new Vector();
    	Pdb pp=new Pdb();
    	for(int i=0;i< vecNoeBMRB.size();i++)
    	{
    		Noe noe=(Noe)vecNoeBMRB.elementAt(i);
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		String res1=noe.getResidueA();
    		String res2=noe.getResidueB();
    		String atom1_new=pp.NameFromBMRBToPDB(res1,atom1);
    		String atom2_new=pp.NameFromBMRBToPDB(res2,atom2);
    		noe.setAtom1Name(atom1_new);
    		noe.setAtom2Name(atom2_new);
    	}//for(int i=0;i< vecNoeBMRB.size();i++)
    	vecNOENew.addAll(vecNoeBMRB);
    	return vecNOENew;
    }
    
    /**
     * long reange noe read from cyana format.
     * 
     * @param noeFile the name of the file
     * @param upCorrection correction of NOE upper bound,
     * use 6A if upCorrection<0
     * @param nameScheme the name scheme
     * 
     * @return a vector of NOE object
     */
    public Vector LongRangeNoeReader(String noeFile, double upCorrection,String nameScheme)
    {
    	Pdb pp=new Pdb();
    	Vector<Noe> inputs = new Vector<Noe>();
    	//int noA, noB;
    	int noA=0, noB=0;
    	String atomA="", atomB="",resA="", resB="";
    	double noeValue;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	    
    	    stop:
    	    while(true) 
    	    {   	    	
    	    	double correction=0.0;
    	    	correction=upCorrection;
    	    	
    	    	double low_correct=0.0;
    	    	st = new StringTokenizer(ss);
    	    	   	     	    	
    	    	if (st.hasMoreTokens())
    	    		noA = new Integer(st.nextToken()).intValue();
    	    
    	    	if (st.hasMoreTokens())
    	    		resA = new String(st.nextToken());
    	          	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomA = new String(st.nextToken());
    	    	
    	    	if(atomA.length()>=2)
    	    	{
    	    		if(atomA.substring(0,2).equalsIgnoreCase("QQ"))
    	    			atomA="H"+atomA.substring(2,atomA.length());
    	    		else if(atomA.substring(0,1).equalsIgnoreCase("Q"))
    	    			atomA="H"+atomA.substring(1,atomA.length());
    	    	}
    	    	if (nameScheme.equalsIgnoreCase("BMRB-NEW"))
    	    		atomA=pp.NameFromBMRBToPDB(resA,atomA);
    	    	    	  
    	    	if (st.hasMoreTokens())
    	    		noB = new Integer(st.nextToken()).intValue();
    	    	
    	    	if (st.hasMoreTokens())
    	    		resB = new String(st.nextToken());
    	    	      	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomB = new String(st.nextToken());    	    	
    	    	
    	    	if(atomB.length()>=2)
    	    	{
    	    		if(atomB.substring(0,2).equalsIgnoreCase("QQ"))
    	    			atomB="H"+atomB.substring(2,atomB.length());
    	    		else if(atomB.substring(0,1).equalsIgnoreCase("Q"))
        	    		atomB="H"+atomB.substring(1,atomB.length());
    	    	}
    	    	
    	    	
    	    	if (nameScheme.equalsIgnoreCase("BMRB-NEW"))
    	    		atomB=pp.NameFromBMRBToPDB(resB,atomB);
    	    	
    	    	if (st.hasMoreTokens())
    	    		upper = new Double(st.nextToken()).doubleValue();
    	    	
    	    	lower=1.8;   	    	
    	    
    	    	
    	    	if(upCorrection<0)
    	    		upper=Const.noeUpBound4Packing;
    	    	else 
    	    		upper=upper+correction;
    	    	inputs.add( new Noe(noA, noB, resA , resB, atomA,  atomB, lower,upper ));
    	    	
    	    	
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    	}
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	} 
    	
    	return inputs; 	

    }
    
   
    /**
     * long reange noe read from cyana format.
     * 
     * @param noeFile the name of the file
     * 
     * @return a vector of NOE object
     */
    public Vector LongRangeNoeReader2(String noeFile)
    {
    	Vector<Noe> inputs = new Vector<Noe>();
    	
    	int noA=0, noB=0;
    	String atomA="", atomB="",resA="", resB="";
    	double noeValue;
    	String ss="";
    	StringTokenizer st = new StringTokenizer("");
    	
    	try
    	{
    	    BufferedReader in = new BufferedReader(new FileReader(noeFile));	
    	    ss = in.readLine();
    	   
    	    stop:
    	    while(true) 
    	    {   	    	
    	    	double correction=0.0;
    	    	double low_correct=0.0;
    	    	st = new StringTokenizer(ss);
    	    		    	     	    	
    	    	if (st.hasMoreTokens())
    	    		noA = new Integer(st.nextToken()).intValue();
    	    	
    	    	if (st.hasMoreTokens())
    	    		resA = new String(st.nextToken());
    	    	   	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomA = new String(st.nextToken());
    	    	  
    	    	if(atomA.substring(0,1).equalsIgnoreCase("Q"))
      		    {
      		    	atomA="C"+atomA.substring(1,atomA.length());
      		    	
      		    	
      		    }
    	    	  	  
    	    	if (st.hasMoreTokens())
    	    		noB = new Integer(st.nextToken()).intValue();
    	    
    	    	if (st.hasMoreTokens())
    	    		resB = new String(st.nextToken());
    	    	  	    	    	    	
    	    	if (st.hasMoreTokens())
    	    		atomB = new String(st.nextToken());
    	    	if(atomB.substring(0,1).equalsIgnoreCase("Q"))
    		    {
    		    	atomB="C"+atomB.substring(1,atomB.length());
    		    	
    		    }
    	    	if((atomA.equalsIgnoreCase("CD")|| atomA.equalsIgnoreCase("CQD") ) && (resA.equalsIgnoreCase("HIS")||resA.equalsIgnoreCase("PHE")|| resA.equalsIgnoreCase("TYR"))  )
    	    	{
    	    		
    	    		atomA="CG";//"CD1";
    	    		correction=1.1+correction;
    		    	
    	    	}
    	    	
    	    	if(atomA.equalsIgnoreCase("CQD") && (resA.equalsIgnoreCase("LEU"))  )
    	    	{
    	    		
    	    		atomA="CG";//"CD1";
    	    		correction=1.5+correction;
    		    
    		    	
    	    	}
    	    	if(atomA.equalsIgnoreCase("CQG") && (resA.equalsIgnoreCase("VAL"))  )
    	    	{
    	    		
    	    		atomA="CB";//"CD1";
    	    		correction=1.5+correction;
    		    	
    		    	
    	    	}
    	    	
    	    	if(atomA.equalsIgnoreCase("CE") && (resA.equalsIgnoreCase("PHE")|| resA.equalsIgnoreCase("TYR"))  )
    	    	{
    	    		atomA="CZ";//"CE1";
    	    		correction=1.3+correction;
    		    	
    		    	
    	    	} 	
    	    	
    	    	if(atomB.equalsIgnoreCase("CD") && (resB.equalsIgnoreCase("PHE")|| resB.equalsIgnoreCase("TYR"))  )
    	    	{
    	    		atomB="CG";//"CD1";
    	    		correction=1.3+correction;
    		    	
    		    	
    	    	}
    	    	
    	    	if(atomB.equalsIgnoreCase("CQD") && (resB.equalsIgnoreCase("LEU"))  )
    	    	{
    	    		
    	    		atomB="CG";//"CD1";
    	    		correction=1.5+correction;
    		    
    		    	
    	    	}
    	    	if(atomB.equalsIgnoreCase("CQG") && (resB.equalsIgnoreCase("VAL"))  )
    	    	{
    	    		
    	    		atomB="CB";//"CD1";
    	    		correction=1.5+correction;
    		    	
    		    	
    	    	}
    	    	if(atomB.equalsIgnoreCase("CE") && (resB.equalsIgnoreCase("PHE")|| resB.equalsIgnoreCase("TYR"))  )
    	    	{
    	    		atomB="CZ";//"CE1";
    	    		correction=1.3+correction;
    		    
    		    	
    	    	}
    	    	
    	    		
    	    	if (st.hasMoreTokens())
    	    		upper = new Double(st.nextToken()).doubleValue();
    	    
    	    	upper=upper+correction;
    	    	lower=1.8;   	    	
    	    
    	    	
    	    	inputs.add( new Noe(noA, noB, resA , resB, atomA,  atomB, lower,6.0));
    	    	
    	    	if ((ss = in.readLine()) == null)
    	    		break stop;
    	    	}
    	}catch (FileNotFoundException e) {
    	    System.out.println("File not found: " + noeFile);
    	}catch (IOException e) {
    	    System.out.println("IOException: the stack trace is:");
    	    e.printStackTrace();
    	}
    
    	return inputs;
    }


}






















