package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	RotaPattern.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/**
 * This class implements data structures and functions that deal with the rotamer properties. 
 *  Written by  Jianyang (Michael) Zeng (2005-2009).
* 
*/
public class RotaPattern 
{
	
	/**residue No of the rotamer. */
	private int resNo; //
	
	/** residue name of the rotamer. */
	private String residueType = null; //
    
    /** all pairs of proton distances. */
    private Vector HdistVec = new Vector(); //
    
    /** all pairs of NOE lists back-computed from distances. */
    private Vector backNoeVec=new Vector(); //
    
    /** vector that stores rotamers after pruning, not used for now. */
    private Vector pdbVec=new Vector();//
    
    //only following variables are used in fact
    /** The pdb for storing each rotamer. */
    private Pdb pdbRotamer=new Pdb();
    
    /** The NOE pattern matching score. */
    private double score=0.0;
    
    /** The coordinates. */
    private double[] coord=new double[3];
    
    /**
     * Instantiates a new rota pattern.
     * 
     * @param coordTemp the coord temp
     * @param dbSc the db sc
     */
    public RotaPattern(double[] coordTemp, double dbSc )
    {   	
    	coord[0]=coordTemp[0];
    	coord[1]=coordTemp[1];
    	coord[2]=coordTemp[2];
    	score=dbSc;
    	resNo=0;
    	residueType=null;    	
    }
    
    /**
     * Instantiates a new rota pattern.
     * 
     * @param pdbRot the pdb rot
     * @param dbSc the db sc
     */
    public RotaPattern(Pdb pdbRot, double dbSc )
    {
    	pdbRotamer=pdbRot;
    	score=dbSc;
    	resNo=0;
    	residueType=null;    	
    }
    
    /**
     * Instantiates a new rota pattern.
     * 
     * @param res_no the res_no
     * @param dbSc the db sc
     */
    public RotaPattern(int res_no, double dbSc )
    {
    	score=dbSc;
    	resNo=res_no;
    	residueType=null;    	
    }
    
    /**
     * Instantiates a new rota pattern.
     */
    public RotaPattern()
    {
    	resNo=0;
    	residueType=null;    	
    }
    
    /**
     * Instantiates a new rota pattern.
     * 
     * @param no the no
     * @param resName the res name
     * @param hdistVec the hdist vec
     * @param NoeVec the noe vec
     */
    public RotaPattern(int no,String resName, Vector hdistVec, Vector NoeVec)
    {
    	resNo=no;
    	residueType=resName;  
    	HdistVec=hdistVec;
    	backNoeVec=NoeVec;    	  	
    }
    
    /**
     * Instantiates a new rota pattern.
     * 
     * @param no the no
     * @param resName the res name
     * @param hdistVec the hdist vec
     * @param NoeVec the noe vec
     * @param pVec the vec
     */
    public RotaPattern(int no,String resName, Vector hdistVec, Vector NoeVec, Vector pVec)
    {
    	resNo=no;
    	residueType=resName;  
    	HdistVec=hdistVec;
    	backNoeVec=NoeVec;
    	pdbVec=pVec;   	
    }
    
    /**
     * Gets the res no.
     * 
     * @return the res no
     */
    public int getResNo()
    { return resNo;}
    
    /**
     * Gets the res type.
     * 
     * @return the res type
     */
    public String getResType()
    {return residueType;}
    
    /**
     * Gets the hdist vec.
     * 
     * @return the hdist vec
     */
    public Vector getHdistVec()
    {return HdistVec;}
    
    /**
     * Gets the back noe vec.
     * 
     * @return the back noe vec
     */
    public Vector getBackNoeVec()
    {return backNoeVec;}    
    
    /**
     * Gets the pdb.
     * 
     * @return the pdb
     */
    public Pdb getPdb()
    {
    	return pdbRotamer;
    }
    
    /**
     * Gets the coord.
     * 
     * @return the coord
     */
    public double[] getCoord()
    {
    	return coord;
    }
    
    /**
     * Gets the score.
     * 
     * @return the score
     */
    public double getScore()
    {
    	return score;
    }
    
    /**
     * Sets the score.
     * 
     * @param sc the new score
     */
    public void setScore(double sc)
    {
    	score=sc;
    }
    
    /**
     * The Class scoreComparator.
     */
    public static class scoreComparator implements Comparator{
    	
	    /* (non-Javadoc)
	     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	     */
	    public int compare(Object o1, Object o2){
    	    RotaPattern n1 = (RotaPattern)o1;
    	    RotaPattern n2 = (RotaPattern)o2;
    	    double d1 = n1.getScore();
    	    double d2 = n2.getScore();
    	    if (d1 >d2)
    		return -1;
    	    else if (d1 < d2)
    		return 1;
    	    else return 0;
    	}
        }
    
    /**
     * calculate the score of each rotamer, and choose rotamers with highest scores.
     * 
     * @param vecRot rotamers
     * @param assignVec resonance assignment
     * @param vecBB backbone structure
     * @param hnNoeVec hn Noe peak list
     * @param cnoeVec ca noe peak list
     * @param resNo residue number
     * @param distBound distance bound to define noe distance
     * @param vecXrayBB xray backbone for computing rmsd
     * @param pickNum the pick num
     * @param maxInd the max ind
     * 
     * @return refined rotamer positions
     */
    public Vector RefineRotLib(Vector vecRot, Vector vecBB,  Vector assignVec,
    		Vector hnNoeVec, Vector cnoeVec,int resNo,double distBound, int pickNum,int[] maxInd, Vector vecXrayBB)
    {
    	int i,j;
    	Vector vecNewRot=new Vector();
    	Pdb pp=new Pdb();
    	Pdb pdbRotamer;
    	
    	Vector vecHdist;
    	String rotamResName="";
    	RotaPattern rotPattern=new RotaPattern();
    	Vector vecBackNoe;
    	Assign asg=new Assign();
    	double dbScore=0.0;
    	Vector vecRotTemp=new Vector();
    	int noeCounter=0;
    
    	int index=-1;
    	double maxSc=-999999.9;
    	int sumPeaks=0;
    	for (i=0;i<vecRot.size();i++)
    	{
    		Vector vecRotamer =new Vector();
    		pdbRotamer=(Pdb)vecRot.elementAt(i);
    		vecRotamer.add(pdbRotamer);
    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		//for computing rmsd;
    		pp.compDistPatternForRotam(vecRotamer,resNo, vecXrayBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		
    		rotamResName=pdbRotamer.getResidue();
    		vecBackNoe=rotPattern.BackCompNoePattern(assignVec,resNo,rotamResName,vecHdist);
    		    		
    		int [] numPeaks=new int[1];
    		dbScore=asg.NoePatternMatchScore(vecBackNoe,hnNoeVec,cnoeVec, numPeaks);
    		sumPeaks=sumPeaks+numPeaks[0];
    		if (maxSc<dbScore)
    		{
    			maxSc=dbScore;
    			index=i;
    		}
    		vecRotTemp.add(new RotaPattern(pdbRotamer, dbScore) );
    	}//for (i=0;i<vecRot.size();i++)
    
    	Collections.sort(vecRotTemp, new RotaPattern.scoreComparator());
    	
    	RotaPattern rotPatn;
    	
    	//we change the ensemble size here...
    	double chosenNum=pickNum;
    	
    	if(rotamResName.equalsIgnoreCase("ASN"))
    		chosenNum=1;
    	else if(rotamResName.equalsIgnoreCase("ARG"))
    		chosenNum=1;
    	else if (rotamResName.equalsIgnoreCase("LYS"))
    		chosenNum=1;
    	else if (rotamResName.equalsIgnoreCase("GLN"))
    		chosenNum=1;
    	else if (rotamResName.equalsIgnoreCase("GLU"))
    		chosenNum=1;
    	else if(rotamResName.equalsIgnoreCase("MET"))
    		chosenNum=1;
    	else 
    		chosenNum=1;
    
		for (i=0;i<Math.min(chosenNum,vecRotTemp.size());i++)
		{
			rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
			vecNewRot.add(rotPatn.getPdb());
		}
    	
    	maxInd[0]=index;
    	maxInd[1]=(int) Math.floor(sumPeaks/vecRot.size());
    	return vecNewRot;
    }
    
    /**
     * Refined function for general case:
     * calculate the score of each rotamer, and choose rotamers with highest scores.
     * 
     * @param vecRot rotamers
     * @param assignVec resonance assignment
     * @param vecBB backbone structure
     * @param resNo residue number
     * @param distBound distance bound to define noe distance
     * @param csErrH the cs err h
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * @param NoesyVec the noesy vec
     * @param pickNum the pick num
     * @param maxInd the max ind
     * 
     * @return refined rotamer positions
     */
    public Vector RefineRotLib(double csErrH, double csErrN, double csErrCA,Vector vecRot, Vector vecBB,  Vector assignVec,
    		Vector NoesyVec,int resNo,double distBound, int pickNum,int[] maxInd)
    {
    	int i,j;
    	Vector vecNewRot=new Vector();
    	Pdb pp=new Pdb();
    	Pdb pdbRotamer;
    	
    	Vector vecHdist;
    	String rotamResName="";
    	RotaPattern rotPattern=new RotaPattern();
    	Vector vecBackNoe;
    	Assign asg=new Assign();
    	double dbScore=0.0;
    	Vector vecRotTemp=new Vector();
    	int noeCounter=0;
    
    	int index=-1;
    	double maxSc=-999999.9;
    	int sumPeaks=0;
    	for (i=0;i<vecRot.size();i++)
    	{
    		Vector vecRotamer =new Vector();
    		pdbRotamer=(Pdb)vecRot.elementAt(i);
    		vecRotamer.add(pdbRotamer);
    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		rotamResName=pdbRotamer.getResidue();
    		vecBackNoe=rotPattern.BackCompNoePattern(assignVec,resNo,rotamResName,vecHdist);
    		    		
    		int [] numPeaks=new int[1];
    		dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoe,NoesyVec, numPeaks,false);
    		
    		sumPeaks=sumPeaks+numPeaks[0];
    		if (maxSc<dbScore)
    		{
    			maxSc=dbScore;
    			index=i;
    		}
    		vecRotTemp.add(new RotaPattern(pdbRotamer, dbScore) );
    	}//for (i=0;i<vecRot.size();i++)
    
    	Collections.sort(vecRotTemp, new RotaPattern.scoreComparator());
    	
    	RotaPattern rotPatn;    	
    	
    	for (i=0;i< vecRotTemp.size();i++)
    	{
    		rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
    		double scDebug=rotPatn.getScore();
    		//System.out.println(" The "+ i +"th best score rotamer: "+ scDebug );
    	}//for (i=0;i< vecRotTemp.size();i++)
    	
    	//we change the ensemble size here...
    	double chosenNum=pickNum;
    	
		for (i=0;i<Math.min(chosenNum,vecRotTemp.size());i++)
		{
			rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
			vecNewRot.add(rotPatn.getPdb());
		}
    	
    	maxInd[0]=index;
    	maxInd[1]=(int) Math.floor(sumPeaks/vecRot.size());
    	return vecNewRot;
    }
    	
    /**
     * Refined function for general case:
     * calculate the score of each rotamer, and return ensemble of rotamers with highest scores.
     * 
     * @param vecRot rotamers
     * @param assignVec resonance assignment
     * @param vecBB backbone structure
     * @param resNo residue number
     * @param distBound distance bound to define noe distance
     * @param vecRotStr rotamer vectors in the structure format.
     * @param csErrH the cs err h
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * @param NoesyVec the noesy vec
     * @param pickNum the pick num
     * @param constant the constant
     * 
     * @return refined rotamer positions
     */
    public Vector RefineRotLibNew(double csErrH, double csErrN, double csErrCA,Vector vecRot, Vector vecBB,  Vector assignVec,
    		Vector NoesyVec,int resNo,double distBound, int pickNum,Vector vecRotStr,double constant)
    {
    	int i,j;
    	Vector vecNewRot=new Vector();
    	Pdb pp=new Pdb();    	
    	Pdb pdbRotamer,pdbRotamerStr;
    	BackNoe bn=new BackNoe();
    	
    	Vector vecHdist;
    	String rotamResName="";
    	RotaPattern rotPattern=new RotaPattern();
    	Vector vecBackNoe;
    	Assign asg=new Assign();
    	double dbScore=0.0;
    	Vector vecRotTemp=new Vector();
    	int noeCounter=0;
    
    	//int index=-1;
    	double maxSc=-999999.9;
    	int sumPeaks=0;
    	double minScore=9999.99;
    	Pdb pdbTemp=(Pdb)vecRot.elementAt(0);
    	String res=pdbTemp.getResidue();//////////////
    	boolean isHydrophili=false;
    	if(res.equalsIgnoreCase("HIS") || res.equalsIgnoreCase("TYR")|| res.equalsIgnoreCase("THR")
    			|| res.equalsIgnoreCase("SER")|| res.equalsIgnoreCase("ASP")|| res.equalsIgnoreCase("ASN")
    			||res.equalsIgnoreCase("LYS") || res.equalsIgnoreCase("ARG")|| res.equalsIgnoreCase("GLU")
    			||res.equalsIgnoreCase("GLN"))
    		isHydrophili=true;
    	if(isHydrophili)
    		distBound=4.0;
    	else
    		distBound=4.5;
    	
    	//first possible stereospecific assignment:
    	for (i=0;i<vecRot.size();i++)
    	{
    		Vector vecRotamer =new Vector();
    		pdbRotamer=(Pdb)vecRot.elementAt(i);
    		pdbRotamerStr=(Pdb)vecRotStr.elementAt(i);
    	    
    		vecRotamer.add(pdbRotamer);
    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		rotamResName=pdbRotamer.getResidue();
    		vecBackNoe=rotPattern.BackCompNoePattern(assignVec,resNo,rotamResName,vecHdist);
    		Vector vecBackNoeNoRepeat=bn.DeleteRepeat(vecBackNoe);
    		    		
    		int [] numPeaks=new int[1];
    		
    		//if(isHydrophili)
    			//dbScore=asg.NoePatternMatchScoreWCaliHdphic(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false,constant);
    		//else    			
    			dbScore=asg.NoePatternMatchScoreWCali(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false,constant);
    		
    		//dbScore=asg.NoePatternMatchScoreWCaliTestJBNMR1(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false,constant);
    		    		    		
    		if(minScore>dbScore)
    			minScore=dbScore;
    		
    		sumPeaks=sumPeaks+numPeaks[0];
    		if (maxSc<dbScore)    		
    			maxSc=dbScore;    			
    	
    		vecRotTemp.add(new RotaPattern(pdbRotamerStr, dbScore) );
    	}//for (i=0;i<vecRot.size();i++)
    
    	//second possible stereospecifc assignment:
    	double maxScSec=-999999.9;
    	Vector vecRotTempSec=new Vector();
    	Vector vecAsgSec=asg.SwapStereoAsg(assignVec,resNo);
       	for (i=0;i<vecRot.size();i++)
    	{
    		Vector vecRotamer =new Vector();
    		pdbRotamer=(Pdb)vecRot.elementAt(i);
    		pdbRotamerStr=(Pdb)vecRotStr.elementAt(i);
    		
    		vecRotamer.add(pdbRotamer);
    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		rotamResName=pdbRotamer.getResidue();
    		vecBackNoe=rotPattern.BackCompNoePattern(vecAsgSec,resNo,rotamResName,vecHdist);
    		Vector vecBackNoeNoRepeat=bn.DeleteRepeat(vecBackNoe);
    		
    		int [] numPeaks=new int[1];
    	//	if(isHydrophili)
    		//	dbScore=asg.NoePatternMatchScoreWCaliHdphic(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false,constant);
    		//else    			
    			dbScore=asg.NoePatternMatchScoreWCali(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false,constant);
    		
    		//dbScore=asg.NoePatternMatchScoreWCaliTestJBNMR1(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false,constant);
    		
    		sumPeaks=sumPeaks+numPeaks[0];
    		if (maxScSec<dbScore)    		
    			maxScSec=dbScore;    	    		
    	
    		vecRotTempSec.add(new RotaPattern(pdbRotamerStr, dbScore) );
    	}//for (i=0;i<vecRot.size();i++)
    
       	if(maxScSec>maxSc)
       	{
       		vecRotTemp=new Vector();
       		vecRotTemp.addAll(vecRotTempSec);
       		assignVec=new Vector();
       		assignVec.addAll(vecAsgSec);
       	}
    	
    	//for hydrophilic amino acids, the NOE patterns are not sufficient to select the correct rotamers
       	//thus, we introduce the probability from rotamer libary for hydrophilic residues.
    	for(i=0; i<vecRotTemp.size();i++ )
    	{
    		
    		RotaPattern rp=(RotaPattern)vecRotTemp.elementAt(i);
    		double sc=rp.getScore();
    		Pdb pdb=(Pdb)vecRot.elementAt(i);
    		double ratioRota=pdb.getRotamerRatio();
    		String resName=pdb.getResidue();
    		//we set the offset for each score, before multipling the probability:
    		//note: the confident factors are different for different residues, which is related to sidechain length
    		double scoreNew=0.0;
    		
    		scoreNew=(Math.log(sc) )+ 0.30*Math.log(ratioRota);//0.3
    		
    		//for longer sidechain, we use larger weight
    		if (resName.equalsIgnoreCase("GLU") || resName.equalsIgnoreCase("GLN")||  //resName.equalsIgnoreCase("ASP")||
    				 resName.equalsIgnoreCase("ARG")||resName.equalsIgnoreCase("LYS"))
     			scoreNew=(Math.log(sc) )+ 0.5*Math.log(ratioRota);//0.5
    		 
    		 
    		rp.setScore(scoreNew);    		
    	
    	}//for(i=0; i<vecRotTemp.size();i++ )
    	
    	
    	Collections.sort(vecRotTemp, new RotaPattern.scoreComparator());
    	
    	RotaPattern rotPatn;
    	
    	
    	for (i=0;i< vecRotTemp.size();i++)
    	{
    		rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
    		double scDebug=rotPatn.getScore();
    		
    	}//for (i=0;i< vecRotTemp.size();i++)
    	
    	//we change the ensemble size here...
    	double chosenNum=pickNum;   	
    
		for (i=0;i<Math.min(chosenNum,vecRotTemp.size());i++)
		{
			rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
			vecNewRot.add(rotPatn.getPdb());
		}
	
    	return vecNewRot;///
    }
    
    /**
     * calibration scheme: using bin scheme.
     * Refined function for general case:
     * calculate the score of each rotamer, and return ensemble of rotamers with highest scores
     * 
     * @param vecRot rotamers
     * @param assignVec resonance assignment
     * @param vecBB backbone structure
     * @param resNo residue number
     * @param distBound distance bound to define noe distance
     * @param vecRotStr rotamer vectors in the structure format.
     * @param csErrH the cs err h
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * @param NoesyVec the noesy vec
     * @param pickNum the pick num
     * 
     * @return refined rotamer positions
     */
    public Vector RefineRotLibNewUsingBins(double csErrH, double csErrN, double csErrCA,Vector vecRot, Vector vecBB,  Vector assignVec,
    		Vector NoesyVec,int resNo,double distBound, int pickNum,Vector vecRotStr)
    {
    	int i,j;
    	Vector vecNewRot=new Vector();
    	Pdb pp=new Pdb();    	
    	Pdb pdbRotamer,pdbRotamerStr;
    	BackNoe bn=new BackNoe();
    	
    	Vector vecHdist;
    	String rotamResName="";
    	RotaPattern rotPattern=new RotaPattern();
    	Vector vecBackNoe;
    	Assign asg=new Assign();
    	double dbScore=0.0;
    	Vector vecRotTemp=new Vector();
    	int noeCounter=0;
    
    	//int index=-1;
    	double maxSc=-999999.9;
    	int sumPeaks=0;
    	double minScore=9999.99;
    	Pdb pdbTemp=(Pdb)vecRot.elementAt(0);
    	String res=pdbTemp.getResidue();//////////////
    
    
    	//first possible stereospecific assignment:
    	for (i=0;i<vecRot.size();i++)
    	{
    		Vector vecRotamer =new Vector();
    		pdbRotamer=(Pdb)vecRot.elementAt(i);
    		pdbRotamerStr=(Pdb)vecRotStr.elementAt(i);
    	    //String res=pdbRotamer.getResidue();
    		
    		vecRotamer.add(pdbRotamer);
    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		rotamResName=pdbRotamer.getResidue();
    		vecBackNoe=rotPattern.BackCompNoePattern(assignVec,resNo,rotamResName,vecHdist);
    		Vector vecBackNoeNoRepeat=bn.DeleteRepeat(vecBackNoe);
    	
    		int [] numPeaks=new int[1];
    		
    		dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false);
    		
    		if(minScore>dbScore)
    			minScore=dbScore;
    		
    		sumPeaks=sumPeaks+numPeaks[0];
    		if (maxSc<dbScore)    		
    			maxSc=dbScore;    			
    		
    		vecRotTemp.add(new RotaPattern(pdbRotamerStr, dbScore) );
    	}//for (i=0;i<vecRot.size();i++)
    
    	//second possible stereospecifc assignment:
    	double maxScSec=-999999.9;
    	Vector vecRotTempSec=new Vector();
    	Vector vecAsgSec=asg.SwapStereoAsg(assignVec,resNo);
       	for (i=0;i<vecRot.size();i++)
    	{
    		Vector vecRotamer =new Vector();
    		pdbRotamer=(Pdb)vecRot.elementAt(i);
    		pdbRotamerStr=(Pdb)vecRotStr.elementAt(i);
    		
    		vecRotamer.add(pdbRotamer);
    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		rotamResName=pdbRotamer.getResidue();
    		vecBackNoe=rotPattern.BackCompNoePattern(vecAsgSec,resNo,rotamResName,vecHdist);
    		Vector vecBackNoeNoRepeat=bn.DeleteRepeat(vecBackNoe);
    		
    		int [] numPeaks=new int[1];
    		dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNoRepeat,NoesyVec, numPeaks,false);
    	
    		sumPeaks=sumPeaks+numPeaks[0];
    		if (maxScSec<dbScore)    		
    			maxScSec=dbScore;    			    		
    	
    		vecRotTempSec.add(new RotaPattern(pdbRotamerStr, dbScore) );
    	}//for (i=0;i<vecRot.size();i++)
    
       	if(maxScSec>maxSc)
       	{
       		vecRotTemp=new Vector();
       		vecRotTemp.addAll(vecRotTempSec);
       		assignVec=new Vector();
       		assignVec.addAll(vecAsgSec);
       	}
    	
    	//for hydrophilic amino acids, the NOE patterns are not sufficient to select the correct rotamers
       	//thus, we introduce the probability from rotamer libary for hydrophilic residues.
    	for(i=0; i<vecRotTemp.size();i++ )
    	{
    		
    		RotaPattern rp=(RotaPattern)vecRotTemp.elementAt(i);
    		double sc=rp.getScore();
    		Pdb pdb=(Pdb)vecRot.elementAt(i);
    		double ratioRota=pdb.getRotamerRatio();
    		String resName=pdb.getResidue();
    		//we set the offset for each score, before multipling the probability:
    		//note: the confident factors are different for different residues, which is related to sidechain length
    		double scoreNew=0.0;
    		if (resName.equalsIgnoreCase("LYS"))    		
    			scoreNew=(Math.log(sc) )+ 0.30*Math.log(ratioRota);
    		else if(resName.equalsIgnoreCase("ARG") )
    			scoreNew=(Math.log(sc) )+ 0.30*Math.log(ratioRota);    		
    		else if (resName.equalsIgnoreCase("GLN"))
    			scoreNew=(Math.log(sc) )+ 0.3*Math.log(ratioRota);
    		else if (resName.equalsIgnoreCase("GLU"))
    			scoreNew=(Math.log(sc) )+ 0.3*Math.log(ratioRota);
    		else if (resName.equalsIgnoreCase("ASN"))
    			scoreNew=(Math.log(sc) )+ 0.30*Math.log(ratioRota);    		
    		else if (resName.equalsIgnoreCase("THR"))
    			scoreNew=(Math.log(sc) )+ 0.00*Math.log(ratioRota);
    		else if (resName.equalsIgnoreCase("SER"))
    			scoreNew=(Math.log(sc) )+ 0.00*Math.log(ratioRota);//0.5    		
    		else if (resName.equalsIgnoreCase("TYR"))
    			scoreNew=(Math.log(sc) )+ 0.00*Math.log(ratioRota);
    		else if (resName.equalsIgnoreCase("ASP"))
    			scoreNew=(Math.log(sc) )+ 0.3*Math.log(ratioRota);
    		else
    			scoreNew=(Math.log(sc) )+ 0.0*Math.log(ratioRota);
    		
    		scoreNew=(Math.log(sc) )+ 0.00*Math.log(ratioRota);
    		rp.setScore(scoreNew);    		
    	
    	}//for(i=0; i<vecRotTemp.size();i++ )
    	
    	
    	Collections.sort(vecRotTemp, new RotaPattern.scoreComparator());
    	
    	RotaPattern rotPatn;
    	
    
    	for (i=0;i< vecRotTemp.size();i++)
    	{
    		rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
    		double scDebug=rotPatn.getScore();
    		//System.out.println(" The "+ i +"th best score rotamer: "+ scDebug );
    	}//for (i=0;i< vecRotTemp.size();i++)
    	
    	//we change the ensemble size here...
    	double chosenNum=pickNum;
    	
    		for (i=0;i<Math.min(chosenNum,vecRotTemp.size());i++)
    		{
    			rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
    			vecNewRot.add(rotPatn.getPdb());
    		}
    
    	return vecNewRot;
    }
    
    /**
     * Refined function for general case:
     * calculate the score of each rotamer, and choose rotamers with highest scores.
     * 
     * @param vecRot rotamers
     * @param assignVec resonance assignment
     * @param vecBB backbone structure
     * @param resNo residue number
     * @param distBound distance bound to define noe distance
     * @param csErrH the cs err h
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * @param NoesyVec the noesy vec
     * @param pickNum the pick num
     * @param maxScore the max score
     * 
     * @return refined rotamer positions
     */
    public Vector RefineRotLib(double csErrH, double csErrN, double csErrCA,Vector vecRot, Vector vecBB,  Vector assignVec,
    		Vector NoesyVec,int resNo,double distBound, int pickNum,double[] maxScore)
    {
    	int i,j;
    	Vector vecNewRot=new Vector();
    	Pdb pp=new Pdb();
    	Pdb pdbRotamer;
    	
    	Vector vecHdist;
    	String rotamResName="";
    	RotaPattern rotPattern=new RotaPattern();
    	Vector vecBackNoe;
    	Assign asg=new Assign();
    	double dbScore=0.0;
    	Vector vecRotTemp=new Vector();
    	int noeCounter=0;
    
    	int index=-1;
    	double maxSc=-999999.9;
    	int sumPeaks=0;
    	for (i=0;i<vecRot.size();i++)
    	{
    		Vector vecRotamer =new Vector();
    		pdbRotamer=(Pdb)vecRot.elementAt(i);
    		vecRotamer.add(pdbRotamer);
    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    		
    		rotamResName=pdbRotamer.getResidue();
    		vecBackNoe=rotPattern.BackCompNoePattern(assignVec,resNo,rotamResName,vecHdist);
    		
    		int [] numPeaks=new int[1];
    		dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoe,NoesyVec, numPeaks,false);
    		sumPeaks=sumPeaks+numPeaks[0];
    		if (maxSc<dbScore)
    		{
    			maxSc=dbScore;
    			index=i;
    		}
    		vecRotTemp.add(new RotaPattern(pdbRotamer, dbScore) );
    	}//for (i=0;i<vecRot.size();i++)
    
    	Collections.sort(vecRotTemp, new RotaPattern.scoreComparator());
    	
    	RotaPattern rotPatn;
    	
    	//we change the ensemble size here...
    	double chosenNum=pickNum;
    	
    	if(rotamResName.equalsIgnoreCase("ASN"))
    		chosenNum=1;
    	else if(rotamResName.equalsIgnoreCase("ARG"))
    		chosenNum=1;
    	else if (rotamResName.equalsIgnoreCase("LYS"))
    		chosenNum=1;
    	else if (rotamResName.equalsIgnoreCase("GLN"))
    		chosenNum=1;
    	else if (rotamResName.equalsIgnoreCase("GLU"))
    		chosenNum=1;
    	else if(rotamResName.equalsIgnoreCase("MET"))
    		chosenNum=1;
    	else 
    		chosenNum=1;
    	
    	
    		for (i=0;i<Math.min(chosenNum,vecRotTemp.size());i++)
    		{
    			rotPatn=(RotaPattern)vecRotTemp.elementAt(i);
    			vecNewRot.add(rotPatn.getPdb());
    		}    	
    	
    	maxScore[0]=index;
    	maxScore[1]=maxSc;
    	return vecNewRot;
    }
    	
          	
    	
    /**
     * for simulate ca NOEs,
     * Back compute an NOE peak from a given pair of NOE distance.
     * 
     * @param asgVec resonance assignment
     * @param hdist the hdist
     * 
     * @return the NOE peak
     */
    public BackNoe SimulateBackCompNOEPeak(final Vector asgVec, final Hdist hdist)
    {
    	long seed = 951179;
    	Random rr = new Random(seed);
    	String atomName1=hdist.getFirstCart().getAtom();
    	String atomName2=hdist.getSecondCart().getAtom();   	
    	
    	Peak pk=new Peak();
    	
    	//int i,j;
    	Map aMap = new TreeMap();
    	int no = 0;
    	String aaType ="";
    	Assign assign = new Assign();
    	int firstResNo=hdist.getFirstResNo();
    	int secondResNo=hdist.getSecondResNo();
    	
    	String firstAtomName=hdist.getFirstAtomName();
    	String secondAtomName=hdist.getSecondAtomName();
    	String firstResName=hdist.getFirstResName();
    	String secondResName=hdist.getSecondResName();
    	String heavyName=pk.GetHeavyAtomFromProton(secondResName,secondAtomName);
    	
    	//in Lincong's resonance list and noe peak lists, CA are labeled with "ca", and HN or H is labeled with "HN"
    	if (heavyName.equals("CA"))
    		heavyName="ca";
    	if (firstAtomName.equalsIgnoreCase("H"))
    		firstAtomName="HN";
    	if (secondAtomName.equalsIgnoreCase("H"))
    		secondAtomName="HN";    	
    	
    	//for methyl group protons, need to be consistent with resonance list
    	if(firstResName.equalsIgnoreCase("ALA") )
    		if(firstAtomName.equalsIgnoreCase("HB1")||firstAtomName.equalsIgnoreCase("HB2")||firstAtomName.equalsIgnoreCase("HB3"))
    			firstAtomName = "HB";
    	if(firstResName.equalsIgnoreCase("ILE")  )
    		if (firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    			firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("ILE"))
    		if (firstAtomName.equalsIgnoreCase("HD11")||firstAtomName.equalsIgnoreCase("HD12")||firstAtomName.equalsIgnoreCase("HD13"))
    			firstAtomName = "HD1";
    	if(firstResName.equalsIgnoreCase("LEU") )
    		if (firstAtomName.equalsIgnoreCase("HD11")||firstAtomName.equalsIgnoreCase("HD12")||firstAtomName.equalsIgnoreCase("HD13"))
    		firstAtomName = "HD1";
    	if(firstResName.equalsIgnoreCase("LEU"))
    		if(firstAtomName.equalsIgnoreCase("HD21")||firstAtomName.equalsIgnoreCase("HD22")||firstAtomName.equalsIgnoreCase("HD23"))
    		firstAtomName = "HD2";
    	if(firstResName.equalsIgnoreCase("VAL"))
    		if(firstAtomName.equalsIgnoreCase("HG11")||firstAtomName.equalsIgnoreCase("HG12")||firstAtomName.equalsIgnoreCase("HG13"))
    		firstAtomName = "HG1";
    	if(firstResName.equalsIgnoreCase("VAL") )
    		if( firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    		firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("THR") )
    		if (firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    		firstAtomName = "HG2";
    	    	    	
    	double dist=hdist.getDistance();
    	
    	double csH1=-999.9;
    	double csHeavy=-999.9;
    	double csH2=-999.9;
    	
    	double csValue = 0.0;
    	String nucleus = "";
    	double csSigmaH=0.03;
    	double csSigmaN=0.08;
    	double csSigmaC=0.05;
    	double csSigmaHeavy=0.0;
    	
    	for (int j=0;j<asgVec.size();j++)
    	{
    		assign =  (Assign)asgVec.elementAt(j);
    		no = assign.getResidueNo();
    	    aaType = assign.getResidueType();
    	    aMap = assign.getMap();
    	       	    
    	    for (Iterator i = aMap.entrySet().iterator(); i.hasNext(); )
    	    {
    			Map.Entry e = (Map.Entry) i.next();
    			csValue =  ((Double)e.getValue()).doubleValue();
    			nucleus =  (String)e.getKey();
    			
    			if (firstResNo==no && firstAtomName.equals(nucleus) )
    			{
    				//csH2=csValue;
    				csH2=csValue+ csSigmaH* rr.nextGaussian(); 
    			}//if (firstResNo==no && firstAtomName==nucleus)
    			
    			if (secondResNo==no && heavyName.equals(nucleus))
    			{
    				if (nucleus.substring(0,1).equalsIgnoreCase("N"))
    					csSigmaHeavy=csSigmaN;
    				else
    					csSigmaHeavy=csSigmaC;
    				//csHeavy=csValue;
    				csHeavy=csValue+csSigmaHeavy*rr.nextGaussian();
    			}

    			if (secondResNo==no && secondAtomName.equals(nucleus))
    			{
    				//csH1=csValue;
    				csH1=csValue+csSigmaH*rr.nextGaussian();
    			}
    			
    		   }//for (Iterator i = aMap.entrySet().iterator(); i.hasNext(); )
    	}//for (i=0;i<asgVec.size();i++)
    	
    	//be careful of the order, here the first res is HN/HA, the second one is in rotamer 
    	BackNoe backNoe=new BackNoe(csH1, csHeavy,csH2,dist, 0.0,secondResNo,firstResNo,
    			secondResName,firstResName,secondAtomName,heavyName,firstAtomName);    	
    	
    	return backNoe;    	
    }
    
    /**
     * Back compute an NOE peak from a given pair of NOE distance. 
     * Changed from function "BackCompNOEPeak".
     * 
     * @param asgVec resonance assignment
     * @param hdist the hdist
     * 
     * @return the NOE peak
     */
    public BackNoe BackCompNOEPeakNew(final Vector asgVec, final Hdist hdist)
    {   	
    	String atomName1=hdist.getFirstCart().getAtom();
    	String atomName2=hdist.getSecondCart().getAtom();   	
    	
    	Peak pk=new Peak();
    	
    	Map aMap = new TreeMap();
    	int no = 0;
    	String aaType ="";
    	H1CS assign = new H1CS();
    	
    	int firstResNo=hdist.getFirstResNo();
    	int secondResNo=hdist.getSecondResNo();    	
    	String firstAtomName=hdist.getFirstAtomName();
    	String secondAtomName=hdist.getSecondAtomName();
    	String firstResName=hdist.getFirstResName();
    	String secondResName=hdist.getSecondResName();
    	String heavyName=pk.GetHeavyAtomFromProton(firstResName,firstAtomName);
    	
     
    	if (firstAtomName.equalsIgnoreCase("H"))
    		firstAtomName="HN";
    	if (secondAtomName.equalsIgnoreCase("H"))
    		secondAtomName="HN";    	
    	//////--------
    	//for methyl group protons, need to be consistent with resonance list
    	if(secondResName.equalsIgnoreCase("ALA") )
    		if(secondAtomName.equalsIgnoreCase("HB1")||secondAtomName.equalsIgnoreCase("HB2")||secondAtomName.equalsIgnoreCase("HB3"))
    			secondAtomName = "HB";
    	if(secondResName.equalsIgnoreCase("ILE")  )
    		if (secondAtomName.equalsIgnoreCase("HG21")||secondAtomName.equalsIgnoreCase("HG22")||secondAtomName.equalsIgnoreCase("HG23"))
    			secondAtomName = "HG2";
    	if(secondResName.equalsIgnoreCase("ILE"))
    		if (secondAtomName.equalsIgnoreCase("HD11")||secondAtomName.equalsIgnoreCase("HD12")||secondAtomName.equalsIgnoreCase("HD13"))
    			secondAtomName = "HD1";
    	if(secondResName.equalsIgnoreCase("LEU") )
    		if (secondAtomName.equalsIgnoreCase("HD11")||secondAtomName.equalsIgnoreCase("HD12")||secondAtomName.equalsIgnoreCase("HD13"))
    			secondAtomName = "HD1";
    	if(secondResName.equalsIgnoreCase("LEU"))
    		if(secondAtomName.equalsIgnoreCase("HD21")||secondAtomName.equalsIgnoreCase("HD22")||secondAtomName.equalsIgnoreCase("HD23"))
    			secondAtomName = "HD2";
    	if(secondResName.equalsIgnoreCase("VAL"))
    		if(secondAtomName.equalsIgnoreCase("HG11")||secondAtomName.equalsIgnoreCase("HG12")||secondAtomName.equalsIgnoreCase("HG13"))
    			secondAtomName = "HG1";
    	if(secondResName.equalsIgnoreCase("VAL") )
    		if( secondAtomName.equalsIgnoreCase("HG21")||secondAtomName.equalsIgnoreCase("HG22")||secondAtomName.equalsIgnoreCase("HG23"))
    			secondAtomName = "HG2";
    	if(secondResName.equalsIgnoreCase("THR") )
    		if (secondAtomName.equalsIgnoreCase("HG21")||secondAtomName.equalsIgnoreCase("HG22")||secondAtomName.equalsIgnoreCase("HG23"))
    			secondAtomName = "HG2";
    	if(secondResName.equalsIgnoreCase("MET") )
    		if (secondAtomName.equalsIgnoreCase("HE1")||secondAtomName.equalsIgnoreCase("HE2")||secondAtomName.equalsIgnoreCase("HE3"))
    			secondAtomName = "HE";    	    	
    	if(secondResName.equalsIgnoreCase("PHE") )
    	{
    		if (secondAtomName.equalsIgnoreCase("HE1")||secondAtomName.equalsIgnoreCase("HE2"))
    			secondAtomName = "HE";   
    		if (secondAtomName.equalsIgnoreCase("HD1")||secondAtomName.equalsIgnoreCase("HD2"))
    			secondAtomName = "HD";
    	}
    	if(secondResName.equalsIgnoreCase("TYR") )
    	{
    		if (secondAtomName.equalsIgnoreCase("HE1")||secondAtomName.equalsIgnoreCase("HE2"))
    			secondAtomName = "HE";   
    		if (secondAtomName.equalsIgnoreCase("HD1")||secondAtomName.equalsIgnoreCase("HD2"))
    			secondAtomName = "HD";
    	}
    	
       	//////--------
    	//for methyl group protons, need to be consistent with resonance list
    	if(firstResName.equalsIgnoreCase("ALA") )
    		if(firstAtomName.equalsIgnoreCase("HB1")||firstAtomName.equalsIgnoreCase("HB2")||firstAtomName.equalsIgnoreCase("HB3"))
    			firstAtomName = "HB";
    	if(firstResName.equalsIgnoreCase("ILE")  )
    		if (firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    			firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("ILE"))
    		if (firstAtomName.equalsIgnoreCase("HD11")||firstAtomName.equalsIgnoreCase("HD12")||firstAtomName.equalsIgnoreCase("HD13"))
    			firstAtomName = "HD1";
    	if(firstResName.equalsIgnoreCase("LEU") )
    		if (firstAtomName.equalsIgnoreCase("HD11")||firstAtomName.equalsIgnoreCase("HD12")||firstAtomName.equalsIgnoreCase("HD13"))
    		firstAtomName = "HD1";
    	if(firstResName.equalsIgnoreCase("LEU"))
    		if(firstAtomName.equalsIgnoreCase("HD21")||firstAtomName.equalsIgnoreCase("HD22")||firstAtomName.equalsIgnoreCase("HD23"))
    		firstAtomName = "HD2";
    	if(firstResName.equalsIgnoreCase("VAL"))
    		if(firstAtomName.equalsIgnoreCase("HG11")||firstAtomName.equalsIgnoreCase("HG12")||firstAtomName.equalsIgnoreCase("HG13"))
    		firstAtomName = "HG1";
    	if(firstResName.equalsIgnoreCase("VAL") )
    		if( firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    		firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("THR") )
    		if (firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    		firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("MET") )
    		if (firstAtomName.equalsIgnoreCase("HE1")||firstAtomName.equalsIgnoreCase("HE2")||firstAtomName.equalsIgnoreCase("HE3"))
    		firstAtomName = "HE";    	    	
    	if(firstResName.equalsIgnoreCase("PHE") )
    	{
    		if (firstAtomName.equalsIgnoreCase("HE1")||firstAtomName.equalsIgnoreCase("HE2"))
    			firstAtomName = "HE";   
    		if (firstAtomName.equalsIgnoreCase("HD1")||firstAtomName.equalsIgnoreCase("HD2"))
    			firstAtomName = "HD";
    	}
    	if(firstResName.equalsIgnoreCase("TYR") )
    	{
    		if (firstAtomName.equalsIgnoreCase("HE1")||firstAtomName.equalsIgnoreCase("HE2"))
    			firstAtomName = "HE";   
    		if (firstAtomName.equalsIgnoreCase("HD1")||firstAtomName.equalsIgnoreCase("HD2"))
    			firstAtomName = "HD";
    	}
    	
    	
    	double dist=hdist.getDistance();
    	
    	double csH1=-999.9;
    	double csHeavy=-999.9;
    	double csH2=-999.9;
    	
    	double csValue = 0.0;
    	String nucleus = "";
    	double csSigmaH=0.0;
    	double csSigmaN=0.0;
    	double csSigmaC=0.0;
    	double csSigmaHeavy=0.0;
    	
    	for (int j=0;j<asgVec.size();j++)
    	{
    		assign =  (H1CS)asgVec.elementAt(j);
    		no = assign.getResidueNo();
    	    aaType = assign.getResidueType();
    	    csValue=assign.getH1CS();
    	    nucleus=assign.getAtomName();
    	    int id=nucleus.indexOf("#");
    	    if(id>0)
    	    	nucleus=nucleus.substring(0, id);
    	    String firstAtomName_temp=firstAtomName;
    	    if(firstAtomName_temp.length()> nucleus.length())
    	    	firstAtomName_temp=firstAtomName_temp.substring(0, nucleus.length());
    	    
    	    String heavyName_temp=heavyName;
    	    if(heavyName_temp.length()> nucleus.length())
    	    	heavyName_temp=heavyName_temp.substring(0, nucleus.length());
    	   
    	    String secondAtomName_temp=secondAtomName;
    	    if(secondAtomName_temp.length()> nucleus.length())
    	    	secondAtomName_temp=secondAtomName_temp.substring(0, nucleus.length());
    	    
    	    if(nucleus.equalsIgnoreCase("H"))
    	    	nucleus="HN";
    	    if (firstResNo==no && firstAtomName.equals(nucleus) )
    		{    		
    			csH1=csValue; 
    		}//if (firstResNo==no && firstAtomName==nucleus)
    			
    		if (firstResNo==no && heavyName.equals(nucleus))
    		{
    			if (nucleus.substring(0,1).equalsIgnoreCase("N"))
    				csSigmaHeavy=csSigmaN;
    			else
    				csSigmaHeavy=csSigmaC;
    			
    			csHeavy=csValue;
    		}

    		if (secondResNo==no && secondAtomName.equals(nucleus))
    		{
    			csH2=csValue;
    		}
    			
    		  
    	}//for (i=0;i<asgVec.size();i++)   	
    	
    	BackNoe backNoe=new BackNoe(csH1, csHeavy,csH2,dist, 0.0,firstResNo, secondResNo,
    			firstResName, secondResName,firstAtomName,heavyName, secondAtomName);
    	    	
    	return backNoe;    	
    }
    /**
     * Back compute an NOE peak from a given pair of NOE distance.
     * 
     * @param asgVec resonance assignment
     * @param hdist the hdist
     * 
     * @return the NOE peak
     */
    public BackNoe BackCompNOEPeak(final Vector asgVec, final Hdist hdist)
    {
    	//note: the first proton in hdist is H from rotamer, while the second proton
    	//is from backbone, ie, Ha or Hn
    	//but in backcomputed noe, it is of the format "Hn, N, H2"
    	long seed = 951179;
    	Random rr = new Random(seed);
    	String atomName1=hdist.getFirstCart().getAtom();
    	String atomName2=hdist.getSecondCart().getAtom();   	
    	
    	Peak pk=new Peak();
    	
    	Map aMap = new TreeMap();
    	int no = 0;
    	String aaType ="";
    	H1CS assign = new H1CS();
    	
    	int firstResNo=hdist.getFirstResNo();
    	int secondResNo=hdist.getSecondResNo();    	
    	String firstAtomName=hdist.getFirstAtomName();
    	String secondAtomName=hdist.getSecondAtomName();
    	String firstResName=hdist.getFirstResName();
    	String secondResName=hdist.getSecondResName();
    	String heavyName=pk.GetHeavyAtomFromProton(secondResName,secondAtomName);
    	
    
    	if (firstAtomName.equalsIgnoreCase("H"))
    		firstAtomName="HN";
    	if (secondAtomName.equalsIgnoreCase("H"))
    		secondAtomName="HN";    	
    	//////--------
    	//for methyl group protons, need to be consistent with resonance list
    	if(secondResName.equalsIgnoreCase("ALA") )
    		if(secondAtomName.equalsIgnoreCase("HB1")||secondAtomName.equalsIgnoreCase("HB2")||secondAtomName.equalsIgnoreCase("HB3"))
    			secondAtomName = "HB";
    	if(secondResName.equalsIgnoreCase("ILE")  )
    		if (secondAtomName.equalsIgnoreCase("HG21")||secondAtomName.equalsIgnoreCase("HG22")||secondAtomName.equalsIgnoreCase("HG23"))
    			secondAtomName = "HG2";
    	if(secondResName.equalsIgnoreCase("ILE"))
    		if (secondAtomName.equalsIgnoreCase("HD11")||secondAtomName.equalsIgnoreCase("HD12")||secondAtomName.equalsIgnoreCase("HD13"))
    			secondAtomName = "HD1";
    	if(secondResName.equalsIgnoreCase("LEU") )
    		if (secondAtomName.equalsIgnoreCase("HD11")||secondAtomName.equalsIgnoreCase("HD12")||secondAtomName.equalsIgnoreCase("HD13"))
    			secondAtomName = "HD1";
    	if(secondResName.equalsIgnoreCase("LEU"))
    		if(secondAtomName.equalsIgnoreCase("HD21")||secondAtomName.equalsIgnoreCase("HD22")||secondAtomName.equalsIgnoreCase("HD23"))
    			secondAtomName = "HD2";
    	if(secondResName.equalsIgnoreCase("VAL"))
    		if(secondAtomName.equalsIgnoreCase("HG11")||secondAtomName.equalsIgnoreCase("HG12")||secondAtomName.equalsIgnoreCase("HG13"))
    			secondAtomName = "HG1";
    	if(secondResName.equalsIgnoreCase("VAL") )
    		if( secondAtomName.equalsIgnoreCase("HG21")||secondAtomName.equalsIgnoreCase("HG22")||secondAtomName.equalsIgnoreCase("HG23"))
    			secondAtomName = "HG2";
    	if(secondResName.equalsIgnoreCase("THR") )
    		if (secondAtomName.equalsIgnoreCase("HG21")||secondAtomName.equalsIgnoreCase("HG22")||secondAtomName.equalsIgnoreCase("HG23"))
    			secondAtomName = "HG2";
    	if(secondResName.equalsIgnoreCase("MET") )
    		if (secondAtomName.equalsIgnoreCase("HE1")||secondAtomName.equalsIgnoreCase("HE2")||secondAtomName.equalsIgnoreCase("HE3"))
    			secondAtomName = "HE";    	    	
    	if(secondResName.equalsIgnoreCase("PHE") )
    	{
    		if (secondAtomName.equalsIgnoreCase("HE1")||secondAtomName.equalsIgnoreCase("HE2"))
    			secondAtomName = "HE";   
    		if (secondAtomName.equalsIgnoreCase("HD1")||secondAtomName.equalsIgnoreCase("HD2"))
    			secondAtomName = "HD";
    	}
    	if(secondResName.equalsIgnoreCase("TYR") )
    	{
    		if (secondAtomName.equalsIgnoreCase("HE1")||secondAtomName.equalsIgnoreCase("HE2"))
    			secondAtomName = "HE";   
    		if (secondAtomName.equalsIgnoreCase("HD1")||secondAtomName.equalsIgnoreCase("HD2"))
    			secondAtomName = "HD";
    	}
    	
       	//////--------
    	//for methyl group protons, need to be consistent with resonance list
    	if(firstResName.equalsIgnoreCase("ALA") )
    		if(firstAtomName.equalsIgnoreCase("HB1")||firstAtomName.equalsIgnoreCase("HB2")||firstAtomName.equalsIgnoreCase("HB3"))
    			firstAtomName = "HB";
    	if(firstResName.equalsIgnoreCase("ILE")  )
    		if (firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    			firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("ILE"))
    		if (firstAtomName.equalsIgnoreCase("HD11")||firstAtomName.equalsIgnoreCase("HD12")||firstAtomName.equalsIgnoreCase("HD13"))
    			firstAtomName = "HD1";
    	if(firstResName.equalsIgnoreCase("LEU") )
    		if (firstAtomName.equalsIgnoreCase("HD11")||firstAtomName.equalsIgnoreCase("HD12")||firstAtomName.equalsIgnoreCase("HD13"))
    		firstAtomName = "HD1";
    	if(firstResName.equalsIgnoreCase("LEU"))
    		if(firstAtomName.equalsIgnoreCase("HD21")||firstAtomName.equalsIgnoreCase("HD22")||firstAtomName.equalsIgnoreCase("HD23"))
    		firstAtomName = "HD2";
    	if(firstResName.equalsIgnoreCase("VAL"))
    		if(firstAtomName.equalsIgnoreCase("HG11")||firstAtomName.equalsIgnoreCase("HG12")||firstAtomName.equalsIgnoreCase("HG13"))
    		firstAtomName = "HG1";
    	if(firstResName.equalsIgnoreCase("VAL") )
    		if( firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    		firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("THR") )
    		if (firstAtomName.equalsIgnoreCase("HG21")||firstAtomName.equalsIgnoreCase("HG22")||firstAtomName.equalsIgnoreCase("HG23"))
    		firstAtomName = "HG2";
    	if(firstResName.equalsIgnoreCase("MET") )
    		if (firstAtomName.equalsIgnoreCase("HE1")||firstAtomName.equalsIgnoreCase("HE2")||firstAtomName.equalsIgnoreCase("HE3"))
    		firstAtomName = "HE";    	    	
    	if(firstResName.equalsIgnoreCase("PHE") )
    	{
    		if (firstAtomName.equalsIgnoreCase("HE1")||firstAtomName.equalsIgnoreCase("HE2"))
    			firstAtomName = "HE";   
    		if (firstAtomName.equalsIgnoreCase("HD1")||firstAtomName.equalsIgnoreCase("HD2"))
    			firstAtomName = "HD";
    	}
    	if(firstResName.equalsIgnoreCase("TYR") )
    	{
    		if (firstAtomName.equalsIgnoreCase("HE1")||firstAtomName.equalsIgnoreCase("HE2"))
    			firstAtomName = "HE";   
    		if (firstAtomName.equalsIgnoreCase("HD1")||firstAtomName.equalsIgnoreCase("HD2"))
    			firstAtomName = "HD";
    	}
    	
    	
    	double dist=hdist.getDistance();
    	
    	double csH1=-999.9;
    	double csHeavy=-999.9;
    	double csH2=-999.9;
    	
    	double csValue = 0.0;
    	String nucleus = "";
    	double csSigmaH=0.0;
    	double csSigmaN=0.0;
    	double csSigmaC=0.0;
    	double csSigmaHeavy=0.0;
    	
    	for (int j=0;j<asgVec.size();j++)
    	{
    		assign =  (H1CS)asgVec.elementAt(j);
    		no = assign.getResidueNo();
    	    aaType = assign.getResidueType();
    	    csValue=assign.getH1CS();
    	    nucleus=assign.getAtomName();
    	    if(nucleus.equalsIgnoreCase("H"))
    	    	nucleus="HN";
    	    if (firstResNo==no && firstAtomName.equals(nucleus) )
    		{    		
    			csH2=csValue+ csSigmaH* rr.nextGaussian(); 
    		}//if (firstResNo==no && firstAtomName==nucleus)
    			
    		if (secondResNo==no && heavyName.equals(nucleus))
    		{
    			if (nucleus.substring(0,1).equalsIgnoreCase("N"))
    				csSigmaHeavy=csSigmaN;
    			else
    				csSigmaHeavy=csSigmaC;
    			
    			csHeavy=csValue+csSigmaHeavy*rr.nextGaussian();
    		}

    		if (secondResNo==no && secondAtomName.equals(nucleus))
    		{
    			csH1=csValue+csSigmaH*rr.nextGaussian();
    		}
    			
    		  
    	}//for (i=0;i<asgVec.size();i++)
    	
    	//be careful of the order, here the first res is HN/HA, the second one is in rotamer 
    	BackNoe backNoe=new BackNoe(csH1, csHeavy,csH2,dist, 0.0,secondResNo,firstResNo,
    			secondResName,firstResName,secondAtomName,heavyName,firstAtomName);
    	    	
    	return backNoe;    	
    }


    /**
     * Back compute all NOE peaks from given pairs of NOE distance patterns for a rotamer.
     * 
     * @param asgVec resonance assignment
     * @param resNo residue No of the rotamer
     * @param resName  residue name of the rotamer
     * @param vecHDist  all pairs of NOE distances from rotamer protons to backbone protons, namely HN/HA/HB
     * 
     * @return the HNOE vector that stores all pairs of bakc-computed NOE peaks
     */
    public Vector BackCompNoePattern(final Vector asgVec, final int resNo,final String resName, final Vector vecHDist)
    {
    	//currently we use class HnNoe to store all NOE peaks, including Ha NOE and HN NOE
    	Vector vecBackNoe=new Vector();
    	int i,j;
    	Hdist hdist;
    	double distance;
    	//int label;//1 for very strong, 2 for strong, 3 for medium, 4 for weak, 5 for very weak.
    	//very strong (1.8-2.5 A), strong (1.8-3.1 A), medium(1.8-3.4A),
    	//weak (1.8-3.9A), and very weak (1.8-5.0A).    	
    	for (i=0;i<vecHDist.size();i++)
    	{
    		hdist=(Hdist)vecHDist.elementAt(i);
    		distance=hdist.getDistance();    		
    		
    		BackNoe backNoe=BackCompNOEPeak(asgVec,hdist);    		
    		vecBackNoe.add(backNoe);
    		
    	}//for (i=0;i<vecHDist.size();i++)
    	
    	
    	return vecBackNoe;
    
    }
    
    /**
     * Back compute all NOE peaks from given pairs of NOE distance patterns for a rotamer.
     * 
     * @param asgVec resonance assignment
     * @param resNo residue No of the rotamer
     * @param resName  residue name of the rotamer
     * @param vecHDist  all pairs of NOE distances from rotamer protons to backbone protons, namely HN/HA/HB
     * 
     * @return the HNOE vector that stores all pairs of bakc-computed NOE peaks
     */
    public Vector<BackNoe> BackCompNoePatternNew(final Vector<H1CS> asgVec, final Vector<Hdist> vecHDist)
    {
    	//currently we use class HnNoe to store all NOE peaks, including Ha NOE and HN NOE
    	Vector vecBackNoe=new Vector();
    	int i,j;
    	Hdist hdist;
    	double distance;
    	//int label;//1 for very strong, 2 for strong, 3 for medium, 4 for weak, 5 for very weak.
    	//very strong (1.8-2.5 A), strong (1.8-3.1 A), medium(1.8-3.4A),
    	//weak (1.8-3.9A), and very weak (1.8-5.0A).    	
    	for (i=0;i<vecHDist.size();i++)
    	{
    		hdist=(Hdist)vecHDist.elementAt(i);
    		distance=hdist.getDistance();    		
    		
    		BackNoe backNoe=BackCompNOEPeakNew(asgVec,hdist);    		
    		vecBackNoe.add(backNoe);
    		
    	}//for (i=0;i<vecHDist.size();i++)
    	
    	
    	return vecBackNoe;
    
    }

}
