/*
 * Decompiled with CFR 0.152.
 */
package smile.clustering;

import java.io.Serializable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.clustering.KMeans;
import smile.math.Math;
import smile.math.matrix.DenseMatrix;
import smile.math.matrix.EVD;
import smile.math.matrix.Matrix;

public class SpectralClustering
implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final Logger logger = LoggerFactory.getLogger(SpectralClustering.class);
    private int k;
    private int[] y;
    private int[] size;
    private double sigma;
    private double distortion;

    public SpectralClustering(double[][] W, int k) {
        if (k < 2) {
            throw new IllegalArgumentException("Invalid number of clusters: " + k);
        }
        this.k = k;
        int n = W.length;
        for (int i = 0; i < n; ++i) {
            if (W[i].length != n) {
                throw new IllegalArgumentException("The adjacency matrix is not square.");
            }
            if (W[i][i] != 0.0) {
                throw new IllegalArgumentException(String.format("Vertex %d has self loop: ", i));
            }
            for (int j = 0; j < i; ++j) {
                if (W[i][j] != W[j][i]) {
                    throw new IllegalArgumentException("The adjacency matrix is not symmetric.");
                }
                if (!(W[i][j] < 0.0)) continue;
                throw new IllegalArgumentException("Negative entry of adjacency matrix: " + W[i][j]);
            }
        }
        double[] D = new double[n];
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < n; ++j) {
                int n2 = i;
                D[n2] = D[n2] + W[i][j];
            }
            if (D[i] == 0.0) {
                throw new IllegalArgumentException("Isolated vertex: " + i);
            }
            D[i] = 1.0 / Math.sqrt((double)D[i]);
        }
        DenseMatrix L = Matrix.zeros((int)n, (int)n);
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < i; ++j) {
                double l = D[i] * W[i][j] * D[j];
                L.set(i, j, l);
                L.set(j, i, l);
            }
        }
        L.setSymmetric(true);
        EVD eigen = L.eigen(k);
        double[][] Y = eigen.getEigenVectors().array();
        for (int i = 0; i < n; ++i) {
            Math.unitize2((double[])Y[i]);
        }
        KMeans kmeans = new KMeans(Y, k);
        this.distortion = kmeans.distortion;
        this.y = kmeans.getClusterLabel();
        this.size = kmeans.getClusterSize();
    }

    public SpectralClustering(double[][] data, int k, double sigma) {
        if (k < 2) {
            throw new IllegalArgumentException("Invalid number of clusters: " + k);
        }
        if (sigma <= 0.0) {
            throw new IllegalArgumentException("Invalid standard deviation of Gaussian kernel: " + sigma);
        }
        this.k = k;
        this.sigma = sigma;
        int n = data.length;
        double gamma = -0.5 / (sigma * sigma);
        DenseMatrix W = Matrix.zeros((int)n, (int)n);
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < i; ++j) {
                double w = Math.exp((double)(gamma * Math.squaredDistance((double[])data[i], (double[])data[j])));
                W.set(i, j, w);
                W.set(j, i, w);
            }
        }
        double[] D = new double[n];
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < n; ++j) {
                int n2 = i;
                D[n2] = D[n2] + W.get(i, j);
            }
            if (D[i] < 1.0E-5) {
                logger.error(String.format("Small D[%d] = %f. The data may contain outliers.", i, D[i]));
            }
            D[i] = 1.0 / Math.sqrt((double)D[i]);
        }
        DenseMatrix L = W;
        for (int i = 0; i < n; ++i) {
            for (int j = 0; j < i; ++j) {
                double l = D[i] * W.get(i, j) * D[j];
                L.set(i, j, l);
                L.set(j, i, l);
            }
        }
        L.setSymmetric(true);
        EVD eigen = L.eigen(k);
        double[][] Y = eigen.getEigenVectors().array();
        for (int i = 0; i < n; ++i) {
            Math.unitize2((double[])Y[i]);
        }
        KMeans kmeans = new KMeans(Y, k);
        this.distortion = kmeans.distortion;
        this.y = kmeans.getClusterLabel();
        this.size = kmeans.getClusterSize();
    }

    public SpectralClustering(double[][] data, int k, int l, double sigma) {
        int i;
        if (l < k || l >= ((double[][])data).length) {
            throw new IllegalArgumentException("Invalid number of random samples: " + l);
        }
        if (k < 2) {
            throw new IllegalArgumentException("Invalid number of clusters: " + k);
        }
        if (sigma <= 0.0) {
            throw new IllegalArgumentException("Invalid standard deviation of Gaussian kernel: " + sigma);
        }
        this.k = k;
        this.sigma = sigma;
        int n = ((double[][])data).length;
        double gamma = -0.5 / (sigma * sigma);
        int[] index = Math.permutate((int)n);
        double[][] x = new double[n][];
        for (int i2 = 0; i2 < n; ++i2) {
            x[i2] = data[index[i2]];
        }
        data = x;
        DenseMatrix C = Matrix.zeros((int)n, (int)l);
        double[] D = new double[n];
        for (i = 0; i < n; ++i) {
            double sum = 0.0;
            for (int j = 0; j < n; ++j) {
                if (i == j) continue;
                double w = Math.exp((double)(gamma * Math.squaredDistance((double[])data[i], (double[])data[j])));
                sum += w;
                if (j >= l) continue;
                C.set(i, j, w);
            }
            if (sum < 1.0E-5) {
                logger.error(String.format("Small D[%d] = %f. The data may contain outliers.", i, sum));
            }
            D[i] = 1.0 / Math.sqrt((double)sum);
        }
        for (i = 0; i < n; ++i) {
            for (int j = 0; j < l; ++j) {
                C.set(i, j, D[i] * C.get(i, j) * D[j]);
            }
        }
        DenseMatrix W = Matrix.zeros((int)l, (int)l);
        for (int i3 = 0; i3 < l; ++i3) {
            for (int j = 0; j < l; ++j) {
                W.set(i3, j, C.get(i3, j));
            }
        }
        W.setSymmetric(true);
        EVD eigen = W.eigen(k);
        double[] e = eigen.getEigenValues();
        double scale = Math.sqrt((double)((double)l / (double)n));
        for (int i4 = 0; i4 < k; ++i4) {
            if (e[i4] <= 0.0) {
                throw new IllegalStateException("Non-positive eigen value: " + e[i4]);
            }
            e[i4] = scale / e[i4];
        }
        DenseMatrix U = eigen.getEigenVectors();
        for (int i5 = 0; i5 < l; ++i5) {
            for (int j = 0; j < k; ++j) {
                U.mul(i5, j, e[j]);
            }
        }
        double[][] Y = ((DenseMatrix)C.abmm((Object)U)).array();
        for (int i6 = 0; i6 < n; ++i6) {
            Math.unitize2((double[])Y[i6]);
        }
        KMeans kmeans = new KMeans(Y, k);
        this.distortion = kmeans.distortion;
        this.size = kmeans.getClusterSize();
        int[] label = kmeans.getClusterLabel();
        this.y = new int[n];
        for (int i7 = 0; i7 < n; ++i7) {
            this.y[index[i7]] = label[i7];
        }
    }

    public int getNumClusters() {
        return this.k;
    }

    public int[] getClusterLabel() {
        return this.y;
    }

    public int[] getClusterSize() {
        return this.size;
    }

    public double getGaussianKernelWidth() {
        return this.sigma;
    }

    public double distortion() {
        return this.distortion;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("Spectral Clustering distortion in feature space: %.5f%n", this.distortion));
        sb.append(String.format("Clusters of %d data points:%n", this.y.length));
        for (int i = 0; i < this.k; ++i) {
            int r = (int)Math.round((double)(1000.0 * (double)this.size[i] / (double)this.y.length));
            sb.append(String.format("%3d\t%5d (%2d.%1d%%)%n", i, this.size[i], r / 10, r % 10));
        }
        return sb.toString();
    }
}

