////////////////////////////////////////////////////////////////////////////////////////////
// SaveMolecule.java
//
//  Version:           0.1
//
//
// authors:
//    initials    name            organization                email
//   ---------   --------------  ------------------------    ------------------------------
//     RHL        Ryan Lilien     Dartmouth College           ryan.lilien@dartmouth.edu
//
////////////////////////////////////////////////////////////////////////////////////////////

/*
 *
 * This Function Rewritten by Ryan Lilien (2001-2004) as the function by Neill White
 *  had several errors and did not conform to the standard pdb format
 *
 * Rewritten by Ryan Lilien based on code by Neill White
 * Many functions have been added, others removed, most have had 
 *  at least some parts rewritten. Code rewrites have often been
 *  major to fix bugs or add functionality.
 * 
 * Based on software copyrighted, 1999, by Neill White. 
 *  The author hereby grants permission to use, copy, modify, and re-distribute
 *  this software and its documentation for any purpose, provided
 *  that existing copyright notices are retained in all copies and that this
 *  notice is included verbatim in any distributions. No written agreement,
 *  license, or royalty fee is required for any of the authorized uses.
 *  Modifications to this software may be distributed provided that
 *  the nature of the modifications are clearly indicated.
 *
 */

/*
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 Contact Info:
   Bruce Donald
   HB 6211
   Computer Science Department
   Dartmouth College
   Hanover, NH 03755
   brd@cs.dartmouth.edu

 If you use or publish any results derived from the use of this program please cite:
   Ryan H. Lilien, Brian W. Stevens, Amy C. Anderson,
   Bruce R. Donald. "A Novel Ensemble-Based Scoring and
   Search Algorithm for Protein Redesign, and its
   Application to Modify the Substrate Specificity of
   the Gramicidin Synthetase A Phenylalanine Adenylation
   Enzyme." Proc. The Eighth Annual International Conference
   on Research in Computational Molecular Biology (RECOMB),
   San Diego, pp 46-57 (2004). 

 Copyright (C) 2004  Ryan H. Lilien and Bruce R. Donald

 <signature of Bruce Donald>, 16 May, 2004
 Bruce Donald, Professor of Computer Science
*/

import java.io.PrintStream;
import java.io.BufferedInputStream;
import java.util.Hashtable;

// Saves a molecule to a pdb format file

class SaveMolecule {

	PrintStream pw;

	
	// Saves molecule m (atom[][] coordinates, not actualCoordinates[]) to the printstream pw. Params include
	//  comment: (string) a one line comment for header
	//  printSegID: (boolean) print atom seg ids
	//  showConnect: (boolean) add connect terms after atom entries
	SaveMolecule (Molecule inM, PrintStream pw, Hashtable params) throws Exception {
		Molecule m = inM;
		inM = null;
		m.resolveCoordinates(); //copy from actualCoordinates to atom coordinates
		DoSaveMolecule(m,pw,params);
	}

	private void DoSaveMolecule (Molecule m, PrintStream pw, Hashtable params) throws Exception {
		this.pw = pw;
		int atomCounter = 1;
		int residueCounter = 0;
		int prevResNum = -1;
		if (m.numberOfAtoms == 0)
			return;

		if (m.connectivity12Valid == false)
			m.establishConnectivity(true);
	
		// Pull out parameters
		String comment = (String) params.get("comment");
		boolean printSegID = ((Boolean) params.get("printSegID")).booleanValue();
		boolean showConnect = ((Boolean) params.get("showConnect")).booleanValue();
		float energy = ((Float) params.get("energy")).floatValue();

		m.resolveCoordinates();
		pw.println("AUTHOR generated by ILMM");
		pw.println("REMARK   6 " + comment);
		pw.println("REMARK   7 " + energy);

		char[] tmpChr = new char[80];
		String tmpStg;
		Integer tmpInt;
		int tmpLen;
		for(int q=0; q<80; q++)  // clear the array
			tmpChr[q]=' ';

		for(int i=0; i<m.numberOfStrands; i++){
			Strand strand = m.strand[i];
			if (strand.numberOfAtoms == 0)
				continue;

			if (i>0)
				pw.println("TER");

			// copy over the one character strand identifier
			tmpStg = strand.name.toUpperCase();
			tmpLen = tmpStg.length();
			if (tmpLen != 0)
				tmpStg.getChars(0,1,tmpChr,21);
			else
				tmpChr[21] = ' ';

			residueCounter = 0;

			for(int j=0; j<strand.numberOfResidues; j++){
				Residue residue = strand.residue[j];
				if (residue.numberOfAtoms == 0)
					continue;
				residueCounter++;

				tmpStg = residue.name.toUpperCase();
				tmpLen = 3-tmpStg.length();
				if (tmpLen <= 0)
					tmpStg.getChars(0,3,tmpChr,17);
				else {
					// from 17-19 (zero based)
					tmpStg.getChars(0,tmpStg.length(),tmpChr,17+tmpLen);
					for(int q=tmpStg.length();q<3;q++)  // add whitespace on left
						tmpChr[19-q]=' ';
				}
				tmpInt = new Integer(residue.getResNumber());
				tmpStg = tmpInt.toString();
				tmpLen = 4-tmpStg.length();
				if (tmpLen <= 0)
					tmpStg.getChars(0,4,tmpChr,22);
				else {
					// from 22-25 (zero based)
					tmpStg.getChars(0,tmpStg.length(),tmpChr,22+tmpLen);
					for(int q=tmpStg.length();q<4;q++)  // add whitespace on left
						tmpChr[25-q]=' ';
				}

				// If the residue number decreases in the middle
				//  of a strand change the strand name to "R"
//				if(tmpInt.intValue() < prevResNum){
//					tmpChr[21] = 'R';
//				}
				prevResNum = tmpInt.intValue();
				
				for(int k=0; k<residue.numberOfAtoms; k++){
					Atom atom = residue.atom[k];
					tmpStg = "ATOM  ";
					tmpStg.getChars(0,6,tmpChr,0);  // Copy over the ATOM term
					tmpStg = "1.00";
					tmpStg.getChars(0,4,tmpChr,56); // Make the occupancy 1.00
					tmpStg = "0.00";
					tmpStg.getChars(0,4,tmpChr,62); // and the temp factor 0.0
					
					tmpInt = new Integer(atomCounter);
					tmpStg = tmpInt.toString();
					tmpLen = 5-tmpStg.length();
					if (tmpLen <= 0)
						tmpStg.getChars(0,5,tmpChr,6);
					else {
						// from 6-10 (zero based)
						tmpStg.getChars(0,tmpStg.length(),tmpChr,6+tmpLen);
						for(int q=tmpStg.length();q<5;q++)  // add whitespace on left
							tmpChr[10-q]=' ';
					}
					atom.modelAtomNumber = atomCounter++;

					// writing the atom name is a little fuzzy, although the
					//  atom name is allocated columns 12-15(zero based), rasmol
					//  likes and other people essentially start with column 13
					//  leaving column 12 blank. So we only allow 3 characters for
					//  the atom name and it should be left justified
					//  unless the first character is a number in which case we
					//  start with column 12
					// there are also exceptions when the atom has a two letter
					//  element code

					tmpStg = getAtomField(atom);
					tmpStg.getChars(0,4,tmpChr,12);
										
					// Write the x coordinate
					tmpStg = coordinate(atom.coord[0]);
					tmpLen = 8-tmpStg.length();
					if (tmpLen < 0) {
						System.out.println("ERROR: coordinate exceeds pdb format size");
						tmpStg.getChars(0,8,tmpChr,30);
					}
					else {
						// from 30-37 (zero based)
						tmpStg.getChars(0,tmpStg.length(),tmpChr,30+tmpLen);
						for(int q=tmpStg.length();q<8;q++)  // add whitespace on left
							tmpChr[37-q]=' ';
					}

					// Write the y coordinate
					tmpStg = coordinate(atom.coord[1]);
					tmpLen = 8-tmpStg.length();
					if (tmpLen < 0) {
						System.out.println("ERROR: coordinate exceeds pdb format size");
						tmpStg.getChars(0,8,tmpChr,38);
					}
					else {
						// from 38-45 (zero based)
						tmpStg.getChars(0,tmpStg.length(),tmpChr,38+tmpLen);
						for(int q=tmpStg.length();q<8;q++)  // add whitespace on left
							tmpChr[45-q]=' ';
					}

					// Write the z coordinate
					tmpStg = coordinate(atom.coord[2]);
					tmpLen = 8-tmpStg.length();
					if (tmpLen < 0) {
						System.out.println("ERROR: coordinate exceeds pdb format size");
						tmpStg.getChars(0,8,tmpChr,46);
					}
					else {
						// from 46-53 (zero based)
						tmpStg.getChars(0,tmpStg.length(),tmpChr,46+tmpLen);
						for(int q=tmpStg.length();q<8;q++)  // add whitespace on left
							tmpChr[53-q]=' ';
					}
					if (printSegID) {
						tmpStg = atom.segID;
						tmpLen = 4-tmpStg.length();
						tmpStg.getChars(0,1,tmpChr,21); // Make the chain identifier the first segID char
						if (tmpLen < 0) {
							System.out.println("ERROR: segment id exceeds pdb format size");
							tmpStg.getChars(0,4,tmpChr,72);
						}
						else {
							// from 72-75 (zero based)
							tmpStg.getChars(0,tmpStg.length(),tmpChr,72);
							for(int q=(72+tmpStg.length());q<76;q++) {
								tmpChr[q]=' ';
							}
						}
					}
					
					pw.println(new String(tmpChr));
				}
			}
		}

		if (showConnect) {
			for (int i=0; i<m.numberOfAtoms; i++){
				int numberOfConnections = m.connected[i][0];
				if (numberOfConnections > 0)
					pw.print("CONECT");
				else 
					continue;
				printAtom(m.atom[ i ].modelAtomNumber);	
				for(int j=1; j<=numberOfConnections; j++){
					int bondedToAtomNumber = m.connected[i][j];
					Atom bondedToAtom = m.atom[bondedToAtomNumber];
					printAtom(bondedToAtom.modelAtomNumber);
				}
				pw.println();
			}
		}
		pw.println("END");
	}

	private void printAtom(int atomNumber){
		if (atomNumber < 10)
			pw.print("    " + atomNumber );
		else if (atomNumber < 100)
			pw.print("   " + atomNumber );
		else if (atomNumber < 1000)
			pw.print("  " + atomNumber);
		else if (atomNumber < 10000)
			pw.print(" " + atomNumber);
		else if (atomNumber < 100000)
			pw.print(atomNumber);
	}

	private String coordinate(float coord){
		if((coord<0.001) && (coord>-0.001))
			coord = 0.0f;
		String coordString = String.valueOf(coord);
		String intPart = " ";
		String floatPart = " ";
		String returnString = null;
		int radix = coordString.indexOf(".");
		if (radix != -1){
			intPart = coordString.substring(0, radix);
			if (radix != coordString.length())
				floatPart = coordString.substring(radix+1);
		}
		else 
			return(coordString);
		if (intPart.length() == 1)
			returnString = "   " + intPart;
		else if (intPart.length() == 2)
			returnString = "  " + intPart;
		else if (intPart.length() == 3)
			returnString = " " + intPart;
		else if (intPart.length() == 4)
			returnString = intPart;
		else if (intPart.length() > 4)
			returnString = intPart.substring(0, 3);
		returnString += ".";
		if (floatPart.length() == 1)
			returnString += floatPart + "00";
		else if (floatPart.length() == 2)
			returnString += floatPart + "0";
		else if (floatPart.length() == 3)
			returnString += floatPart;
		else if (floatPart.length() > 3)
			returnString += floatPart.substring(0, 3);
		return(returnString);
	}
	
	//Returns the atom name field
	private String getAtomField(Atom at){
		if(at.elementType.length()==1) {
			if (at.name.length() == 1)
				return(" " + at.name + "   ");
			else if (at.name.length() == 2){
				if ( (at.name.charAt(0)>='0') && (at.name.charAt(0)<='9') )
					return(at.name + "   ");
				else
					return(" " + at.name + "  ");
			}
			else if (at.name.length() == 3){
				if ( (at.name.charAt(0)>='0') && (at.name.charAt(0)<='9') )
					return(at.name + "  ");
				else
					return(" " + at.name + " ");
			}
			else // (at.name.length() > 3)
				return(at.name.substring(0, 4) + " ");
		}
		else
		{
			if (at.name.length() == 2)
				return(at.name + "   ");
			else if (at.name.length() == 3)
				return(at.name + "  ");
			else // (at.name.length() > 3)
				return(at.name.substring(0, 4) + " ");
		}
	}
}
