/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 * @author        Lincong Wang (2001-2005)
 * @email         wlincong@cs.dartmouth.edu
 * @organization  Dartmouth College
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */

/**
 * Description of the class
 * A class defining various global constants
 */
public class Const {

    /**
     * The default constructor is made private to make sure that this class cannot
     * be instantiated.
     */
    private Const() {
    }

    static final double cst = Math.PI / 180.0; //conversion constant for angle
    static final double eps = 1.0E-10;         //for comparing two float numbers
    //For the definition of these theta angles see our paper
    static final double theta1 = 29.1406 * cst;  //Angles mined from 23 untraHigh resolution X-ray structures.
    static final double theta3 = -20.935 * cst;  //theta3 should be minus c.f. its definition
    static final double theta5 = 26.8237 * cst;
    static final double alpha3 = - theta3;
    static final double theta6 = -0.7518 * cst;
    static final double theta7 = 29.1044 * cst;
    static final double alpha1 = 0.50 * Math.PI - theta7;
    static final double theta9 = -0.0021 * cst;
    static final double theta8 =0.00204369051213495 * cst;//a magic angle for computing precisely the coordinates
    static final double alpha2 =  0.50 * Math.PI - theta5;
    static final double alpha5 = theta1;
    static final double alpha6 = theta6;
    static final double alpha8 = theta9;
    static final double alpha9 = -0.0020436905160222014 * cst;
    static final double[] sevenAngles = new double[]{theta1,theta9,theta3, theta5, theta6, theta7, theta8};

    static final double dN2CA  = 1.458; // --                    //bond lengths for backbone vectors
    static final double dCA2CO = 1.525;
    static final double dCO2N  = 1.329;    
    static final double dCO2O  = 1.231;
    static final double dCA2CB = 1.531;
    static final double dN2H   = 1.042; // 1.020; C changed this on Jun 02, 09
    static final double dCA2HA = 1.119; // 1.090; C changed this on Jun 02, 09    
    
    static final double dCA2CA = 3.8100448409449443;//3.8102585214129507; //the distance between CA(i) and CA(i+1)
    static final double dCA2Nseq = 2.4333047472333475;//; //the distance between CA(i) and N(i+1);
    static final double dCA2NHseq = 2.556836222174780;//; //the distance between CA(i) and NH(i+1);
//     static final double dCO2NH = 2.055; //NH2CO, computed with interAngle = 119.1044;
    
    
    static final double dCO2NH = Math.sqrt(dCO2N * dCO2N + dN2H * dN2H + 2.0 * dCO2N * dN2H * Math.sin(theta7));
    //the plane angle between CO(i)->NH(i+1) vector and CO(i)->N(i+1) vector
    static final double theta10 = - Math.asin(dN2H * Math.sin(theta7+0.50*Math.PI) / dCO2NH ); // is MINUS
    static final double dN2COHBond   = 2.88;               //the distance across the H-bond
    static final double angleNCACOHA = 118.95951 * cst * 0.975; //angles for computing HA and CO atoms
    static final double angleNCAHA   = -19.4 * cst; // -19.4 * cst; 
    static final double deltaHA   = Math.PI - angleNCACOHA; // the dihedral angle
    static final double thetaHA   = -angleNCAHA;  //19.4 * cst; // the plane angle
    static final double angleCACOO   = 59.329 * cst;  
    static final double angleNCaCoCb = (360.0-122.6767595) * cst; //angles for computing HA and CO atoms
    static final double angleNCaCb   = -20.1866 * cst; // -19.4 * cst; 
    static final double deltaCB   = Math.PI - angleNCaCoCb; // the dihedral angle
    static final double thetaCB    = -angleNCaCb; // the plane angle
    static final double deltaO   = -0.7518 * cst; // the dihedral angle
    static final double thetaO   = 32.50530 * cst; // the plane angle

////////////////-------begin--------
// The following set of lines are commented out on Jun 11 due to a better implementation.
// {
    //Relative RDC strength
//    static final double protonGyroRatio = 26.75;
//    static final double carbonGyroRatio = 6.73;
//    static final double nitrogenGyroRatio = 2.71;
//    static final double noeLowerLimit = 1.90;
//    static final double noeUpLimit = 5.01;
//    static final double dnaLimit = 3.00;
//    static final double danLimit = 3.59122;
//    static final double dnnLimit = 4.50;
        
    //the relative strength of DD between different nuclei
    //static final double cahaRatio = 1.00;  //use as standard, 
    
//    static /*final*/ double cahaRatio;  //use as standard,
//    static /*final*/ double nhRatio; //  = Math.pow(dCA2HA /dN2H, 3) * (nitrogenGyroRatio /carbonGyroRatio); // for UBQz zJNb n
//    //static final double nhRatio   = 1.00; //-0.4785;//changed here for eta(based on the scaling factor)
//   // static final double nhRatio   = Math.pow(dCA2HA /dN2H, 3) * (nitrogenGyroRatio /carbonGyroRatio);
//    static /*final*/ double conRatio; //  = Math.pow(dCA2HA /dCO2N,3) * (nitrogenGyroRatio /protonGyroRatio);
//    static /*final*/ double conhRatio; // = Math.pow(dCA2HA /dCO2NH,3)* (nitrogenGyroRatio /carbonGyroRatio); // seems wrong!
//    static /*final*/ double cacoRatio; // =1.00; //Math.pow(rN2H / rCA2CO, 3)*(c13GyroRatio * c13GyroRatio / (n15GyroRatio * h1GyroRatio ));
//
//    static /*final*/ double cacbRatio; // = 1.00;
// }
////////////////-------end--------

    //Matrices used only for testing the degeneracy and computing orientations.
    //static Matrix mat = new Matrix(3,3);
    static final double [][] signInv = {{-1.0, 0.0, 0.0},{0.0, -1.0, 0.0},{0.0, 0.0, -1.0}};//invert coordinates
    static final double [][] xyInv   = {{-1.0, 0.0, 0.0},{0.0, -1.0, 0.0},{0.0, 0.0,  1.0}}; //invert x,y direction
    static final double [][] xzInv   = {{-1.0, 0.0, 0.0},{0.0,  1.0, 0.0},{0.0, 0.0, -1.0}}; //invert x,z direction
    static final double [][] yzInv   = {{ 1.0, 0.0, 0.0},{0.0, -1.0, 0.0},{0.0, 0.0, -1.0}}; //invert y,z direction
    static final double [][] xInv    = {{-1.0, 0.0, 0.0}, {0.0, 1.0, 0.0},{0.0, 0.0,  1.0}}; //invert x direction
    static final double [][] yInv    = {{1.0, 0.0, 0.0}, {0.0, -1.0, 0.0},{0.0, 0.0,  1.0}}; //invert Y direction
    static final double [][] zInv    = {{1.0, 0.0, 0.0},  {0.0, 1.0, 0.0},{0.0, 0.0, -1.0}}; //invert Z direction

    static final Matrix zMat  = new Matrix(zInv);
    static final Matrix xyMat = new Matrix(xyInv);
    static final Matrix xzMat = new Matrix(xzInv);
    static final Matrix yzMat = new Matrix(yzInv);
    static Matrix[] mat4ThreeDirs = new Matrix[]{xyMat, xzMat, yzMat};
    static final Matrix signMat = new Matrix(signInv);
    static final Matrix r1x = Matrix.rotationMat(theta1, "+x");
    static final Matrix r9y = Matrix.rotationMat(theta9, "+y");
    static final Matrix r9y1x = r9y.times(r1x);
    static final Matrix r8z = Matrix.rotationMat(theta8, "+z");
    static final Matrix r3x = Matrix.rotationMat(theta3, "+x");
    static final Matrix r5x = Matrix.rotationMat(theta5, "+x");
    static final Matrix r6y = Matrix.rotationMat(theta6, "+y");
    static final Matrix r7x = Matrix.rotationMat(theta7, "+x");
    static final Matrix r10x = Matrix.rotationMat(theta10, "+x");
    static final Matrix r7x6y5x = r7x.times(r6y.times(r5x));

    static final Matrix r1xAlpha = Matrix.rotationMat(alpha1, "+x");
    static final Matrix r2xAlpha = Matrix.rotationMat(alpha2, "+x");
    static final Matrix r3xAlpha = Matrix.rotationMat(alpha3, "+x");
    static final Matrix r5xAlpha = Matrix.rotationMat(alpha5, "+x");
    static final Matrix r6zAlpha = Matrix.rotationMat(alpha6, "+z");
    static final Matrix r8yAlpha = Matrix.rotationMat(alpha8, "+y");
    static final Matrix r9zAlpha = Matrix.rotationMat(alpha9, "+z");
    static final Matrix r8y3xAlpha = r8yAlpha.times(r3xAlpha); 
    static final Matrix r2x6z1x9zAlpha = r2xAlpha.times(r6zAlpha.times(r1xAlpha.times(r9zAlpha))); 

    //Rotation Matrices used to compute HA and O coordinates
    static final Matrix rHA1 = Matrix.rotationMat(angleNCACOHA,"+y");//theta_{41}=angleNCACOHA=PhiPsiCalPDB(caToCOVec,nToCAVec,caToHAVec);
    static final Matrix rHA2 = Matrix.rotationMat(angleNCAHA,  "+x");//theta_{42}=angleNCAHA=interAngle(nToCAVec,caToHAVec)-Math.PI/2;
    static final Matrix rHA2HA1 = rHA2.times(rHA1);
    static final Matrix rO   = Matrix.rotationMat(angleCACOO,  "+x");//angleCACOO = interAngle(caToCOVec,coToOVec);

    static final Matrix matDeltaHA = Matrix.rotationMat(deltaHA,"+y");//theta_{41}=angleNCACOHA=PhiPsiCalPDB(caToCOVec,nToCAVec,caToHAVec);
    static final Matrix matThetaHA = Matrix.rotationMat(thetaHA, "+x");//theta_{42}=angleNCAHA=interAngle(nToCAVec,caToHAVec)-Math.PI/2;
    static final Matrix matHA = matThetaHA.times(matDeltaHA);

    //Rotation Matrices used to compute CB
    static final Matrix rCb1 = Matrix.rotationMat(angleNCaCoCb,"+y");//theta_{41}=angleNCACOHA=PhiPsiCalPDB(caToCOVec,nToCAVec,caToHAVec);
    static final Matrix rCb2 = Matrix.rotationMat(angleNCaCb,  "+x");//theta_{42}=angleNCAHA=interAngle(nToCAVec,caToHAVec)-Math.PI/2;
    static final Matrix rCb2Cb1 = rCb2.times(rCb1);
    static final Matrix matDeltaCB = Matrix.rotationMat(deltaCB,"+y");//theta_{41}=angleNCACOHA=PhiPsiCalPDB(caToCOVec,nToCAVec,caToHAVec);
    static final Matrix matThetaCB = Matrix.rotationMat(thetaCB, "+x");//theta_{42}=angleNCAHA=interAngle(nToCAVec,caToHAVec)-Math.PI/2;
    static final Matrix matDeltaO = Matrix.rotationMat(deltaO,"+z");
    static final Matrix matThetaO = Matrix.rotationMat(thetaO, "+x");
    static final Matrix matO = matThetaO.times(matDeltaO);

    //Inverse matrices
    static final Matrix rCb1Inv = rCb1.transpose();
    static final Matrix rCb2Inv = rCb2.transpose();
    static final Matrix rCb2Cb1Inv = rCb1Inv.times(rCb2Inv);
    static final Matrix rHA1Inv = rHA1.transpose();
    static final Matrix rHA2Inv = rHA2.transpose();
    static final Matrix rHA2HA1Inv = rHA1Inv.times(rHA2Inv);
    static final Matrix rOInv   = rO.transpose();    
    static final Matrix matDeltaHAInv = matDeltaHA.transpose();
    static final Matrix matThetaHAInv = matThetaHA.transpose();
    static final Matrix matHAInv = matDeltaHAInv.times(matThetaHAInv);
    static final Matrix matDeltaCBInv = matDeltaCB.transpose();
    static final Matrix matThetaCBInv = matThetaCB.transpose();
    static final Matrix matCBInv = matDeltaCBInv.times(matThetaCBInv);

    static final Matrix matDeltaOInv = matDeltaO.transpose();
    static final Matrix matThetaOInv = matThetaO.transpose();
    static final Matrix matOInv = matDeltaOInv.times(matThetaOInv);

    static final Matrix r1xInv  = r1x.transpose();
    static final Matrix r9yInv  = r9y.transpose();
    static final Matrix r1x9yInv  = r1xInv.times(r9yInv);
    static final Matrix r8zInv  = r8z.transpose();
    static final Matrix r8z1xInv  = r8zInv.times(r1xInv); //Const.r8zInv.times(Const.r1xInv.times(coordCA)));
    static final Matrix r3xInv  = r3x.transpose();
    static final Matrix r1x9y3xInv = r1xInv.times(r9yInv.times(r3xInv));
    static final Matrix r5xInv  = r5x.transpose();
    static final Matrix r6yInv  = r6y.transpose();
    static final Matrix r7xInv  = r7x.transpose();
    static final Matrix r10xInv = r10x.transpose();
    static final Matrix r7x6yInv = r6yInv.times(r7xInv);
    static final Matrix r7x6y5xInv = r5xInv.times(r6yInv.times(r7xInv)); //for computing NH direction
    static final Matrix r6y7x8z1xInv = r7x6yInv.times(r8zInv.times(r1xInv)); 
    static final Matrix r1x8z7x6y5xInv = r7x6y5xInv.times(r8zInv.times(r1xInv)); //for computing NCA direction

    static final Matrix r1xAlphaInv = r1xAlpha.transpose();
    static final Matrix r3xAlphaInv = r3xAlpha.transpose();
    static final Matrix r5xAlphaInv = r5xAlpha.transpose();
    static final Matrix r2xAlphaInv = r2xAlpha.transpose();
    static final Matrix r6zAlphaInv = r6zAlpha.transpose();
    static final Matrix r8yAlphaInv = r8yAlpha.transpose();
    static final Matrix r9zAlphaInv = r9zAlpha.transpose();
    static final Matrix r9z1xAlphaInv = r9zAlphaInv.times(r1xAlphaInv);
    static final Matrix r1x2z3xAlphaInv = r1xAlphaInv.times(r2xAlphaInv.times(r3xAlphaInv));
    static final Matrix matL = r1xInv.times(r9yInv);

    /** Vectors for convenience and speed **/
    static final double [] ca2coVec = {0.0, 0.0, dCA2CO}; //constants for computing phi,psi from CA direction
    static final double [] dirA  = r3xInv.times(ca2coVec);
    static final double [] co2nVec = {0.0, dCO2N, 0.0};
    static final double [] dirB1 = r5xInv.times(co2nVec);
    static final double [] coordCA = {0.0, dN2CA, 0.0};
    static final double [] dirB2 = r1x8z7x6y5xInv.times(coordCA);
    static final double [] dirB  = {dirB1[0] + dirB2[0],  dirB1[1] + dirB2[1],  dirB1[2] + dirB2[2]};
    static final double  cosPsiC = dirB[0] / Math.sqrt(dirB[0] * dirB[0] + dirB[1] * dirB[1]); //one number later
    static final double  sinPsiC = dirB[1] / Math.sqrt(dirB[0] * dirB[0] + dirB[1] * dirB[1]);
//     double  psiC1 = 0.0;
//     if ( cosPsiC >= 0.0 )
// 	psiC1 = Math.asin(sinPsiC);
//     else if ( cosPsi < 0)
// 	psiC1 = Math.PI-Math.asin(sinPsiC);
    static final double psiC = Math.PI - Math.asin(sinPsiC); //since ( cosPsi < 0)

    static final double [] dirCAcnt = r8z1xInv.times(coordCA); //a constant needed for the computing nNhCaCal
    static final double [] coordNH = {0.0, 0.0, -dN2H};       
    static final double [] coordNHCnt = {0.0, 0.0, -1.0};       
    static final double [] matRNH  = r5xInv.times(r6yInv.times(r7xInv.times(coordNHCnt)));
    static final double [] matR = r5xInv.times(r6yInv.times(r7xInv.times(r8zInv.times(r1xInv.times(coordCA)))));
    static final double [] unitZ   = new double[]{0, 0, 1};  //For computing the Matrix $M$ in the MATH file
    static final double [] dirCosCHcnt = rHA1Inv.times(rHA2Inv.times(unitZ));
    static final double [] unitMinusZ  = new double[]{0.0, 0.0, -1.0};
    static final double [] unitN2CA  = {0.0, Math.cos(theta1), Math.sin(theta1)};
    static final double [] dirNHcnt = r7x6yInv.times(coordNH);
    static final double [] dirCosNHcnt = r5xInv.times(r6yInv.times(r7xInv.times(unitMinusZ)));
    static final double [] unitCo2Nh = {0.0, dCO2NH, 0.0};
    static final double [] dirCO2NHcnt = r5xInv.times(r6yInv.times(r10xInv.times(unitCo2Nh)));
    static final double [] coordHA = {0.0, 0.0, dCA2HA};
    static final double [] cntHAVec = rHA2HA1Inv.times(coordHA);
    static final double [] phiCnt = {0.0, Math.sin(alpha5), Math.cos(alpha5)};  //a const vector for computing psi by backwards
    static final double [] psiCnt = r3xAlphaInv.times(matHAInv.times(coordHA));  //a const vector for computing psi by backwards
    static final double [] psiCntUnit = r3xAlphaInv.times(matHAInv.times(unitZ));//a const vector for computing psi by backwards

    // A set of constants for computing \phi and \psi angles from dan, dna
    static final double Cx = dCA2HA * dirCosCHcnt[0]; 
    static final double Cy = dCA2HA * dirCosCHcnt[1];
    static final double Cz = dCA2HA * dirCosCHcnt[2];
    static final double [][] phiMat = r1x9yInv.getArray();
    static final double a1 = Cz * phiMat[0][0] - Cx * phiMat[0][2];
    static final double a2 = Cz * phiMat[1][0] - Cx * phiMat[1][2];
    static final double a3 = Cz * phiMat[2][0] - Cx * phiMat[2][2];
    static final double b1 = Cx * phiMat[0][0] + Cz * phiMat[0][2];
    static final double b2 = Cx * phiMat[1][0] + Cz * phiMat[1][2];
    static final double b3 = Cx * phiMat[2][0] + Cz * phiMat[2][2];
    static final double c1 = Cy * phiMat[0][1];
    static final double c2 = Cy * phiMat[1][1] + dN2CA * Math.cos(theta1);
    static final double c3 = Cy * phiMat[2][1] + dN2CA * Math.sin(theta1) + dN2H;
    static final double e0 = Cx * Cx + Cz * Cz + c1*c1 + c2* c2 + c3 * c3;
    static final double e1 = 2* (b1 * c1 + b2 * c2 + b3 * c3);
    static final double e2 = 2* (a1 * c1 + a2 * c2 + a3 * c3);

    static final double [] caToHAVec = r1x9yInv.times(cntHAVec);
    static final double [] ha ={caToHAVec[0], caToHAVec[1] + dN2CA * Math.cos(theta1), caToHAVec[2] + dN2CA * Math.sin(theta1)};
    static final double [] caToCOVec = r1x9y3xInv.times(ca2coVec);
    static final double [] co ={caToCOVec[0], caToCOVec[1] + dN2CA * Math.cos(theta1), caToCOVec[2] + dN2CA * Math.sin(theta1)};
    static final double [][] psiMat = r1x9y3xInv.getArray();
    static final double Cxx = dirCO2NHcnt[0];
    static final double Cyy = dirCO2NHcnt[1];
    static final double Czz = dirCO2NHcnt[2];
    static final double a11 = Cyy * psiMat[0][0] - Cxx * psiMat[0][1];
    static final double a22 = Cyy * psiMat[1][0] - Cxx * psiMat[1][1];
    static final double a33 = Cyy * psiMat[2][0] - Cxx * psiMat[2][1];
    static final double b11 = -Cxx * psiMat[0][0] - Cyy * psiMat[0][1];
    static final double b22 = -Cxx * psiMat[1][0] - Cyy * psiMat[1][1];
    static final double b33 = -Cxx * psiMat[2][0] - Cyy * psiMat[2][1];
    static final double c11 = Czz * psiMat[0][2] + co[0] - ha[0];
    static final double c22 = Czz * psiMat[1][2] + co[1] - ha[1];
    static final double c33 = Czz * psiMat[2][2] + co[2] - ha[2];
    static final double ee0 = Cxx * Cxx + Cyy * Cyy + c11*c11 + c22* c22 + c33 * c33;
    static final double ee1 = 2* (b11 * c11 + b22 * c22 + b33 * c33);
    static final double ee2 = 2* (a11 * c11 + a22 * c22 + a33 * c33);

    /**
     * A set of constants for computing the side-chain conformation of Proline
     * These constants are just back-computed from Pro19 in 1UBQ,
     */
/* Commented out on May 27, 2009
 
    static final double dCa2Cb = 1.49;
    static final double dC2C   = 1.54;
    static final double thetaProCB  =  1.3317861674705895;
    static final double thetaProCBd = -2.093056454581993;
    static final Matrix rxCBInv  = Matrix.rotationMat(thetaProCB,  "-x");
    static final Matrix ryCBdInv = Matrix.rotationMat(thetaProCBd, "-y");
    static final Matrix rCBInv   = ryCBdInv.times(rxCBInv);

    static final double cbTheta1 = 1.3469915625340805;
    static final double cbTheta2 = 1.1816623139143054;
    static final double cbTheta3 = 1.2351203414182543;
    static final double chi1 = -0.5042748429337971;
    static final double cbTheta2d = 2.112251747644531;
    static final double cbTheta3d = -2.0196971725548947; 
    static final sp3Carbon proCB = new sp3Carbon(cbTheta1,cbTheta2,cbTheta3,chi1,cbTheta2d,cbTheta3d,dCa2Cb,dCA2HA,dCA2HA,false);

    static final double cgTheta1 = 1.3414373700743338; 
    static final double cgTheta2 = 1.2393003417313453;
    static final double cgTheta3 = 1.1826139579031945;
    static final double chi2 =     0.7316244067220934;
    static final double cgTheta2d = 2.023077294407706;
    static final double cgTheta3d = -2.1117884874529365; 
    static final sp3Carbon proCG = new sp3Carbon(cgTheta1,cgTheta2,cgTheta3,chi2,cgTheta2d,cgTheta3d,dC2C,dCA2HA,dCA2HA,false);

    static final double cdTheta1 = 1.3843470838350387;
    static final double cdTheta2 = 1.1883376662609455;
    static final double cdTheta3 = 1.2090038592514238;
    static final double chi3 = -0.6692454232754876;
    static final double cdTheta2d =  2.0937261165122902;
    static final double cdTheta3d = -2.0324030990577624; 
    static final sp3Carbon proCD = new sp3Carbon(cdTheta1,cdTheta2,cdTheta3,chi3,cdTheta2d,cdTheta3d,dC2C,dCA2HA,dCA2HA, true);
    //A set of constants mined from ultra-resolution PDBs from CB to other atoms of the same residue
    static final double cb2HG    = 2.06;
    static final double cb2HDLow = 2.71;
    static final double cb2HDUp  = 3.35;
    static final double cb2HD  = 3.35;
    static final double cb2HE  = 4.00;
    static final double cb2HDPhe = 2.67;
    static final double cb2HEPhe = 4.55;
    static final double cb2HZPhe = 5.00;
    static final double cb2HDTyr = 2.67;
    static final double cb2HETyr = 4.55;

Commented out on May 27, 2009 */

    /** A set of constants for computing the backbone \phi and \psi
     * angles when the orientation of the two peptide planes i, i+2
     * are known.
     */
    static final Matrix rzPiInv = Matrix.rotationMat(Math.PI, "-z");
    static final Matrix rot1 = rzPiInv.times(r7x6y5xInv.times(r8zInv)); 
    static final double[] n2caUnit = {0.0, Math.cos(theta1), Math.sin(theta1)};
    static final double[] Cu = rot1.times(n2caUnit);  //for n2ca vector
    static final double[] Cv = rot1.times(unitMinusZ);//for n2nh vector
    static final Matrix rot1Inv = rot1.transpose();
    static final Matrix matA = rot1.times(r1x9yInv);

    static final double[][] arrLeftHand = new double[][]{{-1, 0, 0}, {0,1,0},{0,0,1}}; //for changing  handedness
    public static final Matrix mLeftHand = new Matrix(arrLeftHand);

    static final double phiPro = -65.0 * cst;
    static final Matrix r2yPro =  Matrix.rotationMat(phiPro, "+y");
    static final Matrix r2yInvPro =  r2yPro.transpose(); 

    //The average  Phi/Psi for a typical helix or strand
    static final double phiAveHelix =  -65.3 * cst;  
    static final double psiAveHelix =  -39.4 * cst;
    static final double phiAveBeta  = -120.0 * cst; 
    static final double psiAveBeta  =  138.0 * cst; 

    //The favorable Ramachandran (Ram) regions
    static final double phiHighHelix =  -30.0 * cst;  //helix region
    static final double phiLowHelix  = -100.0 * cst;
    static final double psiHighHelix =  -15.0 * cst;
    static final double psiLowHelix  =  -90.0 * cst;
//     static final double phiHighHelix =  -55.0 * cst;  //helix region
//     static final double phiLowHelix  = -75.0 * cst;
//     static final double psiHighHelix =  -30.0 * cst;
//     static final double psiLowHelix  =  -50.0 * cst;

    static final double phiHighBeta =  -70.0 * cst;  //Beta region
    static final double phiLowBeta  = -170.0 * cst;
    static final double psiHighBeta =  180.0 * cst;
    static final double psiLowBeta  =   80.0 * cst;

    
    static final double gamma_C = 6728.3000000000002;
    static final double gamma_H = 26751.98;
    static final double gamma_N = 2711.5999999999999;
    static final double gamma_P = 10841.0;
    static final double r_cacb = 1.5249999999999999;
    static final double r_caco = 1.5249999999999999;
    static final double r_caha = 1.119;
    static final double r_cahn = 2.5600000000000001;
    static final double r_cohn = 2.069;
    static final double r_nco = 1.329;
    static final double r_nh = 1.0547428700640902;
    static final double r_nhn = 1.042;

    public static final double DmaxN_HN = gamma_N * gamma_H  ;
    public static void main(String... args) {
        System.out.println("Dmax(N-HN) = " + DmaxN_HN);
        System.out.println("dCO2NH: " + dCO2NH);
    }

}

