/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myInputDataAndParameterManager implements java.io.Serializable {

    private static String __master_directory = null;    
    private static String __directory_architecture_file = null; 
    
    private String __input_directory = null;
    private String __output_directory = null;
    private String __log_directory = null; 
    private String __input_data_directory = null; 
    private String __input_parameter_directory = null;

    private String __rdc_file_medium1 = null;
    private String __rdc_file_medium2 = null;
    private String __noe_file = null;
    private String __seq_file = null;    
    private String __talos_dihedral_restraint_file = null;    
    private String __sse_type_and_boundary_spec_file = null;
    
    private String __program_parameter_file = null;    
    private String __loop_configuration_file = null;

    private String __log_file = null;
    private String __best_fragment_pdb_file = null;
    private String __most_recent_fragment_pdb_file = null;
    
    /**
     * This is the only instance (singleton) of this class. For now, we designed
     * this class to be a singleton. Later the decision can be changed, if needed.
     */
    private static final myInputDataAndParameterManager __instance = new myInputDataAndParameterManager();
        
    /**
     * This method returns the only instance of the object of this type.
     *
     * @return the only instance of an object of this type
     */
    public static synchronized myInputDataAndParameterManager getInstance() {
        return __instance;
    }
    
    /**
     * This constructor is private to make sure that the class has a singleton object.
     */
    private myInputDataAndParameterManager() {           
    }
    
    private void myInputDataAndParameterManagerHelper() {
        if (__master_directory == null || __directory_architecture_file == null) {
            System.out.println("Error: the input data and parameter manager is not initialized with the master directory and the directory architecture file names");
            System.exit(1);
        }

        __input_directory = System.getProperty("user.dir") + System.getProperty("file.separator") +
                getMasterDirectory() + System.getProperty("file.separator") +
                getInputDirectory(__directory_architecture_file) + System.getProperty("file.separator");                
        
        if (!(new File(__input_directory).exists())) {
            System.out.println("Error: the directory " + __input_directory + " not present");
            System.exit(1);
        }
        
        __input_data_directory = __input_directory +
                getInputDataDirectory(__directory_architecture_file) + System.getProperty("file.separator");

        if (!(new File(__input_data_directory).exists())) {
            System.out.println("Error: the directory " + __input_data_directory + " not present");
            System.exit(1);
        }       
        
        __input_parameter_directory = __input_directory +
                getInputParameterDirectory(__directory_architecture_file) + System.getProperty("file.separator");
        
        if (!(new File(__input_parameter_directory).exists())) {
            System.out.println("Error: the directory " + __input_parameter_directory + " not present");
            System.exit(1);
        }          
        
        __output_directory = System.getProperty("user.dir") + System.getProperty("file.separator") +
                getMasterDirectory() + System.getProperty("file.separator") +
                getOutputDirectory(__directory_architecture_file) + System.getProperty("file.separator");
        
        if (!(new File(__output_directory).exists())) {
            System.out.println("The directory " + __output_directory+ " not present");
            System.out.println("Creating directory: " + __output_directory);            
            if (new File(__output_directory).mkdir()) {
                System.out.println("The directory: " + __output_directory + " is created"); 
            } else {
                System.out.println("Error: cannot create the directory: " + __output_directory);
                System.exit(1);
            }
        }          
        
        __log_directory = System.getProperty("user.dir") + System.getProperty("file.separator") +
                getMasterDirectory() + System.getProperty("file.separator") +
                getLogDirectory(__directory_architecture_file) + System.getProperty("file.separator");

        if (!(new File(__log_directory).exists())) {
            System.out.println("The directory " + __log_directory + " not present");
            System.out.println("Creating directory: " + __log_directory);            
            if (new File(__log_directory).mkdir()) {
                System.out.println("The directory: " + __log_directory + " is created"); 
            } else {
                System.out.println("Error: cannot create the directory: " + __log_directory);
                System.exit(1);
            }
        }        
        
        __rdc_file_medium1 = __input_data_directory + getRdcFileForMedium1(__directory_architecture_file);
        if (!getRdcFileForMedium2(__directory_architecture_file).equalsIgnoreCase("null")) {
            __rdc_file_medium2 = __input_data_directory + getRdcFileForMedium2(__directory_architecture_file);
        }

        __noe_file = __input_data_directory + getNoeFile(__directory_architecture_file);

        __seq_file = __input_data_directory + getSeqFile(__directory_architecture_file);
        
        __talos_dihedral_restraint_file = __input_data_directory + getTalosDihedralRestraintFile(__directory_architecture_file);                

        __sse_type_and_boundary_spec_file = __input_parameter_directory + getSSeInfoFile(__directory_architecture_file);
        
        __program_parameter_file = __input_parameter_directory + getProgramParameterFile(__directory_architecture_file);
        
        __loop_configuration_file = __input_parameter_directory + getLoopInfoFile(__directory_architecture_file); 
        
        __log_file = __log_directory + getLogFile(__directory_architecture_file);
        
        __best_fragment_pdb_file = __output_directory + getBestFragmentPdbFile(__directory_architecture_file);
        
        __most_recent_fragment_pdb_file = __log_directory + getMostRecentFragmentFileForALeafOfSearchTree(__directory_architecture_file);
    }

    public void setMasterDirectory(String masterDirectory) {        
        __master_directory = masterDirectory;
        if (getDirectoryArchitectureFile() != null) {
            myInputDataAndParameterManagerHelper();
        }
    }

    public String getMasterDirectory() {
        return __master_directory;
    }

    public void setDirectoryArchitectureFile(String dirArchFile) {
        if (getMasterDirectory() == null) {
            System.out.println("Error: The master directory for computation and I/O is not set");
            System.exit(1);
        }
        __directory_architecture_file = getMasterDirectory() + System.getProperty("file.separator") + dirArchFile;
        myInputDataAndParameterManagerHelper();
    }

    public String getDirectoryArchitectureFile() {
        return __directory_architecture_file;
    }

    /**
     * Return the name of the input directory.
     *
     * @return the name of the input directory
     */
    public static String getInputDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "inputDirectory");
    }
    
    public String getInputDirectory() {
        return __input_directory;
    }
    
    /**
     * Return the name of the output directory.
     *
     * @return the name of the output directory
     */
    public static String getOutputDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "outputDirectory");
    }
    
    public String getOutputDirectory() {
        return __output_directory;
    }    
    
    /**
     * Return the name of the log directory.
     *
     * @return the name of the log directory
     */
    
    public static String getLogDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "logDirectory");
    }
    
    public String getLogDirectory() {
        return __log_directory;
    }
              
    /**
     * Return the name of the input data directory.
     *
     * @return the name of the input data directory
     */
    public static String getInputDataDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "inputDataDirectory");
    }
    
    public String getInputDataDirectory() {
        return __input_data_directory;
    }       

    /**
     * Return the name of the input parameter directory.
     *
     * @return the name of the input parameter directory
     */
    public static String getInputParameterDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "inputParameterDirectory");
    }

    public String getInputParameterDirectory() {
        return __input_parameter_directory;
    }
    
    /**
     * Return the name of the program parameter file.
     *
     * @return the name of the program parameter file
     */
    public static String getProgramParameterFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "programParameterFile");
    }
    
    public String getProgramParameterFile() {
        return __program_parameter_file;
    }    

    /**
     * Return the name of the loop specification file.
     *
     * @return the name of the loop specification file
     */
    public static String getLoopInfoFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "loopConfigurationFile");
    }
    
    /**
     * Return the name of the loop configuration file.
     * 
     * @return the name of the loop configuration file
     */    
    public  String getLoopInfoFile() {
        return __loop_configuration_file;
    }    
    
    /**
     * Return the name of the secondary structure specification file.
     * 
     * @return the name of the secondary structure specification file
     */
    public static String getSSeInfoFile(String directoryArchitectureFile) {        
        return inputDirectoryHelper(directoryArchitectureFile, "SSeInfoFile");
    }
    
    public String getSSeInfoFile() {        
        return __sse_type_and_boundary_spec_file;
    }
    /**
     * Return the dipolar coupling data file name for medium 1.
     *
     * @return the dipolar coupling data file name for medium 1
     */
    public static String getRdcFileForMedium1(String directoryArchitectureFile) {        
        return inputDirectoryHelper(directoryArchitectureFile, "RDC and CSA Files in Medium1");
    }

    public String getRdcFileForMedium1() {        
        return __rdc_file_medium1;
    }    
    
    /**
     * Return the dipolar coupling data file name for medium 2.
     *
     * @return the dipolar coupling data file name for medium 2
     */
    public static String getRdcFileForMedium2(String directoryArchitectureFile) {        
        return inputDirectoryHelper(directoryArchitectureFile, "RDC and CSA Files in Medium2");
    }
    
    public String getRdcFileForMedium2() {        
        return __rdc_file_medium2;
    }        
    
    /**
     * Return the NOE restraint file name
     * 
     * @param directoryArchitectureFile the master file containing the name of the NOE restraint file
     * 
     * @return the NOE restraint file name
     */
    public static String getNoeFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "noeRestraintFile");
    }
    
    /**
     * Return the NOE data file.
     *
     * @return the NOE data file
     */
    public String getNoeFile() {        
        return __noe_file;
    }    
    
    /**
     * Return the sequence file name
     * 
     * @param directoryArchitectureFile the master file containing the name of the sequence file
     * 
     * @return the sequence file name
     */
    public static String getSeqFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "sequenceFile");
    }
    
    /**
     * Return the sequence data file.
     *
     * @return the sequence data file
     */
    public String getSeqFile() {        
        return __seq_file;
    }    
    
    /**
     * Return the TALOS+ dihedral restraint file name
     * 
     * @param directoryArchitectureFile the master file containing the name of the TALOS dihedral restraint file
     * 
     * @return the TALOS+ dihedral restraint file name
     */
    public static String getTalosDihedralRestraintFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "dihedralRestraintFile");
    }    
    
    /**
     * Return the name of the TALOS dihedral restraint file.
     *
     * @return the name of the TALOS dihedral restraint file
     */
    public String getTalosDihedralRestraintFile() {
        return __talos_dihedral_restraint_file;
    }     
    
    /**
     * Return the solution tree log file restraint file name
     * 
     * @param directoryArchitectureFile the master file containing the name of the TALOS dihedral restraint file
     * 
     * @return the solution tree log file name
     */    
    public static String getLogFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "solutionTreeLogFile");
    }     
        
    /**
     * Return the name of the solution tree log file.
     * 
     * @return the name of the solution tree log file
     */
    public String getLogFile() {
        return __log_file;
    }
    
    /**
     * Return the best fragment pdb file name
     * 
     * @param directoryArchitectureFile the master file containing the name of the TALOS dihedral restraint file
     * 
     * @return the best fragment pdb file name
     */    
    public static String getBestFragmentPdbFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "bestFragmentPdbFile");
    }    
    
    /**
     * Return the name of the best fragment pdb file name.
     * 
     * @return the name of the best fragment pdb file name
     */
    public String getBestFragmentPdbFile() {
        return __best_fragment_pdb_file;
    } 
    
    public static String getMostRecentFragmentFileForALeafOfSearchTree(String directoryArchitectureFile) { 
        return inputDirectoryHelper(directoryArchitectureFile, "mostRecentFragmentCorrespondingToALeaf");
    }
    
    public String getMostRecentFragmentFileForALeafOfSearchTree() {
        return __most_recent_fragment_pdb_file;
    }
        
        
    /**
     * Read the file whose name is supplied as the first argument, and return the
     * string that corresponds to the tag supplied as the second argument.
     *
     * @param directoryArchitectureFile the file depicting the input directory structure
     * @param description the tag for a particular type of file/directory
     * @return the string that corresponds to the tag
     */
    private static String inputDirectoryHelper(String directoryArchitectureFile, String description) {
        String directoryOrFileName = null;

        try {
            Scanner scanner = new Scanner(new File(directoryArchitectureFile));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (line.length() != 0) {
                    String[] words = line.split(":");

                    if (words[0].equalsIgnoreCase(description) && words.length == 2) {
                        directoryOrFileName = words[1].trim();
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + directoryArchitectureFile + " not found");
            e.printStackTrace();
        } 

        if (directoryOrFileName == null) {
            System.out.println("Error: In the file " + directoryArchitectureFile + " the " + description + " is not found");
            System.exit(1);
        }

        return directoryOrFileName;
    }

    /**
     * Return the dipolar coupling table constructed after reading the RDC files.
     *
     * @return the dipolar coupling table
     */
    public myDipolarCouplingTable getDipolarCouplingTable() {
        if (__rdc_file_medium2 == null) {
            return new myDipolarCouplingTable(__rdc_file_medium1);
        }
        return new myDipolarCouplingTable(__rdc_file_medium1, __rdc_file_medium2);
    }

    public Map<String, Map<String, String>> parseParameterFile() {
        return parseParameterFile(__program_parameter_file);
    }

    /**
     * Parse the parameter file and extract the parameters for alignment tensor,
     * helix and sheet computation.
     *
     * @param parameterFileName the parameter file name
     * @return a table of parameters extracted from the parameter file
     */
    private Map<String, Map<String, String>> parseParameterFile(String parameterFileName) {
        Map<String, Map<String, String>> params = new TreeMap<String, Map<String, String>>();
        try {
            Scanner scanner = new Scanner(new File(parameterFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));

            Vector<String> block = new Vector<String>();

            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.startsWith("@parametersForAlignmentTensorComputation")) {
                    block.clear();
                    block.add(line);
                    while (scanner.hasNextLine()) {
                        line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                        block.add(line);
                        if (line.equalsIgnoreCase("}")) {
                            break;
                        }
                    }
                    Map<String, String> parametersForAlignmentTensorComputation = new TreeMap<String, String>();
                    for (int i = 1; i < block.size() - 1; i++) {
                        String currLine = block.elementAt(i);
                        int index = currLine.indexOf("=");
                        String key = currLine.substring(0, index).trim();
                        String value = currLine.substring(index + 1, currLine.length()).trim();
                        parametersForAlignmentTensorComputation.put(key, value);
                    }
                    params.put("parametersForAlignmentTensorComputation", parametersForAlignmentTensorComputation);
                    block.clear();
                } else if (line.startsWith("@parametersForHelixComputation")) {
                    block.clear();
                    block.add(line);
                    while (scanner.hasNextLine()) {
                        line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                        block.add(line);
                        if (line.equalsIgnoreCase("}")) {
                            break;
                        }
                    }
                    Map<String, String> parametersForHelixComputation = new TreeMap<String, String>();
                    for (int i = 1; i < block.size() - 1; i++) {
                        String currLine = block.elementAt(i);
                        int index = currLine.indexOf("=");
                        String key = currLine.substring(0, index).trim();
                        String value = currLine.substring(index + 1, currLine.length()).trim();
                        parametersForHelixComputation.put(key, value);
                    }
                    params.put("parametersForHelixComputation", parametersForHelixComputation);
                    block.clear();
                } else if (line.startsWith("@parametersForStrandComputation")) {
                    block.clear();
                    block.add(line);
                    while (scanner.hasNextLine()) {
                        line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                        block.add(line);
                        if (line.equalsIgnoreCase("}")) {
                            break;
                        }
                    }
                    Map<String, String> parametersForStrandComputation = new TreeMap<String, String>();
                    for (int i = 1; i < block.size() - 1; i++) {
                        String currLine = block.elementAt(i);
                        int index = currLine.indexOf("=");
                        String key = currLine.substring(0, index).trim();
                        String value = currLine.substring(index + 1, currLine.length()).trim();
                        parametersForStrandComputation.put(key, value);
                    }
                    params.put("parametersForStrandComputation", parametersForStrandComputation);
                    block.clear();
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + parameterFileName + " not found");
            e.printStackTrace();
        } 

        return (params.size() == 0) ? null : params;
    }

    /**
     * This method parses the secondary structure element type and boundary
     * specifications and returns a vector of secondary structure specifications.
     *
     * @return a vector of secondary structure specifications
     */
    public Vector<mySseInfo> parseSseIdFile() {
        mySseInfo sseIds = new mySseInfo();
        Vector<mySseInfo> idVec = sseIds.parseSseIds(__sse_type_and_boundary_spec_file);
        return idVec;
    }

    /**
     * This method parses the secondary structure element type and boundary
     * specifications and returns the beta sheet specifications.
     *
     * @return beta sheet specifications
     */
    public Vector<Vector<myTriple<mySseInfo, mySseInfo, Boolean>>> parseSseIdFileAndReturnSheetSpecifications() {
        mySseInfo sseIds = new mySseInfo();
        return sseIds.parseSseIdFileAndReturnSheetSpecifications(__sse_type_and_boundary_spec_file);
    }

    /**
     * This method parses the secondary structure element type and boundary
     * specifications and returns the SSE specification to be used for the
     * alignment tensor computation.
     *
     * @return the SSE specification to be used for the alignment tensor computation
     */
    public mySseInfo parseSseIdFileAndReturnSseForAlignmentTensorComputation() {
        mySseInfo sseIds = new mySseInfo();
        return sseIds.parseSseIdFileAndReturnSseForAlignmentTensorComputation(__sse_type_and_boundary_spec_file);
    }
    
    /**
     * This method parses the loop specifications and configuration parameters and
     * returns them as a vector of arrays of strings where each string represents 
     * a parameter (which need to be cast to the right type for further use)
     *
     * @return the parameters of the loops as a vector of arrays of strings where
     * each string represents a parameter (which need to be cast to the right type
     * for further use)
     */
    public Vector<String[]> parseLoopConfigurationFileAndReturnLoopAndParameterSpecifications() {

        return parseLoopConfigurationFile(__loop_configuration_file);
    }
    
    
    /**
     * This method parses the loop configuration file and returns both the
     * specifications of the loops and the set of parameters to compute them.
     *
     * @param inputLoopConfigurationFileName the name of the loop configuration
     * file
     * @return the parameters of the loops as a vector of arrays of strings
     * where each string represents a parameter (which need to be cast to the
     * right type for further use)
     */
    
    private Vector<String[]> parseLoopConfigurationFile(String inputLoopConfigurationFileName) {
        String parameterFileName = inputLoopConfigurationFileName;
        Vector<String[]> block = new Vector<String[]>();

        try {
            Scanner scanner = new Scanner(new File(parameterFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            //String patternString = "\\s*(@loop)\\s*\\([0-9]+\\s*\\,\\s*[0-9]+\\s*\\,";

            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.startsWith("@loop")) {
                    line = line.replaceAll("\\(|\\)|\\,", " ");
                    String[] words = line.split("\\s+");
                    for (String s : words) {
                        System.out.print(s + " ");
                    }
                    System.out.println();
                    block.add(words);
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + parameterFileName + " not found");
            e.printStackTrace();
        } 

        return (block.size() == 0) ? null : block;
    }
    
    
    public myPair<Integer, Integer> parseLoopConfigurationFileAndReturnWhichLoopToBeComputedNow() {
        String[] whichLoopToBeComputedNow = parseLoopConfigurationFileAndReturnWhichLoopToBeComputedNow(__loop_configuration_file);        
        if (whichLoopToBeComputedNow != null) {
            if (whichLoopToBeComputedNow.length == 3) {
                return new myPair<Integer, Integer>(Integer.parseInt(whichLoopToBeComputedNow[1]), Integer.parseInt(whichLoopToBeComputedNow[2]));
            }
        }        
        return null;
    }

    private String[] parseLoopConfigurationFileAndReturnWhichLoopToBeComputedNow(String inputLoopConfigurationFileName) {
        String parameterFileName = inputLoopConfigurationFileName;
        Vector<String[]> block = new Vector<String[]>();

        try {
            Scanner scanner = new Scanner(new File(parameterFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            //String patternString = "\\s*(@loop)\\s*\\([0-9]+\\s*\\,\\s*[0-9]+\\s*\\,";

            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.startsWith("@computeNow")) {
                    line = line.replaceAll("\\(|\\)|\\,", " ");
                    String[] words = line.split("\\s+");
                    for (String s : words) {
                        System.out.print(s + " ");
                    }
                    System.out.println();
                    block.add(words);
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + parameterFileName + " not found");
            e.printStackTrace();
        } 

        if (block.size() == 0) return null;
        if (block.size() == 1) return block.firstElement();
        if (block.size() > 1) {
            System.out.println("Error: More than one loops are specified to be computed now");
            System.exit(1);
        }
        return null;
    }
    
    
    public Vector<myDipolarCoupling.Type> parseProgramParameterFileAndGetTypesOfRdcsToBeUsedForAnalyticSolutions() {
        Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolutions = new Vector<myDipolarCoupling.Type>();
        
        try {
            Scanner scanner = new Scanner(new File( __program_parameter_file));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.length() != 0) {
                    line = line.replaceAll("\\(|\\)|\\,", " ");
                    String[] words = line.split("\\s+");

                    // Set the RDC scale properly based on user input
                    if (words[0].equalsIgnoreCase("@typesOfRdcsToBeUsedForAnalyticSolutions") && words.length >= 2) {
                        
                        for (String word : words) {
                            switch (word) {
                                case "CA_HA":
                                    typesOfRdcsToBeUsedForAnalyticSolutions.add(myDipolarCoupling.Type.CA_HA);
                                    break;
                                case "CA_C":
                                    typesOfRdcsToBeUsedForAnalyticSolutions.add(myDipolarCoupling.Type.CA_C);
                                    break;
                                case "CA_CB":
                                    typesOfRdcsToBeUsedForAnalyticSolutions.add(myDipolarCoupling.Type.CA_CB);
                                    break;
                                case "C_N":
                                    typesOfRdcsToBeUsedForAnalyticSolutions.add(myDipolarCoupling.Type.C_N);
                                    break;
                                case "C_HN":
                                    typesOfRdcsToBeUsedForAnalyticSolutions.add(myDipolarCoupling.Type.C_HN);
                                    break;
                                case "N_HN":
                                    typesOfRdcsToBeUsedForAnalyticSolutions.add(myDipolarCoupling.Type.N_HN);
                                    break;
                            }
                        }                                                
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + __program_parameter_file + " not found");
            e.printStackTrace();
        } 
 
        if (typesOfRdcsToBeUsedForAnalyticSolutions.size() < 2) {
            System.out.println("Error: Insufficient number of RDC types for analytic solutions specified in the file " + __program_parameter_file);
            System.exit(1);
        }
        
        return typesOfRdcsToBeUsedForAnalyticSolutions;
    }
    
    
    public Vector<myDipolarCoupling.Type> parseProgramParameterFileAndGetTypesOfRdcsToBeUsedForRefinement() {
        Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForRefinement = new Vector<myDipolarCoupling.Type>();
        
        try {
            Scanner scanner = new Scanner(new File(__program_parameter_file));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.length() != 0) {
                    line = line.replaceAll("\\(|\\)|\\,", " ");
                    String[] words = line.split("\\s+");

                    // Set the RDC scale properly based on user input
                    if (words[0].equalsIgnoreCase("@typeOfRdcsToBeUsedForRefinement") && words.length >= 1) {
                        
                        for (String word : words) {
                            switch (word) {
                                case "CA_HA":
                                    typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.CA_HA);
                                    break;
                                case "CA_C":
                                    typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.CA_C);
                                    break;
                                case "CA_CB":
                                    typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.CA_CB);
                                    break;
                                case "C_N":
                                    typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.C_N);
                                    break;
                                case "C_HN":
                                    typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.C_HN);
                                    break;
                                case "N_HN":
                                    typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.N_HN);
                                    break;
                            }
                        }                                                
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + __program_parameter_file + " not found");
            e.printStackTrace();
        } 
 
        //if (typesOfRdcsToBeUsedForAnalyticSolutions.size() < 2) {
        //    System.out.println("Error: Insufficient number of RDC types for analytic solutions specified in the file " + __program_parameter_file);
        //    System.exit(1);
        //}
        
        return typesOfRdcsToBeUsedForRefinement;
    }    

    
    public Vector<String> parseProgramParameterFileAndGetTypesOfRdcMediaNames() {        
        Vector<String> mediaNames = new Vector<String>();

        try {
            Scanner scanner = new Scanner(new File(__program_parameter_file));
            scanner.useDelimiter(System.getProperty("line.separator"));
            //String patternString = "\\s*(@loop)\\s*\\([0-9]+\\s*\\,\\s*[0-9]+\\s*\\,";

            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.startsWith("@rdcMediumName")) {
                    line = line.replaceAll("\\(|\\)|\\,", " ");
                    String[] words = line.split("\\s+");
                    for (String word : words) {
                        if (word.equalsIgnoreCase("@rdcMediumName")) {
                            continue;
                        }
                        System.out.print(word + " ");
                        mediaNames.add(word);
                    }
                    System.out.println();                    
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + __program_parameter_file + " not found");
            e.printStackTrace();
        }
        
        return (mediaNames.size() == 0) ? null : mediaNames;
    }
    
    public static void main(String... args) {
        String masterDirName = "DS1";//args[0];

        myInputDataAndParameterManager mdpm = myInputDataAndParameterManager.getInstance();
        mdpm.setMasterDirectory(masterDirName);
        mdpm.setDirectoryArchitectureFile("dirArch.txt");
        
        System.out.println(mdpm.getMostRecentFragmentFileForALeafOfSearchTree());
        
        System.out.println(mdpm.getMasterDirectory());
        System.out.println(mdpm.getDirectoryArchitectureFile());
        System.out.println(mdpm.getBestFragmentPdbFile());
        System.out.println(mdpm.getNoeFile());
    }    
}

