/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;

/**
 * Description of the class
 */
public class myNewPacker {

    private double __length_resolution = 0.2; //0.05; // in Angstrom
    private double __angle_resolution = 30;//30.0;  //0.5;  // in Degree
// for the above two quantities (0.10, 30) #sampled points = 20952
// for the above two quantities (0.15, 30) #sampled points = 14400
// for the above two quantities (0.20, 30) #sampled points = 10512
    
    private double __noe_rmsd_threshold = 0.6;
    private double __joint_score_threshold = 2.0;//2.5;//0.75; // Preset to a reasonable bound so not a hardcode

    private boolean __global_best_or_best_first = true; // false means global best

    private boolean __steric_turned_on = true;
    private boolean __hbond_turned_on = true;

    private double __min_noe_ball_radius = 4.0; // Preset to a reasonable bound so not a hardcode
    private double __max_noe_ball_radius = 5.5; // Preset to a reasonable bound so not a hardcode

    boolean __verbose = false;
    
    public myNewPacker() {        
    }

    public myNewPacker(double lengthResolution, double angleResolution) {
        __length_resolution = lengthResolution;
        __angle_resolution = angleResolution;
    }

    public myNewPacker(double lengthResolution, double angleResolution, double noeRmsdThreshold, double jointScoreThreshold) {
        __length_resolution = lengthResolution;
        __angle_resolution = angleResolution;
        __noe_rmsd_threshold = noeRmsdThreshold;
        __joint_score_threshold = jointScoreThreshold;
    }

    public myNewPacker(boolean globalBestOrBestFirst, boolean stericTurnedOn, boolean hbondTurnedOn) {
        __global_best_or_best_first = globalBestOrBestFirst;
        __steric_turned_on = stericTurnedOn;
        __hbond_turned_on = hbondTurnedOn;
    }

    public void setLengthResolution(double lengthResolution) {
        __length_resolution = lengthResolution;
    }

    public void setAngleResolution(double angleResolution) {
        __angle_resolution = angleResolution;
    }

    public void setNoeRmsdThreshold(double noeRmsdThreshold) {
        __noe_rmsd_threshold = noeRmsdThreshold;
    }

    public void setGlobalBestOrBestFirst(boolean globalBestOrBestFirst) {
        __global_best_or_best_first = globalBestOrBestFirst;
    }

    public void turnOnSteric(boolean stericTurnedOn) {
        __steric_turned_on = stericTurnedOn;
    }

    public void turnOnHbond(boolean hbondTurnedOn) {
        __hbond_turned_on = hbondTurnedOn;
    }

    private Vector<Double> __noe_rmsds = new Vector<Double>(16);

    private static Vector<myMatrix> __rotators = new Vector<myMatrix>();

    static {
        __rotators.add(new myMatrix(new double[][]{{1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}})); //identity
        __rotators.add(new myMatrix(Const.xyInv));
        __rotators.add(new myMatrix(Const.yzInv));
        __rotators.add(new myMatrix(Const.xzInv));
    }

    static myProteinSequence thisProteinSeq = null;
    
    static {
        myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();
        String inputFile = thisParameterManager.getSeqFile();

        thisProteinSeq = new myProteinSequence(inputFile);
        System.out.println(thisProteinSeq);
        //System.exit(1);
    }

    private static double __hbond_angle_cutoff = Math.toRadians(40);
    //private static double __hbond_proton_acceptor_distance_ideal = 1.85;
    private static double __hbond_proton_acceptor_distance_lower_bound = 1.50; // Note: Opt?
    private static double __hbond_proton_acceptor_distance_ideal = 1.85;
    private static double __hbond_proton_acceptor_distance_upper_bound = 2.60; // TODO: Opt?

    private double __parallel_strand_axis_angle_cutoff = Math.toRadians(45);
    private double __parallel_strand_hbond_angle_correction = Math.toRadians(15);    

    /**
     * Test if the two strands are parallel.
     *
     * @param firstStrand the first strand
     * @param secondStrand the second strand
     * @return true if the strands are parallel; otherwise, return false
     */
    public boolean areParallelStrands(myProtein firstStrand, myProtein secondStrand) {
        myVector3D firstStrandAxisVector = null;
        myVector3D secondStrandAxisVector = null;

        getFristStrandAxis:
        {
            myPoint p1 = firstStrand.residueAt(firstStrand.getBeginResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
            myPoint p2 = firstStrand.residueAt(firstStrand.getEndResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
            firstStrandAxisVector = myVector3D.normalize(new myVector3D(p1, p2));
        }

        getSecondStrandAxis:
        {
            myPoint p1 = secondStrand.residueAt(secondStrand.getBeginResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
            myPoint p2 = secondStrand.residueAt(secondStrand.getEndResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();

            secondStrandAxisVector = myVector3D.normalize(new myVector3D(p1, p2));
        }

        double angleBetweenAxes = myVector3D.angle(firstStrandAxisVector, secondStrandAxisVector);

        if (angleBetweenAxes <= __parallel_strand_axis_angle_cutoff) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Compute hydrogen bond score between two strands for backbone atoms only.
     *
     * @param p1 the first strand
     * @param p2 the second strand
     * @return numberOfHbonds, angleRmsd and distanceRmsd
     */
    private myPair<Integer, myPair<Double, Double>> hBondPotentialForBackboneAtomsOnly(myProtein p1, myProtein p2) {

        int numberOfHbonds = 0;
        double sumOfSquaredAngleDeviation = 0.0;
        double sumOfSquaredProtonAcceptorDistanceDeviation = 0.0;

        boolean parallelStrands = false; // = areParallelStrands(p1, p2); TODO: implement this fully this is hack

        // TODO: Note that is can be implemented as a linear fn only as for every residue
        // in strand A you need to check at most 3 residues in strand B.
        for (myResidue r1: p1) {
            for (myResidue r2 : p2) {
                r1Tor2: // first block
                {
                    // check for donor in r1 and acceptor in r2
                    myAtom donor = r1.getAtom(myAtomLabel.__N);
                    myAtom proton = r1.getAtom(myAtomLabel.__H);
                    myAtom acceptor = r2.getAtom(myAtomLabel.__O);
                    if (existsHbond(donor, acceptor, proton)) {
                        numberOfHbonds++;
                        double angleOfHbond = myVector3D.angle(new myVector3D(proton.getCoordinates(), donor.getCoordinates()),
                                new myVector3D(proton.getCoordinates(), acceptor.getCoordinates()));

                        double protonAcceptorDistance = myPoint.distance(proton.getCoordinates(), acceptor.getCoordinates());

                        double angleDeviation = Math.PI - angleOfHbond;

                        if (parallelStrands) {
                            if (angleDeviation > __parallel_strand_hbond_angle_correction) {
                                angleDeviation -= __parallel_strand_hbond_angle_correction;
                            }
                        }

                        sumOfSquaredAngleDeviation += angleDeviation * angleDeviation;
                        sumOfSquaredProtonAcceptorDistanceDeviation += (protonAcceptorDistance - __hbond_proton_acceptor_distance_ideal) *
                                (protonAcceptorDistance - __hbond_proton_acceptor_distance_ideal);
                    }
                }

                r2Tor1: // second block
                {
                    // check for acceptor in r1 and donor in r2
                    myAtom donor = r2.getAtom(myAtomLabel.__N);
                    myAtom proton = r2.getAtom(myAtomLabel.__H);
                    myAtom acceptor = r1.getAtom(myAtomLabel.__O);
                    if (existsHbond(donor, acceptor, proton)) {
                        numberOfHbonds++;
                        double angleOfHbond = myVector3D.angle(new myVector3D(proton.getCoordinates(), donor.getCoordinates()),
                                new myVector3D(proton.getCoordinates(), acceptor.getCoordinates()));

                        double protonAcceptorDistance = myPoint.distance(proton.getCoordinates(), acceptor.getCoordinates());

                        double angleDeviation = Math.PI - angleOfHbond;

                        if (parallelStrands) {
                            if (angleDeviation > __parallel_strand_hbond_angle_correction) {
                                angleDeviation -= __parallel_strand_hbond_angle_correction;
                            }
                        }

                        sumOfSquaredAngleDeviation += angleDeviation * angleDeviation;
                        sumOfSquaredProtonAcceptorDistanceDeviation += (protonAcceptorDistance - __hbond_proton_acceptor_distance_ideal) *
                                (protonAcceptorDistance - __hbond_proton_acceptor_distance_ideal);
                    }
                }
            }
        }

        double angleRmsd = Math.sqrt(sumOfSquaredAngleDeviation / numberOfHbonds);
        double distanceRmsd = Math.sqrt(sumOfSquaredProtonAcceptorDistanceDeviation / numberOfHbonds);

        myPair<Integer, myPair<Double, Double>> numberOfHbonds_angleRmsd_DistanceRmsd_triple =
                new myPair<Integer, myPair<Double, Double>>(numberOfHbonds, new myPair<Double, Double>(angleRmsd, distanceRmsd));
        return numberOfHbonds_angleRmsd_DistanceRmsd_triple;
    }

    /**
     * Test if there exists a hydrogen bond between a donor, an acceptor and a proton.
     *
     * @param donor the donor atom
     * @param acceptor the acceptor atom
     * @param proton the proton
     * @return true if there is hydrogen bond; otherwise, return false
     */
    private boolean existsHbond(myAtom donor, myAtom acceptor, myAtom proton) {
        if (donor.isEqualTo(acceptor) || donor.isBondedTo(acceptor) || !donor.isBondedTo(proton) || !proton.getAtomName().startsWith("H")) { // Note: Consider sulphide bonded H in future.
            return false;
        }

//        if (!((donor.getAtomName().startsWith("N") || donor.getAtomName().startsWith("O") || donor.getAtomName().startsWith("S")) &&
//                (acceptor.getAtomName().startsWith("N") || acceptor.getAtomName().startsWith("O") || acceptor.getAtomName().startsWith("S")))) {
//            return false;
//        }

        if (!donor.getAtomName().startsWith("N") || !acceptor.getAtomName().startsWith("O")) { // Handle all combinations of N, S, O
            return false;
        }

        double angleOfHbond = myVector3D.angle(new myVector3D(proton.getCoordinates(), donor.getCoordinates()),
                new myVector3D(proton.getCoordinates(), acceptor.getCoordinates()));

        double protonAcceptorDistance = myPoint.distance(proton.getCoordinates(), acceptor.getCoordinates());

        if (Math.PI - angleOfHbond <= __hbond_angle_cutoff &&
                __hbond_proton_acceptor_distance_lower_bound <= protonAcceptorDistance &&
                protonAcceptorDistance <= __hbond_proton_acceptor_distance_upper_bound) { // means we have detected a legal hbond
            return true;
        }

        return false;
    }

    /**
     * Pack two strands using NOEs. Model the rotamers, optimize hydrogen bonds
     * and sterics, to finally output the best packed strand pairs.
     *
     * @param prot1 the first strand
     * @param prot2 the second strand
     * @param noeVector the vector of NOEs
     * @param seq the sequence
     * @return a triple containing the two packed strands and the packing score
     */
    public myTriple<myProtein, myProtein, Double> __packUsingNoes__(myProtein prot1, myProtein prot2, Vector<myNoe> noeVector, myProteinSequence seq) {
        if (__verbose) {
            System.out.println("I am in __packUsingNoes__");
        }


        myProtein p1Clone = new myProtein(prot1);
        myProtein p2Clone = new myProtein(prot2);

        seq = thisProteinSeq;

        myRotamerLibrary thisRotamerLibrary = myRotamerLibrary.getInstance();

        // Change the poly-alanine models to actual ones
        for (myResidue r : p1Clone) {
            thisRotamerLibrary.mutateResidue(r, seq.getResidueType(r.getResidueNumber()));
        }

        for (myResidue r : p2Clone) {
            thisRotamerLibrary.mutateResidue(r, seq.getResidueType(r.getResidueNumber()));
        }

        Vector<myPoint> pp1 = new Vector<myPoint>(), pp2 = new Vector<myPoint>();
        Map<Integer, myPoint> pp1Map = new TreeMap<Integer, myPoint>();
        Map<Integer, myPoint> pp2Map = new TreeMap<Integer, myPoint>();
        Vector<myNoe> noeVec = new Vector<myNoe>();

        // get the NOEs that will be used here and then get the respective points (atoms)
        // in the respective fragments which describe the NOEs. The noes are added such that
        // each noe has first porton in the first fragment and second proton in the second
        // fragment.
        for (myNoe ee : noeVector) {
            int firstResNum = ee.getResidueNoA();
            int secondResNum = ee.getResidueNoB();
            if (p1Clone.residueAt(firstResNum) != null && p2Clone.residueAt(secondResNum) != null) {
                noeVec.add(new myNoe(ee));
            } else if (p1Clone.residueAt(secondResNum) != null && p2Clone.residueAt(firstResNum) != null) {
                myNoe nee = new myNoe(ee);
                nee.swap();
                noeVec.add(nee);
            } else; //Do not add
        }

        Vector<Vector<myPoint>> setA = new Vector<Vector<myPoint>>();
        Vector<Vector<myPoint>> setB = new Vector<Vector<myPoint>>();

        if (__verbose) {
            for (myNoe ee : noeVec) {
                ee.print();
            }
        }

        for (myNoe ee : noeVec) {
            // Now we have r1 and r2 and we know that there exists an noe between
            // two protons of these two residues. Let's make a set of the possible
            // protons inferred by the rotamer library.

            //test if a backbone atom or a side chain proton.
            Vector<myPoint> pointSetA = new Vector<myPoint>();
            Vector<myPoint> pointSetB = new Vector<myPoint>();

            GetCoordinateSetA:
            {
                int firstResNum = ee.getResidueNoA();
                String firstAtomName = ee.getAtomA();
                myResidue r1 = p1Clone.residueAt(firstResNum);

                //System.out.println("firstResidueName: " + r1.getResidueName() + "  firstAtomName: " + firstAtomName + "  firstResNum: " + firstResNum);

                if (isLoneBackboneProton(firstAtomName)) { // then point set A must contain only one element
                    myPoint thisPoint = new myPoint(r1.getAtom(firstAtomName).getCoordinates());
                    pointSetA.add(thisPoint);
                    if (__verbose) {
                        System.out.println("firstResidueName: " + r1.getResidueName() + "  firstResNum: " + firstResNum + "  newFirstAtomName: " + firstAtomName + "  coordinates: " + thisPoint.toString());
                    }
                } else { // The atom is a side-chain atom and can have multiple coordinates due to different rotamers
                    String firstResidueName = r1.getResidueName();
                    int numberOfRotamerForA = thisRotamerLibrary.numberOfRotamers(firstResidueName);

                    for (int i = 0; i < numberOfRotamerForA; i++) { // TODO: improve for '#'
                        thisRotamerLibrary.mutateResidue(r1, firstResidueName, i);
                        //r1.print();
                        if (firstAtomName.endsWith("#")) {
                            for (int suffix = 1; suffix <= 3; suffix++) {
                                String newFirstAtomName = firstAtomName.replace("#", Integer.toString(suffix));
                                if (r1.canHaveAtom(newFirstAtomName)) {
                                    myPoint thisPoint = new myPoint(r1.getAtom(newFirstAtomName).getCoordinates());
                                    pointSetA.add(thisPoint);
                                    if (__verbose) {
                                        System.out.println("firstResidueName: " + r1.getResidueName() + "  firstResNum: " + firstResNum + "  newFirstAtomName: " + newFirstAtomName + "  coordinates: " + thisPoint.toString());
                                    }
                                }
                            }
                        } else {
                            myPoint thisPoint = new myPoint(r1.getAtom(firstAtomName).getCoordinates());
                            pointSetA.add(thisPoint);
                            if (__verbose) {
                                System.out.println("firstResidueName: " + r1.getResidueName() + "  firstResNum: " + firstResNum + "  newFirstAtomName: " + firstAtomName + "  coordinates: " + thisPoint.toString());
                            }
                        }
                    }
                }

                pp1Map.put(r1.getResidueNumber(), new myPoint(r1.getAtom(myAtomLabel.__CA).getCoordinates()));
            }
            setA.add(pointSetA);

            GetCoordinateSetB:
            {
                int secondResNum = ee.getResidueNoB();
                String secondAtomName = ee.getAtomB();
                myResidue r2 = p2Clone.residueAt(secondResNum);

                //System.out.println("secondResidueName: " + r2.getResidueName() + "  secondAtomName: " + secondAtomName + "  secondResNum: " + secondResNum);

                if (isLoneBackboneProton(secondAtomName)) { // then point set A must contain only one element
                    myPoint thisPoint = new myPoint(r2.getAtom(secondAtomName).getCoordinates());
                    pointSetB.add(thisPoint);
                    if (__verbose) {
                        System.out.println("secondResidueName: " + r2.getResidueName() + "  secondResNum: " + secondResNum + "  newSecondAtomName: " + secondAtomName + "  coordinates: " + thisPoint.toString());
                    }
                } else { // The atom is a side-chain atom and can have multiple coordinates due to different rotamers
                    String secondResidueName = r2.getResidueName();
                    int numberOfRotamerForB = thisRotamerLibrary.numberOfRotamers(secondResidueName);

                    for (int i = 0; i < numberOfRotamerForB; i++) { // TODO: improve for '#'
                        thisRotamerLibrary.mutateResidue(r2, secondResidueName, i);
                        //r2.print();
                        if (secondAtomName.endsWith("#")) {
                            for (int suffix = 1; suffix <= 3; suffix++) {
                                String newSecondAtomName = secondAtomName.replace("#", Integer.toString(suffix));
                                if (r2.canHaveAtom(newSecondAtomName)) {
                                    myPoint thisPoint = new myPoint(r2.getAtom(newSecondAtomName).getCoordinates());
                                    pointSetB.add(thisPoint);
                                    if (__verbose) {
                                        System.out.println("secondResidueName: " + r2.getResidueName() + "  secondResNum: " + secondResNum + "  newSecondAtomName: " + newSecondAtomName + "  coordinates: " + thisPoint.toString());
                                    }
                                }
                            }
                        } else {
                            myPoint thisPoint = new myPoint(r2.getAtom(secondAtomName).getCoordinates());
                            pointSetB.add(thisPoint);
                            if (__verbose) {
                                System.out.println("secondResidueName: " + r2.getResidueName() + "  secondResNum: " + secondResNum + "  newSecondAtomName: " + secondAtomName + "  coordinates: " + thisPoint.toString());
                            }
                        }
                    }
                }

                pp2Map.put(r2.getResidueNumber(), new myPoint(r2.getAtom(myAtomLabel.__CA).getCoordinates()));
            }
            setB.add(pointSetB);

            if (__verbose) {
                System.out.println("Red Blue Set for this Noe:");

                System.out.println("Red set: ");
                for (myPoint p : pointSetA) {
                    System.out.println(p.toString());
                }

                System.out.println("Blue set: ");
                for (myPoint p : pointSetB) {
                    System.out.println(p.toString());
                }
            }
            //inputChar.readChar();
        }

        if (setA.size() != setB.size()) {
            System.out.println("Error: Noe set contains at least one noe whose other end is missing in the structure");
            System.exit(1);
        }

        if (__verbose) {
            System.out.println("Printing the red sets");
            for (Vector<myPoint> vm : setA) {
                for (myPoint p : vm) {
                    System.out.println(p.toString());
                }
                System.out.println();
            }

            System.out.println("Printing the blue sets");
            for (Vector<myPoint> vm : setB) {
                for (myPoint p : vm) {
                    System.out.println(p.toString());
                }
                System.out.println();
            }
        }

//        System.out.println("The NOEs used for packing:");
//        myNoe printerNoe = new myNoe();
//        printerNoe.PrintNOE(noeVec);

        // We previously used map to get rid of duplicates. Now transfer the coordinates to vectors.
        for (Map.Entry<Integer, myPoint> elem : pp1Map.entrySet()) {
            pp1.add(elem.getValue());
        }
        for (Map.Entry<Integer, myPoint> elem : pp2Map.entrySet()) {
            pp2.add(elem.getValue());
        }
        myPoint geomCenter1 = computeGeometricCenter(pp1);
        myPoint geomCenter2 = computeGeometricCenter(pp2);

        //System.out.println("Dist::::::: " + myPoint.distance(geomCenter1, geomCenter2)); System.exit(1);
        //for (myPoint p : pp1) System.out.print(p.toString());
        //System.out.println("Geometric Centers: (1) " + geomCenter1.toString() + "  Geometric Centers: (2) " + geomCenter2.toString());

        Vector<myPoint> pp1New = new Vector<myPoint>(pp1.size()), pp2New = new Vector<myPoint>(pp2.size());
        for (int i = 0; i < pp1.size(); i++) {
            pp1New.add(new myPoint(pp1.elementAt(i)));
        }
        for (int i = 0; i < pp2.size(); i++) {
            pp2New.add(new myPoint(pp2.elementAt(i)));
        }

        // shift both the fragments to origin
        computeTranslation(pp1New, myVector3D.reverse(new myVector3D(geomCenter1.getXYZ())));
        computeTranslation(pp2New, myVector3D.reverse(new myVector3D(geomCenter2.getXYZ())));

        // shift both the fragments' point clouds to the origin
        for (int i = 0; i < setA.size(); i++) { // Note that
            computeTranslation(setA.elementAt(i), myVector3D.reverse(new myVector3D(geomCenter1.getXYZ())));
            computeTranslation(setB.elementAt(i), myVector3D.reverse(new myVector3D(geomCenter2.getXYZ())));
        }

//        for (Vector<myPoint> s : setB) {
//            for (myPoint p : s) {
//                System.out.println(p.toString());
//            }
//        }
//        System.exit(1);
//        myPoint geomCenter1_ = computeGeometricCenter(pp1New);
//        System.out.println("gs1: " + geomCenter1_.toString());
//        myPoint geomCenter2_ = computeGeometricCenter(pp2New);
//        System.out.println("gs2: " + geomCenter2_.toString() + "  " + new myVector3D(geomCenter1_, geomCenter2_).norm());
//System.exit(1);

//        double meanNoe = 0.0;
//        for (int i = 0; i < noeVec.size(); i++) {
//            meanNoe += (noeVec.elementAt(i).getRange()[0] + noeVec.elementAt(i).getRange()[1]) / 2;
//        }
//        meanNoe /= noeVec.size();
//        System.out.println("Mean Noe: " + meanNoe);


        myTriple<myProtein, myProtein, Double> packedFirstFragmentSecondFragmentAndScore = null;
        myProtein packedSseSave = null;
        double clashScoreSave = Double.MAX_VALUE;
        double thetaSave = 0.0, phiSave = 0.0;
        double radiusSave = 0.0;
        double noeRmsdSave = Double.MAX_VALUE;
        Vector<myTriple<Integer, Integer, Double>> vtSave = null;
        double jointScoreSave = Double.MAX_VALUE;

        double meanBallRadius = (__max_noe_ball_radius + __min_noe_ball_radius) / 2; // This is preset, not a hard code.
        double ballThickness = (__max_noe_ball_radius - __min_noe_ball_radius); // This is preset, not a hard code.
        int rIter = (int) (ballThickness / __length_resolution);

        int numberOfPoints = 0;

        bestFirstLabel:
        for (int k = 0; k <= rIter; k++) { // <= to make sure that the boundaries are included
            //double radius = meanNoe + (k - halfrIter) * __length_resolution;
            // fancy zig-zag for loop since we expect the noe satisfaction to be somewhere around the middle of the ball
            int kp1 = (int) ((k + 1) / 2);
            int flipper = 1 - 2 * (k % 2);
            int gridNumber = kp1 * flipper;
            double radius = meanBallRadius + gridNumber * __length_resolution;
            //System.out.println(">>radius: " + radius);

            double newAngleResolution = __angle_resolution *  1 / Math.ceil(radius - __min_noe_ball_radius + 1); // / Math.ceil(4.0 * (k + 1) / (rIter * 1.0));
            //System.out.println(">>newAngleResolution: " + newAngleResolution);
            for (int m = 0; m < 180 / newAngleResolution; m++) {
                double theta = Math.toRadians(m * newAngleResolution);

                for (int j = 0; j < 360 / newAngleResolution; j++) {
                    double phi = Math.toRadians(j * newAngleResolution);

//                    numberOfPoints++;
//                    System.out.println("NumberOfPoints: " + numberOfPoints);
//
//                    boolean bb = true;
//                    if (bb)
//                        continue;

                    Vector<Vector<myPoint>> copyOfSetA = setA; // copyOfVectorOfVectorOfPointsIn3D(setA);
                    Vector<Vector<myPoint>> copyOfSetB = copyOfVectorOfVectorOfPointsIn3D(setB);

                    for (Vector<myPoint> vp : copyOfSetB) {
                        for (int i = 0; i < vp.size(); i++) {
                            myPoint p = myPoint.translate(vp.elementAt(i), new myVector3D(radius * Math.sin(theta) * Math.cos(phi),
                                    radius * Math.sin(theta) * Math.sin(phi), radius * Math.cos(theta)));
                            vp.set(i, p);
                        }
                    }

                    // Compute the noe deviation based on bichromatic clostst pair in 3D.
                    Vector<myTriple<Integer, Integer, Double>> vt = new Vector<myTriple<Integer, Integer, Double>>();
                    for (int i = 0; i < copyOfSetA.size(); i++) {
                        Vector<myPoint> redSet = copyOfSetA.elementAt(i);
                        Vector<myPoint> blueSet = copyOfSetB.elementAt(i);
                        myTriple<Integer, Integer, Double> score = bestNoeWithLeastDeviation(redSet, blueSet, noeVec.elementAt(i));
                        vt.add(score);
                    }

                    // Compute the noe rmsd
                    double noeRmsd = 0.0;
                    for (myTriple<Integer, Integer, Double> t : vt) {
                        //System.out.println("vt: " + t.first() + "  " + t.second() + "  " + t.third());
                        noeRmsd += t.third() * t.third();
                    }
                    //System.out.println();
                    noeRmsd = Math.sqrt(noeRmsd / vt.size());
//noeRmsd = 0.0;
                    if (noeRmsd < __noe_rmsd_threshold) {
                        double clashScore = myMiscConstants.genericThreshold;
                        double hbScore = myMiscConstants.genericThreshold;
                        myProtein packedSse = null;
                        myProtein p1Backbone = null;
                        myProtein p2Backbone = null;

//if (__hbond_turned_on || __steric_turned_on) {
                        codeAddedForStericCheckAndHbond:
                        {
                            myPoint center2New = new myPoint(geomCenter2);
                            center2New.setX(-center2New.getX() + radius * Math.sin(theta) * Math.cos(phi));
                            center2New.setY(-center2New.getY() + radius * Math.sin(theta) * Math.sin(phi));
                            center2New.setZ(-center2New.getZ() + radius * Math.cos(theta));
                            p2Backbone = myProtein.extractBackbone(p2Clone);
                            p2Backbone.translate(center2New.getX() + geomCenter1.getX(), center2New.getY() + geomCenter1.getY(), center2New.getZ() + geomCenter1.getZ());

                            p1Backbone = myProtein.extractBackbone(p1Clone);

                            packedSse = myProtein.cloneAndMergeTwoProteinFragments(p1Backbone, p2Backbone);

                            // Do Hbond stuff before steric since it will save some time when it fails
                            if (__hbond_turned_on) {
                                // Now that we packed the two structures, we will check how well they do in terms of Hbonding.
                                myPair<Integer, myPair<Double, Double>> numberOfHbonds_angleRmsd_DistanceRmsd_triple = hBondPotentialForBackboneAtomsOnly(p1Backbone, p2Backbone);

                                // Next think of how to combine these scores
                                int numberOfHbonds = numberOfHbonds_angleRmsd_DistanceRmsd_triple.first();
                                double angleRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().first(); // expected to be < 0.75 for good strands
                                double hbondDistanceRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().second(); // expected to be < 0.75 for good strands

                                //System.out.println("Number of hBonds: " + numberOfHbonds);

                                double A = 1.0;// to make sure that an odd case like just one perfect hbond does not rule out a strong case like 10 non-perfect hbonds
                                double retScore = (numberOfHbonds == 0) ? myMiscConstants.genericThreshold : (A + angleRmsd + hbondDistanceRmsd) / numberOfHbonds; // test if numberOfHbond^2 can be a better choice

                                //if (retScore != myMiscConstants.genericThreshold) {
                                //System.out.println("Number of hBonds: " + numberOfHbonds);
                                //System.out.println("Hbond score: " + retScore);
                                //}
                                hbScore = retScore;
                            }

                            // Do the steric check.
                            if (__steric_turned_on) {
                                if (__hbond_turned_on && hbScore == myMiscConstants.genericThreshold) {
                                    clashScore = myMiscConstants.genericThreshold;
                                    //System.out.println("This segment of the code is valid");
                                } else { // do steric check
                                    myStericChecker sc = new myStericChecker(1.0, 1.0, 0.5); // ideal: (1.0, 0.8, 0.4);
                                    clashScore = sc.checkStericClash(packedSse);
                                    //System.out.println("Clash Score: " + clashScore);
                                }
                            }
                        }

                        // noeRmsd range: [0.0 - 0.75], hbScore range: [0.001 - 1.0], clashScore range: [0 - 40]

                        double jointScore = noeRmsd;

                        if (__steric_turned_on) {
                            jointScore += clashScore / 40.0; // Note: this is not hardcode, rather a preset value that scales well with other scores
                        }

                        if (__hbond_turned_on) {
                            jointScore += 2 * hbScore; // so that it can be given more importance than noe rmsd
                        }

                        if (jointScore < jointScoreSave) {
                            jointScoreSave = jointScore;

                            clashScoreSave = clashScore;
                            packedSseSave = packedSse;
                            packedFirstFragmentSecondFragmentAndScore = new myTriple<myProtein, myProtein, Double>(p1Backbone, p2Backbone, jointScore);
                            vtSave = vt;

                            thetaSave = theta;
                            phiSave = phi;
                            noeRmsdSave = noeRmsd;
                            radiusSave = radius;

                            System.out.println("Radius: " + radiusSave);
                            System.out.println("Noe rmsd: " + noeRmsdSave); // Monitor the progress

                            if (__steric_turned_on) {
                                System.out.println("Clash score: " + clashScoreSave);
                            }

                            if (__hbond_turned_on) {
                                System.out.println("Hbond score: " + hbScore);
                            }

                            System.out.println("Joint Score: " + jointScoreSave + '\n');
                            if (jointScore < __joint_score_threshold) {
                                packedSseSave.print();
                                if (__global_best_or_best_first == false) {
                                    break bestFirstLabel;
                                }
                            }
                        }
                    }
                }
            }
        }

        //return packedFirstFragmentSecondFragmentAndScore;

        if (__steric_turned_on ||__hbond_turned_on) {//----does not seem to be so good
            if (packedFirstFragmentSecondFragmentAndScore == null)
                return new myTriple<myProtein, myProtein, Double>(null, null, myMiscConstants.scoreTermForSheetPackingThreshold);
            return packedFirstFragmentSecondFragmentAndScore;
        } else {
            System.out.println("Error: Code when steric and hbond filters are turned off is yet to be implemented");
            System.exit(1);
        }


        // Note: The code below does not execute (basically unreachable) as in every reasonable case
        // we set both steric and hbond filters on, so it does not execute. So please remove the code
        // below at your convenience. This was used once while testing and developing this module.

//        System.exit(1);

        System.out.println("Final pcking Results are below");
        System.out.println("Noe rmsd: " + noeRmsdSave);
        for (myTriple<Integer, Integer, Double> t : vtSave) {
            System.out.println("vt: " + t.first() + "  " + t.second() + "  " + t.third());
        }
        packedSseSave.print();
        System.exit(1);

        //System.out.println("NOE rmsd: " + disRmsd);
        //__noe_rmsds.add(disRmsd);
        //System.out.println("NOE distances difference for individual ones:");

        myPoint center2New = new myPoint(geomCenter2);
        center2New.setX(-center2New.getX() + radiusSave * Math.sin(thetaSave) * Math.cos(phiSave));
        center2New.setY(-center2New.getY() + radiusSave * Math.sin(thetaSave) * Math.sin(phiSave));
        center2New.setZ(-center2New.getZ() + radiusSave * Math.cos(thetaSave));

//        p1Clone.translate(-geomCenter1.getX(), -geomCenter1.getY(), -geomCenter1.getZ());
//        p2Clone.translate(center2New.getX(), center2New.getY(), center2New.getZ());

        // TODO: Check if it preserves p1Clone's coordinates
        //p1Clone.translate(-geomCenter1.getX(), -geomCenter1.getY(), -geomCenter1.getZ());
        p2Clone.translate(center2New.getX() + geomCenter1.getX(), center2New.getY() + geomCenter1.getY(), center2New.getZ() + geomCenter1.getZ());

        //p1Clone.print();
        //p2Clone.print();

        //myProtein ppnew = myProtein.cloneAndMergeTwoProteinFragments(myProtein.extractBackbone(p2Clone), myProtein.extractBackbone(p1Clone));
        //ppnew.print();

        System.exit(1);


//        if (disRmsd < 0.3) { // TODO: this is for debugging only, remove this later
//            System.out.println("This is a good pair of sheet + strand, printing this for debugging: ");
//            p1Clone.print();
//            p2Clone.print();
//        }

        // Now that we packed the two structures, we will check how well they do in terms of Hbonding.
        myPair<Integer, myPair<Double, Double>> numberOfHbonds_angleRmsd_DistanceRmsd_triple = hBondPotentialForBackboneAtomsOnly(p1Clone, p2Clone);

        // Next think of how to combine these scores
        int numberOfHbonds = numberOfHbonds_angleRmsd_DistanceRmsd_triple.first();
        double angleRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().first(); // expected to be < 0.75 for good strands
        double hbondDistanceRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().second(); // expected to be < 0.75 for good strands

        System.out.println("Number of hBonds: " + numberOfHbonds);

        double retScore = myMiscConstants.scoreTermForSheetPackingThreshold; // TODO: remove this peice of hCode

        if (noeRmsdSave > myMiscConstants.noeRmsdUpperBoundForSheetPacking || numberOfHbonds == 0) { // TODO: what is a better cutoff for disRmsd??
            return new myTriple<myProtein, myProtein, Double>(null, null, retScore);
        }

        System.out.println("NOE rmsd: " + noeRmsdSave + "    numberOfHbonds: " + numberOfHbonds);

        myProtein toBeStericChecked = myProtein.cloneAndMergeTwoProteinFragments(p1Clone, p2Clone);
        //System.out.println("Passing the protein to the steric checker");
        myStericChecker sc = new myStericChecker(1.0, 1.0, 0.5); // ideal: (1.0, 0.8, 0.4);
        double clashScore = sc.checkStericClash(toBeStericChecked);
        //System.out.println("Returning from the steric checker");
        System.out.println("Clash Score: " + clashScore);

        if (clashScore > myMiscConstants.clashScoreThresholdForSheetPacking) {
            return new myTriple<myProtein, myProtein, Double>(null, null, retScore);
        }

        if (noeRmsdSave < 0.4) { // TODO: this is for debugging only, remove this later
            System.out.println("Note: (this comment is used for Debugging) This can be good pair of sheet + strand that satisfies noes, steric, hbond but NOT sure about RDC: ");
            toBeStericChecked.print();
        }

        // TODO: this is a simple scoring function with the objective that
        // the lower its value, the the better hbonded the two strands are.
        double A = 1.0;// to make sure that an odd case like just one perfect hbond does not rule out a strong case like 10 non-perfect hbonds
        retScore = (A + angleRmsd + hbondDistanceRmsd) / numberOfHbonds; // test if numberOfHbond^2 can be a better choice

        return new myTriple<myProtein, myProtein, Double>(p1Clone, p2Clone, retScore);

        //myProtein ret = myProtein.cloneAndMergeTwoProteinFragments(p1Clone, p2Clone);
        //return ret;
    }

    /**
     * Compute the geometric center of a collection of points in 3D space.
     *
     * @param coll set of points
     * @return the geometric center of the set of points
     */
    public static myPoint computeGeometricCenter(Collection<? extends myPoint> coll) {
        myPoint geomCenter = new myPoint(0, 0, 0);
        Iterator<? extends myPoint> iter = coll.iterator();
        while (iter.hasNext()) {
            myPoint p = iter.next();
            geomCenter.setX(geomCenter.getX() + p.getX());
            geomCenter.setY(geomCenter.getY() + p.getY());
            geomCenter.setZ(geomCenter.getZ() + p.getZ());
        }
        geomCenter.setX(geomCenter.getX() / coll.size());
        geomCenter.setY(geomCenter.getY() / coll.size());
        geomCenter.setZ(geomCenter.getZ() / coll.size());

        return geomCenter;
    }

    /**
     * Translate the set of points by an amount given by the vector
     * @param coll the set of points
     * @param v the vector used to translate the points
     */
    public static void computeTranslation(Collection<? extends myPoint> coll, myVector3D v) {
        Iterator<? extends myPoint> iter = coll.iterator();
        while (iter.hasNext()) {
            myPoint p = iter.next();
            p.translate(v);
        }
    }

    /**
     * Given the protein p in one of the POFs, return a set of four proteins
     * in the four possible POFs.
     *
     * @param p the protein whose four-folds are to be computed
     * @return four-fold structures
     */
    public static Vector<myProtein> getFourFolds(myProtein p) {
        Vector<myProtein> fourFoldFragment = new Vector<myProtein>();
        for (int i = 0; i < 4; i++) {
            fourFoldFragment.add(new myProtein(p));
            fourFoldFragment.elementAt(i).rotate(__rotators.elementAt(i));
        }
        return fourFoldFragment;
    }

    /**
     * Returns true if the atom is a lone backbone atom. Note that the phrase 'proton'
     * in the method name is a misnomer.
     *
     * @param atomName the atom name
     * @return true if it is the only atom; otherwise, return false
     */
    private boolean isLoneBackboneProton(String atomName) {
        atomName = atomName.trim().toUpperCase();
        if (atomName.equalsIgnoreCase("H") || atomName.equalsIgnoreCase("HN") ||
                atomName.equalsIgnoreCase("H1") || atomName.equalsIgnoreCase("H2") ||
                atomName.equalsIgnoreCase("H3") || /*atomName.equalsIgnoreCase("HA2") || atomName.equalsIgnoreCase("HA3")*/
                atomName.equalsIgnoreCase("HA") || atomName.equalsIgnoreCase("CA") ||
                atomName.equalsIgnoreCase("O")  || atomName.equalsIgnoreCase("N") || atomName.equalsIgnoreCase("C")) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Generate a copy of a vector of vector of points in 3D.
     *
     * @param vvp the vector of vector of points in 3D
     * @return a copy of vvp
     */
    private Vector<Vector<myPoint>> copyOfVectorOfVectorOfPointsIn3D(Vector<Vector<myPoint>> vvp) {
        Vector<Vector<myPoint>> vvpCopy = new Vector<Vector<myPoint>>();
        for (Vector<myPoint> vp : vvp) {
            Vector<myPoint> vpCopy = new Vector<myPoint>();
            for (myPoint p : vp) {
                vpCopy.add(new myPoint(p));
            }
            vvpCopy.add(vpCopy);
        }
        return vvpCopy;
    }

    /**
     * Compute the deviation of a point from an interval.
     *
     * @param lb lower bound of the interval
     * @param ub upper bound of the interval
     * @param val the point
     * @return the (absolute value of) deviation
     */
    private double computeDeviation(double lb, double ub, double val) {
        double a = Math.abs(val - lb);
        double b = Math.abs(ub - val);
        return Math.min(a, b);
    }

    /**
     * Find the best NOE with least deviation.
     *
     * @param redSet first set of points in 3D
     * @param blueSet second set of points in 3D
     * @param ee the NOE
     * @return a triple containing the point ids in first and second sets, and the least deviation
     */
    private myTriple<Integer, Integer, Double> bestNoeWithLeastDeviation(Vector<myPoint> redSet, Vector<myPoint> blueSet, myNoe ee) {
        //ee.print();
        Integer iSave = Integer.MAX_VALUE, jSave = Integer.MAX_VALUE;
        Double devSave = Double.MAX_VALUE;
        for (int i = 0; i < redSet.size(); i++) {
            for (int j = 0; j < blueSet.size(); j++) {
                double dist = myPoint.distance(redSet.elementAt(i), blueSet.elementAt(j));
                double noeDeviation = computeDeviation(ee.getLower(), ee.getUpper(), dist);
                if (devSave > noeDeviation) {
                    devSave = noeDeviation;
                    iSave = i;
                    jSave = j;
                }
            }
        }
        return new myTriple(iSave, jSave, devSave);
    }


/**********Everything above is complete, and everything below is being tested**********/

    public static void main0(String... args) {

        Vector<myNoe> hBondVec = null;
        temporaryNoeReader:
        {
            String userDir = System.getProperty("user.dir");
            String fileSeparator = System.getProperty("file.separator");
            String inputDirectory = userDir + fileSeparator + "input_files_1d3z" + fileSeparator + "input_data_files" + fileSeparator;
            String inputFile = inputDirectory + "noe.txt";
            //System.out.println("inputFile: " + inputFile);
            hBondVec = myNoe.parseNoeFileInXplorFormat(inputFile); //new Vector<myNoe>(Arrays.asList(noes));

//TODO: get rid of non bb noes. implement rigorously.
            //for (int i = 0; i < 5; i++)
            //    hBondVec.remove(hBondVec.size() - 1);
            
        int sz = hBondVec.size();

        hBondVec.elementAt(sz-4).setDistLower(10.23);
        hBondVec.elementAt(sz-3).setDistLower(8.32);
        hBondVec.elementAt(sz-2).setDistLower(8.40);
        hBondVec.elementAt(sz-1).setDistLower(8.26);

        hBondVec.elementAt(sz-4).setDistUpper(10.23);
        hBondVec.elementAt(sz-3).setDistUpper(8.32);
        hBondVec.elementAt(sz-2).setDistUpper(8.40);
        hBondVec.elementAt(sz-1).setDistUpper(8.26);

            myNoe.printNoeVector(hBondVec);

        }

        String firstSseFile = "ff11.pdb";
        String secondSseFile = "ff22.pdb";
        
        Vector<myProtein> vp2 = new Vector<myProtein>();
        myPdbParser pParser2 = new myPdbParser(firstSseFile);
        while (pParser2.hasNextProtein()) {
            vp2.add(pParser2.nextProtein());
        }
        myProtein p1 = vp2.elementAt(0);

        Vector<myProtein> vp3 = new Vector<myProtein>();
        myPdbParser pParser3 = new myPdbParser(secondSseFile);
        while (pParser3.hasNextProtein()) {
            vp3.add(pParser3.nextProtein());
        }
        myProtein p2 = vp3.elementAt(0);

        Vector<myProtein> vp = myNewPacker.getFourFolds(p1);

        myNewPacker pk = new myNewPacker();
        
        pk.setGlobalBestOrBestFirst(true);
        pk.turnOnHbond(false);
        pk.__verbose = true;
        pk.turnOnSteric(true);

        for (myProtein wps : vp) {
             myTriple<myProtein, myProtein, Double> score = pk.__packSseUsingNoes__(wps, p2, hBondVec, null);

             System.out.println("Printing the proteins and the packing score");
             System.out.println("Score: " + score.third());
             if (score.first() != null)
                score.first().print();
             if (score.second() != null)
                score.second().print();

             System.out.println("Read a char");
             inputChar.readChar();

             break;

        }
    }


    /****************** The method below is being designed ******************/







    public static void mainD12(String... args) {

        Vector<myNoe> hBondVec = null;
        temporaryNoeReader:
        {
            String userDir = System.getProperty("user.dir");
            String fileSeparator = System.getProperty("file.separator");
            String inputDirectory = userDir + fileSeparator + "input_files_1d3z" + fileSeparator + "input_data_files" + fileSeparator;
            String inputFile = inputDirectory + "noe.txt";
            //System.out.println("inputFile: " + inputFile);
            hBondVec = myNoe.parseNoeFileInXplorFormat(inputFile); //new Vector<myNoe>(Arrays.asList(noes));

//TODO: get rid of non bb noes. implement rigorously.
            //for (int i = 0; i < 5; i++)
            //    hBondVec.remove(hBondVec.size() - 1);

        int sz = hBondVec.size();

        //hBondVec.clear();


//      These are the NOEs for UBQ
        //hBondVec.elementAt(sz-5).setDistLower(9.73);
        hBondVec.elementAt(sz-4).setDistLower(10.23);
        hBondVec.elementAt(sz-3).setDistLower(8.32);
        hBondVec.elementAt(sz-2).setDistLower(8.40);
        hBondVec.elementAt(sz-1).setDistLower(7.38);

        //hBondVec.elementAt(sz-5).setDistUpper(9.73);
        hBondVec.elementAt(sz-4).setDistUpper(10.23);
        hBondVec.elementAt(sz-3).setDistUpper(8.32);
        hBondVec.elementAt(sz-2).setDistUpper(8.40);
        hBondVec.elementAt(sz-1).setDistUpper(7.38);

        hBondVec.remove(sz-1);

        hBondVec.clear();

        //myNoe ubqNoe1 = new myNoe(26, "CA", 17, "CA", 7.68, 7.68);
        //myNoe ubqNoe2 = new myNoe(29, "CA", 15, "CA", 7.78, 7.78);

        //myNoe ubqNoe3 = new myNoe(27, "CA", 43, "CA", 8.32, 8.32);
        //myNoe ubqNoe4 = new myNoe(26, "CA", 15, "CA", 8.40, 8.40);
        //myNoe ubqNoe5 = new myNoe(27, "CA", 41, "CA", 7.38, 7.38);


//assign ( resid     15  and name       HD1# )   ( resid     26 and name     HA )      5.132   3.332  0.0;

//assign ( resid     15  and name       HD1# )   ( resid     30 and name     HA )      3.518   1.718  0.0;

//assign ( resid     31  and name       HN )   ( resid     41 and name     HE21 )      6.000   4.200  0.0;


        myNoe ubqNoe3 = new myNoe(15, "HD1#", 26, "HA", 1.8, 5.132);
        myNoe ubqNoe4 = new myNoe(15, "HD1#", 30, "HA", 1.8, 3.518);
        myNoe ubqNoe5 = new myNoe(31, "H", 41, "HE21", 1.8, 6.0);


//assign ((resid     3 and name    HD1#))   ((resid    30 and name    HD1#))      4.110 4.110 1.028 !

//assign ((resid     5 and name    HG2#))   ((resid    30 and name    HD1#))      2.290 2.290 0.573 !

//assign ((resid    17 and name    HG2#))   ((resid    26 and name    HG2#))      3.584 3.584 0.896 !


        //myNoe ubqNoe3 = new myNoe(3, "HD1#", 30, "HD1#", 1.8, 5.138);
        //myNoe ubqNoe4 = new myNoe(15, "HD2#", 30, "H", 1.8, 5.098);
        

//assign ((resid    23 and name    HG2#))   ((resid    67 and name    HD1#) or
//                                           (resid    67 and name    HD2#))      4.127 4.127 1.032 !

        //myNoe ubqNoe5 = new myNoe(23, "HG2#", 67, "HD1#", 1.8, 5.159);


        //hBondVec.add(ubqNoe1);
        //hBondVec.add(ubqNoe2);
        hBondVec.add(ubqNoe3);
        hBondVec.add(ubqNoe4);
        hBondVec.add(ubqNoe5);

//assign ((resid     30 and name    CA))   ((resid    69 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     3 and name    CA))   ((resid    26 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     27 and name    CA))   ((resid  43 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     15 and name    CA))   ((resid    26 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     27 and name    CA))   ((resid  41 and name    CA))      8.0 8.0 0.0 !

/*
        hBondVec.clear();

        myNoe e1ghh = new myNoe(25,          "CA",        67,          "CA",         7.29,      7.29);
        myNoe e2ghh = new myNoe(28,          "CA",        66,          "CA",         7.29,      7.29);
        myNoe e3ghh = new myNoe(24,          "CA",        70,          "CA",         7.78,      7.78);
        myNoe e4ghh = new myNoe(32,          "CA",        63,          "CA",         8.54,      8.54);
        myNoe e5ghh = new myNoe(21,          "CA",        62,          "CA",        17.61,     17.61);
        myNoe e6ghh = new myNoe(31,          "CA",        70,          "CA",        12.94,     12.94);

        hBondVec.add(e1ghh);
        hBondVec.add(e2ghh);
       hBondVec.add(e3ghh);
       hBondVec.add(e4ghh);
        hBondVec.add(e5ghh);
        hBondVec.add(e6ghh);
*/
/*
        hBondVec.clear();

        myNoe e11ghh = new myNoe(22,          "CA",        41,          "CA",         6.42,      6.42);
        myNoe e12ghh = new myNoe(26,          "CA",        39,          "CA",         6.98,      6.98);
        myNoe e13ghh = new myNoe(50,          "CA",        64,          "CA",         8.88,      8.88);
        myNoe e14ghh = new myNoe( 5,          "CA",        22,          "CA",         8.84,      8.84);

        hBondVec.add(e11ghh);
        hBondVec.add(e12ghh);
        hBondVec.add(e13ghh);
        hBondVec.add(e14ghh);
*/
        myNoe.printNoeVector(hBondVec);

        }


//        String firstSseFile = "f1UbqCaC.pdb";//ff11.pdb";
//        String secondSseFile = "f2UbqCaC.pdb";//ff22.pdb";

        String firstSseFile = "29NovUbqHelix2.pdb";//"ghhOct24H1.pdb";//ff11.pdb";
        String secondSseFile = "29NovUbqSheet.pdb";//"ghhOct24H2.pdb";//ff22.pdb";

        //String firstSseFile = "h1h2packedOct24.pdb";//ghhH1H2Oct24pb.pdb";//ff11.pdb";
        //String secondSseFile = "ghhSSOct24.pdb";//ff22.pdb";

        Vector<myProtein> vp2 = new Vector<myProtein>();
        myPdbParser pParser2 = new myPdbParser(firstSseFile);
        while (pParser2.hasNextProtein()) {
            vp2.add(pParser2.nextProtein());
        }
        myProtein p1 = vp2.elementAt(0);

        Vector<myProtein> vp3 = new Vector<myProtein>();
        myPdbParser pParser3 = new myPdbParser(secondSseFile);
        while (pParser3.hasNextProtein()) {
            vp3.add(pParser3.nextProtein());
        }
        myProtein p2 = vp3.elementAt(0);

        myProtein p = __packRoot__(p1, p2, hBondVec, null);

    }


    private static double individualScore(myProtein p1, myProtein p2) {
        int numberOfAtomsInProtein1 = p1.numberOfAtoms();
        int numberOfAtomsInProtein2 = p2.numberOfAtoms();

        myStericChecker sc1 = new myStericChecker(1.0, 0.8, 0.4);
        double clashScore1 = sc1.checkStericClash(p1);
        //System.out.println("clashScore1: " + clashScore1);
        int clashCount1 = (int) Math.ceil(clashScore1 * numberOfAtomsInProtein1 / 1000.0);

        myStericChecker sc2 = new myStericChecker(1.0, 0.8, 0.4);
        double clashScore2 = sc2.checkStericClash(p2);
        //System.out.println("clashScore2: " + clashScore2);
        int clashCount2 = (int) Math.ceil(clashScore2 * numberOfAtomsInProtein2 / 1000.0);

        double combinedClashScoreFromIndividualContribution = (clashCount1 + clashCount2 + 0.0) / (numberOfAtomsInProtein1 + numberOfAtomsInProtein2 + 0.0) * 1000;

        return combinedClashScoreFromIndividualContribution;
    }

    private static myProtein __packRoot__(myProtein p1, myProtein p2, Vector<myNoe> noeVector, myProteinSequence seq) {
        Vector<myProtein> vp = myNewPacker.getFourFolds(p1);

        myNewPacker pk = new myNewPacker();

        pk.setGlobalBestOrBestFirst(true);
        pk.turnOnHbond(false);
        pk.__verbose = true;
        pk.turnOnSteric(true);

        pk.setNoeRmsdThreshold(0.6);
        pk.setAngleResolution(1);
        pk.setLengthResolution(0.1);

        for (myProtein wps : vp) {
             myTriple<myProtein, myProtein, Double> score = pk.__packSseUsingNoes__(wps, p2, noeVector, null);

             System.out.println("Printing the proteins and the packing score");
             System.out.println("Score: " + score.third());
             if (score.first() != null)
                score.first().print();
             if (score.second() != null)
                score.second().print();

             System.out.println("Read a char");
             inputChar.readChar();

             System.out.println("Stopping after one pair");
             //System.exit(1);
             //break;
        }

//        pk.get16Folds(p1,p2,hBondVec);
        return null;
    }


    public myTriple<myProtein, myProtein, Double> __packSseUsingNoes__(myProtein prot1, myProtein prot2, Vector<myNoe> noeVector, myProteinSequence seq) {
        if (__verbose) {
            System.out.println("I am in __packSseUsingNoes__");
        }

        double indScore = individualScore(prot1, prot2);

        System.out.println("IndividualScore: " + indScore);
        //System.exit(1);

        myProtein p1Clone = new myProtein(prot1);
        myProtein p2Clone = new myProtein(prot2);

        seq = thisProteinSeq;

        myRotamerLibrary thisRotamerLibrary = myRotamerLibrary.getInstance();

        // Change the poly-alanine models to actual ones
        for (myResidue r : p1Clone) {
            thisRotamerLibrary.mutateResidue(r, seq.getResidueType(r.getResidueNumber()));
        }

        for (myResidue r : p2Clone) {
            thisRotamerLibrary.mutateResidue(r, seq.getResidueType(r.getResidueNumber()));
        }

        Vector<myPoint> pp1 = new Vector<myPoint>(), pp2 = new Vector<myPoint>();
        Map<Integer, myPoint> pp1Map = new TreeMap<Integer, myPoint>();
        Map<Integer, myPoint> pp2Map = new TreeMap<Integer, myPoint>();
        Vector<myNoe> noeVec = new Vector<myNoe>();

        // get the NOEs that will be used here and then get the respective points (atoms)
        // in the respective fragments which describe the NOEs. The noes are added such that
        // each noe has first porton in the first fragment and second proton in the second
        // fragment.
        for (myNoe ee : noeVector) {
            int firstResNum = ee.getResidueNoA();
            int secondResNum = ee.getResidueNoB();
            if (p1Clone.residueAt(firstResNum) != null && p2Clone.residueAt(secondResNum) != null) {
                noeVec.add(new myNoe(ee));
            } else if (p1Clone.residueAt(secondResNum) != null && p2Clone.residueAt(firstResNum) != null) {
                myNoe nee = new myNoe(ee);
                nee.swap();
                noeVec.add(nee);
            } else; //Do not add
        }

        Vector<Vector<myPoint>> setA = new Vector<Vector<myPoint>>();
        Vector<Vector<myPoint>> setB = new Vector<Vector<myPoint>>();

        if (__verbose) {
            for (myNoe ee : noeVec) {
                ee.print();
            }
        }

        for (myNoe ee : noeVec) {
            // Now we have r1 and r2 and we know that there exists an noe between
            // two protons of these two residues. Let's make a set of the possible
            // protons inferred by the rotamer library.

            //test if a backbone atom or a side chain proton.
            Vector<myPoint> pointSetA = new Vector<myPoint>();
            Vector<myPoint> pointSetB = new Vector<myPoint>();

            GetCoordinateSetA:
            {
                int firstResNum = ee.getResidueNoA();
                String firstAtomName = ee.getAtomA();
                myResidue r1 = p1Clone.residueAt(firstResNum);

                //System.out.println("firstResidueName: " + r1.getResidueName() + "  firstAtomName: " + firstAtomName + "  firstResNum: " + firstResNum);

                if (isLoneBackboneProton(firstAtomName)) { // then point set A must contain only one element
                    myPoint thisPoint = new myPoint(r1.getAtom(firstAtomName).getCoordinates());
                    pointSetA.add(thisPoint);
                    if (__verbose) {
                        System.out.println("firstResidueName: " + r1.getResidueName() + "  firstResNum: " + firstResNum + "  newFirstAtomName: " + firstAtomName + "  coordinates: " + thisPoint.toString());
                    }
                } else { // The atom is a side-chain atom and can have multiple coordinates due to different rotamers
                    String firstResidueName = r1.getResidueName();
                    int numberOfRotamerForA = thisRotamerLibrary.numberOfRotamers(firstResidueName);

                    for (int i = 0; i < numberOfRotamerForA; i++) { // TODO: improve for '#'
                        thisRotamerLibrary.mutateResidue(r1, firstResidueName, i);
                        //r1.print();
                        if (firstAtomName.endsWith("#")) {
                            for (int suffix = 1; suffix <= 3; suffix++) {
                                String newFirstAtomName = firstAtomName.replace("#", Integer.toString(suffix));
                                if (r1.canHaveAtom(newFirstAtomName)) {
                                    myPoint thisPoint = new myPoint(r1.getAtom(newFirstAtomName).getCoordinates());
                                    pointSetA.add(thisPoint);
                                    if (__verbose) {
                                        System.out.println("firstResidueName: " + r1.getResidueName() + "  firstResNum: " + firstResNum + "  newFirstAtomName: " + newFirstAtomName + "  coordinates: " + thisPoint.toString());
                                    }
                                }
                            }
                        } else {
                            myPoint thisPoint = new myPoint(r1.getAtom(firstAtomName).getCoordinates());
                            pointSetA.add(thisPoint);
                            if (__verbose) {
                                System.out.println("firstResidueName: " + r1.getResidueName() + "  firstResNum: " + firstResNum + "  newFirstAtomName: " + firstAtomName + "  coordinates: " + thisPoint.toString());
                            }
                        }
                    }
                }

                pp1Map.put(r1.getResidueNumber(), new myPoint(r1.getAtom(myAtomLabel.__CA).getCoordinates()));
            }
            setA.add(pointSetA);

            GetCoordinateSetB:
            {
                int secondResNum = ee.getResidueNoB();
                String secondAtomName = ee.getAtomB();
                myResidue r2 = p2Clone.residueAt(secondResNum);

                //System.out.println("secondResidueName: " + r2.getResidueName() + "  secondAtomName: " + secondAtomName + "  secondResNum: " + secondResNum);

                if (isLoneBackboneProton(secondAtomName)) { // then point set A must contain only one element
                    myPoint thisPoint = new myPoint(r2.getAtom(secondAtomName).getCoordinates());
                    pointSetB.add(thisPoint);
                    if (__verbose) {
                        System.out.println("secondResidueName: " + r2.getResidueName() + "  secondResNum: " + secondResNum + "  newSecondAtomName: " + secondAtomName + "  coordinates: " + thisPoint.toString());
                    }
                } else { // The atom is a side-chain atom and can have multiple coordinates due to different rotamers
                    String secondResidueName = r2.getResidueName();
                    int numberOfRotamerForB = thisRotamerLibrary.numberOfRotamers(secondResidueName);

                    for (int i = 0; i < numberOfRotamerForB; i++) { // TODO: improve for '#'
                        thisRotamerLibrary.mutateResidue(r2, secondResidueName, i);
                        //r2.print();
                        if (secondAtomName.endsWith("#")) {
                            for (int suffix = 1; suffix <= 3; suffix++) {
                                String newSecondAtomName = secondAtomName.replace("#", Integer.toString(suffix));
                                if (r2.canHaveAtom(newSecondAtomName)) {
                                    myPoint thisPoint = new myPoint(r2.getAtom(newSecondAtomName).getCoordinates());
                                    pointSetB.add(thisPoint);
                                    if (__verbose) {
                                        System.out.println("secondResidueName: " + r2.getResidueName() + "  secondResNum: " + secondResNum + "  newSecondAtomName: " + newSecondAtomName + "  coordinates: " + thisPoint.toString());
                                    }
                                }
                            }
                        } else {
                            myPoint thisPoint = new myPoint(r2.getAtom(secondAtomName).getCoordinates());
                            pointSetB.add(thisPoint);
                            if (__verbose) {
                                System.out.println("secondResidueName: " + r2.getResidueName() + "  secondResNum: " + secondResNum + "  newSecondAtomName: " + secondAtomName + "  coordinates: " + thisPoint.toString());
                            }
                        }
                    }
                }

                pp2Map.put(r2.getResidueNumber(), new myPoint(r2.getAtom(myAtomLabel.__CA).getCoordinates()));
            }
            setB.add(pointSetB);

            if (__verbose) {
                System.out.println("Red Blue Set for this Noe:");

                System.out.println("Red set: ");
                for (myPoint p : pointSetA) {
                    System.out.println(p.toString());
                }

                System.out.println("Blue set: ");
                for (myPoint p : pointSetB) {
                    System.out.println(p.toString());
                }
            }
            //inputChar.readChar();
        }

        if (setA.size() != setB.size()) {
            System.out.println("Error: Noe set contains at least one noe whose other end is missing in the structure");
            System.exit(1);
        }

        if (__verbose) {
            System.out.println("Printing the red sets");
            for (Vector<myPoint> vm : setA) {
                for (myPoint p : vm) {
                    System.out.println(p.toString());
                }
                System.out.println();
            }

            System.out.println("Printing the blue sets");
            for (Vector<myPoint> vm : setB) {
                for (myPoint p : vm) {
                    System.out.println(p.toString());
                }
                System.out.println();
            }
        }

//        System.out.println("The NOEs used for packing:");
//        myNoe printerNoe = new myNoe();
//        printerNoe.PrintNOE(noeVec);

        // We previously used map to get rid of duplicates. Now transfer the coordinates to vectors.
        for (Map.Entry<Integer, myPoint> elem : pp1Map.entrySet()) {
            pp1.add(elem.getValue());
        }
        for (Map.Entry<Integer, myPoint> elem : pp2Map.entrySet()) {
            pp2.add(elem.getValue());
        }
        myPoint geomCenter1 = computeGeometricCenter(pp1);
        myPoint geomCenter2 = computeGeometricCenter(pp2);

        //System.out.println("Dist::::::: " + myPoint.distance(geomCenter1, geomCenter2)); System.exit(1);
        //for (myPoint p : pp1) System.out.print(p.toString());
        //System.out.println("Geometric Centers: (1) " + geomCenter1.toString() + "  Geometric Centers: (2) " + geomCenter2.toString());

        Vector<myPoint> pp1New = new Vector<myPoint>(pp1.size()), pp2New = new Vector<myPoint>(pp2.size());
        for (int i = 0; i < pp1.size(); i++) {
            pp1New.add(new myPoint(pp1.elementAt(i)));
        }
        for (int i = 0; i < pp2.size(); i++) {
            pp2New.add(new myPoint(pp2.elementAt(i)));
        }

        // shift both the fragments to origin
        computeTranslation(pp1New, myVector3D.reverse(new myVector3D(geomCenter1.getXYZ())));
        computeTranslation(pp2New, myVector3D.reverse(new myVector3D(geomCenter2.getXYZ())));

        // shift both the fragments' point clouds to the origin
        for (int i = 0; i < setA.size(); i++) { // Note that
            computeTranslation(setA.elementAt(i), myVector3D.reverse(new myVector3D(geomCenter1.getXYZ())));
            computeTranslation(setB.elementAt(i), myVector3D.reverse(new myVector3D(geomCenter2.getXYZ())));
        }

//        for (Vector<myPoint> s : setB) {
//            for (myPoint p : s) {
//                System.out.println(p.toString());
//            }
//        }
//        System.exit(1);
//        myPoint geomCenter1_ = computeGeometricCenter(pp1New);
//        System.out.println("gs1: " + geomCenter1_.toString());
//        myPoint geomCenter2_ = computeGeometricCenter(pp2New);
//        System.out.println("gs2: " + geomCenter2_.toString() + "  " + new myVector3D(geomCenter1_, geomCenter2_).norm());
//System.exit(1);

//        double meanNoe = 0.0;
//        for (int i = 0; i < noeVec.size(); i++) {
//            meanNoe += (noeVec.elementAt(i).getRange()[0] + noeVec.elementAt(i).getRange()[1]) / 2;
//        }
//        meanNoe /= noeVec.size();
//        System.out.println("Mean Noe: " + meanNoe);


        myTriple<myProtein, myProtein, Double> packedFirstFragmentSecondFragmentAndScore = null;
        myProtein packedSseSave = null;
        double clashScoreSave = Double.MAX_VALUE;
        double thetaSave = 0.0, phiSave = 0.0;
        double radiusSave = 0.0;
        double noeRmsdSave = Double.MAX_VALUE;
        Vector<myTriple<Integer, Integer, Double>> vtSave = null;
        double jointScoreSave = Double.MAX_VALUE;

        double meanBallRadius = (__max_noe_ball_radius + __min_noe_ball_radius) / 2; // This is preset, not a hard code.
        double ballThickness = (__max_noe_ball_radius - __min_noe_ball_radius); // This is preset, not a hard code.
        int rIter = (int) (ballThickness / __length_resolution);

        int numberOfPoints = 0;

        double noeRmsdAggressiveMin = 999.999;

        bestFirstLabel:
        for (int k = 0; k <= rIter; k++) { // <= to make sure that the boundaries are included
            //double radius = meanNoe + (k - halfrIter) * __length_resolution;
            // fancy zig-zag for loop since we expect the noe satisfaction to be somewhere around the middle of the ball
            int kp1 = (int) ((k + 1) / 2);
            int flipper = 1 - 2 * (k % 2);
            int gridNumber = kp1 * flipper;
            double radius = meanBallRadius + gridNumber * __length_resolution;
            //System.out.println(">>radius: " + radius);

            double newAngleResolution = __angle_resolution *  1 / Math.ceil(radius - __min_noe_ball_radius + 1); // / Math.ceil(4.0 * (k + 1) / (rIter * 1.0));
            //System.out.println(">>newAngleResolution: " + newAngleResolution);
            for (int m = 0; m < 180 / newAngleResolution; m++) {
                double theta = Math.toRadians(m * newAngleResolution);

                for (int j = 0; j < 360 / newAngleResolution; j++) {
                    double phi = Math.toRadians(j * newAngleResolution);

//                    numberOfPoints++;
//                    System.out.println("NumberOfPoints: " + numberOfPoints);
//
//                    boolean bb = true;
//                    if (bb)
//                        continue;

                    Vector<Vector<myPoint>> copyOfSetA = setA; // copyOfVectorOfVectorOfPointsIn3D(setA);
                    Vector<Vector<myPoint>> copyOfSetB = copyOfVectorOfVectorOfPointsIn3D(setB);

                    for (Vector<myPoint> vp : copyOfSetB) {
                        for (int i = 0; i < vp.size(); i++) {
                            myPoint p = myPoint.translate(vp.elementAt(i), new myVector3D(radius * Math.sin(theta) * Math.cos(phi),
                                    radius * Math.sin(theta) * Math.sin(phi), radius * Math.cos(theta)));
                            vp.set(i, p);
                        }
                    }

                    // Compute the noe deviation based on bichromatic clostst pair in 3D.
                    Vector<myTriple<Integer, Integer, Double>> vt = new Vector<myTriple<Integer, Integer, Double>>();
                    for (int i = 0; i < copyOfSetA.size(); i++) {
                        Vector<myPoint> redSet = copyOfSetA.elementAt(i);
                        Vector<myPoint> blueSet = copyOfSetB.elementAt(i);
                        myTriple<Integer, Integer, Double> score = bestNoeWithLeastDeviation(redSet, blueSet, noeVec.elementAt(i));
                        vt.add(score);
                    }

                    // Compute the noe rmsd
                    double noeRmsd = 0.0;
                    for (myTriple<Integer, Integer, Double> t : vt) {
                        //System.out.println("vt: " + t.first() + "  " + t.second() + "  " + t.third());
                        noeRmsd += t.third() * t.third();
                    }
                    //System.out.println();
                    noeRmsd = Math.sqrt(noeRmsd / vt.size());

                    if (noeRmsdAggressiveMin > noeRmsd) {
                        noeRmsdAggressiveMin = noeRmsd;
                        System.out.println("noermsd: " + noeRmsdAggressiveMin);
                    }

                    //System.out.println("noermsd: " + noeRmsd);
//noeRmsd = 0.0;
                    if (noeRmsd < __noe_rmsd_threshold) {
                        //System.out.println("I am here when noe rmsd threshold satisfied");
                        double clashScore = myMiscConstants.genericThreshold;
                        double hbScore = myMiscConstants.genericThreshold;
                        myProtein packedSse = null;
                        myProtein p1Backbone = null;
                        myProtein p2Backbone = null;

//if (__hbond_turned_on || __steric_turned_on) {
                        codeAddedForStericCheckAndHbond:
                        {
                            myPoint center2New = new myPoint(geomCenter2);
                            center2New.setX(-center2New.getX() + radius * Math.sin(theta) * Math.cos(phi));
                            center2New.setY(-center2New.getY() + radius * Math.sin(theta) * Math.sin(phi));
                            center2New.setZ(-center2New.getZ() + radius * Math.cos(theta));
                            p2Backbone = myProtein.extractBackbone(p2Clone);
                            p2Backbone.translate(center2New.getX() + geomCenter1.getX(), center2New.getY() + geomCenter1.getY(), center2New.getZ() + geomCenter1.getZ());

                            p1Backbone = myProtein.extractBackbone(p1Clone);

                            packedSse = myProtein.cloneAndMergeTwoProteinFragments(p1Backbone, p2Backbone);

                            // Do Hbond stuff before steric since it will save some time when it fails
                            if (__hbond_turned_on) {
                                // Now that we packed the two structures, we will check how well they do in terms of Hbonding.
                                myPair<Integer, myPair<Double, Double>> numberOfHbonds_angleRmsd_DistanceRmsd_triple = hBondPotentialForBackboneAtomsOnly(p1Backbone, p2Backbone);

                                // Next think of how to combine these scores
                                int numberOfHbonds = numberOfHbonds_angleRmsd_DistanceRmsd_triple.first();
                                double angleRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().first(); // expected to be < 0.75 for good strands
                                double hbondDistanceRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().second(); // expected to be < 0.75 for good strands

                                //System.out.println("Number of hBonds: " + numberOfHbonds);

                                double A = 1.0;// to make sure that an odd case like just one perfect hbond does not rule out a strong case like 10 non-perfect hbonds
                                double retScore = (numberOfHbonds == 0) ? myMiscConstants.genericThreshold : (A + angleRmsd + hbondDistanceRmsd) / numberOfHbonds; // test if numberOfHbond^2 can be a better choice

                                //if (retScore != myMiscConstants.genericThreshold) {
                                //System.out.println("Number of hBonds: " + numberOfHbonds);
                                //System.out.println("Hbond score: " + retScore);
                                //}
                                hbScore = retScore;
                            }

                            // Do the steric check.
                            if (__steric_turned_on) {
                                if (__hbond_turned_on && hbScore == myMiscConstants.genericThreshold) {
                                    clashScore = myMiscConstants.genericThreshold;
                                    //System.out.println("This segment of the code is valid");
                                } else { // do steric check
                                    myStericChecker sc = new myStericChecker(1.0, 0.8, 0.4);//(1.0, 1.0, 0.5); // ideal: (1.0, 0.8, 0.4);
                                    clashScore = sc.checkStericClash(packedSse);
                                    //System.out.println("Clash Score: " + clashScore);
                                }
                            }
                        }

                        // noeRmsd range: [0.0 - 0.75], hbScore range: [0.001 - 1.0], clashScore range: [0 - 40]

                        double jointScore = noeRmsd;

                        if (__steric_turned_on) {
// DEC 09                            jointScore += clashScore / 40.0; // This is not hardcode, rather a preset value that scales well with other scores
                        }

                        if (__hbond_turned_on) {
                            jointScore += 2 * hbScore; // so that it can be given more importance than noe rmsd
                        }

                    if (!(clashScore > indScore))
                        if (jointScore < jointScoreSave) {
                            jointScoreSave = jointScore;

                            clashScoreSave = clashScore;
                            packedSseSave = packedSse;
                            packedFirstFragmentSecondFragmentAndScore = new myTriple<myProtein, myProtein, Double>(p1Backbone, p2Backbone, jointScore);
                            vtSave = vt;

                            thetaSave = theta;
                            phiSave = phi;
                            noeRmsdSave = noeRmsd;
                            radiusSave = radius;

                            System.out.println("Radius: " + radiusSave);
                            System.out.println("Noe rmsd: " + noeRmsdSave); // Monitor the progress

                            if (__steric_turned_on) {
                                System.out.println("Clash score: " + clashScoreSave);
                            }

                            if (__hbond_turned_on) {
                                System.out.println("Hbond score: " + hbScore);
                            }

                            System.out.println("Joint Score: " + jointScoreSave + '\n');
                            if (jointScore < __joint_score_threshold) {
                                packedSseSave.print();
                                if (__global_best_or_best_first == false) {
                                    break bestFirstLabel;
                                }
                            }
                        }
                    }
                }
            }
        }

        //return packedFirstFragmentSecondFragmentAndScore;

        if (__steric_turned_on ||__hbond_turned_on) {//----does not seem to be so good
            if (packedFirstFragmentSecondFragmentAndScore == null)
                return new myTriple<myProtein, myProtein, Double>(null, null, myMiscConstants.scoreTermForSheetPackingThreshold);
            return packedFirstFragmentSecondFragmentAndScore;
        } else {
            System.out.println("Error: Code when steric and hbond filters are turned off is yet to be implemented");
            System.exit(1);
        }


        // TODO: The code below does not execute (basically unreachable) as in every reasonable case
        // we set both steric and hbond filters on, so it does not execute. So please remove the code
        // below at your convenience.


//        System.exit(1);

        System.out.println("Final pcking Results are below");
        System.out.println("Noe rmsd: " + noeRmsdSave);
        for (myTriple<Integer, Integer, Double> t : vtSave) {
            System.out.println("vt: " + t.first() + "  " + t.second() + "  " + t.third());
        }
        packedSseSave.print();
        System.exit(1);

        //System.out.println("NOE rmsd: " + disRmsd);
        //__noe_rmsds.add(disRmsd);
        //System.out.println("NOE distances difference for individual ones:");

        myPoint center2New = new myPoint(geomCenter2);
        center2New.setX(-center2New.getX() + radiusSave * Math.sin(thetaSave) * Math.cos(phiSave));
        center2New.setY(-center2New.getY() + radiusSave * Math.sin(thetaSave) * Math.sin(phiSave));
        center2New.setZ(-center2New.getZ() + radiusSave * Math.cos(thetaSave));

//        p1Clone.translate(-geomCenter1.getX(), -geomCenter1.getY(), -geomCenter1.getZ());
//        p2Clone.translate(center2New.getX(), center2New.getY(), center2New.getZ());

        // TODO: Check if it preserves p1Clone's coordinates
        //p1Clone.translate(-geomCenter1.getX(), -geomCenter1.getY(), -geomCenter1.getZ());
        p2Clone.translate(center2New.getX() + geomCenter1.getX(), center2New.getY() + geomCenter1.getY(), center2New.getZ() + geomCenter1.getZ());

        //p1Clone.print();
        //p2Clone.print();

        //myProtein ppnew = myProtein.cloneAndMergeTwoProteinFragments(myProtein.extractBackbone(p2Clone), myProtein.extractBackbone(p1Clone));
        //ppnew.print();

        System.exit(1);


//        if (disRmsd < 0.3) { // TODO: this is for debugging only, remove this later
//            System.out.println("This is a good pair of sheet + strand, printing this for debugging: ");
//            p1Clone.print();
//            p2Clone.print();
//        }

        // Now that we packed the two structures, we will check how well they do in terms of Hbonding.
        myPair<Integer, myPair<Double, Double>> numberOfHbonds_angleRmsd_DistanceRmsd_triple = hBondPotentialForBackboneAtomsOnly(p1Clone, p2Clone);

        // Next think of how to combine these scores
        int numberOfHbonds = numberOfHbonds_angleRmsd_DistanceRmsd_triple.first();
        double angleRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().first(); // expected to be < 0.75 for good strands
        double hbondDistanceRmsd = numberOfHbonds_angleRmsd_DistanceRmsd_triple.second().second(); // expected to be < 0.75 for good strands

        System.out.println("Number of hBonds: " + numberOfHbonds);

        double retScore = myMiscConstants.scoreTermForSheetPackingThreshold; // TODO: remove this peice of hCode

        if (noeRmsdSave > myMiscConstants.noeRmsdUpperBoundForSheetPacking || numberOfHbonds == 0) { // TODO: what is a better cutoff for disRmsd??
            return new myTriple<myProtein, myProtein, Double>(null, null, retScore);
        }

        System.out.println("NOE rmsd: " + noeRmsdSave + "    numberOfHbonds: " + numberOfHbonds);

        myProtein toBeStericChecked = myProtein.cloneAndMergeTwoProteinFragments(p1Clone, p2Clone);
        //System.out.println("Passing the protein to the steric checker");
        myStericChecker sc = new myStericChecker(1.0, 1.0, 0.5); // ideal: (1.0, 0.8, 0.4);
        double clashScore = sc.checkStericClash(toBeStericChecked);
        //System.out.println("Returning from the steric checker");
        System.out.println("Clash Score: " + clashScore);

        if (clashScore > myMiscConstants.clashScoreThresholdForSheetPacking) {
            return new myTriple<myProtein, myProtein, Double>(null, null, retScore);
        }

        if (noeRmsdSave < 0.4) { // TODO: this is for debugging only, remove this later
            System.out.println("Note: (this comment is used for Debugging) This can be good pair of sheet + strand that satisfies noes, steric, hbond but NOT sure about RDC: ");
            toBeStericChecked.print();
        }

        // TODO: this is a simple scoring function with the objective that
        // the lower its value, the the better hbonded the two strands are.
        double A = 1.0;// to make sure that an odd case like just one perfect hbond does not rule out a strong case like 10 non-perfect hbonds
        retScore = (A + angleRmsd + hbondDistanceRmsd) / numberOfHbonds; // test if numberOfHbond^2 can be a better choice

        return new myTriple<myProtein, myProtein, Double>(p1Clone, p2Clone, retScore);

        //myProtein ret = myProtein.cloneAndMergeTwoProteinFragments(p1Clone, p2Clone);
        //return ret;
    }









    ////////////////////////////////////////////////////////////////////////////////////////


    public static void main(String... args) {

        Vector<myNoe> hBondVec = null;
        temporaryNoeReader:
        {
            String userDir = System.getProperty("user.dir");
            String fileSeparator = System.getProperty("file.separator");
            String inputDirectory = userDir + fileSeparator + "input_files_1d3z" + fileSeparator + "input_data_files" + fileSeparator;
            String inputFile = inputDirectory + "noe.txt";
            //System.out.println("inputFile: " + inputFile);
            hBondVec = myNoe.parseNoeFileInXplorFormat(inputFile); //new Vector<myNoe>(Arrays.asList(noes));

//TODO: get rid of non bb noes. implement rigorously.
            //for (int i = 0; i < 5; i++)
            //    hBondVec.remove(hBondVec.size() - 1);

        int sz = hBondVec.size();

        //hBondVec.clear();


//      These are the NOEs for UBQ
        //hBondVec.elementAt(sz-5).setDistLower(9.73);
        hBondVec.elementAt(sz-4).setDistLower(10.23);
        hBondVec.elementAt(sz-3).setDistLower(8.32);
        hBondVec.elementAt(sz-2).setDistLower(8.40);
        hBondVec.elementAt(sz-1).setDistLower(7.38);

        //hBondVec.elementAt(sz-5).setDistUpper(9.73);
        hBondVec.elementAt(sz-4).setDistUpper(10.23);
        hBondVec.elementAt(sz-3).setDistUpper(8.32);
        hBondVec.elementAt(sz-2).setDistUpper(8.40);
        hBondVec.elementAt(sz-1).setDistUpper(7.38);

        hBondVec.remove(sz-1);

        hBondVec.clear();

        //myNoe ubqNoe1 = new myNoe(26, "CA", 17, "CA", 7.68, 7.68);
        //myNoe ubqNoe2 = new myNoe(29, "CA", 15, "CA", 7.78, 7.78);

        //myNoe ubqNoe3 = new myNoe(27, "CA", 43, "CA", 8.32, 8.32);
        //myNoe ubqNoe4 = new myNoe(26, "CA", 15, "CA", 8.40, 8.40);
        //myNoe ubqNoe5 = new myNoe(27, "CA", 41, "CA", 7.38, 7.38);


//assign ( resid     15  and name       HD1# )   ( resid     26 and name     HA )      5.132   3.332  0.0;

//assign ( resid     15  and name       HD1# )   ( resid     30 and name     HA )      3.518   1.718  0.0;

//assign ( resid     31  and name       HN )   ( resid     41 and name     HE21 )      6.000   4.200  0.0;


//        myNoe ubqNoe3 = new myNoe(15, "HD1#", 26, "HA", 1.8, 5.132);
//        myNoe ubqNoe4 = new myNoe(15, "HD1#", 30, "HA", 1.8, 3.518);
//        myNoe ubqNoe5 = new myNoe(31, "H", 41, "HE21", 1.8, 6.0);

        myNoe ubqNoe3 = new myNoe(43, "H", 50, "H", 1.8, 4.4);
        myNoe ubqNoe4 = new myNoe(44, "HA", 49, "HA", 1.8, 2.8);
        myNoe ubqNoe5 = new myNoe(45, "H", 48, "H", 2.6, 2.84);        


//assign ((resid     3 and name    HD1#))   ((resid    30 and name    HD1#))      4.110 4.110 1.028 !

//assign ((resid     5 and name    HG2#))   ((resid    30 and name    HD1#))      2.290 2.290 0.573 !

//assign ((resid    17 and name    HG2#))   ((resid    26 and name    HG2#))      3.584 3.584 0.896 !


        //myNoe ubqNoe3 = new myNoe(3, "HD1#", 30, "HD1#", 1.8, 5.138);
        //myNoe ubqNoe4 = new myNoe(15, "HD2#", 30, "H", 1.8, 5.098);


//assign ((resid    23 and name    HG2#))   ((resid    67 and name    HD1#) or
//                                           (resid    67 and name    HD2#))      4.127 4.127 1.032 !

        //myNoe ubqNoe5 = new myNoe(23, "HG2#", 67, "HD1#", 1.8, 5.159);


        //hBondVec.add(ubqNoe1);
        //hBondVec.add(ubqNoe2);
        hBondVec.add(ubqNoe3);
        hBondVec.add(ubqNoe4);
        hBondVec.add(ubqNoe5);

//assign ((resid     30 and name    CA))   ((resid    69 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     3 and name    CA))   ((resid    26 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     27 and name    CA))   ((resid  43 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     15 and name    CA))   ((resid    26 and name    CA))      8.0 8.0 0.0 !
//assign ((resid     27 and name    CA))   ((resid  41 and name    CA))      8.0 8.0 0.0 !

/*
        hBondVec.clear();

        myNoe e1ghh = new myNoe(25,          "CA",        67,          "CA",         7.29,      7.29);
        myNoe e2ghh = new myNoe(28,          "CA",        66,          "CA",         7.29,      7.29);
        myNoe e3ghh = new myNoe(24,          "CA",        70,          "CA",         7.78,      7.78);
        myNoe e4ghh = new myNoe(32,          "CA",        63,          "CA",         8.54,      8.54);
        myNoe e5ghh = new myNoe(21,          "CA",        62,          "CA",        17.61,     17.61);
        myNoe e6ghh = new myNoe(31,          "CA",        70,          "CA",        12.94,     12.94);

        hBondVec.add(e1ghh);
        hBondVec.add(e2ghh);
       hBondVec.add(e3ghh);
       hBondVec.add(e4ghh);
        hBondVec.add(e5ghh);
        hBondVec.add(e6ghh);
*/
/*
        hBondVec.clear();

        myNoe e11ghh = new myNoe(22,          "CA",        41,          "CA",         6.42,      6.42);
        myNoe e12ghh = new myNoe(26,          "CA",        39,          "CA",         6.98,      6.98);
        myNoe e13ghh = new myNoe(50,          "CA",        64,          "CA",         8.88,      8.88);
        myNoe e14ghh = new myNoe( 5,          "CA",        22,          "CA",         8.84,      8.84);

        hBondVec.add(e11ghh);
        hBondVec.add(e12ghh);
        hBondVec.add(e13ghh);
        hBondVec.add(e14ghh);
*/
        myNoe.printNoeVector(hBondVec);

        }


//        String firstSseFile = "f1UbqCaC.pdb";//ff11.pdb";
//        String secondSseFile = "f2UbqCaC.pdb";//ff22.pdb";

        String firstSseFile = "00Str1.pdb";//"ghhOct24H1.pdb";//ff11.pdb";
        String secondSseFile = "00Str2.pdb";//"ghhOct24H2.pdb";//ff22.pdb";

        //String firstSseFile = "h1h2packedOct24.pdb";//ghhH1H2Oct24pb.pdb";//ff11.pdb";
        //String secondSseFile = "ghhSSOct24.pdb";//ff22.pdb";

        Vector<myProtein> vp2 = new Vector<myProtein>();
        myPdbParser pParser2 = new myPdbParser(firstSseFile);
        while (pParser2.hasNextProtein()) {
            vp2.add(pParser2.nextProtein());
        }
        myProtein p1 = vp2.elementAt(0);

        Vector<myProtein> vp3 = new Vector<myProtein>();
        myPdbParser pParser3 = new myPdbParser(secondSseFile);
        while (pParser3.hasNextProtein()) {
            vp3.add(pParser3.nextProtein());
        }
        myProtein p2 = vp3.elementAt(0);

        myProtein p = __packRoot__(p1, p2, hBondVec, null);

    }

}
